/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appmesh;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.appmesh.model.AppMeshException;
import software.amazon.awssdk.services.appmesh.model.AppMeshRequest;
import software.amazon.awssdk.services.appmesh.model.BadRequestException;
import software.amazon.awssdk.services.appmesh.model.ConflictException;
import software.amazon.awssdk.services.appmesh.model.CreateGatewayRouteRequest;
import software.amazon.awssdk.services.appmesh.model.CreateGatewayRouteResponse;
import software.amazon.awssdk.services.appmesh.model.CreateMeshRequest;
import software.amazon.awssdk.services.appmesh.model.CreateMeshResponse;
import software.amazon.awssdk.services.appmesh.model.CreateRouteRequest;
import software.amazon.awssdk.services.appmesh.model.CreateRouteResponse;
import software.amazon.awssdk.services.appmesh.model.CreateVirtualGatewayRequest;
import software.amazon.awssdk.services.appmesh.model.CreateVirtualGatewayResponse;
import software.amazon.awssdk.services.appmesh.model.CreateVirtualNodeRequest;
import software.amazon.awssdk.services.appmesh.model.CreateVirtualNodeResponse;
import software.amazon.awssdk.services.appmesh.model.CreateVirtualRouterRequest;
import software.amazon.awssdk.services.appmesh.model.CreateVirtualRouterResponse;
import software.amazon.awssdk.services.appmesh.model.CreateVirtualServiceRequest;
import software.amazon.awssdk.services.appmesh.model.CreateVirtualServiceResponse;
import software.amazon.awssdk.services.appmesh.model.DeleteGatewayRouteRequest;
import software.amazon.awssdk.services.appmesh.model.DeleteGatewayRouteResponse;
import software.amazon.awssdk.services.appmesh.model.DeleteMeshRequest;
import software.amazon.awssdk.services.appmesh.model.DeleteMeshResponse;
import software.amazon.awssdk.services.appmesh.model.DeleteRouteRequest;
import software.amazon.awssdk.services.appmesh.model.DeleteRouteResponse;
import software.amazon.awssdk.services.appmesh.model.DeleteVirtualGatewayRequest;
import software.amazon.awssdk.services.appmesh.model.DeleteVirtualGatewayResponse;
import software.amazon.awssdk.services.appmesh.model.DeleteVirtualNodeRequest;
import software.amazon.awssdk.services.appmesh.model.DeleteVirtualNodeResponse;
import software.amazon.awssdk.services.appmesh.model.DeleteVirtualRouterRequest;
import software.amazon.awssdk.services.appmesh.model.DeleteVirtualRouterResponse;
import software.amazon.awssdk.services.appmesh.model.DeleteVirtualServiceRequest;
import software.amazon.awssdk.services.appmesh.model.DeleteVirtualServiceResponse;
import software.amazon.awssdk.services.appmesh.model.DescribeGatewayRouteRequest;
import software.amazon.awssdk.services.appmesh.model.DescribeGatewayRouteResponse;
import software.amazon.awssdk.services.appmesh.model.DescribeMeshRequest;
import software.amazon.awssdk.services.appmesh.model.DescribeMeshResponse;
import software.amazon.awssdk.services.appmesh.model.DescribeRouteRequest;
import software.amazon.awssdk.services.appmesh.model.DescribeRouteResponse;
import software.amazon.awssdk.services.appmesh.model.DescribeVirtualGatewayRequest;
import software.amazon.awssdk.services.appmesh.model.DescribeVirtualGatewayResponse;
import software.amazon.awssdk.services.appmesh.model.DescribeVirtualNodeRequest;
import software.amazon.awssdk.services.appmesh.model.DescribeVirtualNodeResponse;
import software.amazon.awssdk.services.appmesh.model.DescribeVirtualRouterRequest;
import software.amazon.awssdk.services.appmesh.model.DescribeVirtualRouterResponse;
import software.amazon.awssdk.services.appmesh.model.DescribeVirtualServiceRequest;
import software.amazon.awssdk.services.appmesh.model.DescribeVirtualServiceResponse;
import software.amazon.awssdk.services.appmesh.model.ForbiddenException;
import software.amazon.awssdk.services.appmesh.model.InternalServerErrorException;
import software.amazon.awssdk.services.appmesh.model.LimitExceededException;
import software.amazon.awssdk.services.appmesh.model.ListGatewayRoutesRequest;
import software.amazon.awssdk.services.appmesh.model.ListGatewayRoutesResponse;
import software.amazon.awssdk.services.appmesh.model.ListMeshesRequest;
import software.amazon.awssdk.services.appmesh.model.ListMeshesResponse;
import software.amazon.awssdk.services.appmesh.model.ListRoutesRequest;
import software.amazon.awssdk.services.appmesh.model.ListRoutesResponse;
import software.amazon.awssdk.services.appmesh.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.appmesh.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.appmesh.model.ListVirtualGatewaysRequest;
import software.amazon.awssdk.services.appmesh.model.ListVirtualGatewaysResponse;
import software.amazon.awssdk.services.appmesh.model.ListVirtualNodesRequest;
import software.amazon.awssdk.services.appmesh.model.ListVirtualNodesResponse;
import software.amazon.awssdk.services.appmesh.model.ListVirtualRoutersRequest;
import software.amazon.awssdk.services.appmesh.model.ListVirtualRoutersResponse;
import software.amazon.awssdk.services.appmesh.model.ListVirtualServicesRequest;
import software.amazon.awssdk.services.appmesh.model.ListVirtualServicesResponse;
import software.amazon.awssdk.services.appmesh.model.NotFoundException;
import software.amazon.awssdk.services.appmesh.model.ResourceInUseException;
import software.amazon.awssdk.services.appmesh.model.ServiceUnavailableException;
import software.amazon.awssdk.services.appmesh.model.TagResourceRequest;
import software.amazon.awssdk.services.appmesh.model.TagResourceResponse;
import software.amazon.awssdk.services.appmesh.model.TooManyRequestsException;
import software.amazon.awssdk.services.appmesh.model.TooManyTagsException;
import software.amazon.awssdk.services.appmesh.model.UntagResourceRequest;
import software.amazon.awssdk.services.appmesh.model.UntagResourceResponse;
import software.amazon.awssdk.services.appmesh.model.UpdateGatewayRouteRequest;
import software.amazon.awssdk.services.appmesh.model.UpdateGatewayRouteResponse;
import software.amazon.awssdk.services.appmesh.model.UpdateMeshRequest;
import software.amazon.awssdk.services.appmesh.model.UpdateMeshResponse;
import software.amazon.awssdk.services.appmesh.model.UpdateRouteRequest;
import software.amazon.awssdk.services.appmesh.model.UpdateRouteResponse;
import software.amazon.awssdk.services.appmesh.model.UpdateVirtualGatewayRequest;
import software.amazon.awssdk.services.appmesh.model.UpdateVirtualGatewayResponse;
import software.amazon.awssdk.services.appmesh.model.UpdateVirtualNodeRequest;
import software.amazon.awssdk.services.appmesh.model.UpdateVirtualNodeResponse;
import software.amazon.awssdk.services.appmesh.model.UpdateVirtualRouterRequest;
import software.amazon.awssdk.services.appmesh.model.UpdateVirtualRouterResponse;
import software.amazon.awssdk.services.appmesh.model.UpdateVirtualServiceRequest;
import software.amazon.awssdk.services.appmesh.model.UpdateVirtualServiceResponse;
import software.amazon.awssdk.services.appmesh.paginators.ListGatewayRoutesIterable;
import software.amazon.awssdk.services.appmesh.paginators.ListMeshesIterable;
import software.amazon.awssdk.services.appmesh.paginators.ListRoutesIterable;
import software.amazon.awssdk.services.appmesh.paginators.ListTagsForResourceIterable;
import software.amazon.awssdk.services.appmesh.paginators.ListVirtualGatewaysIterable;
import software.amazon.awssdk.services.appmesh.paginators.ListVirtualNodesIterable;
import software.amazon.awssdk.services.appmesh.paginators.ListVirtualRoutersIterable;
import software.amazon.awssdk.services.appmesh.paginators.ListVirtualServicesIterable;
import software.amazon.awssdk.services.appmesh.transform.CreateGatewayRouteRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.CreateMeshRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.CreateRouteRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.CreateVirtualGatewayRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.CreateVirtualNodeRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.CreateVirtualRouterRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.CreateVirtualServiceRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DeleteGatewayRouteRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DeleteMeshRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DeleteRouteRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DeleteVirtualGatewayRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DeleteVirtualNodeRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DeleteVirtualRouterRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DeleteVirtualServiceRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DescribeGatewayRouteRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DescribeMeshRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DescribeRouteRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DescribeVirtualGatewayRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DescribeVirtualNodeRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DescribeVirtualRouterRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DescribeVirtualServiceRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.ListGatewayRoutesRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.ListMeshesRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.ListRoutesRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.ListVirtualGatewaysRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.ListVirtualNodesRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.ListVirtualRoutersRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.ListVirtualServicesRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.UpdateGatewayRouteRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.UpdateMeshRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.UpdateRouteRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.UpdateVirtualGatewayRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.UpdateVirtualNodeRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.UpdateVirtualRouterRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.UpdateVirtualServiceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link AppMeshClient}.
 *
 * @see AppMeshClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAppMeshClient implements AppMeshClient {
    private static final Logger log = Logger.loggerFor(DefaultAppMeshClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAppMeshClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a gateway route.
     * </p>
     * <p>
     * A gateway route is attached to a virtual gateway and routes traffic to an existing virtual service. If a route
     * matches a request, it can distribute traffic to a target virtual service.
     * </p>
     * <p>
     * For more information about gateway routes, see <a
     * href="https://docs.aws.amazon.com/app-mesh/latest/userguide/gateway-routes.html">Gateway routes</a>.
     * </p>
     *
     * @param createGatewayRouteRequest
     * @return Result of the CreateGatewayRoute operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws ConflictException
     *         The request contains a client token that was used for a previous update resource call with different
     *         specifications. Try the request again with a new client token.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws LimitExceededException
     *         You have exceeded a service limit for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>AWS App Mesh User Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.CreateGatewayRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/CreateGatewayRoute" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateGatewayRouteResponse createGatewayRoute(CreateGatewayRouteRequest createGatewayRouteRequest)
            throws NotFoundException, BadRequestException, ConflictException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, LimitExceededException, AwsServiceException,
            SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateGatewayRouteResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateGatewayRouteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGatewayRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGatewayRoute");

            return clientHandler.execute(new ClientExecutionParams<CreateGatewayRouteRequest, CreateGatewayRouteResponse>()
                    .withOperationName("CreateGatewayRoute").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createGatewayRouteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateGatewayRouteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a service mesh.
     * </p>
     * <p>
     * A service mesh is a logical boundary for network traffic between services that are represented by resources
     * within the mesh. After you create your service mesh, you can create virtual services, virtual nodes, virtual
     * routers, and routes to distribute traffic between the applications in your mesh.
     * </p>
     * <p>
     * For more information about service meshes, see <a
     * href="https://docs.aws.amazon.com/app-mesh/latest/userguide/meshes.html">Service meshes</a>.
     * </p>
     *
     * @param createMeshRequest
     * @return Result of the CreateMesh operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws ConflictException
     *         The request contains a client token that was used for a previous update resource call with different
     *         specifications. Try the request again with a new client token.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws LimitExceededException
     *         You have exceeded a service limit for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>AWS App Mesh User Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.CreateMesh
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/CreateMesh" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateMeshResponse createMesh(CreateMeshRequest createMeshRequest) throws NotFoundException, BadRequestException,
            ConflictException, TooManyRequestsException, ForbiddenException, ServiceUnavailableException,
            InternalServerErrorException, LimitExceededException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateMeshResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateMeshResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMeshRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMesh");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateMeshRequest, CreateMeshResponse>().withOperationName("CreateMesh")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createMeshRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateMeshRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a route that is associated with a virtual router.
     * </p>
     * <p>
     * You can route several different protocols and define a retry policy for a route. Traffic can be routed to one or
     * more virtual nodes.
     * </p>
     * <p>
     * For more information about routes, see <a
     * href="https://docs.aws.amazon.com/app-mesh/latest/userguide/routes.html">Routes</a>.
     * </p>
     *
     * @param createRouteRequest
     * @return Result of the CreateRoute operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws ConflictException
     *         The request contains a client token that was used for a previous update resource call with different
     *         specifications. Try the request again with a new client token.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws LimitExceededException
     *         You have exceeded a service limit for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>AWS App Mesh User Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.CreateRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/CreateRoute" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateRouteResponse createRoute(CreateRouteRequest createRouteRequest) throws NotFoundException, BadRequestException,
            ConflictException, TooManyRequestsException, ForbiddenException, ServiceUnavailableException,
            InternalServerErrorException, LimitExceededException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateRouteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateRouteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRoute");

            return clientHandler.execute(new ClientExecutionParams<CreateRouteRequest, CreateRouteResponse>()
                    .withOperationName("CreateRoute").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createRouteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateRouteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a virtual gateway.
     * </p>
     * <p>
     * A virtual gateway allows resources outside your mesh to communicate to resources that are inside your mesh. The
     * virtual gateway represents an Envoy proxy running in an Amazon ECS task, in a Kubernetes service, or on an Amazon
     * EC2 instance. Unlike a virtual node, which represents an Envoy running with an application, a virtual gateway
     * represents Envoy deployed by itself.
     * </p>
     * <p>
     * For more information about virtual gateways, see <a
     * href="https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html">Virtual gateways</a>.
     * </p>
     *
     * @param createVirtualGatewayRequest
     * @return Result of the CreateVirtualGateway operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws ConflictException
     *         The request contains a client token that was used for a previous update resource call with different
     *         specifications. Try the request again with a new client token.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws LimitExceededException
     *         You have exceeded a service limit for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>AWS App Mesh User Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.CreateVirtualGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/CreateVirtualGateway" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateVirtualGatewayResponse createVirtualGateway(CreateVirtualGatewayRequest createVirtualGatewayRequest)
            throws NotFoundException, BadRequestException, ConflictException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, LimitExceededException, AwsServiceException,
            SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateVirtualGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateVirtualGatewayResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVirtualGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVirtualGateway");

            return clientHandler.execute(new ClientExecutionParams<CreateVirtualGatewayRequest, CreateVirtualGatewayResponse>()
                    .withOperationName("CreateVirtualGateway").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createVirtualGatewayRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateVirtualGatewayRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a virtual node within a service mesh.
     * </p>
     * <p>
     * A virtual node acts as a logical pointer to a particular task group, such as an Amazon ECS service or a
     * Kubernetes deployment. When you create a virtual node, you can specify the service discovery information for your
     * task group, and whether the proxy running in a task group will communicate with other proxies using Transport
     * Layer Security (TLS).
     * </p>
     * <p>
     * You define a <code>listener</code> for any inbound traffic that your virtual node expects. Any virtual service
     * that your virtual node expects to communicate to is specified as a <code>backend</code>.
     * </p>
     * <p>
     * The response metadata for your new virtual node contains the <code>arn</code> that is associated with the virtual
     * node. Set this value to the full ARN; for example,
     * <code>arn:aws:appmesh:us-west-2:123456789012:myMesh/default/virtualNode/myApp</code>) as the
     * <code>APPMESH_RESOURCE_ARN</code> environment variable for your task group's Envoy proxy container in your task
     * definition or pod spec. This is then mapped to the <code>node.id</code> and <code>node.cluster</code> Envoy
     * parameters.
     * </p>
     * <note>
     * <p>
     * By default, App Mesh uses the name of the resource you specified in <code>APPMESH_RESOURCE_ARN</code> when Envoy
     * is referring to itself in metrics and traces. You can override this behavior by setting the
     * <code>APPMESH_RESOURCE_CLUSTER</code> environment variable with your own name.
     * </p>
     * <p>
     * AWS Cloud Map is not available in the eu-south-1 Region.
     * </p>
     * </note>
     * <p>
     * For more information about virtual nodes, see <a
     * href="https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html">Virtual nodes</a>. You must be
     * using <code>1.15.0</code> or later of the Envoy image when setting these variables. For more information about
     * App Mesh Envoy variables, see <a href="https://docs.aws.amazon.com/app-mesh/latest/userguide/envoy.html">Envoy
     * image</a> in the AWS App Mesh User Guide.
     * </p>
     *
     * @param createVirtualNodeRequest
     * @return Result of the CreateVirtualNode operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws ConflictException
     *         The request contains a client token that was used for a previous update resource call with different
     *         specifications. Try the request again with a new client token.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws LimitExceededException
     *         You have exceeded a service limit for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>AWS App Mesh User Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.CreateVirtualNode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/CreateVirtualNode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateVirtualNodeResponse createVirtualNode(CreateVirtualNodeRequest createVirtualNodeRequest)
            throws NotFoundException, BadRequestException, ConflictException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, LimitExceededException, AwsServiceException,
            SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateVirtualNodeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateVirtualNodeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVirtualNodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVirtualNode");

            return clientHandler.execute(new ClientExecutionParams<CreateVirtualNodeRequest, CreateVirtualNodeResponse>()
                    .withOperationName("CreateVirtualNode").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createVirtualNodeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateVirtualNodeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a virtual router within a service mesh.
     * </p>
     * <p>
     * Specify a <code>listener</code> for any inbound traffic that your virtual router receives. Create a virtual
     * router for each protocol and port that you need to route. Virtual routers handle traffic for one or more virtual
     * services within your mesh. After you create your virtual router, create and associate routes for your virtual
     * router that direct incoming requests to different virtual nodes.
     * </p>
     * <p>
     * For more information about virtual routers, see <a
     * href="https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_routers.html">Virtual routers</a>.
     * </p>
     *
     * @param createVirtualRouterRequest
     * @return Result of the CreateVirtualRouter operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws ConflictException
     *         The request contains a client token that was used for a previous update resource call with different
     *         specifications. Try the request again with a new client token.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws LimitExceededException
     *         You have exceeded a service limit for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>AWS App Mesh User Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.CreateVirtualRouter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/CreateVirtualRouter" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateVirtualRouterResponse createVirtualRouter(CreateVirtualRouterRequest createVirtualRouterRequest)
            throws NotFoundException, BadRequestException, ConflictException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, LimitExceededException, AwsServiceException,
            SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateVirtualRouterResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateVirtualRouterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVirtualRouterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVirtualRouter");

            return clientHandler.execute(new ClientExecutionParams<CreateVirtualRouterRequest, CreateVirtualRouterResponse>()
                    .withOperationName("CreateVirtualRouter").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createVirtualRouterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateVirtualRouterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a virtual service within a service mesh.
     * </p>
     * <p>
     * A virtual service is an abstraction of a real service that is provided by a virtual node directly or indirectly
     * by means of a virtual router. Dependent services call your virtual service by its <code>virtualServiceName</code>
     * , and those requests are routed to the virtual node or virtual router that is specified as the provider for the
     * virtual service.
     * </p>
     * <p>
     * For more information about virtual services, see <a
     * href="https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html">Virtual services</a>.
     * </p>
     *
     * @param createVirtualServiceRequest
     * @return Result of the CreateVirtualService operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws ConflictException
     *         The request contains a client token that was used for a previous update resource call with different
     *         specifications. Try the request again with a new client token.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws LimitExceededException
     *         You have exceeded a service limit for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>AWS App Mesh User Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.CreateVirtualService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/CreateVirtualService" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateVirtualServiceResponse createVirtualService(CreateVirtualServiceRequest createVirtualServiceRequest)
            throws NotFoundException, BadRequestException, ConflictException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, LimitExceededException, AwsServiceException,
            SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateVirtualServiceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateVirtualServiceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVirtualServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVirtualService");

            return clientHandler.execute(new ClientExecutionParams<CreateVirtualServiceRequest, CreateVirtualServiceResponse>()
                    .withOperationName("CreateVirtualService").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createVirtualServiceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateVirtualServiceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an existing gateway route.
     * </p>
     *
     * @param deleteGatewayRouteRequest
     * @return Result of the DeleteGatewayRoute operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ResourceInUseException
     *         You can't delete the specified resource because it's in use or required by another resource.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.DeleteGatewayRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DeleteGatewayRoute" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteGatewayRouteResponse deleteGatewayRoute(DeleteGatewayRouteRequest deleteGatewayRouteRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException, ResourceInUseException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteGatewayRouteResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteGatewayRouteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGatewayRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGatewayRoute");

            return clientHandler.execute(new ClientExecutionParams<DeleteGatewayRouteRequest, DeleteGatewayRouteResponse>()
                    .withOperationName("DeleteGatewayRoute").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteGatewayRouteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteGatewayRouteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an existing service mesh.
     * </p>
     * <p>
     * You must delete all resources (virtual services, routes, virtual routers, and virtual nodes) in the service mesh
     * before you can delete the mesh itself.
     * </p>
     *
     * @param deleteMeshRequest
     * @return Result of the DeleteMesh operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ResourceInUseException
     *         You can't delete the specified resource because it's in use or required by another resource.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.DeleteMesh
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DeleteMesh" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteMeshResponse deleteMesh(DeleteMeshRequest deleteMeshRequest) throws NotFoundException, BadRequestException,
            TooManyRequestsException, ForbiddenException, ResourceInUseException, ServiceUnavailableException,
            InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteMeshResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteMeshResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMeshRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMesh");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteMeshRequest, DeleteMeshResponse>().withOperationName("DeleteMesh")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteMeshRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteMeshRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an existing route.
     * </p>
     *
     * @param deleteRouteRequest
     * @return Result of the DeleteRoute operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ResourceInUseException
     *         You can't delete the specified resource because it's in use or required by another resource.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.DeleteRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DeleteRoute" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteRouteResponse deleteRoute(DeleteRouteRequest deleteRouteRequest) throws NotFoundException, BadRequestException,
            TooManyRequestsException, ForbiddenException, ResourceInUseException, ServiceUnavailableException,
            InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRouteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteRouteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRoute");

            return clientHandler.execute(new ClientExecutionParams<DeleteRouteRequest, DeleteRouteResponse>()
                    .withOperationName("DeleteRoute").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteRouteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteRouteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an existing virtual gateway. You cannot delete a virtual gateway if any gateway routes are associated to
     * it.
     * </p>
     *
     * @param deleteVirtualGatewayRequest
     * @return Result of the DeleteVirtualGateway operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ResourceInUseException
     *         You can't delete the specified resource because it's in use or required by another resource.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.DeleteVirtualGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DeleteVirtualGateway" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteVirtualGatewayResponse deleteVirtualGateway(DeleteVirtualGatewayRequest deleteVirtualGatewayRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException, ResourceInUseException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteVirtualGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteVirtualGatewayResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVirtualGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVirtualGateway");

            return clientHandler.execute(new ClientExecutionParams<DeleteVirtualGatewayRequest, DeleteVirtualGatewayResponse>()
                    .withOperationName("DeleteVirtualGateway").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteVirtualGatewayRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteVirtualGatewayRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an existing virtual node.
     * </p>
     * <p>
     * You must delete any virtual services that list a virtual node as a service provider before you can delete the
     * virtual node itself.
     * </p>
     *
     * @param deleteVirtualNodeRequest
     * @return Result of the DeleteVirtualNode operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ResourceInUseException
     *         You can't delete the specified resource because it's in use or required by another resource.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.DeleteVirtualNode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DeleteVirtualNode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteVirtualNodeResponse deleteVirtualNode(DeleteVirtualNodeRequest deleteVirtualNodeRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException, ResourceInUseException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteVirtualNodeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteVirtualNodeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVirtualNodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVirtualNode");

            return clientHandler.execute(new ClientExecutionParams<DeleteVirtualNodeRequest, DeleteVirtualNodeResponse>()
                    .withOperationName("DeleteVirtualNode").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteVirtualNodeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteVirtualNodeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an existing virtual router.
     * </p>
     * <p>
     * You must delete any routes associated with the virtual router before you can delete the router itself.
     * </p>
     *
     * @param deleteVirtualRouterRequest
     * @return Result of the DeleteVirtualRouter operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ResourceInUseException
     *         You can't delete the specified resource because it's in use or required by another resource.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.DeleteVirtualRouter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DeleteVirtualRouter" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteVirtualRouterResponse deleteVirtualRouter(DeleteVirtualRouterRequest deleteVirtualRouterRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException, ResourceInUseException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteVirtualRouterResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteVirtualRouterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVirtualRouterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVirtualRouter");

            return clientHandler.execute(new ClientExecutionParams<DeleteVirtualRouterRequest, DeleteVirtualRouterResponse>()
                    .withOperationName("DeleteVirtualRouter").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteVirtualRouterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteVirtualRouterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an existing virtual service.
     * </p>
     *
     * @param deleteVirtualServiceRequest
     * @return Result of the DeleteVirtualService operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ResourceInUseException
     *         You can't delete the specified resource because it's in use or required by another resource.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.DeleteVirtualService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DeleteVirtualService" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteVirtualServiceResponse deleteVirtualService(DeleteVirtualServiceRequest deleteVirtualServiceRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException, ResourceInUseException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteVirtualServiceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteVirtualServiceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVirtualServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVirtualService");

            return clientHandler.execute(new ClientExecutionParams<DeleteVirtualServiceRequest, DeleteVirtualServiceResponse>()
                    .withOperationName("DeleteVirtualService").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteVirtualServiceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteVirtualServiceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes an existing gateway route.
     * </p>
     *
     * @param describeGatewayRouteRequest
     * @return Result of the DescribeGatewayRoute operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.DescribeGatewayRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DescribeGatewayRoute" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeGatewayRouteResponse describeGatewayRoute(DescribeGatewayRouteRequest describeGatewayRouteRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeGatewayRouteResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeGatewayRouteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGatewayRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGatewayRoute");

            return clientHandler.execute(new ClientExecutionParams<DescribeGatewayRouteRequest, DescribeGatewayRouteResponse>()
                    .withOperationName("DescribeGatewayRoute").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeGatewayRouteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeGatewayRouteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes an existing service mesh.
     * </p>
     *
     * @param describeMeshRequest
     * @return Result of the DescribeMesh operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.DescribeMesh
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DescribeMesh" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeMeshResponse describeMesh(DescribeMeshRequest describeMeshRequest) throws NotFoundException,
            BadRequestException, TooManyRequestsException, ForbiddenException, ServiceUnavailableException,
            InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeMeshResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeMeshResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeMeshRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMesh");

            return clientHandler.execute(new ClientExecutionParams<DescribeMeshRequest, DescribeMeshResponse>()
                    .withOperationName("DescribeMesh").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeMeshRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeMeshRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes an existing route.
     * </p>
     *
     * @param describeRouteRequest
     * @return Result of the DescribeRoute operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.DescribeRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DescribeRoute" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeRouteResponse describeRoute(DescribeRouteRequest describeRouteRequest) throws NotFoundException,
            BadRequestException, TooManyRequestsException, ForbiddenException, ServiceUnavailableException,
            InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeRouteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeRouteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRoute");

            return clientHandler.execute(new ClientExecutionParams<DescribeRouteRequest, DescribeRouteResponse>()
                    .withOperationName("DescribeRoute").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeRouteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeRouteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes an existing virtual gateway.
     * </p>
     *
     * @param describeVirtualGatewayRequest
     * @return Result of the DescribeVirtualGateway operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.DescribeVirtualGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DescribeVirtualGateway"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeVirtualGatewayResponse describeVirtualGateway(DescribeVirtualGatewayRequest describeVirtualGatewayRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeVirtualGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeVirtualGatewayResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVirtualGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVirtualGateway");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeVirtualGatewayRequest, DescribeVirtualGatewayResponse>()
                            .withOperationName("DescribeVirtualGateway").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeVirtualGatewayRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeVirtualGatewayRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes an existing virtual node.
     * </p>
     *
     * @param describeVirtualNodeRequest
     * @return Result of the DescribeVirtualNode operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.DescribeVirtualNode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DescribeVirtualNode" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeVirtualNodeResponse describeVirtualNode(DescribeVirtualNodeRequest describeVirtualNodeRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeVirtualNodeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeVirtualNodeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVirtualNodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVirtualNode");

            return clientHandler.execute(new ClientExecutionParams<DescribeVirtualNodeRequest, DescribeVirtualNodeResponse>()
                    .withOperationName("DescribeVirtualNode").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeVirtualNodeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeVirtualNodeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes an existing virtual router.
     * </p>
     *
     * @param describeVirtualRouterRequest
     * @return Result of the DescribeVirtualRouter operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.DescribeVirtualRouter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DescribeVirtualRouter" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeVirtualRouterResponse describeVirtualRouter(DescribeVirtualRouterRequest describeVirtualRouterRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeVirtualRouterResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeVirtualRouterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVirtualRouterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVirtualRouter");

            return clientHandler.execute(new ClientExecutionParams<DescribeVirtualRouterRequest, DescribeVirtualRouterResponse>()
                    .withOperationName("DescribeVirtualRouter").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeVirtualRouterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeVirtualRouterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes an existing virtual service.
     * </p>
     *
     * @param describeVirtualServiceRequest
     * @return Result of the DescribeVirtualService operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.DescribeVirtualService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DescribeVirtualService"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeVirtualServiceResponse describeVirtualService(DescribeVirtualServiceRequest describeVirtualServiceRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeVirtualServiceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeVirtualServiceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVirtualServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVirtualService");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeVirtualServiceRequest, DescribeVirtualServiceResponse>()
                            .withOperationName("DescribeVirtualService").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeVirtualServiceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeVirtualServiceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of existing gateway routes that are associated to a virtual gateway.
     * </p>
     *
     * @param listGatewayRoutesRequest
     * @return Result of the ListGatewayRoutes operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.ListGatewayRoutes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListGatewayRoutes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListGatewayRoutesResponse listGatewayRoutes(ListGatewayRoutesRequest listGatewayRoutesRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGatewayRoutesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListGatewayRoutesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGatewayRoutesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGatewayRoutes");

            return clientHandler.execute(new ClientExecutionParams<ListGatewayRoutesRequest, ListGatewayRoutesResponse>()
                    .withOperationName("ListGatewayRoutes").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listGatewayRoutesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListGatewayRoutesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of existing gateway routes that are associated to a virtual gateway.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listGatewayRoutes(software.amazon.awssdk.services.appmesh.model.ListGatewayRoutesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appmesh.paginators.ListGatewayRoutesIterable responses = client.listGatewayRoutesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.appmesh.paginators.ListGatewayRoutesIterable responses = client
     *             .listGatewayRoutesPaginator(request);
     *     for (software.amazon.awssdk.services.appmesh.model.ListGatewayRoutesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appmesh.paginators.ListGatewayRoutesIterable responses = client.listGatewayRoutesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGatewayRoutes(software.amazon.awssdk.services.appmesh.model.ListGatewayRoutesRequest)} operation.</b>
     * </p>
     *
     * @param listGatewayRoutesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.ListGatewayRoutes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListGatewayRoutes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListGatewayRoutesIterable listGatewayRoutesPaginator(ListGatewayRoutesRequest listGatewayRoutesRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        return new ListGatewayRoutesIterable(this, applyPaginatorUserAgent(listGatewayRoutesRequest));
    }

    /**
     * <p>
     * Returns a list of existing service meshes.
     * </p>
     *
     * @param listMeshesRequest
     * @return Result of the ListMeshes operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.ListMeshes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListMeshes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListMeshesResponse listMeshes(ListMeshesRequest listMeshesRequest) throws NotFoundException, BadRequestException,
            TooManyRequestsException, ForbiddenException, ServiceUnavailableException, InternalServerErrorException,
            AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMeshesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListMeshesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMeshesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMeshes");

            return clientHandler
                    .execute(new ClientExecutionParams<ListMeshesRequest, ListMeshesResponse>().withOperationName("ListMeshes")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listMeshesRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListMeshesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of existing service meshes.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listMeshes(software.amazon.awssdk.services.appmesh.model.ListMeshesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appmesh.paginators.ListMeshesIterable responses = client.listMeshesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.appmesh.paginators.ListMeshesIterable responses = client.listMeshesPaginator(request);
     *     for (software.amazon.awssdk.services.appmesh.model.ListMeshesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appmesh.paginators.ListMeshesIterable responses = client.listMeshesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMeshes(software.amazon.awssdk.services.appmesh.model.ListMeshesRequest)} operation.</b>
     * </p>
     *
     * @param listMeshesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.ListMeshes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListMeshes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListMeshesIterable listMeshesPaginator(ListMeshesRequest listMeshesRequest) throws NotFoundException,
            BadRequestException, TooManyRequestsException, ForbiddenException, ServiceUnavailableException,
            InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        return new ListMeshesIterable(this, applyPaginatorUserAgent(listMeshesRequest));
    }

    /**
     * <p>
     * Returns a list of existing routes in a service mesh.
     * </p>
     *
     * @param listRoutesRequest
     * @return Result of the ListRoutes operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.ListRoutes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListRoutes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListRoutesResponse listRoutes(ListRoutesRequest listRoutesRequest) throws NotFoundException, BadRequestException,
            TooManyRequestsException, ForbiddenException, ServiceUnavailableException, InternalServerErrorException,
            AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRoutesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListRoutesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRoutesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRoutes");

            return clientHandler
                    .execute(new ClientExecutionParams<ListRoutesRequest, ListRoutesResponse>().withOperationName("ListRoutes")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listRoutesRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListRoutesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of existing routes in a service mesh.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listRoutes(software.amazon.awssdk.services.appmesh.model.ListRoutesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appmesh.paginators.ListRoutesIterable responses = client.listRoutesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.appmesh.paginators.ListRoutesIterable responses = client.listRoutesPaginator(request);
     *     for (software.amazon.awssdk.services.appmesh.model.ListRoutesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appmesh.paginators.ListRoutesIterable responses = client.listRoutesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRoutes(software.amazon.awssdk.services.appmesh.model.ListRoutesRequest)} operation.</b>
     * </p>
     *
     * @param listRoutesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.ListRoutes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListRoutes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListRoutesIterable listRoutesPaginator(ListRoutesRequest listRoutesRequest) throws NotFoundException,
            BadRequestException, TooManyRequestsException, ForbiddenException, ServiceUnavailableException,
            InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        return new ListRoutesIterable(this, applyPaginatorUserAgent(listRoutesRequest));
    }

    /**
     * <p>
     * List the tags for an App Mesh resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List the tags for an App Mesh resource.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.appmesh.model.ListTagsForResourceRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appmesh.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.appmesh.paginators.ListTagsForResourceIterable responses = client
     *             .listTagsForResourcePaginator(request);
     *     for (software.amazon.awssdk.services.appmesh.model.ListTagsForResourceResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appmesh.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.appmesh.model.ListTagsForResourceRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceIterable listTagsForResourcePaginator(ListTagsForResourceRequest listTagsForResourceRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        return new ListTagsForResourceIterable(this, applyPaginatorUserAgent(listTagsForResourceRequest));
    }

    /**
     * <p>
     * Returns a list of existing virtual gateways in a service mesh.
     * </p>
     *
     * @param listVirtualGatewaysRequest
     * @return Result of the ListVirtualGateways operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.ListVirtualGateways
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListVirtualGateways" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListVirtualGatewaysResponse listVirtualGateways(ListVirtualGatewaysRequest listVirtualGatewaysRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListVirtualGatewaysResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListVirtualGatewaysResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVirtualGatewaysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVirtualGateways");

            return clientHandler.execute(new ClientExecutionParams<ListVirtualGatewaysRequest, ListVirtualGatewaysResponse>()
                    .withOperationName("ListVirtualGateways").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listVirtualGatewaysRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListVirtualGatewaysRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of existing virtual gateways in a service mesh.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listVirtualGateways(software.amazon.awssdk.services.appmesh.model.ListVirtualGatewaysRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appmesh.paginators.ListVirtualGatewaysIterable responses = client.listVirtualGatewaysPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.appmesh.paginators.ListVirtualGatewaysIterable responses = client
     *             .listVirtualGatewaysPaginator(request);
     *     for (software.amazon.awssdk.services.appmesh.model.ListVirtualGatewaysResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appmesh.paginators.ListVirtualGatewaysIterable responses = client.listVirtualGatewaysPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listVirtualGateways(software.amazon.awssdk.services.appmesh.model.ListVirtualGatewaysRequest)}
     * operation.</b>
     * </p>
     *
     * @param listVirtualGatewaysRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.ListVirtualGateways
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListVirtualGateways" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListVirtualGatewaysIterable listVirtualGatewaysPaginator(ListVirtualGatewaysRequest listVirtualGatewaysRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        return new ListVirtualGatewaysIterable(this, applyPaginatorUserAgent(listVirtualGatewaysRequest));
    }

    /**
     * <p>
     * Returns a list of existing virtual nodes.
     * </p>
     *
     * @param listVirtualNodesRequest
     * @return Result of the ListVirtualNodes operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.ListVirtualNodes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListVirtualNodes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListVirtualNodesResponse listVirtualNodes(ListVirtualNodesRequest listVirtualNodesRequest) throws NotFoundException,
            BadRequestException, TooManyRequestsException, ForbiddenException, ServiceUnavailableException,
            InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListVirtualNodesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListVirtualNodesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVirtualNodesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVirtualNodes");

            return clientHandler.execute(new ClientExecutionParams<ListVirtualNodesRequest, ListVirtualNodesResponse>()
                    .withOperationName("ListVirtualNodes").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listVirtualNodesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListVirtualNodesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of existing virtual nodes.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listVirtualNodes(software.amazon.awssdk.services.appmesh.model.ListVirtualNodesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appmesh.paginators.ListVirtualNodesIterable responses = client.listVirtualNodesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.appmesh.paginators.ListVirtualNodesIterable responses = client
     *             .listVirtualNodesPaginator(request);
     *     for (software.amazon.awssdk.services.appmesh.model.ListVirtualNodesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appmesh.paginators.ListVirtualNodesIterable responses = client.listVirtualNodesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listVirtualNodes(software.amazon.awssdk.services.appmesh.model.ListVirtualNodesRequest)} operation.</b>
     * </p>
     *
     * @param listVirtualNodesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.ListVirtualNodes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListVirtualNodes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListVirtualNodesIterable listVirtualNodesPaginator(ListVirtualNodesRequest listVirtualNodesRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        return new ListVirtualNodesIterable(this, applyPaginatorUserAgent(listVirtualNodesRequest));
    }

    /**
     * <p>
     * Returns a list of existing virtual routers in a service mesh.
     * </p>
     *
     * @param listVirtualRoutersRequest
     * @return Result of the ListVirtualRouters operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.ListVirtualRouters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListVirtualRouters" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListVirtualRoutersResponse listVirtualRouters(ListVirtualRoutersRequest listVirtualRoutersRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListVirtualRoutersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListVirtualRoutersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVirtualRoutersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVirtualRouters");

            return clientHandler.execute(new ClientExecutionParams<ListVirtualRoutersRequest, ListVirtualRoutersResponse>()
                    .withOperationName("ListVirtualRouters").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listVirtualRoutersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListVirtualRoutersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of existing virtual routers in a service mesh.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listVirtualRouters(software.amazon.awssdk.services.appmesh.model.ListVirtualRoutersRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appmesh.paginators.ListVirtualRoutersIterable responses = client.listVirtualRoutersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.appmesh.paginators.ListVirtualRoutersIterable responses = client
     *             .listVirtualRoutersPaginator(request);
     *     for (software.amazon.awssdk.services.appmesh.model.ListVirtualRoutersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appmesh.paginators.ListVirtualRoutersIterable responses = client.listVirtualRoutersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listVirtualRouters(software.amazon.awssdk.services.appmesh.model.ListVirtualRoutersRequest)}
     * operation.</b>
     * </p>
     *
     * @param listVirtualRoutersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.ListVirtualRouters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListVirtualRouters" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListVirtualRoutersIterable listVirtualRoutersPaginator(ListVirtualRoutersRequest listVirtualRoutersRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        return new ListVirtualRoutersIterable(this, applyPaginatorUserAgent(listVirtualRoutersRequest));
    }

    /**
     * <p>
     * Returns a list of existing virtual services in a service mesh.
     * </p>
     *
     * @param listVirtualServicesRequest
     * @return Result of the ListVirtualServices operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.ListVirtualServices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListVirtualServices" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListVirtualServicesResponse listVirtualServices(ListVirtualServicesRequest listVirtualServicesRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListVirtualServicesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListVirtualServicesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVirtualServicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVirtualServices");

            return clientHandler.execute(new ClientExecutionParams<ListVirtualServicesRequest, ListVirtualServicesResponse>()
                    .withOperationName("ListVirtualServices").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listVirtualServicesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListVirtualServicesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of existing virtual services in a service mesh.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listVirtualServices(software.amazon.awssdk.services.appmesh.model.ListVirtualServicesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appmesh.paginators.ListVirtualServicesIterable responses = client.listVirtualServicesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.appmesh.paginators.ListVirtualServicesIterable responses = client
     *             .listVirtualServicesPaginator(request);
     *     for (software.amazon.awssdk.services.appmesh.model.ListVirtualServicesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appmesh.paginators.ListVirtualServicesIterable responses = client.listVirtualServicesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listVirtualServices(software.amazon.awssdk.services.appmesh.model.ListVirtualServicesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listVirtualServicesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.ListVirtualServices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListVirtualServices" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListVirtualServicesIterable listVirtualServicesPaginator(ListVirtualServicesRequest listVirtualServicesRequest)
            throws NotFoundException, BadRequestException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        return new ListVirtualServicesIterable(this, applyPaginatorUserAgent(listVirtualServicesRequest));
    }

    /**
     * <p>
     * Associates the specified tags to a resource with the specified <code>resourceArn</code>. If existing tags on a
     * resource aren't specified in the request parameters, they aren't changed. When a resource is deleted, the tags
     * associated with that resource are also deleted.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyTagsException
     *         The request exceeds the maximum allowed number of tags allowed per resource. The current limit is 50 user
     *         tags per resource. You must reduce the number of tags in the request. None of the tags in this request
     *         were applied.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws NotFoundException, BadRequestException,
            TooManyTagsException, TooManyRequestsException, ForbiddenException, ServiceUnavailableException,
            InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes specified tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws NotFoundException,
            BadRequestException, TooManyRequestsException, ForbiddenException, ServiceUnavailableException,
            InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing gateway route that is associated to a specified virtual gateway in a service mesh.
     * </p>
     *
     * @param updateGatewayRouteRequest
     * @return Result of the UpdateGatewayRoute operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws ConflictException
     *         The request contains a client token that was used for a previous update resource call with different
     *         specifications. Try the request again with a new client token.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws LimitExceededException
     *         You have exceeded a service limit for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>AWS App Mesh User Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.UpdateGatewayRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/UpdateGatewayRoute" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateGatewayRouteResponse updateGatewayRoute(UpdateGatewayRouteRequest updateGatewayRouteRequest)
            throws NotFoundException, BadRequestException, ConflictException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, LimitExceededException, AwsServiceException,
            SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateGatewayRouteResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateGatewayRouteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGatewayRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGatewayRoute");

            return clientHandler.execute(new ClientExecutionParams<UpdateGatewayRouteRequest, UpdateGatewayRouteResponse>()
                    .withOperationName("UpdateGatewayRoute").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateGatewayRouteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateGatewayRouteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing service mesh.
     * </p>
     *
     * @param updateMeshRequest
     * @return Result of the UpdateMesh operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws ConflictException
     *         The request contains a client token that was used for a previous update resource call with different
     *         specifications. Try the request again with a new client token.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.UpdateMesh
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/UpdateMesh" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateMeshResponse updateMesh(UpdateMeshRequest updateMeshRequest) throws NotFoundException, BadRequestException,
            ConflictException, TooManyRequestsException, ForbiddenException, ServiceUnavailableException,
            InternalServerErrorException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateMeshResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateMeshResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMeshRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMesh");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateMeshRequest, UpdateMeshResponse>().withOperationName("UpdateMesh")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateMeshRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateMeshRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing route for a specified service mesh and virtual router.
     * </p>
     *
     * @param updateRouteRequest
     * @return Result of the UpdateRoute operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws ConflictException
     *         The request contains a client token that was used for a previous update resource call with different
     *         specifications. Try the request again with a new client token.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws LimitExceededException
     *         You have exceeded a service limit for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>AWS App Mesh User Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.UpdateRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/UpdateRoute" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateRouteResponse updateRoute(UpdateRouteRequest updateRouteRequest) throws NotFoundException, BadRequestException,
            ConflictException, TooManyRequestsException, ForbiddenException, ServiceUnavailableException,
            InternalServerErrorException, LimitExceededException, AwsServiceException, SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateRouteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateRouteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRoute");

            return clientHandler.execute(new ClientExecutionParams<UpdateRouteRequest, UpdateRouteResponse>()
                    .withOperationName("UpdateRoute").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateRouteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateRouteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing virtual gateway in a specified service mesh.
     * </p>
     *
     * @param updateVirtualGatewayRequest
     * @return Result of the UpdateVirtualGateway operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws ConflictException
     *         The request contains a client token that was used for a previous update resource call with different
     *         specifications. Try the request again with a new client token.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws LimitExceededException
     *         You have exceeded a service limit for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>AWS App Mesh User Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.UpdateVirtualGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/UpdateVirtualGateway" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateVirtualGatewayResponse updateVirtualGateway(UpdateVirtualGatewayRequest updateVirtualGatewayRequest)
            throws NotFoundException, BadRequestException, ConflictException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, LimitExceededException, AwsServiceException,
            SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateVirtualGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateVirtualGatewayResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVirtualGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVirtualGateway");

            return clientHandler.execute(new ClientExecutionParams<UpdateVirtualGatewayRequest, UpdateVirtualGatewayResponse>()
                    .withOperationName("UpdateVirtualGateway").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateVirtualGatewayRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateVirtualGatewayRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing virtual node in a specified service mesh.
     * </p>
     *
     * @param updateVirtualNodeRequest
     * @return Result of the UpdateVirtualNode operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws ConflictException
     *         The request contains a client token that was used for a previous update resource call with different
     *         specifications. Try the request again with a new client token.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws LimitExceededException
     *         You have exceeded a service limit for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>AWS App Mesh User Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.UpdateVirtualNode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/UpdateVirtualNode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateVirtualNodeResponse updateVirtualNode(UpdateVirtualNodeRequest updateVirtualNodeRequest)
            throws NotFoundException, BadRequestException, ConflictException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, LimitExceededException, AwsServiceException,
            SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateVirtualNodeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateVirtualNodeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVirtualNodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVirtualNode");

            return clientHandler.execute(new ClientExecutionParams<UpdateVirtualNodeRequest, UpdateVirtualNodeResponse>()
                    .withOperationName("UpdateVirtualNode").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateVirtualNodeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateVirtualNodeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing virtual router in a specified service mesh.
     * </p>
     *
     * @param updateVirtualRouterRequest
     * @return Result of the UpdateVirtualRouter operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws ConflictException
     *         The request contains a client token that was used for a previous update resource call with different
     *         specifications. Try the request again with a new client token.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws LimitExceededException
     *         You have exceeded a service limit for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>AWS App Mesh User Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.UpdateVirtualRouter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/UpdateVirtualRouter" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateVirtualRouterResponse updateVirtualRouter(UpdateVirtualRouterRequest updateVirtualRouterRequest)
            throws NotFoundException, BadRequestException, ConflictException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, LimitExceededException, AwsServiceException,
            SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateVirtualRouterResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateVirtualRouterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVirtualRouterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVirtualRouter");

            return clientHandler.execute(new ClientExecutionParams<UpdateVirtualRouterRequest, UpdateVirtualRouterResponse>()
                    .withOperationName("UpdateVirtualRouter").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateVirtualRouterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateVirtualRouterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing virtual service in a specified service mesh.
     * </p>
     *
     * @param updateVirtualServiceRequest
     * @return Result of the UpdateVirtualService operation returned by the service.
     * @throws NotFoundException
     *         The specified resource doesn't exist. Check your request syntax and try again.
     * @throws BadRequestException
     *         The request syntax was malformed. Check your request syntax and try again.
     * @throws ConflictException
     *         The request contains a client token that was used for a previous update resource call with different
     *         specifications. Try the request again with a new client token.
     * @throws TooManyRequestsException
     *         The maximum request rate permitted by the App Mesh APIs has been exceeded for your account. For best
     *         results, use an increasing or variable sleep interval between requests.
     * @throws ForbiddenException
     *         You don't have permissions to perform this action.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the service.
     * @throws InternalServerErrorException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws LimitExceededException
     *         You have exceeded a service limit for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>AWS App Mesh User Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppMeshException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppMeshClient.UpdateVirtualService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/UpdateVirtualService" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateVirtualServiceResponse updateVirtualService(UpdateVirtualServiceRequest updateVirtualServiceRequest)
            throws NotFoundException, BadRequestException, ConflictException, TooManyRequestsException, ForbiddenException,
            ServiceUnavailableException, InternalServerErrorException, LimitExceededException, AwsServiceException,
            SdkClientException, AppMeshException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateVirtualServiceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateVirtualServiceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVirtualServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVirtualService");

            return clientHandler.execute(new ClientExecutionParams<UpdateVirtualServiceRequest, UpdateVirtualServiceResponse>()
                    .withOperationName("UpdateVirtualService").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateVirtualServiceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateVirtualServiceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AppMeshException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends AppMeshRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
