/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appmesh;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.appmesh.internal.AppMeshServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.appmesh.model.AppMeshException;
import software.amazon.awssdk.services.appmesh.model.BadRequestException;
import software.amazon.awssdk.services.appmesh.model.ConflictException;
import software.amazon.awssdk.services.appmesh.model.CreateGatewayRouteRequest;
import software.amazon.awssdk.services.appmesh.model.CreateGatewayRouteResponse;
import software.amazon.awssdk.services.appmesh.model.CreateMeshRequest;
import software.amazon.awssdk.services.appmesh.model.CreateMeshResponse;
import software.amazon.awssdk.services.appmesh.model.CreateRouteRequest;
import software.amazon.awssdk.services.appmesh.model.CreateRouteResponse;
import software.amazon.awssdk.services.appmesh.model.CreateVirtualGatewayRequest;
import software.amazon.awssdk.services.appmesh.model.CreateVirtualGatewayResponse;
import software.amazon.awssdk.services.appmesh.model.CreateVirtualNodeRequest;
import software.amazon.awssdk.services.appmesh.model.CreateVirtualNodeResponse;
import software.amazon.awssdk.services.appmesh.model.CreateVirtualRouterRequest;
import software.amazon.awssdk.services.appmesh.model.CreateVirtualRouterResponse;
import software.amazon.awssdk.services.appmesh.model.CreateVirtualServiceRequest;
import software.amazon.awssdk.services.appmesh.model.CreateVirtualServiceResponse;
import software.amazon.awssdk.services.appmesh.model.DeleteGatewayRouteRequest;
import software.amazon.awssdk.services.appmesh.model.DeleteGatewayRouteResponse;
import software.amazon.awssdk.services.appmesh.model.DeleteMeshRequest;
import software.amazon.awssdk.services.appmesh.model.DeleteMeshResponse;
import software.amazon.awssdk.services.appmesh.model.DeleteRouteRequest;
import software.amazon.awssdk.services.appmesh.model.DeleteRouteResponse;
import software.amazon.awssdk.services.appmesh.model.DeleteVirtualGatewayRequest;
import software.amazon.awssdk.services.appmesh.model.DeleteVirtualGatewayResponse;
import software.amazon.awssdk.services.appmesh.model.DeleteVirtualNodeRequest;
import software.amazon.awssdk.services.appmesh.model.DeleteVirtualNodeResponse;
import software.amazon.awssdk.services.appmesh.model.DeleteVirtualRouterRequest;
import software.amazon.awssdk.services.appmesh.model.DeleteVirtualRouterResponse;
import software.amazon.awssdk.services.appmesh.model.DeleteVirtualServiceRequest;
import software.amazon.awssdk.services.appmesh.model.DeleteVirtualServiceResponse;
import software.amazon.awssdk.services.appmesh.model.DescribeGatewayRouteRequest;
import software.amazon.awssdk.services.appmesh.model.DescribeGatewayRouteResponse;
import software.amazon.awssdk.services.appmesh.model.DescribeMeshRequest;
import software.amazon.awssdk.services.appmesh.model.DescribeMeshResponse;
import software.amazon.awssdk.services.appmesh.model.DescribeRouteRequest;
import software.amazon.awssdk.services.appmesh.model.DescribeRouteResponse;
import software.amazon.awssdk.services.appmesh.model.DescribeVirtualGatewayRequest;
import software.amazon.awssdk.services.appmesh.model.DescribeVirtualGatewayResponse;
import software.amazon.awssdk.services.appmesh.model.DescribeVirtualNodeRequest;
import software.amazon.awssdk.services.appmesh.model.DescribeVirtualNodeResponse;
import software.amazon.awssdk.services.appmesh.model.DescribeVirtualRouterRequest;
import software.amazon.awssdk.services.appmesh.model.DescribeVirtualRouterResponse;
import software.amazon.awssdk.services.appmesh.model.DescribeVirtualServiceRequest;
import software.amazon.awssdk.services.appmesh.model.DescribeVirtualServiceResponse;
import software.amazon.awssdk.services.appmesh.model.ForbiddenException;
import software.amazon.awssdk.services.appmesh.model.InternalServerErrorException;
import software.amazon.awssdk.services.appmesh.model.LimitExceededException;
import software.amazon.awssdk.services.appmesh.model.ListGatewayRoutesRequest;
import software.amazon.awssdk.services.appmesh.model.ListGatewayRoutesResponse;
import software.amazon.awssdk.services.appmesh.model.ListMeshesRequest;
import software.amazon.awssdk.services.appmesh.model.ListMeshesResponse;
import software.amazon.awssdk.services.appmesh.model.ListRoutesRequest;
import software.amazon.awssdk.services.appmesh.model.ListRoutesResponse;
import software.amazon.awssdk.services.appmesh.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.appmesh.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.appmesh.model.ListVirtualGatewaysRequest;
import software.amazon.awssdk.services.appmesh.model.ListVirtualGatewaysResponse;
import software.amazon.awssdk.services.appmesh.model.ListVirtualNodesRequest;
import software.amazon.awssdk.services.appmesh.model.ListVirtualNodesResponse;
import software.amazon.awssdk.services.appmesh.model.ListVirtualRoutersRequest;
import software.amazon.awssdk.services.appmesh.model.ListVirtualRoutersResponse;
import software.amazon.awssdk.services.appmesh.model.ListVirtualServicesRequest;
import software.amazon.awssdk.services.appmesh.model.ListVirtualServicesResponse;
import software.amazon.awssdk.services.appmesh.model.NotFoundException;
import software.amazon.awssdk.services.appmesh.model.ResourceInUseException;
import software.amazon.awssdk.services.appmesh.model.ServiceUnavailableException;
import software.amazon.awssdk.services.appmesh.model.TagResourceRequest;
import software.amazon.awssdk.services.appmesh.model.TagResourceResponse;
import software.amazon.awssdk.services.appmesh.model.TooManyRequestsException;
import software.amazon.awssdk.services.appmesh.model.TooManyTagsException;
import software.amazon.awssdk.services.appmesh.model.UntagResourceRequest;
import software.amazon.awssdk.services.appmesh.model.UntagResourceResponse;
import software.amazon.awssdk.services.appmesh.model.UpdateGatewayRouteRequest;
import software.amazon.awssdk.services.appmesh.model.UpdateGatewayRouteResponse;
import software.amazon.awssdk.services.appmesh.model.UpdateMeshRequest;
import software.amazon.awssdk.services.appmesh.model.UpdateMeshResponse;
import software.amazon.awssdk.services.appmesh.model.UpdateRouteRequest;
import software.amazon.awssdk.services.appmesh.model.UpdateRouteResponse;
import software.amazon.awssdk.services.appmesh.model.UpdateVirtualGatewayRequest;
import software.amazon.awssdk.services.appmesh.model.UpdateVirtualGatewayResponse;
import software.amazon.awssdk.services.appmesh.model.UpdateVirtualNodeRequest;
import software.amazon.awssdk.services.appmesh.model.UpdateVirtualNodeResponse;
import software.amazon.awssdk.services.appmesh.model.UpdateVirtualRouterRequest;
import software.amazon.awssdk.services.appmesh.model.UpdateVirtualRouterResponse;
import software.amazon.awssdk.services.appmesh.model.UpdateVirtualServiceRequest;
import software.amazon.awssdk.services.appmesh.model.UpdateVirtualServiceResponse;
import software.amazon.awssdk.services.appmesh.transform.CreateGatewayRouteRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.CreateMeshRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.CreateRouteRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.CreateVirtualGatewayRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.CreateVirtualNodeRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.CreateVirtualRouterRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.CreateVirtualServiceRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DeleteGatewayRouteRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DeleteMeshRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DeleteRouteRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DeleteVirtualGatewayRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DeleteVirtualNodeRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DeleteVirtualRouterRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DeleteVirtualServiceRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DescribeGatewayRouteRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DescribeMeshRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DescribeRouteRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DescribeVirtualGatewayRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DescribeVirtualNodeRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DescribeVirtualRouterRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.DescribeVirtualServiceRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.ListGatewayRoutesRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.ListMeshesRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.ListRoutesRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.ListVirtualGatewaysRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.ListVirtualNodesRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.ListVirtualRoutersRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.ListVirtualServicesRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.UpdateGatewayRouteRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.UpdateMeshRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.UpdateRouteRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.UpdateVirtualGatewayRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.UpdateVirtualNodeRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.UpdateVirtualRouterRequestMarshaller;
import software.amazon.awssdk.services.appmesh.transform.UpdateVirtualServiceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AppMeshAsyncClient}.
 *
 * @see AppMeshAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAppMeshAsyncClient implements AppMeshAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAppMeshAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAppMeshAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a gateway route.
     * </p>
     * <p>
     * A gateway route is attached to a virtual gateway and routes traffic to an existing virtual service. If a route
     * matches a request, it can distribute traffic to a target virtual service.
     * </p>
     * <p>
     * For more information about gateway routes, see <a
     * href="https://docs.aws.amazon.com/app-mesh/latest/userguide/gateway-routes.html">Gateway routes</a>.
     * </p>
     *
     * @param createGatewayRouteRequest
     * @return A Java Future containing the result of the CreateGatewayRoute operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>ConflictException The request contains a client token that was used for a previous update resource
     *         call with different specifications. Try the request again with a new client token.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>LimitExceededException You have exceeded a service limit for your account. For more information, see
     *         <a href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>App Mesh User Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.CreateGatewayRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/CreateGatewayRoute" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGatewayRouteResponse> createGatewayRoute(CreateGatewayRouteRequest createGatewayRouteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createGatewayRouteRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGatewayRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGatewayRoute");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGatewayRouteResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateGatewayRouteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGatewayRouteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGatewayRouteRequest, CreateGatewayRouteResponse>()
                            .withOperationName("CreateGatewayRoute").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateGatewayRouteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createGatewayRouteRequest));
            CompletableFuture<CreateGatewayRouteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a service mesh.
     * </p>
     * <p>
     * A service mesh is a logical boundary for network traffic between services that are represented by resources
     * within the mesh. After you create your service mesh, you can create virtual services, virtual nodes, virtual
     * routers, and routes to distribute traffic between the applications in your mesh.
     * </p>
     * <p>
     * For more information about service meshes, see <a
     * href="https://docs.aws.amazon.com/app-mesh/latest/userguide/meshes.html">Service meshes</a>.
     * </p>
     *
     * @param createMeshRequest
     * @return A Java Future containing the result of the CreateMesh operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>ConflictException The request contains a client token that was used for a previous update resource
     *         call with different specifications. Try the request again with a new client token.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>LimitExceededException You have exceeded a service limit for your account. For more information, see
     *         <a href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>App Mesh User Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.CreateMesh
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/CreateMesh" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMeshResponse> createMesh(CreateMeshRequest createMeshRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createMeshRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMeshRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMesh");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMeshResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateMeshResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMeshResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMeshRequest, CreateMeshResponse>().withOperationName("CreateMesh")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateMeshRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createMeshRequest));
            CompletableFuture<CreateMeshResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a route that is associated with a virtual router.
     * </p>
     * <p>
     * You can route several different protocols and define a retry policy for a route. Traffic can be routed to one or
     * more virtual nodes.
     * </p>
     * <p>
     * For more information about routes, see <a
     * href="https://docs.aws.amazon.com/app-mesh/latest/userguide/routes.html">Routes</a>.
     * </p>
     *
     * @param createRouteRequest
     * @return A Java Future containing the result of the CreateRoute operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>ConflictException The request contains a client token that was used for a previous update resource
     *         call with different specifications. Try the request again with a new client token.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>LimitExceededException You have exceeded a service limit for your account. For more information, see
     *         <a href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>App Mesh User Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.CreateRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/CreateRoute" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRouteResponse> createRoute(CreateRouteRequest createRouteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRouteRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRoute");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRouteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateRouteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRouteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRouteRequest, CreateRouteResponse>()
                            .withOperationName("CreateRoute").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateRouteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createRouteRequest));
            CompletableFuture<CreateRouteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a virtual gateway.
     * </p>
     * <p>
     * A virtual gateway allows resources outside your mesh to communicate to resources that are inside your mesh. The
     * virtual gateway represents an Envoy proxy running in an Amazon ECS task, in a Kubernetes service, or on an Amazon
     * EC2 instance. Unlike a virtual node, which represents an Envoy running with an application, a virtual gateway
     * represents Envoy deployed by itself.
     * </p>
     * <p>
     * For more information about virtual gateways, see <a
     * href="https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html">Virtual gateways</a>.
     * </p>
     *
     * @param createVirtualGatewayRequest
     * @return A Java Future containing the result of the CreateVirtualGateway operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>ConflictException The request contains a client token that was used for a previous update resource
     *         call with different specifications. Try the request again with a new client token.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>LimitExceededException You have exceeded a service limit for your account. For more information, see
     *         <a href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>App Mesh User Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.CreateVirtualGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/CreateVirtualGateway" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVirtualGatewayResponse> createVirtualGateway(
            CreateVirtualGatewayRequest createVirtualGatewayRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createVirtualGatewayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVirtualGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVirtualGateway");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVirtualGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVirtualGatewayResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVirtualGatewayResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVirtualGatewayRequest, CreateVirtualGatewayResponse>()
                            .withOperationName("CreateVirtualGateway").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateVirtualGatewayRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createVirtualGatewayRequest));
            CompletableFuture<CreateVirtualGatewayResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a virtual node within a service mesh.
     * </p>
     * <p>
     * A virtual node acts as a logical pointer to a particular task group, such as an Amazon ECS service or a
     * Kubernetes deployment. When you create a virtual node, you can specify the service discovery information for your
     * task group, and whether the proxy running in a task group will communicate with other proxies using Transport
     * Layer Security (TLS).
     * </p>
     * <p>
     * You define a <code>listener</code> for any inbound traffic that your virtual node expects. Any virtual service
     * that your virtual node expects to communicate to is specified as a <code>backend</code>.
     * </p>
     * <p>
     * The response metadata for your new virtual node contains the <code>arn</code> that is associated with the virtual
     * node. Set this value to the full ARN; for example,
     * <code>arn:aws:appmesh:us-west-2:123456789012:myMesh/default/virtualNode/myApp</code>) as the
     * <code>APPMESH_RESOURCE_ARN</code> environment variable for your task group's Envoy proxy container in your task
     * definition or pod spec. This is then mapped to the <code>node.id</code> and <code>node.cluster</code> Envoy
     * parameters.
     * </p>
     * <note>
     * <p>
     * By default, App Mesh uses the name of the resource you specified in <code>APPMESH_RESOURCE_ARN</code> when Envoy
     * is referring to itself in metrics and traces. You can override this behavior by setting the
     * <code>APPMESH_RESOURCE_CLUSTER</code> environment variable with your own name.
     * </p>
     * </note>
     * <p>
     * For more information about virtual nodes, see <a
     * href="https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html">Virtual nodes</a>. You must be
     * using <code>1.15.0</code> or later of the Envoy image when setting these variables. For more information aboutApp
     * Mesh Envoy variables, see <a href="https://docs.aws.amazon.com/app-mesh/latest/userguide/envoy.html">Envoy
     * image</a> in the App Mesh User Guide.
     * </p>
     *
     * @param createVirtualNodeRequest
     * @return A Java Future containing the result of the CreateVirtualNode operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>ConflictException The request contains a client token that was used for a previous update resource
     *         call with different specifications. Try the request again with a new client token.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>LimitExceededException You have exceeded a service limit for your account. For more information, see
     *         <a href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>App Mesh User Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.CreateVirtualNode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/CreateVirtualNode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVirtualNodeResponse> createVirtualNode(CreateVirtualNodeRequest createVirtualNodeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createVirtualNodeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVirtualNodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVirtualNode");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVirtualNodeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVirtualNodeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVirtualNodeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVirtualNodeRequest, CreateVirtualNodeResponse>()
                            .withOperationName("CreateVirtualNode").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateVirtualNodeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createVirtualNodeRequest));
            CompletableFuture<CreateVirtualNodeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a virtual router within a service mesh.
     * </p>
     * <p>
     * Specify a <code>listener</code> for any inbound traffic that your virtual router receives. Create a virtual
     * router for each protocol and port that you need to route. Virtual routers handle traffic for one or more virtual
     * services within your mesh. After you create your virtual router, create and associate routes for your virtual
     * router that direct incoming requests to different virtual nodes.
     * </p>
     * <p>
     * For more information about virtual routers, see <a
     * href="https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_routers.html">Virtual routers</a>.
     * </p>
     *
     * @param createVirtualRouterRequest
     * @return A Java Future containing the result of the CreateVirtualRouter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>ConflictException The request contains a client token that was used for a previous update resource
     *         call with different specifications. Try the request again with a new client token.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>LimitExceededException You have exceeded a service limit for your account. For more information, see
     *         <a href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>App Mesh User Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.CreateVirtualRouter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/CreateVirtualRouter" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVirtualRouterResponse> createVirtualRouter(
            CreateVirtualRouterRequest createVirtualRouterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createVirtualRouterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVirtualRouterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVirtualRouter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVirtualRouterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVirtualRouterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVirtualRouterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVirtualRouterRequest, CreateVirtualRouterResponse>()
                            .withOperationName("CreateVirtualRouter").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateVirtualRouterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createVirtualRouterRequest));
            CompletableFuture<CreateVirtualRouterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a virtual service within a service mesh.
     * </p>
     * <p>
     * A virtual service is an abstraction of a real service that is provided by a virtual node directly or indirectly
     * by means of a virtual router. Dependent services call your virtual service by its <code>virtualServiceName</code>
     * , and those requests are routed to the virtual node or virtual router that is specified as the provider for the
     * virtual service.
     * </p>
     * <p>
     * For more information about virtual services, see <a
     * href="https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html">Virtual services</a>.
     * </p>
     *
     * @param createVirtualServiceRequest
     * @return A Java Future containing the result of the CreateVirtualService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>ConflictException The request contains a client token that was used for a previous update resource
     *         call with different specifications. Try the request again with a new client token.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>LimitExceededException You have exceeded a service limit for your account. For more information, see
     *         <a href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>App Mesh User Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.CreateVirtualService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/CreateVirtualService" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVirtualServiceResponse> createVirtualService(
            CreateVirtualServiceRequest createVirtualServiceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createVirtualServiceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVirtualServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVirtualService");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVirtualServiceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVirtualServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVirtualServiceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVirtualServiceRequest, CreateVirtualServiceResponse>()
                            .withOperationName("CreateVirtualService").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateVirtualServiceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createVirtualServiceRequest));
            CompletableFuture<CreateVirtualServiceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing gateway route.
     * </p>
     *
     * @param deleteGatewayRouteRequest
     * @return A Java Future containing the result of the DeleteGatewayRoute operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ResourceInUseException You can't delete the specified resource because it's in use or required by
     *         another resource.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.DeleteGatewayRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DeleteGatewayRoute" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGatewayRouteResponse> deleteGatewayRoute(DeleteGatewayRouteRequest deleteGatewayRouteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteGatewayRouteRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGatewayRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGatewayRoute");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGatewayRouteResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteGatewayRouteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteGatewayRouteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteGatewayRouteRequest, DeleteGatewayRouteResponse>()
                            .withOperationName("DeleteGatewayRoute").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteGatewayRouteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteGatewayRouteRequest));
            CompletableFuture<DeleteGatewayRouteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing service mesh.
     * </p>
     * <p>
     * You must delete all resources (virtual services, routes, virtual routers, and virtual nodes) in the service mesh
     * before you can delete the mesh itself.
     * </p>
     *
     * @param deleteMeshRequest
     * @return A Java Future containing the result of the DeleteMesh operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ResourceInUseException You can't delete the specified resource because it's in use or required by
     *         another resource.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.DeleteMesh
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DeleteMesh" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMeshResponse> deleteMesh(DeleteMeshRequest deleteMeshRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMeshRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMeshRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMesh");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMeshResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteMeshResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMeshResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMeshRequest, DeleteMeshResponse>().withOperationName("DeleteMesh")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteMeshRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteMeshRequest));
            CompletableFuture<DeleteMeshResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing route.
     * </p>
     *
     * @param deleteRouteRequest
     * @return A Java Future containing the result of the DeleteRoute operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ResourceInUseException You can't delete the specified resource because it's in use or required by
     *         another resource.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.DeleteRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DeleteRoute" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRouteResponse> deleteRoute(DeleteRouteRequest deleteRouteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRouteRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRoute");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRouteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteRouteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRouteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRouteRequest, DeleteRouteResponse>()
                            .withOperationName("DeleteRoute").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteRouteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRouteRequest));
            CompletableFuture<DeleteRouteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing virtual gateway. You cannot delete a virtual gateway if any gateway routes are associated to
     * it.
     * </p>
     *
     * @param deleteVirtualGatewayRequest
     * @return A Java Future containing the result of the DeleteVirtualGateway operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ResourceInUseException You can't delete the specified resource because it's in use or required by
     *         another resource.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.DeleteVirtualGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DeleteVirtualGateway" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVirtualGatewayResponse> deleteVirtualGateway(
            DeleteVirtualGatewayRequest deleteVirtualGatewayRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVirtualGatewayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVirtualGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVirtualGateway");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVirtualGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVirtualGatewayResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVirtualGatewayResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVirtualGatewayRequest, DeleteVirtualGatewayResponse>()
                            .withOperationName("DeleteVirtualGateway").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteVirtualGatewayRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteVirtualGatewayRequest));
            CompletableFuture<DeleteVirtualGatewayResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing virtual node.
     * </p>
     * <p>
     * You must delete any virtual services that list a virtual node as a service provider before you can delete the
     * virtual node itself.
     * </p>
     *
     * @param deleteVirtualNodeRequest
     *        Deletes a virtual node input.
     * @return A Java Future containing the result of the DeleteVirtualNode operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ResourceInUseException You can't delete the specified resource because it's in use or required by
     *         another resource.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.DeleteVirtualNode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DeleteVirtualNode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVirtualNodeResponse> deleteVirtualNode(DeleteVirtualNodeRequest deleteVirtualNodeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVirtualNodeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVirtualNodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVirtualNode");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVirtualNodeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVirtualNodeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVirtualNodeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVirtualNodeRequest, DeleteVirtualNodeResponse>()
                            .withOperationName("DeleteVirtualNode").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteVirtualNodeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteVirtualNodeRequest));
            CompletableFuture<DeleteVirtualNodeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing virtual router.
     * </p>
     * <p>
     * You must delete any routes associated with the virtual router before you can delete the router itself.
     * </p>
     *
     * @param deleteVirtualRouterRequest
     * @return A Java Future containing the result of the DeleteVirtualRouter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ResourceInUseException You can't delete the specified resource because it's in use or required by
     *         another resource.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.DeleteVirtualRouter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DeleteVirtualRouter" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVirtualRouterResponse> deleteVirtualRouter(
            DeleteVirtualRouterRequest deleteVirtualRouterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVirtualRouterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVirtualRouterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVirtualRouter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVirtualRouterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVirtualRouterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVirtualRouterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVirtualRouterRequest, DeleteVirtualRouterResponse>()
                            .withOperationName("DeleteVirtualRouter").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteVirtualRouterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteVirtualRouterRequest));
            CompletableFuture<DeleteVirtualRouterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing virtual service.
     * </p>
     *
     * @param deleteVirtualServiceRequest
     * @return A Java Future containing the result of the DeleteVirtualService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ResourceInUseException You can't delete the specified resource because it's in use or required by
     *         another resource.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.DeleteVirtualService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DeleteVirtualService" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVirtualServiceResponse> deleteVirtualService(
            DeleteVirtualServiceRequest deleteVirtualServiceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVirtualServiceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVirtualServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVirtualService");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVirtualServiceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVirtualServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVirtualServiceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVirtualServiceRequest, DeleteVirtualServiceResponse>()
                            .withOperationName("DeleteVirtualService").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteVirtualServiceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteVirtualServiceRequest));
            CompletableFuture<DeleteVirtualServiceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an existing gateway route.
     * </p>
     *
     * @param describeGatewayRouteRequest
     * @return A Java Future containing the result of the DescribeGatewayRoute operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.DescribeGatewayRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DescribeGatewayRoute" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeGatewayRouteResponse> describeGatewayRoute(
            DescribeGatewayRouteRequest describeGatewayRouteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeGatewayRouteRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGatewayRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGatewayRoute");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeGatewayRouteResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeGatewayRouteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeGatewayRouteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeGatewayRouteRequest, DescribeGatewayRouteResponse>()
                            .withOperationName("DescribeGatewayRoute").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeGatewayRouteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeGatewayRouteRequest));
            CompletableFuture<DescribeGatewayRouteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an existing service mesh.
     * </p>
     *
     * @param describeMeshRequest
     * @return A Java Future containing the result of the DescribeMesh operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.DescribeMesh
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DescribeMesh" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMeshResponse> describeMesh(DescribeMeshRequest describeMeshRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeMeshRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeMeshRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMesh");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeMeshResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeMeshResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeMeshResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeMeshRequest, DescribeMeshResponse>()
                            .withOperationName("DescribeMesh").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeMeshRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeMeshRequest));
            CompletableFuture<DescribeMeshResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an existing route.
     * </p>
     *
     * @param describeRouteRequest
     * @return A Java Future containing the result of the DescribeRoute operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.DescribeRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DescribeRoute" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRouteResponse> describeRoute(DescribeRouteRequest describeRouteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeRouteRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRoute");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRouteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeRouteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeRouteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRouteRequest, DescribeRouteResponse>()
                            .withOperationName("DescribeRoute").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeRouteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeRouteRequest));
            CompletableFuture<DescribeRouteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an existing virtual gateway.
     * </p>
     *
     * @param describeVirtualGatewayRequest
     * @return A Java Future containing the result of the DescribeVirtualGateway operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.DescribeVirtualGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DescribeVirtualGateway"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeVirtualGatewayResponse> describeVirtualGateway(
            DescribeVirtualGatewayRequest describeVirtualGatewayRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeVirtualGatewayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVirtualGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVirtualGateway");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeVirtualGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeVirtualGatewayResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeVirtualGatewayResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeVirtualGatewayRequest, DescribeVirtualGatewayResponse>()
                            .withOperationName("DescribeVirtualGateway").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeVirtualGatewayRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeVirtualGatewayRequest));
            CompletableFuture<DescribeVirtualGatewayResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an existing virtual node.
     * </p>
     *
     * @param describeVirtualNodeRequest
     * @return A Java Future containing the result of the DescribeVirtualNode operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.DescribeVirtualNode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DescribeVirtualNode" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeVirtualNodeResponse> describeVirtualNode(
            DescribeVirtualNodeRequest describeVirtualNodeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeVirtualNodeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVirtualNodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVirtualNode");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeVirtualNodeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeVirtualNodeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeVirtualNodeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeVirtualNodeRequest, DescribeVirtualNodeResponse>()
                            .withOperationName("DescribeVirtualNode").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeVirtualNodeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeVirtualNodeRequest));
            CompletableFuture<DescribeVirtualNodeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an existing virtual router.
     * </p>
     *
     * @param describeVirtualRouterRequest
     * @return A Java Future containing the result of the DescribeVirtualRouter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.DescribeVirtualRouter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DescribeVirtualRouter" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeVirtualRouterResponse> describeVirtualRouter(
            DescribeVirtualRouterRequest describeVirtualRouterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeVirtualRouterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVirtualRouterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVirtualRouter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeVirtualRouterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeVirtualRouterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeVirtualRouterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeVirtualRouterRequest, DescribeVirtualRouterResponse>()
                            .withOperationName("DescribeVirtualRouter").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeVirtualRouterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeVirtualRouterRequest));
            CompletableFuture<DescribeVirtualRouterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an existing virtual service.
     * </p>
     *
     * @param describeVirtualServiceRequest
     * @return A Java Future containing the result of the DescribeVirtualService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.DescribeVirtualService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/DescribeVirtualService"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeVirtualServiceResponse> describeVirtualService(
            DescribeVirtualServiceRequest describeVirtualServiceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeVirtualServiceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVirtualServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVirtualService");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeVirtualServiceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeVirtualServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeVirtualServiceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeVirtualServiceRequest, DescribeVirtualServiceResponse>()
                            .withOperationName("DescribeVirtualService").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeVirtualServiceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeVirtualServiceRequest));
            CompletableFuture<DescribeVirtualServiceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of existing gateway routes that are associated to a virtual gateway.
     * </p>
     *
     * @param listGatewayRoutesRequest
     * @return A Java Future containing the result of the ListGatewayRoutes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.ListGatewayRoutes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListGatewayRoutes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListGatewayRoutesResponse> listGatewayRoutes(ListGatewayRoutesRequest listGatewayRoutesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listGatewayRoutesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGatewayRoutesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGatewayRoutes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGatewayRoutesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListGatewayRoutesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGatewayRoutesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGatewayRoutesRequest, ListGatewayRoutesResponse>()
                            .withOperationName("ListGatewayRoutes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListGatewayRoutesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listGatewayRoutesRequest));
            CompletableFuture<ListGatewayRoutesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of existing service meshes.
     * </p>
     *
     * @param listMeshesRequest
     * @return A Java Future containing the result of the ListMeshes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.ListMeshes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListMeshes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListMeshesResponse> listMeshes(ListMeshesRequest listMeshesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMeshesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMeshesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMeshes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMeshesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListMeshesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMeshesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMeshesRequest, ListMeshesResponse>().withOperationName("ListMeshes")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListMeshesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listMeshesRequest));
            CompletableFuture<ListMeshesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of existing routes in a service mesh.
     * </p>
     *
     * @param listRoutesRequest
     * @return A Java Future containing the result of the ListRoutes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.ListRoutes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListRoutes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListRoutesResponse> listRoutes(ListRoutesRequest listRoutesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRoutesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRoutesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRoutes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRoutesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListRoutesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRoutesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRoutesRequest, ListRoutesResponse>().withOperationName("ListRoutes")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRoutesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRoutesRequest));
            CompletableFuture<ListRoutesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the tags for an App Mesh resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of existing virtual gateways in a service mesh.
     * </p>
     *
     * @param listVirtualGatewaysRequest
     * @return A Java Future containing the result of the ListVirtualGateways operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.ListVirtualGateways
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListVirtualGateways" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListVirtualGatewaysResponse> listVirtualGateways(
            ListVirtualGatewaysRequest listVirtualGatewaysRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listVirtualGatewaysRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVirtualGatewaysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVirtualGateways");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVirtualGatewaysResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVirtualGatewaysResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVirtualGatewaysResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVirtualGatewaysRequest, ListVirtualGatewaysResponse>()
                            .withOperationName("ListVirtualGateways").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListVirtualGatewaysRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listVirtualGatewaysRequest));
            CompletableFuture<ListVirtualGatewaysResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of existing virtual nodes.
     * </p>
     *
     * @param listVirtualNodesRequest
     * @return A Java Future containing the result of the ListVirtualNodes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.ListVirtualNodes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListVirtualNodes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListVirtualNodesResponse> listVirtualNodes(ListVirtualNodesRequest listVirtualNodesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listVirtualNodesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVirtualNodesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVirtualNodes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVirtualNodesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVirtualNodesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVirtualNodesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVirtualNodesRequest, ListVirtualNodesResponse>()
                            .withOperationName("ListVirtualNodes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListVirtualNodesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listVirtualNodesRequest));
            CompletableFuture<ListVirtualNodesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of existing virtual routers in a service mesh.
     * </p>
     *
     * @param listVirtualRoutersRequest
     * @return A Java Future containing the result of the ListVirtualRouters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.ListVirtualRouters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListVirtualRouters" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListVirtualRoutersResponse> listVirtualRouters(ListVirtualRoutersRequest listVirtualRoutersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listVirtualRoutersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVirtualRoutersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVirtualRouters");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVirtualRoutersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVirtualRoutersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVirtualRoutersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVirtualRoutersRequest, ListVirtualRoutersResponse>()
                            .withOperationName("ListVirtualRouters").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListVirtualRoutersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listVirtualRoutersRequest));
            CompletableFuture<ListVirtualRoutersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of existing virtual services in a service mesh.
     * </p>
     *
     * @param listVirtualServicesRequest
     * @return A Java Future containing the result of the ListVirtualServices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.ListVirtualServices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListVirtualServices" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListVirtualServicesResponse> listVirtualServices(
            ListVirtualServicesRequest listVirtualServicesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listVirtualServicesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVirtualServicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVirtualServices");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVirtualServicesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVirtualServicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVirtualServicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVirtualServicesRequest, ListVirtualServicesResponse>()
                            .withOperationName("ListVirtualServices").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListVirtualServicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listVirtualServicesRequest));
            CompletableFuture<ListVirtualServicesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates the specified tags to a resource with the specified <code>resourceArn</code>. If existing tags on a
     * resource aren't specified in the request parameters, they aren't changed. When a resource is deleted, the tags
     * associated with that resource are also deleted.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyTagsException The request exceeds the maximum allowed number of tags allowed per resource. The
     *         current limit is 50 user tags per resource. You must reduce the number of tags in the request. None of
     *         the tags in this request were applied.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes specified tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing gateway route that is associated to a specified virtual gateway in a service mesh.
     * </p>
     *
     * @param updateGatewayRouteRequest
     * @return A Java Future containing the result of the UpdateGatewayRoute operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>ConflictException The request contains a client token that was used for a previous update resource
     *         call with different specifications. Try the request again with a new client token.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>LimitExceededException You have exceeded a service limit for your account. For more information, see
     *         <a href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>App Mesh User Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.UpdateGatewayRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/UpdateGatewayRoute" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGatewayRouteResponse> updateGatewayRoute(UpdateGatewayRouteRequest updateGatewayRouteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateGatewayRouteRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGatewayRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGatewayRoute");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGatewayRouteResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateGatewayRouteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateGatewayRouteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGatewayRouteRequest, UpdateGatewayRouteResponse>()
                            .withOperationName("UpdateGatewayRoute").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateGatewayRouteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateGatewayRouteRequest));
            CompletableFuture<UpdateGatewayRouteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing service mesh.
     * </p>
     *
     * @param updateMeshRequest
     * @return A Java Future containing the result of the UpdateMesh operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>ConflictException The request contains a client token that was used for a previous update resource
     *         call with different specifications. Try the request again with a new client token.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.UpdateMesh
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/UpdateMesh" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateMeshResponse> updateMesh(UpdateMeshRequest updateMeshRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateMeshRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMeshRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMesh");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateMeshResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateMeshResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateMeshResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateMeshRequest, UpdateMeshResponse>().withOperationName("UpdateMesh")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateMeshRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateMeshRequest));
            CompletableFuture<UpdateMeshResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing route for a specified service mesh and virtual router.
     * </p>
     *
     * @param updateRouteRequest
     * @return A Java Future containing the result of the UpdateRoute operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>ConflictException The request contains a client token that was used for a previous update resource
     *         call with different specifications. Try the request again with a new client token.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>LimitExceededException You have exceeded a service limit for your account. For more information, see
     *         <a href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>App Mesh User Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.UpdateRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/UpdateRoute" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRouteResponse> updateRoute(UpdateRouteRequest updateRouteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRouteRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRoute");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRouteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateRouteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRouteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRouteRequest, UpdateRouteResponse>()
                            .withOperationName("UpdateRoute").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateRouteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateRouteRequest));
            CompletableFuture<UpdateRouteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing virtual gateway in a specified service mesh.
     * </p>
     *
     * @param updateVirtualGatewayRequest
     * @return A Java Future containing the result of the UpdateVirtualGateway operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>ConflictException The request contains a client token that was used for a previous update resource
     *         call with different specifications. Try the request again with a new client token.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>LimitExceededException You have exceeded a service limit for your account. For more information, see
     *         <a href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>App Mesh User Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.UpdateVirtualGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/UpdateVirtualGateway" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateVirtualGatewayResponse> updateVirtualGateway(
            UpdateVirtualGatewayRequest updateVirtualGatewayRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateVirtualGatewayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVirtualGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVirtualGateway");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateVirtualGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateVirtualGatewayResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateVirtualGatewayResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateVirtualGatewayRequest, UpdateVirtualGatewayResponse>()
                            .withOperationName("UpdateVirtualGateway").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateVirtualGatewayRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateVirtualGatewayRequest));
            CompletableFuture<UpdateVirtualGatewayResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing virtual node in a specified service mesh.
     * </p>
     *
     * @param updateVirtualNodeRequest
     * @return A Java Future containing the result of the UpdateVirtualNode operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>ConflictException The request contains a client token that was used for a previous update resource
     *         call with different specifications. Try the request again with a new client token.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>LimitExceededException You have exceeded a service limit for your account. For more information, see
     *         <a href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>App Mesh User Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.UpdateVirtualNode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/UpdateVirtualNode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateVirtualNodeResponse> updateVirtualNode(UpdateVirtualNodeRequest updateVirtualNodeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateVirtualNodeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVirtualNodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVirtualNode");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateVirtualNodeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateVirtualNodeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateVirtualNodeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateVirtualNodeRequest, UpdateVirtualNodeResponse>()
                            .withOperationName("UpdateVirtualNode").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateVirtualNodeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateVirtualNodeRequest));
            CompletableFuture<UpdateVirtualNodeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing virtual router in a specified service mesh.
     * </p>
     *
     * @param updateVirtualRouterRequest
     * @return A Java Future containing the result of the UpdateVirtualRouter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>ConflictException The request contains a client token that was used for a previous update resource
     *         call with different specifications. Try the request again with a new client token.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>LimitExceededException You have exceeded a service limit for your account. For more information, see
     *         <a href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>App Mesh User Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.UpdateVirtualRouter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/UpdateVirtualRouter" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateVirtualRouterResponse> updateVirtualRouter(
            UpdateVirtualRouterRequest updateVirtualRouterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateVirtualRouterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVirtualRouterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVirtualRouter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateVirtualRouterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateVirtualRouterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateVirtualRouterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateVirtualRouterRequest, UpdateVirtualRouterResponse>()
                            .withOperationName("UpdateVirtualRouter").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateVirtualRouterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateVirtualRouterRequest));
            CompletableFuture<UpdateVirtualRouterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing virtual service in a specified service mesh.
     * </p>
     *
     * @param updateVirtualServiceRequest
     * @return A Java Future containing the result of the UpdateVirtualService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The specified resource doesn't exist. Check your request syntax and try again.</li>
     *         <li>BadRequestException The request syntax was malformed. Check your request syntax and try again.</li>
     *         <li>ConflictException The request contains a client token that was used for a previous update resource
     *         call with different specifications. Try the request again with a new client token.</li>
     *         <li>TooManyRequestsException The maximum request rate permitted by the App Mesh APIs has been exceeded
     *         for your account. For best results, use an increasing or variable sleep interval between requests.</li>
     *         <li>ForbiddenException You don't have permissions to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the service.</li>
     *         <li>InternalServerErrorException The request processing has failed because of an unknown error,
     *         exception, or failure.</li>
     *         <li>LimitExceededException You have exceeded a service limit for your account. For more information, see
     *         <a href="https://docs.aws.amazon.com/app-mesh/latest/userguide/service-quotas.html">Service Limits</a> in
     *         the <i>App Mesh User Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppMeshException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppMeshAsyncClient.UpdateVirtualService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/UpdateVirtualService" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateVirtualServiceResponse> updateVirtualService(
            UpdateVirtualServiceRequest updateVirtualServiceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateVirtualServiceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVirtualServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "App Mesh");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVirtualService");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateVirtualServiceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateVirtualServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateVirtualServiceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateVirtualServiceRequest, UpdateVirtualServiceResponse>()
                            .withOperationName("UpdateVirtualService").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateVirtualServiceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateVirtualServiceRequest));
            CompletableFuture<UpdateVirtualServiceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final AppMeshServiceClientConfiguration serviceClientConfiguration() {
        return new AppMeshServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AppMeshException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        AppMeshServiceClientConfigurationBuilder serviceConfigBuilder = new AppMeshServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
