/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appmesh.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object representing the specification of a virtual node.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class VirtualNodeSpec implements SdkPojo, Serializable, ToCopyableBuilder<VirtualNodeSpec.Builder, VirtualNodeSpec> {
    private static final SdkField<List<String>> BACKENDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(VirtualNodeSpec::backends))
            .setter(setter(Builder::backends))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("backends").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Listener>> LISTENERS_FIELD = SdkField
            .<List<Listener>> builder(MarshallingType.LIST)
            .getter(getter(VirtualNodeSpec::listeners))
            .setter(setter(Builder::listeners))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("listeners").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Listener> builder(MarshallingType.SDK_POJO)
                                            .constructor(Listener::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ServiceDiscovery> SERVICE_DISCOVERY_FIELD = SdkField
            .<ServiceDiscovery> builder(MarshallingType.SDK_POJO).getter(getter(VirtualNodeSpec::serviceDiscovery))
            .setter(setter(Builder::serviceDiscovery)).constructor(ServiceDiscovery::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceDiscovery").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BACKENDS_FIELD,
            LISTENERS_FIELD, SERVICE_DISCOVERY_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> backends;

    private final List<Listener> listeners;

    private final ServiceDiscovery serviceDiscovery;

    private VirtualNodeSpec(BuilderImpl builder) {
        this.backends = builder.backends;
        this.listeners = builder.listeners;
        this.serviceDiscovery = builder.serviceDiscovery;
    }

    /**
     * <p>
     * The backends to which the virtual node is expected to send outbound traffic.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The backends to which the virtual node is expected to send outbound traffic.
     */
    public List<String> backends() {
        return backends;
    }

    /**
     * <p>
     * The listeners from which the virtual node is expected to receive inbound traffic.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The listeners from which the virtual node is expected to receive inbound traffic.
     */
    public List<Listener> listeners() {
        return listeners;
    }

    /**
     * <p>
     * The service discovery information for the virtual node.
     * </p>
     * 
     * @return The service discovery information for the virtual node.
     */
    public ServiceDiscovery serviceDiscovery() {
        return serviceDiscovery;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(backends());
        hashCode = 31 * hashCode + Objects.hashCode(listeners());
        hashCode = 31 * hashCode + Objects.hashCode(serviceDiscovery());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof VirtualNodeSpec)) {
            return false;
        }
        VirtualNodeSpec other = (VirtualNodeSpec) obj;
        return Objects.equals(backends(), other.backends()) && Objects.equals(listeners(), other.listeners())
                && Objects.equals(serviceDiscovery(), other.serviceDiscovery());
    }

    @Override
    public String toString() {
        return ToString.builder("VirtualNodeSpec").add("Backends", backends()).add("Listeners", listeners())
                .add("ServiceDiscovery", serviceDiscovery()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "backends":
            return Optional.ofNullable(clazz.cast(backends()));
        case "listeners":
            return Optional.ofNullable(clazz.cast(listeners()));
        case "serviceDiscovery":
            return Optional.ofNullable(clazz.cast(serviceDiscovery()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<VirtualNodeSpec, T> g) {
        return obj -> g.apply((VirtualNodeSpec) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, VirtualNodeSpec> {
        /**
         * <p>
         * The backends to which the virtual node is expected to send outbound traffic.
         * </p>
         * 
         * @param backends
         *        The backends to which the virtual node is expected to send outbound traffic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backends(Collection<String> backends);

        /**
         * <p>
         * The backends to which the virtual node is expected to send outbound traffic.
         * </p>
         * 
         * @param backends
         *        The backends to which the virtual node is expected to send outbound traffic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backends(String... backends);

        /**
         * <p>
         * The listeners from which the virtual node is expected to receive inbound traffic.
         * </p>
         * 
         * @param listeners
         *        The listeners from which the virtual node is expected to receive inbound traffic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder listeners(Collection<Listener> listeners);

        /**
         * <p>
         * The listeners from which the virtual node is expected to receive inbound traffic.
         * </p>
         * 
         * @param listeners
         *        The listeners from which the virtual node is expected to receive inbound traffic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder listeners(Listener... listeners);

        /**
         * <p>
         * The listeners from which the virtual node is expected to receive inbound traffic.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Listener>.Builder} avoiding the need to
         * create one manually via {@link List<Listener>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Listener>.Builder#build()} is called immediately and its
         * result is passed to {@link #listeners(List<Listener>)}.
         * 
         * @param listeners
         *        a consumer that will call methods on {@link List<Listener>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #listeners(List<Listener>)
         */
        Builder listeners(Consumer<Listener.Builder>... listeners);

        /**
         * <p>
         * The service discovery information for the virtual node.
         * </p>
         * 
         * @param serviceDiscovery
         *        The service discovery information for the virtual node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceDiscovery(ServiceDiscovery serviceDiscovery);

        /**
         * <p>
         * The service discovery information for the virtual node.
         * </p>
         * This is a convenience that creates an instance of the {@link ServiceDiscovery.Builder} avoiding the need to
         * create one manually via {@link ServiceDiscovery#builder()}.
         *
         * When the {@link Consumer} completes, {@link ServiceDiscovery.Builder#build()} is called immediately and its
         * result is passed to {@link #serviceDiscovery(ServiceDiscovery)}.
         * 
         * @param serviceDiscovery
         *        a consumer that will call methods on {@link ServiceDiscovery.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #serviceDiscovery(ServiceDiscovery)
         */
        default Builder serviceDiscovery(Consumer<ServiceDiscovery.Builder> serviceDiscovery) {
            return serviceDiscovery(ServiceDiscovery.builder().applyMutation(serviceDiscovery).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private List<String> backends = DefaultSdkAutoConstructList.getInstance();

        private List<Listener> listeners = DefaultSdkAutoConstructList.getInstance();

        private ServiceDiscovery serviceDiscovery;

        private BuilderImpl() {
        }

        private BuilderImpl(VirtualNodeSpec model) {
            backends(model.backends);
            listeners(model.listeners);
            serviceDiscovery(model.serviceDiscovery);
        }

        public final Collection<String> getBackends() {
            return backends;
        }

        @Override
        public final Builder backends(Collection<String> backends) {
            this.backends = BackendsCopier.copy(backends);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder backends(String... backends) {
            backends(Arrays.asList(backends));
            return this;
        }

        public final void setBackends(Collection<String> backends) {
            this.backends = BackendsCopier.copy(backends);
        }

        public final Collection<Listener.Builder> getListeners() {
            return listeners != null ? listeners.stream().map(Listener::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder listeners(Collection<Listener> listeners) {
            this.listeners = ListenersCopier.copy(listeners);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder listeners(Listener... listeners) {
            listeners(Arrays.asList(listeners));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder listeners(Consumer<Listener.Builder>... listeners) {
            listeners(Stream.of(listeners).map(c -> Listener.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setListeners(Collection<Listener.BuilderImpl> listeners) {
            this.listeners = ListenersCopier.copyFromBuilder(listeners);
        }

        public final ServiceDiscovery.Builder getServiceDiscovery() {
            return serviceDiscovery != null ? serviceDiscovery.toBuilder() : null;
        }

        @Override
        public final Builder serviceDiscovery(ServiceDiscovery serviceDiscovery) {
            this.serviceDiscovery = serviceDiscovery;
            return this;
        }

        public final void setServiceDiscovery(ServiceDiscovery.BuilderImpl serviceDiscovery) {
            this.serviceDiscovery = serviceDiscovery != null ? serviceDiscovery.build() : null;
        }

        @Override
        public VirtualNodeSpec build() {
            return new VirtualNodeSpec(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
