/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apprunner;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.apprunner.model.AppRunnerException;
import software.amazon.awssdk.services.apprunner.model.AppRunnerRequest;
import software.amazon.awssdk.services.apprunner.model.AssociateCustomDomainRequest;
import software.amazon.awssdk.services.apprunner.model.AssociateCustomDomainResponse;
import software.amazon.awssdk.services.apprunner.model.CreateAutoScalingConfigurationRequest;
import software.amazon.awssdk.services.apprunner.model.CreateAutoScalingConfigurationResponse;
import software.amazon.awssdk.services.apprunner.model.CreateConnectionRequest;
import software.amazon.awssdk.services.apprunner.model.CreateConnectionResponse;
import software.amazon.awssdk.services.apprunner.model.CreateServiceRequest;
import software.amazon.awssdk.services.apprunner.model.CreateServiceResponse;
import software.amazon.awssdk.services.apprunner.model.DeleteAutoScalingConfigurationRequest;
import software.amazon.awssdk.services.apprunner.model.DeleteAutoScalingConfigurationResponse;
import software.amazon.awssdk.services.apprunner.model.DeleteConnectionRequest;
import software.amazon.awssdk.services.apprunner.model.DeleteConnectionResponse;
import software.amazon.awssdk.services.apprunner.model.DeleteServiceRequest;
import software.amazon.awssdk.services.apprunner.model.DeleteServiceResponse;
import software.amazon.awssdk.services.apprunner.model.DescribeAutoScalingConfigurationRequest;
import software.amazon.awssdk.services.apprunner.model.DescribeAutoScalingConfigurationResponse;
import software.amazon.awssdk.services.apprunner.model.DescribeCustomDomainsRequest;
import software.amazon.awssdk.services.apprunner.model.DescribeCustomDomainsResponse;
import software.amazon.awssdk.services.apprunner.model.DescribeServiceRequest;
import software.amazon.awssdk.services.apprunner.model.DescribeServiceResponse;
import software.amazon.awssdk.services.apprunner.model.DisassociateCustomDomainRequest;
import software.amazon.awssdk.services.apprunner.model.DisassociateCustomDomainResponse;
import software.amazon.awssdk.services.apprunner.model.InternalServiceErrorException;
import software.amazon.awssdk.services.apprunner.model.InvalidRequestException;
import software.amazon.awssdk.services.apprunner.model.InvalidStateException;
import software.amazon.awssdk.services.apprunner.model.ListAutoScalingConfigurationsRequest;
import software.amazon.awssdk.services.apprunner.model.ListAutoScalingConfigurationsResponse;
import software.amazon.awssdk.services.apprunner.model.ListConnectionsRequest;
import software.amazon.awssdk.services.apprunner.model.ListConnectionsResponse;
import software.amazon.awssdk.services.apprunner.model.ListOperationsRequest;
import software.amazon.awssdk.services.apprunner.model.ListOperationsResponse;
import software.amazon.awssdk.services.apprunner.model.ListServicesRequest;
import software.amazon.awssdk.services.apprunner.model.ListServicesResponse;
import software.amazon.awssdk.services.apprunner.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.apprunner.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.apprunner.model.PauseServiceRequest;
import software.amazon.awssdk.services.apprunner.model.PauseServiceResponse;
import software.amazon.awssdk.services.apprunner.model.ResourceNotFoundException;
import software.amazon.awssdk.services.apprunner.model.ResumeServiceRequest;
import software.amazon.awssdk.services.apprunner.model.ResumeServiceResponse;
import software.amazon.awssdk.services.apprunner.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.apprunner.model.StartDeploymentRequest;
import software.amazon.awssdk.services.apprunner.model.StartDeploymentResponse;
import software.amazon.awssdk.services.apprunner.model.TagResourceRequest;
import software.amazon.awssdk.services.apprunner.model.TagResourceResponse;
import software.amazon.awssdk.services.apprunner.model.UntagResourceRequest;
import software.amazon.awssdk.services.apprunner.model.UntagResourceResponse;
import software.amazon.awssdk.services.apprunner.model.UpdateServiceRequest;
import software.amazon.awssdk.services.apprunner.model.UpdateServiceResponse;
import software.amazon.awssdk.services.apprunner.paginators.DescribeCustomDomainsIterable;
import software.amazon.awssdk.services.apprunner.paginators.ListAutoScalingConfigurationsIterable;
import software.amazon.awssdk.services.apprunner.paginators.ListConnectionsIterable;
import software.amazon.awssdk.services.apprunner.paginators.ListOperationsIterable;
import software.amazon.awssdk.services.apprunner.paginators.ListServicesIterable;
import software.amazon.awssdk.services.apprunner.transform.AssociateCustomDomainRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.CreateAutoScalingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.CreateConnectionRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.CreateServiceRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.DeleteAutoScalingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.DeleteConnectionRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.DeleteServiceRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.DescribeAutoScalingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.DescribeCustomDomainsRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.DescribeServiceRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.DisassociateCustomDomainRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.ListAutoScalingConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.ListConnectionsRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.ListOperationsRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.ListServicesRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.PauseServiceRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.ResumeServiceRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.StartDeploymentRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.UpdateServiceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link AppRunnerClient}.
 *
 * @see AppRunnerClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAppRunnerClient implements AppRunnerClient {
    private static final Logger log = Logger.loggerFor(DefaultAppRunnerClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAppRunnerClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Associate your own domain name with the App Runner subdomain URL of your App Runner service.
     * </p>
     * <p>
     * After you call <code>AssociateCustomDomain</code> and receive a successful response, use the information in the
     * <a>CustomDomain</a> record that's returned to add CNAME records to your Domain Name System (DNS). For each mapped
     * domain name, add a mapping to the target App Runner subdomain and one or more certificate validation records. App
     * Runner then performs DNS validation to verify that you own or control the domain name that you associated. App
     * Runner tracks domain validity in a certificate stored in <a
     * href="https://docs.aws.amazon.com/acm/latest/userguide">AWS Certificate Manager (ACM)</a>.
     * </p>
     *
     * @param associateCustomDomainRequest
     * @return Result of the AssociateCustomDomain operation returned by the service.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws InvalidStateException
     *         You can't perform this action when the resource is in its current state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.AssociateCustomDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/AssociateCustomDomain"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateCustomDomainResponse associateCustomDomain(AssociateCustomDomainRequest associateCustomDomainRequest)
            throws InvalidRequestException, InternalServiceErrorException, InvalidStateException, AwsServiceException,
            SdkClientException, AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateCustomDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateCustomDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateCustomDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateCustomDomain");

            return clientHandler.execute(new ClientExecutionParams<AssociateCustomDomainRequest, AssociateCustomDomainResponse>()
                    .withOperationName("AssociateCustomDomain").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(associateCustomDomainRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AssociateCustomDomainRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Create an App Runner automatic scaling configuration resource. App Runner requires this resource when you create
     * App Runner services that require non-default auto scaling settings. You can share an auto scaling configuration
     * across multiple services.
     * </p>
     * <p>
     * Create multiple revisions of a configuration by using the same <code>AutoScalingConfigurationName</code> and
     * different <code>AutoScalingConfigurationRevision</code> values. When you create a service, you can set it to use
     * the latest active revision of an auto scaling configuration or a specific revision.
     * </p>
     * <p>
     * Configure a higher <code>MinSize</code> to increase the spread of your App Runner service over more Availability
     * Zones in the Amazon Web Services Region. The tradeoff is a higher minimal cost.
     * </p>
     * <p>
     * Configure a lower <code>MaxSize</code> to control your cost. The tradeoff is lower responsiveness during peak
     * demand.
     * </p>
     *
     * @param createAutoScalingConfigurationRequest
     * @return Result of the CreateAutoScalingConfiguration operation returned by the service.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws ServiceQuotaExceededException
     *         App Runner can't create this resource. You've reached your account quota for this resource type.</p>
     *         <p>
     *         For App Runner per-resource quotas, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/apprunner.html">App Runner endpoints and quotas</a>
     *         in the <i>Amazon Web Services General Reference</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.CreateAutoScalingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/CreateAutoScalingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateAutoScalingConfigurationResponse createAutoScalingConfiguration(
            CreateAutoScalingConfigurationRequest createAutoScalingConfigurationRequest) throws InvalidRequestException,
            InternalServiceErrorException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAutoScalingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateAutoScalingConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createAutoScalingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAutoScalingConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateAutoScalingConfigurationRequest, CreateAutoScalingConfigurationResponse>()
                            .withOperationName("CreateAutoScalingConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createAutoScalingConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateAutoScalingConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Create an App Runner connection resource. App Runner requires a connection resource when you create App Runner
     * services that access private repositories from certain third-party providers. You can share a connection across
     * multiple services.
     * </p>
     * <p>
     * A connection resource is needed to access GitHub repositories. GitHub requires a user interface approval process
     * through the App Runner console before you can use the connection.
     * </p>
     *
     * @param createConnectionRequest
     * @return Result of the CreateConnection operation returned by the service.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws ServiceQuotaExceededException
     *         App Runner can't create this resource. You've reached your account quota for this resource type.</p>
     *         <p>
     *         For App Runner per-resource quotas, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/apprunner.html">App Runner endpoints and quotas</a>
     *         in the <i>Amazon Web Services General Reference</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.CreateConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/CreateConnection" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateConnectionResponse createConnection(CreateConnectionRequest createConnectionRequest)
            throws InvalidRequestException, InternalServiceErrorException, ServiceQuotaExceededException, AwsServiceException,
            SdkClientException, AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateConnectionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateConnectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConnection");

            return clientHandler.execute(new ClientExecutionParams<CreateConnectionRequest, CreateConnectionResponse>()
                    .withOperationName("CreateConnection").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createConnectionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateConnectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Create an App Runner service. After the service is created, the action also automatically starts a deployment.
     * </p>
     * <p>
     * This is an asynchronous operation. On a successful call, you can use the returned <code>OperationId</code> and
     * the <a href="https://docs.aws.amazon.com/apprunner/latest/api/API_ListOperations.html">ListOperations</a> call to
     * track the operation's progress.
     * </p>
     *
     * @param createServiceRequest
     * @return Result of the CreateService operation returned by the service.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws ServiceQuotaExceededException
     *         App Runner can't create this resource. You've reached your account quota for this resource type.</p>
     *         <p>
     *         For App Runner per-resource quotas, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/apprunner.html">App Runner endpoints and quotas</a>
     *         in the <i>Amazon Web Services General Reference</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.CreateService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/CreateService" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateServiceResponse createService(CreateServiceRequest createServiceRequest) throws InvalidRequestException,
            InternalServiceErrorException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateServiceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateService");

            return clientHandler.execute(new ClientExecutionParams<CreateServiceRequest, CreateServiceResponse>()
                    .withOperationName("CreateService").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createServiceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateServiceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete an App Runner automatic scaling configuration resource. You can delete a specific revision or the latest
     * active revision. You can't delete a configuration that's used by one or more App Runner services.
     * </p>
     *
     * @param deleteAutoScalingConfigurationRequest
     * @return Result of the DeleteAutoScalingConfiguration operation returned by the service.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws ResourceNotFoundException
     *         A resource doesn't exist for the specified Amazon Resource Name (ARN) in your Amazon Web Services
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.DeleteAutoScalingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DeleteAutoScalingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteAutoScalingConfigurationResponse deleteAutoScalingConfiguration(
            DeleteAutoScalingConfigurationRequest deleteAutoScalingConfigurationRequest) throws InvalidRequestException,
            InternalServiceErrorException, ResourceNotFoundException, AwsServiceException, SdkClientException, AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAutoScalingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteAutoScalingConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteAutoScalingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAutoScalingConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteAutoScalingConfigurationRequest, DeleteAutoScalingConfigurationResponse>()
                            .withOperationName("DeleteAutoScalingConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteAutoScalingConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteAutoScalingConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete an App Runner connection. You must first ensure that there are no running App Runner services that use
     * this connection. If there are any, the <code>DeleteConnection</code> action fails.
     * </p>
     *
     * @param deleteConnectionRequest
     * @return Result of the DeleteConnection operation returned by the service.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws ResourceNotFoundException
     *         A resource doesn't exist for the specified Amazon Resource Name (ARN) in your Amazon Web Services
     *         account.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.DeleteConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DeleteConnection" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteConnectionResponse deleteConnection(DeleteConnectionRequest deleteConnectionRequest)
            throws InvalidRequestException, ResourceNotFoundException, InternalServiceErrorException, AwsServiceException,
            SdkClientException, AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteConnectionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteConnectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConnection");

            return clientHandler.execute(new ClientExecutionParams<DeleteConnectionRequest, DeleteConnectionResponse>()
                    .withOperationName("DeleteConnection").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteConnectionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteConnectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete an App Runner service.
     * </p>
     * <p>
     * This is an asynchronous operation. On a successful call, you can use the returned <code>OperationId</code> and
     * the <a>ListOperations</a> call to track the operation's progress.
     * </p>
     *
     * @param deleteServiceRequest
     * @return Result of the DeleteService operation returned by the service.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws ResourceNotFoundException
     *         A resource doesn't exist for the specified Amazon Resource Name (ARN) in your Amazon Web Services
     *         account.
     * @throws InvalidStateException
     *         You can't perform this action when the resource is in its current state.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.DeleteService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DeleteService" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteServiceResponse deleteService(DeleteServiceRequest deleteServiceRequest) throws InvalidRequestException,
            ResourceNotFoundException, InvalidStateException, InternalServiceErrorException, AwsServiceException,
            SdkClientException, AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteServiceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteService");

            return clientHandler.execute(new ClientExecutionParams<DeleteServiceRequest, DeleteServiceResponse>()
                    .withOperationName("DeleteService").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteServiceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteServiceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Return a full description of an App Runner automatic scaling configuration resource.
     * </p>
     *
     * @param describeAutoScalingConfigurationRequest
     * @return Result of the DescribeAutoScalingConfiguration operation returned by the service.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws ResourceNotFoundException
     *         A resource doesn't exist for the specified Amazon Resource Name (ARN) in your Amazon Web Services
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.DescribeAutoScalingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DescribeAutoScalingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAutoScalingConfigurationResponse describeAutoScalingConfiguration(
            DescribeAutoScalingConfigurationRequest describeAutoScalingConfigurationRequest) throws InvalidRequestException,
            InternalServiceErrorException, ResourceNotFoundException, AwsServiceException, SdkClientException, AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAutoScalingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAutoScalingConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeAutoScalingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAutoScalingConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAutoScalingConfigurationRequest, DescribeAutoScalingConfigurationResponse>()
                            .withOperationName("DescribeAutoScalingConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeAutoScalingConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAutoScalingConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Return a description of custom domain names that are associated with an App Runner service.
     * </p>
     *
     * @param describeCustomDomainsRequest
     * @return Result of the DescribeCustomDomains operation returned by the service.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws ResourceNotFoundException
     *         A resource doesn't exist for the specified Amazon Resource Name (ARN) in your Amazon Web Services
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.DescribeCustomDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DescribeCustomDomains"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeCustomDomainsResponse describeCustomDomains(DescribeCustomDomainsRequest describeCustomDomainsRequest)
            throws InvalidRequestException, InternalServiceErrorException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeCustomDomainsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeCustomDomainsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeCustomDomainsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCustomDomains");

            return clientHandler.execute(new ClientExecutionParams<DescribeCustomDomainsRequest, DescribeCustomDomainsResponse>()
                    .withOperationName("DescribeCustomDomains").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeCustomDomainsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeCustomDomainsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Return a description of custom domain names that are associated with an App Runner service.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeCustomDomains(software.amazon.awssdk.services.apprunner.model.DescribeCustomDomainsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.apprunner.paginators.DescribeCustomDomainsIterable responses = client.describeCustomDomainsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.apprunner.paginators.DescribeCustomDomainsIterable responses = client
     *             .describeCustomDomainsPaginator(request);
     *     for (software.amazon.awssdk.services.apprunner.model.DescribeCustomDomainsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.apprunner.paginators.DescribeCustomDomainsIterable responses = client.describeCustomDomainsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeCustomDomains(software.amazon.awssdk.services.apprunner.model.DescribeCustomDomainsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeCustomDomainsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws ResourceNotFoundException
     *         A resource doesn't exist for the specified Amazon Resource Name (ARN) in your Amazon Web Services
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.DescribeCustomDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DescribeCustomDomains"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeCustomDomainsIterable describeCustomDomainsPaginator(DescribeCustomDomainsRequest describeCustomDomainsRequest)
            throws InvalidRequestException, InternalServiceErrorException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, AppRunnerException {
        return new DescribeCustomDomainsIterable(this, applyPaginatorUserAgent(describeCustomDomainsRequest));
    }

    /**
     * <p>
     * Return a full description of an App Runner service.
     * </p>
     *
     * @param describeServiceRequest
     * @return Result of the DescribeService operation returned by the service.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws ResourceNotFoundException
     *         A resource doesn't exist for the specified Amazon Resource Name (ARN) in your Amazon Web Services
     *         account.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.DescribeService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DescribeService" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeServiceResponse describeService(DescribeServiceRequest describeServiceRequest) throws InvalidRequestException,
            ResourceNotFoundException, InternalServiceErrorException, AwsServiceException, SdkClientException, AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeServiceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeService");

            return clientHandler.execute(new ClientExecutionParams<DescribeServiceRequest, DescribeServiceResponse>()
                    .withOperationName("DescribeService").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeServiceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeServiceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociate a custom domain name from an App Runner service.
     * </p>
     * <p>
     * Certificates tracking domain validity are associated with a custom domain and are stored in <a
     * href="https://docs.aws.amazon.com/acm/latest/userguide">AWS Certificate Manager (ACM)</a>. These certificates
     * aren't deleted as part of this action. App Runner delays certificate deletion for 30 days after a domain is
     * disassociated from your service.
     * </p>
     *
     * @param disassociateCustomDomainRequest
     * @return Result of the DisassociateCustomDomain operation returned by the service.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws ResourceNotFoundException
     *         A resource doesn't exist for the specified Amazon Resource Name (ARN) in your Amazon Web Services
     *         account.
     * @throws InvalidStateException
     *         You can't perform this action when the resource is in its current state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.DisassociateCustomDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DisassociateCustomDomain"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateCustomDomainResponse disassociateCustomDomain(
            DisassociateCustomDomainRequest disassociateCustomDomainRequest) throws InvalidRequestException,
            InternalServiceErrorException, ResourceNotFoundException, InvalidStateException, AwsServiceException,
            SdkClientException, AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateCustomDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateCustomDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateCustomDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateCustomDomain");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateCustomDomainRequest, DisassociateCustomDomainResponse>()
                            .withOperationName("DisassociateCustomDomain").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disassociateCustomDomainRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateCustomDomainRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of App Runner automatic scaling configurations in your Amazon Web Services account. You can query
     * the revisions for a specific configuration name or the revisions for all configurations in your account. You can
     * optionally query only the latest revision of each requested name.
     * </p>
     *
     * @param listAutoScalingConfigurationsRequest
     * @return Result of the ListAutoScalingConfigurations operation returned by the service.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.ListAutoScalingConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/ListAutoScalingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAutoScalingConfigurationsResponse listAutoScalingConfigurations(
            ListAutoScalingConfigurationsRequest listAutoScalingConfigurationsRequest) throws InvalidRequestException,
            InternalServiceErrorException, AwsServiceException, SdkClientException, AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAutoScalingConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAutoScalingConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listAutoScalingConfigurationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAutoScalingConfigurations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListAutoScalingConfigurationsRequest, ListAutoScalingConfigurationsResponse>()
                            .withOperationName("ListAutoScalingConfigurations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listAutoScalingConfigurationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListAutoScalingConfigurationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of App Runner automatic scaling configurations in your Amazon Web Services account. You can query
     * the revisions for a specific configuration name or the revisions for all configurations in your account. You can
     * optionally query only the latest revision of each requested name.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAutoScalingConfigurations(software.amazon.awssdk.services.apprunner.model.ListAutoScalingConfigurationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.apprunner.paginators.ListAutoScalingConfigurationsIterable responses = client.listAutoScalingConfigurationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.apprunner.paginators.ListAutoScalingConfigurationsIterable responses = client
     *             .listAutoScalingConfigurationsPaginator(request);
     *     for (software.amazon.awssdk.services.apprunner.model.ListAutoScalingConfigurationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.apprunner.paginators.ListAutoScalingConfigurationsIterable responses = client.listAutoScalingConfigurationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAutoScalingConfigurations(software.amazon.awssdk.services.apprunner.model.ListAutoScalingConfigurationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAutoScalingConfigurationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.ListAutoScalingConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/ListAutoScalingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAutoScalingConfigurationsIterable listAutoScalingConfigurationsPaginator(
            ListAutoScalingConfigurationsRequest listAutoScalingConfigurationsRequest) throws InvalidRequestException,
            InternalServiceErrorException, AwsServiceException, SdkClientException, AppRunnerException {
        return new ListAutoScalingConfigurationsIterable(this, applyPaginatorUserAgent(listAutoScalingConfigurationsRequest));
    }

    /**
     * <p>
     * Returns a list of App Runner connections that are associated with your Amazon Web Services account.
     * </p>
     *
     * @param listConnectionsRequest
     * @return Result of the ListConnections operation returned by the service.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.ListConnections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/ListConnections" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListConnectionsResponse listConnections(ListConnectionsRequest listConnectionsRequest) throws InvalidRequestException,
            InternalServiceErrorException, AwsServiceException, SdkClientException, AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListConnectionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListConnectionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConnectionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConnections");

            return clientHandler.execute(new ClientExecutionParams<ListConnectionsRequest, ListConnectionsResponse>()
                    .withOperationName("ListConnections").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listConnectionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListConnectionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of App Runner connections that are associated with your Amazon Web Services account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listConnections(software.amazon.awssdk.services.apprunner.model.ListConnectionsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.apprunner.paginators.ListConnectionsIterable responses = client.listConnectionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.apprunner.paginators.ListConnectionsIterable responses = client
     *             .listConnectionsPaginator(request);
     *     for (software.amazon.awssdk.services.apprunner.model.ListConnectionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.apprunner.paginators.ListConnectionsIterable responses = client.listConnectionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listConnections(software.amazon.awssdk.services.apprunner.model.ListConnectionsRequest)} operation.</b>
     * </p>
     *
     * @param listConnectionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.ListConnections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/ListConnections" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListConnectionsIterable listConnectionsPaginator(ListConnectionsRequest listConnectionsRequest)
            throws InvalidRequestException, InternalServiceErrorException, AwsServiceException, SdkClientException,
            AppRunnerException {
        return new ListConnectionsIterable(this, applyPaginatorUserAgent(listConnectionsRequest));
    }

    /**
     * <p>
     * Return a list of operations that occurred on an App Runner service.
     * </p>
     * <p>
     * The resulting list of <a>OperationSummary</a> objects is sorted in reverse chronological order. The first object
     * on the list represents the last started operation.
     * </p>
     *
     * @param listOperationsRequest
     * @return Result of the ListOperations operation returned by the service.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws ResourceNotFoundException
     *         A resource doesn't exist for the specified Amazon Resource Name (ARN) in your Amazon Web Services
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.ListOperations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/ListOperations" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListOperationsResponse listOperations(ListOperationsRequest listOperationsRequest) throws InvalidRequestException,
            InternalServiceErrorException, ResourceNotFoundException, AwsServiceException, SdkClientException, AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListOperationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListOperationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOperationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOperations");

            return clientHandler.execute(new ClientExecutionParams<ListOperationsRequest, ListOperationsResponse>()
                    .withOperationName("ListOperations").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listOperationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListOperationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Return a list of operations that occurred on an App Runner service.
     * </p>
     * <p>
     * The resulting list of <a>OperationSummary</a> objects is sorted in reverse chronological order. The first object
     * on the list represents the last started operation.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listOperations(software.amazon.awssdk.services.apprunner.model.ListOperationsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.apprunner.paginators.ListOperationsIterable responses = client.listOperationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.apprunner.paginators.ListOperationsIterable responses = client
     *             .listOperationsPaginator(request);
     *     for (software.amazon.awssdk.services.apprunner.model.ListOperationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.apprunner.paginators.ListOperationsIterable responses = client.listOperationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOperations(software.amazon.awssdk.services.apprunner.model.ListOperationsRequest)} operation.</b>
     * </p>
     *
     * @param listOperationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws ResourceNotFoundException
     *         A resource doesn't exist for the specified Amazon Resource Name (ARN) in your Amazon Web Services
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.ListOperations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/ListOperations" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListOperationsIterable listOperationsPaginator(ListOperationsRequest listOperationsRequest)
            throws InvalidRequestException, InternalServiceErrorException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, AppRunnerException {
        return new ListOperationsIterable(this, applyPaginatorUserAgent(listOperationsRequest));
    }

    /**
     * <p>
     * Returns a list of running App Runner services in your Amazon Web Services account.
     * </p>
     *
     * @param listServicesRequest
     * @return Result of the ListServices operation returned by the service.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.ListServices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/ListServices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListServicesResponse listServices(ListServicesRequest listServicesRequest) throws InvalidRequestException,
            InternalServiceErrorException, AwsServiceException, SdkClientException, AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListServicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListServicesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServices");

            return clientHandler.execute(new ClientExecutionParams<ListServicesRequest, ListServicesResponse>()
                    .withOperationName("ListServices").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listServicesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListServicesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of running App Runner services in your Amazon Web Services account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listServices(software.amazon.awssdk.services.apprunner.model.ListServicesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.apprunner.paginators.ListServicesIterable responses = client.listServicesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.apprunner.paginators.ListServicesIterable responses = client.listServicesPaginator(request);
     *     for (software.amazon.awssdk.services.apprunner.model.ListServicesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.apprunner.paginators.ListServicesIterable responses = client.listServicesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listServices(software.amazon.awssdk.services.apprunner.model.ListServicesRequest)} operation.</b>
     * </p>
     *
     * @param listServicesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.ListServices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/ListServices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListServicesIterable listServicesPaginator(ListServicesRequest listServicesRequest) throws InvalidRequestException,
            InternalServiceErrorException, AwsServiceException, SdkClientException, AppRunnerException {
        return new ListServicesIterable(this, applyPaginatorUserAgent(listServicesRequest));
    }

    /**
     * <p>
     * List tags that are associated with for an App Runner resource. The response contains a list of tag key-value
     * pairs.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource doesn't exist for the specified Amazon Resource Name (ARN) in your Amazon Web Services
     *         account.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws InvalidStateException
     *         You can't perform this action when the resource is in its current state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InternalServiceErrorException, InvalidRequestException, InvalidStateException,
            AwsServiceException, SdkClientException, AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Pause an active App Runner service. App Runner reduces compute capacity for the service to zero and loses state
     * (for example, ephemeral storage is removed).
     * </p>
     * <p>
     * This is an asynchronous operation. On a successful call, you can use the returned <code>OperationId</code> and
     * the <a>ListOperations</a> call to track the operation's progress.
     * </p>
     *
     * @param pauseServiceRequest
     * @return Result of the PauseService operation returned by the service.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws ResourceNotFoundException
     *         A resource doesn't exist for the specified Amazon Resource Name (ARN) in your Amazon Web Services
     *         account.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws InvalidStateException
     *         You can't perform this action when the resource is in its current state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.PauseService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/PauseService" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PauseServiceResponse pauseService(PauseServiceRequest pauseServiceRequest) throws InvalidRequestException,
            ResourceNotFoundException, InternalServiceErrorException, InvalidStateException, AwsServiceException,
            SdkClientException, AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PauseServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PauseServiceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, pauseServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PauseService");

            return clientHandler.execute(new ClientExecutionParams<PauseServiceRequest, PauseServiceResponse>()
                    .withOperationName("PauseService").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(pauseServiceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PauseServiceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Resume an active App Runner service. App Runner provisions compute capacity for the service.
     * </p>
     * <p>
     * This is an asynchronous operation. On a successful call, you can use the returned <code>OperationId</code> and
     * the <a>ListOperations</a> call to track the operation's progress.
     * </p>
     *
     * @param resumeServiceRequest
     * @return Result of the ResumeService operation returned by the service.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws ResourceNotFoundException
     *         A resource doesn't exist for the specified Amazon Resource Name (ARN) in your Amazon Web Services
     *         account.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws InvalidStateException
     *         You can't perform this action when the resource is in its current state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.ResumeService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/ResumeService" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ResumeServiceResponse resumeService(ResumeServiceRequest resumeServiceRequest) throws InvalidRequestException,
            ResourceNotFoundException, InternalServiceErrorException, InvalidStateException, AwsServiceException,
            SdkClientException, AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ResumeServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ResumeServiceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resumeServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResumeService");

            return clientHandler.execute(new ClientExecutionParams<ResumeServiceRequest, ResumeServiceResponse>()
                    .withOperationName("ResumeService").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(resumeServiceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ResumeServiceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Initiate a manual deployment of the latest commit in a source code repository or the latest image in a source
     * image repository to an App Runner service.
     * </p>
     * <p>
     * For a source code repository, App Runner retrieves the commit and builds a Docker image. For a source image
     * repository, App Runner retrieves the latest Docker image. In both cases, App Runner then deploys the new image to
     * your service and starts a new container instance.
     * </p>
     * <p>
     * This is an asynchronous operation. On a successful call, you can use the returned <code>OperationId</code> and
     * the <a>ListOperations</a> call to track the operation's progress.
     * </p>
     *
     * @param startDeploymentRequest
     * @return Result of the StartDeployment operation returned by the service.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws ResourceNotFoundException
     *         A resource doesn't exist for the specified Amazon Resource Name (ARN) in your Amazon Web Services
     *         account.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.StartDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/StartDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartDeploymentResponse startDeployment(StartDeploymentRequest startDeploymentRequest) throws InvalidRequestException,
            ResourceNotFoundException, InternalServiceErrorException, AwsServiceException, SdkClientException, AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartDeploymentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDeployment");

            return clientHandler.execute(new ClientExecutionParams<StartDeploymentRequest, StartDeploymentResponse>()
                    .withOperationName("StartDeployment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startDeploymentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartDeploymentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Add tags to, or update the tag values of, an App Runner resource. A tag is a key-value pair.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource doesn't exist for the specified Amazon Resource Name (ARN) in your Amazon Web Services
     *         account.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws InvalidStateException
     *         You can't perform this action when the resource is in its current state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            InternalServiceErrorException, InvalidRequestException, InvalidStateException, AwsServiceException,
            SdkClientException, AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Remove tags from an App Runner resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource doesn't exist for the specified Amazon Resource Name (ARN) in your Amazon Web Services
     *         account.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws InvalidStateException
     *         You can't perform this action when the resource is in its current state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InternalServiceErrorException, InvalidRequestException, InvalidStateException, AwsServiceException,
            SdkClientException, AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update an App Runner service. You can update the source configuration and instance configuration of the service.
     * You can also update the ARN of the auto scaling configuration resource that's associated with the service.
     * However, you can't change the name or the encryption configuration of the service. These can be set only when you
     * create the service.
     * </p>
     * <p>
     * To update the tags applied to your service, use the separate actions <a>TagResource</a> and <a>UntagResource</a>.
     * </p>
     * <p>
     * This is an asynchronous operation. On a successful call, you can use the returned <code>OperationId</code> and
     * the <a>ListOperations</a> call to track the operation's progress.
     * </p>
     *
     * @param updateServiceRequest
     * @return Result of the UpdateService operation returned by the service.
     * @throws InvalidRequestException
     *         One or more input parameters aren't valid. Refer to the API action's document page, correct the input
     *         parameters, and try the action again.
     * @throws ResourceNotFoundException
     *         A resource doesn't exist for the specified Amazon Resource Name (ARN) in your Amazon Web Services
     *         account.
     * @throws InvalidStateException
     *         You can't perform this action when the resource is in its current state.
     * @throws InternalServiceErrorException
     *         An unexpected service exception occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppRunnerClient.UpdateService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/UpdateService" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateServiceResponse updateService(UpdateServiceRequest updateServiceRequest) throws InvalidRequestException,
            ResourceNotFoundException, InvalidStateException, InternalServiceErrorException, AwsServiceException,
            SdkClientException, AppRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateServiceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateService");

            return clientHandler.execute(new ClientExecutionParams<UpdateServiceRequest, UpdateServiceResponse>()
                    .withOperationName("UpdateService").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateServiceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateServiceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AppRunnerException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceErrorException")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidStateException")
                                .exceptionBuilderSupplier(InvalidStateException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends AppRunnerRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
