/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apprunner.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a source image repository.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ImageRepository implements SdkPojo, Serializable, ToCopyableBuilder<ImageRepository.Builder, ImageRepository> {
    private static final SdkField<String> IMAGE_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ImageIdentifier").getter(getter(ImageRepository::imageIdentifier))
            .setter(setter(Builder::imageIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageIdentifier").build()).build();

    private static final SdkField<ImageConfiguration> IMAGE_CONFIGURATION_FIELD = SdkField
            .<ImageConfiguration> builder(MarshallingType.SDK_POJO).memberName("ImageConfiguration")
            .getter(getter(ImageRepository::imageConfiguration)).setter(setter(Builder::imageConfiguration))
            .constructor(ImageConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageConfiguration").build())
            .build();

    private static final SdkField<String> IMAGE_REPOSITORY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ImageRepositoryType").getter(getter(ImageRepository::imageRepositoryTypeAsString))
            .setter(setter(Builder::imageRepositoryType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageRepositoryType").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IMAGE_IDENTIFIER_FIELD,
            IMAGE_CONFIGURATION_FIELD, IMAGE_REPOSITORY_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String imageIdentifier;

    private final ImageConfiguration imageConfiguration;

    private final String imageRepositoryType;

    private ImageRepository(BuilderImpl builder) {
        this.imageIdentifier = builder.imageIdentifier;
        this.imageConfiguration = builder.imageConfiguration;
        this.imageRepositoryType = builder.imageRepositoryType;
    }

    /**
     * <p>
     * The identifier of an image.
     * </p>
     * <p>
     * For an image in Amazon Elastic Container Registry (Amazon ECR), this is an image name. For the image name format,
     * see <a href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/docker-pull-ecr-image.html">Pulling an
     * image</a> in the <i>Amazon ECR User Guide</i>.
     * </p>
     * 
     * @return The identifier of an image.</p>
     *         <p>
     *         For an image in Amazon Elastic Container Registry (Amazon ECR), this is an image name. For the image name
     *         format, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/docker-pull-ecr-image.html">Pulling an
     *         image</a> in the <i>Amazon ECR User Guide</i>.
     */
    public final String imageIdentifier() {
        return imageIdentifier;
    }

    /**
     * <p>
     * Configuration for running the identified image.
     * </p>
     * 
     * @return Configuration for running the identified image.
     */
    public final ImageConfiguration imageConfiguration() {
        return imageConfiguration;
    }

    /**
     * <p>
     * The type of the image repository. This reflects the repository provider and whether the repository is private or
     * public.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #imageRepositoryType} will return {@link ImageRepositoryType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #imageRepositoryTypeAsString}.
     * </p>
     * 
     * @return The type of the image repository. This reflects the repository provider and whether the repository is
     *         private or public.
     * @see ImageRepositoryType
     */
    public final ImageRepositoryType imageRepositoryType() {
        return ImageRepositoryType.fromValue(imageRepositoryType);
    }

    /**
     * <p>
     * The type of the image repository. This reflects the repository provider and whether the repository is private or
     * public.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #imageRepositoryType} will return {@link ImageRepositoryType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #imageRepositoryTypeAsString}.
     * </p>
     * 
     * @return The type of the image repository. This reflects the repository provider and whether the repository is
     *         private or public.
     * @see ImageRepositoryType
     */
    public final String imageRepositoryTypeAsString() {
        return imageRepositoryType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(imageIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(imageConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(imageRepositoryTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImageRepository)) {
            return false;
        }
        ImageRepository other = (ImageRepository) obj;
        return Objects.equals(imageIdentifier(), other.imageIdentifier())
                && Objects.equals(imageConfiguration(), other.imageConfiguration())
                && Objects.equals(imageRepositoryTypeAsString(), other.imageRepositoryTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ImageRepository").add("ImageIdentifier", imageIdentifier())
                .add("ImageConfiguration", imageConfiguration()).add("ImageRepositoryType", imageRepositoryTypeAsString())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ImageIdentifier":
            return Optional.ofNullable(clazz.cast(imageIdentifier()));
        case "ImageConfiguration":
            return Optional.ofNullable(clazz.cast(imageConfiguration()));
        case "ImageRepositoryType":
            return Optional.ofNullable(clazz.cast(imageRepositoryTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ImageRepository, T> g) {
        return obj -> g.apply((ImageRepository) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ImageRepository> {
        /**
         * <p>
         * The identifier of an image.
         * </p>
         * <p>
         * For an image in Amazon Elastic Container Registry (Amazon ECR), this is an image name. For the image name
         * format, see <a
         * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/docker-pull-ecr-image.html">Pulling an image</a>
         * in the <i>Amazon ECR User Guide</i>.
         * </p>
         * 
         * @param imageIdentifier
         *        The identifier of an image.</p>
         *        <p>
         *        For an image in Amazon Elastic Container Registry (Amazon ECR), this is an image name. For the image
         *        name format, see <a
         *        href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/docker-pull-ecr-image.html">Pulling an
         *        image</a> in the <i>Amazon ECR User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageIdentifier(String imageIdentifier);

        /**
         * <p>
         * Configuration for running the identified image.
         * </p>
         * 
         * @param imageConfiguration
         *        Configuration for running the identified image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageConfiguration(ImageConfiguration imageConfiguration);

        /**
         * <p>
         * Configuration for running the identified image.
         * </p>
         * This is a convenience that creates an instance of the {@link ImageConfiguration.Builder} avoiding the need to
         * create one manually via {@link ImageConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link ImageConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #imageConfiguration(ImageConfiguration)}.
         * 
         * @param imageConfiguration
         *        a consumer that will call methods on {@link ImageConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #imageConfiguration(ImageConfiguration)
         */
        default Builder imageConfiguration(Consumer<ImageConfiguration.Builder> imageConfiguration) {
            return imageConfiguration(ImageConfiguration.builder().applyMutation(imageConfiguration).build());
        }

        /**
         * <p>
         * The type of the image repository. This reflects the repository provider and whether the repository is private
         * or public.
         * </p>
         * 
         * @param imageRepositoryType
         *        The type of the image repository. This reflects the repository provider and whether the repository is
         *        private or public.
         * @see ImageRepositoryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImageRepositoryType
         */
        Builder imageRepositoryType(String imageRepositoryType);

        /**
         * <p>
         * The type of the image repository. This reflects the repository provider and whether the repository is private
         * or public.
         * </p>
         * 
         * @param imageRepositoryType
         *        The type of the image repository. This reflects the repository provider and whether the repository is
         *        private or public.
         * @see ImageRepositoryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImageRepositoryType
         */
        Builder imageRepositoryType(ImageRepositoryType imageRepositoryType);
    }

    static final class BuilderImpl implements Builder {
        private String imageIdentifier;

        private ImageConfiguration imageConfiguration;

        private String imageRepositoryType;

        private BuilderImpl() {
        }

        private BuilderImpl(ImageRepository model) {
            imageIdentifier(model.imageIdentifier);
            imageConfiguration(model.imageConfiguration);
            imageRepositoryType(model.imageRepositoryType);
        }

        public final String getImageIdentifier() {
            return imageIdentifier;
        }

        public final void setImageIdentifier(String imageIdentifier) {
            this.imageIdentifier = imageIdentifier;
        }

        @Override
        @Transient
        public final Builder imageIdentifier(String imageIdentifier) {
            this.imageIdentifier = imageIdentifier;
            return this;
        }

        public final ImageConfiguration.Builder getImageConfiguration() {
            return imageConfiguration != null ? imageConfiguration.toBuilder() : null;
        }

        public final void setImageConfiguration(ImageConfiguration.BuilderImpl imageConfiguration) {
            this.imageConfiguration = imageConfiguration != null ? imageConfiguration.build() : null;
        }

        @Override
        @Transient
        public final Builder imageConfiguration(ImageConfiguration imageConfiguration) {
            this.imageConfiguration = imageConfiguration;
            return this;
        }

        public final String getImageRepositoryType() {
            return imageRepositoryType;
        }

        public final void setImageRepositoryType(String imageRepositoryType) {
            this.imageRepositoryType = imageRepositoryType;
        }

        @Override
        @Transient
        public final Builder imageRepositoryType(String imageRepositoryType) {
            this.imageRepositoryType = imageRepositoryType;
            return this;
        }

        @Override
        @Transient
        public final Builder imageRepositoryType(ImageRepositoryType imageRepositoryType) {
            this.imageRepositoryType(imageRepositoryType == null ? null : imageRepositoryType.toString());
            return this;
        }

        @Override
        public ImageRepository build() {
            return new ImageRepository(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
