/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apprunner.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an App Runner service. It can describe a service in any state, including deleted services.
 * </p>
 * <p>
 * This type contains the full information about a service, including configuration details. It's returned by the <a
 * href="https://docs.aws.amazon.com/apprunner/latest/api/API_CreateService.html">CreateService</a>, <a
 * href="https://docs.aws.amazon.com/apprunner/latest/api/API_DescribeService.html">DescribeService</a>, and <a
 * href="https://docs.aws.amazon.com/apprunner/latest/api/API_DeleteService.html">DeleteService</a> actions. A subset of
 * this information is returned by the <a
 * href="https://docs.aws.amazon.com/apprunner/latest/api/API_ListServices.html">ListServices</a> action using the <a
 * href="https://docs.aws.amazon.com/apprunner/latest/api/API_ServiceSummary.html">ServiceSummary</a> type.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Service implements SdkPojo, Serializable, ToCopyableBuilder<Service.Builder, Service> {
    private static final SdkField<String> SERVICE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServiceName").getter(getter(Service::serviceName)).setter(setter(Builder::serviceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceName").build()).build();

    private static final SdkField<String> SERVICE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServiceId").getter(getter(Service::serviceId)).setter(setter(Builder::serviceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceId").build()).build();

    private static final SdkField<String> SERVICE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServiceArn").getter(getter(Service::serviceArn)).setter(setter(Builder::serviceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceArn").build()).build();

    private static final SdkField<String> SERVICE_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServiceUrl").getter(getter(Service::serviceUrl)).setter(setter(Builder::serviceUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceUrl").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt").getter(getter(Service::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAt").build()).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("UpdatedAt").getter(getter(Service::updatedAt)).setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdatedAt").build()).build();

    private static final SdkField<Instant> DELETED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("DeletedAt").getter(getter(Service::deletedAt)).setter(setter(Builder::deletedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeletedAt").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Service::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<SourceConfiguration> SOURCE_CONFIGURATION_FIELD = SdkField
            .<SourceConfiguration> builder(MarshallingType.SDK_POJO).memberName("SourceConfiguration")
            .getter(getter(Service::sourceConfiguration)).setter(setter(Builder::sourceConfiguration))
            .constructor(SourceConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceConfiguration").build())
            .build();

    private static final SdkField<InstanceConfiguration> INSTANCE_CONFIGURATION_FIELD = SdkField
            .<InstanceConfiguration> builder(MarshallingType.SDK_POJO).memberName("InstanceConfiguration")
            .getter(getter(Service::instanceConfiguration)).setter(setter(Builder::instanceConfiguration))
            .constructor(InstanceConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceConfiguration").build())
            .build();

    private static final SdkField<EncryptionConfiguration> ENCRYPTION_CONFIGURATION_FIELD = SdkField
            .<EncryptionConfiguration> builder(MarshallingType.SDK_POJO).memberName("EncryptionConfiguration")
            .getter(getter(Service::encryptionConfiguration)).setter(setter(Builder::encryptionConfiguration))
            .constructor(EncryptionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionConfiguration").build())
            .build();

    private static final SdkField<HealthCheckConfiguration> HEALTH_CHECK_CONFIGURATION_FIELD = SdkField
            .<HealthCheckConfiguration> builder(MarshallingType.SDK_POJO).memberName("HealthCheckConfiguration")
            .getter(getter(Service::healthCheckConfiguration)).setter(setter(Builder::healthCheckConfiguration))
            .constructor(HealthCheckConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HealthCheckConfiguration").build())
            .build();

    private static final SdkField<AutoScalingConfigurationSummary> AUTO_SCALING_CONFIGURATION_SUMMARY_FIELD = SdkField
            .<AutoScalingConfigurationSummary> builder(MarshallingType.SDK_POJO)
            .memberName("AutoScalingConfigurationSummary")
            .getter(getter(Service::autoScalingConfigurationSummary))
            .setter(setter(Builder::autoScalingConfigurationSummary))
            .constructor(AutoScalingConfigurationSummary::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoScalingConfigurationSummary")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SERVICE_NAME_FIELD,
            SERVICE_ID_FIELD, SERVICE_ARN_FIELD, SERVICE_URL_FIELD, CREATED_AT_FIELD, UPDATED_AT_FIELD, DELETED_AT_FIELD,
            STATUS_FIELD, SOURCE_CONFIGURATION_FIELD, INSTANCE_CONFIGURATION_FIELD, ENCRYPTION_CONFIGURATION_FIELD,
            HEALTH_CHECK_CONFIGURATION_FIELD, AUTO_SCALING_CONFIGURATION_SUMMARY_FIELD));

    private static final long serialVersionUID = 1L;

    private final String serviceName;

    private final String serviceId;

    private final String serviceArn;

    private final String serviceUrl;

    private final Instant createdAt;

    private final Instant updatedAt;

    private final Instant deletedAt;

    private final String status;

    private final SourceConfiguration sourceConfiguration;

    private final InstanceConfiguration instanceConfiguration;

    private final EncryptionConfiguration encryptionConfiguration;

    private final HealthCheckConfiguration healthCheckConfiguration;

    private final AutoScalingConfigurationSummary autoScalingConfigurationSummary;

    private Service(BuilderImpl builder) {
        this.serviceName = builder.serviceName;
        this.serviceId = builder.serviceId;
        this.serviceArn = builder.serviceArn;
        this.serviceUrl = builder.serviceUrl;
        this.createdAt = builder.createdAt;
        this.updatedAt = builder.updatedAt;
        this.deletedAt = builder.deletedAt;
        this.status = builder.status;
        this.sourceConfiguration = builder.sourceConfiguration;
        this.instanceConfiguration = builder.instanceConfiguration;
        this.encryptionConfiguration = builder.encryptionConfiguration;
        this.healthCheckConfiguration = builder.healthCheckConfiguration;
        this.autoScalingConfigurationSummary = builder.autoScalingConfigurationSummary;
    }

    /**
     * <p>
     * The customer-provided service name.
     * </p>
     * 
     * @return The customer-provided service name.
     */
    public final String serviceName() {
        return serviceName;
    }

    /**
     * <p>
     * An ID that App Runner generated for this service. It's unique within the Amazon Web Services Region.
     * </p>
     * 
     * @return An ID that App Runner generated for this service. It's unique within the Amazon Web Services Region.
     */
    public final String serviceId() {
        return serviceId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of this service.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of this service.
     */
    public final String serviceArn() {
        return serviceArn;
    }

    /**
     * <p>
     * A subdomain URL that App Runner generated for this service. You can use this URL to access your service web
     * application.
     * </p>
     * 
     * @return A subdomain URL that App Runner generated for this service. You can use this URL to access your service
     *         web application.
     */
    public final String serviceUrl() {
        return serviceUrl;
    }

    /**
     * <p>
     * The time when the App Runner service was created. It's in the Unix time stamp format.
     * </p>
     * 
     * @return The time when the App Runner service was created. It's in the Unix time stamp format.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The time when the App Runner service was last updated at. It's in the Unix time stamp format.
     * </p>
     * 
     * @return The time when the App Runner service was last updated at. It's in the Unix time stamp format.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    /**
     * <p>
     * The time when the App Runner service was deleted. It's in the Unix time stamp format.
     * </p>
     * 
     * @return The time when the App Runner service was deleted. It's in the Unix time stamp format.
     */
    public final Instant deletedAt() {
        return deletedAt;
    }

    /**
     * <p>
     * The current state of the App Runner service. These particular values mean the following.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATE_FAILED</code> – The service failed to create. To troubleshoot this failure, read the failure events
     * and logs, change any parameters that need to be fixed, and retry the call to create the service.
     * </p>
     * <p>
     * The failed service isn't usable, and still counts towards your service quota. When you're done analyzing the
     * failure, delete the service.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETE_FAILED</code> – The service failed to delete and can't be successfully recovered. Retry the service
     * deletion call to ensure that all related resources are removed.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ServiceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current state of the App Runner service. These particular values mean the following.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATE_FAILED</code> – The service failed to create. To troubleshoot this failure, read the failure
     *         events and logs, change any parameters that need to be fixed, and retry the call to create the service.
     *         </p>
     *         <p>
     *         The failed service isn't usable, and still counts towards your service quota. When you're done analyzing
     *         the failure, delete the service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETE_FAILED</code> – The service failed to delete and can't be successfully recovered. Retry the
     *         service deletion call to ensure that all related resources are removed.
     *         </p>
     *         </li>
     * @see ServiceStatus
     */
    public final ServiceStatus status() {
        return ServiceStatus.fromValue(status);
    }

    /**
     * <p>
     * The current state of the App Runner service. These particular values mean the following.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATE_FAILED</code> – The service failed to create. To troubleshoot this failure, read the failure events
     * and logs, change any parameters that need to be fixed, and retry the call to create the service.
     * </p>
     * <p>
     * The failed service isn't usable, and still counts towards your service quota. When you're done analyzing the
     * failure, delete the service.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETE_FAILED</code> – The service failed to delete and can't be successfully recovered. Retry the service
     * deletion call to ensure that all related resources are removed.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ServiceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current state of the App Runner service. These particular values mean the following.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATE_FAILED</code> – The service failed to create. To troubleshoot this failure, read the failure
     *         events and logs, change any parameters that need to be fixed, and retry the call to create the service.
     *         </p>
     *         <p>
     *         The failed service isn't usable, and still counts towards your service quota. When you're done analyzing
     *         the failure, delete the service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETE_FAILED</code> – The service failed to delete and can't be successfully recovered. Retry the
     *         service deletion call to ensure that all related resources are removed.
     *         </p>
     *         </li>
     * @see ServiceStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The source deployed to the App Runner service. It can be a code or an image repository.
     * </p>
     * 
     * @return The source deployed to the App Runner service. It can be a code or an image repository.
     */
    public final SourceConfiguration sourceConfiguration() {
        return sourceConfiguration;
    }

    /**
     * <p>
     * The runtime configuration of instances (scaling units) of this service.
     * </p>
     * 
     * @return The runtime configuration of instances (scaling units) of this service.
     */
    public final InstanceConfiguration instanceConfiguration() {
        return instanceConfiguration;
    }

    /**
     * <p>
     * The encryption key that App Runner uses to encrypt the service logs and the copy of the source repository that
     * App Runner maintains for the service. It can be either a customer-provided encryption key or an Amazon Web
     * Services managed CMK.
     * </p>
     * 
     * @return The encryption key that App Runner uses to encrypt the service logs and the copy of the source repository
     *         that App Runner maintains for the service. It can be either a customer-provided encryption key or an
     *         Amazon Web Services managed CMK.
     */
    public final EncryptionConfiguration encryptionConfiguration() {
        return encryptionConfiguration;
    }

    /**
     * <p>
     * The settings for the health check that App Runner performs to monitor the health of this service.
     * </p>
     * 
     * @return The settings for the health check that App Runner performs to monitor the health of this service.
     */
    public final HealthCheckConfiguration healthCheckConfiguration() {
        return healthCheckConfiguration;
    }

    /**
     * <p>
     * Summary information for the App Runner automatic scaling configuration resource that's associated with this
     * service.
     * </p>
     * 
     * @return Summary information for the App Runner automatic scaling configuration resource that's associated with
     *         this service.
     */
    public final AutoScalingConfigurationSummary autoScalingConfigurationSummary() {
        return autoScalingConfigurationSummary;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(serviceName());
        hashCode = 31 * hashCode + Objects.hashCode(serviceId());
        hashCode = 31 * hashCode + Objects.hashCode(serviceArn());
        hashCode = 31 * hashCode + Objects.hashCode(serviceUrl());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(deletedAt());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(sourceConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(instanceConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(healthCheckConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingConfigurationSummary());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Service)) {
            return false;
        }
        Service other = (Service) obj;
        return Objects.equals(serviceName(), other.serviceName()) && Objects.equals(serviceId(), other.serviceId())
                && Objects.equals(serviceArn(), other.serviceArn()) && Objects.equals(serviceUrl(), other.serviceUrl())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(updatedAt(), other.updatedAt())
                && Objects.equals(deletedAt(), other.deletedAt()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(sourceConfiguration(), other.sourceConfiguration())
                && Objects.equals(instanceConfiguration(), other.instanceConfiguration())
                && Objects.equals(encryptionConfiguration(), other.encryptionConfiguration())
                && Objects.equals(healthCheckConfiguration(), other.healthCheckConfiguration())
                && Objects.equals(autoScalingConfigurationSummary(), other.autoScalingConfigurationSummary());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Service").add("ServiceName", serviceName()).add("ServiceId", serviceId())
                .add("ServiceArn", serviceArn()).add("ServiceUrl", serviceUrl()).add("CreatedAt", createdAt())
                .add("UpdatedAt", updatedAt()).add("DeletedAt", deletedAt()).add("Status", statusAsString())
                .add("SourceConfiguration", sourceConfiguration()).add("InstanceConfiguration", instanceConfiguration())
                .add("EncryptionConfiguration", encryptionConfiguration())
                .add("HealthCheckConfiguration", healthCheckConfiguration())
                .add("AutoScalingConfigurationSummary", autoScalingConfigurationSummary()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ServiceName":
            return Optional.ofNullable(clazz.cast(serviceName()));
        case "ServiceId":
            return Optional.ofNullable(clazz.cast(serviceId()));
        case "ServiceArn":
            return Optional.ofNullable(clazz.cast(serviceArn()));
        case "ServiceUrl":
            return Optional.ofNullable(clazz.cast(serviceUrl()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "UpdatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        case "DeletedAt":
            return Optional.ofNullable(clazz.cast(deletedAt()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "SourceConfiguration":
            return Optional.ofNullable(clazz.cast(sourceConfiguration()));
        case "InstanceConfiguration":
            return Optional.ofNullable(clazz.cast(instanceConfiguration()));
        case "EncryptionConfiguration":
            return Optional.ofNullable(clazz.cast(encryptionConfiguration()));
        case "HealthCheckConfiguration":
            return Optional.ofNullable(clazz.cast(healthCheckConfiguration()));
        case "AutoScalingConfigurationSummary":
            return Optional.ofNullable(clazz.cast(autoScalingConfigurationSummary()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Service, T> g) {
        return obj -> g.apply((Service) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Service> {
        /**
         * <p>
         * The customer-provided service name.
         * </p>
         * 
         * @param serviceName
         *        The customer-provided service name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceName(String serviceName);

        /**
         * <p>
         * An ID that App Runner generated for this service. It's unique within the Amazon Web Services Region.
         * </p>
         * 
         * @param serviceId
         *        An ID that App Runner generated for this service. It's unique within the Amazon Web Services Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceId(String serviceId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of this service.
         * </p>
         * 
         * @param serviceArn
         *        The Amazon Resource Name (ARN) of this service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceArn(String serviceArn);

        /**
         * <p>
         * A subdomain URL that App Runner generated for this service. You can use this URL to access your service web
         * application.
         * </p>
         * 
         * @param serviceUrl
         *        A subdomain URL that App Runner generated for this service. You can use this URL to access your
         *        service web application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceUrl(String serviceUrl);

        /**
         * <p>
         * The time when the App Runner service was created. It's in the Unix time stamp format.
         * </p>
         * 
         * @param createdAt
         *        The time when the App Runner service was created. It's in the Unix time stamp format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The time when the App Runner service was last updated at. It's in the Unix time stamp format.
         * </p>
         * 
         * @param updatedAt
         *        The time when the App Runner service was last updated at. It's in the Unix time stamp format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);

        /**
         * <p>
         * The time when the App Runner service was deleted. It's in the Unix time stamp format.
         * </p>
         * 
         * @param deletedAt
         *        The time when the App Runner service was deleted. It's in the Unix time stamp format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deletedAt(Instant deletedAt);

        /**
         * <p>
         * The current state of the App Runner service. These particular values mean the following.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATE_FAILED</code> – The service failed to create. To troubleshoot this failure, read the failure
         * events and logs, change any parameters that need to be fixed, and retry the call to create the service.
         * </p>
         * <p>
         * The failed service isn't usable, and still counts towards your service quota. When you're done analyzing the
         * failure, delete the service.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETE_FAILED</code> – The service failed to delete and can't be successfully recovered. Retry the
         * service deletion call to ensure that all related resources are removed.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The current state of the App Runner service. These particular values mean the following.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATE_FAILED</code> – The service failed to create. To troubleshoot this failure, read the
         *        failure events and logs, change any parameters that need to be fixed, and retry the call to create the
         *        service.
         *        </p>
         *        <p>
         *        The failed service isn't usable, and still counts towards your service quota. When you're done
         *        analyzing the failure, delete the service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETE_FAILED</code> – The service failed to delete and can't be successfully recovered. Retry
         *        the service deletion call to ensure that all related resources are removed.
         *        </p>
         *        </li>
         * @see ServiceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current state of the App Runner service. These particular values mean the following.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATE_FAILED</code> – The service failed to create. To troubleshoot this failure, read the failure
         * events and logs, change any parameters that need to be fixed, and retry the call to create the service.
         * </p>
         * <p>
         * The failed service isn't usable, and still counts towards your service quota. When you're done analyzing the
         * failure, delete the service.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETE_FAILED</code> – The service failed to delete and can't be successfully recovered. Retry the
         * service deletion call to ensure that all related resources are removed.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The current state of the App Runner service. These particular values mean the following.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATE_FAILED</code> – The service failed to create. To troubleshoot this failure, read the
         *        failure events and logs, change any parameters that need to be fixed, and retry the call to create the
         *        service.
         *        </p>
         *        <p>
         *        The failed service isn't usable, and still counts towards your service quota. When you're done
         *        analyzing the failure, delete the service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETE_FAILED</code> – The service failed to delete and can't be successfully recovered. Retry
         *        the service deletion call to ensure that all related resources are removed.
         *        </p>
         *        </li>
         * @see ServiceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceStatus
         */
        Builder status(ServiceStatus status);

        /**
         * <p>
         * The source deployed to the App Runner service. It can be a code or an image repository.
         * </p>
         * 
         * @param sourceConfiguration
         *        The source deployed to the App Runner service. It can be a code or an image repository.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceConfiguration(SourceConfiguration sourceConfiguration);

        /**
         * <p>
         * The source deployed to the App Runner service. It can be a code or an image repository.
         * </p>
         * This is a convenience that creates an instance of the {@link SourceConfiguration.Builder} avoiding the need
         * to create one manually via {@link SourceConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link SourceConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #sourceConfiguration(SourceConfiguration)}.
         * 
         * @param sourceConfiguration
         *        a consumer that will call methods on {@link SourceConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceConfiguration(SourceConfiguration)
         */
        default Builder sourceConfiguration(Consumer<SourceConfiguration.Builder> sourceConfiguration) {
            return sourceConfiguration(SourceConfiguration.builder().applyMutation(sourceConfiguration).build());
        }

        /**
         * <p>
         * The runtime configuration of instances (scaling units) of this service.
         * </p>
         * 
         * @param instanceConfiguration
         *        The runtime configuration of instances (scaling units) of this service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceConfiguration(InstanceConfiguration instanceConfiguration);

        /**
         * <p>
         * The runtime configuration of instances (scaling units) of this service.
         * </p>
         * This is a convenience that creates an instance of the {@link InstanceConfiguration.Builder} avoiding the need
         * to create one manually via {@link InstanceConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link InstanceConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #instanceConfiguration(InstanceConfiguration)}.
         * 
         * @param instanceConfiguration
         *        a consumer that will call methods on {@link InstanceConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceConfiguration(InstanceConfiguration)
         */
        default Builder instanceConfiguration(Consumer<InstanceConfiguration.Builder> instanceConfiguration) {
            return instanceConfiguration(InstanceConfiguration.builder().applyMutation(instanceConfiguration).build());
        }

        /**
         * <p>
         * The encryption key that App Runner uses to encrypt the service logs and the copy of the source repository
         * that App Runner maintains for the service. It can be either a customer-provided encryption key or an Amazon
         * Web Services managed CMK.
         * </p>
         * 
         * @param encryptionConfiguration
         *        The encryption key that App Runner uses to encrypt the service logs and the copy of the source
         *        repository that App Runner maintains for the service. It can be either a customer-provided encryption
         *        key or an Amazon Web Services managed CMK.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionConfiguration(EncryptionConfiguration encryptionConfiguration);

        /**
         * <p>
         * The encryption key that App Runner uses to encrypt the service logs and the copy of the source repository
         * that App Runner maintains for the service. It can be either a customer-provided encryption key or an Amazon
         * Web Services managed CMK.
         * </p>
         * This is a convenience that creates an instance of the {@link EncryptionConfiguration.Builder} avoiding the
         * need to create one manually via {@link EncryptionConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link EncryptionConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #encryptionConfiguration(EncryptionConfiguration)}.
         * 
         * @param encryptionConfiguration
         *        a consumer that will call methods on {@link EncryptionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionConfiguration(EncryptionConfiguration)
         */
        default Builder encryptionConfiguration(Consumer<EncryptionConfiguration.Builder> encryptionConfiguration) {
            return encryptionConfiguration(EncryptionConfiguration.builder().applyMutation(encryptionConfiguration).build());
        }

        /**
         * <p>
         * The settings for the health check that App Runner performs to monitor the health of this service.
         * </p>
         * 
         * @param healthCheckConfiguration
         *        The settings for the health check that App Runner performs to monitor the health of this service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthCheckConfiguration(HealthCheckConfiguration healthCheckConfiguration);

        /**
         * <p>
         * The settings for the health check that App Runner performs to monitor the health of this service.
         * </p>
         * This is a convenience that creates an instance of the {@link HealthCheckConfiguration.Builder} avoiding the
         * need to create one manually via {@link HealthCheckConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link HealthCheckConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #healthCheckConfiguration(HealthCheckConfiguration)}.
         * 
         * @param healthCheckConfiguration
         *        a consumer that will call methods on {@link HealthCheckConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #healthCheckConfiguration(HealthCheckConfiguration)
         */
        default Builder healthCheckConfiguration(Consumer<HealthCheckConfiguration.Builder> healthCheckConfiguration) {
            return healthCheckConfiguration(HealthCheckConfiguration.builder().applyMutation(healthCheckConfiguration).build());
        }

        /**
         * <p>
         * Summary information for the App Runner automatic scaling configuration resource that's associated with this
         * service.
         * </p>
         * 
         * @param autoScalingConfigurationSummary
         *        Summary information for the App Runner automatic scaling configuration resource that's associated with
         *        this service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingConfigurationSummary(AutoScalingConfigurationSummary autoScalingConfigurationSummary);

        /**
         * <p>
         * Summary information for the App Runner automatic scaling configuration resource that's associated with this
         * service.
         * </p>
         * This is a convenience that creates an instance of the {@link AutoScalingConfigurationSummary.Builder}
         * avoiding the need to create one manually via {@link AutoScalingConfigurationSummary#builder()}.
         *
         * When the {@link Consumer} completes, {@link AutoScalingConfigurationSummary.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #autoScalingConfigurationSummary(AutoScalingConfigurationSummary)}.
         * 
         * @param autoScalingConfigurationSummary
         *        a consumer that will call methods on {@link AutoScalingConfigurationSummary.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #autoScalingConfigurationSummary(AutoScalingConfigurationSummary)
         */
        default Builder autoScalingConfigurationSummary(
                Consumer<AutoScalingConfigurationSummary.Builder> autoScalingConfigurationSummary) {
            return autoScalingConfigurationSummary(AutoScalingConfigurationSummary.builder()
                    .applyMutation(autoScalingConfigurationSummary).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String serviceName;

        private String serviceId;

        private String serviceArn;

        private String serviceUrl;

        private Instant createdAt;

        private Instant updatedAt;

        private Instant deletedAt;

        private String status;

        private SourceConfiguration sourceConfiguration;

        private InstanceConfiguration instanceConfiguration;

        private EncryptionConfiguration encryptionConfiguration;

        private HealthCheckConfiguration healthCheckConfiguration;

        private AutoScalingConfigurationSummary autoScalingConfigurationSummary;

        private BuilderImpl() {
        }

        private BuilderImpl(Service model) {
            serviceName(model.serviceName);
            serviceId(model.serviceId);
            serviceArn(model.serviceArn);
            serviceUrl(model.serviceUrl);
            createdAt(model.createdAt);
            updatedAt(model.updatedAt);
            deletedAt(model.deletedAt);
            status(model.status);
            sourceConfiguration(model.sourceConfiguration);
            instanceConfiguration(model.instanceConfiguration);
            encryptionConfiguration(model.encryptionConfiguration);
            healthCheckConfiguration(model.healthCheckConfiguration);
            autoScalingConfigurationSummary(model.autoScalingConfigurationSummary);
        }

        public final String getServiceName() {
            return serviceName;
        }

        public final void setServiceName(String serviceName) {
            this.serviceName = serviceName;
        }

        @Override
        @Transient
        public final Builder serviceName(String serviceName) {
            this.serviceName = serviceName;
            return this;
        }

        public final String getServiceId() {
            return serviceId;
        }

        public final void setServiceId(String serviceId) {
            this.serviceId = serviceId;
        }

        @Override
        @Transient
        public final Builder serviceId(String serviceId) {
            this.serviceId = serviceId;
            return this;
        }

        public final String getServiceArn() {
            return serviceArn;
        }

        public final void setServiceArn(String serviceArn) {
            this.serviceArn = serviceArn;
        }

        @Override
        @Transient
        public final Builder serviceArn(String serviceArn) {
            this.serviceArn = serviceArn;
            return this;
        }

        public final String getServiceUrl() {
            return serviceUrl;
        }

        public final void setServiceUrl(String serviceUrl) {
            this.serviceUrl = serviceUrl;
        }

        @Override
        @Transient
        public final Builder serviceUrl(String serviceUrl) {
            this.serviceUrl = serviceUrl;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        @Transient
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        @Transient
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public final Instant getDeletedAt() {
            return deletedAt;
        }

        public final void setDeletedAt(Instant deletedAt) {
            this.deletedAt = deletedAt;
        }

        @Override
        @Transient
        public final Builder deletedAt(Instant deletedAt) {
            this.deletedAt = deletedAt;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        @Transient
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        @Transient
        public final Builder status(ServiceStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final SourceConfiguration.Builder getSourceConfiguration() {
            return sourceConfiguration != null ? sourceConfiguration.toBuilder() : null;
        }

        public final void setSourceConfiguration(SourceConfiguration.BuilderImpl sourceConfiguration) {
            this.sourceConfiguration = sourceConfiguration != null ? sourceConfiguration.build() : null;
        }

        @Override
        @Transient
        public final Builder sourceConfiguration(SourceConfiguration sourceConfiguration) {
            this.sourceConfiguration = sourceConfiguration;
            return this;
        }

        public final InstanceConfiguration.Builder getInstanceConfiguration() {
            return instanceConfiguration != null ? instanceConfiguration.toBuilder() : null;
        }

        public final void setInstanceConfiguration(InstanceConfiguration.BuilderImpl instanceConfiguration) {
            this.instanceConfiguration = instanceConfiguration != null ? instanceConfiguration.build() : null;
        }

        @Override
        @Transient
        public final Builder instanceConfiguration(InstanceConfiguration instanceConfiguration) {
            this.instanceConfiguration = instanceConfiguration;
            return this;
        }

        public final EncryptionConfiguration.Builder getEncryptionConfiguration() {
            return encryptionConfiguration != null ? encryptionConfiguration.toBuilder() : null;
        }

        public final void setEncryptionConfiguration(EncryptionConfiguration.BuilderImpl encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration != null ? encryptionConfiguration.build() : null;
        }

        @Override
        @Transient
        public final Builder encryptionConfiguration(EncryptionConfiguration encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration;
            return this;
        }

        public final HealthCheckConfiguration.Builder getHealthCheckConfiguration() {
            return healthCheckConfiguration != null ? healthCheckConfiguration.toBuilder() : null;
        }

        public final void setHealthCheckConfiguration(HealthCheckConfiguration.BuilderImpl healthCheckConfiguration) {
            this.healthCheckConfiguration = healthCheckConfiguration != null ? healthCheckConfiguration.build() : null;
        }

        @Override
        @Transient
        public final Builder healthCheckConfiguration(HealthCheckConfiguration healthCheckConfiguration) {
            this.healthCheckConfiguration = healthCheckConfiguration;
            return this;
        }

        public final AutoScalingConfigurationSummary.Builder getAutoScalingConfigurationSummary() {
            return autoScalingConfigurationSummary != null ? autoScalingConfigurationSummary.toBuilder() : null;
        }

        public final void setAutoScalingConfigurationSummary(
                AutoScalingConfigurationSummary.BuilderImpl autoScalingConfigurationSummary) {
            this.autoScalingConfigurationSummary = autoScalingConfigurationSummary != null ? autoScalingConfigurationSummary
                    .build() : null;
        }

        @Override
        @Transient
        public final Builder autoScalingConfigurationSummary(AutoScalingConfigurationSummary autoScalingConfigurationSummary) {
            this.autoScalingConfigurationSummary = autoScalingConfigurationSummary;
            return this;
        }

        @Override
        public Service build() {
            return new Service(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
