/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apprunner.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the configuration that App Runner uses to run an App Runner service using an image pulled from a source
 * image repository.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ImageConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<ImageConfiguration.Builder, ImageConfiguration> {
    private static final SdkField<Map<String, String>> RUNTIME_ENVIRONMENT_VARIABLES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("RuntimeEnvironmentVariables")
            .getter(getter(ImageConfiguration::runtimeEnvironmentVariables))
            .setter(setter(Builder::runtimeEnvironmentVariables))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuntimeEnvironmentVariables")
                    .build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> START_COMMAND_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StartCommand").getter(getter(ImageConfiguration::startCommand)).setter(setter(Builder::startCommand))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartCommand").build()).build();

    private static final SdkField<String> PORT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Port")
            .getter(getter(ImageConfiguration::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            RUNTIME_ENVIRONMENT_VARIABLES_FIELD, START_COMMAND_FIELD, PORT_FIELD));

    private static final long serialVersionUID = 1L;

    private final Map<String, String> runtimeEnvironmentVariables;

    private final String startCommand;

    private final String port;

    private ImageConfiguration(BuilderImpl builder) {
        this.runtimeEnvironmentVariables = builder.runtimeEnvironmentVariables;
        this.startCommand = builder.startCommand;
        this.port = builder.port;
    }

    /**
     * For responses, this returns true if the service returned a value for the RuntimeEnvironmentVariables property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasRuntimeEnvironmentVariables() {
        return runtimeEnvironmentVariables != null && !(runtimeEnvironmentVariables instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Environment variables that are available to your running App Runner service. An array of key-value pairs. Keys
     * with a prefix of <code>AWSAPPRUNNER</code> are reserved for system use and aren't valid.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRuntimeEnvironmentVariables} method.
     * </p>
     * 
     * @return Environment variables that are available to your running App Runner service. An array of key-value pairs.
     *         Keys with a prefix of <code>AWSAPPRUNNER</code> are reserved for system use and aren't valid.
     */
    public final Map<String, String> runtimeEnvironmentVariables() {
        return runtimeEnvironmentVariables;
    }

    /**
     * <p>
     * An optional command that App Runner runs to start the application in the source image. If specified, this command
     * overrides the Docker image’s default start command.
     * </p>
     * 
     * @return An optional command that App Runner runs to start the application in the source image. If specified, this
     *         command overrides the Docker image’s default start command.
     */
    public final String startCommand() {
        return startCommand;
    }

    /**
     * <p>
     * The port that your application listens to in the container.
     * </p>
     * <p>
     * Default: <code>8080</code>
     * </p>
     * 
     * @return The port that your application listens to in the container.</p>
     *         <p>
     *         Default: <code>8080</code>
     */
    public final String port() {
        return port;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasRuntimeEnvironmentVariables() ? runtimeEnvironmentVariables() : null);
        hashCode = 31 * hashCode + Objects.hashCode(startCommand());
        hashCode = 31 * hashCode + Objects.hashCode(port());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImageConfiguration)) {
            return false;
        }
        ImageConfiguration other = (ImageConfiguration) obj;
        return hasRuntimeEnvironmentVariables() == other.hasRuntimeEnvironmentVariables()
                && Objects.equals(runtimeEnvironmentVariables(), other.runtimeEnvironmentVariables())
                && Objects.equals(startCommand(), other.startCommand()) && Objects.equals(port(), other.port());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("ImageConfiguration")
                .add("RuntimeEnvironmentVariables",
                        runtimeEnvironmentVariables() == null ? null : "*** Sensitive Data Redacted ***")
                .add("StartCommand", startCommand()).add("Port", port()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RuntimeEnvironmentVariables":
            return Optional.ofNullable(clazz.cast(runtimeEnvironmentVariables()));
        case "StartCommand":
            return Optional.ofNullable(clazz.cast(startCommand()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ImageConfiguration, T> g) {
        return obj -> g.apply((ImageConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ImageConfiguration> {
        /**
         * <p>
         * Environment variables that are available to your running App Runner service. An array of key-value pairs.
         * Keys with a prefix of <code>AWSAPPRUNNER</code> are reserved for system use and aren't valid.
         * </p>
         * 
         * @param runtimeEnvironmentVariables
         *        Environment variables that are available to your running App Runner service. An array of key-value
         *        pairs. Keys with a prefix of <code>AWSAPPRUNNER</code> are reserved for system use and aren't valid.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder runtimeEnvironmentVariables(Map<String, String> runtimeEnvironmentVariables);

        /**
         * <p>
         * An optional command that App Runner runs to start the application in the source image. If specified, this
         * command overrides the Docker image’s default start command.
         * </p>
         * 
         * @param startCommand
         *        An optional command that App Runner runs to start the application in the source image. If specified,
         *        this command overrides the Docker image’s default start command.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startCommand(String startCommand);

        /**
         * <p>
         * The port that your application listens to in the container.
         * </p>
         * <p>
         * Default: <code>8080</code>
         * </p>
         * 
         * @param port
         *        The port that your application listens to in the container.</p>
         *        <p>
         *        Default: <code>8080</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(String port);
    }

    static final class BuilderImpl implements Builder {
        private Map<String, String> runtimeEnvironmentVariables = DefaultSdkAutoConstructMap.getInstance();

        private String startCommand;

        private String port;

        private BuilderImpl() {
        }

        private BuilderImpl(ImageConfiguration model) {
            runtimeEnvironmentVariables(model.runtimeEnvironmentVariables);
            startCommand(model.startCommand);
            port(model.port);
        }

        public final Map<String, String> getRuntimeEnvironmentVariables() {
            if (runtimeEnvironmentVariables instanceof SdkAutoConstructMap) {
                return null;
            }
            return runtimeEnvironmentVariables;
        }

        public final void setRuntimeEnvironmentVariables(Map<String, String> runtimeEnvironmentVariables) {
            this.runtimeEnvironmentVariables = RuntimeEnvironmentVariablesCopier.copy(runtimeEnvironmentVariables);
        }

        @Override
        @Transient
        public final Builder runtimeEnvironmentVariables(Map<String, String> runtimeEnvironmentVariables) {
            this.runtimeEnvironmentVariables = RuntimeEnvironmentVariablesCopier.copy(runtimeEnvironmentVariables);
            return this;
        }

        public final String getStartCommand() {
            return startCommand;
        }

        public final void setStartCommand(String startCommand) {
            this.startCommand = startCommand;
        }

        @Override
        @Transient
        public final Builder startCommand(String startCommand) {
            this.startCommand = startCommand;
            return this;
        }

        public final String getPort() {
            return port;
        }

        public final void setPort(String port) {
            this.port = port;
        }

        @Override
        @Transient
        public final Builder port(String port) {
            this.port = port;
            return this;
        }

        @Override
        public ImageConfiguration build() {
            return new ImageConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
