/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apprunner.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an App Runner VPC connector resource. A VPC connector describes the Amazon Virtual Private Cloud (Amazon
 * VPC) that an App Runner service is associated with, and the subnets and security group that are used.
 * </p>
 * <p>
 * Multiple revisions of a connector might have the same <code>Name</code> and different <code>Revision</code> values.
 * </p>
 * <note>
 * <p>
 * At this time, App Runner supports only one revision per name.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class VpcConnector implements SdkPojo, Serializable, ToCopyableBuilder<VpcConnector.Builder, VpcConnector> {
    private static final SdkField<String> VPC_CONNECTOR_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VpcConnectorName").getter(getter(VpcConnector::vpcConnectorName))
            .setter(setter(Builder::vpcConnectorName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcConnectorName").build()).build();

    private static final SdkField<String> VPC_CONNECTOR_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VpcConnectorArn").getter(getter(VpcConnector::vpcConnectorArn)).setter(setter(Builder::vpcConnectorArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcConnectorArn").build()).build();

    private static final SdkField<Integer> VPC_CONNECTOR_REVISION_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("VpcConnectorRevision").getter(getter(VpcConnector::vpcConnectorRevision))
            .setter(setter(Builder::vpcConnectorRevision))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcConnectorRevision").build())
            .build();

    private static final SdkField<List<String>> SUBNETS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Subnets")
            .getter(getter(VpcConnector::subnets))
            .setter(setter(Builder::subnets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subnets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SECURITY_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroups")
            .getter(getter(VpcConnector::securityGroups))
            .setter(setter(Builder::securityGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(VpcConnector::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt").getter(getter(VpcConnector::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAt").build()).build();

    private static final SdkField<Instant> DELETED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("DeletedAt").getter(getter(VpcConnector::deletedAt)).setter(setter(Builder::deletedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeletedAt").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VPC_CONNECTOR_NAME_FIELD,
            VPC_CONNECTOR_ARN_FIELD, VPC_CONNECTOR_REVISION_FIELD, SUBNETS_FIELD, SECURITY_GROUPS_FIELD, STATUS_FIELD,
            CREATED_AT_FIELD, DELETED_AT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String vpcConnectorName;

    private final String vpcConnectorArn;

    private final Integer vpcConnectorRevision;

    private final List<String> subnets;

    private final List<String> securityGroups;

    private final String status;

    private final Instant createdAt;

    private final Instant deletedAt;

    private VpcConnector(BuilderImpl builder) {
        this.vpcConnectorName = builder.vpcConnectorName;
        this.vpcConnectorArn = builder.vpcConnectorArn;
        this.vpcConnectorRevision = builder.vpcConnectorRevision;
        this.subnets = builder.subnets;
        this.securityGroups = builder.securityGroups;
        this.status = builder.status;
        this.createdAt = builder.createdAt;
        this.deletedAt = builder.deletedAt;
    }

    /**
     * <p>
     * The customer-provided VPC connector name.
     * </p>
     * 
     * @return The customer-provided VPC connector name.
     */
    public final String vpcConnectorName() {
        return vpcConnectorName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of this VPC connector.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of this VPC connector.
     */
    public final String vpcConnectorArn() {
        return vpcConnectorArn;
    }

    /**
     * <p>
     * The revision of this VPC connector. It's unique among all the active connectors (<code>"Status": "ACTIVE"</code>)
     * that share the same <code>Name</code>.
     * </p>
     * <note>
     * <p>
     * At this time, App Runner supports only one revision per name.
     * </p>
     * </note>
     * 
     * @return The revision of this VPC connector. It's unique among all the active connectors (
     *         <code>"Status": "ACTIVE"</code>) that share the same <code>Name</code>.</p> <note>
     *         <p>
     *         At this time, App Runner supports only one revision per name.
     *         </p>
     */
    public final Integer vpcConnectorRevision() {
        return vpcConnectorRevision;
    }

    /**
     * For responses, this returns true if the service returned a value for the Subnets property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSubnets() {
        return subnets != null && !(subnets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of IDs of subnets that App Runner uses for your service. All IDs are of subnets of a single Amazon VPC.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSubnets} method.
     * </p>
     * 
     * @return A list of IDs of subnets that App Runner uses for your service. All IDs are of subnets of a single Amazon
     *         VPC.
     */
    public final List<String> subnets() {
        return subnets;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroups property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroups() {
        return securityGroups != null && !(securityGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of IDs of security groups that App Runner uses for access to Amazon Web Services resources under the
     * specified subnets. If not specified, App Runner uses the default security group of the Amazon VPC. The default
     * security group allows all outbound traffic.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroups} method.
     * </p>
     * 
     * @return A list of IDs of security groups that App Runner uses for access to Amazon Web Services resources under
     *         the specified subnets. If not specified, App Runner uses the default security group of the Amazon VPC.
     *         The default security group allows all outbound traffic.
     */
    public final List<String> securityGroups() {
        return securityGroups;
    }

    /**
     * <p>
     * The current state of the VPC connector. If the status of a connector revision is <code>INACTIVE</code>, it was
     * deleted and can't be used. Inactive connector revisions are permanently removed some time after they are deleted.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link VpcConnectorStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current state of the VPC connector. If the status of a connector revision is <code>INACTIVE</code>,
     *         it was deleted and can't be used. Inactive connector revisions are permanently removed some time after
     *         they are deleted.
     * @see VpcConnectorStatus
     */
    public final VpcConnectorStatus status() {
        return VpcConnectorStatus.fromValue(status);
    }

    /**
     * <p>
     * The current state of the VPC connector. If the status of a connector revision is <code>INACTIVE</code>, it was
     * deleted and can't be used. Inactive connector revisions are permanently removed some time after they are deleted.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link VpcConnectorStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current state of the VPC connector. If the status of a connector revision is <code>INACTIVE</code>,
     *         it was deleted and can't be used. Inactive connector revisions are permanently removed some time after
     *         they are deleted.
     * @see VpcConnectorStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The time when the VPC connector was created. It's in Unix time stamp format.
     * </p>
     * 
     * @return The time when the VPC connector was created. It's in Unix time stamp format.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The time when the VPC connector was deleted. It's in Unix time stamp format.
     * </p>
     * 
     * @return The time when the VPC connector was deleted. It's in Unix time stamp format.
     */
    public final Instant deletedAt() {
        return deletedAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(vpcConnectorName());
        hashCode = 31 * hashCode + Objects.hashCode(vpcConnectorArn());
        hashCode = 31 * hashCode + Objects.hashCode(vpcConnectorRevision());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnets() ? subnets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroups() ? securityGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(deletedAt());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof VpcConnector)) {
            return false;
        }
        VpcConnector other = (VpcConnector) obj;
        return Objects.equals(vpcConnectorName(), other.vpcConnectorName())
                && Objects.equals(vpcConnectorArn(), other.vpcConnectorArn())
                && Objects.equals(vpcConnectorRevision(), other.vpcConnectorRevision()) && hasSubnets() == other.hasSubnets()
                && Objects.equals(subnets(), other.subnets()) && hasSecurityGroups() == other.hasSecurityGroups()
                && Objects.equals(securityGroups(), other.securityGroups())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(deletedAt(), other.deletedAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("VpcConnector").add("VpcConnectorName", vpcConnectorName())
                .add("VpcConnectorArn", vpcConnectorArn()).add("VpcConnectorRevision", vpcConnectorRevision())
                .add("Subnets", hasSubnets() ? subnets() : null)
                .add("SecurityGroups", hasSecurityGroups() ? securityGroups() : null).add("Status", statusAsString())
                .add("CreatedAt", createdAt()).add("DeletedAt", deletedAt()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VpcConnectorName":
            return Optional.ofNullable(clazz.cast(vpcConnectorName()));
        case "VpcConnectorArn":
            return Optional.ofNullable(clazz.cast(vpcConnectorArn()));
        case "VpcConnectorRevision":
            return Optional.ofNullable(clazz.cast(vpcConnectorRevision()));
        case "Subnets":
            return Optional.ofNullable(clazz.cast(subnets()));
        case "SecurityGroups":
            return Optional.ofNullable(clazz.cast(securityGroups()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "DeletedAt":
            return Optional.ofNullable(clazz.cast(deletedAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("VpcConnectorName", VPC_CONNECTOR_NAME_FIELD);
        map.put("VpcConnectorArn", VPC_CONNECTOR_ARN_FIELD);
        map.put("VpcConnectorRevision", VPC_CONNECTOR_REVISION_FIELD);
        map.put("Subnets", SUBNETS_FIELD);
        map.put("SecurityGroups", SECURITY_GROUPS_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("CreatedAt", CREATED_AT_FIELD);
        map.put("DeletedAt", DELETED_AT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<VpcConnector, T> g) {
        return obj -> g.apply((VpcConnector) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, VpcConnector> {
        /**
         * <p>
         * The customer-provided VPC connector name.
         * </p>
         * 
         * @param vpcConnectorName
         *        The customer-provided VPC connector name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConnectorName(String vpcConnectorName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of this VPC connector.
         * </p>
         * 
         * @param vpcConnectorArn
         *        The Amazon Resource Name (ARN) of this VPC connector.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConnectorArn(String vpcConnectorArn);

        /**
         * <p>
         * The revision of this VPC connector. It's unique among all the active connectors (
         * <code>"Status": "ACTIVE"</code>) that share the same <code>Name</code>.
         * </p>
         * <note>
         * <p>
         * At this time, App Runner supports only one revision per name.
         * </p>
         * </note>
         * 
         * @param vpcConnectorRevision
         *        The revision of this VPC connector. It's unique among all the active connectors (
         *        <code>"Status": "ACTIVE"</code>) that share the same <code>Name</code>.</p> <note>
         *        <p>
         *        At this time, App Runner supports only one revision per name.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConnectorRevision(Integer vpcConnectorRevision);

        /**
         * <p>
         * A list of IDs of subnets that App Runner uses for your service. All IDs are of subnets of a single Amazon
         * VPC.
         * </p>
         * 
         * @param subnets
         *        A list of IDs of subnets that App Runner uses for your service. All IDs are of subnets of a single
         *        Amazon VPC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnets(Collection<String> subnets);

        /**
         * <p>
         * A list of IDs of subnets that App Runner uses for your service. All IDs are of subnets of a single Amazon
         * VPC.
         * </p>
         * 
         * @param subnets
         *        A list of IDs of subnets that App Runner uses for your service. All IDs are of subnets of a single
         *        Amazon VPC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnets(String... subnets);

        /**
         * <p>
         * A list of IDs of security groups that App Runner uses for access to Amazon Web Services resources under the
         * specified subnets. If not specified, App Runner uses the default security group of the Amazon VPC. The
         * default security group allows all outbound traffic.
         * </p>
         * 
         * @param securityGroups
         *        A list of IDs of security groups that App Runner uses for access to Amazon Web Services resources
         *        under the specified subnets. If not specified, App Runner uses the default security group of the
         *        Amazon VPC. The default security group allows all outbound traffic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(Collection<String> securityGroups);

        /**
         * <p>
         * A list of IDs of security groups that App Runner uses for access to Amazon Web Services resources under the
         * specified subnets. If not specified, App Runner uses the default security group of the Amazon VPC. The
         * default security group allows all outbound traffic.
         * </p>
         * 
         * @param securityGroups
         *        A list of IDs of security groups that App Runner uses for access to Amazon Web Services resources
         *        under the specified subnets. If not specified, App Runner uses the default security group of the
         *        Amazon VPC. The default security group allows all outbound traffic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(String... securityGroups);

        /**
         * <p>
         * The current state of the VPC connector. If the status of a connector revision is <code>INACTIVE</code>, it
         * was deleted and can't be used. Inactive connector revisions are permanently removed some time after they are
         * deleted.
         * </p>
         * 
         * @param status
         *        The current state of the VPC connector. If the status of a connector revision is <code>INACTIVE</code>
         *        , it was deleted and can't be used. Inactive connector revisions are permanently removed some time
         *        after they are deleted.
         * @see VpcConnectorStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcConnectorStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current state of the VPC connector. If the status of a connector revision is <code>INACTIVE</code>, it
         * was deleted and can't be used. Inactive connector revisions are permanently removed some time after they are
         * deleted.
         * </p>
         * 
         * @param status
         *        The current state of the VPC connector. If the status of a connector revision is <code>INACTIVE</code>
         *        , it was deleted and can't be used. Inactive connector revisions are permanently removed some time
         *        after they are deleted.
         * @see VpcConnectorStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcConnectorStatus
         */
        Builder status(VpcConnectorStatus status);

        /**
         * <p>
         * The time when the VPC connector was created. It's in Unix time stamp format.
         * </p>
         * 
         * @param createdAt
         *        The time when the VPC connector was created. It's in Unix time stamp format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The time when the VPC connector was deleted. It's in Unix time stamp format.
         * </p>
         * 
         * @param deletedAt
         *        The time when the VPC connector was deleted. It's in Unix time stamp format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deletedAt(Instant deletedAt);
    }

    static final class BuilderImpl implements Builder {
        private String vpcConnectorName;

        private String vpcConnectorArn;

        private Integer vpcConnectorRevision;

        private List<String> subnets = DefaultSdkAutoConstructList.getInstance();

        private List<String> securityGroups = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private Instant createdAt;

        private Instant deletedAt;

        private BuilderImpl() {
        }

        private BuilderImpl(VpcConnector model) {
            vpcConnectorName(model.vpcConnectorName);
            vpcConnectorArn(model.vpcConnectorArn);
            vpcConnectorRevision(model.vpcConnectorRevision);
            subnets(model.subnets);
            securityGroups(model.securityGroups);
            status(model.status);
            createdAt(model.createdAt);
            deletedAt(model.deletedAt);
        }

        public final String getVpcConnectorName() {
            return vpcConnectorName;
        }

        public final void setVpcConnectorName(String vpcConnectorName) {
            this.vpcConnectorName = vpcConnectorName;
        }

        @Override
        public final Builder vpcConnectorName(String vpcConnectorName) {
            this.vpcConnectorName = vpcConnectorName;
            return this;
        }

        public final String getVpcConnectorArn() {
            return vpcConnectorArn;
        }

        public final void setVpcConnectorArn(String vpcConnectorArn) {
            this.vpcConnectorArn = vpcConnectorArn;
        }

        @Override
        public final Builder vpcConnectorArn(String vpcConnectorArn) {
            this.vpcConnectorArn = vpcConnectorArn;
            return this;
        }

        public final Integer getVpcConnectorRevision() {
            return vpcConnectorRevision;
        }

        public final void setVpcConnectorRevision(Integer vpcConnectorRevision) {
            this.vpcConnectorRevision = vpcConnectorRevision;
        }

        @Override
        public final Builder vpcConnectorRevision(Integer vpcConnectorRevision) {
            this.vpcConnectorRevision = vpcConnectorRevision;
            return this;
        }

        public final Collection<String> getSubnets() {
            if (subnets instanceof SdkAutoConstructList) {
                return null;
            }
            return subnets;
        }

        public final void setSubnets(Collection<String> subnets) {
            this.subnets = StringListCopier.copy(subnets);
        }

        @Override
        public final Builder subnets(Collection<String> subnets) {
            this.subnets = StringListCopier.copy(subnets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnets(String... subnets) {
            subnets(Arrays.asList(subnets));
            return this;
        }

        public final Collection<String> getSecurityGroups() {
            if (securityGroups instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroups;
        }

        public final void setSecurityGroups(Collection<String> securityGroups) {
            this.securityGroups = StringListCopier.copy(securityGroups);
        }

        @Override
        public final Builder securityGroups(Collection<String> securityGroups) {
            this.securityGroups = StringListCopier.copy(securityGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroups(String... securityGroups) {
            securityGroups(Arrays.asList(securityGroups));
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(VpcConnectorStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getDeletedAt() {
            return deletedAt;
        }

        public final void setDeletedAt(Instant deletedAt) {
            this.deletedAt = deletedAt;
        }

        @Override
        public final Builder deletedAt(Instant deletedAt) {
            this.deletedAt = deletedAt;
            return this;
        }

        @Override
        public VpcConnector build() {
            return new VpcConnector(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
