/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appstream.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a virtual machine that is used to create an image.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ImageBuilder implements SdkPojo, Serializable, ToCopyableBuilder<ImageBuilder.Builder, ImageBuilder> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ImageBuilder::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(ImageBuilder::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> IMAGE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ImageArn").getter(getter(ImageBuilder::imageArn)).setter(setter(Builder::imageArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageArn").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(ImageBuilder::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> DISPLAY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DisplayName").getter(getter(ImageBuilder::displayName)).setter(setter(Builder::displayName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisplayName").build()).build();

    private static final SdkField<VpcConfig> VPC_CONFIG_FIELD = SdkField.<VpcConfig> builder(MarshallingType.SDK_POJO)
            .memberName("VpcConfig").getter(getter(ImageBuilder::vpcConfig)).setter(setter(Builder::vpcConfig))
            .constructor(VpcConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcConfig").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceType").getter(getter(ImageBuilder::instanceType)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<String> PLATFORM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Platform").getter(getter(ImageBuilder::platformAsString)).setter(setter(Builder::platform))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Platform").build()).build();

    private static final SdkField<String> IAM_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IamRoleArn").getter(getter(ImageBuilder::iamRoleArn)).setter(setter(Builder::iamRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IamRoleArn").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(ImageBuilder::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<ImageBuilderStateChangeReason> STATE_CHANGE_REASON_FIELD = SdkField
            .<ImageBuilderStateChangeReason> builder(MarshallingType.SDK_POJO).memberName("StateChangeReason")
            .getter(getter(ImageBuilder::stateChangeReason)).setter(setter(Builder::stateChangeReason))
            .constructor(ImageBuilderStateChangeReason::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateChangeReason").build()).build();

    private static final SdkField<Instant> CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTime").getter(getter(ImageBuilder::createdTime)).setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTime").build()).build();

    private static final SdkField<Boolean> ENABLE_DEFAULT_INTERNET_ACCESS_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EnableDefaultInternetAccess")
            .getter(getter(ImageBuilder::enableDefaultInternetAccess))
            .setter(setter(Builder::enableDefaultInternetAccess))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableDefaultInternetAccess")
                    .build()).build();

    private static final SdkField<DomainJoinInfo> DOMAIN_JOIN_INFO_FIELD = SdkField
            .<DomainJoinInfo> builder(MarshallingType.SDK_POJO).memberName("DomainJoinInfo")
            .getter(getter(ImageBuilder::domainJoinInfo)).setter(setter(Builder::domainJoinInfo))
            .constructor(DomainJoinInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainJoinInfo").build()).build();

    private static final SdkField<NetworkAccessConfiguration> NETWORK_ACCESS_CONFIGURATION_FIELD = SdkField
            .<NetworkAccessConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("NetworkAccessConfiguration")
            .getter(getter(ImageBuilder::networkAccessConfiguration))
            .setter(setter(Builder::networkAccessConfiguration))
            .constructor(NetworkAccessConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkAccessConfiguration").build())
            .build();

    private static final SdkField<List<ResourceError>> IMAGE_BUILDER_ERRORS_FIELD = SdkField
            .<List<ResourceError>> builder(MarshallingType.LIST)
            .memberName("ImageBuilderErrors")
            .getter(getter(ImageBuilder::imageBuilderErrors))
            .setter(setter(Builder::imageBuilderErrors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageBuilderErrors").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceError> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceError::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> APPSTREAM_AGENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AppstreamAgentVersion").getter(getter(ImageBuilder::appstreamAgentVersion))
            .setter(setter(Builder::appstreamAgentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AppstreamAgentVersion").build())
            .build();

    private static final SdkField<List<AccessEndpoint>> ACCESS_ENDPOINTS_FIELD = SdkField
            .<List<AccessEndpoint>> builder(MarshallingType.LIST)
            .memberName("AccessEndpoints")
            .getter(getter(ImageBuilder::accessEndpoints))
            .setter(setter(Builder::accessEndpoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessEndpoints").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AccessEndpoint> builder(MarshallingType.SDK_POJO)
                                            .constructor(AccessEndpoint::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> LATEST_APPSTREAM_AGENT_VERSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LatestAppstreamAgentVersion")
            .getter(getter(ImageBuilder::latestAppstreamAgentVersionAsString))
            .setter(setter(Builder::latestAppstreamAgentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LatestAppstreamAgentVersion")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD,
            IMAGE_ARN_FIELD, DESCRIPTION_FIELD, DISPLAY_NAME_FIELD, VPC_CONFIG_FIELD, INSTANCE_TYPE_FIELD, PLATFORM_FIELD,
            IAM_ROLE_ARN_FIELD, STATE_FIELD, STATE_CHANGE_REASON_FIELD, CREATED_TIME_FIELD, ENABLE_DEFAULT_INTERNET_ACCESS_FIELD,
            DOMAIN_JOIN_INFO_FIELD, NETWORK_ACCESS_CONFIGURATION_FIELD, IMAGE_BUILDER_ERRORS_FIELD,
            APPSTREAM_AGENT_VERSION_FIELD, ACCESS_ENDPOINTS_FIELD, LATEST_APPSTREAM_AGENT_VERSION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final String imageArn;

    private final String description;

    private final String displayName;

    private final VpcConfig vpcConfig;

    private final String instanceType;

    private final String platform;

    private final String iamRoleArn;

    private final String state;

    private final ImageBuilderStateChangeReason stateChangeReason;

    private final Instant createdTime;

    private final Boolean enableDefaultInternetAccess;

    private final DomainJoinInfo domainJoinInfo;

    private final NetworkAccessConfiguration networkAccessConfiguration;

    private final List<ResourceError> imageBuilderErrors;

    private final String appstreamAgentVersion;

    private final List<AccessEndpoint> accessEndpoints;

    private final String latestAppstreamAgentVersion;

    private ImageBuilder(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.imageArn = builder.imageArn;
        this.description = builder.description;
        this.displayName = builder.displayName;
        this.vpcConfig = builder.vpcConfig;
        this.instanceType = builder.instanceType;
        this.platform = builder.platform;
        this.iamRoleArn = builder.iamRoleArn;
        this.state = builder.state;
        this.stateChangeReason = builder.stateChangeReason;
        this.createdTime = builder.createdTime;
        this.enableDefaultInternetAccess = builder.enableDefaultInternetAccess;
        this.domainJoinInfo = builder.domainJoinInfo;
        this.networkAccessConfiguration = builder.networkAccessConfiguration;
        this.imageBuilderErrors = builder.imageBuilderErrors;
        this.appstreamAgentVersion = builder.appstreamAgentVersion;
        this.accessEndpoints = builder.accessEndpoints;
        this.latestAppstreamAgentVersion = builder.latestAppstreamAgentVersion;
    }

    /**
     * <p>
     * The name of the image builder.
     * </p>
     * 
     * @return The name of the image builder.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The ARN for the image builder.
     * </p>
     * 
     * @return The ARN for the image builder.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The ARN of the image from which this builder was created.
     * </p>
     * 
     * @return The ARN of the image from which this builder was created.
     */
    public final String imageArn() {
        return imageArn;
    }

    /**
     * <p>
     * The description to display.
     * </p>
     * 
     * @return The description to display.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The image builder name to display.
     * </p>
     * 
     * @return The image builder name to display.
     */
    public final String displayName() {
        return displayName;
    }

    /**
     * <p>
     * The VPC configuration of the image builder.
     * </p>
     * 
     * @return The VPC configuration of the image builder.
     */
    public final VpcConfig vpcConfig() {
        return vpcConfig;
    }

    /**
     * <p>
     * The instance type for the image builder. The following instance types are available:
     * </p>
     * <ul>
     * <li>
     * <p>
     * stream.standard.small
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.standard.medium
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.standard.large
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.compute.large
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.compute.xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.compute.2xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.compute.4xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.compute.8xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.memory.large
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.memory.xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.memory.2xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.memory.4xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.memory.8xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.memory.z1d.large
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.memory.z1d.xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.memory.z1d.2xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.memory.z1d.3xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.memory.z1d.6xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.memory.z1d.12xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.graphics-design.large
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.graphics-design.xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.graphics-design.2xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.graphics-design.4xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.graphics-desktop.2xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.graphics.g4dn.xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.graphics.g4dn.2xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.graphics.g4dn.4xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.graphics.g4dn.8xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.graphics.g4dn.12xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.graphics.g4dn.16xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.graphics-pro.4xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.graphics-pro.8xlarge
     * </p>
     * </li>
     * <li>
     * <p>
     * stream.graphics-pro.16xlarge
     * </p>
     * </li>
     * </ul>
     * 
     * @return The instance type for the image builder. The following instance types are available:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         stream.standard.small
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.standard.medium
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.standard.large
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.compute.large
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.compute.xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.compute.2xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.compute.4xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.compute.8xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.memory.large
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.memory.xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.memory.2xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.memory.4xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.memory.8xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.memory.z1d.large
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.memory.z1d.xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.memory.z1d.2xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.memory.z1d.3xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.memory.z1d.6xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.memory.z1d.12xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.graphics-design.large
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.graphics-design.xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.graphics-design.2xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.graphics-design.4xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.graphics-desktop.2xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.graphics.g4dn.xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.graphics.g4dn.2xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.graphics.g4dn.4xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.graphics.g4dn.8xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.graphics.g4dn.12xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.graphics.g4dn.16xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.graphics-pro.4xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.graphics-pro.8xlarge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         stream.graphics-pro.16xlarge
     *         </p>
     *         </li>
     */
    public final String instanceType() {
        return instanceType;
    }

    /**
     * <p>
     * The operating system platform of the image builder.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #platform} will
     * return {@link PlatformType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #platformAsString}.
     * </p>
     * 
     * @return The operating system platform of the image builder.
     * @see PlatformType
     */
    public final PlatformType platform() {
        return PlatformType.fromValue(platform);
    }

    /**
     * <p>
     * The operating system platform of the image builder.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #platform} will
     * return {@link PlatformType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #platformAsString}.
     * </p>
     * 
     * @return The operating system platform of the image builder.
     * @see PlatformType
     */
    public final String platformAsString() {
        return platform;
    }

    /**
     * <p>
     * The ARN of the IAM role that is applied to the image builder. To assume a role, the image builder calls the AWS
     * Security Token Service (STS) <code>AssumeRole</code> API operation and passes the ARN of the role to use. The
     * operation creates a new session with temporary credentials. AppStream 2.0 retrieves the temporary credentials and
     * creates the <b>appstream_machine_role</b> credential profile on the instance.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/appstream2/latest/developerguide/using-iam-roles-to-grant-permissions-to-applications-scripts-streaming-instances.html"
     * >Using an IAM Role to Grant Permissions to Applications and Scripts Running on AppStream 2.0 Streaming
     * Instances</a> in the <i>Amazon AppStream 2.0 Administration Guide</i>.
     * </p>
     * 
     * @return The ARN of the IAM role that is applied to the image builder. To assume a role, the image builder calls
     *         the AWS Security Token Service (STS) <code>AssumeRole</code> API operation and passes the ARN of the role
     *         to use. The operation creates a new session with temporary credentials. AppStream 2.0 retrieves the
     *         temporary credentials and creates the <b>appstream_machine_role</b> credential profile on the
     *         instance.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/appstream2/latest/developerguide/using-iam-roles-to-grant-permissions-to-applications-scripts-streaming-instances.html"
     *         >Using an IAM Role to Grant Permissions to Applications and Scripts Running on AppStream 2.0 Streaming
     *         Instances</a> in the <i>Amazon AppStream 2.0 Administration Guide</i>.
     */
    public final String iamRoleArn() {
        return iamRoleArn;
    }

    /**
     * <p>
     * The state of the image builder.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link ImageBuilderState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the image builder.
     * @see ImageBuilderState
     */
    public final ImageBuilderState state() {
        return ImageBuilderState.fromValue(state);
    }

    /**
     * <p>
     * The state of the image builder.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link ImageBuilderState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the image builder.
     * @see ImageBuilderState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The reason why the last state change occurred.
     * </p>
     * 
     * @return The reason why the last state change occurred.
     */
    public final ImageBuilderStateChangeReason stateChangeReason() {
        return stateChangeReason;
    }

    /**
     * <p>
     * The time stamp when the image builder was created.
     * </p>
     * 
     * @return The time stamp when the image builder was created.
     */
    public final Instant createdTime() {
        return createdTime;
    }

    /**
     * <p>
     * Enables or disables default internet access for the image builder.
     * </p>
     * 
     * @return Enables or disables default internet access for the image builder.
     */
    public final Boolean enableDefaultInternetAccess() {
        return enableDefaultInternetAccess;
    }

    /**
     * <p>
     * The name of the directory and organizational unit (OU) to use to join the image builder to a Microsoft Active
     * Directory domain.
     * </p>
     * 
     * @return The name of the directory and organizational unit (OU) to use to join the image builder to a Microsoft
     *         Active Directory domain.
     */
    public final DomainJoinInfo domainJoinInfo() {
        return domainJoinInfo;
    }

    /**
     * Returns the value of the NetworkAccessConfiguration property for this object.
     * 
     * @return The value of the NetworkAccessConfiguration property for this object.
     */
    public final NetworkAccessConfiguration networkAccessConfiguration() {
        return networkAccessConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the ImageBuilderErrors property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasImageBuilderErrors() {
        return imageBuilderErrors != null && !(imageBuilderErrors instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The image builder errors.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasImageBuilderErrors} method.
     * </p>
     * 
     * @return The image builder errors.
     */
    public final List<ResourceError> imageBuilderErrors() {
        return imageBuilderErrors;
    }

    /**
     * <p>
     * The version of the AppStream 2.0 agent that is currently being used by the image builder.
     * </p>
     * 
     * @return The version of the AppStream 2.0 agent that is currently being used by the image builder.
     */
    public final String appstreamAgentVersion() {
        return appstreamAgentVersion;
    }

    /**
     * For responses, this returns true if the service returned a value for the AccessEndpoints property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAccessEndpoints() {
        return accessEndpoints != null && !(accessEndpoints instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of virtual private cloud (VPC) interface endpoint objects. Administrators can connect to the image
     * builder only through the specified endpoints.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAccessEndpoints} method.
     * </p>
     * 
     * @return The list of virtual private cloud (VPC) interface endpoint objects. Administrators can connect to the
     *         image builder only through the specified endpoints.
     */
    public final List<AccessEndpoint> accessEndpoints() {
        return accessEndpoints;
    }

    /**
     * <p>
     * Indicates whether the image builder is using the latest AppStream 2.0 agent version or not.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #latestAppstreamAgentVersion} will return {@link LatestAppstreamAgentVersion#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #latestAppstreamAgentVersionAsString}.
     * </p>
     * 
     * @return Indicates whether the image builder is using the latest AppStream 2.0 agent version or not.
     * @see LatestAppstreamAgentVersion
     */
    public final LatestAppstreamAgentVersion latestAppstreamAgentVersion() {
        return LatestAppstreamAgentVersion.fromValue(latestAppstreamAgentVersion);
    }

    /**
     * <p>
     * Indicates whether the image builder is using the latest AppStream 2.0 agent version or not.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #latestAppstreamAgentVersion} will return {@link LatestAppstreamAgentVersion#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #latestAppstreamAgentVersionAsString}.
     * </p>
     * 
     * @return Indicates whether the image builder is using the latest AppStream 2.0 agent version or not.
     * @see LatestAppstreamAgentVersion
     */
    public final String latestAppstreamAgentVersionAsString() {
        return latestAppstreamAgentVersion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(imageArn());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(displayName());
        hashCode = 31 * hashCode + Objects.hashCode(vpcConfig());
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(platformAsString());
        hashCode = 31 * hashCode + Objects.hashCode(iamRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateChangeReason());
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        hashCode = 31 * hashCode + Objects.hashCode(enableDefaultInternetAccess());
        hashCode = 31 * hashCode + Objects.hashCode(domainJoinInfo());
        hashCode = 31 * hashCode + Objects.hashCode(networkAccessConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasImageBuilderErrors() ? imageBuilderErrors() : null);
        hashCode = 31 * hashCode + Objects.hashCode(appstreamAgentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hasAccessEndpoints() ? accessEndpoints() : null);
        hashCode = 31 * hashCode + Objects.hashCode(latestAppstreamAgentVersionAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImageBuilder)) {
            return false;
        }
        ImageBuilder other = (ImageBuilder) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(imageArn(), other.imageArn()) && Objects.equals(description(), other.description())
                && Objects.equals(displayName(), other.displayName()) && Objects.equals(vpcConfig(), other.vpcConfig())
                && Objects.equals(instanceType(), other.instanceType())
                && Objects.equals(platformAsString(), other.platformAsString())
                && Objects.equals(iamRoleArn(), other.iamRoleArn()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(stateChangeReason(), other.stateChangeReason())
                && Objects.equals(createdTime(), other.createdTime())
                && Objects.equals(enableDefaultInternetAccess(), other.enableDefaultInternetAccess())
                && Objects.equals(domainJoinInfo(), other.domainJoinInfo())
                && Objects.equals(networkAccessConfiguration(), other.networkAccessConfiguration())
                && hasImageBuilderErrors() == other.hasImageBuilderErrors()
                && Objects.equals(imageBuilderErrors(), other.imageBuilderErrors())
                && Objects.equals(appstreamAgentVersion(), other.appstreamAgentVersion())
                && hasAccessEndpoints() == other.hasAccessEndpoints()
                && Objects.equals(accessEndpoints(), other.accessEndpoints())
                && Objects.equals(latestAppstreamAgentVersionAsString(), other.latestAppstreamAgentVersionAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ImageBuilder").add("Name", name()).add("Arn", arn()).add("ImageArn", imageArn())
                .add("Description", description()).add("DisplayName", displayName()).add("VpcConfig", vpcConfig())
                .add("InstanceType", instanceType()).add("Platform", platformAsString()).add("IamRoleArn", iamRoleArn())
                .add("State", stateAsString()).add("StateChangeReason", stateChangeReason()).add("CreatedTime", createdTime())
                .add("EnableDefaultInternetAccess", enableDefaultInternetAccess()).add("DomainJoinInfo", domainJoinInfo())
                .add("NetworkAccessConfiguration", networkAccessConfiguration())
                .add("ImageBuilderErrors", hasImageBuilderErrors() ? imageBuilderErrors() : null)
                .add("AppstreamAgentVersion", appstreamAgentVersion())
                .add("AccessEndpoints", hasAccessEndpoints() ? accessEndpoints() : null)
                .add("LatestAppstreamAgentVersion", latestAppstreamAgentVersionAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "ImageArn":
            return Optional.ofNullable(clazz.cast(imageArn()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "DisplayName":
            return Optional.ofNullable(clazz.cast(displayName()));
        case "VpcConfig":
            return Optional.ofNullable(clazz.cast(vpcConfig()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "Platform":
            return Optional.ofNullable(clazz.cast(platformAsString()));
        case "IamRoleArn":
            return Optional.ofNullable(clazz.cast(iamRoleArn()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "StateChangeReason":
            return Optional.ofNullable(clazz.cast(stateChangeReason()));
        case "CreatedTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        case "EnableDefaultInternetAccess":
            return Optional.ofNullable(clazz.cast(enableDefaultInternetAccess()));
        case "DomainJoinInfo":
            return Optional.ofNullable(clazz.cast(domainJoinInfo()));
        case "NetworkAccessConfiguration":
            return Optional.ofNullable(clazz.cast(networkAccessConfiguration()));
        case "ImageBuilderErrors":
            return Optional.ofNullable(clazz.cast(imageBuilderErrors()));
        case "AppstreamAgentVersion":
            return Optional.ofNullable(clazz.cast(appstreamAgentVersion()));
        case "AccessEndpoints":
            return Optional.ofNullable(clazz.cast(accessEndpoints()));
        case "LatestAppstreamAgentVersion":
            return Optional.ofNullable(clazz.cast(latestAppstreamAgentVersionAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ImageBuilder, T> g) {
        return obj -> g.apply((ImageBuilder) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ImageBuilder> {
        /**
         * <p>
         * The name of the image builder.
         * </p>
         * 
         * @param name
         *        The name of the image builder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ARN for the image builder.
         * </p>
         * 
         * @param arn
         *        The ARN for the image builder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The ARN of the image from which this builder was created.
         * </p>
         * 
         * @param imageArn
         *        The ARN of the image from which this builder was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageArn(String imageArn);

        /**
         * <p>
         * The description to display.
         * </p>
         * 
         * @param description
         *        The description to display.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The image builder name to display.
         * </p>
         * 
         * @param displayName
         *        The image builder name to display.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder displayName(String displayName);

        /**
         * <p>
         * The VPC configuration of the image builder.
         * </p>
         * 
         * @param vpcConfig
         *        The VPC configuration of the image builder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConfig(VpcConfig vpcConfig);

        /**
         * <p>
         * The VPC configuration of the image builder.
         * </p>
         * This is a convenience method that creates an instance of the {@link VpcConfig.Builder} avoiding the need to
         * create one manually via {@link VpcConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VpcConfig.Builder#build()} is called immediately and its result
         * is passed to {@link #vpcConfig(VpcConfig)}.
         * 
         * @param vpcConfig
         *        a consumer that will call methods on {@link VpcConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcConfig(VpcConfig)
         */
        default Builder vpcConfig(Consumer<VpcConfig.Builder> vpcConfig) {
            return vpcConfig(VpcConfig.builder().applyMutation(vpcConfig).build());
        }

        /**
         * <p>
         * The instance type for the image builder. The following instance types are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * stream.standard.small
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.standard.medium
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.standard.large
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.compute.large
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.compute.xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.compute.2xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.compute.4xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.compute.8xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.memory.large
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.memory.xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.memory.2xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.memory.4xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.memory.8xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.memory.z1d.large
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.memory.z1d.xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.memory.z1d.2xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.memory.z1d.3xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.memory.z1d.6xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.memory.z1d.12xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.graphics-design.large
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.graphics-design.xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.graphics-design.2xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.graphics-design.4xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.graphics-desktop.2xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.graphics.g4dn.xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.graphics.g4dn.2xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.graphics.g4dn.4xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.graphics.g4dn.8xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.graphics.g4dn.12xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.graphics.g4dn.16xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.graphics-pro.4xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.graphics-pro.8xlarge
         * </p>
         * </li>
         * <li>
         * <p>
         * stream.graphics-pro.16xlarge
         * </p>
         * </li>
         * </ul>
         * 
         * @param instanceType
         *        The instance type for the image builder. The following instance types are available:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        stream.standard.small
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.standard.medium
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.standard.large
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.compute.large
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.compute.xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.compute.2xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.compute.4xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.compute.8xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.memory.large
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.memory.xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.memory.2xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.memory.4xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.memory.8xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.memory.z1d.large
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.memory.z1d.xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.memory.z1d.2xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.memory.z1d.3xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.memory.z1d.6xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.memory.z1d.12xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.graphics-design.large
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.graphics-design.xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.graphics-design.2xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.graphics-design.4xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.graphics-desktop.2xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.graphics.g4dn.xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.graphics.g4dn.2xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.graphics.g4dn.4xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.graphics.g4dn.8xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.graphics.g4dn.12xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.graphics.g4dn.16xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.graphics-pro.4xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.graphics-pro.8xlarge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        stream.graphics-pro.16xlarge
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The operating system platform of the image builder.
         * </p>
         * 
         * @param platform
         *        The operating system platform of the image builder.
         * @see PlatformType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PlatformType
         */
        Builder platform(String platform);

        /**
         * <p>
         * The operating system platform of the image builder.
         * </p>
         * 
         * @param platform
         *        The operating system platform of the image builder.
         * @see PlatformType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PlatformType
         */
        Builder platform(PlatformType platform);

        /**
         * <p>
         * The ARN of the IAM role that is applied to the image builder. To assume a role, the image builder calls the
         * AWS Security Token Service (STS) <code>AssumeRole</code> API operation and passes the ARN of the role to use.
         * The operation creates a new session with temporary credentials. AppStream 2.0 retrieves the temporary
         * credentials and creates the <b>appstream_machine_role</b> credential profile on the instance.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/appstream2/latest/developerguide/using-iam-roles-to-grant-permissions-to-applications-scripts-streaming-instances.html"
         * >Using an IAM Role to Grant Permissions to Applications and Scripts Running on AppStream 2.0 Streaming
         * Instances</a> in the <i>Amazon AppStream 2.0 Administration Guide</i>.
         * </p>
         * 
         * @param iamRoleArn
         *        The ARN of the IAM role that is applied to the image builder. To assume a role, the image builder
         *        calls the AWS Security Token Service (STS) <code>AssumeRole</code> API operation and passes the ARN of
         *        the role to use. The operation creates a new session with temporary credentials. AppStream 2.0
         *        retrieves the temporary credentials and creates the <b>appstream_machine_role</b> credential profile
         *        on the instance.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/appstream2/latest/developerguide/using-iam-roles-to-grant-permissions-to-applications-scripts-streaming-instances.html"
         *        >Using an IAM Role to Grant Permissions to Applications and Scripts Running on AppStream 2.0 Streaming
         *        Instances</a> in the <i>Amazon AppStream 2.0 Administration Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamRoleArn(String iamRoleArn);

        /**
         * <p>
         * The state of the image builder.
         * </p>
         * 
         * @param state
         *        The state of the image builder.
         * @see ImageBuilderState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImageBuilderState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the image builder.
         * </p>
         * 
         * @param state
         *        The state of the image builder.
         * @see ImageBuilderState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImageBuilderState
         */
        Builder state(ImageBuilderState state);

        /**
         * <p>
         * The reason why the last state change occurred.
         * </p>
         * 
         * @param stateChangeReason
         *        The reason why the last state change occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateChangeReason(ImageBuilderStateChangeReason stateChangeReason);

        /**
         * <p>
         * The reason why the last state change occurred.
         * </p>
         * This is a convenience method that creates an instance of the {@link ImageBuilderStateChangeReason.Builder}
         * avoiding the need to create one manually via {@link ImageBuilderStateChangeReason#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ImageBuilderStateChangeReason.Builder#build()} is called
         * immediately and its result is passed to {@link #stateChangeReason(ImageBuilderStateChangeReason)}.
         * 
         * @param stateChangeReason
         *        a consumer that will call methods on {@link ImageBuilderStateChangeReason.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stateChangeReason(ImageBuilderStateChangeReason)
         */
        default Builder stateChangeReason(Consumer<ImageBuilderStateChangeReason.Builder> stateChangeReason) {
            return stateChangeReason(ImageBuilderStateChangeReason.builder().applyMutation(stateChangeReason).build());
        }

        /**
         * <p>
         * The time stamp when the image builder was created.
         * </p>
         * 
         * @param createdTime
         *        The time stamp when the image builder was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);

        /**
         * <p>
         * Enables or disables default internet access for the image builder.
         * </p>
         * 
         * @param enableDefaultInternetAccess
         *        Enables or disables default internet access for the image builder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableDefaultInternetAccess(Boolean enableDefaultInternetAccess);

        /**
         * <p>
         * The name of the directory and organizational unit (OU) to use to join the image builder to a Microsoft Active
         * Directory domain.
         * </p>
         * 
         * @param domainJoinInfo
         *        The name of the directory and organizational unit (OU) to use to join the image builder to a Microsoft
         *        Active Directory domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainJoinInfo(DomainJoinInfo domainJoinInfo);

        /**
         * <p>
         * The name of the directory and organizational unit (OU) to use to join the image builder to a Microsoft Active
         * Directory domain.
         * </p>
         * This is a convenience method that creates an instance of the {@link DomainJoinInfo.Builder} avoiding the need
         * to create one manually via {@link DomainJoinInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DomainJoinInfo.Builder#build()} is called immediately and its
         * result is passed to {@link #domainJoinInfo(DomainJoinInfo)}.
         * 
         * @param domainJoinInfo
         *        a consumer that will call methods on {@link DomainJoinInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #domainJoinInfo(DomainJoinInfo)
         */
        default Builder domainJoinInfo(Consumer<DomainJoinInfo.Builder> domainJoinInfo) {
            return domainJoinInfo(DomainJoinInfo.builder().applyMutation(domainJoinInfo).build());
        }

        /**
         * Sets the value of the NetworkAccessConfiguration property for this object.
         *
         * @param networkAccessConfiguration
         *        The new value for the NetworkAccessConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkAccessConfiguration(NetworkAccessConfiguration networkAccessConfiguration);

        /**
         * Sets the value of the NetworkAccessConfiguration property for this object.
         *
         * This is a convenience method that creates an instance of the {@link NetworkAccessConfiguration.Builder}
         * avoiding the need to create one manually via {@link NetworkAccessConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NetworkAccessConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #networkAccessConfiguration(NetworkAccessConfiguration)}.
         * 
         * @param networkAccessConfiguration
         *        a consumer that will call methods on {@link NetworkAccessConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkAccessConfiguration(NetworkAccessConfiguration)
         */
        default Builder networkAccessConfiguration(Consumer<NetworkAccessConfiguration.Builder> networkAccessConfiguration) {
            return networkAccessConfiguration(NetworkAccessConfiguration.builder().applyMutation(networkAccessConfiguration)
                    .build());
        }

        /**
         * <p>
         * The image builder errors.
         * </p>
         * 
         * @param imageBuilderErrors
         *        The image builder errors.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageBuilderErrors(Collection<ResourceError> imageBuilderErrors);

        /**
         * <p>
         * The image builder errors.
         * </p>
         * 
         * @param imageBuilderErrors
         *        The image builder errors.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageBuilderErrors(ResourceError... imageBuilderErrors);

        /**
         * <p>
         * The image builder errors.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.appstream.model.ResourceError.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.appstream.model.ResourceError#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.appstream.model.ResourceError.Builder#build()} is called immediately
         * and its result is passed to {@link #imageBuilderErrors(List<ResourceError>)}.
         * 
         * @param imageBuilderErrors
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.appstream.model.ResourceError.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #imageBuilderErrors(java.util.Collection<ResourceError>)
         */
        Builder imageBuilderErrors(Consumer<ResourceError.Builder>... imageBuilderErrors);

        /**
         * <p>
         * The version of the AppStream 2.0 agent that is currently being used by the image builder.
         * </p>
         * 
         * @param appstreamAgentVersion
         *        The version of the AppStream 2.0 agent that is currently being used by the image builder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appstreamAgentVersion(String appstreamAgentVersion);

        /**
         * <p>
         * The list of virtual private cloud (VPC) interface endpoint objects. Administrators can connect to the image
         * builder only through the specified endpoints.
         * </p>
         * 
         * @param accessEndpoints
         *        The list of virtual private cloud (VPC) interface endpoint objects. Administrators can connect to the
         *        image builder only through the specified endpoints.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessEndpoints(Collection<AccessEndpoint> accessEndpoints);

        /**
         * <p>
         * The list of virtual private cloud (VPC) interface endpoint objects. Administrators can connect to the image
         * builder only through the specified endpoints.
         * </p>
         * 
         * @param accessEndpoints
         *        The list of virtual private cloud (VPC) interface endpoint objects. Administrators can connect to the
         *        image builder only through the specified endpoints.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessEndpoints(AccessEndpoint... accessEndpoints);

        /**
         * <p>
         * The list of virtual private cloud (VPC) interface endpoint objects. Administrators can connect to the image
         * builder only through the specified endpoints.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.appstream.model.AccessEndpoint.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.appstream.model.AccessEndpoint#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.appstream.model.AccessEndpoint.Builder#build()} is called immediately
         * and its result is passed to {@link #accessEndpoints(List<AccessEndpoint>)}.
         * 
         * @param accessEndpoints
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.appstream.model.AccessEndpoint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accessEndpoints(java.util.Collection<AccessEndpoint>)
         */
        Builder accessEndpoints(Consumer<AccessEndpoint.Builder>... accessEndpoints);

        /**
         * <p>
         * Indicates whether the image builder is using the latest AppStream 2.0 agent version or not.
         * </p>
         * 
         * @param latestAppstreamAgentVersion
         *        Indicates whether the image builder is using the latest AppStream 2.0 agent version or not.
         * @see LatestAppstreamAgentVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LatestAppstreamAgentVersion
         */
        Builder latestAppstreamAgentVersion(String latestAppstreamAgentVersion);

        /**
         * <p>
         * Indicates whether the image builder is using the latest AppStream 2.0 agent version or not.
         * </p>
         * 
         * @param latestAppstreamAgentVersion
         *        Indicates whether the image builder is using the latest AppStream 2.0 agent version or not.
         * @see LatestAppstreamAgentVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LatestAppstreamAgentVersion
         */
        Builder latestAppstreamAgentVersion(LatestAppstreamAgentVersion latestAppstreamAgentVersion);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private String imageArn;

        private String description;

        private String displayName;

        private VpcConfig vpcConfig;

        private String instanceType;

        private String platform;

        private String iamRoleArn;

        private String state;

        private ImageBuilderStateChangeReason stateChangeReason;

        private Instant createdTime;

        private Boolean enableDefaultInternetAccess;

        private DomainJoinInfo domainJoinInfo;

        private NetworkAccessConfiguration networkAccessConfiguration;

        private List<ResourceError> imageBuilderErrors = DefaultSdkAutoConstructList.getInstance();

        private String appstreamAgentVersion;

        private List<AccessEndpoint> accessEndpoints = DefaultSdkAutoConstructList.getInstance();

        private String latestAppstreamAgentVersion;

        private BuilderImpl() {
        }

        private BuilderImpl(ImageBuilder model) {
            name(model.name);
            arn(model.arn);
            imageArn(model.imageArn);
            description(model.description);
            displayName(model.displayName);
            vpcConfig(model.vpcConfig);
            instanceType(model.instanceType);
            platform(model.platform);
            iamRoleArn(model.iamRoleArn);
            state(model.state);
            stateChangeReason(model.stateChangeReason);
            createdTime(model.createdTime);
            enableDefaultInternetAccess(model.enableDefaultInternetAccess);
            domainJoinInfo(model.domainJoinInfo);
            networkAccessConfiguration(model.networkAccessConfiguration);
            imageBuilderErrors(model.imageBuilderErrors);
            appstreamAgentVersion(model.appstreamAgentVersion);
            accessEndpoints(model.accessEndpoints);
            latestAppstreamAgentVersion(model.latestAppstreamAgentVersion);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getImageArn() {
            return imageArn;
        }

        public final void setImageArn(String imageArn) {
            this.imageArn = imageArn;
        }

        @Override
        public final Builder imageArn(String imageArn) {
            this.imageArn = imageArn;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getDisplayName() {
            return displayName;
        }

        public final void setDisplayName(String displayName) {
            this.displayName = displayName;
        }

        @Override
        public final Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public final VpcConfig.Builder getVpcConfig() {
            return vpcConfig != null ? vpcConfig.toBuilder() : null;
        }

        public final void setVpcConfig(VpcConfig.BuilderImpl vpcConfig) {
            this.vpcConfig = vpcConfig != null ? vpcConfig.build() : null;
        }

        @Override
        public final Builder vpcConfig(VpcConfig vpcConfig) {
            this.vpcConfig = vpcConfig;
            return this;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final String getPlatform() {
            return platform;
        }

        public final void setPlatform(String platform) {
            this.platform = platform;
        }

        @Override
        public final Builder platform(String platform) {
            this.platform = platform;
            return this;
        }

        @Override
        public final Builder platform(PlatformType platform) {
            this.platform(platform == null ? null : platform.toString());
            return this;
        }

        public final String getIamRoleArn() {
            return iamRoleArn;
        }

        public final void setIamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
        }

        @Override
        public final Builder iamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(ImageBuilderState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final ImageBuilderStateChangeReason.Builder getStateChangeReason() {
            return stateChangeReason != null ? stateChangeReason.toBuilder() : null;
        }

        public final void setStateChangeReason(ImageBuilderStateChangeReason.BuilderImpl stateChangeReason) {
            this.stateChangeReason = stateChangeReason != null ? stateChangeReason.build() : null;
        }

        @Override
        public final Builder stateChangeReason(ImageBuilderStateChangeReason stateChangeReason) {
            this.stateChangeReason = stateChangeReason;
            return this;
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        @Override
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        public final Boolean getEnableDefaultInternetAccess() {
            return enableDefaultInternetAccess;
        }

        public final void setEnableDefaultInternetAccess(Boolean enableDefaultInternetAccess) {
            this.enableDefaultInternetAccess = enableDefaultInternetAccess;
        }

        @Override
        public final Builder enableDefaultInternetAccess(Boolean enableDefaultInternetAccess) {
            this.enableDefaultInternetAccess = enableDefaultInternetAccess;
            return this;
        }

        public final DomainJoinInfo.Builder getDomainJoinInfo() {
            return domainJoinInfo != null ? domainJoinInfo.toBuilder() : null;
        }

        public final void setDomainJoinInfo(DomainJoinInfo.BuilderImpl domainJoinInfo) {
            this.domainJoinInfo = domainJoinInfo != null ? domainJoinInfo.build() : null;
        }

        @Override
        public final Builder domainJoinInfo(DomainJoinInfo domainJoinInfo) {
            this.domainJoinInfo = domainJoinInfo;
            return this;
        }

        public final NetworkAccessConfiguration.Builder getNetworkAccessConfiguration() {
            return networkAccessConfiguration != null ? networkAccessConfiguration.toBuilder() : null;
        }

        public final void setNetworkAccessConfiguration(NetworkAccessConfiguration.BuilderImpl networkAccessConfiguration) {
            this.networkAccessConfiguration = networkAccessConfiguration != null ? networkAccessConfiguration.build() : null;
        }

        @Override
        public final Builder networkAccessConfiguration(NetworkAccessConfiguration networkAccessConfiguration) {
            this.networkAccessConfiguration = networkAccessConfiguration;
            return this;
        }

        public final List<ResourceError.Builder> getImageBuilderErrors() {
            List<ResourceError.Builder> result = ResourceErrorsCopier.copyToBuilder(this.imageBuilderErrors);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setImageBuilderErrors(Collection<ResourceError.BuilderImpl> imageBuilderErrors) {
            this.imageBuilderErrors = ResourceErrorsCopier.copyFromBuilder(imageBuilderErrors);
        }

        @Override
        public final Builder imageBuilderErrors(Collection<ResourceError> imageBuilderErrors) {
            this.imageBuilderErrors = ResourceErrorsCopier.copy(imageBuilderErrors);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder imageBuilderErrors(ResourceError... imageBuilderErrors) {
            imageBuilderErrors(Arrays.asList(imageBuilderErrors));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder imageBuilderErrors(Consumer<ResourceError.Builder>... imageBuilderErrors) {
            imageBuilderErrors(Stream.of(imageBuilderErrors).map(c -> ResourceError.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getAppstreamAgentVersion() {
            return appstreamAgentVersion;
        }

        public final void setAppstreamAgentVersion(String appstreamAgentVersion) {
            this.appstreamAgentVersion = appstreamAgentVersion;
        }

        @Override
        public final Builder appstreamAgentVersion(String appstreamAgentVersion) {
            this.appstreamAgentVersion = appstreamAgentVersion;
            return this;
        }

        public final List<AccessEndpoint.Builder> getAccessEndpoints() {
            List<AccessEndpoint.Builder> result = AccessEndpointListCopier.copyToBuilder(this.accessEndpoints);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAccessEndpoints(Collection<AccessEndpoint.BuilderImpl> accessEndpoints) {
            this.accessEndpoints = AccessEndpointListCopier.copyFromBuilder(accessEndpoints);
        }

        @Override
        public final Builder accessEndpoints(Collection<AccessEndpoint> accessEndpoints) {
            this.accessEndpoints = AccessEndpointListCopier.copy(accessEndpoints);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accessEndpoints(AccessEndpoint... accessEndpoints) {
            accessEndpoints(Arrays.asList(accessEndpoints));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accessEndpoints(Consumer<AccessEndpoint.Builder>... accessEndpoints) {
            accessEndpoints(Stream.of(accessEndpoints).map(c -> AccessEndpoint.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getLatestAppstreamAgentVersion() {
            return latestAppstreamAgentVersion;
        }

        public final void setLatestAppstreamAgentVersion(String latestAppstreamAgentVersion) {
            this.latestAppstreamAgentVersion = latestAppstreamAgentVersion;
        }

        @Override
        public final Builder latestAppstreamAgentVersion(String latestAppstreamAgentVersion) {
            this.latestAppstreamAgentVersion = latestAppstreamAgentVersion;
            return this;
        }

        @Override
        public final Builder latestAppstreamAgentVersion(LatestAppstreamAgentVersion latestAppstreamAgentVersion) {
            this.latestAppstreamAgentVersion(latestAppstreamAgentVersion == null ? null : latestAppstreamAgentVersion.toString());
            return this;
        }

        @Override
        public ImageBuilder build() {
            return new ImageBuilder(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
