/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appstream.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a virtual machine that is used to create an image.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ImageBuilder implements SdkPojo, Serializable, ToCopyableBuilder<ImageBuilder.Builder, ImageBuilder> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ImageBuilder::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ImageBuilder::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> IMAGE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ImageBuilder::imageArn)).setter(setter(Builder::imageArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageArn").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ImageBuilder::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> DISPLAY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ImageBuilder::displayName)).setter(setter(Builder::displayName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisplayName").build()).build();

    private static final SdkField<VpcConfig> VPC_CONFIG_FIELD = SdkField.<VpcConfig> builder(MarshallingType.SDK_POJO)
            .getter(getter(ImageBuilder::vpcConfig)).setter(setter(Builder::vpcConfig)).constructor(VpcConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcConfig").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ImageBuilder::instanceType)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<String> PLATFORM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ImageBuilder::platformAsString)).setter(setter(Builder::platform))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Platform").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ImageBuilder::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<ImageBuilderStateChangeReason> STATE_CHANGE_REASON_FIELD = SdkField
            .<ImageBuilderStateChangeReason> builder(MarshallingType.SDK_POJO).getter(getter(ImageBuilder::stateChangeReason))
            .setter(setter(Builder::stateChangeReason)).constructor(ImageBuilderStateChangeReason::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateChangeReason").build()).build();

    private static final SdkField<Instant> CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(ImageBuilder::createdTime)).setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTime").build()).build();

    private static final SdkField<Boolean> ENABLE_DEFAULT_INTERNET_ACCESS_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(ImageBuilder::enableDefaultInternetAccess))
            .setter(setter(Builder::enableDefaultInternetAccess))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableDefaultInternetAccess")
                    .build()).build();

    private static final SdkField<DomainJoinInfo> DOMAIN_JOIN_INFO_FIELD = SdkField
            .<DomainJoinInfo> builder(MarshallingType.SDK_POJO).getter(getter(ImageBuilder::domainJoinInfo))
            .setter(setter(Builder::domainJoinInfo)).constructor(DomainJoinInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainJoinInfo").build()).build();

    private static final SdkField<List<ResourceError>> IMAGE_BUILDER_ERRORS_FIELD = SdkField
            .<List<ResourceError>> builder(MarshallingType.LIST)
            .getter(getter(ImageBuilder::imageBuilderErrors))
            .setter(setter(Builder::imageBuilderErrors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageBuilderErrors").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceError> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceError::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> APPSTREAM_AGENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ImageBuilder::appstreamAgentVersion)).setter(setter(Builder::appstreamAgentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AppstreamAgentVersion").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD,
            IMAGE_ARN_FIELD, DESCRIPTION_FIELD, DISPLAY_NAME_FIELD, VPC_CONFIG_FIELD, INSTANCE_TYPE_FIELD, PLATFORM_FIELD,
            STATE_FIELD, STATE_CHANGE_REASON_FIELD, CREATED_TIME_FIELD, ENABLE_DEFAULT_INTERNET_ACCESS_FIELD,
            DOMAIN_JOIN_INFO_FIELD, IMAGE_BUILDER_ERRORS_FIELD, APPSTREAM_AGENT_VERSION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final String imageArn;

    private final String description;

    private final String displayName;

    private final VpcConfig vpcConfig;

    private final String instanceType;

    private final String platform;

    private final String state;

    private final ImageBuilderStateChangeReason stateChangeReason;

    private final Instant createdTime;

    private final Boolean enableDefaultInternetAccess;

    private final DomainJoinInfo domainJoinInfo;

    private final List<ResourceError> imageBuilderErrors;

    private final String appstreamAgentVersion;

    private ImageBuilder(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.imageArn = builder.imageArn;
        this.description = builder.description;
        this.displayName = builder.displayName;
        this.vpcConfig = builder.vpcConfig;
        this.instanceType = builder.instanceType;
        this.platform = builder.platform;
        this.state = builder.state;
        this.stateChangeReason = builder.stateChangeReason;
        this.createdTime = builder.createdTime;
        this.enableDefaultInternetAccess = builder.enableDefaultInternetAccess;
        this.domainJoinInfo = builder.domainJoinInfo;
        this.imageBuilderErrors = builder.imageBuilderErrors;
        this.appstreamAgentVersion = builder.appstreamAgentVersion;
    }

    /**
     * <p>
     * The name of the image builder.
     * </p>
     * 
     * @return The name of the image builder.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The ARN for the image builder.
     * </p>
     * 
     * @return The ARN for the image builder.
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * The ARN of the image from which this builder was created.
     * </p>
     * 
     * @return The ARN of the image from which this builder was created.
     */
    public String imageArn() {
        return imageArn;
    }

    /**
     * <p>
     * The description to display.
     * </p>
     * 
     * @return The description to display.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The image builder name to display.
     * </p>
     * 
     * @return The image builder name to display.
     */
    public String displayName() {
        return displayName;
    }

    /**
     * <p>
     * The VPC configuration of the image builder.
     * </p>
     * 
     * @return The VPC configuration of the image builder.
     */
    public VpcConfig vpcConfig() {
        return vpcConfig;
    }

    /**
     * <p>
     * The instance type for the image builder.
     * </p>
     * 
     * @return The instance type for the image builder.
     */
    public String instanceType() {
        return instanceType;
    }

    /**
     * <p>
     * The operating system platform of the image builder.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #platform} will
     * return {@link PlatformType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #platformAsString}.
     * </p>
     * 
     * @return The operating system platform of the image builder.
     * @see PlatformType
     */
    public PlatformType platform() {
        return PlatformType.fromValue(platform);
    }

    /**
     * <p>
     * The operating system platform of the image builder.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #platform} will
     * return {@link PlatformType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #platformAsString}.
     * </p>
     * 
     * @return The operating system platform of the image builder.
     * @see PlatformType
     */
    public String platformAsString() {
        return platform;
    }

    /**
     * <p>
     * The state of the image builder.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link ImageBuilderState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the image builder.
     * @see ImageBuilderState
     */
    public ImageBuilderState state() {
        return ImageBuilderState.fromValue(state);
    }

    /**
     * <p>
     * The state of the image builder.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link ImageBuilderState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the image builder.
     * @see ImageBuilderState
     */
    public String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The reason why the last state change occurred.
     * </p>
     * 
     * @return The reason why the last state change occurred.
     */
    public ImageBuilderStateChangeReason stateChangeReason() {
        return stateChangeReason;
    }

    /**
     * <p>
     * The time stamp when the image builder was created.
     * </p>
     * 
     * @return The time stamp when the image builder was created.
     */
    public Instant createdTime() {
        return createdTime;
    }

    /**
     * <p>
     * Enables or disables default internet access for the image builder.
     * </p>
     * 
     * @return Enables or disables default internet access for the image builder.
     */
    public Boolean enableDefaultInternetAccess() {
        return enableDefaultInternetAccess;
    }

    /**
     * <p>
     * The name of the directory and organizational unit (OU) to use to join the image builder to a Microsoft Active
     * Directory domain.
     * </p>
     * 
     * @return The name of the directory and organizational unit (OU) to use to join the image builder to a Microsoft
     *         Active Directory domain.
     */
    public DomainJoinInfo domainJoinInfo() {
        return domainJoinInfo;
    }

    /**
     * <p>
     * The image builder errors.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The image builder errors.
     */
    public List<ResourceError> imageBuilderErrors() {
        return imageBuilderErrors;
    }

    /**
     * <p>
     * The version of the AppStream 2.0 agent that is currently being used by the image builder.
     * </p>
     * 
     * @return The version of the AppStream 2.0 agent that is currently being used by the image builder.
     */
    public String appstreamAgentVersion() {
        return appstreamAgentVersion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(imageArn());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(displayName());
        hashCode = 31 * hashCode + Objects.hashCode(vpcConfig());
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(platformAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateChangeReason());
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        hashCode = 31 * hashCode + Objects.hashCode(enableDefaultInternetAccess());
        hashCode = 31 * hashCode + Objects.hashCode(domainJoinInfo());
        hashCode = 31 * hashCode + Objects.hashCode(imageBuilderErrors());
        hashCode = 31 * hashCode + Objects.hashCode(appstreamAgentVersion());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImageBuilder)) {
            return false;
        }
        ImageBuilder other = (ImageBuilder) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(imageArn(), other.imageArn()) && Objects.equals(description(), other.description())
                && Objects.equals(displayName(), other.displayName()) && Objects.equals(vpcConfig(), other.vpcConfig())
                && Objects.equals(instanceType(), other.instanceType())
                && Objects.equals(platformAsString(), other.platformAsString())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(stateChangeReason(), other.stateChangeReason())
                && Objects.equals(createdTime(), other.createdTime())
                && Objects.equals(enableDefaultInternetAccess(), other.enableDefaultInternetAccess())
                && Objects.equals(domainJoinInfo(), other.domainJoinInfo())
                && Objects.equals(imageBuilderErrors(), other.imageBuilderErrors())
                && Objects.equals(appstreamAgentVersion(), other.appstreamAgentVersion());
    }

    @Override
    public String toString() {
        return ToString.builder("ImageBuilder").add("Name", name()).add("Arn", arn()).add("ImageArn", imageArn())
                .add("Description", description()).add("DisplayName", displayName()).add("VpcConfig", vpcConfig())
                .add("InstanceType", instanceType()).add("Platform", platformAsString()).add("State", stateAsString())
                .add("StateChangeReason", stateChangeReason()).add("CreatedTime", createdTime())
                .add("EnableDefaultInternetAccess", enableDefaultInternetAccess()).add("DomainJoinInfo", domainJoinInfo())
                .add("ImageBuilderErrors", imageBuilderErrors()).add("AppstreamAgentVersion", appstreamAgentVersion()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "ImageArn":
            return Optional.ofNullable(clazz.cast(imageArn()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "DisplayName":
            return Optional.ofNullable(clazz.cast(displayName()));
        case "VpcConfig":
            return Optional.ofNullable(clazz.cast(vpcConfig()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "Platform":
            return Optional.ofNullable(clazz.cast(platformAsString()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "StateChangeReason":
            return Optional.ofNullable(clazz.cast(stateChangeReason()));
        case "CreatedTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        case "EnableDefaultInternetAccess":
            return Optional.ofNullable(clazz.cast(enableDefaultInternetAccess()));
        case "DomainJoinInfo":
            return Optional.ofNullable(clazz.cast(domainJoinInfo()));
        case "ImageBuilderErrors":
            return Optional.ofNullable(clazz.cast(imageBuilderErrors()));
        case "AppstreamAgentVersion":
            return Optional.ofNullable(clazz.cast(appstreamAgentVersion()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ImageBuilder, T> g) {
        return obj -> g.apply((ImageBuilder) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ImageBuilder> {
        /**
         * <p>
         * The name of the image builder.
         * </p>
         * 
         * @param name
         *        The name of the image builder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ARN for the image builder.
         * </p>
         * 
         * @param arn
         *        The ARN for the image builder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The ARN of the image from which this builder was created.
         * </p>
         * 
         * @param imageArn
         *        The ARN of the image from which this builder was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageArn(String imageArn);

        /**
         * <p>
         * The description to display.
         * </p>
         * 
         * @param description
         *        The description to display.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The image builder name to display.
         * </p>
         * 
         * @param displayName
         *        The image builder name to display.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder displayName(String displayName);

        /**
         * <p>
         * The VPC configuration of the image builder.
         * </p>
         * 
         * @param vpcConfig
         *        The VPC configuration of the image builder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConfig(VpcConfig vpcConfig);

        /**
         * <p>
         * The VPC configuration of the image builder.
         * </p>
         * This is a convenience that creates an instance of the {@link VpcConfig.Builder} avoiding the need to create
         * one manually via {@link VpcConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link VpcConfig.Builder#build()} is called immediately and its result
         * is passed to {@link #vpcConfig(VpcConfig)}.
         * 
         * @param vpcConfig
         *        a consumer that will call methods on {@link VpcConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcConfig(VpcConfig)
         */
        default Builder vpcConfig(Consumer<VpcConfig.Builder> vpcConfig) {
            return vpcConfig(VpcConfig.builder().applyMutation(vpcConfig).build());
        }

        /**
         * <p>
         * The instance type for the image builder.
         * </p>
         * 
         * @param instanceType
         *        The instance type for the image builder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The operating system platform of the image builder.
         * </p>
         * 
         * @param platform
         *        The operating system platform of the image builder.
         * @see PlatformType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PlatformType
         */
        Builder platform(String platform);

        /**
         * <p>
         * The operating system platform of the image builder.
         * </p>
         * 
         * @param platform
         *        The operating system platform of the image builder.
         * @see PlatformType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PlatformType
         */
        Builder platform(PlatformType platform);

        /**
         * <p>
         * The state of the image builder.
         * </p>
         * 
         * @param state
         *        The state of the image builder.
         * @see ImageBuilderState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImageBuilderState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the image builder.
         * </p>
         * 
         * @param state
         *        The state of the image builder.
         * @see ImageBuilderState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImageBuilderState
         */
        Builder state(ImageBuilderState state);

        /**
         * <p>
         * The reason why the last state change occurred.
         * </p>
         * 
         * @param stateChangeReason
         *        The reason why the last state change occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateChangeReason(ImageBuilderStateChangeReason stateChangeReason);

        /**
         * <p>
         * The reason why the last state change occurred.
         * </p>
         * This is a convenience that creates an instance of the {@link ImageBuilderStateChangeReason.Builder} avoiding
         * the need to create one manually via {@link ImageBuilderStateChangeReason#builder()}.
         *
         * When the {@link Consumer} completes, {@link ImageBuilderStateChangeReason.Builder#build()} is called
         * immediately and its result is passed to {@link #stateChangeReason(ImageBuilderStateChangeReason)}.
         * 
         * @param stateChangeReason
         *        a consumer that will call methods on {@link ImageBuilderStateChangeReason.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stateChangeReason(ImageBuilderStateChangeReason)
         */
        default Builder stateChangeReason(Consumer<ImageBuilderStateChangeReason.Builder> stateChangeReason) {
            return stateChangeReason(ImageBuilderStateChangeReason.builder().applyMutation(stateChangeReason).build());
        }

        /**
         * <p>
         * The time stamp when the image builder was created.
         * </p>
         * 
         * @param createdTime
         *        The time stamp when the image builder was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);

        /**
         * <p>
         * Enables or disables default internet access for the image builder.
         * </p>
         * 
         * @param enableDefaultInternetAccess
         *        Enables or disables default internet access for the image builder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableDefaultInternetAccess(Boolean enableDefaultInternetAccess);

        /**
         * <p>
         * The name of the directory and organizational unit (OU) to use to join the image builder to a Microsoft Active
         * Directory domain.
         * </p>
         * 
         * @param domainJoinInfo
         *        The name of the directory and organizational unit (OU) to use to join the image builder to a Microsoft
         *        Active Directory domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainJoinInfo(DomainJoinInfo domainJoinInfo);

        /**
         * <p>
         * The name of the directory and organizational unit (OU) to use to join the image builder to a Microsoft Active
         * Directory domain.
         * </p>
         * This is a convenience that creates an instance of the {@link DomainJoinInfo.Builder} avoiding the need to
         * create one manually via {@link DomainJoinInfo#builder()}.
         *
         * When the {@link Consumer} completes, {@link DomainJoinInfo.Builder#build()} is called immediately and its
         * result is passed to {@link #domainJoinInfo(DomainJoinInfo)}.
         * 
         * @param domainJoinInfo
         *        a consumer that will call methods on {@link DomainJoinInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #domainJoinInfo(DomainJoinInfo)
         */
        default Builder domainJoinInfo(Consumer<DomainJoinInfo.Builder> domainJoinInfo) {
            return domainJoinInfo(DomainJoinInfo.builder().applyMutation(domainJoinInfo).build());
        }

        /**
         * <p>
         * The image builder errors.
         * </p>
         * 
         * @param imageBuilderErrors
         *        The image builder errors.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageBuilderErrors(Collection<ResourceError> imageBuilderErrors);

        /**
         * <p>
         * The image builder errors.
         * </p>
         * 
         * @param imageBuilderErrors
         *        The image builder errors.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageBuilderErrors(ResourceError... imageBuilderErrors);

        /**
         * <p>
         * The image builder errors.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ResourceError>.Builder} avoiding the need
         * to create one manually via {@link List<ResourceError>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ResourceError>.Builder#build()} is called immediately and
         * its result is passed to {@link #imageBuilderErrors(List<ResourceError>)}.
         * 
         * @param imageBuilderErrors
         *        a consumer that will call methods on {@link List<ResourceError>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #imageBuilderErrors(List<ResourceError>)
         */
        Builder imageBuilderErrors(Consumer<ResourceError.Builder>... imageBuilderErrors);

        /**
         * <p>
         * The version of the AppStream 2.0 agent that is currently being used by the image builder.
         * </p>
         * 
         * @param appstreamAgentVersion
         *        The version of the AppStream 2.0 agent that is currently being used by the image builder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appstreamAgentVersion(String appstreamAgentVersion);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private String imageArn;

        private String description;

        private String displayName;

        private VpcConfig vpcConfig;

        private String instanceType;

        private String platform;

        private String state;

        private ImageBuilderStateChangeReason stateChangeReason;

        private Instant createdTime;

        private Boolean enableDefaultInternetAccess;

        private DomainJoinInfo domainJoinInfo;

        private List<ResourceError> imageBuilderErrors = DefaultSdkAutoConstructList.getInstance();

        private String appstreamAgentVersion;

        private BuilderImpl() {
        }

        private BuilderImpl(ImageBuilder model) {
            name(model.name);
            arn(model.arn);
            imageArn(model.imageArn);
            description(model.description);
            displayName(model.displayName);
            vpcConfig(model.vpcConfig);
            instanceType(model.instanceType);
            platform(model.platform);
            state(model.state);
            stateChangeReason(model.stateChangeReason);
            createdTime(model.createdTime);
            enableDefaultInternetAccess(model.enableDefaultInternetAccess);
            domainJoinInfo(model.domainJoinInfo);
            imageBuilderErrors(model.imageBuilderErrors);
            appstreamAgentVersion(model.appstreamAgentVersion);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getImageArn() {
            return imageArn;
        }

        @Override
        public final Builder imageArn(String imageArn) {
            this.imageArn = imageArn;
            return this;
        }

        public final void setImageArn(String imageArn) {
            this.imageArn = imageArn;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getDisplayName() {
            return displayName;
        }

        @Override
        public final Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public final void setDisplayName(String displayName) {
            this.displayName = displayName;
        }

        public final VpcConfig.Builder getVpcConfig() {
            return vpcConfig != null ? vpcConfig.toBuilder() : null;
        }

        @Override
        public final Builder vpcConfig(VpcConfig vpcConfig) {
            this.vpcConfig = vpcConfig;
            return this;
        }

        public final void setVpcConfig(VpcConfig.BuilderImpl vpcConfig) {
            this.vpcConfig = vpcConfig != null ? vpcConfig.build() : null;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        public final String getPlatformAsString() {
            return platform;
        }

        @Override
        public final Builder platform(String platform) {
            this.platform = platform;
            return this;
        }

        @Override
        public final Builder platform(PlatformType platform) {
            this.platform(platform.toString());
            return this;
        }

        public final void setPlatform(String platform) {
            this.platform = platform;
        }

        public final String getStateAsString() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(ImageBuilderState state) {
            this.state(state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final ImageBuilderStateChangeReason.Builder getStateChangeReason() {
            return stateChangeReason != null ? stateChangeReason.toBuilder() : null;
        }

        @Override
        public final Builder stateChangeReason(ImageBuilderStateChangeReason stateChangeReason) {
            this.stateChangeReason = stateChangeReason;
            return this;
        }

        public final void setStateChangeReason(ImageBuilderStateChangeReason.BuilderImpl stateChangeReason) {
            this.stateChangeReason = stateChangeReason != null ? stateChangeReason.build() : null;
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        @Override
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        public final Boolean getEnableDefaultInternetAccess() {
            return enableDefaultInternetAccess;
        }

        @Override
        public final Builder enableDefaultInternetAccess(Boolean enableDefaultInternetAccess) {
            this.enableDefaultInternetAccess = enableDefaultInternetAccess;
            return this;
        }

        public final void setEnableDefaultInternetAccess(Boolean enableDefaultInternetAccess) {
            this.enableDefaultInternetAccess = enableDefaultInternetAccess;
        }

        public final DomainJoinInfo.Builder getDomainJoinInfo() {
            return domainJoinInfo != null ? domainJoinInfo.toBuilder() : null;
        }

        @Override
        public final Builder domainJoinInfo(DomainJoinInfo domainJoinInfo) {
            this.domainJoinInfo = domainJoinInfo;
            return this;
        }

        public final void setDomainJoinInfo(DomainJoinInfo.BuilderImpl domainJoinInfo) {
            this.domainJoinInfo = domainJoinInfo != null ? domainJoinInfo.build() : null;
        }

        public final Collection<ResourceError.Builder> getImageBuilderErrors() {
            return imageBuilderErrors != null ? imageBuilderErrors.stream().map(ResourceError::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder imageBuilderErrors(Collection<ResourceError> imageBuilderErrors) {
            this.imageBuilderErrors = ResourceErrorsCopier.copy(imageBuilderErrors);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder imageBuilderErrors(ResourceError... imageBuilderErrors) {
            imageBuilderErrors(Arrays.asList(imageBuilderErrors));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder imageBuilderErrors(Consumer<ResourceError.Builder>... imageBuilderErrors) {
            imageBuilderErrors(Stream.of(imageBuilderErrors).map(c -> ResourceError.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setImageBuilderErrors(Collection<ResourceError.BuilderImpl> imageBuilderErrors) {
            this.imageBuilderErrors = ResourceErrorsCopier.copyFromBuilder(imageBuilderErrors);
        }

        public final String getAppstreamAgentVersion() {
            return appstreamAgentVersion;
        }

        @Override
        public final Builder appstreamAgentVersion(String appstreamAgentVersion) {
            this.appstreamAgentVersion = appstreamAgentVersion;
            return this;
        }

        public final void setAppstreamAgentVersion(String appstreamAgentVersion) {
            this.appstreamAgentVersion = appstreamAgentVersion;
        }

        @Override
        public ImageBuilder build() {
            return new ImageBuilder(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
