/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appstream.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an application in the application catalog.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Application implements SdkPojo, Serializable, ToCopyableBuilder<Application.Builder, Application> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Application::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DISPLAY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DisplayName").getter(getter(Application::displayName)).setter(setter(Builder::displayName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisplayName").build()).build();

    private static final SdkField<String> ICON_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IconURL").getter(getter(Application::iconURL)).setter(setter(Builder::iconURL))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IconURL").build()).build();

    private static final SdkField<String> LAUNCH_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LaunchPath").getter(getter(Application::launchPath)).setter(setter(Builder::launchPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LaunchPath").build()).build();

    private static final SdkField<String> LAUNCH_PARAMETERS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LaunchParameters").getter(getter(Application::launchParameters))
            .setter(setter(Builder::launchParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LaunchParameters").build()).build();

    private static final SdkField<Boolean> ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Enabled").getter(getter(Application::enabled)).setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Enabled").build()).build();

    private static final SdkField<Map<String, String>> METADATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Metadata")
            .getter(getter(Application::metadata))
            .setter(setter(Builder::metadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Metadata").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> WORKING_DIRECTORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WorkingDirectory").getter(getter(Application::workingDirectory))
            .setter(setter(Builder::workingDirectory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkingDirectory").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(Application::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(Application::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> APP_BLOCK_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AppBlockArn").getter(getter(Application::appBlockArn)).setter(setter(Builder::appBlockArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AppBlockArn").build()).build();

    private static final SdkField<S3Location> ICON_S3_LOCATION_FIELD = SdkField.<S3Location> builder(MarshallingType.SDK_POJO)
            .memberName("IconS3Location").getter(getter(Application::iconS3Location)).setter(setter(Builder::iconS3Location))
            .constructor(S3Location::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IconS3Location").build()).build();

    private static final SdkField<List<String>> PLATFORMS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Platforms")
            .getter(getter(Application::platformsAsStrings))
            .setter(setter(Builder::platformsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Platforms").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> INSTANCE_FAMILIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("InstanceFamilies")
            .getter(getter(Application::instanceFamilies))
            .setter(setter(Builder::instanceFamilies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceFamilies").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTime").getter(getter(Application::createdTime)).setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            DISPLAY_NAME_FIELD, ICON_URL_FIELD, LAUNCH_PATH_FIELD, LAUNCH_PARAMETERS_FIELD, ENABLED_FIELD, METADATA_FIELD,
            WORKING_DIRECTORY_FIELD, DESCRIPTION_FIELD, ARN_FIELD, APP_BLOCK_ARN_FIELD, ICON_S3_LOCATION_FIELD, PLATFORMS_FIELD,
            INSTANCE_FAMILIES_FIELD, CREATED_TIME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String displayName;

    private final String iconURL;

    private final String launchPath;

    private final String launchParameters;

    private final Boolean enabled;

    private final Map<String, String> metadata;

    private final String workingDirectory;

    private final String description;

    private final String arn;

    private final String appBlockArn;

    private final S3Location iconS3Location;

    private final List<String> platforms;

    private final List<String> instanceFamilies;

    private final Instant createdTime;

    private Application(BuilderImpl builder) {
        this.name = builder.name;
        this.displayName = builder.displayName;
        this.iconURL = builder.iconURL;
        this.launchPath = builder.launchPath;
        this.launchParameters = builder.launchParameters;
        this.enabled = builder.enabled;
        this.metadata = builder.metadata;
        this.workingDirectory = builder.workingDirectory;
        this.description = builder.description;
        this.arn = builder.arn;
        this.appBlockArn = builder.appBlockArn;
        this.iconS3Location = builder.iconS3Location;
        this.platforms = builder.platforms;
        this.instanceFamilies = builder.instanceFamilies;
        this.createdTime = builder.createdTime;
    }

    /**
     * <p>
     * The name of the application.
     * </p>
     * 
     * @return The name of the application.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The application name to display.
     * </p>
     * 
     * @return The application name to display.
     */
    public final String displayName() {
        return displayName;
    }

    /**
     * <p>
     * The URL for the application icon. This URL might be time-limited.
     * </p>
     * 
     * @return The URL for the application icon. This URL might be time-limited.
     */
    public final String iconURL() {
        return iconURL;
    }

    /**
     * <p>
     * The path to the application executable in the instance.
     * </p>
     * 
     * @return The path to the application executable in the instance.
     */
    public final String launchPath() {
        return launchPath;
    }

    /**
     * <p>
     * The arguments that are passed to the application at launch.
     * </p>
     * 
     * @return The arguments that are passed to the application at launch.
     */
    public final String launchParameters() {
        return launchParameters;
    }

    /**
     * <p>
     * If there is a problem, the application can be disabled after image creation.
     * </p>
     * 
     * @return If there is a problem, the application can be disabled after image creation.
     */
    public final Boolean enabled() {
        return enabled;
    }

    /**
     * For responses, this returns true if the service returned a value for the Metadata property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasMetadata() {
        return metadata != null && !(metadata instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Additional attributes that describe the application.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetadata} method.
     * </p>
     * 
     * @return Additional attributes that describe the application.
     */
    public final Map<String, String> metadata() {
        return metadata;
    }

    /**
     * <p>
     * The working directory for the application.
     * </p>
     * 
     * @return The working directory for the application.
     */
    public final String workingDirectory() {
        return workingDirectory;
    }

    /**
     * <p>
     * The description of the application.
     * </p>
     * 
     * @return The description of the application.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The ARN of the application.
     * </p>
     * 
     * @return The ARN of the application.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The app block ARN of the application.
     * </p>
     * 
     * @return The app block ARN of the application.
     */
    public final String appBlockArn() {
        return appBlockArn;
    }

    /**
     * <p>
     * The S3 location of the application icon.
     * </p>
     * 
     * @return The S3 location of the application icon.
     */
    public final S3Location iconS3Location() {
        return iconS3Location;
    }

    /**
     * <p>
     * The platforms on which the application can run.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlatforms} method.
     * </p>
     * 
     * @return The platforms on which the application can run.
     */
    public final List<PlatformType> platforms() {
        return PlatformsCopier.copyStringToEnum(platforms);
    }

    /**
     * For responses, this returns true if the service returned a value for the Platforms property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPlatforms() {
        return platforms != null && !(platforms instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The platforms on which the application can run.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlatforms} method.
     * </p>
     * 
     * @return The platforms on which the application can run.
     */
    public final List<String> platformsAsStrings() {
        return platforms;
    }

    /**
     * For responses, this returns true if the service returned a value for the InstanceFamilies property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInstanceFamilies() {
        return instanceFamilies != null && !(instanceFamilies instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The instance families for the application.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceFamilies} method.
     * </p>
     * 
     * @return The instance families for the application.
     */
    public final List<String> instanceFamilies() {
        return instanceFamilies;
    }

    /**
     * <p>
     * The time at which the application was created within the app block.
     * </p>
     * 
     * @return The time at which the application was created within the app block.
     */
    public final Instant createdTime() {
        return createdTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(displayName());
        hashCode = 31 * hashCode + Objects.hashCode(iconURL());
        hashCode = 31 * hashCode + Objects.hashCode(launchPath());
        hashCode = 31 * hashCode + Objects.hashCode(launchParameters());
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetadata() ? metadata() : null);
        hashCode = 31 * hashCode + Objects.hashCode(workingDirectory());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(appBlockArn());
        hashCode = 31 * hashCode + Objects.hashCode(iconS3Location());
        hashCode = 31 * hashCode + Objects.hashCode(hasPlatforms() ? platformsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceFamilies() ? instanceFamilies() : null);
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Application)) {
            return false;
        }
        Application other = (Application) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(displayName(), other.displayName())
                && Objects.equals(iconURL(), other.iconURL()) && Objects.equals(launchPath(), other.launchPath())
                && Objects.equals(launchParameters(), other.launchParameters()) && Objects.equals(enabled(), other.enabled())
                && hasMetadata() == other.hasMetadata() && Objects.equals(metadata(), other.metadata())
                && Objects.equals(workingDirectory(), other.workingDirectory())
                && Objects.equals(description(), other.description()) && Objects.equals(arn(), other.arn())
                && Objects.equals(appBlockArn(), other.appBlockArn()) && Objects.equals(iconS3Location(), other.iconS3Location())
                && hasPlatforms() == other.hasPlatforms() && Objects.equals(platformsAsStrings(), other.platformsAsStrings())
                && hasInstanceFamilies() == other.hasInstanceFamilies()
                && Objects.equals(instanceFamilies(), other.instanceFamilies())
                && Objects.equals(createdTime(), other.createdTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Application").add("Name", name()).add("DisplayName", displayName()).add("IconURL", iconURL())
                .add("LaunchPath", launchPath()).add("LaunchParameters", launchParameters()).add("Enabled", enabled())
                .add("Metadata", hasMetadata() ? metadata() : null).add("WorkingDirectory", workingDirectory())
                .add("Description", description()).add("Arn", arn()).add("AppBlockArn", appBlockArn())
                .add("IconS3Location", iconS3Location()).add("Platforms", hasPlatforms() ? platformsAsStrings() : null)
                .add("InstanceFamilies", hasInstanceFamilies() ? instanceFamilies() : null).add("CreatedTime", createdTime())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "DisplayName":
            return Optional.ofNullable(clazz.cast(displayName()));
        case "IconURL":
            return Optional.ofNullable(clazz.cast(iconURL()));
        case "LaunchPath":
            return Optional.ofNullable(clazz.cast(launchPath()));
        case "LaunchParameters":
            return Optional.ofNullable(clazz.cast(launchParameters()));
        case "Enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "Metadata":
            return Optional.ofNullable(clazz.cast(metadata()));
        case "WorkingDirectory":
            return Optional.ofNullable(clazz.cast(workingDirectory()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "AppBlockArn":
            return Optional.ofNullable(clazz.cast(appBlockArn()));
        case "IconS3Location":
            return Optional.ofNullable(clazz.cast(iconS3Location()));
        case "Platforms":
            return Optional.ofNullable(clazz.cast(platformsAsStrings()));
        case "InstanceFamilies":
            return Optional.ofNullable(clazz.cast(instanceFamilies()));
        case "CreatedTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("DisplayName", DISPLAY_NAME_FIELD);
        map.put("IconURL", ICON_URL_FIELD);
        map.put("LaunchPath", LAUNCH_PATH_FIELD);
        map.put("LaunchParameters", LAUNCH_PARAMETERS_FIELD);
        map.put("Enabled", ENABLED_FIELD);
        map.put("Metadata", METADATA_FIELD);
        map.put("WorkingDirectory", WORKING_DIRECTORY_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("Arn", ARN_FIELD);
        map.put("AppBlockArn", APP_BLOCK_ARN_FIELD);
        map.put("IconS3Location", ICON_S3_LOCATION_FIELD);
        map.put("Platforms", PLATFORMS_FIELD);
        map.put("InstanceFamilies", INSTANCE_FAMILIES_FIELD);
        map.put("CreatedTime", CREATED_TIME_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Application, T> g) {
        return obj -> g.apply((Application) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Application> {
        /**
         * <p>
         * The name of the application.
         * </p>
         * 
         * @param name
         *        The name of the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The application name to display.
         * </p>
         * 
         * @param displayName
         *        The application name to display.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder displayName(String displayName);

        /**
         * <p>
         * The URL for the application icon. This URL might be time-limited.
         * </p>
         * 
         * @param iconURL
         *        The URL for the application icon. This URL might be time-limited.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iconURL(String iconURL);

        /**
         * <p>
         * The path to the application executable in the instance.
         * </p>
         * 
         * @param launchPath
         *        The path to the application executable in the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchPath(String launchPath);

        /**
         * <p>
         * The arguments that are passed to the application at launch.
         * </p>
         * 
         * @param launchParameters
         *        The arguments that are passed to the application at launch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchParameters(String launchParameters);

        /**
         * <p>
         * If there is a problem, the application can be disabled after image creation.
         * </p>
         * 
         * @param enabled
         *        If there is a problem, the application can be disabled after image creation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * Additional attributes that describe the application.
         * </p>
         * 
         * @param metadata
         *        Additional attributes that describe the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metadata(Map<String, String> metadata);

        /**
         * <p>
         * The working directory for the application.
         * </p>
         * 
         * @param workingDirectory
         *        The working directory for the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workingDirectory(String workingDirectory);

        /**
         * <p>
         * The description of the application.
         * </p>
         * 
         * @param description
         *        The description of the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The ARN of the application.
         * </p>
         * 
         * @param arn
         *        The ARN of the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The app block ARN of the application.
         * </p>
         * 
         * @param appBlockArn
         *        The app block ARN of the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appBlockArn(String appBlockArn);

        /**
         * <p>
         * The S3 location of the application icon.
         * </p>
         * 
         * @param iconS3Location
         *        The S3 location of the application icon.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iconS3Location(S3Location iconS3Location);

        /**
         * <p>
         * The S3 location of the application icon.
         * </p>
         * This is a convenience method that creates an instance of the {@link S3Location.Builder} avoiding the need to
         * create one manually via {@link S3Location#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link S3Location.Builder#build()} is called immediately and its result
         * is passed to {@link #iconS3Location(S3Location)}.
         * 
         * @param iconS3Location
         *        a consumer that will call methods on {@link S3Location.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #iconS3Location(S3Location)
         */
        default Builder iconS3Location(Consumer<S3Location.Builder> iconS3Location) {
            return iconS3Location(S3Location.builder().applyMutation(iconS3Location).build());
        }

        /**
         * <p>
         * The platforms on which the application can run.
         * </p>
         * 
         * @param platforms
         *        The platforms on which the application can run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformsWithStrings(Collection<String> platforms);

        /**
         * <p>
         * The platforms on which the application can run.
         * </p>
         * 
         * @param platforms
         *        The platforms on which the application can run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformsWithStrings(String... platforms);

        /**
         * <p>
         * The platforms on which the application can run.
         * </p>
         * 
         * @param platforms
         *        The platforms on which the application can run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platforms(Collection<PlatformType> platforms);

        /**
         * <p>
         * The platforms on which the application can run.
         * </p>
         * 
         * @param platforms
         *        The platforms on which the application can run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platforms(PlatformType... platforms);

        /**
         * <p>
         * The instance families for the application.
         * </p>
         * 
         * @param instanceFamilies
         *        The instance families for the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceFamilies(Collection<String> instanceFamilies);

        /**
         * <p>
         * The instance families for the application.
         * </p>
         * 
         * @param instanceFamilies
         *        The instance families for the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceFamilies(String... instanceFamilies);

        /**
         * <p>
         * The time at which the application was created within the app block.
         * </p>
         * 
         * @param createdTime
         *        The time at which the application was created within the app block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String displayName;

        private String iconURL;

        private String launchPath;

        private String launchParameters;

        private Boolean enabled;

        private Map<String, String> metadata = DefaultSdkAutoConstructMap.getInstance();

        private String workingDirectory;

        private String description;

        private String arn;

        private String appBlockArn;

        private S3Location iconS3Location;

        private List<String> platforms = DefaultSdkAutoConstructList.getInstance();

        private List<String> instanceFamilies = DefaultSdkAutoConstructList.getInstance();

        private Instant createdTime;

        private BuilderImpl() {
        }

        private BuilderImpl(Application model) {
            name(model.name);
            displayName(model.displayName);
            iconURL(model.iconURL);
            launchPath(model.launchPath);
            launchParameters(model.launchParameters);
            enabled(model.enabled);
            metadata(model.metadata);
            workingDirectory(model.workingDirectory);
            description(model.description);
            arn(model.arn);
            appBlockArn(model.appBlockArn);
            iconS3Location(model.iconS3Location);
            platformsWithStrings(model.platforms);
            instanceFamilies(model.instanceFamilies);
            createdTime(model.createdTime);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDisplayName() {
            return displayName;
        }

        public final void setDisplayName(String displayName) {
            this.displayName = displayName;
        }

        @Override
        public final Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public final String getIconURL() {
            return iconURL;
        }

        public final void setIconURL(String iconURL) {
            this.iconURL = iconURL;
        }

        @Override
        public final Builder iconURL(String iconURL) {
            this.iconURL = iconURL;
            return this;
        }

        public final String getLaunchPath() {
            return launchPath;
        }

        public final void setLaunchPath(String launchPath) {
            this.launchPath = launchPath;
        }

        @Override
        public final Builder launchPath(String launchPath) {
            this.launchPath = launchPath;
            return this;
        }

        public final String getLaunchParameters() {
            return launchParameters;
        }

        public final void setLaunchParameters(String launchParameters) {
            this.launchParameters = launchParameters;
        }

        @Override
        public final Builder launchParameters(String launchParameters) {
            this.launchParameters = launchParameters;
            return this;
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final Map<String, String> getMetadata() {
            if (metadata instanceof SdkAutoConstructMap) {
                return null;
            }
            return metadata;
        }

        public final void setMetadata(Map<String, String> metadata) {
            this.metadata = MetadataCopier.copy(metadata);
        }

        @Override
        public final Builder metadata(Map<String, String> metadata) {
            this.metadata = MetadataCopier.copy(metadata);
            return this;
        }

        public final String getWorkingDirectory() {
            return workingDirectory;
        }

        public final void setWorkingDirectory(String workingDirectory) {
            this.workingDirectory = workingDirectory;
        }

        @Override
        public final Builder workingDirectory(String workingDirectory) {
            this.workingDirectory = workingDirectory;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getAppBlockArn() {
            return appBlockArn;
        }

        public final void setAppBlockArn(String appBlockArn) {
            this.appBlockArn = appBlockArn;
        }

        @Override
        public final Builder appBlockArn(String appBlockArn) {
            this.appBlockArn = appBlockArn;
            return this;
        }

        public final S3Location.Builder getIconS3Location() {
            return iconS3Location != null ? iconS3Location.toBuilder() : null;
        }

        public final void setIconS3Location(S3Location.BuilderImpl iconS3Location) {
            this.iconS3Location = iconS3Location != null ? iconS3Location.build() : null;
        }

        @Override
        public final Builder iconS3Location(S3Location iconS3Location) {
            this.iconS3Location = iconS3Location;
            return this;
        }

        public final Collection<String> getPlatforms() {
            if (platforms instanceof SdkAutoConstructList) {
                return null;
            }
            return platforms;
        }

        public final void setPlatforms(Collection<String> platforms) {
            this.platforms = PlatformsCopier.copy(platforms);
        }

        @Override
        public final Builder platformsWithStrings(Collection<String> platforms) {
            this.platforms = PlatformsCopier.copy(platforms);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder platformsWithStrings(String... platforms) {
            platformsWithStrings(Arrays.asList(platforms));
            return this;
        }

        @Override
        public final Builder platforms(Collection<PlatformType> platforms) {
            this.platforms = PlatformsCopier.copyEnumToString(platforms);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder platforms(PlatformType... platforms) {
            platforms(Arrays.asList(platforms));
            return this;
        }

        public final Collection<String> getInstanceFamilies() {
            if (instanceFamilies instanceof SdkAutoConstructList) {
                return null;
            }
            return instanceFamilies;
        }

        public final void setInstanceFamilies(Collection<String> instanceFamilies) {
            this.instanceFamilies = StringListCopier.copy(instanceFamilies);
        }

        @Override
        public final Builder instanceFamilies(Collection<String> instanceFamilies) {
            this.instanceFamilies = StringListCopier.copy(instanceFamilies);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceFamilies(String... instanceFamilies) {
            instanceFamilies(Arrays.asList(instanceFamilies));
            return this;
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        @Override
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        @Override
        public Application build() {
            return new Application(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
