/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appsync.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the configuration of a source API. A source API is a GraphQL API that is linked to a merged API. There can
 * be multiple source APIs attached to each merged API. When linked to a merged API, the source API's schema, data
 * sources, and resolvers will be combined with other linked source API data to form a new, singular API.
 * </p>
 * <p>
 * Source APIs can originate from your account or from other accounts via Amazon Web Services Resource Access Manager.
 * For more information about sharing resources from other accounts, see <a
 * href="https://docs.aws.amazon.com/ram/latest/userguide/what-is.html">What is Amazon Web Services Resource Access
 * Manager?</a> in the <i>Amazon Web Services Resource Access Manager</i> guide.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SourceApiAssociation implements SdkPojo, Serializable,
        ToCopyableBuilder<SourceApiAssociation.Builder, SourceApiAssociation> {
    private static final SdkField<String> ASSOCIATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("associationId").getter(getter(SourceApiAssociation::associationId))
            .setter(setter(Builder::associationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("associationId").build()).build();

    private static final SdkField<String> ASSOCIATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("associationArn").getter(getter(SourceApiAssociation::associationArn))
            .setter(setter(Builder::associationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("associationArn").build()).build();

    private static final SdkField<String> SOURCE_API_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sourceApiId").getter(getter(SourceApiAssociation::sourceApiId)).setter(setter(Builder::sourceApiId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceApiId").build()).build();

    private static final SdkField<String> SOURCE_API_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sourceApiArn").getter(getter(SourceApiAssociation::sourceApiArn)).setter(setter(Builder::sourceApiArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceApiArn").build()).build();

    private static final SdkField<String> MERGED_API_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("mergedApiArn").getter(getter(SourceApiAssociation::mergedApiArn)).setter(setter(Builder::mergedApiArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("mergedApiArn").build()).build();

    private static final SdkField<String> MERGED_API_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("mergedApiId").getter(getter(SourceApiAssociation::mergedApiId)).setter(setter(Builder::mergedApiId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("mergedApiId").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(SourceApiAssociation::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<SourceApiAssociationConfig> SOURCE_API_ASSOCIATION_CONFIG_FIELD = SdkField
            .<SourceApiAssociationConfig> builder(MarshallingType.SDK_POJO)
            .memberName("sourceApiAssociationConfig")
            .getter(getter(SourceApiAssociation::sourceApiAssociationConfig))
            .setter(setter(Builder::sourceApiAssociationConfig))
            .constructor(SourceApiAssociationConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceApiAssociationConfig").build())
            .build();

    private static final SdkField<String> SOURCE_API_ASSOCIATION_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("sourceApiAssociationStatus")
            .getter(getter(SourceApiAssociation::sourceApiAssociationStatusAsString))
            .setter(setter(Builder::sourceApiAssociationStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceApiAssociationStatus").build())
            .build();

    private static final SdkField<String> SOURCE_API_ASSOCIATION_STATUS_DETAIL_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("sourceApiAssociationStatusDetail")
            .getter(getter(SourceApiAssociation::sourceApiAssociationStatusDetail))
            .setter(setter(Builder::sourceApiAssociationStatusDetail))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceApiAssociationStatusDetail")
                    .build()).build();

    private static final SdkField<Instant> LAST_SUCCESSFUL_MERGE_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastSuccessfulMergeDate").getter(getter(SourceApiAssociation::lastSuccessfulMergeDate))
            .setter(setter(Builder::lastSuccessfulMergeDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastSuccessfulMergeDate").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ASSOCIATION_ID_FIELD,
            ASSOCIATION_ARN_FIELD, SOURCE_API_ID_FIELD, SOURCE_API_ARN_FIELD, MERGED_API_ARN_FIELD, MERGED_API_ID_FIELD,
            DESCRIPTION_FIELD, SOURCE_API_ASSOCIATION_CONFIG_FIELD, SOURCE_API_ASSOCIATION_STATUS_FIELD,
            SOURCE_API_ASSOCIATION_STATUS_DETAIL_FIELD, LAST_SUCCESSFUL_MERGE_DATE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String associationId;

    private final String associationArn;

    private final String sourceApiId;

    private final String sourceApiArn;

    private final String mergedApiArn;

    private final String mergedApiId;

    private final String description;

    private final SourceApiAssociationConfig sourceApiAssociationConfig;

    private final String sourceApiAssociationStatus;

    private final String sourceApiAssociationStatusDetail;

    private final Instant lastSuccessfulMergeDate;

    private SourceApiAssociation(BuilderImpl builder) {
        this.associationId = builder.associationId;
        this.associationArn = builder.associationArn;
        this.sourceApiId = builder.sourceApiId;
        this.sourceApiArn = builder.sourceApiArn;
        this.mergedApiArn = builder.mergedApiArn;
        this.mergedApiId = builder.mergedApiId;
        this.description = builder.description;
        this.sourceApiAssociationConfig = builder.sourceApiAssociationConfig;
        this.sourceApiAssociationStatus = builder.sourceApiAssociationStatus;
        this.sourceApiAssociationStatusDetail = builder.sourceApiAssociationStatusDetail;
        this.lastSuccessfulMergeDate = builder.lastSuccessfulMergeDate;
    }

    /**
     * <p>
     * The ID generated by the AppSync service for the source API association.
     * </p>
     * 
     * @return The ID generated by the AppSync service for the source API association.
     */
    public final String associationId() {
        return associationId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the source API association.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the source API association.
     */
    public final String associationArn() {
        return associationArn;
    }

    /**
     * <p>
     * The ID of the AppSync source API.
     * </p>
     * 
     * @return The ID of the AppSync source API.
     */
    public final String sourceApiId() {
        return sourceApiId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the AppSync source API.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the AppSync source API.
     */
    public final String sourceApiArn() {
        return sourceApiArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the AppSync Merged API.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the AppSync Merged API.
     */
    public final String mergedApiArn() {
        return mergedApiArn;
    }

    /**
     * <p>
     * The ID of the AppSync Merged API.
     * </p>
     * 
     * @return The ID of the AppSync Merged API.
     */
    public final String mergedApiId() {
        return mergedApiId;
    }

    /**
     * <p>
     * The description field.
     * </p>
     * 
     * @return The description field.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The <code>SourceApiAssociationConfig</code> object data.
     * </p>
     * 
     * @return The <code>SourceApiAssociationConfig</code> object data.
     */
    public final SourceApiAssociationConfig sourceApiAssociationConfig() {
        return sourceApiAssociationConfig;
    }

    /**
     * <p>
     * The state of the source API association.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #sourceApiAssociationStatus} will return {@link SourceApiAssociationStatus#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #sourceApiAssociationStatusAsString}.
     * </p>
     * 
     * @return The state of the source API association.
     * @see SourceApiAssociationStatus
     */
    public final SourceApiAssociationStatus sourceApiAssociationStatus() {
        return SourceApiAssociationStatus.fromValue(sourceApiAssociationStatus);
    }

    /**
     * <p>
     * The state of the source API association.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #sourceApiAssociationStatus} will return {@link SourceApiAssociationStatus#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #sourceApiAssociationStatusAsString}.
     * </p>
     * 
     * @return The state of the source API association.
     * @see SourceApiAssociationStatus
     */
    public final String sourceApiAssociationStatusAsString() {
        return sourceApiAssociationStatus;
    }

    /**
     * <p>
     * The detailed message related to the current state of the source API association.
     * </p>
     * 
     * @return The detailed message related to the current state of the source API association.
     */
    public final String sourceApiAssociationStatusDetail() {
        return sourceApiAssociationStatusDetail;
    }

    /**
     * <p>
     * The datetime value of the last successful merge of the source API association. The result will be in UTC format
     * and your local time zone.
     * </p>
     * 
     * @return The datetime value of the last successful merge of the source API association. The result will be in UTC
     *         format and your local time zone.
     */
    public final Instant lastSuccessfulMergeDate() {
        return lastSuccessfulMergeDate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(associationId());
        hashCode = 31 * hashCode + Objects.hashCode(associationArn());
        hashCode = 31 * hashCode + Objects.hashCode(sourceApiId());
        hashCode = 31 * hashCode + Objects.hashCode(sourceApiArn());
        hashCode = 31 * hashCode + Objects.hashCode(mergedApiArn());
        hashCode = 31 * hashCode + Objects.hashCode(mergedApiId());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(sourceApiAssociationConfig());
        hashCode = 31 * hashCode + Objects.hashCode(sourceApiAssociationStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(sourceApiAssociationStatusDetail());
        hashCode = 31 * hashCode + Objects.hashCode(lastSuccessfulMergeDate());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SourceApiAssociation)) {
            return false;
        }
        SourceApiAssociation other = (SourceApiAssociation) obj;
        return Objects.equals(associationId(), other.associationId()) && Objects.equals(associationArn(), other.associationArn())
                && Objects.equals(sourceApiId(), other.sourceApiId()) && Objects.equals(sourceApiArn(), other.sourceApiArn())
                && Objects.equals(mergedApiArn(), other.mergedApiArn()) && Objects.equals(mergedApiId(), other.mergedApiId())
                && Objects.equals(description(), other.description())
                && Objects.equals(sourceApiAssociationConfig(), other.sourceApiAssociationConfig())
                && Objects.equals(sourceApiAssociationStatusAsString(), other.sourceApiAssociationStatusAsString())
                && Objects.equals(sourceApiAssociationStatusDetail(), other.sourceApiAssociationStatusDetail())
                && Objects.equals(lastSuccessfulMergeDate(), other.lastSuccessfulMergeDate());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SourceApiAssociation").add("AssociationId", associationId())
                .add("AssociationArn", associationArn()).add("SourceApiId", sourceApiId()).add("SourceApiArn", sourceApiArn())
                .add("MergedApiArn", mergedApiArn()).add("MergedApiId", mergedApiId()).add("Description", description())
                .add("SourceApiAssociationConfig", sourceApiAssociationConfig())
                .add("SourceApiAssociationStatus", sourceApiAssociationStatusAsString())
                .add("SourceApiAssociationStatusDetail", sourceApiAssociationStatusDetail())
                .add("LastSuccessfulMergeDate", lastSuccessfulMergeDate()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "associationId":
            return Optional.ofNullable(clazz.cast(associationId()));
        case "associationArn":
            return Optional.ofNullable(clazz.cast(associationArn()));
        case "sourceApiId":
            return Optional.ofNullable(clazz.cast(sourceApiId()));
        case "sourceApiArn":
            return Optional.ofNullable(clazz.cast(sourceApiArn()));
        case "mergedApiArn":
            return Optional.ofNullable(clazz.cast(mergedApiArn()));
        case "mergedApiId":
            return Optional.ofNullable(clazz.cast(mergedApiId()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "sourceApiAssociationConfig":
            return Optional.ofNullable(clazz.cast(sourceApiAssociationConfig()));
        case "sourceApiAssociationStatus":
            return Optional.ofNullable(clazz.cast(sourceApiAssociationStatusAsString()));
        case "sourceApiAssociationStatusDetail":
            return Optional.ofNullable(clazz.cast(sourceApiAssociationStatusDetail()));
        case "lastSuccessfulMergeDate":
            return Optional.ofNullable(clazz.cast(lastSuccessfulMergeDate()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("associationId", ASSOCIATION_ID_FIELD);
        map.put("associationArn", ASSOCIATION_ARN_FIELD);
        map.put("sourceApiId", SOURCE_API_ID_FIELD);
        map.put("sourceApiArn", SOURCE_API_ARN_FIELD);
        map.put("mergedApiArn", MERGED_API_ARN_FIELD);
        map.put("mergedApiId", MERGED_API_ID_FIELD);
        map.put("description", DESCRIPTION_FIELD);
        map.put("sourceApiAssociationConfig", SOURCE_API_ASSOCIATION_CONFIG_FIELD);
        map.put("sourceApiAssociationStatus", SOURCE_API_ASSOCIATION_STATUS_FIELD);
        map.put("sourceApiAssociationStatusDetail", SOURCE_API_ASSOCIATION_STATUS_DETAIL_FIELD);
        map.put("lastSuccessfulMergeDate", LAST_SUCCESSFUL_MERGE_DATE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SourceApiAssociation, T> g) {
        return obj -> g.apply((SourceApiAssociation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SourceApiAssociation> {
        /**
         * <p>
         * The ID generated by the AppSync service for the source API association.
         * </p>
         * 
         * @param associationId
         *        The ID generated by the AppSync service for the source API association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationId(String associationId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the source API association.
         * </p>
         * 
         * @param associationArn
         *        The Amazon Resource Name (ARN) of the source API association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationArn(String associationArn);

        /**
         * <p>
         * The ID of the AppSync source API.
         * </p>
         * 
         * @param sourceApiId
         *        The ID of the AppSync source API.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceApiId(String sourceApiId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the AppSync source API.
         * </p>
         * 
         * @param sourceApiArn
         *        The Amazon Resource Name (ARN) of the AppSync source API.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceApiArn(String sourceApiArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the AppSync Merged API.
         * </p>
         * 
         * @param mergedApiArn
         *        The Amazon Resource Name (ARN) of the AppSync Merged API.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mergedApiArn(String mergedApiArn);

        /**
         * <p>
         * The ID of the AppSync Merged API.
         * </p>
         * 
         * @param mergedApiId
         *        The ID of the AppSync Merged API.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mergedApiId(String mergedApiId);

        /**
         * <p>
         * The description field.
         * </p>
         * 
         * @param description
         *        The description field.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The <code>SourceApiAssociationConfig</code> object data.
         * </p>
         * 
         * @param sourceApiAssociationConfig
         *        The <code>SourceApiAssociationConfig</code> object data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceApiAssociationConfig(SourceApiAssociationConfig sourceApiAssociationConfig);

        /**
         * <p>
         * The <code>SourceApiAssociationConfig</code> object data.
         * </p>
         * This is a convenience method that creates an instance of the {@link SourceApiAssociationConfig.Builder}
         * avoiding the need to create one manually via {@link SourceApiAssociationConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SourceApiAssociationConfig.Builder#build()} is called immediately
         * and its result is passed to {@link #sourceApiAssociationConfig(SourceApiAssociationConfig)}.
         * 
         * @param sourceApiAssociationConfig
         *        a consumer that will call methods on {@link SourceApiAssociationConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceApiAssociationConfig(SourceApiAssociationConfig)
         */
        default Builder sourceApiAssociationConfig(Consumer<SourceApiAssociationConfig.Builder> sourceApiAssociationConfig) {
            return sourceApiAssociationConfig(SourceApiAssociationConfig.builder().applyMutation(sourceApiAssociationConfig)
                    .build());
        }

        /**
         * <p>
         * The state of the source API association.
         * </p>
         * 
         * @param sourceApiAssociationStatus
         *        The state of the source API association.
         * @see SourceApiAssociationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SourceApiAssociationStatus
         */
        Builder sourceApiAssociationStatus(String sourceApiAssociationStatus);

        /**
         * <p>
         * The state of the source API association.
         * </p>
         * 
         * @param sourceApiAssociationStatus
         *        The state of the source API association.
         * @see SourceApiAssociationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SourceApiAssociationStatus
         */
        Builder sourceApiAssociationStatus(SourceApiAssociationStatus sourceApiAssociationStatus);

        /**
         * <p>
         * The detailed message related to the current state of the source API association.
         * </p>
         * 
         * @param sourceApiAssociationStatusDetail
         *        The detailed message related to the current state of the source API association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceApiAssociationStatusDetail(String sourceApiAssociationStatusDetail);

        /**
         * <p>
         * The datetime value of the last successful merge of the source API association. The result will be in UTC
         * format and your local time zone.
         * </p>
         * 
         * @param lastSuccessfulMergeDate
         *        The datetime value of the last successful merge of the source API association. The result will be in
         *        UTC format and your local time zone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSuccessfulMergeDate(Instant lastSuccessfulMergeDate);
    }

    static final class BuilderImpl implements Builder {
        private String associationId;

        private String associationArn;

        private String sourceApiId;

        private String sourceApiArn;

        private String mergedApiArn;

        private String mergedApiId;

        private String description;

        private SourceApiAssociationConfig sourceApiAssociationConfig;

        private String sourceApiAssociationStatus;

        private String sourceApiAssociationStatusDetail;

        private Instant lastSuccessfulMergeDate;

        private BuilderImpl() {
        }

        private BuilderImpl(SourceApiAssociation model) {
            associationId(model.associationId);
            associationArn(model.associationArn);
            sourceApiId(model.sourceApiId);
            sourceApiArn(model.sourceApiArn);
            mergedApiArn(model.mergedApiArn);
            mergedApiId(model.mergedApiId);
            description(model.description);
            sourceApiAssociationConfig(model.sourceApiAssociationConfig);
            sourceApiAssociationStatus(model.sourceApiAssociationStatus);
            sourceApiAssociationStatusDetail(model.sourceApiAssociationStatusDetail);
            lastSuccessfulMergeDate(model.lastSuccessfulMergeDate);
        }

        public final String getAssociationId() {
            return associationId;
        }

        public final void setAssociationId(String associationId) {
            this.associationId = associationId;
        }

        @Override
        public final Builder associationId(String associationId) {
            this.associationId = associationId;
            return this;
        }

        public final String getAssociationArn() {
            return associationArn;
        }

        public final void setAssociationArn(String associationArn) {
            this.associationArn = associationArn;
        }

        @Override
        public final Builder associationArn(String associationArn) {
            this.associationArn = associationArn;
            return this;
        }

        public final String getSourceApiId() {
            return sourceApiId;
        }

        public final void setSourceApiId(String sourceApiId) {
            this.sourceApiId = sourceApiId;
        }

        @Override
        public final Builder sourceApiId(String sourceApiId) {
            this.sourceApiId = sourceApiId;
            return this;
        }

        public final String getSourceApiArn() {
            return sourceApiArn;
        }

        public final void setSourceApiArn(String sourceApiArn) {
            this.sourceApiArn = sourceApiArn;
        }

        @Override
        public final Builder sourceApiArn(String sourceApiArn) {
            this.sourceApiArn = sourceApiArn;
            return this;
        }

        public final String getMergedApiArn() {
            return mergedApiArn;
        }

        public final void setMergedApiArn(String mergedApiArn) {
            this.mergedApiArn = mergedApiArn;
        }

        @Override
        public final Builder mergedApiArn(String mergedApiArn) {
            this.mergedApiArn = mergedApiArn;
            return this;
        }

        public final String getMergedApiId() {
            return mergedApiId;
        }

        public final void setMergedApiId(String mergedApiId) {
            this.mergedApiId = mergedApiId;
        }

        @Override
        public final Builder mergedApiId(String mergedApiId) {
            this.mergedApiId = mergedApiId;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final SourceApiAssociationConfig.Builder getSourceApiAssociationConfig() {
            return sourceApiAssociationConfig != null ? sourceApiAssociationConfig.toBuilder() : null;
        }

        public final void setSourceApiAssociationConfig(SourceApiAssociationConfig.BuilderImpl sourceApiAssociationConfig) {
            this.sourceApiAssociationConfig = sourceApiAssociationConfig != null ? sourceApiAssociationConfig.build() : null;
        }

        @Override
        public final Builder sourceApiAssociationConfig(SourceApiAssociationConfig sourceApiAssociationConfig) {
            this.sourceApiAssociationConfig = sourceApiAssociationConfig;
            return this;
        }

        public final String getSourceApiAssociationStatus() {
            return sourceApiAssociationStatus;
        }

        public final void setSourceApiAssociationStatus(String sourceApiAssociationStatus) {
            this.sourceApiAssociationStatus = sourceApiAssociationStatus;
        }

        @Override
        public final Builder sourceApiAssociationStatus(String sourceApiAssociationStatus) {
            this.sourceApiAssociationStatus = sourceApiAssociationStatus;
            return this;
        }

        @Override
        public final Builder sourceApiAssociationStatus(SourceApiAssociationStatus sourceApiAssociationStatus) {
            this.sourceApiAssociationStatus(sourceApiAssociationStatus == null ? null : sourceApiAssociationStatus.toString());
            return this;
        }

        public final String getSourceApiAssociationStatusDetail() {
            return sourceApiAssociationStatusDetail;
        }

        public final void setSourceApiAssociationStatusDetail(String sourceApiAssociationStatusDetail) {
            this.sourceApiAssociationStatusDetail = sourceApiAssociationStatusDetail;
        }

        @Override
        public final Builder sourceApiAssociationStatusDetail(String sourceApiAssociationStatusDetail) {
            this.sourceApiAssociationStatusDetail = sourceApiAssociationStatusDetail;
            return this;
        }

        public final Instant getLastSuccessfulMergeDate() {
            return lastSuccessfulMergeDate;
        }

        public final void setLastSuccessfulMergeDate(Instant lastSuccessfulMergeDate) {
            this.lastSuccessfulMergeDate = lastSuccessfulMergeDate;
        }

        @Override
        public final Builder lastSuccessfulMergeDate(Instant lastSuccessfulMergeDate) {
            this.lastSuccessfulMergeDate = lastSuccessfulMergeDate;
            return this;
        }

        @Override
        public SourceApiAssociation build() {
            return new SourceApiAssociation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
