/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apptest;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.apptest.internal.AppTestServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.apptest.model.AccessDeniedException;
import software.amazon.awssdk.services.apptest.model.AppTestException;
import software.amazon.awssdk.services.apptest.model.ConflictException;
import software.amazon.awssdk.services.apptest.model.CreateTestCaseRequest;
import software.amazon.awssdk.services.apptest.model.CreateTestCaseResponse;
import software.amazon.awssdk.services.apptest.model.CreateTestConfigurationRequest;
import software.amazon.awssdk.services.apptest.model.CreateTestConfigurationResponse;
import software.amazon.awssdk.services.apptest.model.CreateTestSuiteRequest;
import software.amazon.awssdk.services.apptest.model.CreateTestSuiteResponse;
import software.amazon.awssdk.services.apptest.model.DeleteTestCaseRequest;
import software.amazon.awssdk.services.apptest.model.DeleteTestCaseResponse;
import software.amazon.awssdk.services.apptest.model.DeleteTestConfigurationRequest;
import software.amazon.awssdk.services.apptest.model.DeleteTestConfigurationResponse;
import software.amazon.awssdk.services.apptest.model.DeleteTestRunRequest;
import software.amazon.awssdk.services.apptest.model.DeleteTestRunResponse;
import software.amazon.awssdk.services.apptest.model.DeleteTestSuiteRequest;
import software.amazon.awssdk.services.apptest.model.DeleteTestSuiteResponse;
import software.amazon.awssdk.services.apptest.model.GetTestCaseRequest;
import software.amazon.awssdk.services.apptest.model.GetTestCaseResponse;
import software.amazon.awssdk.services.apptest.model.GetTestConfigurationRequest;
import software.amazon.awssdk.services.apptest.model.GetTestConfigurationResponse;
import software.amazon.awssdk.services.apptest.model.GetTestRunStepRequest;
import software.amazon.awssdk.services.apptest.model.GetTestRunStepResponse;
import software.amazon.awssdk.services.apptest.model.GetTestSuiteRequest;
import software.amazon.awssdk.services.apptest.model.GetTestSuiteResponse;
import software.amazon.awssdk.services.apptest.model.InternalServerException;
import software.amazon.awssdk.services.apptest.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.apptest.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.apptest.model.ListTestCasesRequest;
import software.amazon.awssdk.services.apptest.model.ListTestCasesResponse;
import software.amazon.awssdk.services.apptest.model.ListTestConfigurationsRequest;
import software.amazon.awssdk.services.apptest.model.ListTestConfigurationsResponse;
import software.amazon.awssdk.services.apptest.model.ListTestRunStepsRequest;
import software.amazon.awssdk.services.apptest.model.ListTestRunStepsResponse;
import software.amazon.awssdk.services.apptest.model.ListTestRunTestCasesRequest;
import software.amazon.awssdk.services.apptest.model.ListTestRunTestCasesResponse;
import software.amazon.awssdk.services.apptest.model.ListTestRunsRequest;
import software.amazon.awssdk.services.apptest.model.ListTestRunsResponse;
import software.amazon.awssdk.services.apptest.model.ListTestSuitesRequest;
import software.amazon.awssdk.services.apptest.model.ListTestSuitesResponse;
import software.amazon.awssdk.services.apptest.model.ResourceNotFoundException;
import software.amazon.awssdk.services.apptest.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.apptest.model.StartTestRunRequest;
import software.amazon.awssdk.services.apptest.model.StartTestRunResponse;
import software.amazon.awssdk.services.apptest.model.TagResourceRequest;
import software.amazon.awssdk.services.apptest.model.TagResourceResponse;
import software.amazon.awssdk.services.apptest.model.ThrottlingException;
import software.amazon.awssdk.services.apptest.model.UntagResourceRequest;
import software.amazon.awssdk.services.apptest.model.UntagResourceResponse;
import software.amazon.awssdk.services.apptest.model.UpdateTestCaseRequest;
import software.amazon.awssdk.services.apptest.model.UpdateTestCaseResponse;
import software.amazon.awssdk.services.apptest.model.UpdateTestConfigurationRequest;
import software.amazon.awssdk.services.apptest.model.UpdateTestConfigurationResponse;
import software.amazon.awssdk.services.apptest.model.UpdateTestSuiteRequest;
import software.amazon.awssdk.services.apptest.model.UpdateTestSuiteResponse;
import software.amazon.awssdk.services.apptest.model.ValidationException;
import software.amazon.awssdk.services.apptest.transform.CreateTestCaseRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.CreateTestConfigurationRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.CreateTestSuiteRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.DeleteTestCaseRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.DeleteTestConfigurationRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.DeleteTestRunRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.DeleteTestSuiteRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.GetTestCaseRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.GetTestConfigurationRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.GetTestRunStepRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.GetTestSuiteRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.ListTestCasesRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.ListTestConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.ListTestRunStepsRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.ListTestRunTestCasesRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.ListTestRunsRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.ListTestSuitesRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.StartTestRunRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.UpdateTestCaseRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.UpdateTestConfigurationRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.UpdateTestSuiteRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AppTestAsyncClient}.
 *
 * @see AppTestAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAppTestAsyncClient implements AppTestAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAppTestAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAppTestAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a test case.
     * </p>
     *
     * @param createTestCaseRequest
     * @return A Java Future containing the result of the CreateTestCase operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException One or more quotas for AWS Application Testing exceeds the limit.</li>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>ConflictException The parameters provided in the request conflict with existing resources.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.CreateTestCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/CreateTestCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTestCaseResponse> createTestCase(CreateTestCaseRequest createTestCaseRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTestCaseRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTestCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTestCase");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTestCaseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateTestCaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTestCaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTestCaseRequest, CreateTestCaseResponse>()
                            .withOperationName("CreateTestCase").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateTestCaseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createTestCaseRequest));
            CompletableFuture<CreateTestCaseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a test configuration.
     * </p>
     *
     * @param createTestConfigurationRequest
     * @return A Java Future containing the result of the CreateTestConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException One or more quotas for AWS Application Testing exceeds the limit.</li>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>ConflictException The parameters provided in the request conflict with existing resources.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.CreateTestConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/CreateTestConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTestConfigurationResponse> createTestConfiguration(
            CreateTestConfigurationRequest createTestConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTestConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTestConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTestConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTestConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateTestConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTestConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTestConfigurationRequest, CreateTestConfigurationResponse>()
                            .withOperationName("CreateTestConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateTestConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createTestConfigurationRequest));
            CompletableFuture<CreateTestConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a test suite.
     * </p>
     *
     * @param createTestSuiteRequest
     * @return A Java Future containing the result of the CreateTestSuite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException One or more quotas for AWS Application Testing exceeds the limit.</li>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>ConflictException The parameters provided in the request conflict with existing resources.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.CreateTestSuite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/CreateTestSuite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTestSuiteResponse> createTestSuite(CreateTestSuiteRequest createTestSuiteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTestSuiteRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTestSuiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTestSuite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTestSuiteResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateTestSuiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTestSuiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTestSuiteRequest, CreateTestSuiteResponse>()
                            .withOperationName("CreateTestSuite").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateTestSuiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createTestSuiteRequest));
            CompletableFuture<CreateTestSuiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a test case.
     * </p>
     *
     * @param deleteTestCaseRequest
     * @return A Java Future containing the result of the DeleteTestCase operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>ConflictException The parameters provided in the request conflict with existing resources.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.DeleteTestCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/DeleteTestCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTestCaseResponse> deleteTestCase(DeleteTestCaseRequest deleteTestCaseRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTestCaseRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTestCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTestCase");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTestCaseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteTestCaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTestCaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTestCaseRequest, DeleteTestCaseResponse>()
                            .withOperationName("DeleteTestCase").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteTestCaseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTestCaseRequest));
            CompletableFuture<DeleteTestCaseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a test configuration.
     * </p>
     *
     * @param deleteTestConfigurationRequest
     * @return A Java Future containing the result of the DeleteTestConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException One or more quotas for AWS Application Testing exceeds the limit.</li>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>ConflictException The parameters provided in the request conflict with existing resources.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.DeleteTestConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/DeleteTestConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTestConfigurationResponse> deleteTestConfiguration(
            DeleteTestConfigurationRequest deleteTestConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTestConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTestConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTestConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTestConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteTestConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTestConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTestConfigurationRequest, DeleteTestConfigurationResponse>()
                            .withOperationName("DeleteTestConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteTestConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTestConfigurationRequest));
            CompletableFuture<DeleteTestConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a test run.
     * </p>
     *
     * @param deleteTestRunRequest
     * @return A Java Future containing the result of the DeleteTestRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>ConflictException The parameters provided in the request conflict with existing resources.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.DeleteTestRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/DeleteTestRun" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTestRunResponse> deleteTestRun(DeleteTestRunRequest deleteTestRunRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTestRunRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTestRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTestRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTestRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteTestRunResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTestRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTestRunRequest, DeleteTestRunResponse>()
                            .withOperationName("DeleteTestRun").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteTestRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTestRunRequest));
            CompletableFuture<DeleteTestRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a test suite.
     * </p>
     *
     * @param deleteTestSuiteRequest
     * @return A Java Future containing the result of the DeleteTestSuite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException One or more quotas for AWS Application Testing exceeds the limit.</li>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>ConflictException The parameters provided in the request conflict with existing resources.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.DeleteTestSuite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/DeleteTestSuite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTestSuiteResponse> deleteTestSuite(DeleteTestSuiteRequest deleteTestSuiteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTestSuiteRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTestSuiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTestSuite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTestSuiteResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteTestSuiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTestSuiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTestSuiteRequest, DeleteTestSuiteResponse>()
                            .withOperationName("DeleteTestSuite").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteTestSuiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTestSuiteRequest));
            CompletableFuture<DeleteTestSuiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a test case.
     * </p>
     *
     * @param getTestCaseRequest
     * @return A Java Future containing the result of the GetTestCase operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.GetTestCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/GetTestCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTestCaseResponse> getTestCase(GetTestCaseRequest getTestCaseRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTestCaseRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTestCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTestCase");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTestCaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetTestCaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTestCaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTestCaseRequest, GetTestCaseResponse>()
                            .withOperationName("GetTestCase").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTestCaseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTestCaseRequest));
            CompletableFuture<GetTestCaseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a test configuration.
     * </p>
     *
     * @param getTestConfigurationRequest
     * @return A Java Future containing the result of the GetTestConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.GetTestConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/GetTestConfiguration" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTestConfigurationResponse> getTestConfiguration(
            GetTestConfigurationRequest getTestConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTestConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTestConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTestConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTestConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTestConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTestConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTestConfigurationRequest, GetTestConfigurationResponse>()
                            .withOperationName("GetTestConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTestConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTestConfigurationRequest));
            CompletableFuture<GetTestConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a test run step.
     * </p>
     *
     * @param getTestRunStepRequest
     * @return A Java Future containing the result of the GetTestRunStep operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.GetTestRunStep
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/GetTestRunStep" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTestRunStepResponse> getTestRunStep(GetTestRunStepRequest getTestRunStepRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTestRunStepRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTestRunStepRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTestRunStep");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTestRunStepResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTestRunStepResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTestRunStepResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTestRunStepRequest, GetTestRunStepResponse>()
                            .withOperationName("GetTestRunStep").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTestRunStepRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTestRunStepRequest));
            CompletableFuture<GetTestRunStepResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a test suite.
     * </p>
     *
     * @param getTestSuiteRequest
     * @return A Java Future containing the result of the GetTestSuite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.GetTestSuite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/GetTestSuite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTestSuiteResponse> getTestSuite(GetTestSuiteRequest getTestSuiteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTestSuiteRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTestSuiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTestSuite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTestSuiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetTestSuiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTestSuiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTestSuiteRequest, GetTestSuiteResponse>()
                            .withOperationName("GetTestSuite").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTestSuiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTestSuiteRequest));
            CompletableFuture<GetTestSuiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists test cases.
     * </p>
     *
     * @param listTestCasesRequest
     * @return A Java Future containing the result of the ListTestCases operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.ListTestCases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/ListTestCases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTestCasesResponse> listTestCases(ListTestCasesRequest listTestCasesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTestCasesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTestCasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTestCases");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTestCasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTestCasesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTestCasesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTestCasesRequest, ListTestCasesResponse>()
                            .withOperationName("ListTestCases").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTestCasesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTestCasesRequest));
            CompletableFuture<ListTestCasesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists test configurations.
     * </p>
     *
     * @param listTestConfigurationsRequest
     * @return A Java Future containing the result of the ListTestConfigurations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.ListTestConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/ListTestConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTestConfigurationsResponse> listTestConfigurations(
            ListTestConfigurationsRequest listTestConfigurationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTestConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTestConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTestConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTestConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTestConfigurationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTestConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTestConfigurationsRequest, ListTestConfigurationsResponse>()
                            .withOperationName("ListTestConfigurations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTestConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTestConfigurationsRequest));
            CompletableFuture<ListTestConfigurationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists test run steps.
     * </p>
     *
     * @param listTestRunStepsRequest
     * @return A Java Future containing the result of the ListTestRunSteps operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.ListTestRunSteps
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/ListTestRunSteps" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTestRunStepsResponse> listTestRunSteps(ListTestRunStepsRequest listTestRunStepsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTestRunStepsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTestRunStepsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTestRunSteps");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTestRunStepsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTestRunStepsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTestRunStepsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTestRunStepsRequest, ListTestRunStepsResponse>()
                            .withOperationName("ListTestRunSteps").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTestRunStepsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTestRunStepsRequest));
            CompletableFuture<ListTestRunStepsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists test run test cases.
     * </p>
     *
     * @param listTestRunTestCasesRequest
     * @return A Java Future containing the result of the ListTestRunTestCases operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.ListTestRunTestCases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/ListTestRunTestCases" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTestRunTestCasesResponse> listTestRunTestCases(
            ListTestRunTestCasesRequest listTestRunTestCasesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTestRunTestCasesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTestRunTestCasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTestRunTestCases");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTestRunTestCasesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTestRunTestCasesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTestRunTestCasesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTestRunTestCasesRequest, ListTestRunTestCasesResponse>()
                            .withOperationName("ListTestRunTestCases").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTestRunTestCasesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTestRunTestCasesRequest));
            CompletableFuture<ListTestRunTestCasesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists test runs.
     * </p>
     *
     * @param listTestRunsRequest
     * @return A Java Future containing the result of the ListTestRuns operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.ListTestRuns
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/ListTestRuns" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTestRunsResponse> listTestRuns(ListTestRunsRequest listTestRunsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTestRunsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTestRunsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTestRuns");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTestRunsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTestRunsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTestRunsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTestRunsRequest, ListTestRunsResponse>()
                            .withOperationName("ListTestRuns").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTestRunsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTestRunsRequest));
            CompletableFuture<ListTestRunsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists test suites.
     * </p>
     *
     * @param listTestSuitesRequest
     * @return A Java Future containing the result of the ListTestSuites operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.ListTestSuites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/ListTestSuites" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTestSuitesResponse> listTestSuites(ListTestSuitesRequest listTestSuitesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTestSuitesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTestSuitesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTestSuites");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTestSuitesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTestSuitesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTestSuitesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTestSuitesRequest, ListTestSuitesResponse>()
                            .withOperationName("ListTestSuites").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTestSuitesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTestSuitesRequest));
            CompletableFuture<ListTestSuitesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a test run.
     * </p>
     *
     * @param startTestRunRequest
     * @return A Java Future containing the result of the StartTestRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException One or more quotas for AWS Application Testing exceeds the limit.</li>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>ConflictException The parameters provided in the request conflict with existing resources.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.StartTestRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/StartTestRun" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartTestRunResponse> startTestRun(StartTestRunRequest startTestRunRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startTestRunRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startTestRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartTestRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartTestRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartTestRunResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartTestRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartTestRunRequest, StartTestRunResponse>()
                            .withOperationName("StartTestRun").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartTestRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startTestRunRequest));
            CompletableFuture<StartTestRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Specifies tags of a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException One or more quotas for AWS Application Testing exceeds the limit.</li>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Untags a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a test case.
     * </p>
     *
     * @param updateTestCaseRequest
     * @return A Java Future containing the result of the UpdateTestCase operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>ConflictException The parameters provided in the request conflict with existing resources.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.UpdateTestCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/UpdateTestCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTestCaseResponse> updateTestCase(UpdateTestCaseRequest updateTestCaseRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateTestCaseRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTestCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTestCase");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTestCaseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateTestCaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateTestCaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTestCaseRequest, UpdateTestCaseResponse>()
                            .withOperationName("UpdateTestCase").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateTestCaseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateTestCaseRequest));
            CompletableFuture<UpdateTestCaseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a test configuration.
     * </p>
     *
     * @param updateTestConfigurationRequest
     * @return A Java Future containing the result of the UpdateTestConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>ConflictException The parameters provided in the request conflict with existing resources.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.UpdateTestConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/UpdateTestConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTestConfigurationResponse> updateTestConfiguration(
            UpdateTestConfigurationRequest updateTestConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateTestConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTestConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTestConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTestConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateTestConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateTestConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTestConfigurationRequest, UpdateTestConfigurationResponse>()
                            .withOperationName("UpdateTestConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateTestConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateTestConfigurationRequest));
            CompletableFuture<UpdateTestConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a test suite.
     * </p>
     *
     * @param updateTestSuiteRequest
     * @return A Java Future containing the result of the UpdateTestSuite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The number of requests made exceeds the limit.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException The account or role doesn't have the right permissions to make the request.</li>
     *         <li>ValidationException One or more parameter provided in the request is not valid.</li>
     *         <li>ConflictException The parameters provided in the request conflict with existing resources.</li>
     *         <li>InternalServerException An unexpected error occurred during the processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppTestException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppTestAsyncClient.UpdateTestSuite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/UpdateTestSuite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTestSuiteResponse> updateTestSuite(UpdateTestSuiteRequest updateTestSuiteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateTestSuiteRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTestSuiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTestSuite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTestSuiteResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateTestSuiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateTestSuiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTestSuiteRequest, UpdateTestSuiteResponse>()
                            .withOperationName("UpdateTestSuite").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateTestSuiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateTestSuiteRequest));
            CompletableFuture<UpdateTestSuiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final AppTestServiceClientConfiguration serviceClientConfiguration() {
        return new AppTestServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AppTestException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        AppTestServiceClientConfigurationBuilder serviceConfigBuilder = new AppTestServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
