/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apptest;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.apptest.internal.AppTestServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.apptest.model.AccessDeniedException;
import software.amazon.awssdk.services.apptest.model.AppTestException;
import software.amazon.awssdk.services.apptest.model.ConflictException;
import software.amazon.awssdk.services.apptest.model.CreateTestCaseRequest;
import software.amazon.awssdk.services.apptest.model.CreateTestCaseResponse;
import software.amazon.awssdk.services.apptest.model.CreateTestConfigurationRequest;
import software.amazon.awssdk.services.apptest.model.CreateTestConfigurationResponse;
import software.amazon.awssdk.services.apptest.model.CreateTestSuiteRequest;
import software.amazon.awssdk.services.apptest.model.CreateTestSuiteResponse;
import software.amazon.awssdk.services.apptest.model.DeleteTestCaseRequest;
import software.amazon.awssdk.services.apptest.model.DeleteTestCaseResponse;
import software.amazon.awssdk.services.apptest.model.DeleteTestConfigurationRequest;
import software.amazon.awssdk.services.apptest.model.DeleteTestConfigurationResponse;
import software.amazon.awssdk.services.apptest.model.DeleteTestRunRequest;
import software.amazon.awssdk.services.apptest.model.DeleteTestRunResponse;
import software.amazon.awssdk.services.apptest.model.DeleteTestSuiteRequest;
import software.amazon.awssdk.services.apptest.model.DeleteTestSuiteResponse;
import software.amazon.awssdk.services.apptest.model.GetTestCaseRequest;
import software.amazon.awssdk.services.apptest.model.GetTestCaseResponse;
import software.amazon.awssdk.services.apptest.model.GetTestConfigurationRequest;
import software.amazon.awssdk.services.apptest.model.GetTestConfigurationResponse;
import software.amazon.awssdk.services.apptest.model.GetTestRunStepRequest;
import software.amazon.awssdk.services.apptest.model.GetTestRunStepResponse;
import software.amazon.awssdk.services.apptest.model.GetTestSuiteRequest;
import software.amazon.awssdk.services.apptest.model.GetTestSuiteResponse;
import software.amazon.awssdk.services.apptest.model.InternalServerException;
import software.amazon.awssdk.services.apptest.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.apptest.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.apptest.model.ListTestCasesRequest;
import software.amazon.awssdk.services.apptest.model.ListTestCasesResponse;
import software.amazon.awssdk.services.apptest.model.ListTestConfigurationsRequest;
import software.amazon.awssdk.services.apptest.model.ListTestConfigurationsResponse;
import software.amazon.awssdk.services.apptest.model.ListTestRunStepsRequest;
import software.amazon.awssdk.services.apptest.model.ListTestRunStepsResponse;
import software.amazon.awssdk.services.apptest.model.ListTestRunTestCasesRequest;
import software.amazon.awssdk.services.apptest.model.ListTestRunTestCasesResponse;
import software.amazon.awssdk.services.apptest.model.ListTestRunsRequest;
import software.amazon.awssdk.services.apptest.model.ListTestRunsResponse;
import software.amazon.awssdk.services.apptest.model.ListTestSuitesRequest;
import software.amazon.awssdk.services.apptest.model.ListTestSuitesResponse;
import software.amazon.awssdk.services.apptest.model.ResourceNotFoundException;
import software.amazon.awssdk.services.apptest.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.apptest.model.StartTestRunRequest;
import software.amazon.awssdk.services.apptest.model.StartTestRunResponse;
import software.amazon.awssdk.services.apptest.model.TagResourceRequest;
import software.amazon.awssdk.services.apptest.model.TagResourceResponse;
import software.amazon.awssdk.services.apptest.model.ThrottlingException;
import software.amazon.awssdk.services.apptest.model.UntagResourceRequest;
import software.amazon.awssdk.services.apptest.model.UntagResourceResponse;
import software.amazon.awssdk.services.apptest.model.UpdateTestCaseRequest;
import software.amazon.awssdk.services.apptest.model.UpdateTestCaseResponse;
import software.amazon.awssdk.services.apptest.model.UpdateTestConfigurationRequest;
import software.amazon.awssdk.services.apptest.model.UpdateTestConfigurationResponse;
import software.amazon.awssdk.services.apptest.model.UpdateTestSuiteRequest;
import software.amazon.awssdk.services.apptest.model.UpdateTestSuiteResponse;
import software.amazon.awssdk.services.apptest.model.ValidationException;
import software.amazon.awssdk.services.apptest.transform.CreateTestCaseRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.CreateTestConfigurationRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.CreateTestSuiteRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.DeleteTestCaseRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.DeleteTestConfigurationRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.DeleteTestRunRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.DeleteTestSuiteRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.GetTestCaseRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.GetTestConfigurationRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.GetTestRunStepRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.GetTestSuiteRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.ListTestCasesRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.ListTestConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.ListTestRunStepsRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.ListTestRunTestCasesRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.ListTestRunsRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.ListTestSuitesRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.StartTestRunRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.UpdateTestCaseRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.UpdateTestConfigurationRequestMarshaller;
import software.amazon.awssdk.services.apptest.transform.UpdateTestSuiteRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link AppTestClient}.
 *
 * @see AppTestClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAppTestClient implements AppTestClient {
    private static final Logger log = Logger.loggerFor(DefaultAppTestClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAppTestClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a test case.
     * </p>
     *
     * @param createTestCaseRequest
     * @return Result of the CreateTestCase operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         One or more quotas for AWS Application Testing exceeds the limit.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws ConflictException
     *         The parameters provided in the request conflict with existing resources.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.CreateTestCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/CreateTestCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateTestCaseResponse createTestCase(CreateTestCaseRequest createTestCaseRequest)
            throws ServiceQuotaExceededException, ThrottlingException, AccessDeniedException, ValidationException,
            ConflictException, InternalServerException, AwsServiceException, SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateTestCaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateTestCaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTestCaseRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTestCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTestCase");

            return clientHandler.execute(new ClientExecutionParams<CreateTestCaseRequest, CreateTestCaseResponse>()
                    .withOperationName("CreateTestCase").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createTestCaseRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateTestCaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a test configuration.
     * </p>
     *
     * @param createTestConfigurationRequest
     * @return Result of the CreateTestConfiguration operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         One or more quotas for AWS Application Testing exceeds the limit.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws ConflictException
     *         The parameters provided in the request conflict with existing resources.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.CreateTestConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/CreateTestConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateTestConfigurationResponse createTestConfiguration(CreateTestConfigurationRequest createTestConfigurationRequest)
            throws ServiceQuotaExceededException, ThrottlingException, AccessDeniedException, ValidationException,
            ConflictException, InternalServerException, AwsServiceException, SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateTestConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateTestConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTestConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTestConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTestConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateTestConfigurationRequest, CreateTestConfigurationResponse>()
                            .withOperationName("CreateTestConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createTestConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateTestConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a test suite.
     * </p>
     *
     * @param createTestSuiteRequest
     * @return Result of the CreateTestSuite operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         One or more quotas for AWS Application Testing exceeds the limit.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws ConflictException
     *         The parameters provided in the request conflict with existing resources.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.CreateTestSuite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/CreateTestSuite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateTestSuiteResponse createTestSuite(CreateTestSuiteRequest createTestSuiteRequest)
            throws ServiceQuotaExceededException, ThrottlingException, AccessDeniedException, ValidationException,
            ConflictException, InternalServerException, AwsServiceException, SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateTestSuiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateTestSuiteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTestSuiteRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTestSuiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTestSuite");

            return clientHandler.execute(new ClientExecutionParams<CreateTestSuiteRequest, CreateTestSuiteResponse>()
                    .withOperationName("CreateTestSuite").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createTestSuiteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateTestSuiteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a test case.
     * </p>
     *
     * @param deleteTestCaseRequest
     * @return Result of the DeleteTestCase operation returned by the service.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws ConflictException
     *         The parameters provided in the request conflict with existing resources.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.DeleteTestCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/DeleteTestCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteTestCaseResponse deleteTestCase(DeleteTestCaseRequest deleteTestCaseRequest) throws ThrottlingException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ConflictException, InternalServerException,
            AwsServiceException, SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTestCaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteTestCaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTestCaseRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTestCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTestCase");

            return clientHandler.execute(new ClientExecutionParams<DeleteTestCaseRequest, DeleteTestCaseResponse>()
                    .withOperationName("DeleteTestCase").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteTestCaseRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteTestCaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a test configuration.
     * </p>
     *
     * @param deleteTestConfigurationRequest
     * @return Result of the DeleteTestConfiguration operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         One or more quotas for AWS Application Testing exceeds the limit.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws ConflictException
     *         The parameters provided in the request conflict with existing resources.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.DeleteTestConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/DeleteTestConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteTestConfigurationResponse deleteTestConfiguration(DeleteTestConfigurationRequest deleteTestConfigurationRequest)
            throws ServiceQuotaExceededException, ThrottlingException, AccessDeniedException, ValidationException,
            ConflictException, InternalServerException, AwsServiceException, SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTestConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteTestConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTestConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTestConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTestConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteTestConfigurationRequest, DeleteTestConfigurationResponse>()
                            .withOperationName("DeleteTestConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteTestConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteTestConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a test run.
     * </p>
     *
     * @param deleteTestRunRequest
     * @return Result of the DeleteTestRun operation returned by the service.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws ConflictException
     *         The parameters provided in the request conflict with existing resources.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.DeleteTestRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/DeleteTestRun" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteTestRunResponse deleteTestRun(DeleteTestRunRequest deleteTestRunRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, ConflictException, InternalServerException, AwsServiceException,
            SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTestRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteTestRunResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTestRunRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTestRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTestRun");

            return clientHandler.execute(new ClientExecutionParams<DeleteTestRunRequest, DeleteTestRunResponse>()
                    .withOperationName("DeleteTestRun").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteTestRunRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteTestRunRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a test suite.
     * </p>
     *
     * @param deleteTestSuiteRequest
     * @return Result of the DeleteTestSuite operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         One or more quotas for AWS Application Testing exceeds the limit.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws ConflictException
     *         The parameters provided in the request conflict with existing resources.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.DeleteTestSuite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/DeleteTestSuite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteTestSuiteResponse deleteTestSuite(DeleteTestSuiteRequest deleteTestSuiteRequest)
            throws ServiceQuotaExceededException, ThrottlingException, AccessDeniedException, ValidationException,
            ConflictException, InternalServerException, AwsServiceException, SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTestSuiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteTestSuiteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTestSuiteRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTestSuiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTestSuite");

            return clientHandler.execute(new ClientExecutionParams<DeleteTestSuiteRequest, DeleteTestSuiteResponse>()
                    .withOperationName("DeleteTestSuite").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteTestSuiteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteTestSuiteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a test case.
     * </p>
     *
     * @param getTestCaseRequest
     * @return Result of the GetTestCase operation returned by the service.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.GetTestCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/GetTestCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTestCaseResponse getTestCase(GetTestCaseRequest getTestCaseRequest) throws ThrottlingException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTestCaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTestCaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTestCaseRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTestCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTestCase");

            return clientHandler.execute(new ClientExecutionParams<GetTestCaseRequest, GetTestCaseResponse>()
                    .withOperationName("GetTestCase").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getTestCaseRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTestCaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a test configuration.
     * </p>
     *
     * @param getTestConfigurationRequest
     * @return Result of the GetTestConfiguration operation returned by the service.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.GetTestConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/GetTestConfiguration" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetTestConfigurationResponse getTestConfiguration(GetTestConfigurationRequest getTestConfigurationRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTestConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetTestConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTestConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTestConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTestConfiguration");

            return clientHandler.execute(new ClientExecutionParams<GetTestConfigurationRequest, GetTestConfigurationResponse>()
                    .withOperationName("GetTestConfiguration").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getTestConfigurationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTestConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a test run step.
     * </p>
     *
     * @param getTestRunStepRequest
     * @return Result of the GetTestRunStep operation returned by the service.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.GetTestRunStep
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/GetTestRunStep" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTestRunStepResponse getTestRunStep(GetTestRunStepRequest getTestRunStepRequest) throws ThrottlingException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTestRunStepResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTestRunStepResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTestRunStepRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTestRunStepRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTestRunStep");

            return clientHandler.execute(new ClientExecutionParams<GetTestRunStepRequest, GetTestRunStepResponse>()
                    .withOperationName("GetTestRunStep").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getTestRunStepRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTestRunStepRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a test suite.
     * </p>
     *
     * @param getTestSuiteRequest
     * @return Result of the GetTestSuite operation returned by the service.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.GetTestSuite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/GetTestSuite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTestSuiteResponse getTestSuite(GetTestSuiteRequest getTestSuiteRequest) throws ThrottlingException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTestSuiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTestSuiteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTestSuiteRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTestSuiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTestSuite");

            return clientHandler.execute(new ClientExecutionParams<GetTestSuiteRequest, GetTestSuiteResponse>()
                    .withOperationName("GetTestSuite").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getTestSuiteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTestSuiteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists test cases.
     * </p>
     *
     * @param listTestCasesRequest
     * @return Result of the ListTestCases operation returned by the service.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.ListTestCases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/ListTestCases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTestCasesResponse listTestCases(ListTestCasesRequest listTestCasesRequest) throws ThrottlingException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTestCasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTestCasesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTestCasesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTestCasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTestCases");

            return clientHandler.execute(new ClientExecutionParams<ListTestCasesRequest, ListTestCasesResponse>()
                    .withOperationName("ListTestCases").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTestCasesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTestCasesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists test configurations.
     * </p>
     *
     * @param listTestConfigurationsRequest
     * @return Result of the ListTestConfigurations operation returned by the service.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.ListTestConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/ListTestConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTestConfigurationsResponse listTestConfigurations(ListTestConfigurationsRequest listTestConfigurationsRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTestConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTestConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTestConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTestConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTestConfigurations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListTestConfigurationsRequest, ListTestConfigurationsResponse>()
                            .withOperationName("ListTestConfigurations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listTestConfigurationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListTestConfigurationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists test run steps.
     * </p>
     *
     * @param listTestRunStepsRequest
     * @return Result of the ListTestRunSteps operation returned by the service.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.ListTestRunSteps
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/ListTestRunSteps" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTestRunStepsResponse listTestRunSteps(ListTestRunStepsRequest listTestRunStepsRequest) throws ThrottlingException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTestRunStepsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTestRunStepsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTestRunStepsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTestRunStepsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTestRunSteps");

            return clientHandler.execute(new ClientExecutionParams<ListTestRunStepsRequest, ListTestRunStepsResponse>()
                    .withOperationName("ListTestRunSteps").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTestRunStepsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTestRunStepsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists test run test cases.
     * </p>
     *
     * @param listTestRunTestCasesRequest
     * @return Result of the ListTestRunTestCases operation returned by the service.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.ListTestRunTestCases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/ListTestRunTestCases" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTestRunTestCasesResponse listTestRunTestCases(ListTestRunTestCasesRequest listTestRunTestCasesRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTestRunTestCasesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTestRunTestCasesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTestRunTestCasesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTestRunTestCasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTestRunTestCases");

            return clientHandler.execute(new ClientExecutionParams<ListTestRunTestCasesRequest, ListTestRunTestCasesResponse>()
                    .withOperationName("ListTestRunTestCases").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTestRunTestCasesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTestRunTestCasesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists test runs.
     * </p>
     *
     * @param listTestRunsRequest
     * @return Result of the ListTestRuns operation returned by the service.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.ListTestRuns
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/ListTestRuns" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTestRunsResponse listTestRuns(ListTestRunsRequest listTestRunsRequest) throws ThrottlingException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTestRunsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTestRunsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTestRunsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTestRunsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTestRuns");

            return clientHandler.execute(new ClientExecutionParams<ListTestRunsRequest, ListTestRunsResponse>()
                    .withOperationName("ListTestRuns").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTestRunsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTestRunsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists test suites.
     * </p>
     *
     * @param listTestSuitesRequest
     * @return Result of the ListTestSuites operation returned by the service.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.ListTestSuites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/ListTestSuites" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTestSuitesResponse listTestSuites(ListTestSuitesRequest listTestSuitesRequest) throws ThrottlingException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTestSuitesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTestSuitesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTestSuitesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTestSuitesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTestSuites");

            return clientHandler.execute(new ClientExecutionParams<ListTestSuitesRequest, ListTestSuitesResponse>()
                    .withOperationName("ListTestSuites").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTestSuitesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTestSuitesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a test run.
     * </p>
     *
     * @param startTestRunRequest
     * @return Result of the StartTestRun operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         One or more quotas for AWS Application Testing exceeds the limit.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws ConflictException
     *         The parameters provided in the request conflict with existing resources.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.StartTestRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/StartTestRun" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartTestRunResponse startTestRun(StartTestRunRequest startTestRunRequest) throws ServiceQuotaExceededException,
            ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException, ConflictException,
            InternalServerException, AwsServiceException, SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartTestRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartTestRunResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startTestRunRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startTestRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartTestRun");

            return clientHandler.execute(new ClientExecutionParams<StartTestRunRequest, StartTestRunResponse>()
                    .withOperationName("StartTestRun").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startTestRunRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartTestRunRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Specifies tags of a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         One or more quotas for AWS Application Testing exceeds the limit.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ServiceQuotaExceededException,
            ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException,
            AwsServiceException, SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Untags a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ThrottlingException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a test case.
     * </p>
     *
     * @param updateTestCaseRequest
     * @return Result of the UpdateTestCase operation returned by the service.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws ConflictException
     *         The parameters provided in the request conflict with existing resources.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.UpdateTestCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/UpdateTestCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateTestCaseResponse updateTestCase(UpdateTestCaseRequest updateTestCaseRequest) throws ThrottlingException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ConflictException, InternalServerException,
            AwsServiceException, SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateTestCaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateTestCaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateTestCaseRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTestCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTestCase");

            return clientHandler.execute(new ClientExecutionParams<UpdateTestCaseRequest, UpdateTestCaseResponse>()
                    .withOperationName("UpdateTestCase").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateTestCaseRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateTestCaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a test configuration.
     * </p>
     *
     * @param updateTestConfigurationRequest
     * @return Result of the UpdateTestConfiguration operation returned by the service.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws ConflictException
     *         The parameters provided in the request conflict with existing resources.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.UpdateTestConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/UpdateTestConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateTestConfigurationResponse updateTestConfiguration(UpdateTestConfigurationRequest updateTestConfigurationRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException, ConflictException,
            InternalServerException, AwsServiceException, SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateTestConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateTestConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateTestConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTestConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTestConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateTestConfigurationRequest, UpdateTestConfigurationResponse>()
                            .withOperationName("UpdateTestConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateTestConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateTestConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a test suite.
     * </p>
     *
     * @param updateTestSuiteRequest
     * @return Result of the UpdateTestSuite operation returned by the service.
     * @throws ThrottlingException
     *         The number of requests made exceeds the limit.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws AccessDeniedException
     *         The account or role doesn't have the right permissions to make the request.
     * @throws ValidationException
     *         One or more parameter provided in the request is not valid.
     * @throws ConflictException
     *         The parameters provided in the request conflict with existing resources.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppTestException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppTestClient.UpdateTestSuite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apptest-2022-12-06/UpdateTestSuite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateTestSuiteResponse updateTestSuite(UpdateTestSuiteRequest updateTestSuiteRequest) throws ThrottlingException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ConflictException, InternalServerException,
            AwsServiceException, SdkClientException, AppTestException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateTestSuiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateTestSuiteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateTestSuiteRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTestSuiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppTest");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTestSuite");

            return clientHandler.execute(new ClientExecutionParams<UpdateTestSuiteRequest, UpdateTestSuiteResponse>()
                    .withOperationName("UpdateTestSuite").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateTestSuiteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateTestSuiteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        AppTestServiceClientConfigurationBuilder serviceConfigBuilder = new AppTestServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AppTestException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    @Override
    public final AppTestServiceClientConfiguration serviceClientConfiguration() {
        return new AppTestServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
