/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.arczonalshift;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.arczonalshift.internal.ArcZonalShiftServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.arczonalshift.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.arczonalshift.model.AccessDeniedException;
import software.amazon.awssdk.services.arczonalshift.model.ArcZonalShiftException;
import software.amazon.awssdk.services.arczonalshift.model.CancelPracticeRunRequest;
import software.amazon.awssdk.services.arczonalshift.model.CancelPracticeRunResponse;
import software.amazon.awssdk.services.arczonalshift.model.CancelZonalShiftRequest;
import software.amazon.awssdk.services.arczonalshift.model.CancelZonalShiftResponse;
import software.amazon.awssdk.services.arczonalshift.model.ConflictException;
import software.amazon.awssdk.services.arczonalshift.model.CreatePracticeRunConfigurationRequest;
import software.amazon.awssdk.services.arczonalshift.model.CreatePracticeRunConfigurationResponse;
import software.amazon.awssdk.services.arczonalshift.model.DeletePracticeRunConfigurationRequest;
import software.amazon.awssdk.services.arczonalshift.model.DeletePracticeRunConfigurationResponse;
import software.amazon.awssdk.services.arczonalshift.model.GetAutoshiftObserverNotificationStatusRequest;
import software.amazon.awssdk.services.arczonalshift.model.GetAutoshiftObserverNotificationStatusResponse;
import software.amazon.awssdk.services.arczonalshift.model.GetManagedResourceRequest;
import software.amazon.awssdk.services.arczonalshift.model.GetManagedResourceResponse;
import software.amazon.awssdk.services.arczonalshift.model.InternalServerException;
import software.amazon.awssdk.services.arczonalshift.model.ListAutoshiftsRequest;
import software.amazon.awssdk.services.arczonalshift.model.ListAutoshiftsResponse;
import software.amazon.awssdk.services.arczonalshift.model.ListManagedResourcesRequest;
import software.amazon.awssdk.services.arczonalshift.model.ListManagedResourcesResponse;
import software.amazon.awssdk.services.arczonalshift.model.ListZonalShiftsRequest;
import software.amazon.awssdk.services.arczonalshift.model.ListZonalShiftsResponse;
import software.amazon.awssdk.services.arczonalshift.model.ResourceNotFoundException;
import software.amazon.awssdk.services.arczonalshift.model.StartPracticeRunRequest;
import software.amazon.awssdk.services.arczonalshift.model.StartPracticeRunResponse;
import software.amazon.awssdk.services.arczonalshift.model.StartZonalShiftRequest;
import software.amazon.awssdk.services.arczonalshift.model.StartZonalShiftResponse;
import software.amazon.awssdk.services.arczonalshift.model.ThrottlingException;
import software.amazon.awssdk.services.arczonalshift.model.UpdateAutoshiftObserverNotificationStatusRequest;
import software.amazon.awssdk.services.arczonalshift.model.UpdateAutoshiftObserverNotificationStatusResponse;
import software.amazon.awssdk.services.arczonalshift.model.UpdatePracticeRunConfigurationRequest;
import software.amazon.awssdk.services.arczonalshift.model.UpdatePracticeRunConfigurationResponse;
import software.amazon.awssdk.services.arczonalshift.model.UpdateZonalAutoshiftConfigurationRequest;
import software.amazon.awssdk.services.arczonalshift.model.UpdateZonalAutoshiftConfigurationResponse;
import software.amazon.awssdk.services.arczonalshift.model.UpdateZonalShiftRequest;
import software.amazon.awssdk.services.arczonalshift.model.UpdateZonalShiftResponse;
import software.amazon.awssdk.services.arczonalshift.model.ValidationException;
import software.amazon.awssdk.services.arczonalshift.transform.CancelPracticeRunRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.CancelZonalShiftRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.CreatePracticeRunConfigurationRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.DeletePracticeRunConfigurationRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.GetAutoshiftObserverNotificationStatusRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.GetManagedResourceRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.ListAutoshiftsRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.ListManagedResourcesRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.ListZonalShiftsRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.StartPracticeRunRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.StartZonalShiftRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.UpdateAutoshiftObserverNotificationStatusRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.UpdatePracticeRunConfigurationRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.UpdateZonalAutoshiftConfigurationRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.UpdateZonalShiftRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ArcZonalShiftAsyncClient}.
 *
 * @see ArcZonalShiftAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultArcZonalShiftAsyncClient implements ArcZonalShiftAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultArcZonalShiftAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultArcZonalShiftAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "ARC_Zonal_Shift" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Cancel an in-progress practice run zonal shift in Amazon Application Recovery Controller.
     * </p>
     *
     * @param cancelPracticeRunRequest
     * @return A Java Future containing the result of the CancelPracticeRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an internal server error.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ResourceNotFoundException The input requested a resource that was not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcZonalShiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ArcZonalShiftAsyncClient.CancelPracticeRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/CancelPracticeRun"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelPracticeRunResponse> cancelPracticeRun(CancelPracticeRunRequest cancelPracticeRunRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelPracticeRunRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelPracticeRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelPracticeRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelPracticeRunResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelPracticeRunResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CancelPracticeRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelPracticeRunRequest, CancelPracticeRunResponse>()
                            .withOperationName("CancelPracticeRun").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CancelPracticeRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(cancelPracticeRunRequest));
            CompletableFuture<CancelPracticeRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancel a zonal shift in Amazon Application Recovery Controller. To cancel the zonal shift, specify the zonal
     * shift ID.
     * </p>
     * <p>
     * A zonal shift can be one that you've started for a resource in your Amazon Web Services account in an Amazon Web
     * Services Region, or it can be a zonal shift started by a practice run with zonal autoshift.
     * </p>
     *
     * @param cancelZonalShiftRequest
     * @return A Java Future containing the result of the CancelZonalShift operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an internal server error.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ResourceNotFoundException The input requested a resource that was not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcZonalShiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ArcZonalShiftAsyncClient.CancelZonalShift
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/CancelZonalShift"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelZonalShiftResponse> cancelZonalShift(CancelZonalShiftRequest cancelZonalShiftRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelZonalShiftRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelZonalShiftRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelZonalShift");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelZonalShiftResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelZonalShiftResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CancelZonalShiftResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelZonalShiftRequest, CancelZonalShiftResponse>()
                            .withOperationName("CancelZonalShift").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CancelZonalShiftRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(cancelZonalShiftRequest));
            CompletableFuture<CancelZonalShiftResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A practice run configuration for zonal autoshift is required when you enable zonal autoshift. A practice run
     * configuration includes specifications for blocked dates and blocked time windows, and for Amazon CloudWatch
     * alarms that you create to use with practice runs. The alarms that you specify are an <i>outcome alarm</i>, to
     * monitor application health during practice runs and, optionally, a <i>blocking alarm</i>, to block practice runs
     * from starting.
     * </p>
     * <p>
     * When a resource has a practice run configuration, ARC starts zonal shifts for the resource weekly, to shift
     * traffic for practice runs. Practice runs help you to ensure that shifting away traffic from an Availability Zone
     * during an autoshift is safe for your application.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.considerations.html"> Considerations
     * when you configure zonal autoshift</a> in the Amazon Application Recovery Controller Developer Guide.
     * </p>
     *
     * @param createPracticeRunConfigurationRequest
     * @return A Java Future containing the result of the CreatePracticeRunConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an internal server error.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ResourceNotFoundException The input requested a resource that was not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcZonalShiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ArcZonalShiftAsyncClient.CreatePracticeRunConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/CreatePracticeRunConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePracticeRunConfigurationResponse> createPracticeRunConfiguration(
            CreatePracticeRunConfigurationRequest createPracticeRunConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPracticeRunConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createPracticeRunConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePracticeRunConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePracticeRunConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePracticeRunConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreatePracticeRunConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePracticeRunConfigurationRequest, CreatePracticeRunConfigurationResponse>()
                            .withOperationName("CreatePracticeRunConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePracticeRunConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPracticeRunConfigurationRequest));
            CompletableFuture<CreatePracticeRunConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the practice run configuration for a resource. Before you can delete a practice run configuration for a
     * resource., you must disable zonal autoshift for the resource. Practice runs must be configured for zonal
     * autoshift to be enabled.
     * </p>
     *
     * @param deletePracticeRunConfigurationRequest
     * @return A Java Future containing the result of the DeletePracticeRunConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an internal server error.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ResourceNotFoundException The input requested a resource that was not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcZonalShiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ArcZonalShiftAsyncClient.DeletePracticeRunConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/DeletePracticeRunConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePracticeRunConfigurationResponse> deletePracticeRunConfiguration(
            DeletePracticeRunConfigurationRequest deletePracticeRunConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePracticeRunConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deletePracticeRunConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePracticeRunConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePracticeRunConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePracticeRunConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeletePracticeRunConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePracticeRunConfigurationRequest, DeletePracticeRunConfigurationResponse>()
                            .withOperationName("DeletePracticeRunConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePracticeRunConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePracticeRunConfigurationRequest));
            CompletableFuture<DeletePracticeRunConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the status of the autoshift observer notification. Autoshift observer notifications notify you through
     * Amazon EventBridge when there is an autoshift event for zonal autoshift. The status can be <code>ENABLED</code>
     * or <code>DISABLED</code>. When <code>ENABLED</code>, a notification is sent when an autoshift is triggered. When
     * <code>DISABLED</code>, notifications are not sent.
     * </p>
     *
     * @param getAutoshiftObserverNotificationStatusRequest
     * @return A Java Future containing the result of the GetAutoshiftObserverNotificationStatus operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an internal server error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcZonalShiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ArcZonalShiftAsyncClient.GetAutoshiftObserverNotificationStatus
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/GetAutoshiftObserverNotificationStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAutoshiftObserverNotificationStatusResponse> getAutoshiftObserverNotificationStatus(
            GetAutoshiftObserverNotificationStatusRequest getAutoshiftObserverNotificationStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAutoshiftObserverNotificationStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getAutoshiftObserverNotificationStatusRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAutoshiftObserverNotificationStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAutoshiftObserverNotificationStatusResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetAutoshiftObserverNotificationStatusResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetAutoshiftObserverNotificationStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAutoshiftObserverNotificationStatusRequest, GetAutoshiftObserverNotificationStatusResponse>()
                            .withOperationName("GetAutoshiftObserverNotificationStatus").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAutoshiftObserverNotificationStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAutoshiftObserverNotificationStatusRequest));
            CompletableFuture<GetAutoshiftObserverNotificationStatusResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get information about a resource that's been registered for zonal shifts with Amazon Application Recovery
     * Controller in this Amazon Web Services Region. Resources that are registered for zonal shifts are managed
     * resources in ARC. You can start zonal shifts and configure zonal autoshift for managed resources.
     * </p>
     *
     * @param getManagedResourceRequest
     * @return A Java Future containing the result of the GetManagedResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an internal server error.</li>
     *         <li>ResourceNotFoundException The input requested a resource that was not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcZonalShiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ArcZonalShiftAsyncClient.GetManagedResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/GetManagedResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetManagedResourceResponse> getManagedResource(GetManagedResourceRequest getManagedResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getManagedResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getManagedResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetManagedResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetManagedResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetManagedResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetManagedResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetManagedResourceRequest, GetManagedResourceResponse>()
                            .withOperationName("GetManagedResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetManagedResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getManagedResourceRequest));
            CompletableFuture<GetManagedResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the autoshifts for an Amazon Web Services Region. By default, the call returns only <code>ACTIVE</code>
     * autoshifts. Optionally, you can specify the <code>status</code> parameter to return <code>COMPLETED</code>
     * autoshifts.
     * </p>
     *
     * @param listAutoshiftsRequest
     * @return A Java Future containing the result of the ListAutoshifts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an internal server error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcZonalShiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ArcZonalShiftAsyncClient.ListAutoshifts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/ListAutoshifts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAutoshiftsResponse> listAutoshifts(ListAutoshiftsRequest listAutoshiftsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAutoshiftsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAutoshiftsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAutoshifts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAutoshiftsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAutoshiftsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListAutoshiftsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAutoshiftsRequest, ListAutoshiftsResponse>()
                            .withOperationName("ListAutoshifts").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAutoshiftsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAutoshiftsRequest));
            CompletableFuture<ListAutoshiftsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the resources in your Amazon Web Services account in this Amazon Web Services Region that are managed
     * for zonal shifts in Amazon Application Recovery Controller, and information about them. The information includes
     * the zonal autoshift status for the resource, as well as the Amazon Resource Name (ARN), the Availability Zones
     * that each resource is deployed in, and the resource name.
     * </p>
     *
     * @param listManagedResourcesRequest
     * @return A Java Future containing the result of the ListManagedResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an internal server error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcZonalShiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ArcZonalShiftAsyncClient.ListManagedResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/ListManagedResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListManagedResourcesResponse> listManagedResources(
            ListManagedResourcesRequest listManagedResourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listManagedResourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listManagedResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListManagedResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListManagedResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListManagedResourcesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListManagedResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListManagedResourcesRequest, ListManagedResourcesResponse>()
                            .withOperationName("ListManagedResources").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListManagedResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listManagedResourcesRequest));
            CompletableFuture<ListManagedResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all active and completed zonal shifts in Amazon Application Recovery Controller in your Amazon Web Services
     * account in this Amazon Web Services Region. <code>ListZonalShifts</code> returns customer-initiated zonal shifts,
     * as well as practice run zonal shifts that ARC started on your behalf for zonal autoshift.
     * </p>
     * <p>
     * For more information about listing autoshifts, see <a
     * href="https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_ListAutoshifts.html">"&gt;ListAutoshifts</a>.
     * </p>
     *
     * @param listZonalShiftsRequest
     * @return A Java Future containing the result of the ListZonalShifts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an internal server error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcZonalShiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ArcZonalShiftAsyncClient.ListZonalShifts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/ListZonalShifts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListZonalShiftsResponse> listZonalShifts(ListZonalShiftsRequest listZonalShiftsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listZonalShiftsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listZonalShiftsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListZonalShifts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListZonalShiftsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListZonalShiftsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListZonalShiftsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListZonalShiftsRequest, ListZonalShiftsResponse>()
                            .withOperationName("ListZonalShifts").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListZonalShiftsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listZonalShiftsRequest));
            CompletableFuture<ListZonalShiftsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Start an on-demand practice run zonal shift in Amazon Application Recovery Controller. With zonal autoshift
     * enabled, you can start an on-demand practice run to verify preparedness at any time. Amazon Web Services also
     * runs automated practice runs about weekly when you have enabled zonal autoshift.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.considerations.html"> Considerations
     * when you configure zonal autoshift</a> in the Amazon Application Recovery Controller Developer Guide.
     * </p>
     *
     * @param startPracticeRunRequest
     * @return A Java Future containing the result of the StartPracticeRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an internal server error.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ResourceNotFoundException The input requested a resource that was not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcZonalShiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ArcZonalShiftAsyncClient.StartPracticeRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/StartPracticeRun"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartPracticeRunResponse> startPracticeRun(StartPracticeRunRequest startPracticeRunRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startPracticeRunRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startPracticeRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartPracticeRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartPracticeRunResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartPracticeRunResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartPracticeRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartPracticeRunRequest, StartPracticeRunResponse>()
                            .withOperationName("StartPracticeRun").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartPracticeRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startPracticeRunRequest));
            CompletableFuture<StartPracticeRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * You start a zonal shift to temporarily move load balancer traffic away from an Availability Zone in an Amazon Web
     * Services Region, to help your application recover immediately, for example, from a developer's bad code
     * deployment or from an Amazon Web Services infrastructure failure in a single Availability Zone. You can start a
     * zonal shift in ARC only for managed resources in your Amazon Web Services account in an Amazon Web Services
     * Region. Resources are automatically registered with ARC by Amazon Web Services services.
     * </p>
     * <p>
     * Amazon Application Recovery Controller currently supports enabling the following resources for zonal shift and
     * zonal autoshift:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.ec2-auto-scaling-groups.html"
     * >Amazon EC2 Auto Scaling groups</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.eks.html">Amazon
     * Elastic Kubernetes Service</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.app-load-balancers.html"
     * >Application Load Balancer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.network-load-balancers.html"
     * >Network Load Balancer</a>
     * </p>
     * </li>
     * </ul>
     * <p>
     * When you start a zonal shift, traffic for the resource is no longer routed to the Availability Zone. The zonal
     * shift is created immediately in ARC. However, it can take a short time, typically up to a few minutes, for
     * existing, in-progress connections in the Availability Zone to complete.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.html">Zonal
     * shift</a> in the Amazon Application Recovery Controller Developer Guide.
     * </p>
     *
     * @param startZonalShiftRequest
     * @return A Java Future containing the result of the StartZonalShift operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an internal server error.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ResourceNotFoundException The input requested a resource that was not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcZonalShiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ArcZonalShiftAsyncClient.StartZonalShift
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/StartZonalShift"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartZonalShiftResponse> startZonalShift(StartZonalShiftRequest startZonalShiftRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startZonalShiftRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startZonalShiftRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartZonalShift");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartZonalShiftResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartZonalShiftResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartZonalShiftResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartZonalShiftRequest, StartZonalShiftResponse>()
                            .withOperationName("StartZonalShift").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartZonalShiftRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startZonalShiftRequest));
            CompletableFuture<StartZonalShiftResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update the status of autoshift observer notification. Autoshift observer notification enables you to be notified,
     * through Amazon EventBridge, when there is an autoshift event for zonal autoshift.
     * </p>
     * <p>
     * If the status is <code>ENABLED</code>, ARC includes all autoshift events when you use the EventBridge pattern
     * <code>Autoshift In Progress</code>. When the status is <code>DISABLED</code>, ARC includes only autoshift events
     * for autoshifts when one or more of your resources is included in the autoshift.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.how-it-works.html#ZAShiftNotification">
     * Notifications for practice runs and autoshifts</a> in the Amazon Application Recovery Controller Developer Guide.
     * </p>
     *
     * @param updateAutoshiftObserverNotificationStatusRequest
     * @return A Java Future containing the result of the UpdateAutoshiftObserverNotificationStatus operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an internal server error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcZonalShiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ArcZonalShiftAsyncClient.UpdateAutoshiftObserverNotificationStatus
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/UpdateAutoshiftObserverNotificationStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAutoshiftObserverNotificationStatusResponse> updateAutoshiftObserverNotificationStatus(
            UpdateAutoshiftObserverNotificationStatusRequest updateAutoshiftObserverNotificationStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                updateAutoshiftObserverNotificationStatusRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateAutoshiftObserverNotificationStatusRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAutoshiftObserverNotificationStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAutoshiftObserverNotificationStatusResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateAutoshiftObserverNotificationStatusResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateAutoshiftObserverNotificationStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAutoshiftObserverNotificationStatusRequest, UpdateAutoshiftObserverNotificationStatusResponse>()
                            .withOperationName("UpdateAutoshiftObserverNotificationStatus")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateAutoshiftObserverNotificationStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAutoshiftObserverNotificationStatusRequest));
            CompletableFuture<UpdateAutoshiftObserverNotificationStatusResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update a practice run configuration to change one or more of the following: add, change, or remove the blocking
     * alarm; change the outcome alarm; or add, change, or remove blocking dates or time windows.
     * </p>
     *
     * @param updatePracticeRunConfigurationRequest
     * @return A Java Future containing the result of the UpdatePracticeRunConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an internal server error.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ResourceNotFoundException The input requested a resource that was not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcZonalShiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ArcZonalShiftAsyncClient.UpdatePracticeRunConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/UpdatePracticeRunConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePracticeRunConfigurationResponse> updatePracticeRunConfiguration(
            UpdatePracticeRunConfigurationRequest updatePracticeRunConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePracticeRunConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updatePracticeRunConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePracticeRunConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePracticeRunConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePracticeRunConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdatePracticeRunConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePracticeRunConfigurationRequest, UpdatePracticeRunConfigurationResponse>()
                            .withOperationName("UpdatePracticeRunConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdatePracticeRunConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePracticeRunConfigurationRequest));
            CompletableFuture<UpdatePracticeRunConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The zonal autoshift configuration for a resource includes the practice run configuration and the status for
     * running autoshifts, zonal autoshift status. When a resource has a practice run configuration, ARC starts weekly
     * zonal shifts for the resource, to shift traffic away from an Availability Zone. Weekly practice runs help you to
     * make sure that your application can continue to operate normally with the loss of one Availability Zone.
     * </p>
     * <p>
     * You can update the zonal autoshift status to enable or disable zonal autoshift. When zonal autoshift is
     * <code>ENABLED</code>, you authorize Amazon Web Services to shift away resource traffic for an application from an
     * Availability Zone during events, on your behalf, to help reduce time to recovery. Traffic is also shifted away
     * for the required weekly practice runs.
     * </p>
     *
     * @param updateZonalAutoshiftConfigurationRequest
     * @return A Java Future containing the result of the UpdateZonalAutoshiftConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an internal server error.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ResourceNotFoundException The input requested a resource that was not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcZonalShiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ArcZonalShiftAsyncClient.UpdateZonalAutoshiftConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/UpdateZonalAutoshiftConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateZonalAutoshiftConfigurationResponse> updateZonalAutoshiftConfiguration(
            UpdateZonalAutoshiftConfigurationRequest updateZonalAutoshiftConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateZonalAutoshiftConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateZonalAutoshiftConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateZonalAutoshiftConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateZonalAutoshiftConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateZonalAutoshiftConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateZonalAutoshiftConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateZonalAutoshiftConfigurationRequest, UpdateZonalAutoshiftConfigurationResponse>()
                            .withOperationName("UpdateZonalAutoshiftConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateZonalAutoshiftConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateZonalAutoshiftConfigurationRequest));
            CompletableFuture<UpdateZonalAutoshiftConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update an active zonal shift in Amazon Application Recovery Controller in your Amazon Web Services account. You
     * can update a zonal shift to set a new expiration, or edit or replace the comment for the zonal shift.
     * </p>
     *
     * @param updateZonalShiftRequest
     * @return A Java Future containing the result of the UpdateZonalShift operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an internal server error.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ResourceNotFoundException The input requested a resource that was not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcZonalShiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ArcZonalShiftAsyncClient.UpdateZonalShift
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/UpdateZonalShift"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateZonalShiftResponse> updateZonalShift(UpdateZonalShiftRequest updateZonalShiftRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateZonalShiftRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateZonalShiftRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateZonalShift");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateZonalShiftResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateZonalShiftResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateZonalShiftResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateZonalShiftRequest, UpdateZonalShiftResponse>()
                            .withOperationName("UpdateZonalShift").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateZonalShiftRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateZonalShiftRequest));
            CompletableFuture<UpdateZonalShiftResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final ArcZonalShiftServiceClientConfiguration serviceClientConfiguration() {
        return new ArcZonalShiftServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(ArcZonalShiftException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        ArcZonalShiftServiceClientConfigurationBuilder serviceConfigBuilder = new ArcZonalShiftServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
