/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.arczonalshift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Lists information about zonal shifts in Amazon Application Recovery Controller, including zonal shifts that you start
 * yourself and zonal shifts that ARC starts on your behalf for practice runs with zonal autoshift.
 * </p>
 * <p>
 * Zonal shifts are temporary, including customer-initiated zonal shifts and the zonal autoshift practice run zonal
 * shifts that ARC starts weekly, on your behalf. A zonal shift that a customer starts can be active for up to three
 * days (72 hours). A practice run zonal shift has a 30 minute duration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ZonalShiftSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<ZonalShiftSummary.Builder, ZonalShiftSummary> {
    private static final SdkField<String> ZONAL_SHIFT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("zonalShiftId").getter(getter(ZonalShiftSummary::zonalShiftId)).setter(setter(Builder::zonalShiftId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("zonalShiftId").build()).build();

    private static final SdkField<String> RESOURCE_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceIdentifier").getter(getter(ZonalShiftSummary::resourceIdentifier))
            .setter(setter(Builder::resourceIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceIdentifier").build())
            .build();

    private static final SdkField<String> AWAY_FROM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("awayFrom").getter(getter(ZonalShiftSummary::awayFrom)).setter(setter(Builder::awayFrom))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("awayFrom").build()).build();

    private static final SdkField<Instant> EXPIRY_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("expiryTime").getter(getter(ZonalShiftSummary::expiryTime)).setter(setter(Builder::expiryTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("expiryTime").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime").getter(getter(ZonalShiftSummary::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(ZonalShiftSummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> COMMENT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("comment")
            .getter(getter(ZonalShiftSummary::comment)).setter(setter(Builder::comment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("comment").build()).build();

    private static final SdkField<String> SHIFT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("shiftType").getter(getter(ZonalShiftSummary::shiftTypeAsString)).setter(setter(Builder::shiftType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("shiftType").build()).build();

    private static final SdkField<String> PRACTICE_RUN_OUTCOME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("practiceRunOutcome").getter(getter(ZonalShiftSummary::practiceRunOutcomeAsString))
            .setter(setter(Builder::practiceRunOutcome))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("practiceRunOutcome").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ZONAL_SHIFT_ID_FIELD,
            RESOURCE_IDENTIFIER_FIELD, AWAY_FROM_FIELD, EXPIRY_TIME_FIELD, START_TIME_FIELD, STATUS_FIELD, COMMENT_FIELD,
            SHIFT_TYPE_FIELD, PRACTICE_RUN_OUTCOME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String zonalShiftId;

    private final String resourceIdentifier;

    private final String awayFrom;

    private final Instant expiryTime;

    private final Instant startTime;

    private final String status;

    private final String comment;

    private final String shiftType;

    private final String practiceRunOutcome;

    private ZonalShiftSummary(BuilderImpl builder) {
        this.zonalShiftId = builder.zonalShiftId;
        this.resourceIdentifier = builder.resourceIdentifier;
        this.awayFrom = builder.awayFrom;
        this.expiryTime = builder.expiryTime;
        this.startTime = builder.startTime;
        this.status = builder.status;
        this.comment = builder.comment;
        this.shiftType = builder.shiftType;
        this.practiceRunOutcome = builder.practiceRunOutcome;
    }

    /**
     * <p>
     * The identifier of a zonal shift.
     * </p>
     * 
     * @return The identifier of a zonal shift.
     */
    public final String zonalShiftId() {
        return zonalShiftId;
    }

    /**
     * <p>
     * The identifier for the resource to include in a zonal shift. The identifier is the Amazon Resource Name (ARN) for
     * the resource.
     * </p>
     * <p>
     * Amazon Application Recovery Controller currently supports enabling the following resources for zonal shift and
     * zonal autoshift:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.ec2-auto-scaling-groups.html"
     * >Amazon EC2 Auto Scaling groups</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.eks.html">Amazon
     * Elastic Kubernetes Service</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.app-load-balancers.html"
     * >Application Load Balancers</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.network-load-balancers.html"
     * >Network Load Balancers</a>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The identifier for the resource to include in a zonal shift. The identifier is the Amazon Resource Name
     *         (ARN) for the resource.</p>
     *         <p>
     *         Amazon Application Recovery Controller currently supports enabling the following resources for zonal
     *         shift and zonal autoshift:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <a href=
     *         "https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.ec2-auto-scaling-groups.html"
     *         >Amazon EC2 Auto Scaling groups</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a
     *         href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.eks.html">Amazon
     *         Elastic Kubernetes Service</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href=
     *         "https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.app-load-balancers.html"
     *         >Application Load Balancers</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href=
     *         "https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.network-load-balancers.html"
     *         >Network Load Balancers</a>
     *         </p>
     *         </li>
     */
    public final String resourceIdentifier() {
        return resourceIdentifier;
    }

    /**
     * <p>
     * The Availability Zone (for example, <code>use1-az1</code>) that traffic is moved away from for a resource when
     * you start a zonal shift. Until the zonal shift expires or you cancel it, traffic for the resource is instead
     * moved to other Availability Zones in the Amazon Web Services Region.
     * </p>
     * 
     * @return The Availability Zone (for example, <code>use1-az1</code>) that traffic is moved away from for a resource
     *         when you start a zonal shift. Until the zonal shift expires or you cancel it, traffic for the resource is
     *         instead moved to other Availability Zones in the Amazon Web Services Region.
     */
    public final String awayFrom() {
        return awayFrom;
    }

    /**
     * <p>
     * The expiry time (expiration time) for a customer-initiated zonal shift. A zonal shift is temporary and must be
     * set to expire when you start the zonal shift. You can initially set a zonal shift to expire in a maximum of three
     * days (72 hours). However, you can update a zonal shift to set a new expiration at any time.
     * </p>
     * <p>
     * When you start a zonal shift, you specify how long you want it to be active, which ARC converts to an expiry time
     * (expiration time). You can cancel a zonal shift when you're ready to restore traffic to the Availability Zone, or
     * just wait for it to expire. Or you can update the zonal shift to specify another length of time to expire in.
     * </p>
     * 
     * @return The expiry time (expiration time) for a customer-initiated zonal shift. A zonal shift is temporary and
     *         must be set to expire when you start the zonal shift. You can initially set a zonal shift to expire in a
     *         maximum of three days (72 hours). However, you can update a zonal shift to set a new expiration at any
     *         time. </p>
     *         <p>
     *         When you start a zonal shift, you specify how long you want it to be active, which ARC converts to an
     *         expiry time (expiration time). You can cancel a zonal shift when you're ready to restore traffic to the
     *         Availability Zone, or just wait for it to expire. Or you can update the zonal shift to specify another
     *         length of time to expire in.
     */
    public final Instant expiryTime() {
        return expiryTime;
    }

    /**
     * <p>
     * The time (UTC) when the zonal shift starts.
     * </p>
     * 
     * @return The time (UTC) when the zonal shift starts.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * A status for a zonal shift.
     * </p>
     * <p>
     * The <code>Status</code> for a zonal shift can have one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>ACTIVE:</b> The zonal shift has been started and is active.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>EXPIRED:</b> The zonal shift has expired (the expiry time was exceeded).
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CANCELED:</b> The zonal shift was canceled.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ZonalShiftStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return A status for a zonal shift.</p>
     *         <p>
     *         The <code>Status</code> for a zonal shift can have one of the following values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>ACTIVE:</b> The zonal shift has been started and is active.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>EXPIRED:</b> The zonal shift has expired (the expiry time was exceeded).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CANCELED:</b> The zonal shift was canceled.
     *         </p>
     *         </li>
     * @see ZonalShiftStatus
     */
    public final ZonalShiftStatus status() {
        return ZonalShiftStatus.fromValue(status);
    }

    /**
     * <p>
     * A status for a zonal shift.
     * </p>
     * <p>
     * The <code>Status</code> for a zonal shift can have one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>ACTIVE:</b> The zonal shift has been started and is active.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>EXPIRED:</b> The zonal shift has expired (the expiry time was exceeded).
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CANCELED:</b> The zonal shift was canceled.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ZonalShiftStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return A status for a zonal shift.</p>
     *         <p>
     *         The <code>Status</code> for a zonal shift can have one of the following values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>ACTIVE:</b> The zonal shift has been started and is active.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>EXPIRED:</b> The zonal shift has expired (the expiry time was exceeded).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CANCELED:</b> The zonal shift was canceled.
     *         </p>
     *         </li>
     * @see ZonalShiftStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A comment that you enter about the zonal shift. Only the latest comment is retained; no comment history is
     * maintained. That is, a new comment overwrites any existing comment string.
     * </p>
     * 
     * @return A comment that you enter about the zonal shift. Only the latest comment is retained; no comment history
     *         is maintained. That is, a new comment overwrites any existing comment string.
     */
    public final String comment() {
        return comment;
    }

    /**
     * <p>
     * Defines the zonal shift type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #shiftType} will
     * return {@link ShiftType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #shiftTypeAsString}.
     * </p>
     * 
     * @return Defines the zonal shift type.
     * @see ShiftType
     */
    public final ShiftType shiftType() {
        return ShiftType.fromValue(shiftType);
    }

    /**
     * <p>
     * Defines the zonal shift type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #shiftType} will
     * return {@link ShiftType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #shiftTypeAsString}.
     * </p>
     * 
     * @return Defines the zonal shift type.
     * @see ShiftType
     */
    public final String shiftTypeAsString() {
        return shiftType;
    }

    /**
     * <p>
     * The outcome, or end state, of a practice run. The following values can be returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>PENDING:</b> Outcome value when the practice run is in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>SUCCEEDED:</b> Outcome value when the outcome alarm specified for the practice run configuration does not go
     * into an <code>ALARM</code> state during the practice run, and the practice run was not interrupted before it
     * completed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>INTERRUPTED:</b> Outcome value when the practice run did not run for the expected 30 minutes or there was
     * another problem with the practice run that created an inconclusive outcome.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>FAILED:</b> Outcome value when the outcome alarm specified for the practice run configuration goes into an
     * <code>ALARM</code> state during the practice run, and the practice run was not interrupted before it completed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CAPACITY_CHECK_FAILED:</b> The check for balanced capacity across Availability Zones for your load balancing
     * and Auto Scaling group resources failed.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about practice run outcomes, see <a
     * href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.configure.html"> Considerations when
     * you configure zonal autoshift</a> in the Amazon Application Recovery Controller Developer Guide.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #practiceRunOutcome} will return {@link PracticeRunOutcome#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #practiceRunOutcomeAsString}.
     * </p>
     * 
     * @return The outcome, or end state, of a practice run. The following values can be returned:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>PENDING:</b> Outcome value when the practice run is in progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>SUCCEEDED:</b> Outcome value when the outcome alarm specified for the practice run configuration does
     *         not go into an <code>ALARM</code> state during the practice run, and the practice run was not interrupted
     *         before it completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>INTERRUPTED:</b> Outcome value when the practice run did not run for the expected 30 minutes or there
     *         was another problem with the practice run that created an inconclusive outcome.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FAILED:</b> Outcome value when the outcome alarm specified for the practice run configuration goes
     *         into an <code>ALARM</code> state during the practice run, and the practice run was not interrupted before
     *         it completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CAPACITY_CHECK_FAILED:</b> The check for balanced capacity across Availability Zones for your load
     *         balancing and Auto Scaling group resources failed.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information about practice run outcomes, see <a
     *         href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.configure.html">
     *         Considerations when you configure zonal autoshift</a> in the Amazon Application Recovery Controller
     *         Developer Guide.
     * @see PracticeRunOutcome
     */
    public final PracticeRunOutcome practiceRunOutcome() {
        return PracticeRunOutcome.fromValue(practiceRunOutcome);
    }

    /**
     * <p>
     * The outcome, or end state, of a practice run. The following values can be returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>PENDING:</b> Outcome value when the practice run is in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>SUCCEEDED:</b> Outcome value when the outcome alarm specified for the practice run configuration does not go
     * into an <code>ALARM</code> state during the practice run, and the practice run was not interrupted before it
     * completed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>INTERRUPTED:</b> Outcome value when the practice run did not run for the expected 30 minutes or there was
     * another problem with the practice run that created an inconclusive outcome.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>FAILED:</b> Outcome value when the outcome alarm specified for the practice run configuration goes into an
     * <code>ALARM</code> state during the practice run, and the practice run was not interrupted before it completed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CAPACITY_CHECK_FAILED:</b> The check for balanced capacity across Availability Zones for your load balancing
     * and Auto Scaling group resources failed.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about practice run outcomes, see <a
     * href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.configure.html"> Considerations when
     * you configure zonal autoshift</a> in the Amazon Application Recovery Controller Developer Guide.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #practiceRunOutcome} will return {@link PracticeRunOutcome#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #practiceRunOutcomeAsString}.
     * </p>
     * 
     * @return The outcome, or end state, of a practice run. The following values can be returned:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>PENDING:</b> Outcome value when the practice run is in progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>SUCCEEDED:</b> Outcome value when the outcome alarm specified for the practice run configuration does
     *         not go into an <code>ALARM</code> state during the practice run, and the practice run was not interrupted
     *         before it completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>INTERRUPTED:</b> Outcome value when the practice run did not run for the expected 30 minutes or there
     *         was another problem with the practice run that created an inconclusive outcome.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FAILED:</b> Outcome value when the outcome alarm specified for the practice run configuration goes
     *         into an <code>ALARM</code> state during the practice run, and the practice run was not interrupted before
     *         it completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CAPACITY_CHECK_FAILED:</b> The check for balanced capacity across Availability Zones for your load
     *         balancing and Auto Scaling group resources failed.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information about practice run outcomes, see <a
     *         href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.configure.html">
     *         Considerations when you configure zonal autoshift</a> in the Amazon Application Recovery Controller
     *         Developer Guide.
     * @see PracticeRunOutcome
     */
    public final String practiceRunOutcomeAsString() {
        return practiceRunOutcome;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(zonalShiftId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(awayFrom());
        hashCode = 31 * hashCode + Objects.hashCode(expiryTime());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(comment());
        hashCode = 31 * hashCode + Objects.hashCode(shiftTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(practiceRunOutcomeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ZonalShiftSummary)) {
            return false;
        }
        ZonalShiftSummary other = (ZonalShiftSummary) obj;
        return Objects.equals(zonalShiftId(), other.zonalShiftId())
                && Objects.equals(resourceIdentifier(), other.resourceIdentifier())
                && Objects.equals(awayFrom(), other.awayFrom()) && Objects.equals(expiryTime(), other.expiryTime())
                && Objects.equals(startTime(), other.startTime()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(comment(), other.comment()) && Objects.equals(shiftTypeAsString(), other.shiftTypeAsString())
                && Objects.equals(practiceRunOutcomeAsString(), other.practiceRunOutcomeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ZonalShiftSummary").add("ZonalShiftId", zonalShiftId())
                .add("ResourceIdentifier", resourceIdentifier()).add("AwayFrom", awayFrom()).add("ExpiryTime", expiryTime())
                .add("StartTime", startTime()).add("Status", statusAsString()).add("Comment", comment())
                .add("ShiftType", shiftTypeAsString()).add("PracticeRunOutcome", practiceRunOutcomeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "zonalShiftId":
            return Optional.ofNullable(clazz.cast(zonalShiftId()));
        case "resourceIdentifier":
            return Optional.ofNullable(clazz.cast(resourceIdentifier()));
        case "awayFrom":
            return Optional.ofNullable(clazz.cast(awayFrom()));
        case "expiryTime":
            return Optional.ofNullable(clazz.cast(expiryTime()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "comment":
            return Optional.ofNullable(clazz.cast(comment()));
        case "shiftType":
            return Optional.ofNullable(clazz.cast(shiftTypeAsString()));
        case "practiceRunOutcome":
            return Optional.ofNullable(clazz.cast(practiceRunOutcomeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("zonalShiftId", ZONAL_SHIFT_ID_FIELD);
        map.put("resourceIdentifier", RESOURCE_IDENTIFIER_FIELD);
        map.put("awayFrom", AWAY_FROM_FIELD);
        map.put("expiryTime", EXPIRY_TIME_FIELD);
        map.put("startTime", START_TIME_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("comment", COMMENT_FIELD);
        map.put("shiftType", SHIFT_TYPE_FIELD);
        map.put("practiceRunOutcome", PRACTICE_RUN_OUTCOME_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ZonalShiftSummary, T> g) {
        return obj -> g.apply((ZonalShiftSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ZonalShiftSummary> {
        /**
         * <p>
         * The identifier of a zonal shift.
         * </p>
         * 
         * @param zonalShiftId
         *        The identifier of a zonal shift.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder zonalShiftId(String zonalShiftId);

        /**
         * <p>
         * The identifier for the resource to include in a zonal shift. The identifier is the Amazon Resource Name (ARN)
         * for the resource.
         * </p>
         * <p>
         * Amazon Application Recovery Controller currently supports enabling the following resources for zonal shift
         * and zonal autoshift:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <a href=
         * "https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.ec2-auto-scaling-groups.html"
         * >Amazon EC2 Auto Scaling groups</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.eks.html">Amazon
         * Elastic Kubernetes Service</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href=
         * "https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.app-load-balancers.html"
         * >Application Load Balancers</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href=
         * "https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.network-load-balancers.html"
         * >Network Load Balancers</a>
         * </p>
         * </li>
         * </ul>
         * 
         * @param resourceIdentifier
         *        The identifier for the resource to include in a zonal shift. The identifier is the Amazon Resource
         *        Name (ARN) for the resource.</p>
         *        <p>
         *        Amazon Application Recovery Controller currently supports enabling the following resources for zonal
         *        shift and zonal autoshift:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <a href=
         *        "https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.ec2-auto-scaling-groups.html"
         *        >Amazon EC2 Auto Scaling groups</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.eks.html">
         *        Amazon Elastic Kubernetes Service</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href=
         *        "https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.app-load-balancers.html"
         *        >Application Load Balancers</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href=
         *        "https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.network-load-balancers.html"
         *        >Network Load Balancers</a>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIdentifier(String resourceIdentifier);

        /**
         * <p>
         * The Availability Zone (for example, <code>use1-az1</code>) that traffic is moved away from for a resource
         * when you start a zonal shift. Until the zonal shift expires or you cancel it, traffic for the resource is
         * instead moved to other Availability Zones in the Amazon Web Services Region.
         * </p>
         * 
         * @param awayFrom
         *        The Availability Zone (for example, <code>use1-az1</code>) that traffic is moved away from for a
         *        resource when you start a zonal shift. Until the zonal shift expires or you cancel it, traffic for the
         *        resource is instead moved to other Availability Zones in the Amazon Web Services Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awayFrom(String awayFrom);

        /**
         * <p>
         * The expiry time (expiration time) for a customer-initiated zonal shift. A zonal shift is temporary and must
         * be set to expire when you start the zonal shift. You can initially set a zonal shift to expire in a maximum
         * of three days (72 hours). However, you can update a zonal shift to set a new expiration at any time.
         * </p>
         * <p>
         * When you start a zonal shift, you specify how long you want it to be active, which ARC converts to an expiry
         * time (expiration time). You can cancel a zonal shift when you're ready to restore traffic to the Availability
         * Zone, or just wait for it to expire. Or you can update the zonal shift to specify another length of time to
         * expire in.
         * </p>
         * 
         * @param expiryTime
         *        The expiry time (expiration time) for a customer-initiated zonal shift. A zonal shift is temporary and
         *        must be set to expire when you start the zonal shift. You can initially set a zonal shift to expire in
         *        a maximum of three days (72 hours). However, you can update a zonal shift to set a new expiration at
         *        any time. </p>
         *        <p>
         *        When you start a zonal shift, you specify how long you want it to be active, which ARC converts to an
         *        expiry time (expiration time). You can cancel a zonal shift when you're ready to restore traffic to
         *        the Availability Zone, or just wait for it to expire. Or you can update the zonal shift to specify
         *        another length of time to expire in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expiryTime(Instant expiryTime);

        /**
         * <p>
         * The time (UTC) when the zonal shift starts.
         * </p>
         * 
         * @param startTime
         *        The time (UTC) when the zonal shift starts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * A status for a zonal shift.
         * </p>
         * <p>
         * The <code>Status</code> for a zonal shift can have one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>ACTIVE:</b> The zonal shift has been started and is active.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>EXPIRED:</b> The zonal shift has expired (the expiry time was exceeded).
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CANCELED:</b> The zonal shift was canceled.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        A status for a zonal shift.</p>
         *        <p>
         *        The <code>Status</code> for a zonal shift can have one of the following values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>ACTIVE:</b> The zonal shift has been started and is active.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>EXPIRED:</b> The zonal shift has expired (the expiry time was exceeded).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CANCELED:</b> The zonal shift was canceled.
         *        </p>
         *        </li>
         * @see ZonalShiftStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ZonalShiftStatus
         */
        Builder status(String status);

        /**
         * <p>
         * A status for a zonal shift.
         * </p>
         * <p>
         * The <code>Status</code> for a zonal shift can have one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>ACTIVE:</b> The zonal shift has been started and is active.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>EXPIRED:</b> The zonal shift has expired (the expiry time was exceeded).
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CANCELED:</b> The zonal shift was canceled.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        A status for a zonal shift.</p>
         *        <p>
         *        The <code>Status</code> for a zonal shift can have one of the following values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>ACTIVE:</b> The zonal shift has been started and is active.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>EXPIRED:</b> The zonal shift has expired (the expiry time was exceeded).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CANCELED:</b> The zonal shift was canceled.
         *        </p>
         *        </li>
         * @see ZonalShiftStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ZonalShiftStatus
         */
        Builder status(ZonalShiftStatus status);

        /**
         * <p>
         * A comment that you enter about the zonal shift. Only the latest comment is retained; no comment history is
         * maintained. That is, a new comment overwrites any existing comment string.
         * </p>
         * 
         * @param comment
         *        A comment that you enter about the zonal shift. Only the latest comment is retained; no comment
         *        history is maintained. That is, a new comment overwrites any existing comment string.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder comment(String comment);

        /**
         * <p>
         * Defines the zonal shift type.
         * </p>
         * 
         * @param shiftType
         *        Defines the zonal shift type.
         * @see ShiftType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ShiftType
         */
        Builder shiftType(String shiftType);

        /**
         * <p>
         * Defines the zonal shift type.
         * </p>
         * 
         * @param shiftType
         *        Defines the zonal shift type.
         * @see ShiftType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ShiftType
         */
        Builder shiftType(ShiftType shiftType);

        /**
         * <p>
         * The outcome, or end state, of a practice run. The following values can be returned:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>PENDING:</b> Outcome value when the practice run is in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>SUCCEEDED:</b> Outcome value when the outcome alarm specified for the practice run configuration does not
         * go into an <code>ALARM</code> state during the practice run, and the practice run was not interrupted before
         * it completed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>INTERRUPTED:</b> Outcome value when the practice run did not run for the expected 30 minutes or there was
         * another problem with the practice run that created an inconclusive outcome.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>FAILED:</b> Outcome value when the outcome alarm specified for the practice run configuration goes into an
         * <code>ALARM</code> state during the practice run, and the practice run was not interrupted before it
         * completed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CAPACITY_CHECK_FAILED:</b> The check for balanced capacity across Availability Zones for your load
         * balancing and Auto Scaling group resources failed.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information about practice run outcomes, see <a
         * href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.configure.html"> Considerations
         * when you configure zonal autoshift</a> in the Amazon Application Recovery Controller Developer Guide.
         * </p>
         * 
         * @param practiceRunOutcome
         *        The outcome, or end state, of a practice run. The following values can be returned:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>PENDING:</b> Outcome value when the practice run is in progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>SUCCEEDED:</b> Outcome value when the outcome alarm specified for the practice run configuration
         *        does not go into an <code>ALARM</code> state during the practice run, and the practice run was not
         *        interrupted before it completed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>INTERRUPTED:</b> Outcome value when the practice run did not run for the expected 30 minutes or
         *        there was another problem with the practice run that created an inconclusive outcome.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>FAILED:</b> Outcome value when the outcome alarm specified for the practice run configuration goes
         *        into an <code>ALARM</code> state during the practice run, and the practice run was not interrupted
         *        before it completed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CAPACITY_CHECK_FAILED:</b> The check for balanced capacity across Availability Zones for your load
         *        balancing and Auto Scaling group resources failed.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information about practice run outcomes, see <a
         *        href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.configure.html">
         *        Considerations when you configure zonal autoshift</a> in the Amazon Application Recovery Controller
         *        Developer Guide.
         * @see PracticeRunOutcome
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PracticeRunOutcome
         */
        Builder practiceRunOutcome(String practiceRunOutcome);

        /**
         * <p>
         * The outcome, or end state, of a practice run. The following values can be returned:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>PENDING:</b> Outcome value when the practice run is in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>SUCCEEDED:</b> Outcome value when the outcome alarm specified for the practice run configuration does not
         * go into an <code>ALARM</code> state during the practice run, and the practice run was not interrupted before
         * it completed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>INTERRUPTED:</b> Outcome value when the practice run did not run for the expected 30 minutes or there was
         * another problem with the practice run that created an inconclusive outcome.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>FAILED:</b> Outcome value when the outcome alarm specified for the practice run configuration goes into an
         * <code>ALARM</code> state during the practice run, and the practice run was not interrupted before it
         * completed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CAPACITY_CHECK_FAILED:</b> The check for balanced capacity across Availability Zones for your load
         * balancing and Auto Scaling group resources failed.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information about practice run outcomes, see <a
         * href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.configure.html"> Considerations
         * when you configure zonal autoshift</a> in the Amazon Application Recovery Controller Developer Guide.
         * </p>
         * 
         * @param practiceRunOutcome
         *        The outcome, or end state, of a practice run. The following values can be returned:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>PENDING:</b> Outcome value when the practice run is in progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>SUCCEEDED:</b> Outcome value when the outcome alarm specified for the practice run configuration
         *        does not go into an <code>ALARM</code> state during the practice run, and the practice run was not
         *        interrupted before it completed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>INTERRUPTED:</b> Outcome value when the practice run did not run for the expected 30 minutes or
         *        there was another problem with the practice run that created an inconclusive outcome.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>FAILED:</b> Outcome value when the outcome alarm specified for the practice run configuration goes
         *        into an <code>ALARM</code> state during the practice run, and the practice run was not interrupted
         *        before it completed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CAPACITY_CHECK_FAILED:</b> The check for balanced capacity across Availability Zones for your load
         *        balancing and Auto Scaling group resources failed.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information about practice run outcomes, see <a
         *        href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.configure.html">
         *        Considerations when you configure zonal autoshift</a> in the Amazon Application Recovery Controller
         *        Developer Guide.
         * @see PracticeRunOutcome
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PracticeRunOutcome
         */
        Builder practiceRunOutcome(PracticeRunOutcome practiceRunOutcome);
    }

    static final class BuilderImpl implements Builder {
        private String zonalShiftId;

        private String resourceIdentifier;

        private String awayFrom;

        private Instant expiryTime;

        private Instant startTime;

        private String status;

        private String comment;

        private String shiftType;

        private String practiceRunOutcome;

        private BuilderImpl() {
        }

        private BuilderImpl(ZonalShiftSummary model) {
            zonalShiftId(model.zonalShiftId);
            resourceIdentifier(model.resourceIdentifier);
            awayFrom(model.awayFrom);
            expiryTime(model.expiryTime);
            startTime(model.startTime);
            status(model.status);
            comment(model.comment);
            shiftType(model.shiftType);
            practiceRunOutcome(model.practiceRunOutcome);
        }

        public final String getZonalShiftId() {
            return zonalShiftId;
        }

        public final void setZonalShiftId(String zonalShiftId) {
            this.zonalShiftId = zonalShiftId;
        }

        @Override
        public final Builder zonalShiftId(String zonalShiftId) {
            this.zonalShiftId = zonalShiftId;
            return this;
        }

        public final String getResourceIdentifier() {
            return resourceIdentifier;
        }

        public final void setResourceIdentifier(String resourceIdentifier) {
            this.resourceIdentifier = resourceIdentifier;
        }

        @Override
        public final Builder resourceIdentifier(String resourceIdentifier) {
            this.resourceIdentifier = resourceIdentifier;
            return this;
        }

        public final String getAwayFrom() {
            return awayFrom;
        }

        public final void setAwayFrom(String awayFrom) {
            this.awayFrom = awayFrom;
        }

        @Override
        public final Builder awayFrom(String awayFrom) {
            this.awayFrom = awayFrom;
            return this;
        }

        public final Instant getExpiryTime() {
            return expiryTime;
        }

        public final void setExpiryTime(Instant expiryTime) {
            this.expiryTime = expiryTime;
        }

        @Override
        public final Builder expiryTime(Instant expiryTime) {
            this.expiryTime = expiryTime;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ZonalShiftStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getComment() {
            return comment;
        }

        public final void setComment(String comment) {
            this.comment = comment;
        }

        @Override
        public final Builder comment(String comment) {
            this.comment = comment;
            return this;
        }

        public final String getShiftType() {
            return shiftType;
        }

        public final void setShiftType(String shiftType) {
            this.shiftType = shiftType;
        }

        @Override
        public final Builder shiftType(String shiftType) {
            this.shiftType = shiftType;
            return this;
        }

        @Override
        public final Builder shiftType(ShiftType shiftType) {
            this.shiftType(shiftType == null ? null : shiftType.toString());
            return this;
        }

        public final String getPracticeRunOutcome() {
            return practiceRunOutcome;
        }

        public final void setPracticeRunOutcome(String practiceRunOutcome) {
            this.practiceRunOutcome = practiceRunOutcome;
        }

        @Override
        public final Builder practiceRunOutcome(String practiceRunOutcome) {
            this.practiceRunOutcome = practiceRunOutcome;
            return this;
        }

        @Override
        public final Builder practiceRunOutcome(PracticeRunOutcome practiceRunOutcome) {
            this.practiceRunOutcome(practiceRunOutcome == null ? null : practiceRunOutcome.toString());
            return this;
        }

        @Override
        public ZonalShiftSummary build() {
            return new ZonalShiftSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
