/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.arczonalshift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex structure that lists an autoshift that is currently active for a managed resource and information about the
 * autoshift.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.how-it-works.html">How zonal autoshift
 * and practice runs work</a> in the Amazon Application Recovery Controller Developer Guide.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutoshiftInResource implements SdkPojo, Serializable,
        ToCopyableBuilder<AutoshiftInResource.Builder, AutoshiftInResource> {
    private static final SdkField<String> APPLIED_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("appliedStatus").getter(getter(AutoshiftInResource::appliedStatusAsString))
            .setter(setter(Builder::appliedStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("appliedStatus").build()).build();

    private static final SdkField<String> AWAY_FROM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("awayFrom").getter(getter(AutoshiftInResource::awayFrom)).setter(setter(Builder::awayFrom))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("awayFrom").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime").getter(getter(AutoshiftInResource::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APPLIED_STATUS_FIELD,
            AWAY_FROM_FIELD, START_TIME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String appliedStatus;

    private final String awayFrom;

    private final Instant startTime;

    private AutoshiftInResource(BuilderImpl builder) {
        this.appliedStatus = builder.appliedStatus;
        this.awayFrom = builder.awayFrom;
        this.startTime = builder.startTime;
    }

    /**
     * <p>
     * The <code>appliedStatus</code> field specifies which application traffic shift is in effect for a resource when
     * there is more than one active traffic shift. There can be more than one application traffic shift in progress at
     * the same time - that is, practice run zonal shifts, customer-initiated zonal shifts, or an autoshift. The
     * <code>appliedStatus</code> field for a shift that is in progress for a resource can have one of two values:
     * <code>APPLIED</code> or <code>NOT_APPLIED</code>. The zonal shift or autoshift that is currently in effect for
     * the resource has an <code>appliedStatus</code> set to <code>APPLIED</code>.
     * </p>
     * <p>
     * The overall principle for precedence is that zonal shifts that you start as a customer take precedence
     * autoshifts, which take precedence over practice runs. That is, customer-initiated zonal shifts &gt; autoshifts
     * &gt; practice run zonal shifts.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.how-it-works.html">How zonal
     * autoshift and practice runs work</a> in the Amazon Application Recovery Controller Developer Guide.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #appliedStatus}
     * will return {@link AutoshiftAppliedStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #appliedStatusAsString}.
     * </p>
     * 
     * @return The <code>appliedStatus</code> field specifies which application traffic shift is in effect for a
     *         resource when there is more than one active traffic shift. There can be more than one application traffic
     *         shift in progress at the same time - that is, practice run zonal shifts, customer-initiated zonal shifts,
     *         or an autoshift. The <code>appliedStatus</code> field for a shift that is in progress for a resource can
     *         have one of two values: <code>APPLIED</code> or <code>NOT_APPLIED</code>. The zonal shift or autoshift
     *         that is currently in effect for the resource has an <code>appliedStatus</code> set to
     *         <code>APPLIED</code>.</p>
     *         <p>
     *         The overall principle for precedence is that zonal shifts that you start as a customer take precedence
     *         autoshifts, which take precedence over practice runs. That is, customer-initiated zonal shifts &gt;
     *         autoshifts &gt; practice run zonal shifts.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.how-it-works.html">How zonal
     *         autoshift and practice runs work</a> in the Amazon Application Recovery Controller Developer Guide.
     * @see AutoshiftAppliedStatus
     */
    public final AutoshiftAppliedStatus appliedStatus() {
        return AutoshiftAppliedStatus.fromValue(appliedStatus);
    }

    /**
     * <p>
     * The <code>appliedStatus</code> field specifies which application traffic shift is in effect for a resource when
     * there is more than one active traffic shift. There can be more than one application traffic shift in progress at
     * the same time - that is, practice run zonal shifts, customer-initiated zonal shifts, or an autoshift. The
     * <code>appliedStatus</code> field for a shift that is in progress for a resource can have one of two values:
     * <code>APPLIED</code> or <code>NOT_APPLIED</code>. The zonal shift or autoshift that is currently in effect for
     * the resource has an <code>appliedStatus</code> set to <code>APPLIED</code>.
     * </p>
     * <p>
     * The overall principle for precedence is that zonal shifts that you start as a customer take precedence
     * autoshifts, which take precedence over practice runs. That is, customer-initiated zonal shifts &gt; autoshifts
     * &gt; practice run zonal shifts.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.how-it-works.html">How zonal
     * autoshift and practice runs work</a> in the Amazon Application Recovery Controller Developer Guide.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #appliedStatus}
     * will return {@link AutoshiftAppliedStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #appliedStatusAsString}.
     * </p>
     * 
     * @return The <code>appliedStatus</code> field specifies which application traffic shift is in effect for a
     *         resource when there is more than one active traffic shift. There can be more than one application traffic
     *         shift in progress at the same time - that is, practice run zonal shifts, customer-initiated zonal shifts,
     *         or an autoshift. The <code>appliedStatus</code> field for a shift that is in progress for a resource can
     *         have one of two values: <code>APPLIED</code> or <code>NOT_APPLIED</code>. The zonal shift or autoshift
     *         that is currently in effect for the resource has an <code>appliedStatus</code> set to
     *         <code>APPLIED</code>.</p>
     *         <p>
     *         The overall principle for precedence is that zonal shifts that you start as a customer take precedence
     *         autoshifts, which take precedence over practice runs. That is, customer-initiated zonal shifts &gt;
     *         autoshifts &gt; practice run zonal shifts.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.how-it-works.html">How zonal
     *         autoshift and practice runs work</a> in the Amazon Application Recovery Controller Developer Guide.
     * @see AutoshiftAppliedStatus
     */
    public final String appliedStatusAsString() {
        return appliedStatus;
    }

    /**
     * <p>
     * The Availability Zone (for example, <code>use1-az1</code>) that traffic is shifted away from for a resource, when
     * Amazon Web Services starts an autoshift. Until the autoshift ends, traffic for the resource is instead directed
     * to other Availability Zones in the Amazon Web Services Region. An autoshift can end for a resource, for example,
     * when Amazon Web Services ends the autoshift for the Availability Zone or when you disable zonal autoshift for the
     * resource.
     * </p>
     * 
     * @return The Availability Zone (for example, <code>use1-az1</code>) that traffic is shifted away from for a
     *         resource, when Amazon Web Services starts an autoshift. Until the autoshift ends, traffic for the
     *         resource is instead directed to other Availability Zones in the Amazon Web Services Region. An autoshift
     *         can end for a resource, for example, when Amazon Web Services ends the autoshift for the Availability
     *         Zone or when you disable zonal autoshift for the resource.
     */
    public final String awayFrom() {
        return awayFrom;
    }

    /**
     * <p>
     * The time (UTC) when the autoshift started.
     * </p>
     * 
     * @return The time (UTC) when the autoshift started.
     */
    public final Instant startTime() {
        return startTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(appliedStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(awayFrom());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutoshiftInResource)) {
            return false;
        }
        AutoshiftInResource other = (AutoshiftInResource) obj;
        return Objects.equals(appliedStatusAsString(), other.appliedStatusAsString())
                && Objects.equals(awayFrom(), other.awayFrom()) && Objects.equals(startTime(), other.startTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AutoshiftInResource").add("AppliedStatus", appliedStatusAsString()).add("AwayFrom", awayFrom())
                .add("StartTime", startTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "appliedStatus":
            return Optional.ofNullable(clazz.cast(appliedStatusAsString()));
        case "awayFrom":
            return Optional.ofNullable(clazz.cast(awayFrom()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("appliedStatus", APPLIED_STATUS_FIELD);
        map.put("awayFrom", AWAY_FROM_FIELD);
        map.put("startTime", START_TIME_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AutoshiftInResource, T> g) {
        return obj -> g.apply((AutoshiftInResource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AutoshiftInResource> {
        /**
         * <p>
         * The <code>appliedStatus</code> field specifies which application traffic shift is in effect for a resource
         * when there is more than one active traffic shift. There can be more than one application traffic shift in
         * progress at the same time - that is, practice run zonal shifts, customer-initiated zonal shifts, or an
         * autoshift. The <code>appliedStatus</code> field for a shift that is in progress for a resource can have one
         * of two values: <code>APPLIED</code> or <code>NOT_APPLIED</code>. The zonal shift or autoshift that is
         * currently in effect for the resource has an <code>appliedStatus</code> set to <code>APPLIED</code>.
         * </p>
         * <p>
         * The overall principle for precedence is that zonal shifts that you start as a customer take precedence
         * autoshifts, which take precedence over practice runs. That is, customer-initiated zonal shifts &gt;
         * autoshifts &gt; practice run zonal shifts.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.how-it-works.html">How zonal
         * autoshift and practice runs work</a> in the Amazon Application Recovery Controller Developer Guide.
         * </p>
         * 
         * @param appliedStatus
         *        The <code>appliedStatus</code> field specifies which application traffic shift is in effect for a
         *        resource when there is more than one active traffic shift. There can be more than one application
         *        traffic shift in progress at the same time - that is, practice run zonal shifts, customer-initiated
         *        zonal shifts, or an autoshift. The <code>appliedStatus</code> field for a shift that is in progress
         *        for a resource can have one of two values: <code>APPLIED</code> or <code>NOT_APPLIED</code>. The zonal
         *        shift or autoshift that is currently in effect for the resource has an <code>appliedStatus</code> set
         *        to <code>APPLIED</code>.</p>
         *        <p>
         *        The overall principle for precedence is that zonal shifts that you start as a customer take precedence
         *        autoshifts, which take precedence over practice runs. That is, customer-initiated zonal shifts &gt;
         *        autoshifts &gt; practice run zonal shifts.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.how-it-works.html">How
         *        zonal autoshift and practice runs work</a> in the Amazon Application Recovery Controller Developer
         *        Guide.
         * @see AutoshiftAppliedStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutoshiftAppliedStatus
         */
        Builder appliedStatus(String appliedStatus);

        /**
         * <p>
         * The <code>appliedStatus</code> field specifies which application traffic shift is in effect for a resource
         * when there is more than one active traffic shift. There can be more than one application traffic shift in
         * progress at the same time - that is, practice run zonal shifts, customer-initiated zonal shifts, or an
         * autoshift. The <code>appliedStatus</code> field for a shift that is in progress for a resource can have one
         * of two values: <code>APPLIED</code> or <code>NOT_APPLIED</code>. The zonal shift or autoshift that is
         * currently in effect for the resource has an <code>appliedStatus</code> set to <code>APPLIED</code>.
         * </p>
         * <p>
         * The overall principle for precedence is that zonal shifts that you start as a customer take precedence
         * autoshifts, which take precedence over practice runs. That is, customer-initiated zonal shifts &gt;
         * autoshifts &gt; practice run zonal shifts.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.how-it-works.html">How zonal
         * autoshift and practice runs work</a> in the Amazon Application Recovery Controller Developer Guide.
         * </p>
         * 
         * @param appliedStatus
         *        The <code>appliedStatus</code> field specifies which application traffic shift is in effect for a
         *        resource when there is more than one active traffic shift. There can be more than one application
         *        traffic shift in progress at the same time - that is, practice run zonal shifts, customer-initiated
         *        zonal shifts, or an autoshift. The <code>appliedStatus</code> field for a shift that is in progress
         *        for a resource can have one of two values: <code>APPLIED</code> or <code>NOT_APPLIED</code>. The zonal
         *        shift or autoshift that is currently in effect for the resource has an <code>appliedStatus</code> set
         *        to <code>APPLIED</code>.</p>
         *        <p>
         *        The overall principle for precedence is that zonal shifts that you start as a customer take precedence
         *        autoshifts, which take precedence over practice runs. That is, customer-initiated zonal shifts &gt;
         *        autoshifts &gt; practice run zonal shifts.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.how-it-works.html">How
         *        zonal autoshift and practice runs work</a> in the Amazon Application Recovery Controller Developer
         *        Guide.
         * @see AutoshiftAppliedStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutoshiftAppliedStatus
         */
        Builder appliedStatus(AutoshiftAppliedStatus appliedStatus);

        /**
         * <p>
         * The Availability Zone (for example, <code>use1-az1</code>) that traffic is shifted away from for a resource,
         * when Amazon Web Services starts an autoshift. Until the autoshift ends, traffic for the resource is instead
         * directed to other Availability Zones in the Amazon Web Services Region. An autoshift can end for a resource,
         * for example, when Amazon Web Services ends the autoshift for the Availability Zone or when you disable zonal
         * autoshift for the resource.
         * </p>
         * 
         * @param awayFrom
         *        The Availability Zone (for example, <code>use1-az1</code>) that traffic is shifted away from for a
         *        resource, when Amazon Web Services starts an autoshift. Until the autoshift ends, traffic for the
         *        resource is instead directed to other Availability Zones in the Amazon Web Services Region. An
         *        autoshift can end for a resource, for example, when Amazon Web Services ends the autoshift for the
         *        Availability Zone or when you disable zonal autoshift for the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awayFrom(String awayFrom);

        /**
         * <p>
         * The time (UTC) when the autoshift started.
         * </p>
         * 
         * @param startTime
         *        The time (UTC) when the autoshift started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);
    }

    static final class BuilderImpl implements Builder {
        private String appliedStatus;

        private String awayFrom;

        private Instant startTime;

        private BuilderImpl() {
        }

        private BuilderImpl(AutoshiftInResource model) {
            appliedStatus(model.appliedStatus);
            awayFrom(model.awayFrom);
            startTime(model.startTime);
        }

        public final String getAppliedStatus() {
            return appliedStatus;
        }

        public final void setAppliedStatus(String appliedStatus) {
            this.appliedStatus = appliedStatus;
        }

        @Override
        public final Builder appliedStatus(String appliedStatus) {
            this.appliedStatus = appliedStatus;
            return this;
        }

        @Override
        public final Builder appliedStatus(AutoshiftAppliedStatus appliedStatus) {
            this.appliedStatus(appliedStatus == null ? null : appliedStatus.toString());
            return this;
        }

        public final String getAwayFrom() {
            return awayFrom;
        }

        public final void setAwayFrom(String awayFrom) {
            this.awayFrom = awayFrom;
        }

        @Override
        public final Builder awayFrom(String awayFrom) {
            this.awayFrom = awayFrom;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        @Override
        public AutoshiftInResource build() {
            return new AutoshiftInResource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
