/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.artifact;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.artifact.model.AccessDeniedException;
import software.amazon.awssdk.services.artifact.model.ArtifactException;
import software.amazon.awssdk.services.artifact.model.ConflictException;
import software.amazon.awssdk.services.artifact.model.GetAccountSettingsRequest;
import software.amazon.awssdk.services.artifact.model.GetAccountSettingsResponse;
import software.amazon.awssdk.services.artifact.model.GetReportMetadataRequest;
import software.amazon.awssdk.services.artifact.model.GetReportMetadataResponse;
import software.amazon.awssdk.services.artifact.model.GetReportRequest;
import software.amazon.awssdk.services.artifact.model.GetReportResponse;
import software.amazon.awssdk.services.artifact.model.GetTermForReportRequest;
import software.amazon.awssdk.services.artifact.model.GetTermForReportResponse;
import software.amazon.awssdk.services.artifact.model.InternalServerException;
import software.amazon.awssdk.services.artifact.model.ListCustomerAgreementsRequest;
import software.amazon.awssdk.services.artifact.model.ListCustomerAgreementsResponse;
import software.amazon.awssdk.services.artifact.model.ListReportsRequest;
import software.amazon.awssdk.services.artifact.model.ListReportsResponse;
import software.amazon.awssdk.services.artifact.model.PutAccountSettingsRequest;
import software.amazon.awssdk.services.artifact.model.PutAccountSettingsResponse;
import software.amazon.awssdk.services.artifact.model.ResourceNotFoundException;
import software.amazon.awssdk.services.artifact.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.artifact.model.ThrottlingException;
import software.amazon.awssdk.services.artifact.model.ValidationException;
import software.amazon.awssdk.services.artifact.paginators.ListCustomerAgreementsIterable;
import software.amazon.awssdk.services.artifact.paginators.ListReportsIterable;

/**
 * Service client for accessing AWS Artifact. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * This reference provides descriptions of the low-level AWS Artifact Service API.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface ArtifactClient extends AwsClient {
    String SERVICE_NAME = "artifact";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "artifact";

    /**
     * <p>
     * Get the account settings for Artifact.
     * </p>
     *
     * @param getAccountSettingsRequest
     * @return Result of the GetAccountSettings operation returned by the service.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ConflictException
     *         Request to create/modify content would result in a conflict.
     * @throws InternalServerException
     *         An unknown server exception has occurred.
     * @throws ValidationException
     *         Request fails to satisfy the constraints specified by an AWS service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArtifactException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArtifactClient.GetAccountSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/artifact-2018-05-10/GetAccountSettings" target="_top">AWS
     *      API Documentation</a>
     */
    default GetAccountSettingsResponse getAccountSettings(GetAccountSettingsRequest getAccountSettingsRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ConflictException,
            InternalServerException, ValidationException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            ArtifactException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Get the account settings for Artifact.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetAccountSettingsRequest.Builder} avoiding the
     * need to create one manually via {@link GetAccountSettingsRequest#builder()}
     * </p>
     *
     * @param getAccountSettingsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.artifact.model.GetAccountSettingsRequest.Builder} to create a
     *        request.
     * @return Result of the GetAccountSettings operation returned by the service.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ConflictException
     *         Request to create/modify content would result in a conflict.
     * @throws InternalServerException
     *         An unknown server exception has occurred.
     * @throws ValidationException
     *         Request fails to satisfy the constraints specified by an AWS service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArtifactException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArtifactClient.GetAccountSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/artifact-2018-05-10/GetAccountSettings" target="_top">AWS
     *      API Documentation</a>
     */
    default GetAccountSettingsResponse getAccountSettings(Consumer<GetAccountSettingsRequest.Builder> getAccountSettingsRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ConflictException,
            InternalServerException, ValidationException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            ArtifactException {
        return getAccountSettings(GetAccountSettingsRequest.builder().applyMutation(getAccountSettingsRequest).build());
    }

    /**
     * <p>
     * Get the content for a single report.
     * </p>
     *
     * @param getReportRequest
     * @return Result of the GetReport operation returned by the service.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ConflictException
     *         Request to create/modify content would result in a conflict.
     * @throws InternalServerException
     *         An unknown server exception has occurred.
     * @throws ValidationException
     *         Request fails to satisfy the constraints specified by an AWS service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArtifactException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArtifactClient.GetReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/artifact-2018-05-10/GetReport" target="_top">AWS API
     *      Documentation</a>
     */
    default GetReportResponse getReport(GetReportRequest getReportRequest) throws ResourceNotFoundException, ThrottlingException,
            AccessDeniedException, ConflictException, InternalServerException, ValidationException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, ArtifactException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Get the content for a single report.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetReportRequest.Builder} avoiding the need to
     * create one manually via {@link GetReportRequest#builder()}
     * </p>
     *
     * @param getReportRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.artifact.model.GetReportRequest.Builder} to create a request.
     * @return Result of the GetReport operation returned by the service.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ConflictException
     *         Request to create/modify content would result in a conflict.
     * @throws InternalServerException
     *         An unknown server exception has occurred.
     * @throws ValidationException
     *         Request fails to satisfy the constraints specified by an AWS service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArtifactException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArtifactClient.GetReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/artifact-2018-05-10/GetReport" target="_top">AWS API
     *      Documentation</a>
     */
    default GetReportResponse getReport(Consumer<GetReportRequest.Builder> getReportRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ConflictException, InternalServerException, ValidationException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, ArtifactException {
        return getReport(GetReportRequest.builder().applyMutation(getReportRequest).build());
    }

    /**
     * <p>
     * Get the metadata for a single report.
     * </p>
     *
     * @param getReportMetadataRequest
     * @return Result of the GetReportMetadata operation returned by the service.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An unknown server exception has occurred.
     * @throws ValidationException
     *         Request fails to satisfy the constraints specified by an AWS service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArtifactException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArtifactClient.GetReportMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/artifact-2018-05-10/GetReportMetadata" target="_top">AWS
     *      API Documentation</a>
     */
    default GetReportMetadataResponse getReportMetadata(GetReportMetadataRequest getReportMetadataRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException,
            ValidationException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, ArtifactException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Get the metadata for a single report.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetReportMetadataRequest.Builder} avoiding the need
     * to create one manually via {@link GetReportMetadataRequest#builder()}
     * </p>
     *
     * @param getReportMetadataRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.artifact.model.GetReportMetadataRequest.Builder} to create a
     *        request.
     * @return Result of the GetReportMetadata operation returned by the service.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An unknown server exception has occurred.
     * @throws ValidationException
     *         Request fails to satisfy the constraints specified by an AWS service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArtifactException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArtifactClient.GetReportMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/artifact-2018-05-10/GetReportMetadata" target="_top">AWS
     *      API Documentation</a>
     */
    default GetReportMetadataResponse getReportMetadata(Consumer<GetReportMetadataRequest.Builder> getReportMetadataRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException,
            ValidationException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, ArtifactException {
        return getReportMetadata(GetReportMetadataRequest.builder().applyMutation(getReportMetadataRequest).build());
    }

    /**
     * <p>
     * Get the Term content associated with a single report.
     * </p>
     *
     * @param getTermForReportRequest
     * @return Result of the GetTermForReport operation returned by the service.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ConflictException
     *         Request to create/modify content would result in a conflict.
     * @throws InternalServerException
     *         An unknown server exception has occurred.
     * @throws ValidationException
     *         Request fails to satisfy the constraints specified by an AWS service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArtifactException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArtifactClient.GetTermForReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/artifact-2018-05-10/GetTermForReport" target="_top">AWS API
     *      Documentation</a>
     */
    default GetTermForReportResponse getTermForReport(GetTermForReportRequest getTermForReportRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ConflictException,
            InternalServerException, ValidationException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            ArtifactException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Get the Term content associated with a single report.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetTermForReportRequest.Builder} avoiding the need
     * to create one manually via {@link GetTermForReportRequest#builder()}
     * </p>
     *
     * @param getTermForReportRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.artifact.model.GetTermForReportRequest.Builder} to create a
     *        request.
     * @return Result of the GetTermForReport operation returned by the service.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ConflictException
     *         Request to create/modify content would result in a conflict.
     * @throws InternalServerException
     *         An unknown server exception has occurred.
     * @throws ValidationException
     *         Request fails to satisfy the constraints specified by an AWS service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArtifactException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArtifactClient.GetTermForReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/artifact-2018-05-10/GetTermForReport" target="_top">AWS API
     *      Documentation</a>
     */
    default GetTermForReportResponse getTermForReport(Consumer<GetTermForReportRequest.Builder> getTermForReportRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ConflictException,
            InternalServerException, ValidationException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            ArtifactException {
        return getTermForReport(GetTermForReportRequest.builder().applyMutation(getTermForReportRequest).build());
    }

    /**
     * <p>
     * List active customer-agreements applicable to calling identity.
     * </p>
     *
     * @param listCustomerAgreementsRequest
     * @return Result of the ListCustomerAgreements operation returned by the service.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An unknown server exception has occurred.
     * @throws ValidationException
     *         Request fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArtifactException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArtifactClient.ListCustomerAgreements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/artifact-2018-05-10/ListCustomerAgreements"
     *      target="_top">AWS API Documentation</a>
     */
    default ListCustomerAgreementsResponse listCustomerAgreements(ListCustomerAgreementsRequest listCustomerAgreementsRequest)
            throws ThrottlingException, AccessDeniedException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, ArtifactException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * List active customer-agreements applicable to calling identity.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListCustomerAgreementsRequest.Builder} avoiding the
     * need to create one manually via {@link ListCustomerAgreementsRequest#builder()}
     * </p>
     *
     * @param listCustomerAgreementsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.artifact.model.ListCustomerAgreementsRequest.Builder} to create a
     *        request.
     * @return Result of the ListCustomerAgreements operation returned by the service.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An unknown server exception has occurred.
     * @throws ValidationException
     *         Request fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArtifactException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArtifactClient.ListCustomerAgreements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/artifact-2018-05-10/ListCustomerAgreements"
     *      target="_top">AWS API Documentation</a>
     */
    default ListCustomerAgreementsResponse listCustomerAgreements(
            Consumer<ListCustomerAgreementsRequest.Builder> listCustomerAgreementsRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, AwsServiceException, SdkClientException,
            ArtifactException {
        return listCustomerAgreements(ListCustomerAgreementsRequest.builder().applyMutation(listCustomerAgreementsRequest)
                .build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listCustomerAgreements(software.amazon.awssdk.services.artifact.model.ListCustomerAgreementsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.artifact.paginators.ListCustomerAgreementsIterable responses = client.listCustomerAgreementsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.artifact.paginators.ListCustomerAgreementsIterable responses = client
     *             .listCustomerAgreementsPaginator(request);
     *     for (software.amazon.awssdk.services.artifact.model.ListCustomerAgreementsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.artifact.paginators.ListCustomerAgreementsIterable responses = client.listCustomerAgreementsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCustomerAgreements(software.amazon.awssdk.services.artifact.model.ListCustomerAgreementsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listCustomerAgreementsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An unknown server exception has occurred.
     * @throws ValidationException
     *         Request fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArtifactException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArtifactClient.ListCustomerAgreements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/artifact-2018-05-10/ListCustomerAgreements"
     *      target="_top">AWS API Documentation</a>
     */
    default ListCustomerAgreementsIterable listCustomerAgreementsPaginator(
            ListCustomerAgreementsRequest listCustomerAgreementsRequest) throws ThrottlingException, AccessDeniedException,
            InternalServerException, ValidationException, AwsServiceException, SdkClientException, ArtifactException {
        return new ListCustomerAgreementsIterable(this, listCustomerAgreementsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listCustomerAgreements(software.amazon.awssdk.services.artifact.model.ListCustomerAgreementsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.artifact.paginators.ListCustomerAgreementsIterable responses = client.listCustomerAgreementsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.artifact.paginators.ListCustomerAgreementsIterable responses = client
     *             .listCustomerAgreementsPaginator(request);
     *     for (software.amazon.awssdk.services.artifact.model.ListCustomerAgreementsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.artifact.paginators.ListCustomerAgreementsIterable responses = client.listCustomerAgreementsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCustomerAgreements(software.amazon.awssdk.services.artifact.model.ListCustomerAgreementsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListCustomerAgreementsRequest.Builder} avoiding the
     * need to create one manually via {@link ListCustomerAgreementsRequest#builder()}
     * </p>
     *
     * @param listCustomerAgreementsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.artifact.model.ListCustomerAgreementsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An unknown server exception has occurred.
     * @throws ValidationException
     *         Request fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArtifactException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArtifactClient.ListCustomerAgreements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/artifact-2018-05-10/ListCustomerAgreements"
     *      target="_top">AWS API Documentation</a>
     */
    default ListCustomerAgreementsIterable listCustomerAgreementsPaginator(
            Consumer<ListCustomerAgreementsRequest.Builder> listCustomerAgreementsRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, AwsServiceException, SdkClientException,
            ArtifactException {
        return listCustomerAgreementsPaginator(ListCustomerAgreementsRequest.builder()
                .applyMutation(listCustomerAgreementsRequest).build());
    }

    /**
     * <p>
     * List available reports.
     * </p>
     *
     * @param listReportsRequest
     * @return Result of the ListReports operation returned by the service.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An unknown server exception has occurred.
     * @throws ValidationException
     *         Request fails to satisfy the constraints specified by an AWS service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArtifactException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArtifactClient.ListReports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/artifact-2018-05-10/ListReports" target="_top">AWS API
     *      Documentation</a>
     */
    default ListReportsResponse listReports(ListReportsRequest listReportsRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, InternalServerException, ValidationException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, ArtifactException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * List available reports.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListReportsRequest.Builder} avoiding the need to
     * create one manually via {@link ListReportsRequest#builder()}
     * </p>
     *
     * @param listReportsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.artifact.model.ListReportsRequest.Builder} to create a request.
     * @return Result of the ListReports operation returned by the service.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An unknown server exception has occurred.
     * @throws ValidationException
     *         Request fails to satisfy the constraints specified by an AWS service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArtifactException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArtifactClient.ListReports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/artifact-2018-05-10/ListReports" target="_top">AWS API
     *      Documentation</a>
     */
    default ListReportsResponse listReports(Consumer<ListReportsRequest.Builder> listReportsRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException,
            ValidationException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, ArtifactException {
        return listReports(ListReportsRequest.builder().applyMutation(listReportsRequest).build());
    }

    /**
     * <p>
     * This is a variant of {@link #listReports(software.amazon.awssdk.services.artifact.model.ListReportsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.artifact.paginators.ListReportsIterable responses = client.listReportsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.artifact.paginators.ListReportsIterable responses = client.listReportsPaginator(request);
     *     for (software.amazon.awssdk.services.artifact.model.ListReportsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.artifact.paginators.ListReportsIterable responses = client.listReportsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listReports(software.amazon.awssdk.services.artifact.model.ListReportsRequest)} operation.</b>
     * </p>
     *
     * @param listReportsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An unknown server exception has occurred.
     * @throws ValidationException
     *         Request fails to satisfy the constraints specified by an AWS service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArtifactException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArtifactClient.ListReports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/artifact-2018-05-10/ListReports" target="_top">AWS API
     *      Documentation</a>
     */
    default ListReportsIterable listReportsPaginator(ListReportsRequest listReportsRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, InternalServerException, ValidationException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, ArtifactException {
        return new ListReportsIterable(this, listReportsRequest);
    }

    /**
     * <p>
     * This is a variant of {@link #listReports(software.amazon.awssdk.services.artifact.model.ListReportsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.artifact.paginators.ListReportsIterable responses = client.listReportsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.artifact.paginators.ListReportsIterable responses = client.listReportsPaginator(request);
     *     for (software.amazon.awssdk.services.artifact.model.ListReportsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.artifact.paginators.ListReportsIterable responses = client.listReportsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listReports(software.amazon.awssdk.services.artifact.model.ListReportsRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListReportsRequest.Builder} avoiding the need to
     * create one manually via {@link ListReportsRequest#builder()}
     * </p>
     *
     * @param listReportsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.artifact.model.ListReportsRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An unknown server exception has occurred.
     * @throws ValidationException
     *         Request fails to satisfy the constraints specified by an AWS service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArtifactException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArtifactClient.ListReports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/artifact-2018-05-10/ListReports" target="_top">AWS API
     *      Documentation</a>
     */
    default ListReportsIterable listReportsPaginator(Consumer<ListReportsRequest.Builder> listReportsRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException,
            ValidationException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, ArtifactException {
        return listReportsPaginator(ListReportsRequest.builder().applyMutation(listReportsRequest).build());
    }

    /**
     * <p>
     * Put the account settings for Artifact.
     * </p>
     *
     * @param putAccountSettingsRequest
     * @return Result of the PutAccountSettings operation returned by the service.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ConflictException
     *         Request to create/modify content would result in a conflict.
     * @throws InternalServerException
     *         An unknown server exception has occurred.
     * @throws ValidationException
     *         Request fails to satisfy the constraints specified by an AWS service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArtifactException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArtifactClient.PutAccountSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/artifact-2018-05-10/PutAccountSettings" target="_top">AWS
     *      API Documentation</a>
     */
    default PutAccountSettingsResponse putAccountSettings(PutAccountSettingsRequest putAccountSettingsRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ConflictException,
            InternalServerException, ValidationException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            ArtifactException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Put the account settings for Artifact.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutAccountSettingsRequest.Builder} avoiding the
     * need to create one manually via {@link PutAccountSettingsRequest#builder()}
     * </p>
     *
     * @param putAccountSettingsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.artifact.model.PutAccountSettingsRequest.Builder} to create a
     *        request.
     * @return Result of the PutAccountSettings operation returned by the service.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ConflictException
     *         Request to create/modify content would result in a conflict.
     * @throws InternalServerException
     *         An unknown server exception has occurred.
     * @throws ValidationException
     *         Request fails to satisfy the constraints specified by an AWS service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArtifactException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArtifactClient.PutAccountSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/artifact-2018-05-10/PutAccountSettings" target="_top">AWS
     *      API Documentation</a>
     */
    default PutAccountSettingsResponse putAccountSettings(Consumer<PutAccountSettingsRequest.Builder> putAccountSettingsRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ConflictException,
            InternalServerException, ValidationException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            ArtifactException {
        return putAccountSettings(PutAccountSettingsRequest.builder().applyMutation(putAccountSettingsRequest).build());
    }

    /**
     * Create a {@link ArtifactClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static ArtifactClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link ArtifactClient}.
     */
    static ArtifactClientBuilder builder() {
        return new DefaultArtifactClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default ArtifactServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
