/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.athena;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.services.athena.model.BatchGetNamedQueryRequest;
import software.amazon.awssdk.services.athena.model.BatchGetNamedQueryResponse;
import software.amazon.awssdk.services.athena.model.BatchGetQueryExecutionRequest;
import software.amazon.awssdk.services.athena.model.BatchGetQueryExecutionResponse;
import software.amazon.awssdk.services.athena.model.CreateNamedQueryRequest;
import software.amazon.awssdk.services.athena.model.CreateNamedQueryResponse;
import software.amazon.awssdk.services.athena.model.CreateWorkGroupRequest;
import software.amazon.awssdk.services.athena.model.CreateWorkGroupResponse;
import software.amazon.awssdk.services.athena.model.DeleteNamedQueryRequest;
import software.amazon.awssdk.services.athena.model.DeleteNamedQueryResponse;
import software.amazon.awssdk.services.athena.model.DeleteWorkGroupRequest;
import software.amazon.awssdk.services.athena.model.DeleteWorkGroupResponse;
import software.amazon.awssdk.services.athena.model.GetNamedQueryRequest;
import software.amazon.awssdk.services.athena.model.GetNamedQueryResponse;
import software.amazon.awssdk.services.athena.model.GetQueryExecutionRequest;
import software.amazon.awssdk.services.athena.model.GetQueryExecutionResponse;
import software.amazon.awssdk.services.athena.model.GetQueryResultsRequest;
import software.amazon.awssdk.services.athena.model.GetQueryResultsResponse;
import software.amazon.awssdk.services.athena.model.GetWorkGroupRequest;
import software.amazon.awssdk.services.athena.model.GetWorkGroupResponse;
import software.amazon.awssdk.services.athena.model.ListNamedQueriesRequest;
import software.amazon.awssdk.services.athena.model.ListNamedQueriesResponse;
import software.amazon.awssdk.services.athena.model.ListQueryExecutionsRequest;
import software.amazon.awssdk.services.athena.model.ListQueryExecutionsResponse;
import software.amazon.awssdk.services.athena.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.athena.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.athena.model.ListWorkGroupsRequest;
import software.amazon.awssdk.services.athena.model.ListWorkGroupsResponse;
import software.amazon.awssdk.services.athena.model.StartQueryExecutionRequest;
import software.amazon.awssdk.services.athena.model.StartQueryExecutionResponse;
import software.amazon.awssdk.services.athena.model.StopQueryExecutionRequest;
import software.amazon.awssdk.services.athena.model.StopQueryExecutionResponse;
import software.amazon.awssdk.services.athena.model.TagResourceRequest;
import software.amazon.awssdk.services.athena.model.TagResourceResponse;
import software.amazon.awssdk.services.athena.model.UntagResourceRequest;
import software.amazon.awssdk.services.athena.model.UntagResourceResponse;
import software.amazon.awssdk.services.athena.model.UpdateWorkGroupRequest;
import software.amazon.awssdk.services.athena.model.UpdateWorkGroupResponse;
import software.amazon.awssdk.services.athena.paginators.GetQueryResultsPublisher;
import software.amazon.awssdk.services.athena.paginators.ListNamedQueriesPublisher;
import software.amazon.awssdk.services.athena.paginators.ListQueryExecutionsPublisher;
import software.amazon.awssdk.services.athena.paginators.ListWorkGroupsPublisher;

/**
 * Service client for accessing Amazon Athena asynchronously. This can be created using the static {@link #builder()}
 * method.
 *
 * <p>
 * Amazon Athena is an interactive query service that lets you use standard SQL to analyze data directly in Amazon S3.
 * You can point Athena at your data in Amazon S3 and run ad-hoc queries and get results in seconds. Athena is
 * serverless, so there is no infrastructure to set up or manage. You pay only for the queries you run. Athena scales
 * automatically—executing queries in parallel—so results are fast, even with large datasets and complex queries. For
 * more information, see <a href="http://docs.aws.amazon.com/athena/latest/ug/what-is.html">What is Amazon Athena</a> in
 * the <i>Amazon Athena User Guide</i>.
 * </p>
 * <p>
 * If you connect to Athena using the JDBC driver, use version 1.1.0 of the driver or later with the Amazon Athena API.
 * Earlier version drivers do not support the API. For more information and to download the driver, see <a
 * href="https://docs.aws.amazon.com/athena/latest/ug/connect-with-jdbc.html">Accessing Amazon Athena with JDBC</a>.
 * </p>
 * <p>
 * For code samples using the AWS SDK for Java, see <a
 * href="https://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the <i>Amazon
 * Athena User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public interface AthenaAsyncClient extends SdkClient {
    String SERVICE_NAME = "athena";

    /**
     * Create a {@link AthenaAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static AthenaAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link AthenaAsyncClient}.
     */
    static AthenaAsyncClientBuilder builder() {
        return new DefaultAthenaAsyncClientBuilder();
    }

    /**
     * <p>
     * Returns the details of a single named query or a list of up to 50 queries, which you provide as an array of query
     * ID strings. Requires you to have access to the workgroup in which the queries were saved. Use
     * <a>ListNamedQueriesInput</a> to get the list of named query IDs in the specified workgroup. If information could
     * not be retrieved for a submitted query ID, information about the query ID submitted is listed under
     * <a>UnprocessedNamedQueryId</a>. Named queries differ from executed queries. Use
     * <a>BatchGetQueryExecutionInput</a> to get details about each unique query execution, and
     * <a>ListQueryExecutionsInput</a> to get a list of query execution IDs.
     * </p>
     *
     * @param batchGetNamedQueryRequest
     * @return A Java Future containing the result of the BatchGetNamedQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.BatchGetNamedQuery
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/BatchGetNamedQuery" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<BatchGetNamedQueryResponse> batchGetNamedQuery(BatchGetNamedQueryRequest batchGetNamedQueryRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the details of a single named query or a list of up to 50 queries, which you provide as an array of query
     * ID strings. Requires you to have access to the workgroup in which the queries were saved. Use
     * <a>ListNamedQueriesInput</a> to get the list of named query IDs in the specified workgroup. If information could
     * not be retrieved for a submitted query ID, information about the query ID submitted is listed under
     * <a>UnprocessedNamedQueryId</a>. Named queries differ from executed queries. Use
     * <a>BatchGetQueryExecutionInput</a> to get details about each unique query execution, and
     * <a>ListQueryExecutionsInput</a> to get a list of query execution IDs.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link BatchGetNamedQueryRequest.Builder} avoiding the
     * need to create one manually via {@link BatchGetNamedQueryRequest#builder()}
     * </p>
     *
     * @param batchGetNamedQueryRequest
     *        A {@link Consumer} that will call methods on {@link BatchGetNamedQueryInput.Builder} to create a request.
     * @return A Java Future containing the result of the BatchGetNamedQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.BatchGetNamedQuery
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/BatchGetNamedQuery" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<BatchGetNamedQueryResponse> batchGetNamedQuery(
            Consumer<BatchGetNamedQueryRequest.Builder> batchGetNamedQueryRequest) {
        return batchGetNamedQuery(BatchGetNamedQueryRequest.builder().applyMutation(batchGetNamedQueryRequest).build());
    }

    /**
     * <p>
     * Returns the details of a single query execution or a list of up to 50 query executions, which you provide as an
     * array of query execution ID strings. Requires you to have access to the workgroup in which the queries ran. To
     * get a list of query execution IDs, use <a>ListQueryExecutionsInput&#36WorkGroup</a>. Query executions differ from
     * named (saved) queries. Use <a>BatchGetNamedQueryInput</a> to get details about named queries.
     * </p>
     *
     * @param batchGetQueryExecutionRequest
     * @return A Java Future containing the result of the BatchGetQueryExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.BatchGetQueryExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/BatchGetQueryExecution" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<BatchGetQueryExecutionResponse> batchGetQueryExecution(
            BatchGetQueryExecutionRequest batchGetQueryExecutionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the details of a single query execution or a list of up to 50 query executions, which you provide as an
     * array of query execution ID strings. Requires you to have access to the workgroup in which the queries ran. To
     * get a list of query execution IDs, use <a>ListQueryExecutionsInput&#36WorkGroup</a>. Query executions differ from
     * named (saved) queries. Use <a>BatchGetNamedQueryInput</a> to get details about named queries.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link BatchGetQueryExecutionRequest.Builder} avoiding the
     * need to create one manually via {@link BatchGetQueryExecutionRequest#builder()}
     * </p>
     *
     * @param batchGetQueryExecutionRequest
     *        A {@link Consumer} that will call methods on {@link BatchGetQueryExecutionInput.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the BatchGetQueryExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.BatchGetQueryExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/BatchGetQueryExecution" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<BatchGetQueryExecutionResponse> batchGetQueryExecution(
            Consumer<BatchGetQueryExecutionRequest.Builder> batchGetQueryExecutionRequest) {
        return batchGetQueryExecution(BatchGetQueryExecutionRequest.builder().applyMutation(batchGetQueryExecutionRequest)
                .build());
    }

    /**
     * <p>
     * Creates a named query in the specified workgroup. Requires that you have access to the workgroup.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     *
     * @param createNamedQueryRequest
     * @return A Java Future containing the result of the CreateNamedQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.CreateNamedQuery
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/CreateNamedQuery" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateNamedQueryResponse> createNamedQuery(CreateNamedQueryRequest createNamedQueryRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a named query in the specified workgroup. Requires that you have access to the workgroup.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateNamedQueryRequest.Builder} avoiding the need
     * to create one manually via {@link CreateNamedQueryRequest#builder()}
     * </p>
     *
     * @param createNamedQueryRequest
     *        A {@link Consumer} that will call methods on {@link CreateNamedQueryInput.Builder} to create a request.
     * @return A Java Future containing the result of the CreateNamedQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.CreateNamedQuery
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/CreateNamedQuery" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateNamedQueryResponse> createNamedQuery(
            Consumer<CreateNamedQueryRequest.Builder> createNamedQueryRequest) {
        return createNamedQuery(CreateNamedQueryRequest.builder().applyMutation(createNamedQueryRequest).build());
    }

    /**
     * <p>
     * Creates a workgroup with the specified name.
     * </p>
     *
     * @param createWorkGroupRequest
     * @return A Java Future containing the result of the CreateWorkGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.CreateWorkGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/CreateWorkGroup" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateWorkGroupResponse> createWorkGroup(CreateWorkGroupRequest createWorkGroupRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a workgroup with the specified name.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateWorkGroupRequest.Builder} avoiding the need
     * to create one manually via {@link CreateWorkGroupRequest#builder()}
     * </p>
     *
     * @param createWorkGroupRequest
     *        A {@link Consumer} that will call methods on {@link CreateWorkGroupInput.Builder} to create a request.
     * @return A Java Future containing the result of the CreateWorkGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.CreateWorkGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/CreateWorkGroup" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateWorkGroupResponse> createWorkGroup(
            Consumer<CreateWorkGroupRequest.Builder> createWorkGroupRequest) {
        return createWorkGroup(CreateWorkGroupRequest.builder().applyMutation(createWorkGroupRequest).build());
    }

    /**
     * <p>
     * Deletes the named query if you have access to the workgroup in which the query was saved.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     *
     * @param deleteNamedQueryRequest
     * @return A Java Future containing the result of the DeleteNamedQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.DeleteNamedQuery
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/DeleteNamedQuery" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteNamedQueryResponse> deleteNamedQuery(DeleteNamedQueryRequest deleteNamedQueryRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the named query if you have access to the workgroup in which the query was saved.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteNamedQueryRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteNamedQueryRequest#builder()}
     * </p>
     *
     * @param deleteNamedQueryRequest
     *        A {@link Consumer} that will call methods on {@link DeleteNamedQueryInput.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteNamedQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.DeleteNamedQuery
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/DeleteNamedQuery" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteNamedQueryResponse> deleteNamedQuery(
            Consumer<DeleteNamedQueryRequest.Builder> deleteNamedQueryRequest) {
        return deleteNamedQuery(DeleteNamedQueryRequest.builder().applyMutation(deleteNamedQueryRequest).build());
    }

    /**
     * <p>
     * Deletes the workgroup with the specified name. The primary workgroup cannot be deleted.
     * </p>
     *
     * @param deleteWorkGroupRequest
     * @return A Java Future containing the result of the DeleteWorkGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.DeleteWorkGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/DeleteWorkGroup" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteWorkGroupResponse> deleteWorkGroup(DeleteWorkGroupRequest deleteWorkGroupRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the workgroup with the specified name. The primary workgroup cannot be deleted.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteWorkGroupRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteWorkGroupRequest#builder()}
     * </p>
     *
     * @param deleteWorkGroupRequest
     *        A {@link Consumer} that will call methods on {@link DeleteWorkGroupInput.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteWorkGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.DeleteWorkGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/DeleteWorkGroup" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteWorkGroupResponse> deleteWorkGroup(
            Consumer<DeleteWorkGroupRequest.Builder> deleteWorkGroupRequest) {
        return deleteWorkGroup(DeleteWorkGroupRequest.builder().applyMutation(deleteWorkGroupRequest).build());
    }

    /**
     * <p>
     * Returns information about a single query. Requires that you have access to the workgroup in which the query was
     * saved.
     * </p>
     *
     * @param getNamedQueryRequest
     * @return A Java Future containing the result of the GetNamedQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.GetNamedQuery
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/GetNamedQuery" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetNamedQueryResponse> getNamedQuery(GetNamedQueryRequest getNamedQueryRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns information about a single query. Requires that you have access to the workgroup in which the query was
     * saved.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetNamedQueryRequest.Builder} avoiding the need to
     * create one manually via {@link GetNamedQueryRequest#builder()}
     * </p>
     *
     * @param getNamedQueryRequest
     *        A {@link Consumer} that will call methods on {@link GetNamedQueryInput.Builder} to create a request.
     * @return A Java Future containing the result of the GetNamedQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.GetNamedQuery
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/GetNamedQuery" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetNamedQueryResponse> getNamedQuery(Consumer<GetNamedQueryRequest.Builder> getNamedQueryRequest) {
        return getNamedQuery(GetNamedQueryRequest.builder().applyMutation(getNamedQueryRequest).build());
    }

    /**
     * <p>
     * Returns information about a single execution of a query if you have access to the workgroup in which the query
     * ran. Each time a query executes, information about the query execution is saved with a unique ID.
     * </p>
     *
     * @param getQueryExecutionRequest
     * @return A Java Future containing the result of the GetQueryExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.GetQueryExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/GetQueryExecution" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetQueryExecutionResponse> getQueryExecution(GetQueryExecutionRequest getQueryExecutionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns information about a single execution of a query if you have access to the workgroup in which the query
     * ran. Each time a query executes, information about the query execution is saved with a unique ID.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetQueryExecutionRequest.Builder} avoiding the need
     * to create one manually via {@link GetQueryExecutionRequest#builder()}
     * </p>
     *
     * @param getQueryExecutionRequest
     *        A {@link Consumer} that will call methods on {@link GetQueryExecutionInput.Builder} to create a request.
     * @return A Java Future containing the result of the GetQueryExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.GetQueryExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/GetQueryExecution" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetQueryExecutionResponse> getQueryExecution(
            Consumer<GetQueryExecutionRequest.Builder> getQueryExecutionRequest) {
        return getQueryExecution(GetQueryExecutionRequest.builder().applyMutation(getQueryExecutionRequest).build());
    }

    /**
     * <p>
     * Streams the results of a single query execution specified by <code>QueryExecutionId</code> from the Athena query
     * results location in Amazon S3. For more information, see <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/querying.html">Query Results</a> in the <i>Amazon Athena User
     * Guide</i>. This request does not execute the query but returns results. Use <a>StartQueryExecution</a> to run a
     * query.
     * </p>
     * <p>
     * To stream query results successfully, the IAM principal with permission to call <code>GetQueryResults</code> also
     * must have permissions to the Amazon S3 <code>GetObject</code> action for the Athena query results location.
     * </p>
     * <important>
     * <p>
     * IAM principals with permission to the Amazon S3 <code>GetObject</code> action for the query results location are
     * able to retrieve query results from Amazon S3 even if permission to the <code>GetQueryResults</code> action is
     * denied. To restrict user or role access, ensure that Amazon S3 permissions to the Athena query location are
     * denied.
     * </p>
     * </important>
     *
     * @param getQueryResultsRequest
     * @return A Java Future containing the result of the GetQueryResults operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.GetQueryResults
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/GetQueryResults" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetQueryResultsResponse> getQueryResults(GetQueryResultsRequest getQueryResultsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Streams the results of a single query execution specified by <code>QueryExecutionId</code> from the Athena query
     * results location in Amazon S3. For more information, see <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/querying.html">Query Results</a> in the <i>Amazon Athena User
     * Guide</i>. This request does not execute the query but returns results. Use <a>StartQueryExecution</a> to run a
     * query.
     * </p>
     * <p>
     * To stream query results successfully, the IAM principal with permission to call <code>GetQueryResults</code> also
     * must have permissions to the Amazon S3 <code>GetObject</code> action for the Athena query results location.
     * </p>
     * <important>
     * <p>
     * IAM principals with permission to the Amazon S3 <code>GetObject</code> action for the query results location are
     * able to retrieve query results from Amazon S3 even if permission to the <code>GetQueryResults</code> action is
     * denied. To restrict user or role access, ensure that Amazon S3 permissions to the Athena query location are
     * denied.
     * </p>
     * </important><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetQueryResultsRequest.Builder} avoiding the need
     * to create one manually via {@link GetQueryResultsRequest#builder()}
     * </p>
     *
     * @param getQueryResultsRequest
     *        A {@link Consumer} that will call methods on {@link GetQueryResultsInput.Builder} to create a request.
     * @return A Java Future containing the result of the GetQueryResults operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.GetQueryResults
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/GetQueryResults" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetQueryResultsResponse> getQueryResults(
            Consumer<GetQueryResultsRequest.Builder> getQueryResultsRequest) {
        return getQueryResults(GetQueryResultsRequest.builder().applyMutation(getQueryResultsRequest).build());
    }

    /**
     * <p>
     * Streams the results of a single query execution specified by <code>QueryExecutionId</code> from the Athena query
     * results location in Amazon S3. For more information, see <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/querying.html">Query Results</a> in the <i>Amazon Athena User
     * Guide</i>. This request does not execute the query but returns results. Use <a>StartQueryExecution</a> to run a
     * query.
     * </p>
     * <p>
     * To stream query results successfully, the IAM principal with permission to call <code>GetQueryResults</code> also
     * must have permissions to the Amazon S3 <code>GetObject</code> action for the Athena query results location.
     * </p>
     * <important>
     * <p>
     * IAM principals with permission to the Amazon S3 <code>GetObject</code> action for the query results location are
     * able to retrieve query results from Amazon S3 even if permission to the <code>GetQueryResults</code> action is
     * denied. To restrict user or role access, ensure that Amazon S3 permissions to the Athena query location are
     * denied.
     * </p>
     * </important><br/>
     * <p>
     * This is a variant of
     * {@link #getQueryResults(software.amazon.awssdk.services.athena.model.GetQueryResultsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.GetQueryResultsPublisher publisher = client.getQueryResultsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.GetQueryResultsPublisher publisher = client.getQueryResultsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.athena.model.GetQueryResultsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.athena.model.GetQueryResultsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getQueryResults(software.amazon.awssdk.services.athena.model.GetQueryResultsRequest)} operation.</b>
     * </p>
     *
     * @param getQueryResultsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.GetQueryResults
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/GetQueryResults" target="_top">AWS API
     *      Documentation</a>
     */
    default GetQueryResultsPublisher getQueryResultsPaginator(GetQueryResultsRequest getQueryResultsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Streams the results of a single query execution specified by <code>QueryExecutionId</code> from the Athena query
     * results location in Amazon S3. For more information, see <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/querying.html">Query Results</a> in the <i>Amazon Athena User
     * Guide</i>. This request does not execute the query but returns results. Use <a>StartQueryExecution</a> to run a
     * query.
     * </p>
     * <p>
     * To stream query results successfully, the IAM principal with permission to call <code>GetQueryResults</code> also
     * must have permissions to the Amazon S3 <code>GetObject</code> action for the Athena query results location.
     * </p>
     * <important>
     * <p>
     * IAM principals with permission to the Amazon S3 <code>GetObject</code> action for the query results location are
     * able to retrieve query results from Amazon S3 even if permission to the <code>GetQueryResults</code> action is
     * denied. To restrict user or role access, ensure that Amazon S3 permissions to the Athena query location are
     * denied.
     * </p>
     * </important><br/>
     * <p>
     * This is a variant of
     * {@link #getQueryResults(software.amazon.awssdk.services.athena.model.GetQueryResultsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.GetQueryResultsPublisher publisher = client.getQueryResultsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.GetQueryResultsPublisher publisher = client.getQueryResultsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.athena.model.GetQueryResultsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.athena.model.GetQueryResultsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getQueryResults(software.amazon.awssdk.services.athena.model.GetQueryResultsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link GetQueryResultsRequest.Builder} avoiding the need
     * to create one manually via {@link GetQueryResultsRequest#builder()}
     * </p>
     *
     * @param getQueryResultsRequest
     *        A {@link Consumer} that will call methods on {@link GetQueryResultsInput.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.GetQueryResults
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/GetQueryResults" target="_top">AWS API
     *      Documentation</a>
     */
    default GetQueryResultsPublisher getQueryResultsPaginator(Consumer<GetQueryResultsRequest.Builder> getQueryResultsRequest) {
        return getQueryResultsPaginator(GetQueryResultsRequest.builder().applyMutation(getQueryResultsRequest).build());
    }

    /**
     * <p>
     * Returns information about the workgroup with the specified name.
     * </p>
     *
     * @param getWorkGroupRequest
     * @return A Java Future containing the result of the GetWorkGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.GetWorkGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/GetWorkGroup" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetWorkGroupResponse> getWorkGroup(GetWorkGroupRequest getWorkGroupRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns information about the workgroup with the specified name.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetWorkGroupRequest.Builder} avoiding the need to
     * create one manually via {@link GetWorkGroupRequest#builder()}
     * </p>
     *
     * @param getWorkGroupRequest
     *        A {@link Consumer} that will call methods on {@link GetWorkGroupInput.Builder} to create a request.
     * @return A Java Future containing the result of the GetWorkGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.GetWorkGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/GetWorkGroup" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetWorkGroupResponse> getWorkGroup(Consumer<GetWorkGroupRequest.Builder> getWorkGroupRequest) {
        return getWorkGroup(GetWorkGroupRequest.builder().applyMutation(getWorkGroupRequest).build());
    }

    /**
     * <p>
     * Provides a list of available query IDs only for queries saved in the specified workgroup. Requires that you have
     * access to the workgroup.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     *
     * @param listNamedQueriesRequest
     * @return A Java Future containing the result of the ListNamedQueries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListNamedQueries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListNamedQueries" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListNamedQueriesResponse> listNamedQueries(ListNamedQueriesRequest listNamedQueriesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Provides a list of available query IDs only for queries saved in the specified workgroup. Requires that you have
     * access to the workgroup.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListNamedQueriesRequest.Builder} avoiding the need
     * to create one manually via {@link ListNamedQueriesRequest#builder()}
     * </p>
     *
     * @param listNamedQueriesRequest
     *        A {@link Consumer} that will call methods on {@link ListNamedQueriesInput.Builder} to create a request.
     * @return A Java Future containing the result of the ListNamedQueries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListNamedQueries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListNamedQueries" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListNamedQueriesResponse> listNamedQueries(
            Consumer<ListNamedQueriesRequest.Builder> listNamedQueriesRequest) {
        return listNamedQueries(ListNamedQueriesRequest.builder().applyMutation(listNamedQueriesRequest).build());
    }

    /**
     * <p>
     * Provides a list of available query IDs only for queries saved in the specified workgroup. Requires that you have
     * access to the workgroup.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     *
     * @return A Java Future containing the result of the ListNamedQueries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListNamedQueries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListNamedQueries" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListNamedQueriesResponse> listNamedQueries() {
        return listNamedQueries(ListNamedQueriesRequest.builder().build());
    }

    /**
     * <p>
     * Provides a list of available query IDs only for queries saved in the specified workgroup. Requires that you have
     * access to the workgroup.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNamedQueries(software.amazon.awssdk.services.athena.model.ListNamedQueriesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListNamedQueriesPublisher publisher = client.listNamedQueriesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListNamedQueriesPublisher publisher = client.listNamedQueriesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.athena.model.ListNamedQueriesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.athena.model.ListNamedQueriesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNamedQueries(software.amazon.awssdk.services.athena.model.ListNamedQueriesRequest)} operation.</b>
     * </p>
     *
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListNamedQueries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListNamedQueries" target="_top">AWS API
     *      Documentation</a>
     */
    default ListNamedQueriesPublisher listNamedQueriesPaginator() {
        return listNamedQueriesPaginator(ListNamedQueriesRequest.builder().build());
    }

    /**
     * <p>
     * Provides a list of available query IDs only for queries saved in the specified workgroup. Requires that you have
     * access to the workgroup.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNamedQueries(software.amazon.awssdk.services.athena.model.ListNamedQueriesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListNamedQueriesPublisher publisher = client.listNamedQueriesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListNamedQueriesPublisher publisher = client.listNamedQueriesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.athena.model.ListNamedQueriesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.athena.model.ListNamedQueriesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNamedQueries(software.amazon.awssdk.services.athena.model.ListNamedQueriesRequest)} operation.</b>
     * </p>
     *
     * @param listNamedQueriesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListNamedQueries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListNamedQueries" target="_top">AWS API
     *      Documentation</a>
     */
    default ListNamedQueriesPublisher listNamedQueriesPaginator(ListNamedQueriesRequest listNamedQueriesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Provides a list of available query IDs only for queries saved in the specified workgroup. Requires that you have
     * access to the workgroup.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNamedQueries(software.amazon.awssdk.services.athena.model.ListNamedQueriesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListNamedQueriesPublisher publisher = client.listNamedQueriesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListNamedQueriesPublisher publisher = client.listNamedQueriesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.athena.model.ListNamedQueriesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.athena.model.ListNamedQueriesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNamedQueries(software.amazon.awssdk.services.athena.model.ListNamedQueriesRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListNamedQueriesRequest.Builder} avoiding the need
     * to create one manually via {@link ListNamedQueriesRequest#builder()}
     * </p>
     *
     * @param listNamedQueriesRequest
     *        A {@link Consumer} that will call methods on {@link ListNamedQueriesInput.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListNamedQueries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListNamedQueries" target="_top">AWS API
     *      Documentation</a>
     */
    default ListNamedQueriesPublisher listNamedQueriesPaginator(Consumer<ListNamedQueriesRequest.Builder> listNamedQueriesRequest) {
        return listNamedQueriesPaginator(ListNamedQueriesRequest.builder().applyMutation(listNamedQueriesRequest).build());
    }

    /**
     * <p>
     * Provides a list of available query execution IDs for the queries in the specified workgroup. Requires you to have
     * access to the workgroup in which the queries ran.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     *
     * @param listQueryExecutionsRequest
     * @return A Java Future containing the result of the ListQueryExecutions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListQueryExecutions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListQueryExecutions" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListQueryExecutionsResponse> listQueryExecutions(
            ListQueryExecutionsRequest listQueryExecutionsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Provides a list of available query execution IDs for the queries in the specified workgroup. Requires you to have
     * access to the workgroup in which the queries ran.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListQueryExecutionsRequest.Builder} avoiding the
     * need to create one manually via {@link ListQueryExecutionsRequest#builder()}
     * </p>
     *
     * @param listQueryExecutionsRequest
     *        A {@link Consumer} that will call methods on {@link ListQueryExecutionsInput.Builder} to create a request.
     * @return A Java Future containing the result of the ListQueryExecutions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListQueryExecutions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListQueryExecutions" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListQueryExecutionsResponse> listQueryExecutions(
            Consumer<ListQueryExecutionsRequest.Builder> listQueryExecutionsRequest) {
        return listQueryExecutions(ListQueryExecutionsRequest.builder().applyMutation(listQueryExecutionsRequest).build());
    }

    /**
     * <p>
     * Provides a list of available query execution IDs for the queries in the specified workgroup. Requires you to have
     * access to the workgroup in which the queries ran.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     *
     * @return A Java Future containing the result of the ListQueryExecutions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListQueryExecutions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListQueryExecutions" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListQueryExecutionsResponse> listQueryExecutions() {
        return listQueryExecutions(ListQueryExecutionsRequest.builder().build());
    }

    /**
     * <p>
     * Provides a list of available query execution IDs for the queries in the specified workgroup. Requires you to have
     * access to the workgroup in which the queries ran.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listQueryExecutions(software.amazon.awssdk.services.athena.model.ListQueryExecutionsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListQueryExecutionsPublisher publisher = client.listQueryExecutionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListQueryExecutionsPublisher publisher = client.listQueryExecutionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.athena.model.ListQueryExecutionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.athena.model.ListQueryExecutionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listQueryExecutions(software.amazon.awssdk.services.athena.model.ListQueryExecutionsRequest)}
     * operation.</b>
     * </p>
     *
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListQueryExecutions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListQueryExecutions" target="_top">AWS API
     *      Documentation</a>
     */
    default ListQueryExecutionsPublisher listQueryExecutionsPaginator() {
        return listQueryExecutionsPaginator(ListQueryExecutionsRequest.builder().build());
    }

    /**
     * <p>
     * Provides a list of available query execution IDs for the queries in the specified workgroup. Requires you to have
     * access to the workgroup in which the queries ran.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listQueryExecutions(software.amazon.awssdk.services.athena.model.ListQueryExecutionsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListQueryExecutionsPublisher publisher = client.listQueryExecutionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListQueryExecutionsPublisher publisher = client.listQueryExecutionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.athena.model.ListQueryExecutionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.athena.model.ListQueryExecutionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listQueryExecutions(software.amazon.awssdk.services.athena.model.ListQueryExecutionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listQueryExecutionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListQueryExecutions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListQueryExecutions" target="_top">AWS API
     *      Documentation</a>
     */
    default ListQueryExecutionsPublisher listQueryExecutionsPaginator(ListQueryExecutionsRequest listQueryExecutionsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Provides a list of available query execution IDs for the queries in the specified workgroup. Requires you to have
     * access to the workgroup in which the queries ran.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listQueryExecutions(software.amazon.awssdk.services.athena.model.ListQueryExecutionsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListQueryExecutionsPublisher publisher = client.listQueryExecutionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListQueryExecutionsPublisher publisher = client.listQueryExecutionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.athena.model.ListQueryExecutionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.athena.model.ListQueryExecutionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listQueryExecutions(software.amazon.awssdk.services.athena.model.ListQueryExecutionsRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListQueryExecutionsRequest.Builder} avoiding the
     * need to create one manually via {@link ListQueryExecutionsRequest#builder()}
     * </p>
     *
     * @param listQueryExecutionsRequest
     *        A {@link Consumer} that will call methods on {@link ListQueryExecutionsInput.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListQueryExecutions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListQueryExecutions" target="_top">AWS API
     *      Documentation</a>
     */
    default ListQueryExecutionsPublisher listQueryExecutionsPaginator(
            Consumer<ListQueryExecutionsRequest.Builder> listQueryExecutionsRequest) {
        return listQueryExecutionsPaginator(ListQueryExecutionsRequest.builder().applyMutation(listQueryExecutionsRequest)
                .build());
    }

    /**
     * <p>
     * Lists the tags associated with this workgroup.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>ResourceNotFoundException A resource, such as a workgroup, was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the tags associated with this workgroup.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on {@link ListTagsForResourceInput.Builder} to create a request.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>ResourceNotFoundException A resource, such as a workgroup, was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Lists available workgroups for the account.
     * </p>
     *
     * @param listWorkGroupsRequest
     * @return A Java Future containing the result of the ListWorkGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListWorkGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListWorkGroups" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListWorkGroupsResponse> listWorkGroups(ListWorkGroupsRequest listWorkGroupsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists available workgroups for the account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListWorkGroupsRequest.Builder} avoiding the need to
     * create one manually via {@link ListWorkGroupsRequest#builder()}
     * </p>
     *
     * @param listWorkGroupsRequest
     *        A {@link Consumer} that will call methods on {@link ListWorkGroupsInput.Builder} to create a request.
     * @return A Java Future containing the result of the ListWorkGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListWorkGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListWorkGroups" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListWorkGroupsResponse> listWorkGroups(Consumer<ListWorkGroupsRequest.Builder> listWorkGroupsRequest) {
        return listWorkGroups(ListWorkGroupsRequest.builder().applyMutation(listWorkGroupsRequest).build());
    }

    /**
     * <p>
     * Lists available workgroups for the account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listWorkGroups(software.amazon.awssdk.services.athena.model.ListWorkGroupsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListWorkGroupsPublisher publisher = client.listWorkGroupsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListWorkGroupsPublisher publisher = client.listWorkGroupsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.athena.model.ListWorkGroupsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.athena.model.ListWorkGroupsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listWorkGroups(software.amazon.awssdk.services.athena.model.ListWorkGroupsRequest)} operation.</b>
     * </p>
     *
     * @param listWorkGroupsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListWorkGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListWorkGroups" target="_top">AWS API
     *      Documentation</a>
     */
    default ListWorkGroupsPublisher listWorkGroupsPaginator(ListWorkGroupsRequest listWorkGroupsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists available workgroups for the account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listWorkGroups(software.amazon.awssdk.services.athena.model.ListWorkGroupsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListWorkGroupsPublisher publisher = client.listWorkGroupsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListWorkGroupsPublisher publisher = client.listWorkGroupsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.athena.model.ListWorkGroupsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.athena.model.ListWorkGroupsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listWorkGroups(software.amazon.awssdk.services.athena.model.ListWorkGroupsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListWorkGroupsRequest.Builder} avoiding the need to
     * create one manually via {@link ListWorkGroupsRequest#builder()}
     * </p>
     *
     * @param listWorkGroupsRequest
     *        A {@link Consumer} that will call methods on {@link ListWorkGroupsInput.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListWorkGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListWorkGroups" target="_top">AWS API
     *      Documentation</a>
     */
    default ListWorkGroupsPublisher listWorkGroupsPaginator(Consumer<ListWorkGroupsRequest.Builder> listWorkGroupsRequest) {
        return listWorkGroupsPaginator(ListWorkGroupsRequest.builder().applyMutation(listWorkGroupsRequest).build());
    }

    /**
     * <p>
     * Runs the SQL query statements contained in the <code>Query</code>. Requires you to have access to the workgroup
     * in which the query ran.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     *
     * @param startQueryExecutionRequest
     * @return A Java Future containing the result of the StartQueryExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>TooManyRequestsException Indicates that the request was throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.StartQueryExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/StartQueryExecution" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<StartQueryExecutionResponse> startQueryExecution(
            StartQueryExecutionRequest startQueryExecutionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Runs the SQL query statements contained in the <code>Query</code>. Requires you to have access to the workgroup
     * in which the query ran.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartQueryExecutionRequest.Builder} avoiding the
     * need to create one manually via {@link StartQueryExecutionRequest#builder()}
     * </p>
     *
     * @param startQueryExecutionRequest
     *        A {@link Consumer} that will call methods on {@link StartQueryExecutionInput.Builder} to create a request.
     * @return A Java Future containing the result of the StartQueryExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>TooManyRequestsException Indicates that the request was throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.StartQueryExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/StartQueryExecution" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<StartQueryExecutionResponse> startQueryExecution(
            Consumer<StartQueryExecutionRequest.Builder> startQueryExecutionRequest) {
        return startQueryExecution(StartQueryExecutionRequest.builder().applyMutation(startQueryExecutionRequest).build());
    }

    /**
     * <p>
     * Stops a query execution. Requires you to have access to the workgroup in which the query ran.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     *
     * @param stopQueryExecutionRequest
     * @return A Java Future containing the result of the StopQueryExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.StopQueryExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/StopQueryExecution" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<StopQueryExecutionResponse> stopQueryExecution(StopQueryExecutionRequest stopQueryExecutionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Stops a query execution. Requires you to have access to the workgroup in which the query ran.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StopQueryExecutionRequest.Builder} avoiding the
     * need to create one manually via {@link StopQueryExecutionRequest#builder()}
     * </p>
     *
     * @param stopQueryExecutionRequest
     *        A {@link Consumer} that will call methods on {@link StopQueryExecutionInput.Builder} to create a request.
     * @return A Java Future containing the result of the StopQueryExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.StopQueryExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/StopQueryExecution" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<StopQueryExecutionResponse> stopQueryExecution(
            Consumer<StopQueryExecutionRequest.Builder> stopQueryExecutionRequest) {
        return stopQueryExecution(StopQueryExecutionRequest.builder().applyMutation(stopQueryExecutionRequest).build());
    }

    /**
     * <p>
     * Adds one or more tags to the resource, such as a workgroup. A tag is a label that you assign to an AWS Athena
     * resource (a workgroup). Each tag consists of a key and an optional value, both of which you define. Tags enable
     * you to categorize resources (workgroups) in Athena, for example, by purpose, owner, or environment. Use a
     * consistent set of tag keys to make it easier to search and filter workgroups in your account. For best practices,
     * see <a href="https://aws.amazon.com/answers/account-management/aws-tagging-strategies/">AWS Tagging
     * Strategies</a>. The key length is from 1 (minimum) to 128 (maximum) Unicode characters in UTF-8. The tag value
     * length is from 0 (minimum) to 256 (maximum) Unicode characters in UTF-8. You can use letters and numbers
     * representable in UTF-8, and the following characters: + - = . _ : / @. Tag keys and values are case-sensitive.
     * Tag keys must be unique per resource. If you specify more than one, separate them by commas.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>ResourceNotFoundException A resource, such as a workgroup, was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds one or more tags to the resource, such as a workgroup. A tag is a label that you assign to an AWS Athena
     * resource (a workgroup). Each tag consists of a key and an optional value, both of which you define. Tags enable
     * you to categorize resources (workgroups) in Athena, for example, by purpose, owner, or environment. Use a
     * consistent set of tag keys to make it easier to search and filter workgroups in your account. For best practices,
     * see <a href="https://aws.amazon.com/answers/account-management/aws-tagging-strategies/">AWS Tagging
     * Strategies</a>. The key length is from 1 (minimum) to 128 (maximum) Unicode characters in UTF-8. The tag value
     * length is from 0 (minimum) to 256 (maximum) Unicode characters in UTF-8. You can use letters and numbers
     * representable in UTF-8, and the following characters: + - = . _ : / @. Tag keys and values are case-sensitive.
     * Tag keys must be unique per resource. If you specify more than one, separate them by commas.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on {@link TagResourceInput.Builder} to create a request.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>ResourceNotFoundException A resource, such as a workgroup, was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes one or more tags from the workgroup resource. Takes as an input a list of TagKey Strings separated by
     * commas, and removes their tags at the same time.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>ResourceNotFoundException A resource, such as a workgroup, was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes one or more tags from the workgroup resource. Takes as an input a list of TagKey Strings separated by
     * commas, and removes their tags at the same time.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on {@link UntagResourceInput.Builder} to create a request.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>ResourceNotFoundException A resource, such as a workgroup, was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest) {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates the workgroup with the specified name. The workgroup's name cannot be changed.
     * </p>
     *
     * @param updateWorkGroupRequest
     * @return A Java Future containing the result of the UpdateWorkGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.UpdateWorkGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/UpdateWorkGroup" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateWorkGroupResponse> updateWorkGroup(UpdateWorkGroupRequest updateWorkGroupRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates the workgroup with the specified name. The workgroup's name cannot be changed.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateWorkGroupRequest.Builder} avoiding the need
     * to create one manually via {@link UpdateWorkGroupRequest#builder()}
     * </p>
     *
     * @param updateWorkGroupRequest
     *        A {@link Consumer} that will call methods on {@link UpdateWorkGroupInput.Builder} to create a request.
     * @return A Java Future containing the result of the UpdateWorkGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.UpdateWorkGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/UpdateWorkGroup" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateWorkGroupResponse> updateWorkGroup(
            Consumer<UpdateWorkGroupRequest.Builder> updateWorkGroupRequest) {
        return updateWorkGroup(UpdateWorkGroupRequest.builder().applyMutation(updateWorkGroupRequest).build());
    }
}
