/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.athena.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateWorkGroupRequest extends AthenaRequest implements
        ToCopyableBuilder<CreateWorkGroupRequest.Builder, CreateWorkGroupRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateWorkGroupRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<WorkGroupConfiguration> CONFIGURATION_FIELD = SdkField
            .<WorkGroupConfiguration> builder(MarshallingType.SDK_POJO).getter(getter(CreateWorkGroupRequest::configuration))
            .setter(setter(Builder::configuration)).constructor(WorkGroupConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Configuration").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateWorkGroupRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(CreateWorkGroupRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            CONFIGURATION_FIELD, DESCRIPTION_FIELD, TAGS_FIELD));

    private final String name;

    private final WorkGroupConfiguration configuration;

    private final String description;

    private final List<Tag> tags;

    private CreateWorkGroupRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.configuration = builder.configuration;
        this.description = builder.description;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The workgroup name.
     * </p>
     * 
     * @return The workgroup name.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The configuration for the workgroup, which includes the location in Amazon S3 where query results are stored, the
     * encryption configuration, if any, used for encrypting query results, whether the Amazon CloudWatch Metrics are
     * enabled for the workgroup, the limit for the amount of bytes scanned (cutoff) per query, if it is specified, and
     * whether workgroup's settings (specified with EnforceWorkGroupConfiguration) in the WorkGroupConfiguration
     * override client-side settings. See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
     * </p>
     * 
     * @return The configuration for the workgroup, which includes the location in Amazon S3 where query results are
     *         stored, the encryption configuration, if any, used for encrypting query results, whether the Amazon
     *         CloudWatch Metrics are enabled for the workgroup, the limit for the amount of bytes scanned (cutoff) per
     *         query, if it is specified, and whether workgroup's settings (specified with
     *         EnforceWorkGroupConfiguration) in the WorkGroupConfiguration override client-side settings. See
     *         <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
     */
    public WorkGroupConfiguration configuration() {
        return configuration;
    }

    /**
     * <p>
     * The workgroup description.
     * </p>
     * 
     * @return The workgroup description.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * One or more tags, separated by commas, that you want to attach to the workgroup as you create it.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return One or more tags, separated by commas, that you want to attach to the workgroup as you create it.
     */
    public List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(configuration());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateWorkGroupRequest)) {
            return false;
        }
        CreateWorkGroupRequest other = (CreateWorkGroupRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(configuration(), other.configuration())
                && Objects.equals(description(), other.description()) && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateWorkGroupRequest").add("Name", name()).add("Configuration", configuration())
                .add("Description", description()).add("Tags", tags()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateWorkGroupRequest, T> g) {
        return obj -> g.apply((CreateWorkGroupRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends AthenaRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateWorkGroupRequest> {
        /**
         * <p>
         * The workgroup name.
         * </p>
         * 
         * @param name
         *        The workgroup name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The configuration for the workgroup, which includes the location in Amazon S3 where query results are stored,
         * the encryption configuration, if any, used for encrypting query results, whether the Amazon CloudWatch
         * Metrics are enabled for the workgroup, the limit for the amount of bytes scanned (cutoff) per query, if it is
         * specified, and whether workgroup's settings (specified with EnforceWorkGroupConfiguration) in the
         * WorkGroupConfiguration override client-side settings. See
         * <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
         * </p>
         * 
         * @param configuration
         *        The configuration for the workgroup, which includes the location in Amazon S3 where query results are
         *        stored, the encryption configuration, if any, used for encrypting query results, whether the Amazon
         *        CloudWatch Metrics are enabled for the workgroup, the limit for the amount of bytes scanned (cutoff)
         *        per query, if it is specified, and whether workgroup's settings (specified with
         *        EnforceWorkGroupConfiguration) in the WorkGroupConfiguration override client-side settings. See
         *        <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(WorkGroupConfiguration configuration);

        /**
         * <p>
         * The configuration for the workgroup, which includes the location in Amazon S3 where query results are stored,
         * the encryption configuration, if any, used for encrypting query results, whether the Amazon CloudWatch
         * Metrics are enabled for the workgroup, the limit for the amount of bytes scanned (cutoff) per query, if it is
         * specified, and whether workgroup's settings (specified with EnforceWorkGroupConfiguration) in the
         * WorkGroupConfiguration override client-side settings. See
         * <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link WorkGroupConfiguration.Builder} avoiding the
         * need to create one manually via {@link WorkGroupConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link WorkGroupConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #configuration(WorkGroupConfiguration)}.
         * 
         * @param configuration
         *        a consumer that will call methods on {@link WorkGroupConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configuration(WorkGroupConfiguration)
         */
        default Builder configuration(Consumer<WorkGroupConfiguration.Builder> configuration) {
            return configuration(WorkGroupConfiguration.builder().applyMutation(configuration).build());
        }

        /**
         * <p>
         * The workgroup description.
         * </p>
         * 
         * @param description
         *        The workgroup description.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * One or more tags, separated by commas, that you want to attach to the workgroup as you create it.
         * </p>
         * 
         * @param tags
         *        One or more tags, separated by commas, that you want to attach to the workgroup as you create it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * One or more tags, separated by commas, that you want to attach to the workgroup as you create it.
         * </p>
         * 
         * @param tags
         *        One or more tags, separated by commas, that you want to attach to the workgroup as you create it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * One or more tags, separated by commas, that you want to attach to the workgroup as you create it.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends AthenaRequest.BuilderImpl implements Builder {
        private String name;

        private WorkGroupConfiguration configuration;

        private String description;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateWorkGroupRequest model) {
            super(model);
            name(model.name);
            configuration(model.configuration);
            description(model.description);
            tags(model.tags);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final WorkGroupConfiguration.Builder getConfiguration() {
            return configuration != null ? configuration.toBuilder() : null;
        }

        @Override
        public final Builder configuration(WorkGroupConfiguration configuration) {
            this.configuration = configuration;
            return this;
        }

        public final void setConfiguration(WorkGroupConfiguration.BuilderImpl configuration) {
            this.configuration = configuration != null ? configuration.build() : null;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateWorkGroupRequest build() {
            return new CreateWorkGroupRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
