/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.athena.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a single instance of a query execution.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class QueryExecution implements SdkPojo, Serializable, ToCopyableBuilder<QueryExecution.Builder, QueryExecution> {
    private static final SdkField<String> QUERY_EXECUTION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("QueryExecutionId").getter(getter(QueryExecution::queryExecutionId))
            .setter(setter(Builder::queryExecutionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueryExecutionId").build()).build();

    private static final SdkField<String> QUERY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Query")
            .getter(getter(QueryExecution::query)).setter(setter(Builder::query))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Query").build()).build();

    private static final SdkField<String> STATEMENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatementType").getter(getter(QueryExecution::statementTypeAsString))
            .setter(setter(Builder::statementType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatementType").build()).build();

    private static final SdkField<ResultConfiguration> RESULT_CONFIGURATION_FIELD = SdkField
            .<ResultConfiguration> builder(MarshallingType.SDK_POJO).memberName("ResultConfiguration")
            .getter(getter(QueryExecution::resultConfiguration)).setter(setter(Builder::resultConfiguration))
            .constructor(ResultConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResultConfiguration").build())
            .build();

    private static final SdkField<QueryExecutionContext> QUERY_EXECUTION_CONTEXT_FIELD = SdkField
            .<QueryExecutionContext> builder(MarshallingType.SDK_POJO).memberName("QueryExecutionContext")
            .getter(getter(QueryExecution::queryExecutionContext)).setter(setter(Builder::queryExecutionContext))
            .constructor(QueryExecutionContext::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueryExecutionContext").build())
            .build();

    private static final SdkField<QueryExecutionStatus> STATUS_FIELD = SdkField
            .<QueryExecutionStatus> builder(MarshallingType.SDK_POJO).memberName("Status").getter(getter(QueryExecution::status))
            .setter(setter(Builder::status)).constructor(QueryExecutionStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<QueryExecutionStatistics> STATISTICS_FIELD = SdkField
            .<QueryExecutionStatistics> builder(MarshallingType.SDK_POJO).memberName("Statistics")
            .getter(getter(QueryExecution::statistics)).setter(setter(Builder::statistics))
            .constructor(QueryExecutionStatistics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Statistics").build()).build();

    private static final SdkField<String> WORK_GROUP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WorkGroup").getter(getter(QueryExecution::workGroup)).setter(setter(Builder::workGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkGroup").build()).build();

    private static final SdkField<EngineVersion> ENGINE_VERSION_FIELD = SdkField
            .<EngineVersion> builder(MarshallingType.SDK_POJO).memberName("EngineVersion")
            .getter(getter(QueryExecution::engineVersion)).setter(setter(Builder::engineVersion))
            .constructor(EngineVersion::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineVersion").build()).build();

    private static final SdkField<List<String>> EXECUTION_PARAMETERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExecutionParameters")
            .getter(getter(QueryExecution::executionParameters))
            .setter(setter(Builder::executionParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionParameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(QUERY_EXECUTION_ID_FIELD,
            QUERY_FIELD, STATEMENT_TYPE_FIELD, RESULT_CONFIGURATION_FIELD, QUERY_EXECUTION_CONTEXT_FIELD, STATUS_FIELD,
            STATISTICS_FIELD, WORK_GROUP_FIELD, ENGINE_VERSION_FIELD, EXECUTION_PARAMETERS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String queryExecutionId;

    private final String query;

    private final String statementType;

    private final ResultConfiguration resultConfiguration;

    private final QueryExecutionContext queryExecutionContext;

    private final QueryExecutionStatus status;

    private final QueryExecutionStatistics statistics;

    private final String workGroup;

    private final EngineVersion engineVersion;

    private final List<String> executionParameters;

    private QueryExecution(BuilderImpl builder) {
        this.queryExecutionId = builder.queryExecutionId;
        this.query = builder.query;
        this.statementType = builder.statementType;
        this.resultConfiguration = builder.resultConfiguration;
        this.queryExecutionContext = builder.queryExecutionContext;
        this.status = builder.status;
        this.statistics = builder.statistics;
        this.workGroup = builder.workGroup;
        this.engineVersion = builder.engineVersion;
        this.executionParameters = builder.executionParameters;
    }

    /**
     * <p>
     * The unique identifier for each query execution.
     * </p>
     * 
     * @return The unique identifier for each query execution.
     */
    public final String queryExecutionId() {
        return queryExecutionId;
    }

    /**
     * <p>
     * The SQL query statements which the query execution ran.
     * </p>
     * 
     * @return The SQL query statements which the query execution ran.
     */
    public final String query() {
        return query;
    }

    /**
     * <p>
     * The type of query statement that was run. <code>DDL</code> indicates DDL query statements. <code>DML</code>
     * indicates DML (Data Manipulation Language) query statements, such as <code>CREATE TABLE AS SELECT</code>.
     * <code>UTILITY</code> indicates query statements other than DDL and DML, such as <code>SHOW CREATE TABLE</code>,
     * or <code>DESCRIBE TABLE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statementType}
     * will return {@link StatementType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statementTypeAsString}.
     * </p>
     * 
     * @return The type of query statement that was run. <code>DDL</code> indicates DDL query statements.
     *         <code>DML</code> indicates DML (Data Manipulation Language) query statements, such as
     *         <code>CREATE TABLE AS SELECT</code>. <code>UTILITY</code> indicates query statements other than DDL and
     *         DML, such as <code>SHOW CREATE TABLE</code>, or <code>DESCRIBE TABLE</code>.
     * @see StatementType
     */
    public final StatementType statementType() {
        return StatementType.fromValue(statementType);
    }

    /**
     * <p>
     * The type of query statement that was run. <code>DDL</code> indicates DDL query statements. <code>DML</code>
     * indicates DML (Data Manipulation Language) query statements, such as <code>CREATE TABLE AS SELECT</code>.
     * <code>UTILITY</code> indicates query statements other than DDL and DML, such as <code>SHOW CREATE TABLE</code>,
     * or <code>DESCRIBE TABLE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statementType}
     * will return {@link StatementType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statementTypeAsString}.
     * </p>
     * 
     * @return The type of query statement that was run. <code>DDL</code> indicates DDL query statements.
     *         <code>DML</code> indicates DML (Data Manipulation Language) query statements, such as
     *         <code>CREATE TABLE AS SELECT</code>. <code>UTILITY</code> indicates query statements other than DDL and
     *         DML, such as <code>SHOW CREATE TABLE</code>, or <code>DESCRIBE TABLE</code>.
     * @see StatementType
     */
    public final String statementTypeAsString() {
        return statementType;
    }

    /**
     * <p>
     * The location in Amazon S3 where query results were stored and the encryption option, if any, used for query
     * results. These are known as "client-side settings". If workgroup settings override client-side settings, then the
     * query uses the location for the query results and the encryption configuration that are specified for the
     * workgroup.
     * </p>
     * 
     * @return The location in Amazon S3 where query results were stored and the encryption option, if any, used for
     *         query results. These are known as "client-side settings". If workgroup settings override client-side
     *         settings, then the query uses the location for the query results and the encryption configuration that
     *         are specified for the workgroup.
     */
    public final ResultConfiguration resultConfiguration() {
        return resultConfiguration;
    }

    /**
     * <p>
     * The database in which the query execution occurred.
     * </p>
     * 
     * @return The database in which the query execution occurred.
     */
    public final QueryExecutionContext queryExecutionContext() {
        return queryExecutionContext;
    }

    /**
     * <p>
     * The completion date, current state, submission time, and state change reason (if applicable) for the query
     * execution.
     * </p>
     * 
     * @return The completion date, current state, submission time, and state change reason (if applicable) for the
     *         query execution.
     */
    public final QueryExecutionStatus status() {
        return status;
    }

    /**
     * <p>
     * Query execution statistics, such as the amount of data scanned, the amount of time that the query took to
     * process, and the type of statement that was run.
     * </p>
     * 
     * @return Query execution statistics, such as the amount of data scanned, the amount of time that the query took to
     *         process, and the type of statement that was run.
     */
    public final QueryExecutionStatistics statistics() {
        return statistics;
    }

    /**
     * <p>
     * The name of the workgroup in which the query ran.
     * </p>
     * 
     * @return The name of the workgroup in which the query ran.
     */
    public final String workGroup() {
        return workGroup;
    }

    /**
     * <p>
     * The engine version that executed the query.
     * </p>
     * 
     * @return The engine version that executed the query.
     */
    public final EngineVersion engineVersion() {
        return engineVersion;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExecutionParameters property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExecutionParameters() {
        return executionParameters != null && !(executionParameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of values for the parameters in a query. The values are applied sequentially to the parameters in the
     * query in the order in which the parameters occur.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExecutionParameters} method.
     * </p>
     * 
     * @return A list of values for the parameters in a query. The values are applied sequentially to the parameters in
     *         the query in the order in which the parameters occur.
     */
    public final List<String> executionParameters() {
        return executionParameters;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(queryExecutionId());
        hashCode = 31 * hashCode + Objects.hashCode(query());
        hashCode = 31 * hashCode + Objects.hashCode(statementTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resultConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(queryExecutionContext());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(statistics());
        hashCode = 31 * hashCode + Objects.hashCode(workGroup());
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hasExecutionParameters() ? executionParameters() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof QueryExecution)) {
            return false;
        }
        QueryExecution other = (QueryExecution) obj;
        return Objects.equals(queryExecutionId(), other.queryExecutionId()) && Objects.equals(query(), other.query())
                && Objects.equals(statementTypeAsString(), other.statementTypeAsString())
                && Objects.equals(resultConfiguration(), other.resultConfiguration())
                && Objects.equals(queryExecutionContext(), other.queryExecutionContext())
                && Objects.equals(status(), other.status()) && Objects.equals(statistics(), other.statistics())
                && Objects.equals(workGroup(), other.workGroup()) && Objects.equals(engineVersion(), other.engineVersion())
                && hasExecutionParameters() == other.hasExecutionParameters()
                && Objects.equals(executionParameters(), other.executionParameters());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("QueryExecution").add("QueryExecutionId", queryExecutionId()).add("Query", query())
                .add("StatementType", statementTypeAsString()).add("ResultConfiguration", resultConfiguration())
                .add("QueryExecutionContext", queryExecutionContext()).add("Status", status()).add("Statistics", statistics())
                .add("WorkGroup", workGroup()).add("EngineVersion", engineVersion())
                .add("ExecutionParameters", hasExecutionParameters() ? executionParameters() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "QueryExecutionId":
            return Optional.ofNullable(clazz.cast(queryExecutionId()));
        case "Query":
            return Optional.ofNullable(clazz.cast(query()));
        case "StatementType":
            return Optional.ofNullable(clazz.cast(statementTypeAsString()));
        case "ResultConfiguration":
            return Optional.ofNullable(clazz.cast(resultConfiguration()));
        case "QueryExecutionContext":
            return Optional.ofNullable(clazz.cast(queryExecutionContext()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "Statistics":
            return Optional.ofNullable(clazz.cast(statistics()));
        case "WorkGroup":
            return Optional.ofNullable(clazz.cast(workGroup()));
        case "EngineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        case "ExecutionParameters":
            return Optional.ofNullable(clazz.cast(executionParameters()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<QueryExecution, T> g) {
        return obj -> g.apply((QueryExecution) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, QueryExecution> {
        /**
         * <p>
         * The unique identifier for each query execution.
         * </p>
         * 
         * @param queryExecutionId
         *        The unique identifier for each query execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryExecutionId(String queryExecutionId);

        /**
         * <p>
         * The SQL query statements which the query execution ran.
         * </p>
         * 
         * @param query
         *        The SQL query statements which the query execution ran.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder query(String query);

        /**
         * <p>
         * The type of query statement that was run. <code>DDL</code> indicates DDL query statements. <code>DML</code>
         * indicates DML (Data Manipulation Language) query statements, such as <code>CREATE TABLE AS SELECT</code>.
         * <code>UTILITY</code> indicates query statements other than DDL and DML, such as
         * <code>SHOW CREATE TABLE</code>, or <code>DESCRIBE TABLE</code>.
         * </p>
         * 
         * @param statementType
         *        The type of query statement that was run. <code>DDL</code> indicates DDL query statements.
         *        <code>DML</code> indicates DML (Data Manipulation Language) query statements, such as
         *        <code>CREATE TABLE AS SELECT</code>. <code>UTILITY</code> indicates query statements other than DDL
         *        and DML, such as <code>SHOW CREATE TABLE</code>, or <code>DESCRIBE TABLE</code>.
         * @see StatementType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatementType
         */
        Builder statementType(String statementType);

        /**
         * <p>
         * The type of query statement that was run. <code>DDL</code> indicates DDL query statements. <code>DML</code>
         * indicates DML (Data Manipulation Language) query statements, such as <code>CREATE TABLE AS SELECT</code>.
         * <code>UTILITY</code> indicates query statements other than DDL and DML, such as
         * <code>SHOW CREATE TABLE</code>, or <code>DESCRIBE TABLE</code>.
         * </p>
         * 
         * @param statementType
         *        The type of query statement that was run. <code>DDL</code> indicates DDL query statements.
         *        <code>DML</code> indicates DML (Data Manipulation Language) query statements, such as
         *        <code>CREATE TABLE AS SELECT</code>. <code>UTILITY</code> indicates query statements other than DDL
         *        and DML, such as <code>SHOW CREATE TABLE</code>, or <code>DESCRIBE TABLE</code>.
         * @see StatementType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatementType
         */
        Builder statementType(StatementType statementType);

        /**
         * <p>
         * The location in Amazon S3 where query results were stored and the encryption option, if any, used for query
         * results. These are known as "client-side settings". If workgroup settings override client-side settings, then
         * the query uses the location for the query results and the encryption configuration that are specified for the
         * workgroup.
         * </p>
         * 
         * @param resultConfiguration
         *        The location in Amazon S3 where query results were stored and the encryption option, if any, used for
         *        query results. These are known as "client-side settings". If workgroup settings override client-side
         *        settings, then the query uses the location for the query results and the encryption configuration that
         *        are specified for the workgroup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resultConfiguration(ResultConfiguration resultConfiguration);

        /**
         * <p>
         * The location in Amazon S3 where query results were stored and the encryption option, if any, used for query
         * results. These are known as "client-side settings". If workgroup settings override client-side settings, then
         * the query uses the location for the query results and the encryption configuration that are specified for the
         * workgroup.
         * </p>
         * This is a convenience method that creates an instance of the {@link ResultConfiguration.Builder} avoiding the
         * need to create one manually via {@link ResultConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResultConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #resultConfiguration(ResultConfiguration)}.
         * 
         * @param resultConfiguration
         *        a consumer that will call methods on {@link ResultConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resultConfiguration(ResultConfiguration)
         */
        default Builder resultConfiguration(Consumer<ResultConfiguration.Builder> resultConfiguration) {
            return resultConfiguration(ResultConfiguration.builder().applyMutation(resultConfiguration).build());
        }

        /**
         * <p>
         * The database in which the query execution occurred.
         * </p>
         * 
         * @param queryExecutionContext
         *        The database in which the query execution occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryExecutionContext(QueryExecutionContext queryExecutionContext);

        /**
         * <p>
         * The database in which the query execution occurred.
         * </p>
         * This is a convenience method that creates an instance of the {@link QueryExecutionContext.Builder} avoiding
         * the need to create one manually via {@link QueryExecutionContext#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link QueryExecutionContext.Builder#build()} is called immediately and
         * its result is passed to {@link #queryExecutionContext(QueryExecutionContext)}.
         * 
         * @param queryExecutionContext
         *        a consumer that will call methods on {@link QueryExecutionContext.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #queryExecutionContext(QueryExecutionContext)
         */
        default Builder queryExecutionContext(Consumer<QueryExecutionContext.Builder> queryExecutionContext) {
            return queryExecutionContext(QueryExecutionContext.builder().applyMutation(queryExecutionContext).build());
        }

        /**
         * <p>
         * The completion date, current state, submission time, and state change reason (if applicable) for the query
         * execution.
         * </p>
         * 
         * @param status
         *        The completion date, current state, submission time, and state change reason (if applicable) for the
         *        query execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(QueryExecutionStatus status);

        /**
         * <p>
         * The completion date, current state, submission time, and state change reason (if applicable) for the query
         * execution.
         * </p>
         * This is a convenience method that creates an instance of the {@link QueryExecutionStatus.Builder} avoiding
         * the need to create one manually via {@link QueryExecutionStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link QueryExecutionStatus.Builder#build()} is called immediately and
         * its result is passed to {@link #status(QueryExecutionStatus)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link QueryExecutionStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(QueryExecutionStatus)
         */
        default Builder status(Consumer<QueryExecutionStatus.Builder> status) {
            return status(QueryExecutionStatus.builder().applyMutation(status).build());
        }

        /**
         * <p>
         * Query execution statistics, such as the amount of data scanned, the amount of time that the query took to
         * process, and the type of statement that was run.
         * </p>
         * 
         * @param statistics
         *        Query execution statistics, such as the amount of data scanned, the amount of time that the query took
         *        to process, and the type of statement that was run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statistics(QueryExecutionStatistics statistics);

        /**
         * <p>
         * Query execution statistics, such as the amount of data scanned, the amount of time that the query took to
         * process, and the type of statement that was run.
         * </p>
         * This is a convenience method that creates an instance of the {@link QueryExecutionStatistics.Builder}
         * avoiding the need to create one manually via {@link QueryExecutionStatistics#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link QueryExecutionStatistics.Builder#build()} is called immediately
         * and its result is passed to {@link #statistics(QueryExecutionStatistics)}.
         * 
         * @param statistics
         *        a consumer that will call methods on {@link QueryExecutionStatistics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #statistics(QueryExecutionStatistics)
         */
        default Builder statistics(Consumer<QueryExecutionStatistics.Builder> statistics) {
            return statistics(QueryExecutionStatistics.builder().applyMutation(statistics).build());
        }

        /**
         * <p>
         * The name of the workgroup in which the query ran.
         * </p>
         * 
         * @param workGroup
         *        The name of the workgroup in which the query ran.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workGroup(String workGroup);

        /**
         * <p>
         * The engine version that executed the query.
         * </p>
         * 
         * @param engineVersion
         *        The engine version that executed the query.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(EngineVersion engineVersion);

        /**
         * <p>
         * The engine version that executed the query.
         * </p>
         * This is a convenience method that creates an instance of the {@link EngineVersion.Builder} avoiding the need
         * to create one manually via {@link EngineVersion#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EngineVersion.Builder#build()} is called immediately and its
         * result is passed to {@link #engineVersion(EngineVersion)}.
         * 
         * @param engineVersion
         *        a consumer that will call methods on {@link EngineVersion.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #engineVersion(EngineVersion)
         */
        default Builder engineVersion(Consumer<EngineVersion.Builder> engineVersion) {
            return engineVersion(EngineVersion.builder().applyMutation(engineVersion).build());
        }

        /**
         * <p>
         * A list of values for the parameters in a query. The values are applied sequentially to the parameters in the
         * query in the order in which the parameters occur.
         * </p>
         * 
         * @param executionParameters
         *        A list of values for the parameters in a query. The values are applied sequentially to the parameters
         *        in the query in the order in which the parameters occur.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionParameters(Collection<String> executionParameters);

        /**
         * <p>
         * A list of values for the parameters in a query. The values are applied sequentially to the parameters in the
         * query in the order in which the parameters occur.
         * </p>
         * 
         * @param executionParameters
         *        A list of values for the parameters in a query. The values are applied sequentially to the parameters
         *        in the query in the order in which the parameters occur.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionParameters(String... executionParameters);
    }

    static final class BuilderImpl implements Builder {
        private String queryExecutionId;

        private String query;

        private String statementType;

        private ResultConfiguration resultConfiguration;

        private QueryExecutionContext queryExecutionContext;

        private QueryExecutionStatus status;

        private QueryExecutionStatistics statistics;

        private String workGroup;

        private EngineVersion engineVersion;

        private List<String> executionParameters = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(QueryExecution model) {
            queryExecutionId(model.queryExecutionId);
            query(model.query);
            statementType(model.statementType);
            resultConfiguration(model.resultConfiguration);
            queryExecutionContext(model.queryExecutionContext);
            status(model.status);
            statistics(model.statistics);
            workGroup(model.workGroup);
            engineVersion(model.engineVersion);
            executionParameters(model.executionParameters);
        }

        public final String getQueryExecutionId() {
            return queryExecutionId;
        }

        public final void setQueryExecutionId(String queryExecutionId) {
            this.queryExecutionId = queryExecutionId;
        }

        @Override
        public final Builder queryExecutionId(String queryExecutionId) {
            this.queryExecutionId = queryExecutionId;
            return this;
        }

        public final String getQuery() {
            return query;
        }

        public final void setQuery(String query) {
            this.query = query;
        }

        @Override
        public final Builder query(String query) {
            this.query = query;
            return this;
        }

        public final String getStatementType() {
            return statementType;
        }

        public final void setStatementType(String statementType) {
            this.statementType = statementType;
        }

        @Override
        public final Builder statementType(String statementType) {
            this.statementType = statementType;
            return this;
        }

        @Override
        public final Builder statementType(StatementType statementType) {
            this.statementType(statementType == null ? null : statementType.toString());
            return this;
        }

        public final ResultConfiguration.Builder getResultConfiguration() {
            return resultConfiguration != null ? resultConfiguration.toBuilder() : null;
        }

        public final void setResultConfiguration(ResultConfiguration.BuilderImpl resultConfiguration) {
            this.resultConfiguration = resultConfiguration != null ? resultConfiguration.build() : null;
        }

        @Override
        public final Builder resultConfiguration(ResultConfiguration resultConfiguration) {
            this.resultConfiguration = resultConfiguration;
            return this;
        }

        public final QueryExecutionContext.Builder getQueryExecutionContext() {
            return queryExecutionContext != null ? queryExecutionContext.toBuilder() : null;
        }

        public final void setQueryExecutionContext(QueryExecutionContext.BuilderImpl queryExecutionContext) {
            this.queryExecutionContext = queryExecutionContext != null ? queryExecutionContext.build() : null;
        }

        @Override
        public final Builder queryExecutionContext(QueryExecutionContext queryExecutionContext) {
            this.queryExecutionContext = queryExecutionContext;
            return this;
        }

        public final QueryExecutionStatus.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        public final void setStatus(QueryExecutionStatus.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        @Override
        public final Builder status(QueryExecutionStatus status) {
            this.status = status;
            return this;
        }

        public final QueryExecutionStatistics.Builder getStatistics() {
            return statistics != null ? statistics.toBuilder() : null;
        }

        public final void setStatistics(QueryExecutionStatistics.BuilderImpl statistics) {
            this.statistics = statistics != null ? statistics.build() : null;
        }

        @Override
        public final Builder statistics(QueryExecutionStatistics statistics) {
            this.statistics = statistics;
            return this;
        }

        public final String getWorkGroup() {
            return workGroup;
        }

        public final void setWorkGroup(String workGroup) {
            this.workGroup = workGroup;
        }

        @Override
        public final Builder workGroup(String workGroup) {
            this.workGroup = workGroup;
            return this;
        }

        public final EngineVersion.Builder getEngineVersion() {
            return engineVersion != null ? engineVersion.toBuilder() : null;
        }

        public final void setEngineVersion(EngineVersion.BuilderImpl engineVersion) {
            this.engineVersion = engineVersion != null ? engineVersion.build() : null;
        }

        @Override
        public final Builder engineVersion(EngineVersion engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        public final Collection<String> getExecutionParameters() {
            if (executionParameters instanceof SdkAutoConstructList) {
                return null;
            }
            return executionParameters;
        }

        public final void setExecutionParameters(Collection<String> executionParameters) {
            this.executionParameters = ExecutionParametersCopier.copy(executionParameters);
        }

        @Override
        public final Builder executionParameters(Collection<String> executionParameters) {
            this.executionParameters = ExecutionParametersCopier.copy(executionParameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder executionParameters(String... executionParameters) {
            executionParameters(Arrays.asList(executionParameters));
            return this;
        }

        @Override
        public QueryExecution build() {
            return new QueryExecution(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
