/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.athena.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The location in Amazon S3 where query results are stored and the encryption option, if any, used for query results.
 * These are known as "client-side settings". If workgroup settings override client-side settings, then the query uses
 * the workgroup settings.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResultConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<ResultConfiguration.Builder, ResultConfiguration> {
    private static final SdkField<String> OUTPUT_LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OutputLocation").getter(getter(ResultConfiguration::outputLocation))
            .setter(setter(Builder::outputLocation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputLocation").build()).build();

    private static final SdkField<EncryptionConfiguration> ENCRYPTION_CONFIGURATION_FIELD = SdkField
            .<EncryptionConfiguration> builder(MarshallingType.SDK_POJO).memberName("EncryptionConfiguration")
            .getter(getter(ResultConfiguration::encryptionConfiguration)).setter(setter(Builder::encryptionConfiguration))
            .constructor(EncryptionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionConfiguration").build())
            .build();

    private static final SdkField<String> EXPECTED_BUCKET_OWNER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExpectedBucketOwner").getter(getter(ResultConfiguration::expectedBucketOwner))
            .setter(setter(Builder::expectedBucketOwner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpectedBucketOwner").build())
            .build();

    private static final SdkField<AclConfiguration> ACL_CONFIGURATION_FIELD = SdkField
            .<AclConfiguration> builder(MarshallingType.SDK_POJO).memberName("AclConfiguration")
            .getter(getter(ResultConfiguration::aclConfiguration)).setter(setter(Builder::aclConfiguration))
            .constructor(AclConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AclConfiguration").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OUTPUT_LOCATION_FIELD,
            ENCRYPTION_CONFIGURATION_FIELD, EXPECTED_BUCKET_OWNER_FIELD, ACL_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String outputLocation;

    private final EncryptionConfiguration encryptionConfiguration;

    private final String expectedBucketOwner;

    private final AclConfiguration aclConfiguration;

    private ResultConfiguration(BuilderImpl builder) {
        this.outputLocation = builder.outputLocation;
        this.encryptionConfiguration = builder.encryptionConfiguration;
        this.expectedBucketOwner = builder.expectedBucketOwner;
        this.aclConfiguration = builder.aclConfiguration;
    }

    /**
     * <p>
     * The location in Amazon S3 where your query results are stored, such as <code>s3://path/to/query/bucket/</code>.
     * To run the query, you must specify the query results location using one of the ways: either for individual
     * queries using either this setting (client-side), or in the workgroup, using <a>WorkGroupConfiguration</a>. If
     * none of them is set, Athena issues an error that no output location is provided. For more information, see <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/querying.html">Query Results</a>. If workgroup settings
     * override client-side settings, then the query uses the settings specified for the workgroup. See
     * <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
     * </p>
     * 
     * @return The location in Amazon S3 where your query results are stored, such as
     *         <code>s3://path/to/query/bucket/</code>. To run the query, you must specify the query results location
     *         using one of the ways: either for individual queries using either this setting (client-side), or in the
     *         workgroup, using <a>WorkGroupConfiguration</a>. If none of them is set, Athena issues an error that no
     *         output location is provided. For more information, see <a
     *         href="https://docs.aws.amazon.com/athena/latest/ug/querying.html">Query Results</a>. If workgroup
     *         settings override client-side settings, then the query uses the settings specified for the workgroup. See
     *         <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
     */
    public final String outputLocation() {
        return outputLocation;
    }

    /**
     * <p>
     * If query results are encrypted in Amazon S3, indicates the encryption option used (for example,
     * <code>SSE_KMS</code> or <code>CSE_KMS</code>) and key information. This is a client-side setting. If workgroup
     * settings override client-side settings, then the query uses the encryption configuration that is specified for
     * the workgroup, and also uses the location for storing query results specified in the workgroup. See
     * <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings Override
     * Client-Side Settings</a>.
     * </p>
     * 
     * @return If query results are encrypted in Amazon S3, indicates the encryption option used (for example,
     *         <code>SSE_KMS</code> or <code>CSE_KMS</code>) and key information. This is a client-side setting. If
     *         workgroup settings override client-side settings, then the query uses the encryption configuration that
     *         is specified for the workgroup, and also uses the location for storing query results specified in the
     *         workgroup. See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
     *         href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
     *         Override Client-Side Settings</a>.
     */
    public final EncryptionConfiguration encryptionConfiguration() {
        return encryptionConfiguration;
    }

    /**
     * <p>
     * The Amazon Web Services account ID that you expect to be the owner of the Amazon S3 bucket specified by
     * <a>ResultConfiguration$OutputLocation</a>. If set, Athena uses the value for <code>ExpectedBucketOwner</code>
     * when it makes Amazon S3 calls to your specified output location. If the <code>ExpectedBucketOwner</code> Amazon
     * Web Services account ID does not match the actual owner of the Amazon S3 bucket, the call fails with a
     * permissions error.
     * </p>
     * <p>
     * This is a client-side setting. If workgroup settings override client-side settings, then the query uses the
     * <code>ExpectedBucketOwner</code> setting that is specified for the workgroup, and also uses the location for
     * storing query results specified in the workgroup. See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>
     * and <a href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
     * Override Client-Side Settings</a>.
     * </p>
     * 
     * @return The Amazon Web Services account ID that you expect to be the owner of the Amazon S3 bucket specified by
     *         <a>ResultConfiguration$OutputLocation</a>. If set, Athena uses the value for
     *         <code>ExpectedBucketOwner</code> when it makes Amazon S3 calls to your specified output location. If the
     *         <code>ExpectedBucketOwner</code> Amazon Web Services account ID does not match the actual owner of the
     *         Amazon S3 bucket, the call fails with a permissions error.</p>
     *         <p>
     *         This is a client-side setting. If workgroup settings override client-side settings, then the query uses
     *         the <code>ExpectedBucketOwner</code> setting that is specified for the workgroup, and also uses the
     *         location for storing query results specified in the workgroup. See
     *         <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
     *         href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
     *         Override Client-Side Settings</a>.
     */
    public final String expectedBucketOwner() {
        return expectedBucketOwner;
    }

    /**
     * <p>
     * Indicates that an Amazon S3 canned ACL should be set to control ownership of stored query results. Currently the
     * only supported canned ACL is <code>BUCKET_OWNER_FULL_CONTROL</code>. This is a client-side setting. If workgroup
     * settings override client-side settings, then the query uses the ACL configuration that is specified for the
     * workgroup, and also uses the location for storing query results specified in the workgroup. For more information,
     * see <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings Override
     * Client-Side Settings</a>.
     * </p>
     * 
     * @return Indicates that an Amazon S3 canned ACL should be set to control ownership of stored query results.
     *         Currently the only supported canned ACL is <code>BUCKET_OWNER_FULL_CONTROL</code>. This is a client-side
     *         setting. If workgroup settings override client-side settings, then the query uses the ACL configuration
     *         that is specified for the workgroup, and also uses the location for storing query results specified in
     *         the workgroup. For more information, see <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and
     *         <a href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup
     *         Settings Override Client-Side Settings</a>.
     */
    public final AclConfiguration aclConfiguration() {
        return aclConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(outputLocation());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(expectedBucketOwner());
        hashCode = 31 * hashCode + Objects.hashCode(aclConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResultConfiguration)) {
            return false;
        }
        ResultConfiguration other = (ResultConfiguration) obj;
        return Objects.equals(outputLocation(), other.outputLocation())
                && Objects.equals(encryptionConfiguration(), other.encryptionConfiguration())
                && Objects.equals(expectedBucketOwner(), other.expectedBucketOwner())
                && Objects.equals(aclConfiguration(), other.aclConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResultConfiguration").add("OutputLocation", outputLocation())
                .add("EncryptionConfiguration", encryptionConfiguration()).add("ExpectedBucketOwner", expectedBucketOwner())
                .add("AclConfiguration", aclConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OutputLocation":
            return Optional.ofNullable(clazz.cast(outputLocation()));
        case "EncryptionConfiguration":
            return Optional.ofNullable(clazz.cast(encryptionConfiguration()));
        case "ExpectedBucketOwner":
            return Optional.ofNullable(clazz.cast(expectedBucketOwner()));
        case "AclConfiguration":
            return Optional.ofNullable(clazz.cast(aclConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ResultConfiguration, T> g) {
        return obj -> g.apply((ResultConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResultConfiguration> {
        /**
         * <p>
         * The location in Amazon S3 where your query results are stored, such as
         * <code>s3://path/to/query/bucket/</code>. To run the query, you must specify the query results location using
         * one of the ways: either for individual queries using either this setting (client-side), or in the workgroup,
         * using <a>WorkGroupConfiguration</a>. If none of them is set, Athena issues an error that no output location
         * is provided. For more information, see <a
         * href="https://docs.aws.amazon.com/athena/latest/ug/querying.html">Query Results</a>. If workgroup settings
         * override client-side settings, then the query uses the settings specified for the workgroup. See
         * <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
         * </p>
         * 
         * @param outputLocation
         *        The location in Amazon S3 where your query results are stored, such as
         *        <code>s3://path/to/query/bucket/</code>. To run the query, you must specify the query results location
         *        using one of the ways: either for individual queries using either this setting (client-side), or in
         *        the workgroup, using <a>WorkGroupConfiguration</a>. If none of them is set, Athena issues an error
         *        that no output location is provided. For more information, see <a
         *        href="https://docs.aws.amazon.com/athena/latest/ug/querying.html">Query Results</a>. If workgroup
         *        settings override client-side settings, then the query uses the settings specified for the workgroup.
         *        See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputLocation(String outputLocation);

        /**
         * <p>
         * If query results are encrypted in Amazon S3, indicates the encryption option used (for example,
         * <code>SSE_KMS</code> or <code>CSE_KMS</code>) and key information. This is a client-side setting. If
         * workgroup settings override client-side settings, then the query uses the encryption configuration that is
         * specified for the workgroup, and also uses the location for storing query results specified in the workgroup.
         * See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
         * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
         * Override Client-Side Settings</a>.
         * </p>
         * 
         * @param encryptionConfiguration
         *        If query results are encrypted in Amazon S3, indicates the encryption option used (for example,
         *        <code>SSE_KMS</code> or <code>CSE_KMS</code>) and key information. This is a client-side setting. If
         *        workgroup settings override client-side settings, then the query uses the encryption configuration
         *        that is specified for the workgroup, and also uses the location for storing query results specified in
         *        the workgroup. See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
         *        href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup
         *        Settings Override Client-Side Settings</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionConfiguration(EncryptionConfiguration encryptionConfiguration);

        /**
         * <p>
         * If query results are encrypted in Amazon S3, indicates the encryption option used (for example,
         * <code>SSE_KMS</code> or <code>CSE_KMS</code>) and key information. This is a client-side setting. If
         * workgroup settings override client-side settings, then the query uses the encryption configuration that is
         * specified for the workgroup, and also uses the location for storing query results specified in the workgroup.
         * See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
         * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
         * Override Client-Side Settings</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link EncryptionConfiguration.Builder} avoiding
         * the need to create one manually via {@link EncryptionConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EncryptionConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #encryptionConfiguration(EncryptionConfiguration)}.
         * 
         * @param encryptionConfiguration
         *        a consumer that will call methods on {@link EncryptionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionConfiguration(EncryptionConfiguration)
         */
        default Builder encryptionConfiguration(Consumer<EncryptionConfiguration.Builder> encryptionConfiguration) {
            return encryptionConfiguration(EncryptionConfiguration.builder().applyMutation(encryptionConfiguration).build());
        }

        /**
         * <p>
         * The Amazon Web Services account ID that you expect to be the owner of the Amazon S3 bucket specified by
         * <a>ResultConfiguration$OutputLocation</a>. If set, Athena uses the value for <code>ExpectedBucketOwner</code>
         * when it makes Amazon S3 calls to your specified output location. If the <code>ExpectedBucketOwner</code>
         * Amazon Web Services account ID does not match the actual owner of the Amazon S3 bucket, the call fails with a
         * permissions error.
         * </p>
         * <p>
         * This is a client-side setting. If workgroup settings override client-side settings, then the query uses the
         * <code>ExpectedBucketOwner</code> setting that is specified for the workgroup, and also uses the location for
         * storing query results specified in the workgroup. See
         * <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
         * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
         * Override Client-Side Settings</a>.
         * </p>
         * 
         * @param expectedBucketOwner
         *        The Amazon Web Services account ID that you expect to be the owner of the Amazon S3 bucket specified
         *        by <a>ResultConfiguration$OutputLocation</a>. If set, Athena uses the value for
         *        <code>ExpectedBucketOwner</code> when it makes Amazon S3 calls to your specified output location. If
         *        the <code>ExpectedBucketOwner</code> Amazon Web Services account ID does not match the actual owner of
         *        the Amazon S3 bucket, the call fails with a permissions error.</p>
         *        <p>
         *        This is a client-side setting. If workgroup settings override client-side settings, then the query
         *        uses the <code>ExpectedBucketOwner</code> setting that is specified for the workgroup, and also uses
         *        the location for storing query results specified in the workgroup. See
         *        <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
         *        href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup
         *        Settings Override Client-Side Settings</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expectedBucketOwner(String expectedBucketOwner);

        /**
         * <p>
         * Indicates that an Amazon S3 canned ACL should be set to control ownership of stored query results. Currently
         * the only supported canned ACL is <code>BUCKET_OWNER_FULL_CONTROL</code>. This is a client-side setting. If
         * workgroup settings override client-side settings, then the query uses the ACL configuration that is specified
         * for the workgroup, and also uses the location for storing query results specified in the workgroup. For more
         * information, see <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
         * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
         * Override Client-Side Settings</a>.
         * </p>
         * 
         * @param aclConfiguration
         *        Indicates that an Amazon S3 canned ACL should be set to control ownership of stored query results.
         *        Currently the only supported canned ACL is <code>BUCKET_OWNER_FULL_CONTROL</code>. This is a
         *        client-side setting. If workgroup settings override client-side settings, then the query uses the ACL
         *        configuration that is specified for the workgroup, and also uses the location for storing query
         *        results specified in the workgroup. For more information, see
         *        <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
         *        href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup
         *        Settings Override Client-Side Settings</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aclConfiguration(AclConfiguration aclConfiguration);

        /**
         * <p>
         * Indicates that an Amazon S3 canned ACL should be set to control ownership of stored query results. Currently
         * the only supported canned ACL is <code>BUCKET_OWNER_FULL_CONTROL</code>. This is a client-side setting. If
         * workgroup settings override client-side settings, then the query uses the ACL configuration that is specified
         * for the workgroup, and also uses the location for storing query results specified in the workgroup. For more
         * information, see <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
         * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
         * Override Client-Side Settings</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link AclConfiguration.Builder} avoiding the
         * need to create one manually via {@link AclConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AclConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #aclConfiguration(AclConfiguration)}.
         * 
         * @param aclConfiguration
         *        a consumer that will call methods on {@link AclConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #aclConfiguration(AclConfiguration)
         */
        default Builder aclConfiguration(Consumer<AclConfiguration.Builder> aclConfiguration) {
            return aclConfiguration(AclConfiguration.builder().applyMutation(aclConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String outputLocation;

        private EncryptionConfiguration encryptionConfiguration;

        private String expectedBucketOwner;

        private AclConfiguration aclConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(ResultConfiguration model) {
            outputLocation(model.outputLocation);
            encryptionConfiguration(model.encryptionConfiguration);
            expectedBucketOwner(model.expectedBucketOwner);
            aclConfiguration(model.aclConfiguration);
        }

        public final String getOutputLocation() {
            return outputLocation;
        }

        public final void setOutputLocation(String outputLocation) {
            this.outputLocation = outputLocation;
        }

        @Override
        public final Builder outputLocation(String outputLocation) {
            this.outputLocation = outputLocation;
            return this;
        }

        public final EncryptionConfiguration.Builder getEncryptionConfiguration() {
            return encryptionConfiguration != null ? encryptionConfiguration.toBuilder() : null;
        }

        public final void setEncryptionConfiguration(EncryptionConfiguration.BuilderImpl encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration != null ? encryptionConfiguration.build() : null;
        }

        @Override
        public final Builder encryptionConfiguration(EncryptionConfiguration encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration;
            return this;
        }

        public final String getExpectedBucketOwner() {
            return expectedBucketOwner;
        }

        public final void setExpectedBucketOwner(String expectedBucketOwner) {
            this.expectedBucketOwner = expectedBucketOwner;
        }

        @Override
        public final Builder expectedBucketOwner(String expectedBucketOwner) {
            this.expectedBucketOwner = expectedBucketOwner;
            return this;
        }

        public final AclConfiguration.Builder getAclConfiguration() {
            return aclConfiguration != null ? aclConfiguration.toBuilder() : null;
        }

        public final void setAclConfiguration(AclConfiguration.BuilderImpl aclConfiguration) {
            this.aclConfiguration = aclConfiguration != null ? aclConfiguration.build() : null;
        }

        @Override
        public final Builder aclConfiguration(AclConfiguration aclConfiguration) {
            this.aclConfiguration = aclConfiguration;
            return this;
        }

        @Override
        public ResultConfiguration build() {
            return new ResultConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
