/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.athena.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The information about the updates in the query results, such as output location and encryption configuration for the
 * query results.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResultConfigurationUpdates implements SdkPojo, Serializable,
        ToCopyableBuilder<ResultConfigurationUpdates.Builder, ResultConfigurationUpdates> {
    private static final SdkField<String> OUTPUT_LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OutputLocation").getter(getter(ResultConfigurationUpdates::outputLocation))
            .setter(setter(Builder::outputLocation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputLocation").build()).build();

    private static final SdkField<Boolean> REMOVE_OUTPUT_LOCATION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("RemoveOutputLocation").getter(getter(ResultConfigurationUpdates::removeOutputLocation))
            .setter(setter(Builder::removeOutputLocation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RemoveOutputLocation").build())
            .build();

    private static final SdkField<EncryptionConfiguration> ENCRYPTION_CONFIGURATION_FIELD = SdkField
            .<EncryptionConfiguration> builder(MarshallingType.SDK_POJO).memberName("EncryptionConfiguration")
            .getter(getter(ResultConfigurationUpdates::encryptionConfiguration)).setter(setter(Builder::encryptionConfiguration))
            .constructor(EncryptionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionConfiguration").build())
            .build();

    private static final SdkField<Boolean> REMOVE_ENCRYPTION_CONFIGURATION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("RemoveEncryptionConfiguration")
            .getter(getter(ResultConfigurationUpdates::removeEncryptionConfiguration))
            .setter(setter(Builder::removeEncryptionConfiguration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RemoveEncryptionConfiguration")
                    .build()).build();

    private static final SdkField<String> EXPECTED_BUCKET_OWNER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExpectedBucketOwner").getter(getter(ResultConfigurationUpdates::expectedBucketOwner))
            .setter(setter(Builder::expectedBucketOwner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpectedBucketOwner").build())
            .build();

    private static final SdkField<Boolean> REMOVE_EXPECTED_BUCKET_OWNER_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN).memberName("RemoveExpectedBucketOwner")
            .getter(getter(ResultConfigurationUpdates::removeExpectedBucketOwner))
            .setter(setter(Builder::removeExpectedBucketOwner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RemoveExpectedBucketOwner").build())
            .build();

    private static final SdkField<AclConfiguration> ACL_CONFIGURATION_FIELD = SdkField
            .<AclConfiguration> builder(MarshallingType.SDK_POJO).memberName("AclConfiguration")
            .getter(getter(ResultConfigurationUpdates::aclConfiguration)).setter(setter(Builder::aclConfiguration))
            .constructor(AclConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AclConfiguration").build()).build();

    private static final SdkField<Boolean> REMOVE_ACL_CONFIGURATION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("RemoveAclConfiguration").getter(getter(ResultConfigurationUpdates::removeAclConfiguration))
            .setter(setter(Builder::removeAclConfiguration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RemoveAclConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OUTPUT_LOCATION_FIELD,
            REMOVE_OUTPUT_LOCATION_FIELD, ENCRYPTION_CONFIGURATION_FIELD, REMOVE_ENCRYPTION_CONFIGURATION_FIELD,
            EXPECTED_BUCKET_OWNER_FIELD, REMOVE_EXPECTED_BUCKET_OWNER_FIELD, ACL_CONFIGURATION_FIELD,
            REMOVE_ACL_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String outputLocation;

    private final Boolean removeOutputLocation;

    private final EncryptionConfiguration encryptionConfiguration;

    private final Boolean removeEncryptionConfiguration;

    private final String expectedBucketOwner;

    private final Boolean removeExpectedBucketOwner;

    private final AclConfiguration aclConfiguration;

    private final Boolean removeAclConfiguration;

    private ResultConfigurationUpdates(BuilderImpl builder) {
        this.outputLocation = builder.outputLocation;
        this.removeOutputLocation = builder.removeOutputLocation;
        this.encryptionConfiguration = builder.encryptionConfiguration;
        this.removeEncryptionConfiguration = builder.removeEncryptionConfiguration;
        this.expectedBucketOwner = builder.expectedBucketOwner;
        this.removeExpectedBucketOwner = builder.removeExpectedBucketOwner;
        this.aclConfiguration = builder.aclConfiguration;
        this.removeAclConfiguration = builder.removeAclConfiguration;
    }

    /**
     * <p>
     * The location in Amazon S3 where your query and calculation results are stored, such as
     * <code>s3://path/to/query/bucket/</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/querying.html">Working with query results, recent queries, and
     * output files</a>. If workgroup settings override client-side settings, then the query uses the location for the
     * query results and the encryption configuration that are specified for the workgroup. The
     * "workgroup settings override" is specified in <code>EnforceWorkGroupConfiguration</code> (true/false) in the
     * <code>WorkGroupConfiguration</code>. See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
     * </p>
     * 
     * @return The location in Amazon S3 where your query and calculation results are stored, such as
     *         <code>s3://path/to/query/bucket/</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/athena/latest/ug/querying.html">Working with query results, recent
     *         queries, and output files</a>. If workgroup settings override client-side settings, then the query uses
     *         the location for the query results and the encryption configuration that are specified for the workgroup.
     *         The "workgroup settings override" is specified in <code>EnforceWorkGroupConfiguration</code> (true/false)
     *         in the <code>WorkGroupConfiguration</code>. See
     *         <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
     */
    public final String outputLocation() {
        return outputLocation;
    }

    /**
     * <p>
     * If set to "true", indicates that the previously-specified query results location (also known as a client-side
     * setting) for queries in this workgroup should be ignored and set to null. If set to "false" or not set, and a
     * value is present in the <code>OutputLocation</code> in <code>ResultConfigurationUpdates</code> (the client-side
     * setting), the <code>OutputLocation</code> in the workgroup's <code>ResultConfiguration</code> will be updated
     * with the new value. For more information, see <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings Override
     * Client-Side Settings</a>.
     * </p>
     * 
     * @return If set to "true", indicates that the previously-specified query results location (also known as a
     *         client-side setting) for queries in this workgroup should be ignored and set to null. If set to "false"
     *         or not set, and a value is present in the <code>OutputLocation</code> in
     *         <code>ResultConfigurationUpdates</code> (the client-side setting), the <code>OutputLocation</code> in the
     *         workgroup's <code>ResultConfiguration</code> will be updated with the new value. For more information,
     *         see <a href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup
     *         Settings Override Client-Side Settings</a>.
     */
    public final Boolean removeOutputLocation() {
        return removeOutputLocation;
    }

    /**
     * <p>
     * The encryption configuration for query and calculation results.
     * </p>
     * 
     * @return The encryption configuration for query and calculation results.
     */
    public final EncryptionConfiguration encryptionConfiguration() {
        return encryptionConfiguration;
    }

    /**
     * <p>
     * If set to "true", indicates that the previously-specified encryption configuration (also known as the client-side
     * setting) for queries in this workgroup should be ignored and set to null. If set to "false" or not set, and a
     * value is present in the <code>EncryptionConfiguration</code> in <code>ResultConfigurationUpdates</code> (the
     * client-side setting), the <code>EncryptionConfiguration</code> in the workgroup's
     * <code>ResultConfiguration</code> will be updated with the new value. For more information, see <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings Override
     * Client-Side Settings</a>.
     * </p>
     * 
     * @return If set to "true", indicates that the previously-specified encryption configuration (also known as the
     *         client-side setting) for queries in this workgroup should be ignored and set to null. If set to "false"
     *         or not set, and a value is present in the <code>EncryptionConfiguration</code> in
     *         <code>ResultConfigurationUpdates</code> (the client-side setting), the
     *         <code>EncryptionConfiguration</code> in the workgroup's <code>ResultConfiguration</code> will be updated
     *         with the new value. For more information, see <a
     *         href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
     *         Override Client-Side Settings</a>.
     */
    public final Boolean removeEncryptionConfiguration() {
        return removeEncryptionConfiguration;
    }

    /**
     * <p>
     * The Amazon Web Services account ID that you expect to be the owner of the Amazon S3 bucket specified by
     * <a>ResultConfiguration$OutputLocation</a>. If set, Athena uses the value for <code>ExpectedBucketOwner</code>
     * when it makes Amazon S3 calls to your specified output location. If the <code>ExpectedBucketOwner</code> Amazon
     * Web Services account ID does not match the actual owner of the Amazon S3 bucket, the call fails with a
     * permissions error.
     * </p>
     * <p>
     * If workgroup settings override client-side settings, then the query uses the <code>ExpectedBucketOwner</code>
     * setting that is specified for the workgroup, and also uses the location for storing query results specified in
     * the workgroup. See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings Override
     * Client-Side Settings</a>.
     * </p>
     * 
     * @return The Amazon Web Services account ID that you expect to be the owner of the Amazon S3 bucket specified by
     *         <a>ResultConfiguration$OutputLocation</a>. If set, Athena uses the value for
     *         <code>ExpectedBucketOwner</code> when it makes Amazon S3 calls to your specified output location. If the
     *         <code>ExpectedBucketOwner</code> Amazon Web Services account ID does not match the actual owner of the
     *         Amazon S3 bucket, the call fails with a permissions error.</p>
     *         <p>
     *         If workgroup settings override client-side settings, then the query uses the
     *         <code>ExpectedBucketOwner</code> setting that is specified for the workgroup, and also uses the location
     *         for storing query results specified in the workgroup. See
     *         <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
     *         href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
     *         Override Client-Side Settings</a>.
     */
    public final String expectedBucketOwner() {
        return expectedBucketOwner;
    }

    /**
     * <p>
     * If set to "true", removes the Amazon Web Services account ID previously specified for
     * <a>ResultConfiguration$ExpectedBucketOwner</a>. If set to "false" or not set, and a value is present in the
     * <code>ExpectedBucketOwner</code> in <code>ResultConfigurationUpdates</code> (the client-side setting), the
     * <code>ExpectedBucketOwner</code> in the workgroup's <code>ResultConfiguration</code> is updated with the new
     * value. For more information, see <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings Override
     * Client-Side Settings</a>.
     * </p>
     * 
     * @return If set to "true", removes the Amazon Web Services account ID previously specified for
     *         <a>ResultConfiguration$ExpectedBucketOwner</a>. If set to "false" or not set, and a value is present in
     *         the <code>ExpectedBucketOwner</code> in <code>ResultConfigurationUpdates</code> (the client-side
     *         setting), the <code>ExpectedBucketOwner</code> in the workgroup's <code>ResultConfiguration</code> is
     *         updated with the new value. For more information, see <a
     *         href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
     *         Override Client-Side Settings</a>.
     */
    public final Boolean removeExpectedBucketOwner() {
        return removeExpectedBucketOwner;
    }

    /**
     * <p>
     * The ACL configuration for the query results.
     * </p>
     * 
     * @return The ACL configuration for the query results.
     */
    public final AclConfiguration aclConfiguration() {
        return aclConfiguration;
    }

    /**
     * <p>
     * If set to <code>true</code>, indicates that the previously-specified ACL configuration for queries in this
     * workgroup should be ignored and set to null. If set to <code>false</code> or not set, and a value is present in
     * the <code>AclConfiguration</code> of <code>ResultConfigurationUpdates</code>, the <code>AclConfiguration</code>
     * in the workgroup's <code>ResultConfiguration</code> is updated with the new value. For more information, see <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings Override
     * Client-Side Settings</a>.
     * </p>
     * 
     * @return If set to <code>true</code>, indicates that the previously-specified ACL configuration for queries in
     *         this workgroup should be ignored and set to null. If set to <code>false</code> or not set, and a value is
     *         present in the <code>AclConfiguration</code> of <code>ResultConfigurationUpdates</code>, the
     *         <code>AclConfiguration</code> in the workgroup's <code>ResultConfiguration</code> is updated with the new
     *         value. For more information, see <a
     *         href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
     *         Override Client-Side Settings</a>.
     */
    public final Boolean removeAclConfiguration() {
        return removeAclConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(outputLocation());
        hashCode = 31 * hashCode + Objects.hashCode(removeOutputLocation());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(removeEncryptionConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(expectedBucketOwner());
        hashCode = 31 * hashCode + Objects.hashCode(removeExpectedBucketOwner());
        hashCode = 31 * hashCode + Objects.hashCode(aclConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(removeAclConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResultConfigurationUpdates)) {
            return false;
        }
        ResultConfigurationUpdates other = (ResultConfigurationUpdates) obj;
        return Objects.equals(outputLocation(), other.outputLocation())
                && Objects.equals(removeOutputLocation(), other.removeOutputLocation())
                && Objects.equals(encryptionConfiguration(), other.encryptionConfiguration())
                && Objects.equals(removeEncryptionConfiguration(), other.removeEncryptionConfiguration())
                && Objects.equals(expectedBucketOwner(), other.expectedBucketOwner())
                && Objects.equals(removeExpectedBucketOwner(), other.removeExpectedBucketOwner())
                && Objects.equals(aclConfiguration(), other.aclConfiguration())
                && Objects.equals(removeAclConfiguration(), other.removeAclConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResultConfigurationUpdates").add("OutputLocation", outputLocation())
                .add("RemoveOutputLocation", removeOutputLocation()).add("EncryptionConfiguration", encryptionConfiguration())
                .add("RemoveEncryptionConfiguration", removeEncryptionConfiguration())
                .add("ExpectedBucketOwner", expectedBucketOwner()).add("RemoveExpectedBucketOwner", removeExpectedBucketOwner())
                .add("AclConfiguration", aclConfiguration()).add("RemoveAclConfiguration", removeAclConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OutputLocation":
            return Optional.ofNullable(clazz.cast(outputLocation()));
        case "RemoveOutputLocation":
            return Optional.ofNullable(clazz.cast(removeOutputLocation()));
        case "EncryptionConfiguration":
            return Optional.ofNullable(clazz.cast(encryptionConfiguration()));
        case "RemoveEncryptionConfiguration":
            return Optional.ofNullable(clazz.cast(removeEncryptionConfiguration()));
        case "ExpectedBucketOwner":
            return Optional.ofNullable(clazz.cast(expectedBucketOwner()));
        case "RemoveExpectedBucketOwner":
            return Optional.ofNullable(clazz.cast(removeExpectedBucketOwner()));
        case "AclConfiguration":
            return Optional.ofNullable(clazz.cast(aclConfiguration()));
        case "RemoveAclConfiguration":
            return Optional.ofNullable(clazz.cast(removeAclConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ResultConfigurationUpdates, T> g) {
        return obj -> g.apply((ResultConfigurationUpdates) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResultConfigurationUpdates> {
        /**
         * <p>
         * The location in Amazon S3 where your query and calculation results are stored, such as
         * <code>s3://path/to/query/bucket/</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/athena/latest/ug/querying.html">Working with query results, recent queries,
         * and output files</a>. If workgroup settings override client-side settings, then the query uses the location
         * for the query results and the encryption configuration that are specified for the workgroup. The
         * "workgroup settings override" is specified in <code>EnforceWorkGroupConfiguration</code> (true/false) in the
         * <code>WorkGroupConfiguration</code>. See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
         * </p>
         * 
         * @param outputLocation
         *        The location in Amazon S3 where your query and calculation results are stored, such as
         *        <code>s3://path/to/query/bucket/</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/athena/latest/ug/querying.html">Working with query results, recent
         *        queries, and output files</a>. If workgroup settings override client-side settings, then the query
         *        uses the location for the query results and the encryption configuration that are specified for the
         *        workgroup. The "workgroup settings override" is specified in
         *        <code>EnforceWorkGroupConfiguration</code> (true/false) in the <code>WorkGroupConfiguration</code>.
         *        See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputLocation(String outputLocation);

        /**
         * <p>
         * If set to "true", indicates that the previously-specified query results location (also known as a client-side
         * setting) for queries in this workgroup should be ignored and set to null. If set to "false" or not set, and a
         * value is present in the <code>OutputLocation</code> in <code>ResultConfigurationUpdates</code> (the
         * client-side setting), the <code>OutputLocation</code> in the workgroup's <code>ResultConfiguration</code>
         * will be updated with the new value. For more information, see <a
         * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
         * Override Client-Side Settings</a>.
         * </p>
         * 
         * @param removeOutputLocation
         *        If set to "true", indicates that the previously-specified query results location (also known as a
         *        client-side setting) for queries in this workgroup should be ignored and set to null. If set to
         *        "false" or not set, and a value is present in the <code>OutputLocation</code> in
         *        <code>ResultConfigurationUpdates</code> (the client-side setting), the <code>OutputLocation</code> in
         *        the workgroup's <code>ResultConfiguration</code> will be updated with the new value. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup
         *        Settings Override Client-Side Settings</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder removeOutputLocation(Boolean removeOutputLocation);

        /**
         * <p>
         * The encryption configuration for query and calculation results.
         * </p>
         * 
         * @param encryptionConfiguration
         *        The encryption configuration for query and calculation results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionConfiguration(EncryptionConfiguration encryptionConfiguration);

        /**
         * <p>
         * The encryption configuration for query and calculation results.
         * </p>
         * This is a convenience method that creates an instance of the {@link EncryptionConfiguration.Builder} avoiding
         * the need to create one manually via {@link EncryptionConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EncryptionConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #encryptionConfiguration(EncryptionConfiguration)}.
         * 
         * @param encryptionConfiguration
         *        a consumer that will call methods on {@link EncryptionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionConfiguration(EncryptionConfiguration)
         */
        default Builder encryptionConfiguration(Consumer<EncryptionConfiguration.Builder> encryptionConfiguration) {
            return encryptionConfiguration(EncryptionConfiguration.builder().applyMutation(encryptionConfiguration).build());
        }

        /**
         * <p>
         * If set to "true", indicates that the previously-specified encryption configuration (also known as the
         * client-side setting) for queries in this workgroup should be ignored and set to null. If set to "false" or
         * not set, and a value is present in the <code>EncryptionConfiguration</code> in
         * <code>ResultConfigurationUpdates</code> (the client-side setting), the <code>EncryptionConfiguration</code>
         * in the workgroup's <code>ResultConfiguration</code> will be updated with the new value. For more information,
         * see <a href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup
         * Settings Override Client-Side Settings</a>.
         * </p>
         * 
         * @param removeEncryptionConfiguration
         *        If set to "true", indicates that the previously-specified encryption configuration (also known as the
         *        client-side setting) for queries in this workgroup should be ignored and set to null. If set to
         *        "false" or not set, and a value is present in the <code>EncryptionConfiguration</code> in
         *        <code>ResultConfigurationUpdates</code> (the client-side setting), the
         *        <code>EncryptionConfiguration</code> in the workgroup's <code>ResultConfiguration</code> will be
         *        updated with the new value. For more information, see <a
         *        href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup
         *        Settings Override Client-Side Settings</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder removeEncryptionConfiguration(Boolean removeEncryptionConfiguration);

        /**
         * <p>
         * The Amazon Web Services account ID that you expect to be the owner of the Amazon S3 bucket specified by
         * <a>ResultConfiguration$OutputLocation</a>. If set, Athena uses the value for <code>ExpectedBucketOwner</code>
         * when it makes Amazon S3 calls to your specified output location. If the <code>ExpectedBucketOwner</code>
         * Amazon Web Services account ID does not match the actual owner of the Amazon S3 bucket, the call fails with a
         * permissions error.
         * </p>
         * <p>
         * If workgroup settings override client-side settings, then the query uses the <code>ExpectedBucketOwner</code>
         * setting that is specified for the workgroup, and also uses the location for storing query results specified
         * in the workgroup. See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
         * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
         * Override Client-Side Settings</a>.
         * </p>
         * 
         * @param expectedBucketOwner
         *        The Amazon Web Services account ID that you expect to be the owner of the Amazon S3 bucket specified
         *        by <a>ResultConfiguration$OutputLocation</a>. If set, Athena uses the value for
         *        <code>ExpectedBucketOwner</code> when it makes Amazon S3 calls to your specified output location. If
         *        the <code>ExpectedBucketOwner</code> Amazon Web Services account ID does not match the actual owner of
         *        the Amazon S3 bucket, the call fails with a permissions error.</p>
         *        <p>
         *        If workgroup settings override client-side settings, then the query uses the
         *        <code>ExpectedBucketOwner</code> setting that is specified for the workgroup, and also uses the
         *        location for storing query results specified in the workgroup. See
         *        <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
         *        href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup
         *        Settings Override Client-Side Settings</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expectedBucketOwner(String expectedBucketOwner);

        /**
         * <p>
         * If set to "true", removes the Amazon Web Services account ID previously specified for
         * <a>ResultConfiguration$ExpectedBucketOwner</a>. If set to "false" or not set, and a value is present in the
         * <code>ExpectedBucketOwner</code> in <code>ResultConfigurationUpdates</code> (the client-side setting), the
         * <code>ExpectedBucketOwner</code> in the workgroup's <code>ResultConfiguration</code> is updated with the new
         * value. For more information, see <a
         * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
         * Override Client-Side Settings</a>.
         * </p>
         * 
         * @param removeExpectedBucketOwner
         *        If set to "true", removes the Amazon Web Services account ID previously specified for
         *        <a>ResultConfiguration$ExpectedBucketOwner</a>. If set to "false" or not set, and a value is present
         *        in the <code>ExpectedBucketOwner</code> in <code>ResultConfigurationUpdates</code> (the client-side
         *        setting), the <code>ExpectedBucketOwner</code> in the workgroup's <code>ResultConfiguration</code> is
         *        updated with the new value. For more information, see <a
         *        href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup
         *        Settings Override Client-Side Settings</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder removeExpectedBucketOwner(Boolean removeExpectedBucketOwner);

        /**
         * <p>
         * The ACL configuration for the query results.
         * </p>
         * 
         * @param aclConfiguration
         *        The ACL configuration for the query results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aclConfiguration(AclConfiguration aclConfiguration);

        /**
         * <p>
         * The ACL configuration for the query results.
         * </p>
         * This is a convenience method that creates an instance of the {@link AclConfiguration.Builder} avoiding the
         * need to create one manually via {@link AclConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AclConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #aclConfiguration(AclConfiguration)}.
         * 
         * @param aclConfiguration
         *        a consumer that will call methods on {@link AclConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #aclConfiguration(AclConfiguration)
         */
        default Builder aclConfiguration(Consumer<AclConfiguration.Builder> aclConfiguration) {
            return aclConfiguration(AclConfiguration.builder().applyMutation(aclConfiguration).build());
        }

        /**
         * <p>
         * If set to <code>true</code>, indicates that the previously-specified ACL configuration for queries in this
         * workgroup should be ignored and set to null. If set to <code>false</code> or not set, and a value is present
         * in the <code>AclConfiguration</code> of <code>ResultConfigurationUpdates</code>, the
         * <code>AclConfiguration</code> in the workgroup's <code>ResultConfiguration</code> is updated with the new
         * value. For more information, see <a
         * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
         * Override Client-Side Settings</a>.
         * </p>
         * 
         * @param removeAclConfiguration
         *        If set to <code>true</code>, indicates that the previously-specified ACL configuration for queries in
         *        this workgroup should be ignored and set to null. If set to <code>false</code> or not set, and a value
         *        is present in the <code>AclConfiguration</code> of <code>ResultConfigurationUpdates</code>, the
         *        <code>AclConfiguration</code> in the workgroup's <code>ResultConfiguration</code> is updated with the
         *        new value. For more information, see <a
         *        href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup
         *        Settings Override Client-Side Settings</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder removeAclConfiguration(Boolean removeAclConfiguration);
    }

    static final class BuilderImpl implements Builder {
        private String outputLocation;

        private Boolean removeOutputLocation;

        private EncryptionConfiguration encryptionConfiguration;

        private Boolean removeEncryptionConfiguration;

        private String expectedBucketOwner;

        private Boolean removeExpectedBucketOwner;

        private AclConfiguration aclConfiguration;

        private Boolean removeAclConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(ResultConfigurationUpdates model) {
            outputLocation(model.outputLocation);
            removeOutputLocation(model.removeOutputLocation);
            encryptionConfiguration(model.encryptionConfiguration);
            removeEncryptionConfiguration(model.removeEncryptionConfiguration);
            expectedBucketOwner(model.expectedBucketOwner);
            removeExpectedBucketOwner(model.removeExpectedBucketOwner);
            aclConfiguration(model.aclConfiguration);
            removeAclConfiguration(model.removeAclConfiguration);
        }

        public final String getOutputLocation() {
            return outputLocation;
        }

        public final void setOutputLocation(String outputLocation) {
            this.outputLocation = outputLocation;
        }

        @Override
        public final Builder outputLocation(String outputLocation) {
            this.outputLocation = outputLocation;
            return this;
        }

        public final Boolean getRemoveOutputLocation() {
            return removeOutputLocation;
        }

        public final void setRemoveOutputLocation(Boolean removeOutputLocation) {
            this.removeOutputLocation = removeOutputLocation;
        }

        @Override
        public final Builder removeOutputLocation(Boolean removeOutputLocation) {
            this.removeOutputLocation = removeOutputLocation;
            return this;
        }

        public final EncryptionConfiguration.Builder getEncryptionConfiguration() {
            return encryptionConfiguration != null ? encryptionConfiguration.toBuilder() : null;
        }

        public final void setEncryptionConfiguration(EncryptionConfiguration.BuilderImpl encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration != null ? encryptionConfiguration.build() : null;
        }

        @Override
        public final Builder encryptionConfiguration(EncryptionConfiguration encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration;
            return this;
        }

        public final Boolean getRemoveEncryptionConfiguration() {
            return removeEncryptionConfiguration;
        }

        public final void setRemoveEncryptionConfiguration(Boolean removeEncryptionConfiguration) {
            this.removeEncryptionConfiguration = removeEncryptionConfiguration;
        }

        @Override
        public final Builder removeEncryptionConfiguration(Boolean removeEncryptionConfiguration) {
            this.removeEncryptionConfiguration = removeEncryptionConfiguration;
            return this;
        }

        public final String getExpectedBucketOwner() {
            return expectedBucketOwner;
        }

        public final void setExpectedBucketOwner(String expectedBucketOwner) {
            this.expectedBucketOwner = expectedBucketOwner;
        }

        @Override
        public final Builder expectedBucketOwner(String expectedBucketOwner) {
            this.expectedBucketOwner = expectedBucketOwner;
            return this;
        }

        public final Boolean getRemoveExpectedBucketOwner() {
            return removeExpectedBucketOwner;
        }

        public final void setRemoveExpectedBucketOwner(Boolean removeExpectedBucketOwner) {
            this.removeExpectedBucketOwner = removeExpectedBucketOwner;
        }

        @Override
        public final Builder removeExpectedBucketOwner(Boolean removeExpectedBucketOwner) {
            this.removeExpectedBucketOwner = removeExpectedBucketOwner;
            return this;
        }

        public final AclConfiguration.Builder getAclConfiguration() {
            return aclConfiguration != null ? aclConfiguration.toBuilder() : null;
        }

        public final void setAclConfiguration(AclConfiguration.BuilderImpl aclConfiguration) {
            this.aclConfiguration = aclConfiguration != null ? aclConfiguration.build() : null;
        }

        @Override
        public final Builder aclConfiguration(AclConfiguration aclConfiguration) {
            this.aclConfiguration = aclConfiguration;
            return this;
        }

        public final Boolean getRemoveAclConfiguration() {
            return removeAclConfiguration;
        }

        public final void setRemoveAclConfiguration(Boolean removeAclConfiguration) {
            this.removeAclConfiguration = removeAclConfiguration;
        }

        @Override
        public final Builder removeAclConfiguration(Boolean removeAclConfiguration) {
            this.removeAclConfiguration = removeAclConfiguration;
            return this;
        }

        @Override
        public ResultConfigurationUpdates build() {
            return new ResultConfigurationUpdates(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
