/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.athena.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration of the workgroup, which includes the location in Amazon S3 where query and calculation results are
 * stored, the encryption option, if any, used for query and calculation results, whether the Amazon CloudWatch Metrics
 * are enabled for the workgroup and whether workgroup settings override query settings, and the data usage limits for
 * the amount of data scanned per query or per workgroup. The workgroup settings override is specified in
 * <code>EnforceWorkGroupConfiguration</code> (true/false) in the <code>WorkGroupConfiguration</code>. See
 * <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WorkGroupConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<WorkGroupConfiguration.Builder, WorkGroupConfiguration> {
    private static final SdkField<ResultConfiguration> RESULT_CONFIGURATION_FIELD = SdkField
            .<ResultConfiguration> builder(MarshallingType.SDK_POJO).memberName("ResultConfiguration")
            .getter(getter(WorkGroupConfiguration::resultConfiguration)).setter(setter(Builder::resultConfiguration))
            .constructor(ResultConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResultConfiguration").build())
            .build();

    private static final SdkField<Boolean> ENFORCE_WORK_GROUP_CONFIGURATION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EnforceWorkGroupConfiguration")
            .getter(getter(WorkGroupConfiguration::enforceWorkGroupConfiguration))
            .setter(setter(Builder::enforceWorkGroupConfiguration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnforceWorkGroupConfiguration")
                    .build()).build();

    private static final SdkField<Boolean> PUBLISH_CLOUD_WATCH_METRICS_ENABLED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("PublishCloudWatchMetricsEnabled")
            .getter(getter(WorkGroupConfiguration::publishCloudWatchMetricsEnabled))
            .setter(setter(Builder::publishCloudWatchMetricsEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PublishCloudWatchMetricsEnabled")
                    .build()).build();

    private static final SdkField<Long> BYTES_SCANNED_CUTOFF_PER_QUERY_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("BytesScannedCutoffPerQuery")
            .getter(getter(WorkGroupConfiguration::bytesScannedCutoffPerQuery))
            .setter(setter(Builder::bytesScannedCutoffPerQuery))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BytesScannedCutoffPerQuery").build())
            .build();

    private static final SdkField<Boolean> REQUESTER_PAYS_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("RequesterPaysEnabled").getter(getter(WorkGroupConfiguration::requesterPaysEnabled))
            .setter(setter(Builder::requesterPaysEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequesterPaysEnabled").build())
            .build();

    private static final SdkField<EngineVersion> ENGINE_VERSION_FIELD = SdkField
            .<EngineVersion> builder(MarshallingType.SDK_POJO).memberName("EngineVersion")
            .getter(getter(WorkGroupConfiguration::engineVersion)).setter(setter(Builder::engineVersion))
            .constructor(EngineVersion::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineVersion").build()).build();

    private static final SdkField<String> ADDITIONAL_CONFIGURATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AdditionalConfiguration").getter(getter(WorkGroupConfiguration::additionalConfiguration))
            .setter(setter(Builder::additionalConfiguration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdditionalConfiguration").build())
            .build();

    private static final SdkField<String> EXECUTION_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExecutionRole").getter(getter(WorkGroupConfiguration::executionRole))
            .setter(setter(Builder::executionRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionRole").build()).build();

    private static final SdkField<CustomerContentEncryptionConfiguration> CUSTOMER_CONTENT_ENCRYPTION_CONFIGURATION_FIELD = SdkField
            .<CustomerContentEncryptionConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("CustomerContentEncryptionConfiguration")
            .getter(getter(WorkGroupConfiguration::customerContentEncryptionConfiguration))
            .setter(setter(Builder::customerContentEncryptionConfiguration))
            .constructor(CustomerContentEncryptionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("CustomerContentEncryptionConfiguration").build()).build();

    private static final SdkField<Boolean> ENABLE_MINIMUM_ENCRYPTION_CONFIGURATION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EnableMinimumEncryptionConfiguration")
            .getter(getter(WorkGroupConfiguration::enableMinimumEncryptionConfiguration))
            .setter(setter(Builder::enableMinimumEncryptionConfiguration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("EnableMinimumEncryptionConfiguration").build()).build();

    private static final SdkField<IdentityCenterConfiguration> IDENTITY_CENTER_CONFIGURATION_FIELD = SdkField
            .<IdentityCenterConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("IdentityCenterConfiguration")
            .getter(getter(WorkGroupConfiguration::identityCenterConfiguration))
            .setter(setter(Builder::identityCenterConfiguration))
            .constructor(IdentityCenterConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentityCenterConfiguration")
                    .build()).build();

    private static final SdkField<QueryResultsS3AccessGrantsConfiguration> QUERY_RESULTS_S3_ACCESS_GRANTS_CONFIGURATION_FIELD = SdkField
            .<QueryResultsS3AccessGrantsConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("QueryResultsS3AccessGrantsConfiguration")
            .getter(getter(WorkGroupConfiguration::queryResultsS3AccessGrantsConfiguration))
            .setter(setter(Builder::queryResultsS3AccessGrantsConfiguration))
            .constructor(QueryResultsS3AccessGrantsConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("QueryResultsS3AccessGrantsConfiguration").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESULT_CONFIGURATION_FIELD,
            ENFORCE_WORK_GROUP_CONFIGURATION_FIELD, PUBLISH_CLOUD_WATCH_METRICS_ENABLED_FIELD,
            BYTES_SCANNED_CUTOFF_PER_QUERY_FIELD, REQUESTER_PAYS_ENABLED_FIELD, ENGINE_VERSION_FIELD,
            ADDITIONAL_CONFIGURATION_FIELD, EXECUTION_ROLE_FIELD, CUSTOMER_CONTENT_ENCRYPTION_CONFIGURATION_FIELD,
            ENABLE_MINIMUM_ENCRYPTION_CONFIGURATION_FIELD, IDENTITY_CENTER_CONFIGURATION_FIELD,
            QUERY_RESULTS_S3_ACCESS_GRANTS_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final ResultConfiguration resultConfiguration;

    private final Boolean enforceWorkGroupConfiguration;

    private final Boolean publishCloudWatchMetricsEnabled;

    private final Long bytesScannedCutoffPerQuery;

    private final Boolean requesterPaysEnabled;

    private final EngineVersion engineVersion;

    private final String additionalConfiguration;

    private final String executionRole;

    private final CustomerContentEncryptionConfiguration customerContentEncryptionConfiguration;

    private final Boolean enableMinimumEncryptionConfiguration;

    private final IdentityCenterConfiguration identityCenterConfiguration;

    private final QueryResultsS3AccessGrantsConfiguration queryResultsS3AccessGrantsConfiguration;

    private WorkGroupConfiguration(BuilderImpl builder) {
        this.resultConfiguration = builder.resultConfiguration;
        this.enforceWorkGroupConfiguration = builder.enforceWorkGroupConfiguration;
        this.publishCloudWatchMetricsEnabled = builder.publishCloudWatchMetricsEnabled;
        this.bytesScannedCutoffPerQuery = builder.bytesScannedCutoffPerQuery;
        this.requesterPaysEnabled = builder.requesterPaysEnabled;
        this.engineVersion = builder.engineVersion;
        this.additionalConfiguration = builder.additionalConfiguration;
        this.executionRole = builder.executionRole;
        this.customerContentEncryptionConfiguration = builder.customerContentEncryptionConfiguration;
        this.enableMinimumEncryptionConfiguration = builder.enableMinimumEncryptionConfiguration;
        this.identityCenterConfiguration = builder.identityCenterConfiguration;
        this.queryResultsS3AccessGrantsConfiguration = builder.queryResultsS3AccessGrantsConfiguration;
    }

    /**
     * <p>
     * The configuration for the workgroup, which includes the location in Amazon S3 where query and calculation results
     * are stored and the encryption option, if any, used for query and calculation results. To run the query, you must
     * specify the query results location using one of the ways: either in the workgroup using this setting, or for
     * individual queries (client-side), using <a>ResultConfiguration$OutputLocation</a>. If none of them is set, Athena
     * issues an error that no output location is provided.
     * </p>
     * 
     * @return The configuration for the workgroup, which includes the location in Amazon S3 where query and calculation
     *         results are stored and the encryption option, if any, used for query and calculation results. To run the
     *         query, you must specify the query results location using one of the ways: either in the workgroup using
     *         this setting, or for individual queries (client-side), using <a>ResultConfiguration$OutputLocation</a>.
     *         If none of them is set, Athena issues an error that no output location is provided.
     */
    public final ResultConfiguration resultConfiguration() {
        return resultConfiguration;
    }

    /**
     * <p>
     * If set to "true", the settings for the workgroup override client-side settings. If set to "false", client-side
     * settings are used. For more information, see <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings Override
     * Client-Side Settings</a>.
     * </p>
     * 
     * @return If set to "true", the settings for the workgroup override client-side settings. If set to "false",
     *         client-side settings are used. For more information, see <a
     *         href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
     *         Override Client-Side Settings</a>.
     */
    public final Boolean enforceWorkGroupConfiguration() {
        return enforceWorkGroupConfiguration;
    }

    /**
     * <p>
     * Indicates that the Amazon CloudWatch metrics are enabled for the workgroup.
     * </p>
     * 
     * @return Indicates that the Amazon CloudWatch metrics are enabled for the workgroup.
     */
    public final Boolean publishCloudWatchMetricsEnabled() {
        return publishCloudWatchMetricsEnabled;
    }

    /**
     * <p>
     * The upper data usage limit (cutoff) for the amount of bytes a single query in a workgroup is allowed to scan.
     * </p>
     * 
     * @return The upper data usage limit (cutoff) for the amount of bytes a single query in a workgroup is allowed to
     *         scan.
     */
    public final Long bytesScannedCutoffPerQuery() {
        return bytesScannedCutoffPerQuery;
    }

    /**
     * <p>
     * If set to <code>true</code>, allows members assigned to a workgroup to reference Amazon S3 Requester Pays buckets
     * in queries. If set to <code>false</code>, workgroup members cannot query data from Requester Pays buckets, and
     * queries that retrieve data from Requester Pays buckets cause an error. The default is <code>false</code>. For
     * more information about Requester Pays buckets, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/RequesterPaysBuckets.html">Requester Pays Buckets</a> in
     * the <i>Amazon Simple Storage Service Developer Guide</i>.
     * </p>
     * 
     * @return If set to <code>true</code>, allows members assigned to a workgroup to reference Amazon S3 Requester Pays
     *         buckets in queries. If set to <code>false</code>, workgroup members cannot query data from Requester Pays
     *         buckets, and queries that retrieve data from Requester Pays buckets cause an error. The default is
     *         <code>false</code>. For more information about Requester Pays buckets, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/RequesterPaysBuckets.html">Requester Pays
     *         Buckets</a> in the <i>Amazon Simple Storage Service Developer Guide</i>.
     */
    public final Boolean requesterPaysEnabled() {
        return requesterPaysEnabled;
    }

    /**
     * <p>
     * The engine version that all queries running on the workgroup use. Queries on the
     * <code>AmazonAthenaPreviewFunctionality</code> workgroup run on the preview engine regardless of this setting.
     * </p>
     * 
     * @return The engine version that all queries running on the workgroup use. Queries on the
     *         <code>AmazonAthenaPreviewFunctionality</code> workgroup run on the preview engine regardless of this
     *         setting.
     */
    public final EngineVersion engineVersion() {
        return engineVersion;
    }

    /**
     * <p>
     * Specifies a user defined JSON string that is passed to the notebook engine.
     * </p>
     * 
     * @return Specifies a user defined JSON string that is passed to the notebook engine.
     */
    public final String additionalConfiguration() {
        return additionalConfiguration;
    }

    /**
     * <p>
     * The ARN of the execution role used to access user resources for Spark sessions and IAM Identity Center enabled
     * workgroups. This property applies only to Spark enabled workgroups and IAM Identity Center enabled workgroups.
     * The property is required for IAM Identity Center enabled workgroups.
     * </p>
     * 
     * @return The ARN of the execution role used to access user resources for Spark sessions and IAM Identity Center
     *         enabled workgroups. This property applies only to Spark enabled workgroups and IAM Identity Center
     *         enabled workgroups. The property is required for IAM Identity Center enabled workgroups.
     */
    public final String executionRole() {
        return executionRole;
    }

    /**
     * <p>
     * Specifies the KMS key that is used to encrypt the user's data stores in Athena. This setting does not apply to
     * Athena SQL workgroups.
     * </p>
     * 
     * @return Specifies the KMS key that is used to encrypt the user's data stores in Athena. This setting does not
     *         apply to Athena SQL workgroups.
     */
    public final CustomerContentEncryptionConfiguration customerContentEncryptionConfiguration() {
        return customerContentEncryptionConfiguration;
    }

    /**
     * <p>
     * Enforces a minimal level of encryption for the workgroup for query and calculation results that are written to
     * Amazon S3. When enabled, workgroup users can set encryption only to the minimum level set by the administrator or
     * higher when they submit queries.
     * </p>
     * <p>
     * The <code>EnforceWorkGroupConfiguration</code> setting takes precedence over the
     * <code>EnableMinimumEncryptionConfiguration</code> flag. This means that if
     * <code>EnforceWorkGroupConfiguration</code> is true, the <code>EnableMinimumEncryptionConfiguration</code> flag is
     * ignored, and the workgroup configuration for encryption is used.
     * </p>
     * 
     * @return Enforces a minimal level of encryption for the workgroup for query and calculation results that are
     *         written to Amazon S3. When enabled, workgroup users can set encryption only to the minimum level set by
     *         the administrator or higher when they submit queries.</p>
     *         <p>
     *         The <code>EnforceWorkGroupConfiguration</code> setting takes precedence over the
     *         <code>EnableMinimumEncryptionConfiguration</code> flag. This means that if
     *         <code>EnforceWorkGroupConfiguration</code> is true, the <code>EnableMinimumEncryptionConfiguration</code>
     *         flag is ignored, and the workgroup configuration for encryption is used.
     */
    public final Boolean enableMinimumEncryptionConfiguration() {
        return enableMinimumEncryptionConfiguration;
    }

    /**
     * <p>
     * Specifies whether the workgroup is IAM Identity Center supported.
     * </p>
     * 
     * @return Specifies whether the workgroup is IAM Identity Center supported.
     */
    public final IdentityCenterConfiguration identityCenterConfiguration() {
        return identityCenterConfiguration;
    }

    /**
     * <p>
     * Specifies whether Amazon S3 access grants are enabled for query results.
     * </p>
     * 
     * @return Specifies whether Amazon S3 access grants are enabled for query results.
     */
    public final QueryResultsS3AccessGrantsConfiguration queryResultsS3AccessGrantsConfiguration() {
        return queryResultsS3AccessGrantsConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resultConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(enforceWorkGroupConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(publishCloudWatchMetricsEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(bytesScannedCutoffPerQuery());
        hashCode = 31 * hashCode + Objects.hashCode(requesterPaysEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(additionalConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(executionRole());
        hashCode = 31 * hashCode + Objects.hashCode(customerContentEncryptionConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(enableMinimumEncryptionConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(identityCenterConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(queryResultsS3AccessGrantsConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WorkGroupConfiguration)) {
            return false;
        }
        WorkGroupConfiguration other = (WorkGroupConfiguration) obj;
        return Objects.equals(resultConfiguration(), other.resultConfiguration())
                && Objects.equals(enforceWorkGroupConfiguration(), other.enforceWorkGroupConfiguration())
                && Objects.equals(publishCloudWatchMetricsEnabled(), other.publishCloudWatchMetricsEnabled())
                && Objects.equals(bytesScannedCutoffPerQuery(), other.bytesScannedCutoffPerQuery())
                && Objects.equals(requesterPaysEnabled(), other.requesterPaysEnabled())
                && Objects.equals(engineVersion(), other.engineVersion())
                && Objects.equals(additionalConfiguration(), other.additionalConfiguration())
                && Objects.equals(executionRole(), other.executionRole())
                && Objects.equals(customerContentEncryptionConfiguration(), other.customerContentEncryptionConfiguration())
                && Objects.equals(enableMinimumEncryptionConfiguration(), other.enableMinimumEncryptionConfiguration())
                && Objects.equals(identityCenterConfiguration(), other.identityCenterConfiguration())
                && Objects.equals(queryResultsS3AccessGrantsConfiguration(), other.queryResultsS3AccessGrantsConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("WorkGroupConfiguration").add("ResultConfiguration", resultConfiguration())
                .add("EnforceWorkGroupConfiguration", enforceWorkGroupConfiguration())
                .add("PublishCloudWatchMetricsEnabled", publishCloudWatchMetricsEnabled())
                .add("BytesScannedCutoffPerQuery", bytesScannedCutoffPerQuery())
                .add("RequesterPaysEnabled", requesterPaysEnabled()).add("EngineVersion", engineVersion())
                .add("AdditionalConfiguration", additionalConfiguration()).add("ExecutionRole", executionRole())
                .add("CustomerContentEncryptionConfiguration", customerContentEncryptionConfiguration())
                .add("EnableMinimumEncryptionConfiguration", enableMinimumEncryptionConfiguration())
                .add("IdentityCenterConfiguration", identityCenterConfiguration())
                .add("QueryResultsS3AccessGrantsConfiguration", queryResultsS3AccessGrantsConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ResultConfiguration":
            return Optional.ofNullable(clazz.cast(resultConfiguration()));
        case "EnforceWorkGroupConfiguration":
            return Optional.ofNullable(clazz.cast(enforceWorkGroupConfiguration()));
        case "PublishCloudWatchMetricsEnabled":
            return Optional.ofNullable(clazz.cast(publishCloudWatchMetricsEnabled()));
        case "BytesScannedCutoffPerQuery":
            return Optional.ofNullable(clazz.cast(bytesScannedCutoffPerQuery()));
        case "RequesterPaysEnabled":
            return Optional.ofNullable(clazz.cast(requesterPaysEnabled()));
        case "EngineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        case "AdditionalConfiguration":
            return Optional.ofNullable(clazz.cast(additionalConfiguration()));
        case "ExecutionRole":
            return Optional.ofNullable(clazz.cast(executionRole()));
        case "CustomerContentEncryptionConfiguration":
            return Optional.ofNullable(clazz.cast(customerContentEncryptionConfiguration()));
        case "EnableMinimumEncryptionConfiguration":
            return Optional.ofNullable(clazz.cast(enableMinimumEncryptionConfiguration()));
        case "IdentityCenterConfiguration":
            return Optional.ofNullable(clazz.cast(identityCenterConfiguration()));
        case "QueryResultsS3AccessGrantsConfiguration":
            return Optional.ofNullable(clazz.cast(queryResultsS3AccessGrantsConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<WorkGroupConfiguration, T> g) {
        return obj -> g.apply((WorkGroupConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WorkGroupConfiguration> {
        /**
         * <p>
         * The configuration for the workgroup, which includes the location in Amazon S3 where query and calculation
         * results are stored and the encryption option, if any, used for query and calculation results. To run the
         * query, you must specify the query results location using one of the ways: either in the workgroup using this
         * setting, or for individual queries (client-side), using <a>ResultConfiguration$OutputLocation</a>. If none of
         * them is set, Athena issues an error that no output location is provided.
         * </p>
         * 
         * @param resultConfiguration
         *        The configuration for the workgroup, which includes the location in Amazon S3 where query and
         *        calculation results are stored and the encryption option, if any, used for query and calculation
         *        results. To run the query, you must specify the query results location using one of the ways: either
         *        in the workgroup using this setting, or for individual queries (client-side), using
         *        <a>ResultConfiguration$OutputLocation</a>. If none of them is set, Athena issues an error that no
         *        output location is provided.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resultConfiguration(ResultConfiguration resultConfiguration);

        /**
         * <p>
         * The configuration for the workgroup, which includes the location in Amazon S3 where query and calculation
         * results are stored and the encryption option, if any, used for query and calculation results. To run the
         * query, you must specify the query results location using one of the ways: either in the workgroup using this
         * setting, or for individual queries (client-side), using <a>ResultConfiguration$OutputLocation</a>. If none of
         * them is set, Athena issues an error that no output location is provided.
         * </p>
         * This is a convenience method that creates an instance of the {@link ResultConfiguration.Builder} avoiding the
         * need to create one manually via {@link ResultConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResultConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #resultConfiguration(ResultConfiguration)}.
         * 
         * @param resultConfiguration
         *        a consumer that will call methods on {@link ResultConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resultConfiguration(ResultConfiguration)
         */
        default Builder resultConfiguration(Consumer<ResultConfiguration.Builder> resultConfiguration) {
            return resultConfiguration(ResultConfiguration.builder().applyMutation(resultConfiguration).build());
        }

        /**
         * <p>
         * If set to "true", the settings for the workgroup override client-side settings. If set to "false",
         * client-side settings are used. For more information, see <a
         * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
         * Override Client-Side Settings</a>.
         * </p>
         * 
         * @param enforceWorkGroupConfiguration
         *        If set to "true", the settings for the workgroup override client-side settings. If set to "false",
         *        client-side settings are used. For more information, see <a
         *        href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup
         *        Settings Override Client-Side Settings</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enforceWorkGroupConfiguration(Boolean enforceWorkGroupConfiguration);

        /**
         * <p>
         * Indicates that the Amazon CloudWatch metrics are enabled for the workgroup.
         * </p>
         * 
         * @param publishCloudWatchMetricsEnabled
         *        Indicates that the Amazon CloudWatch metrics are enabled for the workgroup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publishCloudWatchMetricsEnabled(Boolean publishCloudWatchMetricsEnabled);

        /**
         * <p>
         * The upper data usage limit (cutoff) for the amount of bytes a single query in a workgroup is allowed to scan.
         * </p>
         * 
         * @param bytesScannedCutoffPerQuery
         *        The upper data usage limit (cutoff) for the amount of bytes a single query in a workgroup is allowed
         *        to scan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bytesScannedCutoffPerQuery(Long bytesScannedCutoffPerQuery);

        /**
         * <p>
         * If set to <code>true</code>, allows members assigned to a workgroup to reference Amazon S3 Requester Pays
         * buckets in queries. If set to <code>false</code>, workgroup members cannot query data from Requester Pays
         * buckets, and queries that retrieve data from Requester Pays buckets cause an error. The default is
         * <code>false</code>. For more information about Requester Pays buckets, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/RequesterPaysBuckets.html">Requester Pays Buckets</a>
         * in the <i>Amazon Simple Storage Service Developer Guide</i>.
         * </p>
         * 
         * @param requesterPaysEnabled
         *        If set to <code>true</code>, allows members assigned to a workgroup to reference Amazon S3 Requester
         *        Pays buckets in queries. If set to <code>false</code>, workgroup members cannot query data from
         *        Requester Pays buckets, and queries that retrieve data from Requester Pays buckets cause an error. The
         *        default is <code>false</code>. For more information about Requester Pays buckets, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/RequesterPaysBuckets.html">Requester Pays
         *        Buckets</a> in the <i>Amazon Simple Storage Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requesterPaysEnabled(Boolean requesterPaysEnabled);

        /**
         * <p>
         * The engine version that all queries running on the workgroup use. Queries on the
         * <code>AmazonAthenaPreviewFunctionality</code> workgroup run on the preview engine regardless of this setting.
         * </p>
         * 
         * @param engineVersion
         *        The engine version that all queries running on the workgroup use. Queries on the
         *        <code>AmazonAthenaPreviewFunctionality</code> workgroup run on the preview engine regardless of this
         *        setting.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(EngineVersion engineVersion);

        /**
         * <p>
         * The engine version that all queries running on the workgroup use. Queries on the
         * <code>AmazonAthenaPreviewFunctionality</code> workgroup run on the preview engine regardless of this setting.
         * </p>
         * This is a convenience method that creates an instance of the {@link EngineVersion.Builder} avoiding the need
         * to create one manually via {@link EngineVersion#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EngineVersion.Builder#build()} is called immediately and its
         * result is passed to {@link #engineVersion(EngineVersion)}.
         * 
         * @param engineVersion
         *        a consumer that will call methods on {@link EngineVersion.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #engineVersion(EngineVersion)
         */
        default Builder engineVersion(Consumer<EngineVersion.Builder> engineVersion) {
            return engineVersion(EngineVersion.builder().applyMutation(engineVersion).build());
        }

        /**
         * <p>
         * Specifies a user defined JSON string that is passed to the notebook engine.
         * </p>
         * 
         * @param additionalConfiguration
         *        Specifies a user defined JSON string that is passed to the notebook engine.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalConfiguration(String additionalConfiguration);

        /**
         * <p>
         * The ARN of the execution role used to access user resources for Spark sessions and IAM Identity Center
         * enabled workgroups. This property applies only to Spark enabled workgroups and IAM Identity Center enabled
         * workgroups. The property is required for IAM Identity Center enabled workgroups.
         * </p>
         * 
         * @param executionRole
         *        The ARN of the execution role used to access user resources for Spark sessions and IAM Identity Center
         *        enabled workgroups. This property applies only to Spark enabled workgroups and IAM Identity Center
         *        enabled workgroups. The property is required for IAM Identity Center enabled workgroups.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionRole(String executionRole);

        /**
         * <p>
         * Specifies the KMS key that is used to encrypt the user's data stores in Athena. This setting does not apply
         * to Athena SQL workgroups.
         * </p>
         * 
         * @param customerContentEncryptionConfiguration
         *        Specifies the KMS key that is used to encrypt the user's data stores in Athena. This setting does not
         *        apply to Athena SQL workgroups.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerContentEncryptionConfiguration(
                CustomerContentEncryptionConfiguration customerContentEncryptionConfiguration);

        /**
         * <p>
         * Specifies the KMS key that is used to encrypt the user's data stores in Athena. This setting does not apply
         * to Athena SQL workgroups.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link CustomerContentEncryptionConfiguration.Builder} avoiding the need to create one manually via
         * {@link CustomerContentEncryptionConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CustomerContentEncryptionConfiguration.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #customerContentEncryptionConfiguration(CustomerContentEncryptionConfiguration)}.
         * 
         * @param customerContentEncryptionConfiguration
         *        a consumer that will call methods on {@link CustomerContentEncryptionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #customerContentEncryptionConfiguration(CustomerContentEncryptionConfiguration)
         */
        default Builder customerContentEncryptionConfiguration(
                Consumer<CustomerContentEncryptionConfiguration.Builder> customerContentEncryptionConfiguration) {
            return customerContentEncryptionConfiguration(CustomerContentEncryptionConfiguration.builder()
                    .applyMutation(customerContentEncryptionConfiguration).build());
        }

        /**
         * <p>
         * Enforces a minimal level of encryption for the workgroup for query and calculation results that are written
         * to Amazon S3. When enabled, workgroup users can set encryption only to the minimum level set by the
         * administrator or higher when they submit queries.
         * </p>
         * <p>
         * The <code>EnforceWorkGroupConfiguration</code> setting takes precedence over the
         * <code>EnableMinimumEncryptionConfiguration</code> flag. This means that if
         * <code>EnforceWorkGroupConfiguration</code> is true, the <code>EnableMinimumEncryptionConfiguration</code>
         * flag is ignored, and the workgroup configuration for encryption is used.
         * </p>
         * 
         * @param enableMinimumEncryptionConfiguration
         *        Enforces a minimal level of encryption for the workgroup for query and calculation results that are
         *        written to Amazon S3. When enabled, workgroup users can set encryption only to the minimum level set
         *        by the administrator or higher when they submit queries.</p>
         *        <p>
         *        The <code>EnforceWorkGroupConfiguration</code> setting takes precedence over the
         *        <code>EnableMinimumEncryptionConfiguration</code> flag. This means that if
         *        <code>EnforceWorkGroupConfiguration</code> is true, the
         *        <code>EnableMinimumEncryptionConfiguration</code> flag is ignored, and the workgroup configuration for
         *        encryption is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableMinimumEncryptionConfiguration(Boolean enableMinimumEncryptionConfiguration);

        /**
         * <p>
         * Specifies whether the workgroup is IAM Identity Center supported.
         * </p>
         * 
         * @param identityCenterConfiguration
         *        Specifies whether the workgroup is IAM Identity Center supported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identityCenterConfiguration(IdentityCenterConfiguration identityCenterConfiguration);

        /**
         * <p>
         * Specifies whether the workgroup is IAM Identity Center supported.
         * </p>
         * This is a convenience method that creates an instance of the {@link IdentityCenterConfiguration.Builder}
         * avoiding the need to create one manually via {@link IdentityCenterConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link IdentityCenterConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #identityCenterConfiguration(IdentityCenterConfiguration)}.
         * 
         * @param identityCenterConfiguration
         *        a consumer that will call methods on {@link IdentityCenterConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #identityCenterConfiguration(IdentityCenterConfiguration)
         */
        default Builder identityCenterConfiguration(Consumer<IdentityCenterConfiguration.Builder> identityCenterConfiguration) {
            return identityCenterConfiguration(IdentityCenterConfiguration.builder().applyMutation(identityCenterConfiguration)
                    .build());
        }

        /**
         * <p>
         * Specifies whether Amazon S3 access grants are enabled for query results.
         * </p>
         * 
         * @param queryResultsS3AccessGrantsConfiguration
         *        Specifies whether Amazon S3 access grants are enabled for query results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryResultsS3AccessGrantsConfiguration(
                QueryResultsS3AccessGrantsConfiguration queryResultsS3AccessGrantsConfiguration);

        /**
         * <p>
         * Specifies whether Amazon S3 access grants are enabled for query results.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link QueryResultsS3AccessGrantsConfiguration.Builder} avoiding the need to create one manually via
         * {@link QueryResultsS3AccessGrantsConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link QueryResultsS3AccessGrantsConfiguration.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #queryResultsS3AccessGrantsConfiguration(QueryResultsS3AccessGrantsConfiguration)}.
         * 
         * @param queryResultsS3AccessGrantsConfiguration
         *        a consumer that will call methods on {@link QueryResultsS3AccessGrantsConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #queryResultsS3AccessGrantsConfiguration(QueryResultsS3AccessGrantsConfiguration)
         */
        default Builder queryResultsS3AccessGrantsConfiguration(
                Consumer<QueryResultsS3AccessGrantsConfiguration.Builder> queryResultsS3AccessGrantsConfiguration) {
            return queryResultsS3AccessGrantsConfiguration(QueryResultsS3AccessGrantsConfiguration.builder()
                    .applyMutation(queryResultsS3AccessGrantsConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private ResultConfiguration resultConfiguration;

        private Boolean enforceWorkGroupConfiguration;

        private Boolean publishCloudWatchMetricsEnabled;

        private Long bytesScannedCutoffPerQuery;

        private Boolean requesterPaysEnabled;

        private EngineVersion engineVersion;

        private String additionalConfiguration;

        private String executionRole;

        private CustomerContentEncryptionConfiguration customerContentEncryptionConfiguration;

        private Boolean enableMinimumEncryptionConfiguration;

        private IdentityCenterConfiguration identityCenterConfiguration;

        private QueryResultsS3AccessGrantsConfiguration queryResultsS3AccessGrantsConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(WorkGroupConfiguration model) {
            resultConfiguration(model.resultConfiguration);
            enforceWorkGroupConfiguration(model.enforceWorkGroupConfiguration);
            publishCloudWatchMetricsEnabled(model.publishCloudWatchMetricsEnabled);
            bytesScannedCutoffPerQuery(model.bytesScannedCutoffPerQuery);
            requesterPaysEnabled(model.requesterPaysEnabled);
            engineVersion(model.engineVersion);
            additionalConfiguration(model.additionalConfiguration);
            executionRole(model.executionRole);
            customerContentEncryptionConfiguration(model.customerContentEncryptionConfiguration);
            enableMinimumEncryptionConfiguration(model.enableMinimumEncryptionConfiguration);
            identityCenterConfiguration(model.identityCenterConfiguration);
            queryResultsS3AccessGrantsConfiguration(model.queryResultsS3AccessGrantsConfiguration);
        }

        public final ResultConfiguration.Builder getResultConfiguration() {
            return resultConfiguration != null ? resultConfiguration.toBuilder() : null;
        }

        public final void setResultConfiguration(ResultConfiguration.BuilderImpl resultConfiguration) {
            this.resultConfiguration = resultConfiguration != null ? resultConfiguration.build() : null;
        }

        @Override
        public final Builder resultConfiguration(ResultConfiguration resultConfiguration) {
            this.resultConfiguration = resultConfiguration;
            return this;
        }

        public final Boolean getEnforceWorkGroupConfiguration() {
            return enforceWorkGroupConfiguration;
        }

        public final void setEnforceWorkGroupConfiguration(Boolean enforceWorkGroupConfiguration) {
            this.enforceWorkGroupConfiguration = enforceWorkGroupConfiguration;
        }

        @Override
        public final Builder enforceWorkGroupConfiguration(Boolean enforceWorkGroupConfiguration) {
            this.enforceWorkGroupConfiguration = enforceWorkGroupConfiguration;
            return this;
        }

        public final Boolean getPublishCloudWatchMetricsEnabled() {
            return publishCloudWatchMetricsEnabled;
        }

        public final void setPublishCloudWatchMetricsEnabled(Boolean publishCloudWatchMetricsEnabled) {
            this.publishCloudWatchMetricsEnabled = publishCloudWatchMetricsEnabled;
        }

        @Override
        public final Builder publishCloudWatchMetricsEnabled(Boolean publishCloudWatchMetricsEnabled) {
            this.publishCloudWatchMetricsEnabled = publishCloudWatchMetricsEnabled;
            return this;
        }

        public final Long getBytesScannedCutoffPerQuery() {
            return bytesScannedCutoffPerQuery;
        }

        public final void setBytesScannedCutoffPerQuery(Long bytesScannedCutoffPerQuery) {
            this.bytesScannedCutoffPerQuery = bytesScannedCutoffPerQuery;
        }

        @Override
        public final Builder bytesScannedCutoffPerQuery(Long bytesScannedCutoffPerQuery) {
            this.bytesScannedCutoffPerQuery = bytesScannedCutoffPerQuery;
            return this;
        }

        public final Boolean getRequesterPaysEnabled() {
            return requesterPaysEnabled;
        }

        public final void setRequesterPaysEnabled(Boolean requesterPaysEnabled) {
            this.requesterPaysEnabled = requesterPaysEnabled;
        }

        @Override
        public final Builder requesterPaysEnabled(Boolean requesterPaysEnabled) {
            this.requesterPaysEnabled = requesterPaysEnabled;
            return this;
        }

        public final EngineVersion.Builder getEngineVersion() {
            return engineVersion != null ? engineVersion.toBuilder() : null;
        }

        public final void setEngineVersion(EngineVersion.BuilderImpl engineVersion) {
            this.engineVersion = engineVersion != null ? engineVersion.build() : null;
        }

        @Override
        public final Builder engineVersion(EngineVersion engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        public final String getAdditionalConfiguration() {
            return additionalConfiguration;
        }

        public final void setAdditionalConfiguration(String additionalConfiguration) {
            this.additionalConfiguration = additionalConfiguration;
        }

        @Override
        public final Builder additionalConfiguration(String additionalConfiguration) {
            this.additionalConfiguration = additionalConfiguration;
            return this;
        }

        public final String getExecutionRole() {
            return executionRole;
        }

        public final void setExecutionRole(String executionRole) {
            this.executionRole = executionRole;
        }

        @Override
        public final Builder executionRole(String executionRole) {
            this.executionRole = executionRole;
            return this;
        }

        public final CustomerContentEncryptionConfiguration.Builder getCustomerContentEncryptionConfiguration() {
            return customerContentEncryptionConfiguration != null ? customerContentEncryptionConfiguration.toBuilder() : null;
        }

        public final void setCustomerContentEncryptionConfiguration(
                CustomerContentEncryptionConfiguration.BuilderImpl customerContentEncryptionConfiguration) {
            this.customerContentEncryptionConfiguration = customerContentEncryptionConfiguration != null ? customerContentEncryptionConfiguration
                    .build() : null;
        }

        @Override
        public final Builder customerContentEncryptionConfiguration(
                CustomerContentEncryptionConfiguration customerContentEncryptionConfiguration) {
            this.customerContentEncryptionConfiguration = customerContentEncryptionConfiguration;
            return this;
        }

        public final Boolean getEnableMinimumEncryptionConfiguration() {
            return enableMinimumEncryptionConfiguration;
        }

        public final void setEnableMinimumEncryptionConfiguration(Boolean enableMinimumEncryptionConfiguration) {
            this.enableMinimumEncryptionConfiguration = enableMinimumEncryptionConfiguration;
        }

        @Override
        public final Builder enableMinimumEncryptionConfiguration(Boolean enableMinimumEncryptionConfiguration) {
            this.enableMinimumEncryptionConfiguration = enableMinimumEncryptionConfiguration;
            return this;
        }

        public final IdentityCenterConfiguration.Builder getIdentityCenterConfiguration() {
            return identityCenterConfiguration != null ? identityCenterConfiguration.toBuilder() : null;
        }

        public final void setIdentityCenterConfiguration(IdentityCenterConfiguration.BuilderImpl identityCenterConfiguration) {
            this.identityCenterConfiguration = identityCenterConfiguration != null ? identityCenterConfiguration.build() : null;
        }

        @Override
        public final Builder identityCenterConfiguration(IdentityCenterConfiguration identityCenterConfiguration) {
            this.identityCenterConfiguration = identityCenterConfiguration;
            return this;
        }

        public final QueryResultsS3AccessGrantsConfiguration.Builder getQueryResultsS3AccessGrantsConfiguration() {
            return queryResultsS3AccessGrantsConfiguration != null ? queryResultsS3AccessGrantsConfiguration.toBuilder() : null;
        }

        public final void setQueryResultsS3AccessGrantsConfiguration(
                QueryResultsS3AccessGrantsConfiguration.BuilderImpl queryResultsS3AccessGrantsConfiguration) {
            this.queryResultsS3AccessGrantsConfiguration = queryResultsS3AccessGrantsConfiguration != null ? queryResultsS3AccessGrantsConfiguration
                    .build() : null;
        }

        @Override
        public final Builder queryResultsS3AccessGrantsConfiguration(
                QueryResultsS3AccessGrantsConfiguration queryResultsS3AccessGrantsConfiguration) {
            this.queryResultsS3AccessGrantsConfiguration = queryResultsS3AccessGrantsConfiguration;
            return this;
        }

        @Override
        public WorkGroupConfiguration build() {
            return new WorkGroupConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
