/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.athena.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the status of a notebook calculation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CalculationStatus implements SdkPojo, Serializable,
        ToCopyableBuilder<CalculationStatus.Builder, CalculationStatus> {
    private static final SdkField<Instant> SUBMISSION_DATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("SubmissionDateTime").getter(getter(CalculationStatus::submissionDateTime))
            .setter(setter(Builder::submissionDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubmissionDateTime").build())
            .build();

    private static final SdkField<Instant> COMPLETION_DATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CompletionDateTime").getter(getter(CalculationStatus::completionDateTime))
            .setter(setter(Builder::completionDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletionDateTime").build())
            .build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(CalculationStatus::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<String> STATE_CHANGE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StateChangeReason").getter(getter(CalculationStatus::stateChangeReason))
            .setter(setter(Builder::stateChangeReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateChangeReason").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SUBMISSION_DATE_TIME_FIELD,
            COMPLETION_DATE_TIME_FIELD, STATE_FIELD, STATE_CHANGE_REASON_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Instant submissionDateTime;

    private final Instant completionDateTime;

    private final String state;

    private final String stateChangeReason;

    private CalculationStatus(BuilderImpl builder) {
        this.submissionDateTime = builder.submissionDateTime;
        this.completionDateTime = builder.completionDateTime;
        this.state = builder.state;
        this.stateChangeReason = builder.stateChangeReason;
    }

    /**
     * <p>
     * The date and time the calculation was submitted for processing.
     * </p>
     * 
     * @return The date and time the calculation was submitted for processing.
     */
    public final Instant submissionDateTime() {
        return submissionDateTime;
    }

    /**
     * <p>
     * The date and time the calculation completed processing.
     * </p>
     * 
     * @return The date and time the calculation completed processing.
     */
    public final Instant completionDateTime() {
        return completionDateTime;
    }

    /**
     * <p>
     * The state of the calculation execution. A description of each state follows.
     * </p>
     * <p>
     * <code>CREATING</code> - The calculation is in the process of being created.
     * </p>
     * <p>
     * <code>CREATED</code> - The calculation has been created and is ready to run.
     * </p>
     * <p>
     * <code>QUEUED</code> - The calculation has been queued for processing.
     * </p>
     * <p>
     * <code>RUNNING</code> - The calculation is running.
     * </p>
     * <p>
     * <code>CANCELING</code> - A request to cancel the calculation has been received and the system is working to stop
     * it.
     * </p>
     * <p>
     * <code>CANCELED</code> - The calculation is no longer running as the result of a cancel request.
     * </p>
     * <p>
     * <code>COMPLETED</code> - The calculation has completed without error.
     * </p>
     * <p>
     * <code>FAILED</code> - The calculation failed and is no longer running.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link CalculationExecutionState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the calculation execution. A description of each state follows.</p>
     *         <p>
     *         <code>CREATING</code> - The calculation is in the process of being created.
     *         </p>
     *         <p>
     *         <code>CREATED</code> - The calculation has been created and is ready to run.
     *         </p>
     *         <p>
     *         <code>QUEUED</code> - The calculation has been queued for processing.
     *         </p>
     *         <p>
     *         <code>RUNNING</code> - The calculation is running.
     *         </p>
     *         <p>
     *         <code>CANCELING</code> - A request to cancel the calculation has been received and the system is working
     *         to stop it.
     *         </p>
     *         <p>
     *         <code>CANCELED</code> - The calculation is no longer running as the result of a cancel request.
     *         </p>
     *         <p>
     *         <code>COMPLETED</code> - The calculation has completed without error.
     *         </p>
     *         <p>
     *         <code>FAILED</code> - The calculation failed and is no longer running.
     * @see CalculationExecutionState
     */
    public final CalculationExecutionState state() {
        return CalculationExecutionState.fromValue(state);
    }

    /**
     * <p>
     * The state of the calculation execution. A description of each state follows.
     * </p>
     * <p>
     * <code>CREATING</code> - The calculation is in the process of being created.
     * </p>
     * <p>
     * <code>CREATED</code> - The calculation has been created and is ready to run.
     * </p>
     * <p>
     * <code>QUEUED</code> - The calculation has been queued for processing.
     * </p>
     * <p>
     * <code>RUNNING</code> - The calculation is running.
     * </p>
     * <p>
     * <code>CANCELING</code> - A request to cancel the calculation has been received and the system is working to stop
     * it.
     * </p>
     * <p>
     * <code>CANCELED</code> - The calculation is no longer running as the result of a cancel request.
     * </p>
     * <p>
     * <code>COMPLETED</code> - The calculation has completed without error.
     * </p>
     * <p>
     * <code>FAILED</code> - The calculation failed and is no longer running.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link CalculationExecutionState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the calculation execution. A description of each state follows.</p>
     *         <p>
     *         <code>CREATING</code> - The calculation is in the process of being created.
     *         </p>
     *         <p>
     *         <code>CREATED</code> - The calculation has been created and is ready to run.
     *         </p>
     *         <p>
     *         <code>QUEUED</code> - The calculation has been queued for processing.
     *         </p>
     *         <p>
     *         <code>RUNNING</code> - The calculation is running.
     *         </p>
     *         <p>
     *         <code>CANCELING</code> - A request to cancel the calculation has been received and the system is working
     *         to stop it.
     *         </p>
     *         <p>
     *         <code>CANCELED</code> - The calculation is no longer running as the result of a cancel request.
     *         </p>
     *         <p>
     *         <code>COMPLETED</code> - The calculation has completed without error.
     *         </p>
     *         <p>
     *         <code>FAILED</code> - The calculation failed and is no longer running.
     * @see CalculationExecutionState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The reason for the calculation state change (for example, the calculation was canceled because the session was
     * terminated).
     * </p>
     * 
     * @return The reason for the calculation state change (for example, the calculation was canceled because the
     *         session was terminated).
     */
    public final String stateChangeReason() {
        return stateChangeReason;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(submissionDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(completionDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateChangeReason());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CalculationStatus)) {
            return false;
        }
        CalculationStatus other = (CalculationStatus) obj;
        return Objects.equals(submissionDateTime(), other.submissionDateTime())
                && Objects.equals(completionDateTime(), other.completionDateTime())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(stateChangeReason(), other.stateChangeReason());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CalculationStatus").add("SubmissionDateTime", submissionDateTime())
                .add("CompletionDateTime", completionDateTime()).add("State", stateAsString())
                .add("StateChangeReason", stateChangeReason()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SubmissionDateTime":
            return Optional.ofNullable(clazz.cast(submissionDateTime()));
        case "CompletionDateTime":
            return Optional.ofNullable(clazz.cast(completionDateTime()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "StateChangeReason":
            return Optional.ofNullable(clazz.cast(stateChangeReason()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("SubmissionDateTime", SUBMISSION_DATE_TIME_FIELD);
        map.put("CompletionDateTime", COMPLETION_DATE_TIME_FIELD);
        map.put("State", STATE_FIELD);
        map.put("StateChangeReason", STATE_CHANGE_REASON_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CalculationStatus, T> g) {
        return obj -> g.apply((CalculationStatus) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CalculationStatus> {
        /**
         * <p>
         * The date and time the calculation was submitted for processing.
         * </p>
         * 
         * @param submissionDateTime
         *        The date and time the calculation was submitted for processing.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder submissionDateTime(Instant submissionDateTime);

        /**
         * <p>
         * The date and time the calculation completed processing.
         * </p>
         * 
         * @param completionDateTime
         *        The date and time the calculation completed processing.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completionDateTime(Instant completionDateTime);

        /**
         * <p>
         * The state of the calculation execution. A description of each state follows.
         * </p>
         * <p>
         * <code>CREATING</code> - The calculation is in the process of being created.
         * </p>
         * <p>
         * <code>CREATED</code> - The calculation has been created and is ready to run.
         * </p>
         * <p>
         * <code>QUEUED</code> - The calculation has been queued for processing.
         * </p>
         * <p>
         * <code>RUNNING</code> - The calculation is running.
         * </p>
         * <p>
         * <code>CANCELING</code> - A request to cancel the calculation has been received and the system is working to
         * stop it.
         * </p>
         * <p>
         * <code>CANCELED</code> - The calculation is no longer running as the result of a cancel request.
         * </p>
         * <p>
         * <code>COMPLETED</code> - The calculation has completed without error.
         * </p>
         * <p>
         * <code>FAILED</code> - The calculation failed and is no longer running.
         * </p>
         * 
         * @param state
         *        The state of the calculation execution. A description of each state follows.</p>
         *        <p>
         *        <code>CREATING</code> - The calculation is in the process of being created.
         *        </p>
         *        <p>
         *        <code>CREATED</code> - The calculation has been created and is ready to run.
         *        </p>
         *        <p>
         *        <code>QUEUED</code> - The calculation has been queued for processing.
         *        </p>
         *        <p>
         *        <code>RUNNING</code> - The calculation is running.
         *        </p>
         *        <p>
         *        <code>CANCELING</code> - A request to cancel the calculation has been received and the system is
         *        working to stop it.
         *        </p>
         *        <p>
         *        <code>CANCELED</code> - The calculation is no longer running as the result of a cancel request.
         *        </p>
         *        <p>
         *        <code>COMPLETED</code> - The calculation has completed without error.
         *        </p>
         *        <p>
         *        <code>FAILED</code> - The calculation failed and is no longer running.
         * @see CalculationExecutionState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CalculationExecutionState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the calculation execution. A description of each state follows.
         * </p>
         * <p>
         * <code>CREATING</code> - The calculation is in the process of being created.
         * </p>
         * <p>
         * <code>CREATED</code> - The calculation has been created and is ready to run.
         * </p>
         * <p>
         * <code>QUEUED</code> - The calculation has been queued for processing.
         * </p>
         * <p>
         * <code>RUNNING</code> - The calculation is running.
         * </p>
         * <p>
         * <code>CANCELING</code> - A request to cancel the calculation has been received and the system is working to
         * stop it.
         * </p>
         * <p>
         * <code>CANCELED</code> - The calculation is no longer running as the result of a cancel request.
         * </p>
         * <p>
         * <code>COMPLETED</code> - The calculation has completed without error.
         * </p>
         * <p>
         * <code>FAILED</code> - The calculation failed and is no longer running.
         * </p>
         * 
         * @param state
         *        The state of the calculation execution. A description of each state follows.</p>
         *        <p>
         *        <code>CREATING</code> - The calculation is in the process of being created.
         *        </p>
         *        <p>
         *        <code>CREATED</code> - The calculation has been created and is ready to run.
         *        </p>
         *        <p>
         *        <code>QUEUED</code> - The calculation has been queued for processing.
         *        </p>
         *        <p>
         *        <code>RUNNING</code> - The calculation is running.
         *        </p>
         *        <p>
         *        <code>CANCELING</code> - A request to cancel the calculation has been received and the system is
         *        working to stop it.
         *        </p>
         *        <p>
         *        <code>CANCELED</code> - The calculation is no longer running as the result of a cancel request.
         *        </p>
         *        <p>
         *        <code>COMPLETED</code> - The calculation has completed without error.
         *        </p>
         *        <p>
         *        <code>FAILED</code> - The calculation failed and is no longer running.
         * @see CalculationExecutionState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CalculationExecutionState
         */
        Builder state(CalculationExecutionState state);

        /**
         * <p>
         * The reason for the calculation state change (for example, the calculation was canceled because the session
         * was terminated).
         * </p>
         * 
         * @param stateChangeReason
         *        The reason for the calculation state change (for example, the calculation was canceled because the
         *        session was terminated).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateChangeReason(String stateChangeReason);
    }

    static final class BuilderImpl implements Builder {
        private Instant submissionDateTime;

        private Instant completionDateTime;

        private String state;

        private String stateChangeReason;

        private BuilderImpl() {
        }

        private BuilderImpl(CalculationStatus model) {
            submissionDateTime(model.submissionDateTime);
            completionDateTime(model.completionDateTime);
            state(model.state);
            stateChangeReason(model.stateChangeReason);
        }

        public final Instant getSubmissionDateTime() {
            return submissionDateTime;
        }

        public final void setSubmissionDateTime(Instant submissionDateTime) {
            this.submissionDateTime = submissionDateTime;
        }

        @Override
        public final Builder submissionDateTime(Instant submissionDateTime) {
            this.submissionDateTime = submissionDateTime;
            return this;
        }

        public final Instant getCompletionDateTime() {
            return completionDateTime;
        }

        public final void setCompletionDateTime(Instant completionDateTime) {
            this.completionDateTime = completionDateTime;
        }

        @Override
        public final Builder completionDateTime(Instant completionDateTime) {
            this.completionDateTime = completionDateTime;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(CalculationExecutionState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getStateChangeReason() {
            return stateChangeReason;
        }

        public final void setStateChangeReason(String stateChangeReason) {
            this.stateChangeReason = stateChangeReason;
        }

        @Override
        public final Builder stateChangeReason(String stateChangeReason) {
            this.stateChangeReason = stateChangeReason;
            return this;
        }

        @Override
        public CalculationStatus build() {
            return new CalculationStatus(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
