/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.athena.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The summary information for the data catalog, which includes its name and type.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DataCatalogSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<DataCatalogSummary.Builder, DataCatalogSummary> {
    private static final SdkField<String> CATALOG_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CatalogName").getter(getter(DataCatalogSummary::catalogName)).setter(setter(Builder::catalogName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CatalogName").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(DataCatalogSummary::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(DataCatalogSummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> CONNECTION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConnectionType").getter(getter(DataCatalogSummary::connectionTypeAsString))
            .setter(setter(Builder::connectionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionType").build()).build();

    private static final SdkField<String> ERROR_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Error")
            .getter(getter(DataCatalogSummary::error)).setter(setter(Builder::error))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Error").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CATALOG_NAME_FIELD,
            TYPE_FIELD, STATUS_FIELD, CONNECTION_TYPE_FIELD, ERROR_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String catalogName;

    private final String type;

    private final String status;

    private final String connectionType;

    private final String error;

    private DataCatalogSummary(BuilderImpl builder) {
        this.catalogName = builder.catalogName;
        this.type = builder.type;
        this.status = builder.status;
        this.connectionType = builder.connectionType;
        this.error = builder.error;
    }

    /**
     * <p>
     * The name of the data catalog. The catalog name is unique for the Amazon Web Services account and can use a
     * maximum of 127 alphanumeric, underscore, at sign, or hyphen characters. The remainder of the length constraint of
     * 256 is reserved for use by Athena.
     * </p>
     * 
     * @return The name of the data catalog. The catalog name is unique for the Amazon Web Services account and can use
     *         a maximum of 127 alphanumeric, underscore, at sign, or hyphen characters. The remainder of the length
     *         constraint of 256 is reserved for use by Athena.
     */
    public final String catalogName() {
        return catalogName;
    }

    /**
     * <p>
     * The data catalog type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DataCatalogType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The data catalog type.
     * @see DataCatalogType
     */
    public final DataCatalogType type() {
        return DataCatalogType.fromValue(type);
    }

    /**
     * <p>
     * The data catalog type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DataCatalogType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The data catalog type.
     * @see DataCatalogType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The status of the creation or deletion of the data catalog.
     * </p>
     * <ul>
     * <li>
     * <p>
     * The <code>LAMBDA</code>, <code>GLUE</code>, and <code>HIVE</code> data catalog types are created synchronously.
     * Their status is either <code>CREATE_COMPLETE</code> or <code>CREATE_FAILED</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>FEDERATED</code> data catalog type is created asynchronously.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Data catalog creation status:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATE_IN_PROGRESS</code>: Federated data catalog creation in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATE_COMPLETE</code>: Data catalog creation complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATE_FAILED</code>: Data catalog could not be created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATE_FAILED_CLEANUP_IN_PROGRESS</code>: Federated data catalog creation failed and is being removed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATE_FAILED_CLEANUP_COMPLETE</code>: Federated data catalog creation failed and was removed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATE_FAILED_CLEANUP_FAILED</code>: Federated data catalog creation failed but could not be removed.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Data catalog deletion status:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DELETE_IN_PROGRESS</code>: Federated data catalog deletion in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETE_COMPLETE</code>: Federated data catalog deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETE_FAILED</code>: Federated data catalog could not be deleted.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DataCatalogStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the creation or deletion of the data catalog.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The <code>LAMBDA</code>, <code>GLUE</code>, and <code>HIVE</code> data catalog types are created
     *         synchronously. Their status is either <code>CREATE_COMPLETE</code> or <code>CREATE_FAILED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>FEDERATED</code> data catalog type is created asynchronously.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Data catalog creation status:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATE_IN_PROGRESS</code>: Federated data catalog creation in progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATE_COMPLETE</code>: Data catalog creation complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATE_FAILED</code>: Data catalog could not be created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATE_FAILED_CLEANUP_IN_PROGRESS</code>: Federated data catalog creation failed and is being
     *         removed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATE_FAILED_CLEANUP_COMPLETE</code>: Federated data catalog creation failed and was removed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATE_FAILED_CLEANUP_FAILED</code>: Federated data catalog creation failed but could not be
     *         removed.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Data catalog deletion status:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DELETE_IN_PROGRESS</code>: Federated data catalog deletion in progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETE_COMPLETE</code>: Federated data catalog deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETE_FAILED</code>: Federated data catalog could not be deleted.
     *         </p>
     *         </li>
     * @see DataCatalogStatus
     */
    public final DataCatalogStatus status() {
        return DataCatalogStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the creation or deletion of the data catalog.
     * </p>
     * <ul>
     * <li>
     * <p>
     * The <code>LAMBDA</code>, <code>GLUE</code>, and <code>HIVE</code> data catalog types are created synchronously.
     * Their status is either <code>CREATE_COMPLETE</code> or <code>CREATE_FAILED</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>FEDERATED</code> data catalog type is created asynchronously.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Data catalog creation status:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATE_IN_PROGRESS</code>: Federated data catalog creation in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATE_COMPLETE</code>: Data catalog creation complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATE_FAILED</code>: Data catalog could not be created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATE_FAILED_CLEANUP_IN_PROGRESS</code>: Federated data catalog creation failed and is being removed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATE_FAILED_CLEANUP_COMPLETE</code>: Federated data catalog creation failed and was removed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATE_FAILED_CLEANUP_FAILED</code>: Federated data catalog creation failed but could not be removed.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Data catalog deletion status:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DELETE_IN_PROGRESS</code>: Federated data catalog deletion in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETE_COMPLETE</code>: Federated data catalog deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETE_FAILED</code>: Federated data catalog could not be deleted.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DataCatalogStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the creation or deletion of the data catalog.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The <code>LAMBDA</code>, <code>GLUE</code>, and <code>HIVE</code> data catalog types are created
     *         synchronously. Their status is either <code>CREATE_COMPLETE</code> or <code>CREATE_FAILED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>FEDERATED</code> data catalog type is created asynchronously.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Data catalog creation status:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATE_IN_PROGRESS</code>: Federated data catalog creation in progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATE_COMPLETE</code>: Data catalog creation complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATE_FAILED</code>: Data catalog could not be created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATE_FAILED_CLEANUP_IN_PROGRESS</code>: Federated data catalog creation failed and is being
     *         removed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATE_FAILED_CLEANUP_COMPLETE</code>: Federated data catalog creation failed and was removed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATE_FAILED_CLEANUP_FAILED</code>: Federated data catalog creation failed but could not be
     *         removed.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Data catalog deletion status:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DELETE_IN_PROGRESS</code>: Federated data catalog deletion in progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETE_COMPLETE</code>: Federated data catalog deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETE_FAILED</code>: Federated data catalog could not be deleted.
     *         </p>
     *         </li>
     * @see DataCatalogStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The type of connection for a <code>FEDERATED</code> data catalog (for example, <code>REDSHIFT</code>,
     * <code>MYSQL</code>, or <code>SQLSERVER</code>). For information about individual connectors, see <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/connectors-available.html">Available data source
     * connectors</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #connectionType}
     * will return {@link ConnectionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #connectionTypeAsString}.
     * </p>
     * 
     * @return The type of connection for a <code>FEDERATED</code> data catalog (for example, <code>REDSHIFT</code>,
     *         <code>MYSQL</code>, or <code>SQLSERVER</code>). For information about individual connectors, see <a
     *         href="https://docs.aws.amazon.com/athena/latest/ug/connectors-available.html">Available data source
     *         connectors</a>.
     * @see ConnectionType
     */
    public final ConnectionType connectionType() {
        return ConnectionType.fromValue(connectionType);
    }

    /**
     * <p>
     * The type of connection for a <code>FEDERATED</code> data catalog (for example, <code>REDSHIFT</code>,
     * <code>MYSQL</code>, or <code>SQLSERVER</code>). For information about individual connectors, see <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/connectors-available.html">Available data source
     * connectors</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #connectionType}
     * will return {@link ConnectionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #connectionTypeAsString}.
     * </p>
     * 
     * @return The type of connection for a <code>FEDERATED</code> data catalog (for example, <code>REDSHIFT</code>,
     *         <code>MYSQL</code>, or <code>SQLSERVER</code>). For information about individual connectors, see <a
     *         href="https://docs.aws.amazon.com/athena/latest/ug/connectors-available.html">Available data source
     *         connectors</a>.
     * @see ConnectionType
     */
    public final String connectionTypeAsString() {
        return connectionType;
    }

    /**
     * <p>
     * Text of the error that occurred during data catalog creation or deletion.
     * </p>
     * 
     * @return Text of the error that occurred during data catalog creation or deletion.
     */
    public final String error() {
        return error;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(catalogName());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(connectionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(error());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DataCatalogSummary)) {
            return false;
        }
        DataCatalogSummary other = (DataCatalogSummary) obj;
        return Objects.equals(catalogName(), other.catalogName()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(connectionTypeAsString(), other.connectionTypeAsString())
                && Objects.equals(error(), other.error());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DataCatalogSummary").add("CatalogName", catalogName()).add("Type", typeAsString())
                .add("Status", statusAsString()).add("ConnectionType", connectionTypeAsString()).add("Error", error()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CatalogName":
            return Optional.ofNullable(clazz.cast(catalogName()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "ConnectionType":
            return Optional.ofNullable(clazz.cast(connectionTypeAsString()));
        case "Error":
            return Optional.ofNullable(clazz.cast(error()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("CatalogName", CATALOG_NAME_FIELD);
        map.put("Type", TYPE_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("ConnectionType", CONNECTION_TYPE_FIELD);
        map.put("Error", ERROR_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DataCatalogSummary, T> g) {
        return obj -> g.apply((DataCatalogSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DataCatalogSummary> {
        /**
         * <p>
         * The name of the data catalog. The catalog name is unique for the Amazon Web Services account and can use a
         * maximum of 127 alphanumeric, underscore, at sign, or hyphen characters. The remainder of the length
         * constraint of 256 is reserved for use by Athena.
         * </p>
         * 
         * @param catalogName
         *        The name of the data catalog. The catalog name is unique for the Amazon Web Services account and can
         *        use a maximum of 127 alphanumeric, underscore, at sign, or hyphen characters. The remainder of the
         *        length constraint of 256 is reserved for use by Athena.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder catalogName(String catalogName);

        /**
         * <p>
         * The data catalog type.
         * </p>
         * 
         * @param type
         *        The data catalog type.
         * @see DataCatalogType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataCatalogType
         */
        Builder type(String type);

        /**
         * <p>
         * The data catalog type.
         * </p>
         * 
         * @param type
         *        The data catalog type.
         * @see DataCatalogType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataCatalogType
         */
        Builder type(DataCatalogType type);

        /**
         * <p>
         * The status of the creation or deletion of the data catalog.
         * </p>
         * <ul>
         * <li>
         * <p>
         * The <code>LAMBDA</code>, <code>GLUE</code>, and <code>HIVE</code> data catalog types are created
         * synchronously. Their status is either <code>CREATE_COMPLETE</code> or <code>CREATE_FAILED</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>FEDERATED</code> data catalog type is created asynchronously.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Data catalog creation status:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATE_IN_PROGRESS</code>: Federated data catalog creation in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATE_COMPLETE</code>: Data catalog creation complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATE_FAILED</code>: Data catalog could not be created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATE_FAILED_CLEANUP_IN_PROGRESS</code>: Federated data catalog creation failed and is being removed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATE_FAILED_CLEANUP_COMPLETE</code>: Federated data catalog creation failed and was removed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATE_FAILED_CLEANUP_FAILED</code>: Federated data catalog creation failed but could not be removed.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Data catalog deletion status:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DELETE_IN_PROGRESS</code>: Federated data catalog deletion in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETE_COMPLETE</code>: Federated data catalog deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETE_FAILED</code>: Federated data catalog could not be deleted.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the creation or deletion of the data catalog.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The <code>LAMBDA</code>, <code>GLUE</code>, and <code>HIVE</code> data catalog types are created
         *        synchronously. Their status is either <code>CREATE_COMPLETE</code> or <code>CREATE_FAILED</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>FEDERATED</code> data catalog type is created asynchronously.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Data catalog creation status:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATE_IN_PROGRESS</code>: Federated data catalog creation in progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATE_COMPLETE</code>: Data catalog creation complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATE_FAILED</code>: Data catalog could not be created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATE_FAILED_CLEANUP_IN_PROGRESS</code>: Federated data catalog creation failed and is being
         *        removed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATE_FAILED_CLEANUP_COMPLETE</code>: Federated data catalog creation failed and was removed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATE_FAILED_CLEANUP_FAILED</code>: Federated data catalog creation failed but could not be
         *        removed.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Data catalog deletion status:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DELETE_IN_PROGRESS</code>: Federated data catalog deletion in progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETE_COMPLETE</code>: Federated data catalog deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETE_FAILED</code>: Federated data catalog could not be deleted.
         *        </p>
         *        </li>
         * @see DataCatalogStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataCatalogStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the creation or deletion of the data catalog.
         * </p>
         * <ul>
         * <li>
         * <p>
         * The <code>LAMBDA</code>, <code>GLUE</code>, and <code>HIVE</code> data catalog types are created
         * synchronously. Their status is either <code>CREATE_COMPLETE</code> or <code>CREATE_FAILED</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>FEDERATED</code> data catalog type is created asynchronously.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Data catalog creation status:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATE_IN_PROGRESS</code>: Federated data catalog creation in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATE_COMPLETE</code>: Data catalog creation complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATE_FAILED</code>: Data catalog could not be created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATE_FAILED_CLEANUP_IN_PROGRESS</code>: Federated data catalog creation failed and is being removed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATE_FAILED_CLEANUP_COMPLETE</code>: Federated data catalog creation failed and was removed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATE_FAILED_CLEANUP_FAILED</code>: Federated data catalog creation failed but could not be removed.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Data catalog deletion status:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DELETE_IN_PROGRESS</code>: Federated data catalog deletion in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETE_COMPLETE</code>: Federated data catalog deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETE_FAILED</code>: Federated data catalog could not be deleted.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the creation or deletion of the data catalog.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The <code>LAMBDA</code>, <code>GLUE</code>, and <code>HIVE</code> data catalog types are created
         *        synchronously. Their status is either <code>CREATE_COMPLETE</code> or <code>CREATE_FAILED</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>FEDERATED</code> data catalog type is created asynchronously.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Data catalog creation status:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATE_IN_PROGRESS</code>: Federated data catalog creation in progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATE_COMPLETE</code>: Data catalog creation complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATE_FAILED</code>: Data catalog could not be created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATE_FAILED_CLEANUP_IN_PROGRESS</code>: Federated data catalog creation failed and is being
         *        removed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATE_FAILED_CLEANUP_COMPLETE</code>: Federated data catalog creation failed and was removed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATE_FAILED_CLEANUP_FAILED</code>: Federated data catalog creation failed but could not be
         *        removed.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Data catalog deletion status:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DELETE_IN_PROGRESS</code>: Federated data catalog deletion in progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETE_COMPLETE</code>: Federated data catalog deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETE_FAILED</code>: Federated data catalog could not be deleted.
         *        </p>
         *        </li>
         * @see DataCatalogStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataCatalogStatus
         */
        Builder status(DataCatalogStatus status);

        /**
         * <p>
         * The type of connection for a <code>FEDERATED</code> data catalog (for example, <code>REDSHIFT</code>,
         * <code>MYSQL</code>, or <code>SQLSERVER</code>). For information about individual connectors, see <a
         * href="https://docs.aws.amazon.com/athena/latest/ug/connectors-available.html">Available data source
         * connectors</a>.
         * </p>
         * 
         * @param connectionType
         *        The type of connection for a <code>FEDERATED</code> data catalog (for example, <code>REDSHIFT</code>,
         *        <code>MYSQL</code>, or <code>SQLSERVER</code>). For information about individual connectors, see <a
         *        href="https://docs.aws.amazon.com/athena/latest/ug/connectors-available.html">Available data source
         *        connectors</a>.
         * @see ConnectionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConnectionType
         */
        Builder connectionType(String connectionType);

        /**
         * <p>
         * The type of connection for a <code>FEDERATED</code> data catalog (for example, <code>REDSHIFT</code>,
         * <code>MYSQL</code>, or <code>SQLSERVER</code>). For information about individual connectors, see <a
         * href="https://docs.aws.amazon.com/athena/latest/ug/connectors-available.html">Available data source
         * connectors</a>.
         * </p>
         * 
         * @param connectionType
         *        The type of connection for a <code>FEDERATED</code> data catalog (for example, <code>REDSHIFT</code>,
         *        <code>MYSQL</code>, or <code>SQLSERVER</code>). For information about individual connectors, see <a
         *        href="https://docs.aws.amazon.com/athena/latest/ug/connectors-available.html">Available data source
         *        connectors</a>.
         * @see ConnectionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConnectionType
         */
        Builder connectionType(ConnectionType connectionType);

        /**
         * <p>
         * Text of the error that occurred during data catalog creation or deletion.
         * </p>
         * 
         * @param error
         *        Text of the error that occurred during data catalog creation or deletion.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder error(String error);
    }

    static final class BuilderImpl implements Builder {
        private String catalogName;

        private String type;

        private String status;

        private String connectionType;

        private String error;

        private BuilderImpl() {
        }

        private BuilderImpl(DataCatalogSummary model) {
            catalogName(model.catalogName);
            type(model.type);
            status(model.status);
            connectionType(model.connectionType);
            error(model.error);
        }

        public final String getCatalogName() {
            return catalogName;
        }

        public final void setCatalogName(String catalogName) {
            this.catalogName = catalogName;
        }

        @Override
        public final Builder catalogName(String catalogName) {
            this.catalogName = catalogName;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(DataCatalogType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(DataCatalogStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getConnectionType() {
            return connectionType;
        }

        public final void setConnectionType(String connectionType) {
            this.connectionType = connectionType;
        }

        @Override
        public final Builder connectionType(String connectionType) {
            this.connectionType = connectionType;
            return this;
        }

        @Override
        public final Builder connectionType(ConnectionType connectionType) {
            this.connectionType(connectionType == null ? null : connectionType.toString());
            return this;
        }

        public final String getError() {
            return error;
        }

        public final void setError(String error) {
            this.error = error;
        }

        @Override
        public final Builder error(String error) {
            this.error = error;
            return this;
        }

        @Override
        public DataCatalogSummary build() {
            return new DataCatalogSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
