/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.athena.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a single instance of a query execution.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class QueryExecution implements SdkPojo, Serializable, ToCopyableBuilder<QueryExecution.Builder, QueryExecution> {
    private static final SdkField<String> QUERY_EXECUTION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("QueryExecutionId").getter(getter(QueryExecution::queryExecutionId))
            .setter(setter(Builder::queryExecutionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueryExecutionId").build()).build();

    private static final SdkField<String> QUERY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Query")
            .getter(getter(QueryExecution::query)).setter(setter(Builder::query))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Query").build()).build();

    private static final SdkField<String> STATEMENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatementType").getter(getter(QueryExecution::statementTypeAsString))
            .setter(setter(Builder::statementType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatementType").build()).build();

    private static final SdkField<ResultConfiguration> RESULT_CONFIGURATION_FIELD = SdkField
            .<ResultConfiguration> builder(MarshallingType.SDK_POJO).memberName("ResultConfiguration")
            .getter(getter(QueryExecution::resultConfiguration)).setter(setter(Builder::resultConfiguration))
            .constructor(ResultConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResultConfiguration").build())
            .build();

    private static final SdkField<ResultReuseConfiguration> RESULT_REUSE_CONFIGURATION_FIELD = SdkField
            .<ResultReuseConfiguration> builder(MarshallingType.SDK_POJO).memberName("ResultReuseConfiguration")
            .getter(getter(QueryExecution::resultReuseConfiguration)).setter(setter(Builder::resultReuseConfiguration))
            .constructor(ResultReuseConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResultReuseConfiguration").build())
            .build();

    private static final SdkField<QueryExecutionContext> QUERY_EXECUTION_CONTEXT_FIELD = SdkField
            .<QueryExecutionContext> builder(MarshallingType.SDK_POJO).memberName("QueryExecutionContext")
            .getter(getter(QueryExecution::queryExecutionContext)).setter(setter(Builder::queryExecutionContext))
            .constructor(QueryExecutionContext::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueryExecutionContext").build())
            .build();

    private static final SdkField<QueryExecutionStatus> STATUS_FIELD = SdkField
            .<QueryExecutionStatus> builder(MarshallingType.SDK_POJO).memberName("Status").getter(getter(QueryExecution::status))
            .setter(setter(Builder::status)).constructor(QueryExecutionStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<QueryExecutionStatistics> STATISTICS_FIELD = SdkField
            .<QueryExecutionStatistics> builder(MarshallingType.SDK_POJO).memberName("Statistics")
            .getter(getter(QueryExecution::statistics)).setter(setter(Builder::statistics))
            .constructor(QueryExecutionStatistics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Statistics").build()).build();

    private static final SdkField<String> WORK_GROUP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WorkGroup").getter(getter(QueryExecution::workGroup)).setter(setter(Builder::workGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkGroup").build()).build();

    private static final SdkField<EngineVersion> ENGINE_VERSION_FIELD = SdkField
            .<EngineVersion> builder(MarshallingType.SDK_POJO).memberName("EngineVersion")
            .getter(getter(QueryExecution::engineVersion)).setter(setter(Builder::engineVersion))
            .constructor(EngineVersion::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineVersion").build()).build();

    private static final SdkField<List<String>> EXECUTION_PARAMETERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExecutionParameters")
            .getter(getter(QueryExecution::executionParameters))
            .setter(setter(Builder::executionParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionParameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SUBSTATEMENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SubstatementType").getter(getter(QueryExecution::substatementType))
            .setter(setter(Builder::substatementType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubstatementType").build()).build();

    private static final SdkField<QueryResultsS3AccessGrantsConfiguration> QUERY_RESULTS_S3_ACCESS_GRANTS_CONFIGURATION_FIELD = SdkField
            .<QueryResultsS3AccessGrantsConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("QueryResultsS3AccessGrantsConfiguration")
            .getter(getter(QueryExecution::queryResultsS3AccessGrantsConfiguration))
            .setter(setter(Builder::queryResultsS3AccessGrantsConfiguration))
            .constructor(QueryResultsS3AccessGrantsConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("QueryResultsS3AccessGrantsConfiguration").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(QUERY_EXECUTION_ID_FIELD,
            QUERY_FIELD, STATEMENT_TYPE_FIELD, RESULT_CONFIGURATION_FIELD, RESULT_REUSE_CONFIGURATION_FIELD,
            QUERY_EXECUTION_CONTEXT_FIELD, STATUS_FIELD, STATISTICS_FIELD, WORK_GROUP_FIELD, ENGINE_VERSION_FIELD,
            EXECUTION_PARAMETERS_FIELD, SUBSTATEMENT_TYPE_FIELD, QUERY_RESULTS_S3_ACCESS_GRANTS_CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String queryExecutionId;

    private final String query;

    private final String statementType;

    private final ResultConfiguration resultConfiguration;

    private final ResultReuseConfiguration resultReuseConfiguration;

    private final QueryExecutionContext queryExecutionContext;

    private final QueryExecutionStatus status;

    private final QueryExecutionStatistics statistics;

    private final String workGroup;

    private final EngineVersion engineVersion;

    private final List<String> executionParameters;

    private final String substatementType;

    private final QueryResultsS3AccessGrantsConfiguration queryResultsS3AccessGrantsConfiguration;

    private QueryExecution(BuilderImpl builder) {
        this.queryExecutionId = builder.queryExecutionId;
        this.query = builder.query;
        this.statementType = builder.statementType;
        this.resultConfiguration = builder.resultConfiguration;
        this.resultReuseConfiguration = builder.resultReuseConfiguration;
        this.queryExecutionContext = builder.queryExecutionContext;
        this.status = builder.status;
        this.statistics = builder.statistics;
        this.workGroup = builder.workGroup;
        this.engineVersion = builder.engineVersion;
        this.executionParameters = builder.executionParameters;
        this.substatementType = builder.substatementType;
        this.queryResultsS3AccessGrantsConfiguration = builder.queryResultsS3AccessGrantsConfiguration;
    }

    /**
     * <p>
     * The unique identifier for each query execution.
     * </p>
     * 
     * @return The unique identifier for each query execution.
     */
    public final String queryExecutionId() {
        return queryExecutionId;
    }

    /**
     * <p>
     * The SQL query statements which the query execution ran.
     * </p>
     * 
     * @return The SQL query statements which the query execution ran.
     */
    public final String query() {
        return query;
    }

    /**
     * <p>
     * The type of query statement that was run. <code>DDL</code> indicates DDL query statements. <code>DML</code>
     * indicates DML (Data Manipulation Language) query statements, such as <code>CREATE TABLE AS SELECT</code>.
     * <code>UTILITY</code> indicates query statements other than DDL and DML, such as <code>SHOW CREATE TABLE</code>,
     * <code>EXPLAIN</code>, <code>DESCRIBE</code>, or <code>SHOW TABLES</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statementType}
     * will return {@link StatementType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statementTypeAsString}.
     * </p>
     * 
     * @return The type of query statement that was run. <code>DDL</code> indicates DDL query statements.
     *         <code>DML</code> indicates DML (Data Manipulation Language) query statements, such as
     *         <code>CREATE TABLE AS SELECT</code>. <code>UTILITY</code> indicates query statements other than DDL and
     *         DML, such as <code>SHOW CREATE TABLE</code>, <code>EXPLAIN</code>, <code>DESCRIBE</code>, or
     *         <code>SHOW TABLES</code>.
     * @see StatementType
     */
    public final StatementType statementType() {
        return StatementType.fromValue(statementType);
    }

    /**
     * <p>
     * The type of query statement that was run. <code>DDL</code> indicates DDL query statements. <code>DML</code>
     * indicates DML (Data Manipulation Language) query statements, such as <code>CREATE TABLE AS SELECT</code>.
     * <code>UTILITY</code> indicates query statements other than DDL and DML, such as <code>SHOW CREATE TABLE</code>,
     * <code>EXPLAIN</code>, <code>DESCRIBE</code>, or <code>SHOW TABLES</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statementType}
     * will return {@link StatementType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statementTypeAsString}.
     * </p>
     * 
     * @return The type of query statement that was run. <code>DDL</code> indicates DDL query statements.
     *         <code>DML</code> indicates DML (Data Manipulation Language) query statements, such as
     *         <code>CREATE TABLE AS SELECT</code>. <code>UTILITY</code> indicates query statements other than DDL and
     *         DML, such as <code>SHOW CREATE TABLE</code>, <code>EXPLAIN</code>, <code>DESCRIBE</code>, or
     *         <code>SHOW TABLES</code>.
     * @see StatementType
     */
    public final String statementTypeAsString() {
        return statementType;
    }

    /**
     * <p>
     * The location in Amazon S3 where query and calculation results are stored and the encryption option, if any, used
     * for query results. These are known as "client-side settings". If workgroup settings override client-side
     * settings, then the query uses the location for the query results and the encryption configuration that are
     * specified for the workgroup.
     * </p>
     * 
     * @return The location in Amazon S3 where query and calculation results are stored and the encryption option, if
     *         any, used for query results. These are known as "client-side settings". If workgroup settings override
     *         client-side settings, then the query uses the location for the query results and the encryption
     *         configuration that are specified for the workgroup.
     */
    public final ResultConfiguration resultConfiguration() {
        return resultConfiguration;
    }

    /**
     * <p>
     * Specifies the query result reuse behavior that was used for the query.
     * </p>
     * 
     * @return Specifies the query result reuse behavior that was used for the query.
     */
    public final ResultReuseConfiguration resultReuseConfiguration() {
        return resultReuseConfiguration;
    }

    /**
     * <p>
     * The database in which the query execution occurred.
     * </p>
     * 
     * @return The database in which the query execution occurred.
     */
    public final QueryExecutionContext queryExecutionContext() {
        return queryExecutionContext;
    }

    /**
     * <p>
     * The completion date, current state, submission time, and state change reason (if applicable) for the query
     * execution.
     * </p>
     * 
     * @return The completion date, current state, submission time, and state change reason (if applicable) for the
     *         query execution.
     */
    public final QueryExecutionStatus status() {
        return status;
    }

    /**
     * <p>
     * Query execution statistics, such as the amount of data scanned, the amount of time that the query took to
     * process, and the type of statement that was run.
     * </p>
     * 
     * @return Query execution statistics, such as the amount of data scanned, the amount of time that the query took to
     *         process, and the type of statement that was run.
     */
    public final QueryExecutionStatistics statistics() {
        return statistics;
    }

    /**
     * <p>
     * The name of the workgroup in which the query ran.
     * </p>
     * 
     * @return The name of the workgroup in which the query ran.
     */
    public final String workGroup() {
        return workGroup;
    }

    /**
     * <p>
     * The engine version that executed the query.
     * </p>
     * 
     * @return The engine version that executed the query.
     */
    public final EngineVersion engineVersion() {
        return engineVersion;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExecutionParameters property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExecutionParameters() {
        return executionParameters != null && !(executionParameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of values for the parameters in a query. The values are applied sequentially to the parameters in the
     * query in the order in which the parameters occur. The list of parameters is not returned in the response.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExecutionParameters} method.
     * </p>
     * 
     * @return A list of values for the parameters in a query. The values are applied sequentially to the parameters in
     *         the query in the order in which the parameters occur. The list of parameters is not returned in the
     *         response.
     */
    public final List<String> executionParameters() {
        return executionParameters;
    }

    /**
     * <p>
     * The kind of query statement that was run.
     * </p>
     * 
     * @return The kind of query statement that was run.
     */
    public final String substatementType() {
        return substatementType;
    }

    /**
     * <p>
     * Specifies whether Amazon S3 access grants are enabled for query results.
     * </p>
     * 
     * @return Specifies whether Amazon S3 access grants are enabled for query results.
     */
    public final QueryResultsS3AccessGrantsConfiguration queryResultsS3AccessGrantsConfiguration() {
        return queryResultsS3AccessGrantsConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(queryExecutionId());
        hashCode = 31 * hashCode + Objects.hashCode(query());
        hashCode = 31 * hashCode + Objects.hashCode(statementTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resultConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(resultReuseConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(queryExecutionContext());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(statistics());
        hashCode = 31 * hashCode + Objects.hashCode(workGroup());
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hasExecutionParameters() ? executionParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(substatementType());
        hashCode = 31 * hashCode + Objects.hashCode(queryResultsS3AccessGrantsConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof QueryExecution)) {
            return false;
        }
        QueryExecution other = (QueryExecution) obj;
        return Objects.equals(queryExecutionId(), other.queryExecutionId()) && Objects.equals(query(), other.query())
                && Objects.equals(statementTypeAsString(), other.statementTypeAsString())
                && Objects.equals(resultConfiguration(), other.resultConfiguration())
                && Objects.equals(resultReuseConfiguration(), other.resultReuseConfiguration())
                && Objects.equals(queryExecutionContext(), other.queryExecutionContext())
                && Objects.equals(status(), other.status()) && Objects.equals(statistics(), other.statistics())
                && Objects.equals(workGroup(), other.workGroup()) && Objects.equals(engineVersion(), other.engineVersion())
                && hasExecutionParameters() == other.hasExecutionParameters()
                && Objects.equals(executionParameters(), other.executionParameters())
                && Objects.equals(substatementType(), other.substatementType())
                && Objects.equals(queryResultsS3AccessGrantsConfiguration(), other.queryResultsS3AccessGrantsConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("QueryExecution").add("QueryExecutionId", queryExecutionId()).add("Query", query())
                .add("StatementType", statementTypeAsString()).add("ResultConfiguration", resultConfiguration())
                .add("ResultReuseConfiguration", resultReuseConfiguration())
                .add("QueryExecutionContext", queryExecutionContext()).add("Status", status()).add("Statistics", statistics())
                .add("WorkGroup", workGroup()).add("EngineVersion", engineVersion())
                .add("ExecutionParameters", hasExecutionParameters() ? executionParameters() : null)
                .add("SubstatementType", substatementType())
                .add("QueryResultsS3AccessGrantsConfiguration", queryResultsS3AccessGrantsConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "QueryExecutionId":
            return Optional.ofNullable(clazz.cast(queryExecutionId()));
        case "Query":
            return Optional.ofNullable(clazz.cast(query()));
        case "StatementType":
            return Optional.ofNullable(clazz.cast(statementTypeAsString()));
        case "ResultConfiguration":
            return Optional.ofNullable(clazz.cast(resultConfiguration()));
        case "ResultReuseConfiguration":
            return Optional.ofNullable(clazz.cast(resultReuseConfiguration()));
        case "QueryExecutionContext":
            return Optional.ofNullable(clazz.cast(queryExecutionContext()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "Statistics":
            return Optional.ofNullable(clazz.cast(statistics()));
        case "WorkGroup":
            return Optional.ofNullable(clazz.cast(workGroup()));
        case "EngineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        case "ExecutionParameters":
            return Optional.ofNullable(clazz.cast(executionParameters()));
        case "SubstatementType":
            return Optional.ofNullable(clazz.cast(substatementType()));
        case "QueryResultsS3AccessGrantsConfiguration":
            return Optional.ofNullable(clazz.cast(queryResultsS3AccessGrantsConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("QueryExecutionId", QUERY_EXECUTION_ID_FIELD);
        map.put("Query", QUERY_FIELD);
        map.put("StatementType", STATEMENT_TYPE_FIELD);
        map.put("ResultConfiguration", RESULT_CONFIGURATION_FIELD);
        map.put("ResultReuseConfiguration", RESULT_REUSE_CONFIGURATION_FIELD);
        map.put("QueryExecutionContext", QUERY_EXECUTION_CONTEXT_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("Statistics", STATISTICS_FIELD);
        map.put("WorkGroup", WORK_GROUP_FIELD);
        map.put("EngineVersion", ENGINE_VERSION_FIELD);
        map.put("ExecutionParameters", EXECUTION_PARAMETERS_FIELD);
        map.put("SubstatementType", SUBSTATEMENT_TYPE_FIELD);
        map.put("QueryResultsS3AccessGrantsConfiguration", QUERY_RESULTS_S3_ACCESS_GRANTS_CONFIGURATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<QueryExecution, T> g) {
        return obj -> g.apply((QueryExecution) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, QueryExecution> {
        /**
         * <p>
         * The unique identifier for each query execution.
         * </p>
         * 
         * @param queryExecutionId
         *        The unique identifier for each query execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryExecutionId(String queryExecutionId);

        /**
         * <p>
         * The SQL query statements which the query execution ran.
         * </p>
         * 
         * @param query
         *        The SQL query statements which the query execution ran.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder query(String query);

        /**
         * <p>
         * The type of query statement that was run. <code>DDL</code> indicates DDL query statements. <code>DML</code>
         * indicates DML (Data Manipulation Language) query statements, such as <code>CREATE TABLE AS SELECT</code>.
         * <code>UTILITY</code> indicates query statements other than DDL and DML, such as
         * <code>SHOW CREATE TABLE</code>, <code>EXPLAIN</code>, <code>DESCRIBE</code>, or <code>SHOW TABLES</code>.
         * </p>
         * 
         * @param statementType
         *        The type of query statement that was run. <code>DDL</code> indicates DDL query statements.
         *        <code>DML</code> indicates DML (Data Manipulation Language) query statements, such as
         *        <code>CREATE TABLE AS SELECT</code>. <code>UTILITY</code> indicates query statements other than DDL
         *        and DML, such as <code>SHOW CREATE TABLE</code>, <code>EXPLAIN</code>, <code>DESCRIBE</code>, or
         *        <code>SHOW TABLES</code>.
         * @see StatementType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatementType
         */
        Builder statementType(String statementType);

        /**
         * <p>
         * The type of query statement that was run. <code>DDL</code> indicates DDL query statements. <code>DML</code>
         * indicates DML (Data Manipulation Language) query statements, such as <code>CREATE TABLE AS SELECT</code>.
         * <code>UTILITY</code> indicates query statements other than DDL and DML, such as
         * <code>SHOW CREATE TABLE</code>, <code>EXPLAIN</code>, <code>DESCRIBE</code>, or <code>SHOW TABLES</code>.
         * </p>
         * 
         * @param statementType
         *        The type of query statement that was run. <code>DDL</code> indicates DDL query statements.
         *        <code>DML</code> indicates DML (Data Manipulation Language) query statements, such as
         *        <code>CREATE TABLE AS SELECT</code>. <code>UTILITY</code> indicates query statements other than DDL
         *        and DML, such as <code>SHOW CREATE TABLE</code>, <code>EXPLAIN</code>, <code>DESCRIBE</code>, or
         *        <code>SHOW TABLES</code>.
         * @see StatementType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatementType
         */
        Builder statementType(StatementType statementType);

        /**
         * <p>
         * The location in Amazon S3 where query and calculation results are stored and the encryption option, if any,
         * used for query results. These are known as "client-side settings". If workgroup settings override client-side
         * settings, then the query uses the location for the query results and the encryption configuration that are
         * specified for the workgroup.
         * </p>
         * 
         * @param resultConfiguration
         *        The location in Amazon S3 where query and calculation results are stored and the encryption option, if
         *        any, used for query results. These are known as "client-side settings". If workgroup settings override
         *        client-side settings, then the query uses the location for the query results and the encryption
         *        configuration that are specified for the workgroup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resultConfiguration(ResultConfiguration resultConfiguration);

        /**
         * <p>
         * The location in Amazon S3 where query and calculation results are stored and the encryption option, if any,
         * used for query results. These are known as "client-side settings". If workgroup settings override client-side
         * settings, then the query uses the location for the query results and the encryption configuration that are
         * specified for the workgroup.
         * </p>
         * This is a convenience method that creates an instance of the {@link ResultConfiguration.Builder} avoiding the
         * need to create one manually via {@link ResultConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResultConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #resultConfiguration(ResultConfiguration)}.
         * 
         * @param resultConfiguration
         *        a consumer that will call methods on {@link ResultConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resultConfiguration(ResultConfiguration)
         */
        default Builder resultConfiguration(Consumer<ResultConfiguration.Builder> resultConfiguration) {
            return resultConfiguration(ResultConfiguration.builder().applyMutation(resultConfiguration).build());
        }

        /**
         * <p>
         * Specifies the query result reuse behavior that was used for the query.
         * </p>
         * 
         * @param resultReuseConfiguration
         *        Specifies the query result reuse behavior that was used for the query.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resultReuseConfiguration(ResultReuseConfiguration resultReuseConfiguration);

        /**
         * <p>
         * Specifies the query result reuse behavior that was used for the query.
         * </p>
         * This is a convenience method that creates an instance of the {@link ResultReuseConfiguration.Builder}
         * avoiding the need to create one manually via {@link ResultReuseConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResultReuseConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #resultReuseConfiguration(ResultReuseConfiguration)}.
         * 
         * @param resultReuseConfiguration
         *        a consumer that will call methods on {@link ResultReuseConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resultReuseConfiguration(ResultReuseConfiguration)
         */
        default Builder resultReuseConfiguration(Consumer<ResultReuseConfiguration.Builder> resultReuseConfiguration) {
            return resultReuseConfiguration(ResultReuseConfiguration.builder().applyMutation(resultReuseConfiguration).build());
        }

        /**
         * <p>
         * The database in which the query execution occurred.
         * </p>
         * 
         * @param queryExecutionContext
         *        The database in which the query execution occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryExecutionContext(QueryExecutionContext queryExecutionContext);

        /**
         * <p>
         * The database in which the query execution occurred.
         * </p>
         * This is a convenience method that creates an instance of the {@link QueryExecutionContext.Builder} avoiding
         * the need to create one manually via {@link QueryExecutionContext#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link QueryExecutionContext.Builder#build()} is called immediately and
         * its result is passed to {@link #queryExecutionContext(QueryExecutionContext)}.
         * 
         * @param queryExecutionContext
         *        a consumer that will call methods on {@link QueryExecutionContext.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #queryExecutionContext(QueryExecutionContext)
         */
        default Builder queryExecutionContext(Consumer<QueryExecutionContext.Builder> queryExecutionContext) {
            return queryExecutionContext(QueryExecutionContext.builder().applyMutation(queryExecutionContext).build());
        }

        /**
         * <p>
         * The completion date, current state, submission time, and state change reason (if applicable) for the query
         * execution.
         * </p>
         * 
         * @param status
         *        The completion date, current state, submission time, and state change reason (if applicable) for the
         *        query execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(QueryExecutionStatus status);

        /**
         * <p>
         * The completion date, current state, submission time, and state change reason (if applicable) for the query
         * execution.
         * </p>
         * This is a convenience method that creates an instance of the {@link QueryExecutionStatus.Builder} avoiding
         * the need to create one manually via {@link QueryExecutionStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link QueryExecutionStatus.Builder#build()} is called immediately and
         * its result is passed to {@link #status(QueryExecutionStatus)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link QueryExecutionStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(QueryExecutionStatus)
         */
        default Builder status(Consumer<QueryExecutionStatus.Builder> status) {
            return status(QueryExecutionStatus.builder().applyMutation(status).build());
        }

        /**
         * <p>
         * Query execution statistics, such as the amount of data scanned, the amount of time that the query took to
         * process, and the type of statement that was run.
         * </p>
         * 
         * @param statistics
         *        Query execution statistics, such as the amount of data scanned, the amount of time that the query took
         *        to process, and the type of statement that was run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statistics(QueryExecutionStatistics statistics);

        /**
         * <p>
         * Query execution statistics, such as the amount of data scanned, the amount of time that the query took to
         * process, and the type of statement that was run.
         * </p>
         * This is a convenience method that creates an instance of the {@link QueryExecutionStatistics.Builder}
         * avoiding the need to create one manually via {@link QueryExecutionStatistics#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link QueryExecutionStatistics.Builder#build()} is called immediately
         * and its result is passed to {@link #statistics(QueryExecutionStatistics)}.
         * 
         * @param statistics
         *        a consumer that will call methods on {@link QueryExecutionStatistics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #statistics(QueryExecutionStatistics)
         */
        default Builder statistics(Consumer<QueryExecutionStatistics.Builder> statistics) {
            return statistics(QueryExecutionStatistics.builder().applyMutation(statistics).build());
        }

        /**
         * <p>
         * The name of the workgroup in which the query ran.
         * </p>
         * 
         * @param workGroup
         *        The name of the workgroup in which the query ran.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workGroup(String workGroup);

        /**
         * <p>
         * The engine version that executed the query.
         * </p>
         * 
         * @param engineVersion
         *        The engine version that executed the query.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(EngineVersion engineVersion);

        /**
         * <p>
         * The engine version that executed the query.
         * </p>
         * This is a convenience method that creates an instance of the {@link EngineVersion.Builder} avoiding the need
         * to create one manually via {@link EngineVersion#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EngineVersion.Builder#build()} is called immediately and its
         * result is passed to {@link #engineVersion(EngineVersion)}.
         * 
         * @param engineVersion
         *        a consumer that will call methods on {@link EngineVersion.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #engineVersion(EngineVersion)
         */
        default Builder engineVersion(Consumer<EngineVersion.Builder> engineVersion) {
            return engineVersion(EngineVersion.builder().applyMutation(engineVersion).build());
        }

        /**
         * <p>
         * A list of values for the parameters in a query. The values are applied sequentially to the parameters in the
         * query in the order in which the parameters occur. The list of parameters is not returned in the response.
         * </p>
         * 
         * @param executionParameters
         *        A list of values for the parameters in a query. The values are applied sequentially to the parameters
         *        in the query in the order in which the parameters occur. The list of parameters is not returned in the
         *        response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionParameters(Collection<String> executionParameters);

        /**
         * <p>
         * A list of values for the parameters in a query. The values are applied sequentially to the parameters in the
         * query in the order in which the parameters occur. The list of parameters is not returned in the response.
         * </p>
         * 
         * @param executionParameters
         *        A list of values for the parameters in a query. The values are applied sequentially to the parameters
         *        in the query in the order in which the parameters occur. The list of parameters is not returned in the
         *        response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionParameters(String... executionParameters);

        /**
         * <p>
         * The kind of query statement that was run.
         * </p>
         * 
         * @param substatementType
         *        The kind of query statement that was run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder substatementType(String substatementType);

        /**
         * <p>
         * Specifies whether Amazon S3 access grants are enabled for query results.
         * </p>
         * 
         * @param queryResultsS3AccessGrantsConfiguration
         *        Specifies whether Amazon S3 access grants are enabled for query results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryResultsS3AccessGrantsConfiguration(
                QueryResultsS3AccessGrantsConfiguration queryResultsS3AccessGrantsConfiguration);

        /**
         * <p>
         * Specifies whether Amazon S3 access grants are enabled for query results.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link QueryResultsS3AccessGrantsConfiguration.Builder} avoiding the need to create one manually via
         * {@link QueryResultsS3AccessGrantsConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link QueryResultsS3AccessGrantsConfiguration.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #queryResultsS3AccessGrantsConfiguration(QueryResultsS3AccessGrantsConfiguration)}.
         * 
         * @param queryResultsS3AccessGrantsConfiguration
         *        a consumer that will call methods on {@link QueryResultsS3AccessGrantsConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #queryResultsS3AccessGrantsConfiguration(QueryResultsS3AccessGrantsConfiguration)
         */
        default Builder queryResultsS3AccessGrantsConfiguration(
                Consumer<QueryResultsS3AccessGrantsConfiguration.Builder> queryResultsS3AccessGrantsConfiguration) {
            return queryResultsS3AccessGrantsConfiguration(QueryResultsS3AccessGrantsConfiguration.builder()
                    .applyMutation(queryResultsS3AccessGrantsConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String queryExecutionId;

        private String query;

        private String statementType;

        private ResultConfiguration resultConfiguration;

        private ResultReuseConfiguration resultReuseConfiguration;

        private QueryExecutionContext queryExecutionContext;

        private QueryExecutionStatus status;

        private QueryExecutionStatistics statistics;

        private String workGroup;

        private EngineVersion engineVersion;

        private List<String> executionParameters = DefaultSdkAutoConstructList.getInstance();

        private String substatementType;

        private QueryResultsS3AccessGrantsConfiguration queryResultsS3AccessGrantsConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(QueryExecution model) {
            queryExecutionId(model.queryExecutionId);
            query(model.query);
            statementType(model.statementType);
            resultConfiguration(model.resultConfiguration);
            resultReuseConfiguration(model.resultReuseConfiguration);
            queryExecutionContext(model.queryExecutionContext);
            status(model.status);
            statistics(model.statistics);
            workGroup(model.workGroup);
            engineVersion(model.engineVersion);
            executionParameters(model.executionParameters);
            substatementType(model.substatementType);
            queryResultsS3AccessGrantsConfiguration(model.queryResultsS3AccessGrantsConfiguration);
        }

        public final String getQueryExecutionId() {
            return queryExecutionId;
        }

        public final void setQueryExecutionId(String queryExecutionId) {
            this.queryExecutionId = queryExecutionId;
        }

        @Override
        public final Builder queryExecutionId(String queryExecutionId) {
            this.queryExecutionId = queryExecutionId;
            return this;
        }

        public final String getQuery() {
            return query;
        }

        public final void setQuery(String query) {
            this.query = query;
        }

        @Override
        public final Builder query(String query) {
            this.query = query;
            return this;
        }

        public final String getStatementType() {
            return statementType;
        }

        public final void setStatementType(String statementType) {
            this.statementType = statementType;
        }

        @Override
        public final Builder statementType(String statementType) {
            this.statementType = statementType;
            return this;
        }

        @Override
        public final Builder statementType(StatementType statementType) {
            this.statementType(statementType == null ? null : statementType.toString());
            return this;
        }

        public final ResultConfiguration.Builder getResultConfiguration() {
            return resultConfiguration != null ? resultConfiguration.toBuilder() : null;
        }

        public final void setResultConfiguration(ResultConfiguration.BuilderImpl resultConfiguration) {
            this.resultConfiguration = resultConfiguration != null ? resultConfiguration.build() : null;
        }

        @Override
        public final Builder resultConfiguration(ResultConfiguration resultConfiguration) {
            this.resultConfiguration = resultConfiguration;
            return this;
        }

        public final ResultReuseConfiguration.Builder getResultReuseConfiguration() {
            return resultReuseConfiguration != null ? resultReuseConfiguration.toBuilder() : null;
        }

        public final void setResultReuseConfiguration(ResultReuseConfiguration.BuilderImpl resultReuseConfiguration) {
            this.resultReuseConfiguration = resultReuseConfiguration != null ? resultReuseConfiguration.build() : null;
        }

        @Override
        public final Builder resultReuseConfiguration(ResultReuseConfiguration resultReuseConfiguration) {
            this.resultReuseConfiguration = resultReuseConfiguration;
            return this;
        }

        public final QueryExecutionContext.Builder getQueryExecutionContext() {
            return queryExecutionContext != null ? queryExecutionContext.toBuilder() : null;
        }

        public final void setQueryExecutionContext(QueryExecutionContext.BuilderImpl queryExecutionContext) {
            this.queryExecutionContext = queryExecutionContext != null ? queryExecutionContext.build() : null;
        }

        @Override
        public final Builder queryExecutionContext(QueryExecutionContext queryExecutionContext) {
            this.queryExecutionContext = queryExecutionContext;
            return this;
        }

        public final QueryExecutionStatus.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        public final void setStatus(QueryExecutionStatus.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        @Override
        public final Builder status(QueryExecutionStatus status) {
            this.status = status;
            return this;
        }

        public final QueryExecutionStatistics.Builder getStatistics() {
            return statistics != null ? statistics.toBuilder() : null;
        }

        public final void setStatistics(QueryExecutionStatistics.BuilderImpl statistics) {
            this.statistics = statistics != null ? statistics.build() : null;
        }

        @Override
        public final Builder statistics(QueryExecutionStatistics statistics) {
            this.statistics = statistics;
            return this;
        }

        public final String getWorkGroup() {
            return workGroup;
        }

        public final void setWorkGroup(String workGroup) {
            this.workGroup = workGroup;
        }

        @Override
        public final Builder workGroup(String workGroup) {
            this.workGroup = workGroup;
            return this;
        }

        public final EngineVersion.Builder getEngineVersion() {
            return engineVersion != null ? engineVersion.toBuilder() : null;
        }

        public final void setEngineVersion(EngineVersion.BuilderImpl engineVersion) {
            this.engineVersion = engineVersion != null ? engineVersion.build() : null;
        }

        @Override
        public final Builder engineVersion(EngineVersion engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        public final Collection<String> getExecutionParameters() {
            if (executionParameters instanceof SdkAutoConstructList) {
                return null;
            }
            return executionParameters;
        }

        public final void setExecutionParameters(Collection<String> executionParameters) {
            this.executionParameters = ExecutionParametersCopier.copy(executionParameters);
        }

        @Override
        public final Builder executionParameters(Collection<String> executionParameters) {
            this.executionParameters = ExecutionParametersCopier.copy(executionParameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder executionParameters(String... executionParameters) {
            executionParameters(Arrays.asList(executionParameters));
            return this;
        }

        public final String getSubstatementType() {
            return substatementType;
        }

        public final void setSubstatementType(String substatementType) {
            this.substatementType = substatementType;
        }

        @Override
        public final Builder substatementType(String substatementType) {
            this.substatementType = substatementType;
            return this;
        }

        public final QueryResultsS3AccessGrantsConfiguration.Builder getQueryResultsS3AccessGrantsConfiguration() {
            return queryResultsS3AccessGrantsConfiguration != null ? queryResultsS3AccessGrantsConfiguration.toBuilder() : null;
        }

        public final void setQueryResultsS3AccessGrantsConfiguration(
                QueryResultsS3AccessGrantsConfiguration.BuilderImpl queryResultsS3AccessGrantsConfiguration) {
            this.queryResultsS3AccessGrantsConfiguration = queryResultsS3AccessGrantsConfiguration != null ? queryResultsS3AccessGrantsConfiguration
                    .build() : null;
        }

        @Override
        public final Builder queryResultsS3AccessGrantsConfiguration(
                QueryResultsS3AccessGrantsConfiguration queryResultsS3AccessGrantsConfiguration) {
            this.queryResultsS3AccessGrantsConfiguration = queryResultsS3AccessGrantsConfiguration;
            return this;
        }

        @Override
        public QueryExecution build() {
            return new QueryExecution(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
