/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.athena.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * If query and calculation results are encrypted in Amazon S3, indicates the encryption option used (for example,
 * <code>SSE_KMS</code> or <code>CSE_KMS</code>) and key information.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EncryptionConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<EncryptionConfiguration.Builder, EncryptionConfiguration> {
    private static final SdkField<String> ENCRYPTION_OPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EncryptionOption").getter(getter(EncryptionConfiguration::encryptionOptionAsString))
            .setter(setter(Builder::encryptionOption))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionOption").build()).build();

    private static final SdkField<String> KMS_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("KmsKey")
            .getter(getter(EncryptionConfiguration::kmsKey)).setter(setter(Builder::kmsKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKey").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENCRYPTION_OPTION_FIELD,
            KMS_KEY_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String encryptionOption;

    private final String kmsKey;

    private EncryptionConfiguration(BuilderImpl builder) {
        this.encryptionOption = builder.encryptionOption;
        this.kmsKey = builder.kmsKey;
    }

    /**
     * <p>
     * Indicates whether Amazon S3 server-side encryption with Amazon S3-managed keys (<code>SSE_S3</code>), server-side
     * encryption with KMS-managed keys (<code>SSE_KMS</code>), or client-side encryption with KMS-managed keys (
     * <code>CSE_KMS</code>) is used.
     * </p>
     * <p>
     * If a query runs in a workgroup and the workgroup overrides client-side settings, then the workgroup's setting for
     * encryption is used. It specifies whether query results must be encrypted, for all queries that run in this
     * workgroup.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionOption}
     * will return {@link EncryptionOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionOptionAsString}.
     * </p>
     * 
     * @return Indicates whether Amazon S3 server-side encryption with Amazon S3-managed keys (<code>SSE_S3</code>),
     *         server-side encryption with KMS-managed keys (<code>SSE_KMS</code>), or client-side encryption with
     *         KMS-managed keys (<code>CSE_KMS</code>) is used.</p>
     *         <p>
     *         If a query runs in a workgroup and the workgroup overrides client-side settings, then the workgroup's
     *         setting for encryption is used. It specifies whether query results must be encrypted, for all queries
     *         that run in this workgroup.
     * @see EncryptionOption
     */
    public final EncryptionOption encryptionOption() {
        return EncryptionOption.fromValue(encryptionOption);
    }

    /**
     * <p>
     * Indicates whether Amazon S3 server-side encryption with Amazon S3-managed keys (<code>SSE_S3</code>), server-side
     * encryption with KMS-managed keys (<code>SSE_KMS</code>), or client-side encryption with KMS-managed keys (
     * <code>CSE_KMS</code>) is used.
     * </p>
     * <p>
     * If a query runs in a workgroup and the workgroup overrides client-side settings, then the workgroup's setting for
     * encryption is used. It specifies whether query results must be encrypted, for all queries that run in this
     * workgroup.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionOption}
     * will return {@link EncryptionOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionOptionAsString}.
     * </p>
     * 
     * @return Indicates whether Amazon S3 server-side encryption with Amazon S3-managed keys (<code>SSE_S3</code>),
     *         server-side encryption with KMS-managed keys (<code>SSE_KMS</code>), or client-side encryption with
     *         KMS-managed keys (<code>CSE_KMS</code>) is used.</p>
     *         <p>
     *         If a query runs in a workgroup and the workgroup overrides client-side settings, then the workgroup's
     *         setting for encryption is used. It specifies whether query results must be encrypted, for all queries
     *         that run in this workgroup.
     * @see EncryptionOption
     */
    public final String encryptionOptionAsString() {
        return encryptionOption;
    }

    /**
     * <p>
     * For <code>SSE_KMS</code> and <code>CSE_KMS</code>, this is the KMS key ARN or ID.
     * </p>
     * 
     * @return For <code>SSE_KMS</code> and <code>CSE_KMS</code>, this is the KMS key ARN or ID.
     */
    public final String kmsKey() {
        return kmsKey;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(encryptionOptionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKey());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EncryptionConfiguration)) {
            return false;
        }
        EncryptionConfiguration other = (EncryptionConfiguration) obj;
        return Objects.equals(encryptionOptionAsString(), other.encryptionOptionAsString())
                && Objects.equals(kmsKey(), other.kmsKey());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EncryptionConfiguration").add("EncryptionOption", encryptionOptionAsString())
                .add("KmsKey", kmsKey()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EncryptionOption":
            return Optional.ofNullable(clazz.cast(encryptionOptionAsString()));
        case "KmsKey":
            return Optional.ofNullable(clazz.cast(kmsKey()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("EncryptionOption", ENCRYPTION_OPTION_FIELD);
        map.put("KmsKey", KMS_KEY_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<EncryptionConfiguration, T> g) {
        return obj -> g.apply((EncryptionConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EncryptionConfiguration> {
        /**
         * <p>
         * Indicates whether Amazon S3 server-side encryption with Amazon S3-managed keys (<code>SSE_S3</code>),
         * server-side encryption with KMS-managed keys (<code>SSE_KMS</code>), or client-side encryption with
         * KMS-managed keys (<code>CSE_KMS</code>) is used.
         * </p>
         * <p>
         * If a query runs in a workgroup and the workgroup overrides client-side settings, then the workgroup's setting
         * for encryption is used. It specifies whether query results must be encrypted, for all queries that run in
         * this workgroup.
         * </p>
         * 
         * @param encryptionOption
         *        Indicates whether Amazon S3 server-side encryption with Amazon S3-managed keys (<code>SSE_S3</code>),
         *        server-side encryption with KMS-managed keys (<code>SSE_KMS</code>), or client-side encryption with
         *        KMS-managed keys (<code>CSE_KMS</code>) is used.</p>
         *        <p>
         *        If a query runs in a workgroup and the workgroup overrides client-side settings, then the workgroup's
         *        setting for encryption is used. It specifies whether query results must be encrypted, for all queries
         *        that run in this workgroup.
         * @see EncryptionOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionOption
         */
        Builder encryptionOption(String encryptionOption);

        /**
         * <p>
         * Indicates whether Amazon S3 server-side encryption with Amazon S3-managed keys (<code>SSE_S3</code>),
         * server-side encryption with KMS-managed keys (<code>SSE_KMS</code>), or client-side encryption with
         * KMS-managed keys (<code>CSE_KMS</code>) is used.
         * </p>
         * <p>
         * If a query runs in a workgroup and the workgroup overrides client-side settings, then the workgroup's setting
         * for encryption is used. It specifies whether query results must be encrypted, for all queries that run in
         * this workgroup.
         * </p>
         * 
         * @param encryptionOption
         *        Indicates whether Amazon S3 server-side encryption with Amazon S3-managed keys (<code>SSE_S3</code>),
         *        server-side encryption with KMS-managed keys (<code>SSE_KMS</code>), or client-side encryption with
         *        KMS-managed keys (<code>CSE_KMS</code>) is used.</p>
         *        <p>
         *        If a query runs in a workgroup and the workgroup overrides client-side settings, then the workgroup's
         *        setting for encryption is used. It specifies whether query results must be encrypted, for all queries
         *        that run in this workgroup.
         * @see EncryptionOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionOption
         */
        Builder encryptionOption(EncryptionOption encryptionOption);

        /**
         * <p>
         * For <code>SSE_KMS</code> and <code>CSE_KMS</code>, this is the KMS key ARN or ID.
         * </p>
         * 
         * @param kmsKey
         *        For <code>SSE_KMS</code> and <code>CSE_KMS</code>, this is the KMS key ARN or ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKey(String kmsKey);
    }

    static final class BuilderImpl implements Builder {
        private String encryptionOption;

        private String kmsKey;

        private BuilderImpl() {
        }

        private BuilderImpl(EncryptionConfiguration model) {
            encryptionOption(model.encryptionOption);
            kmsKey(model.kmsKey);
        }

        public final String getEncryptionOption() {
            return encryptionOption;
        }

        public final void setEncryptionOption(String encryptionOption) {
            this.encryptionOption = encryptionOption;
        }

        @Override
        public final Builder encryptionOption(String encryptionOption) {
            this.encryptionOption = encryptionOption;
            return this;
        }

        @Override
        public final Builder encryptionOption(EncryptionOption encryptionOption) {
            this.encryptionOption(encryptionOption == null ? null : encryptionOption.toString());
            return this;
        }

        public final String getKmsKey() {
            return kmsKey;
        }

        public final void setKmsKey(String kmsKey) {
            this.kmsKey = kmsKey;
        }

        @Override
        public final Builder kmsKey(String kmsKey) {
            this.kmsKey = kmsKey;
            return this;
        }

        @Override
        public EncryptionConfiguration build() {
            return new EncryptionConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
