/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.auditmanager.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A metadata object associated with an assessment in AWS Audit Manager.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AssessmentMetadataItem implements SdkPojo, Serializable,
        ToCopyableBuilder<AssessmentMetadataItem.Builder, AssessmentMetadataItem> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(AssessmentMetadataItem::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(AssessmentMetadataItem::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> COMPLIANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("complianceType").getter(getter(AssessmentMetadataItem::complianceType))
            .setter(setter(Builder::complianceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("complianceType").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(AssessmentMetadataItem::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<List<Role>> ROLES_FIELD = SdkField
            .<List<Role>> builder(MarshallingType.LIST)
            .memberName("roles")
            .getter(getter(AssessmentMetadataItem::roles))
            .setter(setter(Builder::roles))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roles").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Role> builder(MarshallingType.SDK_POJO)
                                            .constructor(Role::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Delegation>> DELEGATIONS_FIELD = SdkField
            .<List<Delegation>> builder(MarshallingType.LIST)
            .memberName("delegations")
            .getter(getter(AssessmentMetadataItem::delegations))
            .setter(setter(Builder::delegations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("delegations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Delegation> builder(MarshallingType.SDK_POJO)
                                            .constructor(Delegation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("creationTime").getter(getter(AssessmentMetadataItem::creationTime))
            .setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("creationTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdated").getter(getter(AssessmentMetadataItem::lastUpdated)).setter(setter(Builder::lastUpdated))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdated").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ID_FIELD,
            COMPLIANCE_TYPE_FIELD, STATUS_FIELD, ROLES_FIELD, DELEGATIONS_FIELD, CREATION_TIME_FIELD, LAST_UPDATED_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String id;

    private final String complianceType;

    private final String status;

    private final List<Role> roles;

    private final List<Delegation> delegations;

    private final Instant creationTime;

    private final Instant lastUpdated;

    private AssessmentMetadataItem(BuilderImpl builder) {
        this.name = builder.name;
        this.id = builder.id;
        this.complianceType = builder.complianceType;
        this.status = builder.status;
        this.roles = builder.roles;
        this.delegations = builder.delegations;
        this.creationTime = builder.creationTime;
        this.lastUpdated = builder.lastUpdated;
    }

    /**
     * <p>
     * The name of the assessment.
     * </p>
     * 
     * @return The name of the assessment.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The unique identifier for the assessment.
     * </p>
     * 
     * @return The unique identifier for the assessment.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * The name of the compliance standard related to the assessment, such as PCI-DSS.
     * </p>
     * 
     * @return The name of the compliance standard related to the assessment, such as PCI-DSS.
     */
    public String complianceType() {
        return complianceType;
    }

    /**
     * <p>
     * The current status of the assessment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AssessmentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the assessment.
     * @see AssessmentStatus
     */
    public AssessmentStatus status() {
        return AssessmentStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the assessment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AssessmentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the assessment.
     * @see AssessmentStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * Returns true if the Roles property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasRoles() {
        return roles != null && !(roles instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The roles associated with the assessment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRoles()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The roles associated with the assessment.
     */
    public List<Role> roles() {
        return roles;
    }

    /**
     * Returns true if the Delegations property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasDelegations() {
        return delegations != null && !(delegations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The delegations associated with the assessment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDelegations()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The delegations associated with the assessment.
     */
    public List<Delegation> delegations() {
        return delegations;
    }

    /**
     * <p>
     * Specifies when the assessment was created.
     * </p>
     * 
     * @return Specifies when the assessment was created.
     */
    public Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The time of the most recent update.
     * </p>
     * 
     * @return The time of the most recent update.
     */
    public Instant lastUpdated() {
        return lastUpdated;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(complianceType());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasRoles() ? roles() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDelegations() ? delegations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdated());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AssessmentMetadataItem)) {
            return false;
        }
        AssessmentMetadataItem other = (AssessmentMetadataItem) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(id(), other.id())
                && Objects.equals(complianceType(), other.complianceType())
                && Objects.equals(statusAsString(), other.statusAsString()) && hasRoles() == other.hasRoles()
                && Objects.equals(roles(), other.roles()) && hasDelegations() == other.hasDelegations()
                && Objects.equals(delegations(), other.delegations()) && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(lastUpdated(), other.lastUpdated());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("AssessmentMetadataItem").add("Name", name()).add("Id", id())
                .add("ComplianceType", complianceType()).add("Status", statusAsString())
                .add("Roles", hasRoles() ? roles() : null).add("Delegations", hasDelegations() ? delegations() : null)
                .add("CreationTime", creationTime()).add("LastUpdated", lastUpdated()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "complianceType":
            return Optional.ofNullable(clazz.cast(complianceType()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "roles":
            return Optional.ofNullable(clazz.cast(roles()));
        case "delegations":
            return Optional.ofNullable(clazz.cast(delegations()));
        case "creationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "lastUpdated":
            return Optional.ofNullable(clazz.cast(lastUpdated()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AssessmentMetadataItem, T> g) {
        return obj -> g.apply((AssessmentMetadataItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AssessmentMetadataItem> {
        /**
         * <p>
         * The name of the assessment.
         * </p>
         * 
         * @param name
         *        The name of the assessment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The unique identifier for the assessment.
         * </p>
         * 
         * @param id
         *        The unique identifier for the assessment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The name of the compliance standard related to the assessment, such as PCI-DSS.
         * </p>
         * 
         * @param complianceType
         *        The name of the compliance standard related to the assessment, such as PCI-DSS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder complianceType(String complianceType);

        /**
         * <p>
         * The current status of the assessment.
         * </p>
         * 
         * @param status
         *        The current status of the assessment.
         * @see AssessmentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AssessmentStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the assessment.
         * </p>
         * 
         * @param status
         *        The current status of the assessment.
         * @see AssessmentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AssessmentStatus
         */
        Builder status(AssessmentStatus status);

        /**
         * <p>
         * The roles associated with the assessment.
         * </p>
         * 
         * @param roles
         *        The roles associated with the assessment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roles(Collection<Role> roles);

        /**
         * <p>
         * The roles associated with the assessment.
         * </p>
         * 
         * @param roles
         *        The roles associated with the assessment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roles(Role... roles);

        /**
         * <p>
         * The roles associated with the assessment.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Role>.Builder} avoiding the need to create
         * one manually via {@link List<Role>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Role>.Builder#build()} is called immediately and its result
         * is passed to {@link #roles(List<Role>)}.
         * 
         * @param roles
         *        a consumer that will call methods on {@link List<Role>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #roles(List<Role>)
         */
        Builder roles(Consumer<Role.Builder>... roles);

        /**
         * <p>
         * The delegations associated with the assessment.
         * </p>
         * 
         * @param delegations
         *        The delegations associated with the assessment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder delegations(Collection<Delegation> delegations);

        /**
         * <p>
         * The delegations associated with the assessment.
         * </p>
         * 
         * @param delegations
         *        The delegations associated with the assessment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder delegations(Delegation... delegations);

        /**
         * <p>
         * The delegations associated with the assessment.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Delegation>.Builder} avoiding the need to
         * create one manually via {@link List<Delegation>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Delegation>.Builder#build()} is called immediately and its
         * result is passed to {@link #delegations(List<Delegation>)}.
         * 
         * @param delegations
         *        a consumer that will call methods on {@link List<Delegation>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #delegations(List<Delegation>)
         */
        Builder delegations(Consumer<Delegation.Builder>... delegations);

        /**
         * <p>
         * Specifies when the assessment was created.
         * </p>
         * 
         * @param creationTime
         *        Specifies when the assessment was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The time of the most recent update.
         * </p>
         * 
         * @param lastUpdated
         *        The time of the most recent update.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdated(Instant lastUpdated);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String id;

        private String complianceType;

        private String status;

        private List<Role> roles = DefaultSdkAutoConstructList.getInstance();

        private List<Delegation> delegations = DefaultSdkAutoConstructList.getInstance();

        private Instant creationTime;

        private Instant lastUpdated;

        private BuilderImpl() {
        }

        private BuilderImpl(AssessmentMetadataItem model) {
            name(model.name);
            id(model.id);
            complianceType(model.complianceType);
            status(model.status);
            roles(model.roles);
            delegations(model.delegations);
            creationTime(model.creationTime);
            lastUpdated(model.lastUpdated);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getComplianceType() {
            return complianceType;
        }

        @Override
        public final Builder complianceType(String complianceType) {
            this.complianceType = complianceType;
            return this;
        }

        public final void setComplianceType(String complianceType) {
            this.complianceType = complianceType;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(AssessmentStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Collection<Role.Builder> getRoles() {
            if (roles instanceof SdkAutoConstructList) {
                return null;
            }
            return roles != null ? roles.stream().map(Role::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder roles(Collection<Role> roles) {
            this.roles = RolesCopier.copy(roles);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder roles(Role... roles) {
            roles(Arrays.asList(roles));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder roles(Consumer<Role.Builder>... roles) {
            roles(Stream.of(roles).map(c -> Role.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setRoles(Collection<Role.BuilderImpl> roles) {
            this.roles = RolesCopier.copyFromBuilder(roles);
        }

        public final Collection<Delegation.Builder> getDelegations() {
            if (delegations instanceof SdkAutoConstructList) {
                return null;
            }
            return delegations != null ? delegations.stream().map(Delegation::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder delegations(Collection<Delegation> delegations) {
            this.delegations = DelegationsCopier.copy(delegations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder delegations(Delegation... delegations) {
            delegations(Arrays.asList(delegations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder delegations(Consumer<Delegation.Builder>... delegations) {
            delegations(Stream.of(delegations).map(c -> Delegation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setDelegations(Collection<Delegation.BuilderImpl> delegations) {
            this.delegations = DelegationsCopier.copyFromBuilder(delegations);
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        public final Instant getLastUpdated() {
            return lastUpdated;
        }

        @Override
        public final Builder lastUpdated(Instant lastUpdated) {
            this.lastUpdated = lastUpdated;
            return this;
        }

        public final void setLastUpdated(Instant lastUpdated) {
            this.lastUpdated = lastUpdated;
        }

        @Override
        public AssessmentMetadataItem build() {
            return new AssessmentMetadataItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
