/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.auditmanager.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a set of controls in an AWS Audit Manager assessment.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AssessmentControlSet implements SdkPojo, Serializable,
        ToCopyableBuilder<AssessmentControlSet.Builder, AssessmentControlSet> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(AssessmentControlSet::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(AssessmentControlSet::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(AssessmentControlSet::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<List<Role>> ROLES_FIELD = SdkField
            .<List<Role>> builder(MarshallingType.LIST)
            .memberName("roles")
            .getter(getter(AssessmentControlSet::roles))
            .setter(setter(Builder::roles))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roles").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Role> builder(MarshallingType.SDK_POJO)
                                            .constructor(Role::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<AssessmentControl>> CONTROLS_FIELD = SdkField
            .<List<AssessmentControl>> builder(MarshallingType.LIST)
            .memberName("controls")
            .getter(getter(AssessmentControlSet::controls))
            .setter(setter(Builder::controls))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("controls").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AssessmentControl> builder(MarshallingType.SDK_POJO)
                                            .constructor(AssessmentControl::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Delegation>> DELEGATIONS_FIELD = SdkField
            .<List<Delegation>> builder(MarshallingType.LIST)
            .memberName("delegations")
            .getter(getter(AssessmentControlSet::delegations))
            .setter(setter(Builder::delegations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("delegations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Delegation> builder(MarshallingType.SDK_POJO)
                                            .constructor(Delegation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> SYSTEM_EVIDENCE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("systemEvidenceCount").getter(getter(AssessmentControlSet::systemEvidenceCount))
            .setter(setter(Builder::systemEvidenceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("systemEvidenceCount").build())
            .build();

    private static final SdkField<Integer> MANUAL_EVIDENCE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("manualEvidenceCount").getter(getter(AssessmentControlSet::manualEvidenceCount))
            .setter(setter(Builder::manualEvidenceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("manualEvidenceCount").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, DESCRIPTION_FIELD,
            STATUS_FIELD, ROLES_FIELD, CONTROLS_FIELD, DELEGATIONS_FIELD, SYSTEM_EVIDENCE_COUNT_FIELD,
            MANUAL_EVIDENCE_COUNT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String description;

    private final String status;

    private final List<Role> roles;

    private final List<AssessmentControl> controls;

    private final List<Delegation> delegations;

    private final Integer systemEvidenceCount;

    private final Integer manualEvidenceCount;

    private AssessmentControlSet(BuilderImpl builder) {
        this.id = builder.id;
        this.description = builder.description;
        this.status = builder.status;
        this.roles = builder.roles;
        this.controls = builder.controls;
        this.delegations = builder.delegations;
        this.systemEvidenceCount = builder.systemEvidenceCount;
        this.manualEvidenceCount = builder.manualEvidenceCount;
    }

    /**
     * <p>
     * The identifier of the control set in the assessment. This is the control set name in a plain string format.
     * </p>
     * 
     * @return The identifier of the control set in the assessment. This is the control set name in a plain string
     *         format.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The description for the control set.
     * </p>
     * 
     * @return The description for the control set.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * Specifies the current status of the control set.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ControlSetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Specifies the current status of the control set.
     * @see ControlSetStatus
     */
    public final ControlSetStatus status() {
        return ControlSetStatus.fromValue(status);
    }

    /**
     * <p>
     * Specifies the current status of the control set.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ControlSetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Specifies the current status of the control set.
     * @see ControlSetStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * Returns true if the Roles property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasRoles() {
        return roles != null && !(roles instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The roles associated with the control set.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRoles()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The roles associated with the control set.
     */
    public final List<Role> roles() {
        return roles;
    }

    /**
     * Returns true if the Controls property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasControls() {
        return controls != null && !(controls instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of controls contained with the control set.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasControls()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The list of controls contained with the control set.
     */
    public final List<AssessmentControl> controls() {
        return controls;
    }

    /**
     * Returns true if the Delegations property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasDelegations() {
        return delegations != null && !(delegations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The delegations associated with the control set.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDelegations()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The delegations associated with the control set.
     */
    public final List<Delegation> delegations() {
        return delegations;
    }

    /**
     * <p>
     * The total number of evidence objects retrieved automatically for the control set.
     * </p>
     * 
     * @return The total number of evidence objects retrieved automatically for the control set.
     */
    public final Integer systemEvidenceCount() {
        return systemEvidenceCount;
    }

    /**
     * <p>
     * The total number of evidence objects uploaded manually to the control set.
     * </p>
     * 
     * @return The total number of evidence objects uploaded manually to the control set.
     */
    public final Integer manualEvidenceCount() {
        return manualEvidenceCount;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasRoles() ? roles() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasControls() ? controls() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDelegations() ? delegations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(systemEvidenceCount());
        hashCode = 31 * hashCode + Objects.hashCode(manualEvidenceCount());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AssessmentControlSet)) {
            return false;
        }
        AssessmentControlSet other = (AssessmentControlSet) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(description(), other.description())
                && Objects.equals(statusAsString(), other.statusAsString()) && hasRoles() == other.hasRoles()
                && Objects.equals(roles(), other.roles()) && hasControls() == other.hasControls()
                && Objects.equals(controls(), other.controls()) && hasDelegations() == other.hasDelegations()
                && Objects.equals(delegations(), other.delegations())
                && Objects.equals(systemEvidenceCount(), other.systemEvidenceCount())
                && Objects.equals(manualEvidenceCount(), other.manualEvidenceCount());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AssessmentControlSet").add("Id", id()).add("Description", description())
                .add("Status", statusAsString()).add("Roles", hasRoles() ? roles() : null)
                .add("Controls", hasControls() ? controls() : null).add("Delegations", hasDelegations() ? delegations() : null)
                .add("SystemEvidenceCount", systemEvidenceCount()).add("ManualEvidenceCount", manualEvidenceCount()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "roles":
            return Optional.ofNullable(clazz.cast(roles()));
        case "controls":
            return Optional.ofNullable(clazz.cast(controls()));
        case "delegations":
            return Optional.ofNullable(clazz.cast(delegations()));
        case "systemEvidenceCount":
            return Optional.ofNullable(clazz.cast(systemEvidenceCount()));
        case "manualEvidenceCount":
            return Optional.ofNullable(clazz.cast(manualEvidenceCount()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AssessmentControlSet, T> g) {
        return obj -> g.apply((AssessmentControlSet) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AssessmentControlSet> {
        /**
         * <p>
         * The identifier of the control set in the assessment. This is the control set name in a plain string format.
         * </p>
         * 
         * @param id
         *        The identifier of the control set in the assessment. This is the control set name in a plain string
         *        format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The description for the control set.
         * </p>
         * 
         * @param description
         *        The description for the control set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Specifies the current status of the control set.
         * </p>
         * 
         * @param status
         *        Specifies the current status of the control set.
         * @see ControlSetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ControlSetStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Specifies the current status of the control set.
         * </p>
         * 
         * @param status
         *        Specifies the current status of the control set.
         * @see ControlSetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ControlSetStatus
         */
        Builder status(ControlSetStatus status);

        /**
         * <p>
         * The roles associated with the control set.
         * </p>
         * 
         * @param roles
         *        The roles associated with the control set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roles(Collection<Role> roles);

        /**
         * <p>
         * The roles associated with the control set.
         * </p>
         * 
         * @param roles
         *        The roles associated with the control set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roles(Role... roles);

        /**
         * <p>
         * The roles associated with the control set.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Role>.Builder} avoiding the need to create
         * one manually via {@link List<Role>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Role>.Builder#build()} is called immediately and its result
         * is passed to {@link #roles(List<Role>)}.
         * 
         * @param roles
         *        a consumer that will call methods on {@link List<Role>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #roles(List<Role>)
         */
        Builder roles(Consumer<Role.Builder>... roles);

        /**
         * <p>
         * The list of controls contained with the control set.
         * </p>
         * 
         * @param controls
         *        The list of controls contained with the control set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder controls(Collection<AssessmentControl> controls);

        /**
         * <p>
         * The list of controls contained with the control set.
         * </p>
         * 
         * @param controls
         *        The list of controls contained with the control set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder controls(AssessmentControl... controls);

        /**
         * <p>
         * The list of controls contained with the control set.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AssessmentControl>.Builder} avoiding the
         * need to create one manually via {@link List<AssessmentControl>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AssessmentControl>.Builder#build()} is called immediately
         * and its result is passed to {@link #controls(List<AssessmentControl>)}.
         * 
         * @param controls
         *        a consumer that will call methods on {@link List<AssessmentControl>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #controls(List<AssessmentControl>)
         */
        Builder controls(Consumer<AssessmentControl.Builder>... controls);

        /**
         * <p>
         * The delegations associated with the control set.
         * </p>
         * 
         * @param delegations
         *        The delegations associated with the control set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder delegations(Collection<Delegation> delegations);

        /**
         * <p>
         * The delegations associated with the control set.
         * </p>
         * 
         * @param delegations
         *        The delegations associated with the control set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder delegations(Delegation... delegations);

        /**
         * <p>
         * The delegations associated with the control set.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Delegation>.Builder} avoiding the need to
         * create one manually via {@link List<Delegation>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Delegation>.Builder#build()} is called immediately and its
         * result is passed to {@link #delegations(List<Delegation>)}.
         * 
         * @param delegations
         *        a consumer that will call methods on {@link List<Delegation>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #delegations(List<Delegation>)
         */
        Builder delegations(Consumer<Delegation.Builder>... delegations);

        /**
         * <p>
         * The total number of evidence objects retrieved automatically for the control set.
         * </p>
         * 
         * @param systemEvidenceCount
         *        The total number of evidence objects retrieved automatically for the control set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder systemEvidenceCount(Integer systemEvidenceCount);

        /**
         * <p>
         * The total number of evidence objects uploaded manually to the control set.
         * </p>
         * 
         * @param manualEvidenceCount
         *        The total number of evidence objects uploaded manually to the control set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder manualEvidenceCount(Integer manualEvidenceCount);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String description;

        private String status;

        private List<Role> roles = DefaultSdkAutoConstructList.getInstance();

        private List<AssessmentControl> controls = DefaultSdkAutoConstructList.getInstance();

        private List<Delegation> delegations = DefaultSdkAutoConstructList.getInstance();

        private Integer systemEvidenceCount;

        private Integer manualEvidenceCount;

        private BuilderImpl() {
        }

        private BuilderImpl(AssessmentControlSet model) {
            id(model.id);
            description(model.description);
            status(model.status);
            roles(model.roles);
            controls(model.controls);
            delegations(model.delegations);
            systemEvidenceCount(model.systemEvidenceCount);
            manualEvidenceCount(model.manualEvidenceCount);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ControlSetStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Collection<Role.Builder> getRoles() {
            if (roles instanceof SdkAutoConstructList) {
                return null;
            }
            return roles != null ? roles.stream().map(Role::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder roles(Collection<Role> roles) {
            this.roles = RolesCopier.copy(roles);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder roles(Role... roles) {
            roles(Arrays.asList(roles));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder roles(Consumer<Role.Builder>... roles) {
            roles(Stream.of(roles).map(c -> Role.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setRoles(Collection<Role.BuilderImpl> roles) {
            this.roles = RolesCopier.copyFromBuilder(roles);
        }

        public final Collection<AssessmentControl.Builder> getControls() {
            if (controls instanceof SdkAutoConstructList) {
                return null;
            }
            return controls != null ? controls.stream().map(AssessmentControl::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder controls(Collection<AssessmentControl> controls) {
            this.controls = AssessmentControlsCopier.copy(controls);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder controls(AssessmentControl... controls) {
            controls(Arrays.asList(controls));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder controls(Consumer<AssessmentControl.Builder>... controls) {
            controls(Stream.of(controls).map(c -> AssessmentControl.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setControls(Collection<AssessmentControl.BuilderImpl> controls) {
            this.controls = AssessmentControlsCopier.copyFromBuilder(controls);
        }

        public final Collection<Delegation.Builder> getDelegations() {
            if (delegations instanceof SdkAutoConstructList) {
                return null;
            }
            return delegations != null ? delegations.stream().map(Delegation::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder delegations(Collection<Delegation> delegations) {
            this.delegations = DelegationsCopier.copy(delegations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder delegations(Delegation... delegations) {
            delegations(Arrays.asList(delegations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder delegations(Consumer<Delegation.Builder>... delegations) {
            delegations(Stream.of(delegations).map(c -> Delegation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setDelegations(Collection<Delegation.BuilderImpl> delegations) {
            this.delegations = DelegationsCopier.copyFromBuilder(delegations);
        }

        public final Integer getSystemEvidenceCount() {
            return systemEvidenceCount;
        }

        @Override
        public final Builder systemEvidenceCount(Integer systemEvidenceCount) {
            this.systemEvidenceCount = systemEvidenceCount;
            return this;
        }

        public final void setSystemEvidenceCount(Integer systemEvidenceCount) {
            this.systemEvidenceCount = systemEvidenceCount;
        }

        public final Integer getManualEvidenceCount() {
            return manualEvidenceCount;
        }

        @Override
        public final Builder manualEvidenceCount(Integer manualEvidenceCount) {
            this.manualEvidenceCount = manualEvidenceCount;
            return this;
        }

        public final void setManualEvidenceCount(Integer manualEvidenceCount) {
            this.manualEvidenceCount = manualEvidenceCount;
        }

        @Override
        public AssessmentControlSet build() {
            return new AssessmentControlSet(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
