/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.auditmanager.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A keyword that relates to the control data source.
 * </p>
 * <p>
 * For manual evidence, this keyword indicates if the manual evidence is a file or text.
 * </p>
 * <p>
 * For automated evidence, this keyword identifies a specific CloudTrail event, Config rule, Security Hub control, or
 * Amazon Web Services API name.
 * </p>
 * <p>
 * To learn more about the supported keywords that you can use when mapping a control data source, see the following
 * pages in the <i>Audit Manager User Guide</i>:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/audit-manager/latest/userguide/control-data-sources-config.html">Config rules
 * supported by Audit Manager</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/audit-manager/latest/userguide/control-data-sources-ash.html">Security Hub
 * controls supported by Audit Manager</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/audit-manager/latest/userguide/control-data-sources-api.html">API calls
 * supported by Audit Manager</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/audit-manager/latest/userguide/control-data-sources-cloudtrail.html">CloudTrail
 * event names supported by Audit Manager</a>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SourceKeyword implements SdkPojo, Serializable, ToCopyableBuilder<SourceKeyword.Builder, SourceKeyword> {
    private static final SdkField<String> KEYWORD_INPUT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("keywordInputType").getter(getter(SourceKeyword::keywordInputTypeAsString))
            .setter(setter(Builder::keywordInputType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("keywordInputType").build()).build();

    private static final SdkField<String> KEYWORD_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("keywordValue").getter(getter(SourceKeyword::keywordValue)).setter(setter(Builder::keywordValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("keywordValue").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KEYWORD_INPUT_TYPE_FIELD,
            KEYWORD_VALUE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String keywordInputType;

    private final String keywordValue;

    private SourceKeyword(BuilderImpl builder) {
        this.keywordInputType = builder.keywordInputType;
        this.keywordValue = builder.keywordValue;
    }

    /**
     * <p>
     * The input method for the keyword.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SELECT_FROM_LIST</code> is used when mapping a data source for automated evidence.
     * </p>
     * <ul>
     * <li>
     * <p>
     * When <code>keywordInputType</code> is <code>SELECT_FROM_LIST</code>, a keyword must be selected to collect
     * automated evidence. For example, this keyword can be a CloudTrail event name, a rule name for Config, a Security
     * Hub control, or the name of an Amazon Web Services API call.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>UPLOAD_FILE</code> and <code>INPUT_TEXT</code> are only used when mapping a data source for manual
     * evidence.
     * </p>
     * <ul>
     * <li>
     * <p>
     * When <code>keywordInputType</code> is <code>UPLOAD_FILE</code>, a file must be uploaded as manual evidence.
     * </p>
     * </li>
     * <li>
     * <p>
     * When <code>keywordInputType</code> is <code>INPUT_TEXT</code>, text must be entered as manual evidence.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keywordInputType}
     * will return {@link KeywordInputType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #keywordInputTypeAsString}.
     * </p>
     * 
     * @return The input method for the keyword. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SELECT_FROM_LIST</code> is used when mapping a data source for automated evidence.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         When <code>keywordInputType</code> is <code>SELECT_FROM_LIST</code>, a keyword must be selected to
     *         collect automated evidence. For example, this keyword can be a CloudTrail event name, a rule name for
     *         Config, a Security Hub control, or the name of an Amazon Web Services API call.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPLOAD_FILE</code> and <code>INPUT_TEXT</code> are only used when mapping a data source for manual
     *         evidence.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         When <code>keywordInputType</code> is <code>UPLOAD_FILE</code>, a file must be uploaded as manual
     *         evidence.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         When <code>keywordInputType</code> is <code>INPUT_TEXT</code>, text must be entered as manual evidence.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     * @see KeywordInputType
     */
    public final KeywordInputType keywordInputType() {
        return KeywordInputType.fromValue(keywordInputType);
    }

    /**
     * <p>
     * The input method for the keyword.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SELECT_FROM_LIST</code> is used when mapping a data source for automated evidence.
     * </p>
     * <ul>
     * <li>
     * <p>
     * When <code>keywordInputType</code> is <code>SELECT_FROM_LIST</code>, a keyword must be selected to collect
     * automated evidence. For example, this keyword can be a CloudTrail event name, a rule name for Config, a Security
     * Hub control, or the name of an Amazon Web Services API call.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>UPLOAD_FILE</code> and <code>INPUT_TEXT</code> are only used when mapping a data source for manual
     * evidence.
     * </p>
     * <ul>
     * <li>
     * <p>
     * When <code>keywordInputType</code> is <code>UPLOAD_FILE</code>, a file must be uploaded as manual evidence.
     * </p>
     * </li>
     * <li>
     * <p>
     * When <code>keywordInputType</code> is <code>INPUT_TEXT</code>, text must be entered as manual evidence.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keywordInputType}
     * will return {@link KeywordInputType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #keywordInputTypeAsString}.
     * </p>
     * 
     * @return The input method for the keyword. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SELECT_FROM_LIST</code> is used when mapping a data source for automated evidence.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         When <code>keywordInputType</code> is <code>SELECT_FROM_LIST</code>, a keyword must be selected to
     *         collect automated evidence. For example, this keyword can be a CloudTrail event name, a rule name for
     *         Config, a Security Hub control, or the name of an Amazon Web Services API call.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPLOAD_FILE</code> and <code>INPUT_TEXT</code> are only used when mapping a data source for manual
     *         evidence.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         When <code>keywordInputType</code> is <code>UPLOAD_FILE</code>, a file must be uploaded as manual
     *         evidence.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         When <code>keywordInputType</code> is <code>INPUT_TEXT</code>, text must be entered as manual evidence.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     * @see KeywordInputType
     */
    public final String keywordInputTypeAsString() {
        return keywordInputType;
    }

    /**
     * <p>
     * The value of the keyword that's used when mapping a control data source. For example, this can be a CloudTrail
     * event name, a rule name for Config, a Security Hub control, or the name of an Amazon Web Services API call.
     * </p>
     * <p>
     * If you’re mapping a data source to a rule in Config, the <code>keywordValue</code> that you specify depends on
     * the type of rule:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For <a
     * href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_use-managed-rules.html">managed
     * rules</a>, you can use the rule identifier as the <code>keywordValue</code>. You can find the rule identifier
     * from the <a href="https://docs.aws.amazon.com/config/latest/developerguide/managed-rules-by-aws-config.html">list
     * of Config managed rules</a>. For some rules, the rule identifier is different from the rule name. For example,
     * the rule name <code>restricted-ssh</code> has the following rule identifier: <code>INCOMING_SSH_DISABLED</code>.
     * Make sure to use the rule identifier, not the rule name.
     * </p>
     * <p>
     * Keyword example for managed rules:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Managed rule name: <a
     * href="https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-acl-prohibited.html"
     * >s3-bucket-acl-prohibited</a>
     * </p>
     * <p>
     * <code>keywordValue</code>: <code>S3_BUCKET_ACL_PROHIBITED</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * For <a href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_develop-rules.html">custom
     * rules</a>, you form the <code>keywordValue</code> by adding the <code>Custom_</code> prefix to the rule name.
     * This prefix distinguishes the custom rule from a managed rule.
     * </p>
     * <p>
     * Keyword example for custom rules:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Custom rule name: my-custom-config-rule
     * </p>
     * <p>
     * <code>keywordValue</code>: <code>Custom_my-custom-config-rule</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * For <a
     * href="https://docs.aws.amazon.com/config/latest/developerguide/service-linked-awsconfig-rules.html">service
     * -linked rules</a>, you form the <code>keywordValue</code> by adding the <code>Custom_</code> prefix to the rule
     * name. In addition, you remove the suffix ID that appears at the end of the rule name.
     * </p>
     * <p>
     * Keyword examples for service-linked rules:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Service-linked rule name: CustomRuleForAccount-conformance-pack-szsm1uv0w
     * </p>
     * <p>
     * <code>keywordValue</code>: <code>Custom_CustomRuleForAccount-conformance-pack</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Service-linked rule name: OrgConfigRule-s3-bucket-versioning-enabled-dbgzf8ba
     * </p>
     * <p>
     * <code>keywordValue</code>: <code>Custom_OrgConfigRule-s3-bucket-versioning-enabled</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <important>
     * <p>
     * The <code>keywordValue</code> is case sensitive. If you enter a value incorrectly, Audit Manager might not
     * recognize the data source mapping. As a result, you might not successfully collect evidence from that data source
     * as intended.
     * </p>
     * <p>
     * Keep in mind the following requirements, depending on the data source type that you're using.
     * </p>
     * <ol>
     * <li>
     * <p>
     * For Config:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For managed rules, make sure that the <code>keywordValue</code> is the rule identifier in
     * <code>ALL_CAPS_WITH_UNDERSCORES</code>. For example, <code>CLOUDWATCH_LOG_GROUP_ENCRYPTED</code>. For accuracy,
     * we recommend that you reference the list of <a
     * href="https://docs.aws.amazon.com/audit-manager/latest/userguide/control-data-sources-config.html">supported
     * Config managed rules</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For custom rules, make sure that the <code>keywordValue</code> has the <code>Custom_</code> prefix followed by
     * the custom rule name. The format of the custom rule name itself may vary. For accuracy, we recommend that you
     * visit the <a href="https://console.aws.amazon.com/config/">Config console</a> to verify your custom rule name.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * For Security Hub: The format varies for Security Hub control names. For accuracy, we recommend that you reference
     * the list of <a
     * href="https://docs.aws.amazon.com/audit-manager/latest/userguide/control-data-sources-ash.html">supported
     * Security Hub controls</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amazon Web Services API calls: Make sure that the <code>keywordValue</code> is written as
     * <code>serviceprefix_ActionName</code>. For example, <code>iam_ListGroups</code>. For accuracy, we recommend that
     * you reference the list of <a
     * href="https://docs.aws.amazon.com/audit-manager/latest/userguide/control-data-sources-api.html">supported API
     * calls</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For CloudTrail: Make sure that the <code>keywordValue</code> is written as <code>serviceprefix_ActionName</code>.
     * For example, <code>cloudtrail_StartLogging</code>. For accuracy, we recommend that you review the Amazon Web
     * Service prefix and action names in the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/reference_policies_actions-resources-contextkeys.html"
     * >Service Authorization Reference</a>.
     * </p>
     * </li>
     * </ol>
     * </important>
     * 
     * @return The value of the keyword that's used when mapping a control data source. For example, this can be a
     *         CloudTrail event name, a rule name for Config, a Security Hub control, or the name of an Amazon Web
     *         Services API call. </p>
     *         <p>
     *         If you’re mapping a data source to a rule in Config, the <code>keywordValue</code> that you specify
     *         depends on the type of rule:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For <a
     *         href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_use-managed-rules.html"
     *         >managed rules</a>, you can use the rule identifier as the <code>keywordValue</code>. You can find the
     *         rule identifier from the <a
     *         href="https://docs.aws.amazon.com/config/latest/developerguide/managed-rules-by-aws-config.html">list of
     *         Config managed rules</a>. For some rules, the rule identifier is different from the rule name. For
     *         example, the rule name <code>restricted-ssh</code> has the following rule identifier:
     *         <code>INCOMING_SSH_DISABLED</code>. Make sure to use the rule identifier, not the rule name.
     *         </p>
     *         <p>
     *         Keyword example for managed rules:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Managed rule name: <a
     *         href="https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-acl-prohibited.html"
     *         >s3-bucket-acl-prohibited</a>
     *         </p>
     *         <p>
     *         <code>keywordValue</code>: <code>S3_BUCKET_ACL_PROHIBITED</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         For <a
     *         href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_develop-rules.html">custom
     *         rules</a>, you form the <code>keywordValue</code> by adding the <code>Custom_</code> prefix to the rule
     *         name. This prefix distinguishes the custom rule from a managed rule.
     *         </p>
     *         <p>
     *         Keyword example for custom rules:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Custom rule name: my-custom-config-rule
     *         </p>
     *         <p>
     *         <code>keywordValue</code>: <code>Custom_my-custom-config-rule</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         For <a
     *         href="https://docs.aws.amazon.com/config/latest/developerguide/service-linked-awsconfig-rules.html"
     *         >service-linked rules</a>, you form the <code>keywordValue</code> by adding the <code>Custom_</code>
     *         prefix to the rule name. In addition, you remove the suffix ID that appears at the end of the rule name.
     *         </p>
     *         <p>
     *         Keyword examples for service-linked rules:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Service-linked rule name: CustomRuleForAccount-conformance-pack-szsm1uv0w
     *         </p>
     *         <p>
     *         <code>keywordValue</code>: <code>Custom_CustomRuleForAccount-conformance-pack</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Service-linked rule name: OrgConfigRule-s3-bucket-versioning-enabled-dbgzf8ba
     *         </p>
     *         <p>
     *         <code>keywordValue</code>: <code>Custom_OrgConfigRule-s3-bucket-versioning-enabled</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         <important>
     *         <p>
     *         The <code>keywordValue</code> is case sensitive. If you enter a value incorrectly, Audit Manager might
     *         not recognize the data source mapping. As a result, you might not successfully collect evidence from that
     *         data source as intended.
     *         </p>
     *         <p>
     *         Keep in mind the following requirements, depending on the data source type that you're using.
     *         </p>
     *         <ol>
     *         <li>
     *         <p>
     *         For Config:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For managed rules, make sure that the <code>keywordValue</code> is the rule identifier in
     *         <code>ALL_CAPS_WITH_UNDERSCORES</code>. For example, <code>CLOUDWATCH_LOG_GROUP_ENCRYPTED</code>. For
     *         accuracy, we recommend that you reference the list of <a
     *         href="https://docs.aws.amazon.com/audit-manager/latest/userguide/control-data-sources-config.html"
     *         >supported Config managed rules</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For custom rules, make sure that the <code>keywordValue</code> has the <code>Custom_</code> prefix
     *         followed by the custom rule name. The format of the custom rule name itself may vary. For accuracy, we
     *         recommend that you visit the <a href="https://console.aws.amazon.com/config/">Config console</a> to
     *         verify your custom rule name.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         For Security Hub: The format varies for Security Hub control names. For accuracy, we recommend that you
     *         reference the list of <a
     *         href="https://docs.aws.amazon.com/audit-manager/latest/userguide/control-data-sources-ash.html">supported
     *         Security Hub controls</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For Amazon Web Services API calls: Make sure that the <code>keywordValue</code> is written as
     *         <code>serviceprefix_ActionName</code>. For example, <code>iam_ListGroups</code>. For accuracy, we
     *         recommend that you reference the list of <a
     *         href="https://docs.aws.amazon.com/audit-manager/latest/userguide/control-data-sources-api.html">supported
     *         API calls</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For CloudTrail: Make sure that the <code>keywordValue</code> is written as
     *         <code>serviceprefix_ActionName</code>. For example, <code>cloudtrail_StartLogging</code>. For accuracy,
     *         we recommend that you review the Amazon Web Service prefix and action names in the <a href=
     *         "https://docs.aws.amazon.com/service-authorization/latest/reference/reference_policies_actions-resources-contextkeys.html"
     *         >Service Authorization Reference</a>.
     *         </p>
     *         </li>
     *         </ol>
     */
    public final String keywordValue() {
        return keywordValue;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(keywordInputTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(keywordValue());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SourceKeyword)) {
            return false;
        }
        SourceKeyword other = (SourceKeyword) obj;
        return Objects.equals(keywordInputTypeAsString(), other.keywordInputTypeAsString())
                && Objects.equals(keywordValue(), other.keywordValue());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SourceKeyword").add("KeywordInputType", keywordInputTypeAsString())
                .add("KeywordValue", keywordValue()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "keywordInputType":
            return Optional.ofNullable(clazz.cast(keywordInputTypeAsString()));
        case "keywordValue":
            return Optional.ofNullable(clazz.cast(keywordValue()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SourceKeyword, T> g) {
        return obj -> g.apply((SourceKeyword) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SourceKeyword> {
        /**
         * <p>
         * The input method for the keyword.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SELECT_FROM_LIST</code> is used when mapping a data source for automated evidence.
         * </p>
         * <ul>
         * <li>
         * <p>
         * When <code>keywordInputType</code> is <code>SELECT_FROM_LIST</code>, a keyword must be selected to collect
         * automated evidence. For example, this keyword can be a CloudTrail event name, a rule name for Config, a
         * Security Hub control, or the name of an Amazon Web Services API call.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>UPLOAD_FILE</code> and <code>INPUT_TEXT</code> are only used when mapping a data source for manual
         * evidence.
         * </p>
         * <ul>
         * <li>
         * <p>
         * When <code>keywordInputType</code> is <code>UPLOAD_FILE</code>, a file must be uploaded as manual evidence.
         * </p>
         * </li>
         * <li>
         * <p>
         * When <code>keywordInputType</code> is <code>INPUT_TEXT</code>, text must be entered as manual evidence.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param keywordInputType
         *        The input method for the keyword. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SELECT_FROM_LIST</code> is used when mapping a data source for automated evidence.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        When <code>keywordInputType</code> is <code>SELECT_FROM_LIST</code>, a keyword must be selected to
         *        collect automated evidence. For example, this keyword can be a CloudTrail event name, a rule name for
         *        Config, a Security Hub control, or the name of an Amazon Web Services API call.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPLOAD_FILE</code> and <code>INPUT_TEXT</code> are only used when mapping a data source for
         *        manual evidence.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        When <code>keywordInputType</code> is <code>UPLOAD_FILE</code>, a file must be uploaded as manual
         *        evidence.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        When <code>keywordInputType</code> is <code>INPUT_TEXT</code>, text must be entered as manual
         *        evidence.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @see KeywordInputType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeywordInputType
         */
        Builder keywordInputType(String keywordInputType);

        /**
         * <p>
         * The input method for the keyword.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SELECT_FROM_LIST</code> is used when mapping a data source for automated evidence.
         * </p>
         * <ul>
         * <li>
         * <p>
         * When <code>keywordInputType</code> is <code>SELECT_FROM_LIST</code>, a keyword must be selected to collect
         * automated evidence. For example, this keyword can be a CloudTrail event name, a rule name for Config, a
         * Security Hub control, or the name of an Amazon Web Services API call.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>UPLOAD_FILE</code> and <code>INPUT_TEXT</code> are only used when mapping a data source for manual
         * evidence.
         * </p>
         * <ul>
         * <li>
         * <p>
         * When <code>keywordInputType</code> is <code>UPLOAD_FILE</code>, a file must be uploaded as manual evidence.
         * </p>
         * </li>
         * <li>
         * <p>
         * When <code>keywordInputType</code> is <code>INPUT_TEXT</code>, text must be entered as manual evidence.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param keywordInputType
         *        The input method for the keyword. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SELECT_FROM_LIST</code> is used when mapping a data source for automated evidence.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        When <code>keywordInputType</code> is <code>SELECT_FROM_LIST</code>, a keyword must be selected to
         *        collect automated evidence. For example, this keyword can be a CloudTrail event name, a rule name for
         *        Config, a Security Hub control, or the name of an Amazon Web Services API call.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPLOAD_FILE</code> and <code>INPUT_TEXT</code> are only used when mapping a data source for
         *        manual evidence.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        When <code>keywordInputType</code> is <code>UPLOAD_FILE</code>, a file must be uploaded as manual
         *        evidence.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        When <code>keywordInputType</code> is <code>INPUT_TEXT</code>, text must be entered as manual
         *        evidence.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @see KeywordInputType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeywordInputType
         */
        Builder keywordInputType(KeywordInputType keywordInputType);

        /**
         * <p>
         * The value of the keyword that's used when mapping a control data source. For example, this can be a
         * CloudTrail event name, a rule name for Config, a Security Hub control, or the name of an Amazon Web Services
         * API call.
         * </p>
         * <p>
         * If you’re mapping a data source to a rule in Config, the <code>keywordValue</code> that you specify depends
         * on the type of rule:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For <a
         * href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_use-managed-rules.html">
         * managed rules</a>, you can use the rule identifier as the <code>keywordValue</code>. You can find the rule
         * identifier from the <a
         * href="https://docs.aws.amazon.com/config/latest/developerguide/managed-rules-by-aws-config.html">list of
         * Config managed rules</a>. For some rules, the rule identifier is different from the rule name. For example,
         * the rule name <code>restricted-ssh</code> has the following rule identifier:
         * <code>INCOMING_SSH_DISABLED</code>. Make sure to use the rule identifier, not the rule name.
         * </p>
         * <p>
         * Keyword example for managed rules:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Managed rule name: <a
         * href="https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-acl-prohibited.html"
         * >s3-bucket-acl-prohibited</a>
         * </p>
         * <p>
         * <code>keywordValue</code>: <code>S3_BUCKET_ACL_PROHIBITED</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * For <a
         * href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_develop-rules.html">custom
         * rules</a>, you form the <code>keywordValue</code> by adding the <code>Custom_</code> prefix to the rule name.
         * This prefix distinguishes the custom rule from a managed rule.
         * </p>
         * <p>
         * Keyword example for custom rules:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Custom rule name: my-custom-config-rule
         * </p>
         * <p>
         * <code>keywordValue</code>: <code>Custom_my-custom-config-rule</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * For <a
         * href="https://docs.aws.amazon.com/config/latest/developerguide/service-linked-awsconfig-rules.html">service
         * -linked rules</a>, you form the <code>keywordValue</code> by adding the <code>Custom_</code> prefix to the
         * rule name. In addition, you remove the suffix ID that appears at the end of the rule name.
         * </p>
         * <p>
         * Keyword examples for service-linked rules:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Service-linked rule name: CustomRuleForAccount-conformance-pack-szsm1uv0w
         * </p>
         * <p>
         * <code>keywordValue</code>: <code>Custom_CustomRuleForAccount-conformance-pack</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Service-linked rule name: OrgConfigRule-s3-bucket-versioning-enabled-dbgzf8ba
         * </p>
         * <p>
         * <code>keywordValue</code>: <code>Custom_OrgConfigRule-s3-bucket-versioning-enabled</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * <important>
         * <p>
         * The <code>keywordValue</code> is case sensitive. If you enter a value incorrectly, Audit Manager might not
         * recognize the data source mapping. As a result, you might not successfully collect evidence from that data
         * source as intended.
         * </p>
         * <p>
         * Keep in mind the following requirements, depending on the data source type that you're using.
         * </p>
         * <ol>
         * <li>
         * <p>
         * For Config:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For managed rules, make sure that the <code>keywordValue</code> is the rule identifier in
         * <code>ALL_CAPS_WITH_UNDERSCORES</code>. For example, <code>CLOUDWATCH_LOG_GROUP_ENCRYPTED</code>. For
         * accuracy, we recommend that you reference the list of <a
         * href="https://docs.aws.amazon.com/audit-manager/latest/userguide/control-data-sources-config.html">supported
         * Config managed rules</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For custom rules, make sure that the <code>keywordValue</code> has the <code>Custom_</code> prefix followed
         * by the custom rule name. The format of the custom rule name itself may vary. For accuracy, we recommend that
         * you visit the <a href="https://console.aws.amazon.com/config/">Config console</a> to verify your custom rule
         * name.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * For Security Hub: The format varies for Security Hub control names. For accuracy, we recommend that you
         * reference the list of <a
         * href="https://docs.aws.amazon.com/audit-manager/latest/userguide/control-data-sources-ash.html">supported
         * Security Hub controls</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Amazon Web Services API calls: Make sure that the <code>keywordValue</code> is written as
         * <code>serviceprefix_ActionName</code>. For example, <code>iam_ListGroups</code>. For accuracy, we recommend
         * that you reference the list of <a
         * href="https://docs.aws.amazon.com/audit-manager/latest/userguide/control-data-sources-api.html">supported API
         * calls</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For CloudTrail: Make sure that the <code>keywordValue</code> is written as
         * <code>serviceprefix_ActionName</code>. For example, <code>cloudtrail_StartLogging</code>. For accuracy, we
         * recommend that you review the Amazon Web Service prefix and action names in the <a href=
         * "https://docs.aws.amazon.com/service-authorization/latest/reference/reference_policies_actions-resources-contextkeys.html"
         * >Service Authorization Reference</a>.
         * </p>
         * </li>
         * </ol>
         * </important>
         * 
         * @param keywordValue
         *        The value of the keyword that's used when mapping a control data source. For example, this can be a
         *        CloudTrail event name, a rule name for Config, a Security Hub control, or the name of an Amazon Web
         *        Services API call. </p>
         *        <p>
         *        If you’re mapping a data source to a rule in Config, the <code>keywordValue</code> that you specify
         *        depends on the type of rule:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For <a href=
         *        "https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_use-managed-rules.html"
         *        >managed rules</a>, you can use the rule identifier as the <code>keywordValue</code>. You can find the
         *        rule identifier from the <a
         *        href="https://docs.aws.amazon.com/config/latest/developerguide/managed-rules-by-aws-config.html">list
         *        of Config managed rules</a>. For some rules, the rule identifier is different from the rule name. For
         *        example, the rule name <code>restricted-ssh</code> has the following rule identifier:
         *        <code>INCOMING_SSH_DISABLED</code>. Make sure to use the rule identifier, not the rule name.
         *        </p>
         *        <p>
         *        Keyword example for managed rules:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Managed rule name: <a
         *        href="https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-acl-prohibited.html"
         *        >s3-bucket-acl-prohibited</a>
         *        </p>
         *        <p>
         *        <code>keywordValue</code>: <code>S3_BUCKET_ACL_PROHIBITED</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        For <a
         *        href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_develop-rules.html"
         *        >custom rules</a>, you form the <code>keywordValue</code> by adding the <code>Custom_</code> prefix to
         *        the rule name. This prefix distinguishes the custom rule from a managed rule.
         *        </p>
         *        <p>
         *        Keyword example for custom rules:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Custom rule name: my-custom-config-rule
         *        </p>
         *        <p>
         *        <code>keywordValue</code>: <code>Custom_my-custom-config-rule</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        For <a
         *        href="https://docs.aws.amazon.com/config/latest/developerguide/service-linked-awsconfig-rules.html"
         *        >service-linked rules</a>, you form the <code>keywordValue</code> by adding the <code>Custom_</code>
         *        prefix to the rule name. In addition, you remove the suffix ID that appears at the end of the rule
         *        name.
         *        </p>
         *        <p>
         *        Keyword examples for service-linked rules:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Service-linked rule name: CustomRuleForAccount-conformance-pack-szsm1uv0w
         *        </p>
         *        <p>
         *        <code>keywordValue</code>: <code>Custom_CustomRuleForAccount-conformance-pack</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Service-linked rule name: OrgConfigRule-s3-bucket-versioning-enabled-dbgzf8ba
         *        </p>
         *        <p>
         *        <code>keywordValue</code>: <code>Custom_OrgConfigRule-s3-bucket-versioning-enabled</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        <important>
         *        <p>
         *        The <code>keywordValue</code> is case sensitive. If you enter a value incorrectly, Audit Manager might
         *        not recognize the data source mapping. As a result, you might not successfully collect evidence from
         *        that data source as intended.
         *        </p>
         *        <p>
         *        Keep in mind the following requirements, depending on the data source type that you're using.
         *        </p>
         *        <ol>
         *        <li>
         *        <p>
         *        For Config:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For managed rules, make sure that the <code>keywordValue</code> is the rule identifier in
         *        <code>ALL_CAPS_WITH_UNDERSCORES</code>. For example, <code>CLOUDWATCH_LOG_GROUP_ENCRYPTED</code>. For
         *        accuracy, we recommend that you reference the list of <a
         *        href="https://docs.aws.amazon.com/audit-manager/latest/userguide/control-data-sources-config.html"
         *        >supported Config managed rules</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For custom rules, make sure that the <code>keywordValue</code> has the <code>Custom_</code> prefix
         *        followed by the custom rule name. The format of the custom rule name itself may vary. For accuracy, we
         *        recommend that you visit the <a href="https://console.aws.amazon.com/config/">Config console</a> to
         *        verify your custom rule name.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        For Security Hub: The format varies for Security Hub control names. For accuracy, we recommend that
         *        you reference the list of <a
         *        href="https://docs.aws.amazon.com/audit-manager/latest/userguide/control-data-sources-ash.html"
         *        >supported Security Hub controls</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Amazon Web Services API calls: Make sure that the <code>keywordValue</code> is written as
         *        <code>serviceprefix_ActionName</code>. For example, <code>iam_ListGroups</code>. For accuracy, we
         *        recommend that you reference the list of <a
         *        href="https://docs.aws.amazon.com/audit-manager/latest/userguide/control-data-sources-api.html"
         *        >supported API calls</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For CloudTrail: Make sure that the <code>keywordValue</code> is written as
         *        <code>serviceprefix_ActionName</code>. For example, <code>cloudtrail_StartLogging</code>. For
         *        accuracy, we recommend that you review the Amazon Web Service prefix and action names in the <a href=
         *        "https://docs.aws.amazon.com/service-authorization/latest/reference/reference_policies_actions-resources-contextkeys.html"
         *        >Service Authorization Reference</a>.
         *        </p>
         *        </li>
         *        </ol>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keywordValue(String keywordValue);
    }

    static final class BuilderImpl implements Builder {
        private String keywordInputType;

        private String keywordValue;

        private BuilderImpl() {
        }

        private BuilderImpl(SourceKeyword model) {
            keywordInputType(model.keywordInputType);
            keywordValue(model.keywordValue);
        }

        public final String getKeywordInputType() {
            return keywordInputType;
        }

        public final void setKeywordInputType(String keywordInputType) {
            this.keywordInputType = keywordInputType;
        }

        @Override
        public final Builder keywordInputType(String keywordInputType) {
            this.keywordInputType = keywordInputType;
            return this;
        }

        @Override
        public final Builder keywordInputType(KeywordInputType keywordInputType) {
            this.keywordInputType(keywordInputType == null ? null : keywordInputType.toString());
            return this;
        }

        public final String getKeywordValue() {
            return keywordValue;
        }

        public final void setKeywordValue(String keywordValue) {
            this.keywordValue = keywordValue;
        }

        @Override
        public final Builder keywordValue(String keywordValue) {
            this.keywordValue = keywordValue;
            return this;
        }

        @Override
        public SourceKeyword build() {
            return new SourceKeyword(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
