/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.auth.credentials;

import java.io.IOException;
import java.net.URI;
import java.time.Clock;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Collections;
import java.util.Map;
import java.util.function.Supplier;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.SdkTestInternalApi;
import software.amazon.awssdk.auth.credentials.AwsCredentials;
import software.amazon.awssdk.auth.credentials.HttpCredentialsProvider;
import software.amazon.awssdk.auth.credentials.internal.Ec2MetadataConfigProvider;
import software.amazon.awssdk.auth.credentials.internal.HttpCredentialsLoader;
import software.amazon.awssdk.auth.credentials.internal.StaticResourcesEndpointProvider;
import software.amazon.awssdk.core.SdkSystemSetting;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.exception.SdkServiceException;
import software.amazon.awssdk.profiles.ProfileFile;
import software.amazon.awssdk.regions.util.HttpResourcesUtils;
import software.amazon.awssdk.regions.util.ResourcesEndpointProvider;
import software.amazon.awssdk.utils.ComparableUtils;
import software.amazon.awssdk.utils.Logger;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.Validate;
import software.amazon.awssdk.utils.cache.CachedSupplier;
import software.amazon.awssdk.utils.cache.NonBlocking;
import software.amazon.awssdk.utils.cache.RefreshResult;

@SdkPublicApi
public final class InstanceProfileCredentialsProvider
implements HttpCredentialsProvider {
    private static final Logger log = Logger.loggerFor(InstanceProfileCredentialsProvider.class);
    private static final String EC2_METADATA_TOKEN_HEADER = "x-aws-ec2-metadata-token";
    private static final String SECURITY_CREDENTIALS_RESOURCE = "/latest/meta-data/iam/security-credentials/";
    private static final String TOKEN_RESOURCE = "/latest/api/token";
    private static final String EC2_METADATA_TOKEN_TTL_HEADER = "x-aws-ec2-metadata-token-ttl-seconds";
    private static final String DEFAULT_TOKEN_TTL = "21600";
    private final Clock clock;
    private final String endpoint;
    private final Ec2MetadataConfigProvider configProvider;
    private final HttpCredentialsLoader httpCredentialsLoader;
    private final CachedSupplier<AwsCredentials> credentialsCache;
    private volatile HttpCredentialsLoader.LoadedCredentials cachedCredentials;

    private InstanceProfileCredentialsProvider(BuilderImpl builder) {
        this.clock = builder.clock;
        this.endpoint = builder.endpoint;
        this.httpCredentialsLoader = HttpCredentialsLoader.create();
        this.configProvider = Ec2MetadataConfigProvider.builder().profileFile(builder.profileFile == null ? null : () -> builder.profileFile).profileName(builder.profileName == null ? null : builder.profileName).build();
        if (Boolean.TRUE.equals(builder.asyncCredentialUpdateEnabled)) {
            Validate.paramNotBlank((CharSequence)builder.asyncThreadName, (String)"asyncThreadName");
            this.credentialsCache = CachedSupplier.builder(this::refreshCredentials).prefetchStrategy((CachedSupplier.PrefetchStrategy)new NonBlocking(builder.asyncThreadName)).build();
        } else {
            this.credentialsCache = CachedSupplier.builder(this::refreshCredentials).build();
        }
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static InstanceProfileCredentialsProvider create() {
        return InstanceProfileCredentialsProvider.builder().build();
    }

    @Override
    public AwsCredentials resolveCredentials() {
        return (AwsCredentials)this.credentialsCache.get();
    }

    private RefreshResult<AwsCredentials> refreshCredentials() {
        HttpCredentialsLoader.LoadedCredentials credentials;
        if (this.isLocalCredentialLoadingDisabled()) {
            throw SdkClientException.create((String)"IMDS credentials have been disabled by environment variable or system property.");
        }
        try {
            this.cachedCredentials = credentials = this.httpCredentialsLoader.loadCredentials(this.createEndpointProvider());
        }
        catch (IOException | RuntimeException e) {
            credentials = this.cachedCredentials;
            if (credentials != null) {
                credentials.getExpiration().ifPresent(expiration -> {
                    Supplier<String> errorMessage = () -> "Failure encountered when attempting to refresh credentials from IMDS.";
                    Instant fifteenMinutesFromNow = Instant.now().plus(15L, ChronoUnit.MINUTES);
                    if (expiration.isBefore(fifteenMinutesFromNow)) {
                        log.warn(errorMessage, (Throwable)e);
                    } else {
                        log.debug(errorMessage, (Throwable)e);
                    }
                });
            }
            throw SdkClientException.create((String)"Failed to load credentials from IMDS.", (Throwable)e);
        }
        return RefreshResult.builder((Object)credentials.getAwsCredentials()).staleTime(null).prefetchTime(this.prefetchTime(credentials.getExpiration().orElse(null))).build();
    }

    private boolean isLocalCredentialLoadingDisabled() {
        return SdkSystemSetting.AWS_EC2_METADATA_DISABLED.getBooleanValueOrThrow();
    }

    private Instant prefetchTime(Instant expiration) {
        Instant now = this.clock.instant();
        if (expiration == null) {
            return now.plus(60L, ChronoUnit.MINUTES);
        }
        Instant sixtyMinutesBeforeExpiration = expiration.minus(60L, ChronoUnit.MINUTES);
        if (now.isBefore(sixtyMinutesBeforeExpiration)) {
            return (Instant)ComparableUtils.minimum((Comparable[])new Instant[]{sixtyMinutesBeforeExpiration, now.plus(60L, ChronoUnit.MINUTES)});
        }
        Instant fiveMinutesBeforeExpiration = expiration.minus(5L, ChronoUnit.MINUTES);
        if (now.isBefore(fiveMinutesBeforeExpiration)) {
            return (Instant)ComparableUtils.minimum((Comparable[])new Instant[]{fiveMinutesBeforeExpiration, now.plus(30L, ChronoUnit.MINUTES)});
        }
        Instant fifteenSecondsBeforeExpiration = expiration.minus(15L, ChronoUnit.SECONDS);
        if (now.isBefore(fifteenSecondsBeforeExpiration)) {
            return fifteenSecondsBeforeExpiration;
        }
        log.warn(() -> "IMDS credential expiration has been extended due to an IMDS availability outage. A refresh of these credentials will be attempted again in 5 minutes.");
        return now.plus(5L, ChronoUnit.MINUTES);
    }

    public void close() {
        this.credentialsCache.close();
    }

    public String toString() {
        return ToString.create((String)"InstanceProfileCredentialsProvider");
    }

    private ResourcesEndpointProvider createEndpointProvider() throws IOException {
        String imdsHostname = this.getImdsEndpoint();
        String token = this.getToken(imdsHostname);
        String[] securityCredentials = this.getSecurityCredentials(imdsHostname, token);
        return new StaticResourcesEndpointProvider(URI.create(imdsHostname + SECURITY_CREDENTIALS_RESOURCE + securityCredentials[0]), this.getTokenHeaders(token));
    }

    private String getImdsEndpoint() {
        if (this.endpoint != null) {
            return this.endpoint;
        }
        return this.configProvider.getEndpoint();
    }

    private String getToken(String imdsHostname) {
        Map<String, String> tokenTtlHeaders = Collections.singletonMap(EC2_METADATA_TOKEN_TTL_HEADER, DEFAULT_TOKEN_TTL);
        StaticResourcesEndpointProvider tokenEndpoint = new StaticResourcesEndpointProvider(this.getTokenEndpoint(imdsHostname), tokenTtlHeaders);
        try {
            return HttpResourcesUtils.instance().readResource((ResourcesEndpointProvider)tokenEndpoint, "PUT");
        }
        catch (SdkServiceException e) {
            if (e.statusCode() == 400) {
                throw SdkClientException.builder().message("Unable to fetch metadata token.").cause((Throwable)e).build();
            }
            log.debug(() -> "Ignoring non-fatal exception while attempting to load metadata token from instance profile.", (Throwable)e);
            return null;
        }
        catch (Exception e) {
            log.debug(() -> "Ignoring non-fatal exception while attempting to load metadata token from instance profile.", (Throwable)e);
            return null;
        }
    }

    private URI getTokenEndpoint(String imdsHostname) {
        String finalHost = imdsHostname;
        if (finalHost.endsWith("/")) {
            finalHost = finalHost.substring(0, finalHost.length() - 1);
        }
        return URI.create(finalHost + TOKEN_RESOURCE);
    }

    private String[] getSecurityCredentials(String imdsHostname, String metadataToken) throws IOException {
        StaticResourcesEndpointProvider securityCredentialsEndpoint = new StaticResourcesEndpointProvider(URI.create(imdsHostname + SECURITY_CREDENTIALS_RESOURCE), this.getTokenHeaders(metadataToken));
        String securityCredentialsList = HttpResourcesUtils.instance().readResource((ResourcesEndpointProvider)securityCredentialsEndpoint);
        String[] securityCredentials = securityCredentialsList.trim().split("\n");
        if (securityCredentials.length == 0) {
            throw SdkClientException.builder().message("Unable to load credentials path").build();
        }
        return securityCredentials;
    }

    private Map<String, String> getTokenHeaders(String metadataToken) {
        if (metadataToken == null) {
            return Collections.emptyMap();
        }
        return Collections.singletonMap(EC2_METADATA_TOKEN_HEADER, metadataToken);
    }

    @SdkTestInternalApi
    static final class BuilderImpl
    implements Builder {
        private Clock clock = Clock.systemUTC();
        private String endpoint;
        private Boolean asyncCredentialUpdateEnabled;
        private String asyncThreadName;
        private ProfileFile profileFile;
        private String profileName;

        private BuilderImpl() {
            this.asyncThreadName("instance-profile-credentials-provider");
        }

        Builder clock(Clock clock) {
            this.clock = clock;
            return this;
        }

        @Override
        public Builder endpoint(String endpoint) {
            this.endpoint = endpoint;
            return this;
        }

        public void setEndpoint(String endpoint) {
            this.endpoint(endpoint);
        }

        @Override
        public Builder asyncCredentialUpdateEnabled(Boolean asyncCredentialUpdateEnabled) {
            this.asyncCredentialUpdateEnabled = asyncCredentialUpdateEnabled;
            return this;
        }

        public void setAsyncCredentialUpdateEnabled(boolean asyncCredentialUpdateEnabled) {
            this.asyncCredentialUpdateEnabled(asyncCredentialUpdateEnabled);
        }

        @Override
        public Builder asyncThreadName(String asyncThreadName) {
            this.asyncThreadName = asyncThreadName;
            return this;
        }

        public void setAsyncThreadName(String asyncThreadName) {
            this.asyncThreadName(asyncThreadName);
        }

        @Override
        public Builder profileFile(ProfileFile profileFile) {
            this.profileFile = profileFile;
            return this;
        }

        public void setProfileFile(ProfileFile profileFile) {
            this.profileFile(profileFile);
        }

        @Override
        public Builder profileName(String profileName) {
            this.profileName = profileName;
            return this;
        }

        public void setProfileName(String profileName) {
            this.profileName(profileName);
        }

        @Override
        public InstanceProfileCredentialsProvider build() {
            return new InstanceProfileCredentialsProvider(this);
        }
    }

    public static interface Builder
    extends HttpCredentialsProvider.Builder<InstanceProfileCredentialsProvider, Builder> {
        public Builder profileFile(ProfileFile var1);

        public Builder profileName(String var1);

        @Override
        public InstanceProfileCredentialsProvider build();
    }
}

