/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscaling.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Generated;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a scaling policy.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ScalingPolicy implements ToCopyableBuilder<ScalingPolicy.Builder, ScalingPolicy> {
    private final String autoScalingGroupName;

    private final String policyName;

    private final String policyARN;

    private final String policyType;

    private final String adjustmentType;

    private final Integer minAdjustmentStep;

    private final Integer minAdjustmentMagnitude;

    private final Integer scalingAdjustment;

    private final Integer cooldown;

    private final List<StepAdjustment> stepAdjustments;

    private final String metricAggregationType;

    private final Integer estimatedInstanceWarmup;

    private final List<Alarm> alarms;

    private final TargetTrackingConfiguration targetTrackingConfiguration;

    private ScalingPolicy(BuilderImpl builder) {
        this.autoScalingGroupName = builder.autoScalingGroupName;
        this.policyName = builder.policyName;
        this.policyARN = builder.policyARN;
        this.policyType = builder.policyType;
        this.adjustmentType = builder.adjustmentType;
        this.minAdjustmentStep = builder.minAdjustmentStep;
        this.minAdjustmentMagnitude = builder.minAdjustmentMagnitude;
        this.scalingAdjustment = builder.scalingAdjustment;
        this.cooldown = builder.cooldown;
        this.stepAdjustments = builder.stepAdjustments;
        this.metricAggregationType = builder.metricAggregationType;
        this.estimatedInstanceWarmup = builder.estimatedInstanceWarmup;
        this.alarms = builder.alarms;
        this.targetTrackingConfiguration = builder.targetTrackingConfiguration;
    }

    /**
     * <p>
     * The name of the Auto Scaling group associated with this scaling policy.
     * </p>
     * 
     * @return The name of the Auto Scaling group associated with this scaling policy.
     */
    public String autoScalingGroupName() {
        return autoScalingGroupName;
    }

    /**
     * <p>
     * The name of the scaling policy.
     * </p>
     * 
     * @return The name of the scaling policy.
     */
    public String policyName() {
        return policyName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the policy.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the policy.
     */
    public String policyARN() {
        return policyARN;
    }

    /**
     * <p>
     * The policy type. Valid values are <code>SimpleScaling</code> and <code>StepScaling</code>.
     * </p>
     * 
     * @return The policy type. Valid values are <code>SimpleScaling</code> and <code>StepScaling</code>.
     */
    public String policyType() {
        return policyType;
    }

    /**
     * <p>
     * The adjustment type, which specifies how <code>ScalingAdjustment</code> is interpreted. Valid values are
     * <code>ChangeInCapacity</code>, <code>ExactCapacity</code>, and <code>PercentChangeInCapacity</code>.
     * </p>
     * 
     * @return The adjustment type, which specifies how <code>ScalingAdjustment</code> is interpreted. Valid values are
     *         <code>ChangeInCapacity</code>, <code>ExactCapacity</code>, and <code>PercentChangeInCapacity</code>.
     */
    public String adjustmentType() {
        return adjustmentType;
    }

    /**
     * <p>
     * Available for backward compatibility. Use <code>MinAdjustmentMagnitude</code> instead.
     * </p>
     * 
     * @return Available for backward compatibility. Use <code>MinAdjustmentMagnitude</code> instead.
     */
    public Integer minAdjustmentStep() {
        return minAdjustmentStep;
    }

    /**
     * <p>
     * The minimum number of instances to scale. If the value of <code>AdjustmentType</code> is
     * <code>PercentChangeInCapacity</code>, the scaling policy changes the <code>DesiredCapacity</code> of the Auto
     * Scaling group by at least this many instances. Otherwise, the error is <code>ValidationError</code>.
     * </p>
     * 
     * @return The minimum number of instances to scale. If the value of <code>AdjustmentType</code> is
     *         <code>PercentChangeInCapacity</code>, the scaling policy changes the <code>DesiredCapacity</code> of the
     *         Auto Scaling group by at least this many instances. Otherwise, the error is <code>ValidationError</code>.
     */
    public Integer minAdjustmentMagnitude() {
        return minAdjustmentMagnitude;
    }

    /**
     * <p>
     * The amount by which to scale, based on the specified adjustment type. A positive value adds to the current
     * capacity while a negative number removes from the current capacity.
     * </p>
     * 
     * @return The amount by which to scale, based on the specified adjustment type. A positive value adds to the
     *         current capacity while a negative number removes from the current capacity.
     */
    public Integer scalingAdjustment() {
        return scalingAdjustment;
    }

    /**
     * <p>
     * The amount of time, in seconds, after a scaling activity completes before any further dynamic scaling activities
     * can start.
     * </p>
     * 
     * @return The amount of time, in seconds, after a scaling activity completes before any further dynamic scaling
     *         activities can start.
     */
    public Integer cooldown() {
        return cooldown;
    }

    /**
     * <p>
     * A set of adjustments that enable you to scale based on the size of the alarm breach.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A set of adjustments that enable you to scale based on the size of the alarm breach.
     */
    public List<StepAdjustment> stepAdjustments() {
        return stepAdjustments;
    }

    /**
     * <p>
     * The aggregation type for the CloudWatch metrics. Valid values are <code>Minimum</code>, <code>Maximum</code>, and
     * <code>Average</code>.
     * </p>
     * 
     * @return The aggregation type for the CloudWatch metrics. Valid values are <code>Minimum</code>,
     *         <code>Maximum</code>, and <code>Average</code>.
     */
    public String metricAggregationType() {
        return metricAggregationType;
    }

    /**
     * <p>
     * The estimated time, in seconds, until a newly launched instance can contribute to the CloudWatch metrics.
     * </p>
     * 
     * @return The estimated time, in seconds, until a newly launched instance can contribute to the CloudWatch metrics.
     */
    public Integer estimatedInstanceWarmup() {
        return estimatedInstanceWarmup;
    }

    /**
     * <p>
     * The CloudWatch alarms related to the policy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The CloudWatch alarms related to the policy.
     */
    public List<Alarm> alarms() {
        return alarms;
    }

    /**
     * <p>
     * A target tracking policy.
     * </p>
     * 
     * @return A target tracking policy.
     */
    public TargetTrackingConfiguration targetTrackingConfiguration() {
        return targetTrackingConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(policyName());
        hashCode = 31 * hashCode + Objects.hashCode(policyARN());
        hashCode = 31 * hashCode + Objects.hashCode(policyType());
        hashCode = 31 * hashCode + Objects.hashCode(adjustmentType());
        hashCode = 31 * hashCode + Objects.hashCode(minAdjustmentStep());
        hashCode = 31 * hashCode + Objects.hashCode(minAdjustmentMagnitude());
        hashCode = 31 * hashCode + Objects.hashCode(scalingAdjustment());
        hashCode = 31 * hashCode + Objects.hashCode(cooldown());
        hashCode = 31 * hashCode + Objects.hashCode(stepAdjustments());
        hashCode = 31 * hashCode + Objects.hashCode(metricAggregationType());
        hashCode = 31 * hashCode + Objects.hashCode(estimatedInstanceWarmup());
        hashCode = 31 * hashCode + Objects.hashCode(alarms());
        hashCode = 31 * hashCode + Objects.hashCode(targetTrackingConfiguration());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ScalingPolicy)) {
            return false;
        }
        ScalingPolicy other = (ScalingPolicy) obj;
        return Objects.equals(autoScalingGroupName(), other.autoScalingGroupName())
                && Objects.equals(policyName(), other.policyName()) && Objects.equals(policyARN(), other.policyARN())
                && Objects.equals(policyType(), other.policyType()) && Objects.equals(adjustmentType(), other.adjustmentType())
                && Objects.equals(minAdjustmentStep(), other.minAdjustmentStep())
                && Objects.equals(minAdjustmentMagnitude(), other.minAdjustmentMagnitude())
                && Objects.equals(scalingAdjustment(), other.scalingAdjustment()) && Objects.equals(cooldown(), other.cooldown())
                && Objects.equals(stepAdjustments(), other.stepAdjustments())
                && Objects.equals(metricAggregationType(), other.metricAggregationType())
                && Objects.equals(estimatedInstanceWarmup(), other.estimatedInstanceWarmup())
                && Objects.equals(alarms(), other.alarms())
                && Objects.equals(targetTrackingConfiguration(), other.targetTrackingConfiguration());
    }

    @Override
    public String toString() {
        return ToString.builder("ScalingPolicy").add("AutoScalingGroupName", autoScalingGroupName())
                .add("PolicyName", policyName()).add("PolicyARN", policyARN()).add("PolicyType", policyType())
                .add("AdjustmentType", adjustmentType()).add("MinAdjustmentStep", minAdjustmentStep())
                .add("MinAdjustmentMagnitude", minAdjustmentMagnitude()).add("ScalingAdjustment", scalingAdjustment())
                .add("Cooldown", cooldown()).add("StepAdjustments", stepAdjustments())
                .add("MetricAggregationType", metricAggregationType()).add("EstimatedInstanceWarmup", estimatedInstanceWarmup())
                .add("Alarms", alarms()).add("TargetTrackingConfiguration", targetTrackingConfiguration()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AutoScalingGroupName":
            return Optional.of(clazz.cast(autoScalingGroupName()));
        case "PolicyName":
            return Optional.of(clazz.cast(policyName()));
        case "PolicyARN":
            return Optional.of(clazz.cast(policyARN()));
        case "PolicyType":
            return Optional.of(clazz.cast(policyType()));
        case "AdjustmentType":
            return Optional.of(clazz.cast(adjustmentType()));
        case "MinAdjustmentStep":
            return Optional.of(clazz.cast(minAdjustmentStep()));
        case "MinAdjustmentMagnitude":
            return Optional.of(clazz.cast(minAdjustmentMagnitude()));
        case "ScalingAdjustment":
            return Optional.of(clazz.cast(scalingAdjustment()));
        case "Cooldown":
            return Optional.of(clazz.cast(cooldown()));
        case "StepAdjustments":
            return Optional.of(clazz.cast(stepAdjustments()));
        case "MetricAggregationType":
            return Optional.of(clazz.cast(metricAggregationType()));
        case "EstimatedInstanceWarmup":
            return Optional.of(clazz.cast(estimatedInstanceWarmup()));
        case "Alarms":
            return Optional.of(clazz.cast(alarms()));
        case "TargetTrackingConfiguration":
            return Optional.of(clazz.cast(targetTrackingConfiguration()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, ScalingPolicy> {
        /**
         * <p>
         * The name of the Auto Scaling group associated with this scaling policy.
         * </p>
         * 
         * @param autoScalingGroupName
         *        The name of the Auto Scaling group associated with this scaling policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingGroupName(String autoScalingGroupName);

        /**
         * <p>
         * The name of the scaling policy.
         * </p>
         * 
         * @param policyName
         *        The name of the scaling policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyName(String policyName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the policy.
         * </p>
         * 
         * @param policyARN
         *        The Amazon Resource Name (ARN) of the policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyARN(String policyARN);

        /**
         * <p>
         * The policy type. Valid values are <code>SimpleScaling</code> and <code>StepScaling</code>.
         * </p>
         * 
         * @param policyType
         *        The policy type. Valid values are <code>SimpleScaling</code> and <code>StepScaling</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyType(String policyType);

        /**
         * <p>
         * The adjustment type, which specifies how <code>ScalingAdjustment</code> is interpreted. Valid values are
         * <code>ChangeInCapacity</code>, <code>ExactCapacity</code>, and <code>PercentChangeInCapacity</code>.
         * </p>
         * 
         * @param adjustmentType
         *        The adjustment type, which specifies how <code>ScalingAdjustment</code> is interpreted. Valid values
         *        are <code>ChangeInCapacity</code>, <code>ExactCapacity</code>, and
         *        <code>PercentChangeInCapacity</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder adjustmentType(String adjustmentType);

        /**
         * <p>
         * Available for backward compatibility. Use <code>MinAdjustmentMagnitude</code> instead.
         * </p>
         * 
         * @param minAdjustmentStep
         *        Available for backward compatibility. Use <code>MinAdjustmentMagnitude</code> instead.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minAdjustmentStep(Integer minAdjustmentStep);

        /**
         * <p>
         * The minimum number of instances to scale. If the value of <code>AdjustmentType</code> is
         * <code>PercentChangeInCapacity</code>, the scaling policy changes the <code>DesiredCapacity</code> of the Auto
         * Scaling group by at least this many instances. Otherwise, the error is <code>ValidationError</code>.
         * </p>
         * 
         * @param minAdjustmentMagnitude
         *        The minimum number of instances to scale. If the value of <code>AdjustmentType</code> is
         *        <code>PercentChangeInCapacity</code>, the scaling policy changes the <code>DesiredCapacity</code> of
         *        the Auto Scaling group by at least this many instances. Otherwise, the error is
         *        <code>ValidationError</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minAdjustmentMagnitude(Integer minAdjustmentMagnitude);

        /**
         * <p>
         * The amount by which to scale, based on the specified adjustment type. A positive value adds to the current
         * capacity while a negative number removes from the current capacity.
         * </p>
         * 
         * @param scalingAdjustment
         *        The amount by which to scale, based on the specified adjustment type. A positive value adds to the
         *        current capacity while a negative number removes from the current capacity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scalingAdjustment(Integer scalingAdjustment);

        /**
         * <p>
         * The amount of time, in seconds, after a scaling activity completes before any further dynamic scaling
         * activities can start.
         * </p>
         * 
         * @param cooldown
         *        The amount of time, in seconds, after a scaling activity completes before any further dynamic scaling
         *        activities can start.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cooldown(Integer cooldown);

        /**
         * <p>
         * A set of adjustments that enable you to scale based on the size of the alarm breach.
         * </p>
         * 
         * @param stepAdjustments
         *        A set of adjustments that enable you to scale based on the size of the alarm breach.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stepAdjustments(Collection<StepAdjustment> stepAdjustments);

        /**
         * <p>
         * A set of adjustments that enable you to scale based on the size of the alarm breach.
         * </p>
         * 
         * @param stepAdjustments
         *        A set of adjustments that enable you to scale based on the size of the alarm breach.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stepAdjustments(StepAdjustment... stepAdjustments);

        /**
         * <p>
         * A set of adjustments that enable you to scale based on the size of the alarm breach.
         * </p>
         * This is a convenience that creates an instance of the {@link List<StepAdjustment>.Builder} avoiding the need
         * to create one manually via {@link List<StepAdjustment>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<StepAdjustment>.Builder#build()} is called immediately and
         * its result is passed to {@link #stepAdjustments(List<StepAdjustment>)}.
         * 
         * @param stepAdjustments
         *        a consumer that will call methods on {@link List<StepAdjustment>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stepAdjustments(List<StepAdjustment>)
         */
        Builder stepAdjustments(Consumer<StepAdjustment.Builder>... stepAdjustments);

        /**
         * <p>
         * The aggregation type for the CloudWatch metrics. Valid values are <code>Minimum</code>, <code>Maximum</code>,
         * and <code>Average</code>.
         * </p>
         * 
         * @param metricAggregationType
         *        The aggregation type for the CloudWatch metrics. Valid values are <code>Minimum</code>,
         *        <code>Maximum</code>, and <code>Average</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricAggregationType(String metricAggregationType);

        /**
         * <p>
         * The estimated time, in seconds, until a newly launched instance can contribute to the CloudWatch metrics.
         * </p>
         * 
         * @param estimatedInstanceWarmup
         *        The estimated time, in seconds, until a newly launched instance can contribute to the CloudWatch
         *        metrics.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder estimatedInstanceWarmup(Integer estimatedInstanceWarmup);

        /**
         * <p>
         * The CloudWatch alarms related to the policy.
         * </p>
         * 
         * @param alarms
         *        The CloudWatch alarms related to the policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarms(Collection<Alarm> alarms);

        /**
         * <p>
         * The CloudWatch alarms related to the policy.
         * </p>
         * 
         * @param alarms
         *        The CloudWatch alarms related to the policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarms(Alarm... alarms);

        /**
         * <p>
         * The CloudWatch alarms related to the policy.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Alarm>.Builder} avoiding the need to create
         * one manually via {@link List<Alarm>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Alarm>.Builder#build()} is called immediately and its result
         * is passed to {@link #alarms(List<Alarm>)}.
         * 
         * @param alarms
         *        a consumer that will call methods on {@link List<Alarm>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #alarms(List<Alarm>)
         */
        Builder alarms(Consumer<Alarm.Builder>... alarms);

        /**
         * <p>
         * A target tracking policy.
         * </p>
         * 
         * @param targetTrackingConfiguration
         *        A target tracking policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetTrackingConfiguration(TargetTrackingConfiguration targetTrackingConfiguration);

        /**
         * <p>
         * A target tracking policy.
         * </p>
         * This is a convenience that creates an instance of the {@link TargetTrackingConfiguration.Builder} avoiding
         * the need to create one manually via {@link TargetTrackingConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link TargetTrackingConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #targetTrackingConfiguration(TargetTrackingConfiguration)}.
         * 
         * @param targetTrackingConfiguration
         *        a consumer that will call methods on {@link TargetTrackingConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetTrackingConfiguration(TargetTrackingConfiguration)
         */
        default Builder targetTrackingConfiguration(Consumer<TargetTrackingConfiguration.Builder> targetTrackingConfiguration) {
            return targetTrackingConfiguration(TargetTrackingConfiguration.builder().apply(targetTrackingConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String autoScalingGroupName;

        private String policyName;

        private String policyARN;

        private String policyType;

        private String adjustmentType;

        private Integer minAdjustmentStep;

        private Integer minAdjustmentMagnitude;

        private Integer scalingAdjustment;

        private Integer cooldown;

        private List<StepAdjustment> stepAdjustments;

        private String metricAggregationType;

        private Integer estimatedInstanceWarmup;

        private List<Alarm> alarms;

        private TargetTrackingConfiguration targetTrackingConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(ScalingPolicy model) {
            autoScalingGroupName(model.autoScalingGroupName);
            policyName(model.policyName);
            policyARN(model.policyARN);
            policyType(model.policyType);
            adjustmentType(model.adjustmentType);
            minAdjustmentStep(model.minAdjustmentStep);
            minAdjustmentMagnitude(model.minAdjustmentMagnitude);
            scalingAdjustment(model.scalingAdjustment);
            cooldown(model.cooldown);
            stepAdjustments(model.stepAdjustments);
            metricAggregationType(model.metricAggregationType);
            estimatedInstanceWarmup(model.estimatedInstanceWarmup);
            alarms(model.alarms);
            targetTrackingConfiguration(model.targetTrackingConfiguration);
        }

        public final String getAutoScalingGroupName() {
            return autoScalingGroupName;
        }

        @Override
        public final Builder autoScalingGroupName(String autoScalingGroupName) {
            this.autoScalingGroupName = autoScalingGroupName;
            return this;
        }

        public final void setAutoScalingGroupName(String autoScalingGroupName) {
            this.autoScalingGroupName = autoScalingGroupName;
        }

        public final String getPolicyName() {
            return policyName;
        }

        @Override
        public final Builder policyName(String policyName) {
            this.policyName = policyName;
            return this;
        }

        public final void setPolicyName(String policyName) {
            this.policyName = policyName;
        }

        public final String getPolicyARN() {
            return policyARN;
        }

        @Override
        public final Builder policyARN(String policyARN) {
            this.policyARN = policyARN;
            return this;
        }

        public final void setPolicyARN(String policyARN) {
            this.policyARN = policyARN;
        }

        public final String getPolicyType() {
            return policyType;
        }

        @Override
        public final Builder policyType(String policyType) {
            this.policyType = policyType;
            return this;
        }

        public final void setPolicyType(String policyType) {
            this.policyType = policyType;
        }

        public final String getAdjustmentType() {
            return adjustmentType;
        }

        @Override
        public final Builder adjustmentType(String adjustmentType) {
            this.adjustmentType = adjustmentType;
            return this;
        }

        public final void setAdjustmentType(String adjustmentType) {
            this.adjustmentType = adjustmentType;
        }

        public final Integer getMinAdjustmentStep() {
            return minAdjustmentStep;
        }

        @Override
        public final Builder minAdjustmentStep(Integer minAdjustmentStep) {
            this.minAdjustmentStep = minAdjustmentStep;
            return this;
        }

        public final void setMinAdjustmentStep(Integer minAdjustmentStep) {
            this.minAdjustmentStep = minAdjustmentStep;
        }

        public final Integer getMinAdjustmentMagnitude() {
            return minAdjustmentMagnitude;
        }

        @Override
        public final Builder minAdjustmentMagnitude(Integer minAdjustmentMagnitude) {
            this.minAdjustmentMagnitude = minAdjustmentMagnitude;
            return this;
        }

        public final void setMinAdjustmentMagnitude(Integer minAdjustmentMagnitude) {
            this.minAdjustmentMagnitude = minAdjustmentMagnitude;
        }

        public final Integer getScalingAdjustment() {
            return scalingAdjustment;
        }

        @Override
        public final Builder scalingAdjustment(Integer scalingAdjustment) {
            this.scalingAdjustment = scalingAdjustment;
            return this;
        }

        public final void setScalingAdjustment(Integer scalingAdjustment) {
            this.scalingAdjustment = scalingAdjustment;
        }

        public final Integer getCooldown() {
            return cooldown;
        }

        @Override
        public final Builder cooldown(Integer cooldown) {
            this.cooldown = cooldown;
            return this;
        }

        public final void setCooldown(Integer cooldown) {
            this.cooldown = cooldown;
        }

        public final Collection<StepAdjustment.Builder> getStepAdjustments() {
            return stepAdjustments != null ? stepAdjustments.stream().map(StepAdjustment::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder stepAdjustments(Collection<StepAdjustment> stepAdjustments) {
            this.stepAdjustments = StepAdjustmentsCopier.copy(stepAdjustments);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stepAdjustments(StepAdjustment... stepAdjustments) {
            stepAdjustments(Arrays.asList(stepAdjustments));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stepAdjustments(Consumer<StepAdjustment.Builder>... stepAdjustments) {
            stepAdjustments(Stream.of(stepAdjustments).map(c -> StepAdjustment.builder().apply(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setStepAdjustments(Collection<StepAdjustment.BuilderImpl> stepAdjustments) {
            this.stepAdjustments = StepAdjustmentsCopier.copyFromBuilder(stepAdjustments);
        }

        public final String getMetricAggregationType() {
            return metricAggregationType;
        }

        @Override
        public final Builder metricAggregationType(String metricAggregationType) {
            this.metricAggregationType = metricAggregationType;
            return this;
        }

        public final void setMetricAggregationType(String metricAggregationType) {
            this.metricAggregationType = metricAggregationType;
        }

        public final Integer getEstimatedInstanceWarmup() {
            return estimatedInstanceWarmup;
        }

        @Override
        public final Builder estimatedInstanceWarmup(Integer estimatedInstanceWarmup) {
            this.estimatedInstanceWarmup = estimatedInstanceWarmup;
            return this;
        }

        public final void setEstimatedInstanceWarmup(Integer estimatedInstanceWarmup) {
            this.estimatedInstanceWarmup = estimatedInstanceWarmup;
        }

        public final Collection<Alarm.Builder> getAlarms() {
            return alarms != null ? alarms.stream().map(Alarm::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder alarms(Collection<Alarm> alarms) {
            this.alarms = AlarmsCopier.copy(alarms);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder alarms(Alarm... alarms) {
            alarms(Arrays.asList(alarms));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder alarms(Consumer<Alarm.Builder>... alarms) {
            alarms(Stream.of(alarms).map(c -> Alarm.builder().apply(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setAlarms(Collection<Alarm.BuilderImpl> alarms) {
            this.alarms = AlarmsCopier.copyFromBuilder(alarms);
        }

        public final TargetTrackingConfiguration.Builder getTargetTrackingConfiguration() {
            return targetTrackingConfiguration != null ? targetTrackingConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder targetTrackingConfiguration(TargetTrackingConfiguration targetTrackingConfiguration) {
            this.targetTrackingConfiguration = targetTrackingConfiguration;
            return this;
        }

        public final void setTargetTrackingConfiguration(TargetTrackingConfiguration.BuilderImpl targetTrackingConfiguration) {
            this.targetTrackingConfiguration = targetTrackingConfiguration != null ? targetTrackingConfiguration.build() : null;
        }

        @Override
        public ScalingPolicy build() {
            return new ScalingPolicy(this);
        }
    }
}
