/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscaling.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an Auto Scaling group.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutoScalingGroup implements ToCopyableBuilder<AutoScalingGroup.Builder, AutoScalingGroup> {
    private final String autoScalingGroupName;

    private final String autoScalingGroupARN;

    private final String launchConfigurationName;

    private final Integer minSize;

    private final Integer maxSize;

    private final Integer desiredCapacity;

    private final Integer defaultCooldown;

    private final List<String> availabilityZones;

    private final List<String> loadBalancerNames;

    private final List<String> targetGroupARNs;

    private final String healthCheckType;

    private final Integer healthCheckGracePeriod;

    private final List<Instance> instances;

    private final Instant createdTime;

    private final List<SuspendedProcess> suspendedProcesses;

    private final String placementGroup;

    private final String vpcZoneIdentifier;

    private final List<EnabledMetric> enabledMetrics;

    private final String status;

    private final List<TagDescription> tags;

    private final List<String> terminationPolicies;

    private final Boolean newInstancesProtectedFromScaleIn;

    private AutoScalingGroup(BuilderImpl builder) {
        this.autoScalingGroupName = builder.autoScalingGroupName;
        this.autoScalingGroupARN = builder.autoScalingGroupARN;
        this.launchConfigurationName = builder.launchConfigurationName;
        this.minSize = builder.minSize;
        this.maxSize = builder.maxSize;
        this.desiredCapacity = builder.desiredCapacity;
        this.defaultCooldown = builder.defaultCooldown;
        this.availabilityZones = builder.availabilityZones;
        this.loadBalancerNames = builder.loadBalancerNames;
        this.targetGroupARNs = builder.targetGroupARNs;
        this.healthCheckType = builder.healthCheckType;
        this.healthCheckGracePeriod = builder.healthCheckGracePeriod;
        this.instances = builder.instances;
        this.createdTime = builder.createdTime;
        this.suspendedProcesses = builder.suspendedProcesses;
        this.placementGroup = builder.placementGroup;
        this.vpcZoneIdentifier = builder.vpcZoneIdentifier;
        this.enabledMetrics = builder.enabledMetrics;
        this.status = builder.status;
        this.tags = builder.tags;
        this.terminationPolicies = builder.terminationPolicies;
        this.newInstancesProtectedFromScaleIn = builder.newInstancesProtectedFromScaleIn;
    }

    /**
     * <p>
     * The name of the group.
     * </p>
     * 
     * @return The name of the group.
     */
    public String autoScalingGroupName() {
        return autoScalingGroupName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the group.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the group.
     */
    public String autoScalingGroupARN() {
        return autoScalingGroupARN;
    }

    /**
     * <p>
     * The name of the associated launch configuration.
     * </p>
     * 
     * @return The name of the associated launch configuration.
     */
    public String launchConfigurationName() {
        return launchConfigurationName;
    }

    /**
     * <p>
     * The minimum size of the group.
     * </p>
     * 
     * @return The minimum size of the group.
     */
    public Integer minSize() {
        return minSize;
    }

    /**
     * <p>
     * The maximum size of the group.
     * </p>
     * 
     * @return The maximum size of the group.
     */
    public Integer maxSize() {
        return maxSize;
    }

    /**
     * <p>
     * The desired size of the group.
     * </p>
     * 
     * @return The desired size of the group.
     */
    public Integer desiredCapacity() {
        return desiredCapacity;
    }

    /**
     * <p>
     * The amount of time, in seconds, after a scaling activity completes before another scaling activity can start.
     * </p>
     * 
     * @return The amount of time, in seconds, after a scaling activity completes before another scaling activity can
     *         start.
     */
    public Integer defaultCooldown() {
        return defaultCooldown;
    }

    /**
     * <p>
     * One or more Availability Zones for the group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return One or more Availability Zones for the group.
     */
    public List<String> availabilityZones() {
        return availabilityZones;
    }

    /**
     * <p>
     * One or more load balancers associated with the group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return One or more load balancers associated with the group.
     */
    public List<String> loadBalancerNames() {
        return loadBalancerNames;
    }

    /**
     * <p>
     * The Amazon Resource Names (ARN) of the target groups for your load balancer.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The Amazon Resource Names (ARN) of the target groups for your load balancer.
     */
    public List<String> targetGroupARNs() {
        return targetGroupARNs;
    }

    /**
     * <p>
     * The service to use for the health checks. The valid values are <code>EC2</code> and <code>ELB</code>.
     * </p>
     * 
     * @return The service to use for the health checks. The valid values are <code>EC2</code> and <code>ELB</code>.
     */
    public String healthCheckType() {
        return healthCheckType;
    }

    /**
     * <p>
     * The amount of time, in seconds, that Auto Scaling waits before checking the health status of an EC2 instance that
     * has come into service.
     * </p>
     * 
     * @return The amount of time, in seconds, that Auto Scaling waits before checking the health status of an EC2
     *         instance that has come into service.
     */
    public Integer healthCheckGracePeriod() {
        return healthCheckGracePeriod;
    }

    /**
     * <p>
     * The EC2 instances associated with the group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The EC2 instances associated with the group.
     */
    public List<Instance> instances() {
        return instances;
    }

    /**
     * <p>
     * The date and time the group was created.
     * </p>
     * 
     * @return The date and time the group was created.
     */
    public Instant createdTime() {
        return createdTime;
    }

    /**
     * <p>
     * The suspended processes associated with the group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The suspended processes associated with the group.
     */
    public List<SuspendedProcess> suspendedProcesses() {
        return suspendedProcesses;
    }

    /**
     * <p>
     * The name of the placement group into which you'll launch your instances, if any. For more information, see <a
     * href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html">Placement Groups</a> in the
     * <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * 
     * @return The name of the placement group into which you'll launch your instances, if any. For more information,
     *         see <a href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html">Placement
     *         Groups</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     */
    public String placementGroup() {
        return placementGroup;
    }

    /**
     * <p>
     * One or more subnet IDs, if applicable, separated by commas.
     * </p>
     * <p>
     * If you specify <code>VPCZoneIdentifier</code> and <code>AvailabilityZones</code>, ensure that the Availability
     * Zones of the subnets match the values for <code>AvailabilityZones</code>.
     * </p>
     * 
     * @return One or more subnet IDs, if applicable, separated by commas.</p>
     *         <p>
     *         If you specify <code>VPCZoneIdentifier</code> and <code>AvailabilityZones</code>, ensure that the
     *         Availability Zones of the subnets match the values for <code>AvailabilityZones</code>.
     */
    public String vpcZoneIdentifier() {
        return vpcZoneIdentifier;
    }

    /**
     * <p>
     * The metrics enabled for the group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The metrics enabled for the group.
     */
    public List<EnabledMetric> enabledMetrics() {
        return enabledMetrics;
    }

    /**
     * <p>
     * The current state of the group when <a>DeleteAutoScalingGroup</a> is in progress.
     * </p>
     * 
     * @return The current state of the group when <a>DeleteAutoScalingGroup</a> is in progress.
     */
    public String status() {
        return status;
    }

    /**
     * <p>
     * The tags for the group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The tags for the group.
     */
    public List<TagDescription> tags() {
        return tags;
    }

    /**
     * <p>
     * The termination policies for the group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The termination policies for the group.
     */
    public List<String> terminationPolicies() {
        return terminationPolicies;
    }

    /**
     * <p>
     * Indicates whether newly launched instances are protected from termination by Auto Scaling when scaling in.
     * </p>
     * 
     * @return Indicates whether newly launched instances are protected from termination by Auto Scaling when scaling
     *         in.
     */
    public Boolean newInstancesProtectedFromScaleIn() {
        return newInstancesProtectedFromScaleIn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingGroupARN());
        hashCode = 31 * hashCode + Objects.hashCode(launchConfigurationName());
        hashCode = 31 * hashCode + Objects.hashCode(minSize());
        hashCode = 31 * hashCode + Objects.hashCode(maxSize());
        hashCode = 31 * hashCode + Objects.hashCode(desiredCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(defaultCooldown());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZones());
        hashCode = 31 * hashCode + Objects.hashCode(loadBalancerNames());
        hashCode = 31 * hashCode + Objects.hashCode(targetGroupARNs());
        hashCode = 31 * hashCode + Objects.hashCode(healthCheckType());
        hashCode = 31 * hashCode + Objects.hashCode(healthCheckGracePeriod());
        hashCode = 31 * hashCode + Objects.hashCode(instances());
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        hashCode = 31 * hashCode + Objects.hashCode(suspendedProcesses());
        hashCode = 31 * hashCode + Objects.hashCode(placementGroup());
        hashCode = 31 * hashCode + Objects.hashCode(vpcZoneIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(enabledMetrics());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(terminationPolicies());
        hashCode = 31 * hashCode + Objects.hashCode(newInstancesProtectedFromScaleIn());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutoScalingGroup)) {
            return false;
        }
        AutoScalingGroup other = (AutoScalingGroup) obj;
        return Objects.equals(autoScalingGroupName(), other.autoScalingGroupName())
                && Objects.equals(autoScalingGroupARN(), other.autoScalingGroupARN())
                && Objects.equals(launchConfigurationName(), other.launchConfigurationName())
                && Objects.equals(minSize(), other.minSize()) && Objects.equals(maxSize(), other.maxSize())
                && Objects.equals(desiredCapacity(), other.desiredCapacity())
                && Objects.equals(defaultCooldown(), other.defaultCooldown())
                && Objects.equals(availabilityZones(), other.availabilityZones())
                && Objects.equals(loadBalancerNames(), other.loadBalancerNames())
                && Objects.equals(targetGroupARNs(), other.targetGroupARNs())
                && Objects.equals(healthCheckType(), other.healthCheckType())
                && Objects.equals(healthCheckGracePeriod(), other.healthCheckGracePeriod())
                && Objects.equals(instances(), other.instances()) && Objects.equals(createdTime(), other.createdTime())
                && Objects.equals(suspendedProcesses(), other.suspendedProcesses())
                && Objects.equals(placementGroup(), other.placementGroup())
                && Objects.equals(vpcZoneIdentifier(), other.vpcZoneIdentifier())
                && Objects.equals(enabledMetrics(), other.enabledMetrics()) && Objects.equals(status(), other.status())
                && Objects.equals(tags(), other.tags()) && Objects.equals(terminationPolicies(), other.terminationPolicies())
                && Objects.equals(newInstancesProtectedFromScaleIn(), other.newInstancesProtectedFromScaleIn());
    }

    @Override
    public String toString() {
        return ToString.builder("AutoScalingGroup").add("AutoScalingGroupName", autoScalingGroupName())
                .add("AutoScalingGroupARN", autoScalingGroupARN()).add("LaunchConfigurationName", launchConfigurationName())
                .add("MinSize", minSize()).add("MaxSize", maxSize()).add("DesiredCapacity", desiredCapacity())
                .add("DefaultCooldown", defaultCooldown()).add("AvailabilityZones", availabilityZones())
                .add("LoadBalancerNames", loadBalancerNames()).add("TargetGroupARNs", targetGroupARNs())
                .add("HealthCheckType", healthCheckType()).add("HealthCheckGracePeriod", healthCheckGracePeriod())
                .add("Instances", instances()).add("CreatedTime", createdTime()).add("SuspendedProcesses", suspendedProcesses())
                .add("PlacementGroup", placementGroup()).add("VPCZoneIdentifier", vpcZoneIdentifier())
                .add("EnabledMetrics", enabledMetrics()).add("Status", status()).add("Tags", tags())
                .add("TerminationPolicies", terminationPolicies())
                .add("NewInstancesProtectedFromScaleIn", newInstancesProtectedFromScaleIn()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AutoScalingGroupName":
            return Optional.ofNullable(clazz.cast(autoScalingGroupName()));
        case "AutoScalingGroupARN":
            return Optional.ofNullable(clazz.cast(autoScalingGroupARN()));
        case "LaunchConfigurationName":
            return Optional.ofNullable(clazz.cast(launchConfigurationName()));
        case "MinSize":
            return Optional.ofNullable(clazz.cast(minSize()));
        case "MaxSize":
            return Optional.ofNullable(clazz.cast(maxSize()));
        case "DesiredCapacity":
            return Optional.ofNullable(clazz.cast(desiredCapacity()));
        case "DefaultCooldown":
            return Optional.ofNullable(clazz.cast(defaultCooldown()));
        case "AvailabilityZones":
            return Optional.ofNullable(clazz.cast(availabilityZones()));
        case "LoadBalancerNames":
            return Optional.ofNullable(clazz.cast(loadBalancerNames()));
        case "TargetGroupARNs":
            return Optional.ofNullable(clazz.cast(targetGroupARNs()));
        case "HealthCheckType":
            return Optional.ofNullable(clazz.cast(healthCheckType()));
        case "HealthCheckGracePeriod":
            return Optional.ofNullable(clazz.cast(healthCheckGracePeriod()));
        case "Instances":
            return Optional.ofNullable(clazz.cast(instances()));
        case "CreatedTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        case "SuspendedProcesses":
            return Optional.ofNullable(clazz.cast(suspendedProcesses()));
        case "PlacementGroup":
            return Optional.ofNullable(clazz.cast(placementGroup()));
        case "VPCZoneIdentifier":
            return Optional.ofNullable(clazz.cast(vpcZoneIdentifier()));
        case "EnabledMetrics":
            return Optional.ofNullable(clazz.cast(enabledMetrics()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "TerminationPolicies":
            return Optional.ofNullable(clazz.cast(terminationPolicies()));
        case "NewInstancesProtectedFromScaleIn":
            return Optional.ofNullable(clazz.cast(newInstancesProtectedFromScaleIn()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, AutoScalingGroup> {
        /**
         * <p>
         * The name of the group.
         * </p>
         * 
         * @param autoScalingGroupName
         *        The name of the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingGroupName(String autoScalingGroupName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the group.
         * </p>
         * 
         * @param autoScalingGroupARN
         *        The Amazon Resource Name (ARN) of the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingGroupARN(String autoScalingGroupARN);

        /**
         * <p>
         * The name of the associated launch configuration.
         * </p>
         * 
         * @param launchConfigurationName
         *        The name of the associated launch configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchConfigurationName(String launchConfigurationName);

        /**
         * <p>
         * The minimum size of the group.
         * </p>
         * 
         * @param minSize
         *        The minimum size of the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minSize(Integer minSize);

        /**
         * <p>
         * The maximum size of the group.
         * </p>
         * 
         * @param maxSize
         *        The maximum size of the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxSize(Integer maxSize);

        /**
         * <p>
         * The desired size of the group.
         * </p>
         * 
         * @param desiredCapacity
         *        The desired size of the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder desiredCapacity(Integer desiredCapacity);

        /**
         * <p>
         * The amount of time, in seconds, after a scaling activity completes before another scaling activity can start.
         * </p>
         * 
         * @param defaultCooldown
         *        The amount of time, in seconds, after a scaling activity completes before another scaling activity can
         *        start.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultCooldown(Integer defaultCooldown);

        /**
         * <p>
         * One or more Availability Zones for the group.
         * </p>
         * 
         * @param availabilityZones
         *        One or more Availability Zones for the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(Collection<String> availabilityZones);

        /**
         * <p>
         * One or more Availability Zones for the group.
         * </p>
         * 
         * @param availabilityZones
         *        One or more Availability Zones for the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(String... availabilityZones);

        /**
         * <p>
         * One or more load balancers associated with the group.
         * </p>
         * 
         * @param loadBalancerNames
         *        One or more load balancers associated with the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loadBalancerNames(Collection<String> loadBalancerNames);

        /**
         * <p>
         * One or more load balancers associated with the group.
         * </p>
         * 
         * @param loadBalancerNames
         *        One or more load balancers associated with the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loadBalancerNames(String... loadBalancerNames);

        /**
         * <p>
         * The Amazon Resource Names (ARN) of the target groups for your load balancer.
         * </p>
         * 
         * @param targetGroupARNs
         *        The Amazon Resource Names (ARN) of the target groups for your load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetGroupARNs(Collection<String> targetGroupARNs);

        /**
         * <p>
         * The Amazon Resource Names (ARN) of the target groups for your load balancer.
         * </p>
         * 
         * @param targetGroupARNs
         *        The Amazon Resource Names (ARN) of the target groups for your load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetGroupARNs(String... targetGroupARNs);

        /**
         * <p>
         * The service to use for the health checks. The valid values are <code>EC2</code> and <code>ELB</code>.
         * </p>
         * 
         * @param healthCheckType
         *        The service to use for the health checks. The valid values are <code>EC2</code> and <code>ELB</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthCheckType(String healthCheckType);

        /**
         * <p>
         * The amount of time, in seconds, that Auto Scaling waits before checking the health status of an EC2 instance
         * that has come into service.
         * </p>
         * 
         * @param healthCheckGracePeriod
         *        The amount of time, in seconds, that Auto Scaling waits before checking the health status of an EC2
         *        instance that has come into service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthCheckGracePeriod(Integer healthCheckGracePeriod);

        /**
         * <p>
         * The EC2 instances associated with the group.
         * </p>
         * 
         * @param instances
         *        The EC2 instances associated with the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instances(Collection<Instance> instances);

        /**
         * <p>
         * The EC2 instances associated with the group.
         * </p>
         * 
         * @param instances
         *        The EC2 instances associated with the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instances(Instance... instances);

        /**
         * <p>
         * The EC2 instances associated with the group.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Instance>.Builder} avoiding the need to
         * create one manually via {@link List<Instance>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Instance>.Builder#build()} is called immediately and its
         * result is passed to {@link #instances(List<Instance>)}.
         * 
         * @param instances
         *        a consumer that will call methods on {@link List<Instance>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instances(List<Instance>)
         */
        Builder instances(Consumer<Instance.Builder>... instances);

        /**
         * <p>
         * The date and time the group was created.
         * </p>
         * 
         * @param createdTime
         *        The date and time the group was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);

        /**
         * <p>
         * The suspended processes associated with the group.
         * </p>
         * 
         * @param suspendedProcesses
         *        The suspended processes associated with the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder suspendedProcesses(Collection<SuspendedProcess> suspendedProcesses);

        /**
         * <p>
         * The suspended processes associated with the group.
         * </p>
         * 
         * @param suspendedProcesses
         *        The suspended processes associated with the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder suspendedProcesses(SuspendedProcess... suspendedProcesses);

        /**
         * <p>
         * The suspended processes associated with the group.
         * </p>
         * This is a convenience that creates an instance of the {@link List<SuspendedProcess>.Builder} avoiding the
         * need to create one manually via {@link List<SuspendedProcess>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<SuspendedProcess>.Builder#build()} is called immediately and
         * its result is passed to {@link #suspendedProcesses(List<SuspendedProcess>)}.
         * 
         * @param suspendedProcesses
         *        a consumer that will call methods on {@link List<SuspendedProcess>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #suspendedProcesses(List<SuspendedProcess>)
         */
        Builder suspendedProcesses(Consumer<SuspendedProcess.Builder>... suspendedProcesses);

        /**
         * <p>
         * The name of the placement group into which you'll launch your instances, if any. For more information, see <a
         * href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html">Placement Groups</a> in the
         * <i>Amazon Elastic Compute Cloud User Guide</i>.
         * </p>
         * 
         * @param placementGroup
         *        The name of the placement group into which you'll launch your instances, if any. For more information,
         *        see <a href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html">Placement
         *        Groups</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder placementGroup(String placementGroup);

        /**
         * <p>
         * One or more subnet IDs, if applicable, separated by commas.
         * </p>
         * <p>
         * If you specify <code>VPCZoneIdentifier</code> and <code>AvailabilityZones</code>, ensure that the
         * Availability Zones of the subnets match the values for <code>AvailabilityZones</code>.
         * </p>
         * 
         * @param vpcZoneIdentifier
         *        One or more subnet IDs, if applicable, separated by commas.</p>
         *        <p>
         *        If you specify <code>VPCZoneIdentifier</code> and <code>AvailabilityZones</code>, ensure that the
         *        Availability Zones of the subnets match the values for <code>AvailabilityZones</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcZoneIdentifier(String vpcZoneIdentifier);

        /**
         * <p>
         * The metrics enabled for the group.
         * </p>
         * 
         * @param enabledMetrics
         *        The metrics enabled for the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabledMetrics(Collection<EnabledMetric> enabledMetrics);

        /**
         * <p>
         * The metrics enabled for the group.
         * </p>
         * 
         * @param enabledMetrics
         *        The metrics enabled for the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabledMetrics(EnabledMetric... enabledMetrics);

        /**
         * <p>
         * The metrics enabled for the group.
         * </p>
         * This is a convenience that creates an instance of the {@link List<EnabledMetric>.Builder} avoiding the need
         * to create one manually via {@link List<EnabledMetric>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<EnabledMetric>.Builder#build()} is called immediately and
         * its result is passed to {@link #enabledMetrics(List<EnabledMetric>)}.
         * 
         * @param enabledMetrics
         *        a consumer that will call methods on {@link List<EnabledMetric>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #enabledMetrics(List<EnabledMetric>)
         */
        Builder enabledMetrics(Consumer<EnabledMetric.Builder>... enabledMetrics);

        /**
         * <p>
         * The current state of the group when <a>DeleteAutoScalingGroup</a> is in progress.
         * </p>
         * 
         * @param status
         *        The current state of the group when <a>DeleteAutoScalingGroup</a> is in progress.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * The tags for the group.
         * </p>
         * 
         * @param tags
         *        The tags for the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<TagDescription> tags);

        /**
         * <p>
         * The tags for the group.
         * </p>
         * 
         * @param tags
         *        The tags for the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(TagDescription... tags);

        /**
         * <p>
         * The tags for the group.
         * </p>
         * This is a convenience that creates an instance of the {@link List<TagDescription>.Builder} avoiding the need
         * to create one manually via {@link List<TagDescription>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<TagDescription>.Builder#build()} is called immediately and
         * its result is passed to {@link #tags(List<TagDescription>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<TagDescription>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<TagDescription>)
         */
        Builder tags(Consumer<TagDescription.Builder>... tags);

        /**
         * <p>
         * The termination policies for the group.
         * </p>
         * 
         * @param terminationPolicies
         *        The termination policies for the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder terminationPolicies(Collection<String> terminationPolicies);

        /**
         * <p>
         * The termination policies for the group.
         * </p>
         * 
         * @param terminationPolicies
         *        The termination policies for the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder terminationPolicies(String... terminationPolicies);

        /**
         * <p>
         * Indicates whether newly launched instances are protected from termination by Auto Scaling when scaling in.
         * </p>
         * 
         * @param newInstancesProtectedFromScaleIn
         *        Indicates whether newly launched instances are protected from termination by Auto Scaling when scaling
         *        in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder newInstancesProtectedFromScaleIn(Boolean newInstancesProtectedFromScaleIn);
    }

    static final class BuilderImpl implements Builder {
        private String autoScalingGroupName;

        private String autoScalingGroupARN;

        private String launchConfigurationName;

        private Integer minSize;

        private Integer maxSize;

        private Integer desiredCapacity;

        private Integer defaultCooldown;

        private List<String> availabilityZones = DefaultSdkAutoConstructList.getInstance();

        private List<String> loadBalancerNames = DefaultSdkAutoConstructList.getInstance();

        private List<String> targetGroupARNs = DefaultSdkAutoConstructList.getInstance();

        private String healthCheckType;

        private Integer healthCheckGracePeriod;

        private List<Instance> instances = DefaultSdkAutoConstructList.getInstance();

        private Instant createdTime;

        private List<SuspendedProcess> suspendedProcesses = DefaultSdkAutoConstructList.getInstance();

        private String placementGroup;

        private String vpcZoneIdentifier;

        private List<EnabledMetric> enabledMetrics = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private List<TagDescription> tags = DefaultSdkAutoConstructList.getInstance();

        private List<String> terminationPolicies = DefaultSdkAutoConstructList.getInstance();

        private Boolean newInstancesProtectedFromScaleIn;

        private BuilderImpl() {
        }

        private BuilderImpl(AutoScalingGroup model) {
            autoScalingGroupName(model.autoScalingGroupName);
            autoScalingGroupARN(model.autoScalingGroupARN);
            launchConfigurationName(model.launchConfigurationName);
            minSize(model.minSize);
            maxSize(model.maxSize);
            desiredCapacity(model.desiredCapacity);
            defaultCooldown(model.defaultCooldown);
            availabilityZones(model.availabilityZones);
            loadBalancerNames(model.loadBalancerNames);
            targetGroupARNs(model.targetGroupARNs);
            healthCheckType(model.healthCheckType);
            healthCheckGracePeriod(model.healthCheckGracePeriod);
            instances(model.instances);
            createdTime(model.createdTime);
            suspendedProcesses(model.suspendedProcesses);
            placementGroup(model.placementGroup);
            vpcZoneIdentifier(model.vpcZoneIdentifier);
            enabledMetrics(model.enabledMetrics);
            status(model.status);
            tags(model.tags);
            terminationPolicies(model.terminationPolicies);
            newInstancesProtectedFromScaleIn(model.newInstancesProtectedFromScaleIn);
        }

        public final String getAutoScalingGroupName() {
            return autoScalingGroupName;
        }

        @Override
        public final Builder autoScalingGroupName(String autoScalingGroupName) {
            this.autoScalingGroupName = autoScalingGroupName;
            return this;
        }

        public final void setAutoScalingGroupName(String autoScalingGroupName) {
            this.autoScalingGroupName = autoScalingGroupName;
        }

        public final String getAutoScalingGroupARN() {
            return autoScalingGroupARN;
        }

        @Override
        public final Builder autoScalingGroupARN(String autoScalingGroupARN) {
            this.autoScalingGroupARN = autoScalingGroupARN;
            return this;
        }

        public final void setAutoScalingGroupARN(String autoScalingGroupARN) {
            this.autoScalingGroupARN = autoScalingGroupARN;
        }

        public final String getLaunchConfigurationName() {
            return launchConfigurationName;
        }

        @Override
        public final Builder launchConfigurationName(String launchConfigurationName) {
            this.launchConfigurationName = launchConfigurationName;
            return this;
        }

        public final void setLaunchConfigurationName(String launchConfigurationName) {
            this.launchConfigurationName = launchConfigurationName;
        }

        public final Integer getMinSize() {
            return minSize;
        }

        @Override
        public final Builder minSize(Integer minSize) {
            this.minSize = minSize;
            return this;
        }

        public final void setMinSize(Integer minSize) {
            this.minSize = minSize;
        }

        public final Integer getMaxSize() {
            return maxSize;
        }

        @Override
        public final Builder maxSize(Integer maxSize) {
            this.maxSize = maxSize;
            return this;
        }

        public final void setMaxSize(Integer maxSize) {
            this.maxSize = maxSize;
        }

        public final Integer getDesiredCapacity() {
            return desiredCapacity;
        }

        @Override
        public final Builder desiredCapacity(Integer desiredCapacity) {
            this.desiredCapacity = desiredCapacity;
            return this;
        }

        public final void setDesiredCapacity(Integer desiredCapacity) {
            this.desiredCapacity = desiredCapacity;
        }

        public final Integer getDefaultCooldown() {
            return defaultCooldown;
        }

        @Override
        public final Builder defaultCooldown(Integer defaultCooldown) {
            this.defaultCooldown = defaultCooldown;
            return this;
        }

        public final void setDefaultCooldown(Integer defaultCooldown) {
            this.defaultCooldown = defaultCooldown;
        }

        public final Collection<String> getAvailabilityZones() {
            return availabilityZones;
        }

        @Override
        public final Builder availabilityZones(Collection<String> availabilityZones) {
            this.availabilityZones = AvailabilityZonesCopier.copy(availabilityZones);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availabilityZones(String... availabilityZones) {
            availabilityZones(Arrays.asList(availabilityZones));
            return this;
        }

        public final void setAvailabilityZones(Collection<String> availabilityZones) {
            this.availabilityZones = AvailabilityZonesCopier.copy(availabilityZones);
        }

        public final Collection<String> getLoadBalancerNames() {
            return loadBalancerNames;
        }

        @Override
        public final Builder loadBalancerNames(Collection<String> loadBalancerNames) {
            this.loadBalancerNames = LoadBalancerNamesCopier.copy(loadBalancerNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder loadBalancerNames(String... loadBalancerNames) {
            loadBalancerNames(Arrays.asList(loadBalancerNames));
            return this;
        }

        public final void setLoadBalancerNames(Collection<String> loadBalancerNames) {
            this.loadBalancerNames = LoadBalancerNamesCopier.copy(loadBalancerNames);
        }

        public final Collection<String> getTargetGroupARNs() {
            return targetGroupARNs;
        }

        @Override
        public final Builder targetGroupARNs(Collection<String> targetGroupARNs) {
            this.targetGroupARNs = TargetGroupARNsCopier.copy(targetGroupARNs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetGroupARNs(String... targetGroupARNs) {
            targetGroupARNs(Arrays.asList(targetGroupARNs));
            return this;
        }

        public final void setTargetGroupARNs(Collection<String> targetGroupARNs) {
            this.targetGroupARNs = TargetGroupARNsCopier.copy(targetGroupARNs);
        }

        public final String getHealthCheckType() {
            return healthCheckType;
        }

        @Override
        public final Builder healthCheckType(String healthCheckType) {
            this.healthCheckType = healthCheckType;
            return this;
        }

        public final void setHealthCheckType(String healthCheckType) {
            this.healthCheckType = healthCheckType;
        }

        public final Integer getHealthCheckGracePeriod() {
            return healthCheckGracePeriod;
        }

        @Override
        public final Builder healthCheckGracePeriod(Integer healthCheckGracePeriod) {
            this.healthCheckGracePeriod = healthCheckGracePeriod;
            return this;
        }

        public final void setHealthCheckGracePeriod(Integer healthCheckGracePeriod) {
            this.healthCheckGracePeriod = healthCheckGracePeriod;
        }

        public final Collection<Instance.Builder> getInstances() {
            return instances != null ? instances.stream().map(Instance::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder instances(Collection<Instance> instances) {
            this.instances = InstancesCopier.copy(instances);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instances(Instance... instances) {
            instances(Arrays.asList(instances));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instances(Consumer<Instance.Builder>... instances) {
            instances(Stream.of(instances).map(c -> Instance.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setInstances(Collection<Instance.BuilderImpl> instances) {
            this.instances = InstancesCopier.copyFromBuilder(instances);
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        @Override
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        public final Collection<SuspendedProcess.Builder> getSuspendedProcesses() {
            return suspendedProcesses != null ? suspendedProcesses.stream().map(SuspendedProcess::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder suspendedProcesses(Collection<SuspendedProcess> suspendedProcesses) {
            this.suspendedProcesses = SuspendedProcessesCopier.copy(suspendedProcesses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder suspendedProcesses(SuspendedProcess... suspendedProcesses) {
            suspendedProcesses(Arrays.asList(suspendedProcesses));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder suspendedProcesses(Consumer<SuspendedProcess.Builder>... suspendedProcesses) {
            suspendedProcesses(Stream.of(suspendedProcesses).map(c -> SuspendedProcess.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setSuspendedProcesses(Collection<SuspendedProcess.BuilderImpl> suspendedProcesses) {
            this.suspendedProcesses = SuspendedProcessesCopier.copyFromBuilder(suspendedProcesses);
        }

        public final String getPlacementGroup() {
            return placementGroup;
        }

        @Override
        public final Builder placementGroup(String placementGroup) {
            this.placementGroup = placementGroup;
            return this;
        }

        public final void setPlacementGroup(String placementGroup) {
            this.placementGroup = placementGroup;
        }

        public final String getVPCZoneIdentifier() {
            return vpcZoneIdentifier;
        }

        @Override
        public final Builder vpcZoneIdentifier(String vpcZoneIdentifier) {
            this.vpcZoneIdentifier = vpcZoneIdentifier;
            return this;
        }

        public final void setVPCZoneIdentifier(String vpcZoneIdentifier) {
            this.vpcZoneIdentifier = vpcZoneIdentifier;
        }

        public final Collection<EnabledMetric.Builder> getEnabledMetrics() {
            return enabledMetrics != null ? enabledMetrics.stream().map(EnabledMetric::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder enabledMetrics(Collection<EnabledMetric> enabledMetrics) {
            this.enabledMetrics = EnabledMetricsCopier.copy(enabledMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder enabledMetrics(EnabledMetric... enabledMetrics) {
            enabledMetrics(Arrays.asList(enabledMetrics));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder enabledMetrics(Consumer<EnabledMetric.Builder>... enabledMetrics) {
            enabledMetrics(Stream.of(enabledMetrics).map(c -> EnabledMetric.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setEnabledMetrics(Collection<EnabledMetric.BuilderImpl> enabledMetrics) {
            this.enabledMetrics = EnabledMetricsCopier.copyFromBuilder(enabledMetrics);
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Collection<TagDescription.Builder> getTags() {
            return tags != null ? tags.stream().map(TagDescription::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<TagDescription> tags) {
            this.tags = TagDescriptionListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(TagDescription... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<TagDescription.Builder>... tags) {
            tags(Stream.of(tags).map(c -> TagDescription.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<TagDescription.BuilderImpl> tags) {
            this.tags = TagDescriptionListCopier.copyFromBuilder(tags);
        }

        public final Collection<String> getTerminationPolicies() {
            return terminationPolicies;
        }

        @Override
        public final Builder terminationPolicies(Collection<String> terminationPolicies) {
            this.terminationPolicies = TerminationPoliciesCopier.copy(terminationPolicies);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder terminationPolicies(String... terminationPolicies) {
            terminationPolicies(Arrays.asList(terminationPolicies));
            return this;
        }

        public final void setTerminationPolicies(Collection<String> terminationPolicies) {
            this.terminationPolicies = TerminationPoliciesCopier.copy(terminationPolicies);
        }

        public final Boolean getNewInstancesProtectedFromScaleIn() {
            return newInstancesProtectedFromScaleIn;
        }

        @Override
        public final Builder newInstancesProtectedFromScaleIn(Boolean newInstancesProtectedFromScaleIn) {
            this.newInstancesProtectedFromScaleIn = newInstancesProtectedFromScaleIn;
            return this;
        }

        public final void setNewInstancesProtectedFromScaleIn(Boolean newInstancesProtectedFromScaleIn) {
            this.newInstancesProtectedFromScaleIn = newInstancesProtectedFromScaleIn;
        }

        @Override
        public AutoScalingGroup build() {
            return new AutoScalingGroup(this);
        }
    }
}
