/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscaling.paginators;

import java.util.Iterator;
import java.util.function.Function;
import javax.annotation.Generated;
import software.amazon.awssdk.core.pagination.NextPageFetcher;
import software.amazon.awssdk.core.pagination.PaginatedItemsIterable;
import software.amazon.awssdk.core.pagination.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.SdkIterable;
import software.amazon.awssdk.services.autoscaling.AutoScalingClient;
import software.amazon.awssdk.services.autoscaling.model.DescribeScheduledActionsRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeScheduledActionsResponse;
import software.amazon.awssdk.services.autoscaling.model.ScheduledUpdateGroupAction;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.autoscaling.AutoScalingClient#describeScheduledActionsIterable(software.amazon.awssdk.services.autoscaling.model.DescribeScheduledActionsRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.autoscaling.model.DescribeScheduledActionsResponse} that can be used to
 * iterate through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.autoscaling.paginators.DescribeScheduledActionsPaginator responses = client.describeScheduledActionsIterable(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.autoscaling.paginators.DescribeScheduledActionsPaginator responses = client
 *             .describeScheduledActionsIterable(request);
 *     for (software.amazon.awssdk.services.autoscaling.model.DescribeScheduledActionsResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.autoscaling.paginators.DescribeScheduledActionsPaginator responses = client.describeScheduledActionsIterable(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #describeScheduledActions(software.amazon.awssdk.services.autoscaling.model.DescribeScheduledActionsRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DescribeScheduledActionsPaginator implements SdkIterable<DescribeScheduledActionsResponse> {
    private final AutoScalingClient client;

    private final DescribeScheduledActionsRequest firstRequest;

    private final NextPageFetcher nextPageFetcher;

    public DescribeScheduledActionsPaginator(final AutoScalingClient client, final DescribeScheduledActionsRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new DescribeScheduledActionsResponseFetcher();
    }

    @Override
    public Iterator<DescribeScheduledActionsResponse> iterator() {
        return new PaginatedResponsesIterator(nextPageFetcher);
    }

    /**
     * Returns an iterable to iterate through the paginated
     * {@link DescribeScheduledActionsResponse#scheduledUpdateGroupActions()} member. The returned iterable is used to
     * iterate through the results across all response pages and not a single page.
     *
     * This method is useful if you are interested in iterating over the paginated member in the response pages instead
     * of the top level pages. Similar to iteration over pages, this method internally makes service calls to get the
     * next list of results until the iteration stops or there are no more results.
     */
    public SdkIterable<ScheduledUpdateGroupAction> scheduledUpdateGroupActions() {
        Function<DescribeScheduledActionsResponse, Iterator<ScheduledUpdateGroupAction>> getIterator = response -> response != null ? response
                .scheduledUpdateGroupActions().iterator() : null;

        return new PaginatedItemsIterable(this, getIterator);
    }

    private class DescribeScheduledActionsResponseFetcher implements NextPageFetcher<DescribeScheduledActionsResponse> {
        @Override
        public boolean hasNextPage(DescribeScheduledActionsResponse previousPage) {
            return previousPage.nextToken() != null;
        }

        @Override
        public DescribeScheduledActionsResponse nextPage(DescribeScheduledActionsResponse previousPage) {
            if (previousPage == null) {
                return client.describeScheduledActions(firstRequest);
            }
            return client.describeScheduledActions(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
