/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscaling.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configures a customized metric for a target tracking policy.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class CustomizedMetricSpecification implements
        ToCopyableBuilder<CustomizedMetricSpecification.Builder, CustomizedMetricSpecification> {
    private final String metricName;

    private final String namespace;

    private final List<MetricDimension> dimensions;

    private final String statistic;

    private final String unit;

    private CustomizedMetricSpecification(BuilderImpl builder) {
        this.metricName = builder.metricName;
        this.namespace = builder.namespace;
        this.dimensions = builder.dimensions;
        this.statistic = builder.statistic;
        this.unit = builder.unit;
    }

    /**
     * <p>
     * The name of the metric.
     * </p>
     * 
     * @return The name of the metric.
     */
    public String metricName() {
        return metricName;
    }

    /**
     * <p>
     * The namespace of the metric.
     * </p>
     * 
     * @return The namespace of the metric.
     */
    public String namespace() {
        return namespace;
    }

    /**
     * <p>
     * The dimensions of the metric.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The dimensions of the metric.
     */
    public List<MetricDimension> dimensions() {
        return dimensions;
    }

    /**
     * <p>
     * The statistic of the metric.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statistic} will
     * return {@link MetricStatistic#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statisticString}.
     * </p>
     * 
     * @return The statistic of the metric.
     * @see MetricStatistic
     */
    public MetricStatistic statistic() {
        return MetricStatistic.fromValue(statistic);
    }

    /**
     * <p>
     * The statistic of the metric.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statistic} will
     * return {@link MetricStatistic#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statisticString}.
     * </p>
     * 
     * @return The statistic of the metric.
     * @see MetricStatistic
     */
    public String statisticString() {
        return statistic;
    }

    /**
     * <p>
     * The unit of the metric.
     * </p>
     * 
     * @return The unit of the metric.
     */
    public String unit() {
        return unit;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(metricName());
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(dimensions());
        hashCode = 31 * hashCode + Objects.hashCode(statisticString());
        hashCode = 31 * hashCode + Objects.hashCode(unit());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CustomizedMetricSpecification)) {
            return false;
        }
        CustomizedMetricSpecification other = (CustomizedMetricSpecification) obj;
        return Objects.equals(metricName(), other.metricName()) && Objects.equals(namespace(), other.namespace())
                && Objects.equals(dimensions(), other.dimensions()) && Objects.equals(statisticString(), other.statisticString())
                && Objects.equals(unit(), other.unit());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (metricName() != null) {
            sb.append("MetricName: ").append(metricName()).append(",");
        }
        if (namespace() != null) {
            sb.append("Namespace: ").append(namespace()).append(",");
        }
        if (dimensions() != null) {
            sb.append("Dimensions: ").append(dimensions()).append(",");
        }
        if (statisticString() != null) {
            sb.append("Statistic: ").append(statisticString()).append(",");
        }
        if (unit() != null) {
            sb.append("Unit: ").append(unit()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MetricName":
            return Optional.of(clazz.cast(metricName()));
        case "Namespace":
            return Optional.of(clazz.cast(namespace()));
        case "Dimensions":
            return Optional.of(clazz.cast(dimensions()));
        case "Statistic":
            return Optional.of(clazz.cast(statisticString()));
        case "Unit":
            return Optional.of(clazz.cast(unit()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, CustomizedMetricSpecification> {
        /**
         * <p>
         * The name of the metric.
         * </p>
         * 
         * @param metricName
         *        The name of the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * The namespace of the metric.
         * </p>
         * 
         * @param namespace
         *        The namespace of the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * The dimensions of the metric.
         * </p>
         * 
         * @param dimensions
         *        The dimensions of the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Collection<MetricDimension> dimensions);

        /**
         * <p>
         * The dimensions of the metric.
         * </p>
         * 
         * @param dimensions
         *        The dimensions of the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(MetricDimension... dimensions);

        /**
         * <p>
         * The statistic of the metric.
         * </p>
         * 
         * @param statistic
         *        The statistic of the metric.
         * @see MetricStatistic
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricStatistic
         */
        Builder statistic(String statistic);

        /**
         * <p>
         * The statistic of the metric.
         * </p>
         * 
         * @param statistic
         *        The statistic of the metric.
         * @see MetricStatistic
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricStatistic
         */
        Builder statistic(MetricStatistic statistic);

        /**
         * <p>
         * The unit of the metric.
         * </p>
         * 
         * @param unit
         *        The unit of the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unit(String unit);
    }

    static final class BuilderImpl implements Builder {
        private String metricName;

        private String namespace;

        private List<MetricDimension> dimensions;

        private String statistic;

        private String unit;

        private BuilderImpl() {
        }

        private BuilderImpl(CustomizedMetricSpecification model) {
            metricName(model.metricName);
            namespace(model.namespace);
            dimensions(model.dimensions);
            statistic(model.statistic);
            unit(model.unit);
        }

        public final String getMetricName() {
            return metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        public final String getNamespace() {
            return namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        public final Collection<MetricDimension.Builder> getDimensions() {
            return dimensions != null ? dimensions.stream().map(MetricDimension::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder dimensions(Collection<MetricDimension> dimensions) {
            this.dimensions = MetricDimensionsCopier.copy(dimensions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(MetricDimension... dimensions) {
            dimensions(Arrays.asList(dimensions));
            return this;
        }

        public final void setDimensions(Collection<MetricDimension.BuilderImpl> dimensions) {
            this.dimensions = MetricDimensionsCopier.copyFromBuilder(dimensions);
        }

        public final String getStatistic() {
            return statistic;
        }

        @Override
        public final Builder statistic(String statistic) {
            this.statistic = statistic;
            return this;
        }

        @Override
        public final Builder statistic(MetricStatistic statistic) {
            this.statistic(statistic.toString());
            return this;
        }

        public final void setStatistic(String statistic) {
            this.statistic = statistic;
        }

        public final String getUnit() {
            return unit;
        }

        @Override
        public final Builder unit(String unit) {
            this.unit = unit;
            return this;
        }

        public final void setUnit(String unit) {
            this.unit = unit;
        }

        @Override
        public CustomizedMetricSpecification build() {
            return new CustomizedMetricSpecification(this);
        }
    }
}
