/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscaling.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a launch template and overrides.
 * </p>
 * <p>
 * The overrides are used to override the instance type specified by the launch template with multiple instance types
 * that can be used to launch On-Demand Instances and Spot Instances.
 * </p>
 * <p>
 * When you update the launch template or overrides, existing Amazon EC2 instances continue to run. When scale out
 * occurs, Amazon EC2 Auto Scaling launches instances to match the new settings. When scale in occurs, Amazon EC2 Auto
 * Scaling terminates instances according to the group's termination policies.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LaunchTemplate implements SdkPojo, Serializable, ToCopyableBuilder<LaunchTemplate.Builder, LaunchTemplate> {
    private static final SdkField<LaunchTemplateSpecification> LAUNCH_TEMPLATE_SPECIFICATION_FIELD = SdkField
            .<LaunchTemplateSpecification> builder(MarshallingType.SDK_POJO)
            .getter(getter(LaunchTemplate::launchTemplateSpecification))
            .setter(setter(Builder::launchTemplateSpecification))
            .constructor(LaunchTemplateSpecification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LaunchTemplateSpecification")
                    .build()).build();

    private static final SdkField<List<LaunchTemplateOverrides>> OVERRIDES_FIELD = SdkField
            .<List<LaunchTemplateOverrides>> builder(MarshallingType.LIST)
            .getter(getter(LaunchTemplate::overrides))
            .setter(setter(Builder::overrides))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Overrides").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LaunchTemplateOverrides> builder(MarshallingType.SDK_POJO)
                                            .constructor(LaunchTemplateOverrides::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            LAUNCH_TEMPLATE_SPECIFICATION_FIELD, OVERRIDES_FIELD));

    private static final long serialVersionUID = 1L;

    private final LaunchTemplateSpecification launchTemplateSpecification;

    private final List<LaunchTemplateOverrides> overrides;

    private LaunchTemplate(BuilderImpl builder) {
        this.launchTemplateSpecification = builder.launchTemplateSpecification;
        this.overrides = builder.overrides;
    }

    /**
     * <p>
     * The launch template to use. You must specify either the launch template ID or launch template name in the
     * request.
     * </p>
     * 
     * @return The launch template to use. You must specify either the launch template ID or launch template name in the
     *         request.
     */
    public LaunchTemplateSpecification launchTemplateSpecification() {
        return launchTemplateSpecification;
    }

    /**
     * <p>
     * An optional setting. Any parameters that you specify override the same parameters in the launch template.
     * Currently, the only supported override is instance type. You can specify between 1 and 20 instance types.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An optional setting. Any parameters that you specify override the same parameters in the launch template.
     *         Currently, the only supported override is instance type. You can specify between 1 and 20 instance types.
     */
    public List<LaunchTemplateOverrides> overrides() {
        return overrides;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(launchTemplateSpecification());
        hashCode = 31 * hashCode + Objects.hashCode(overrides());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LaunchTemplate)) {
            return false;
        }
        LaunchTemplate other = (LaunchTemplate) obj;
        return Objects.equals(launchTemplateSpecification(), other.launchTemplateSpecification())
                && Objects.equals(overrides(), other.overrides());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("LaunchTemplate").add("LaunchTemplateSpecification", launchTemplateSpecification())
                .add("Overrides", overrides()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "LaunchTemplateSpecification":
            return Optional.ofNullable(clazz.cast(launchTemplateSpecification()));
        case "Overrides":
            return Optional.ofNullable(clazz.cast(overrides()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LaunchTemplate, T> g) {
        return obj -> g.apply((LaunchTemplate) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LaunchTemplate> {
        /**
         * <p>
         * The launch template to use. You must specify either the launch template ID or launch template name in the
         * request.
         * </p>
         * 
         * @param launchTemplateSpecification
         *        The launch template to use. You must specify either the launch template ID or launch template name in
         *        the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchTemplateSpecification(LaunchTemplateSpecification launchTemplateSpecification);

        /**
         * <p>
         * The launch template to use. You must specify either the launch template ID or launch template name in the
         * request.
         * </p>
         * This is a convenience that creates an instance of the {@link LaunchTemplateSpecification.Builder} avoiding
         * the need to create one manually via {@link LaunchTemplateSpecification#builder()}.
         *
         * When the {@link Consumer} completes, {@link LaunchTemplateSpecification.Builder#build()} is called
         * immediately and its result is passed to {@link #launchTemplateSpecification(LaunchTemplateSpecification)}.
         * 
         * @param launchTemplateSpecification
         *        a consumer that will call methods on {@link LaunchTemplateSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #launchTemplateSpecification(LaunchTemplateSpecification)
         */
        default Builder launchTemplateSpecification(Consumer<LaunchTemplateSpecification.Builder> launchTemplateSpecification) {
            return launchTemplateSpecification(LaunchTemplateSpecification.builder().applyMutation(launchTemplateSpecification)
                    .build());
        }

        /**
         * <p>
         * An optional setting. Any parameters that you specify override the same parameters in the launch template.
         * Currently, the only supported override is instance type. You can specify between 1 and 20 instance types.
         * </p>
         * 
         * @param overrides
         *        An optional setting. Any parameters that you specify override the same parameters in the launch
         *        template. Currently, the only supported override is instance type. You can specify between 1 and 20
         *        instance types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overrides(Collection<LaunchTemplateOverrides> overrides);

        /**
         * <p>
         * An optional setting. Any parameters that you specify override the same parameters in the launch template.
         * Currently, the only supported override is instance type. You can specify between 1 and 20 instance types.
         * </p>
         * 
         * @param overrides
         *        An optional setting. Any parameters that you specify override the same parameters in the launch
         *        template. Currently, the only supported override is instance type. You can specify between 1 and 20
         *        instance types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overrides(LaunchTemplateOverrides... overrides);

        /**
         * <p>
         * An optional setting. Any parameters that you specify override the same parameters in the launch template.
         * Currently, the only supported override is instance type. You can specify between 1 and 20 instance types.
         * </p>
         * This is a convenience that creates an instance of the {@link List<LaunchTemplateOverrides>.Builder} avoiding
         * the need to create one manually via {@link List<LaunchTemplateOverrides>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<LaunchTemplateOverrides>.Builder#build()} is called
         * immediately and its result is passed to {@link #overrides(List<LaunchTemplateOverrides>)}.
         * 
         * @param overrides
         *        a consumer that will call methods on {@link List<LaunchTemplateOverrides>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #overrides(List<LaunchTemplateOverrides>)
         */
        Builder overrides(Consumer<LaunchTemplateOverrides.Builder>... overrides);
    }

    static final class BuilderImpl implements Builder {
        private LaunchTemplateSpecification launchTemplateSpecification;

        private List<LaunchTemplateOverrides> overrides = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LaunchTemplate model) {
            launchTemplateSpecification(model.launchTemplateSpecification);
            overrides(model.overrides);
        }

        public final LaunchTemplateSpecification.Builder getLaunchTemplateSpecification() {
            return launchTemplateSpecification != null ? launchTemplateSpecification.toBuilder() : null;
        }

        @Override
        public final Builder launchTemplateSpecification(LaunchTemplateSpecification launchTemplateSpecification) {
            this.launchTemplateSpecification = launchTemplateSpecification;
            return this;
        }

        public final void setLaunchTemplateSpecification(LaunchTemplateSpecification.BuilderImpl launchTemplateSpecification) {
            this.launchTemplateSpecification = launchTemplateSpecification != null ? launchTemplateSpecification.build() : null;
        }

        public final Collection<LaunchTemplateOverrides.Builder> getOverrides() {
            return overrides != null ? overrides.stream().map(LaunchTemplateOverrides::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder overrides(Collection<LaunchTemplateOverrides> overrides) {
            this.overrides = OverridesCopier.copy(overrides);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder overrides(LaunchTemplateOverrides... overrides) {
            overrides(Arrays.asList(overrides));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder overrides(Consumer<LaunchTemplateOverrides.Builder>... overrides) {
            overrides(Stream.of(overrides).map(c -> LaunchTemplateOverrides.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setOverrides(Collection<LaunchTemplateOverrides.BuilderImpl> overrides) {
            this.overrides = OverridesCopier.copyFromBuilder(overrides);
        }

        @Override
        public LaunchTemplate build() {
            return new LaunchTemplate(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
