/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscaling;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.query.AwsQueryProtocolFactory;
import software.amazon.awssdk.services.autoscaling.model.ActiveInstanceRefreshNotFoundException;
import software.amazon.awssdk.services.autoscaling.model.AlreadyExistsException;
import software.amazon.awssdk.services.autoscaling.model.AttachInstancesRequest;
import software.amazon.awssdk.services.autoscaling.model.AttachInstancesResponse;
import software.amazon.awssdk.services.autoscaling.model.AttachLoadBalancerTargetGroupsRequest;
import software.amazon.awssdk.services.autoscaling.model.AttachLoadBalancerTargetGroupsResponse;
import software.amazon.awssdk.services.autoscaling.model.AttachLoadBalancersRequest;
import software.amazon.awssdk.services.autoscaling.model.AttachLoadBalancersResponse;
import software.amazon.awssdk.services.autoscaling.model.AutoScalingException;
import software.amazon.awssdk.services.autoscaling.model.AutoScalingRequest;
import software.amazon.awssdk.services.autoscaling.model.BatchDeleteScheduledActionRequest;
import software.amazon.awssdk.services.autoscaling.model.BatchDeleteScheduledActionResponse;
import software.amazon.awssdk.services.autoscaling.model.BatchPutScheduledUpdateGroupActionRequest;
import software.amazon.awssdk.services.autoscaling.model.BatchPutScheduledUpdateGroupActionResponse;
import software.amazon.awssdk.services.autoscaling.model.CancelInstanceRefreshRequest;
import software.amazon.awssdk.services.autoscaling.model.CancelInstanceRefreshResponse;
import software.amazon.awssdk.services.autoscaling.model.CompleteLifecycleActionRequest;
import software.amazon.awssdk.services.autoscaling.model.CompleteLifecycleActionResponse;
import software.amazon.awssdk.services.autoscaling.model.CreateAutoScalingGroupRequest;
import software.amazon.awssdk.services.autoscaling.model.CreateAutoScalingGroupResponse;
import software.amazon.awssdk.services.autoscaling.model.CreateLaunchConfigurationRequest;
import software.amazon.awssdk.services.autoscaling.model.CreateLaunchConfigurationResponse;
import software.amazon.awssdk.services.autoscaling.model.CreateOrUpdateTagsRequest;
import software.amazon.awssdk.services.autoscaling.model.CreateOrUpdateTagsResponse;
import software.amazon.awssdk.services.autoscaling.model.DeleteAutoScalingGroupRequest;
import software.amazon.awssdk.services.autoscaling.model.DeleteAutoScalingGroupResponse;
import software.amazon.awssdk.services.autoscaling.model.DeleteLaunchConfigurationRequest;
import software.amazon.awssdk.services.autoscaling.model.DeleteLaunchConfigurationResponse;
import software.amazon.awssdk.services.autoscaling.model.DeleteLifecycleHookRequest;
import software.amazon.awssdk.services.autoscaling.model.DeleteLifecycleHookResponse;
import software.amazon.awssdk.services.autoscaling.model.DeleteNotificationConfigurationRequest;
import software.amazon.awssdk.services.autoscaling.model.DeleteNotificationConfigurationResponse;
import software.amazon.awssdk.services.autoscaling.model.DeletePolicyRequest;
import software.amazon.awssdk.services.autoscaling.model.DeletePolicyResponse;
import software.amazon.awssdk.services.autoscaling.model.DeleteScheduledActionRequest;
import software.amazon.awssdk.services.autoscaling.model.DeleteScheduledActionResponse;
import software.amazon.awssdk.services.autoscaling.model.DeleteTagsRequest;
import software.amazon.awssdk.services.autoscaling.model.DeleteTagsResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeAccountLimitsRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeAccountLimitsResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeAdjustmentTypesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeAdjustmentTypesResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingGroupsRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingGroupsResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingInstancesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingInstancesResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingNotificationTypesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingNotificationTypesResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeInstanceRefreshesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeInstanceRefreshesResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeLaunchConfigurationsRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeLaunchConfigurationsResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeLifecycleHookTypesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeLifecycleHookTypesResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeLifecycleHooksRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeLifecycleHooksResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeLoadBalancerTargetGroupsRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeLoadBalancerTargetGroupsResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeLoadBalancersRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeLoadBalancersResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeMetricCollectionTypesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeMetricCollectionTypesResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeNotificationConfigurationsRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeNotificationConfigurationsResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribePoliciesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribePoliciesResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeScalingActivitiesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeScalingActivitiesResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeScalingProcessTypesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeScalingProcessTypesResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeScheduledActionsRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeScheduledActionsResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeTagsRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeTagsResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeTerminationPolicyTypesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeTerminationPolicyTypesResponse;
import software.amazon.awssdk.services.autoscaling.model.DetachInstancesRequest;
import software.amazon.awssdk.services.autoscaling.model.DetachInstancesResponse;
import software.amazon.awssdk.services.autoscaling.model.DetachLoadBalancerTargetGroupsRequest;
import software.amazon.awssdk.services.autoscaling.model.DetachLoadBalancerTargetGroupsResponse;
import software.amazon.awssdk.services.autoscaling.model.DetachLoadBalancersRequest;
import software.amazon.awssdk.services.autoscaling.model.DetachLoadBalancersResponse;
import software.amazon.awssdk.services.autoscaling.model.DisableMetricsCollectionRequest;
import software.amazon.awssdk.services.autoscaling.model.DisableMetricsCollectionResponse;
import software.amazon.awssdk.services.autoscaling.model.EnableMetricsCollectionRequest;
import software.amazon.awssdk.services.autoscaling.model.EnableMetricsCollectionResponse;
import software.amazon.awssdk.services.autoscaling.model.EnterStandbyRequest;
import software.amazon.awssdk.services.autoscaling.model.EnterStandbyResponse;
import software.amazon.awssdk.services.autoscaling.model.ExecutePolicyRequest;
import software.amazon.awssdk.services.autoscaling.model.ExecutePolicyResponse;
import software.amazon.awssdk.services.autoscaling.model.ExitStandbyRequest;
import software.amazon.awssdk.services.autoscaling.model.ExitStandbyResponse;
import software.amazon.awssdk.services.autoscaling.model.InstanceRefreshInProgressException;
import software.amazon.awssdk.services.autoscaling.model.InvalidNextTokenException;
import software.amazon.awssdk.services.autoscaling.model.LimitExceededException;
import software.amazon.awssdk.services.autoscaling.model.PutLifecycleHookRequest;
import software.amazon.awssdk.services.autoscaling.model.PutLifecycleHookResponse;
import software.amazon.awssdk.services.autoscaling.model.PutNotificationConfigurationRequest;
import software.amazon.awssdk.services.autoscaling.model.PutNotificationConfigurationResponse;
import software.amazon.awssdk.services.autoscaling.model.PutScalingPolicyRequest;
import software.amazon.awssdk.services.autoscaling.model.PutScalingPolicyResponse;
import software.amazon.awssdk.services.autoscaling.model.PutScheduledUpdateGroupActionRequest;
import software.amazon.awssdk.services.autoscaling.model.PutScheduledUpdateGroupActionResponse;
import software.amazon.awssdk.services.autoscaling.model.RecordLifecycleActionHeartbeatRequest;
import software.amazon.awssdk.services.autoscaling.model.RecordLifecycleActionHeartbeatResponse;
import software.amazon.awssdk.services.autoscaling.model.ResourceContentionException;
import software.amazon.awssdk.services.autoscaling.model.ResourceInUseException;
import software.amazon.awssdk.services.autoscaling.model.ResumeProcessesRequest;
import software.amazon.awssdk.services.autoscaling.model.ResumeProcessesResponse;
import software.amazon.awssdk.services.autoscaling.model.ScalingActivityInProgressException;
import software.amazon.awssdk.services.autoscaling.model.ServiceLinkedRoleFailureException;
import software.amazon.awssdk.services.autoscaling.model.SetDesiredCapacityRequest;
import software.amazon.awssdk.services.autoscaling.model.SetDesiredCapacityResponse;
import software.amazon.awssdk.services.autoscaling.model.SetInstanceHealthRequest;
import software.amazon.awssdk.services.autoscaling.model.SetInstanceHealthResponse;
import software.amazon.awssdk.services.autoscaling.model.SetInstanceProtectionRequest;
import software.amazon.awssdk.services.autoscaling.model.SetInstanceProtectionResponse;
import software.amazon.awssdk.services.autoscaling.model.StartInstanceRefreshRequest;
import software.amazon.awssdk.services.autoscaling.model.StartInstanceRefreshResponse;
import software.amazon.awssdk.services.autoscaling.model.SuspendProcessesRequest;
import software.amazon.awssdk.services.autoscaling.model.SuspendProcessesResponse;
import software.amazon.awssdk.services.autoscaling.model.TerminateInstanceInAutoScalingGroupRequest;
import software.amazon.awssdk.services.autoscaling.model.TerminateInstanceInAutoScalingGroupResponse;
import software.amazon.awssdk.services.autoscaling.model.UpdateAutoScalingGroupRequest;
import software.amazon.awssdk.services.autoscaling.model.UpdateAutoScalingGroupResponse;
import software.amazon.awssdk.services.autoscaling.paginators.DescribeAutoScalingGroupsIterable;
import software.amazon.awssdk.services.autoscaling.paginators.DescribeAutoScalingInstancesIterable;
import software.amazon.awssdk.services.autoscaling.paginators.DescribeLaunchConfigurationsIterable;
import software.amazon.awssdk.services.autoscaling.paginators.DescribeNotificationConfigurationsIterable;
import software.amazon.awssdk.services.autoscaling.paginators.DescribePoliciesIterable;
import software.amazon.awssdk.services.autoscaling.paginators.DescribeScalingActivitiesIterable;
import software.amazon.awssdk.services.autoscaling.paginators.DescribeScheduledActionsIterable;
import software.amazon.awssdk.services.autoscaling.paginators.DescribeTagsIterable;
import software.amazon.awssdk.services.autoscaling.transform.AttachInstancesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.AttachLoadBalancerTargetGroupsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.AttachLoadBalancersRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.BatchDeleteScheduledActionRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.BatchPutScheduledUpdateGroupActionRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.CancelInstanceRefreshRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.CompleteLifecycleActionRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.CreateAutoScalingGroupRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.CreateLaunchConfigurationRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.CreateOrUpdateTagsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DeleteAutoScalingGroupRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DeleteLaunchConfigurationRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DeleteLifecycleHookRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DeleteNotificationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DeletePolicyRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DeleteScheduledActionRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DeleteTagsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeAccountLimitsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeAdjustmentTypesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeAutoScalingGroupsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeAutoScalingInstancesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeAutoScalingNotificationTypesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeInstanceRefreshesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeLaunchConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeLifecycleHookTypesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeLifecycleHooksRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeLoadBalancerTargetGroupsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeLoadBalancersRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeMetricCollectionTypesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeNotificationConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribePoliciesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeScalingActivitiesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeScalingProcessTypesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeScheduledActionsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeTagsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeTerminationPolicyTypesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DetachInstancesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DetachLoadBalancerTargetGroupsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DetachLoadBalancersRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DisableMetricsCollectionRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.EnableMetricsCollectionRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.EnterStandbyRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.ExecutePolicyRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.ExitStandbyRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.PutLifecycleHookRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.PutNotificationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.PutScalingPolicyRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.PutScheduledUpdateGroupActionRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.RecordLifecycleActionHeartbeatRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.ResumeProcessesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.SetDesiredCapacityRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.SetInstanceHealthRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.SetInstanceProtectionRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.StartInstanceRefreshRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.SuspendProcessesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.TerminateInstanceInAutoScalingGroupRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.UpdateAutoScalingGroupRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.waiters.AutoScalingWaiter;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link AutoScalingClient}.
 *
 * @see AutoScalingClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAutoScalingClient implements AutoScalingClient {
    private static final Logger log = Logger.loggerFor(DefaultAutoScalingClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsQueryProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAutoScalingClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Attaches one or more EC2 instances to the specified Auto Scaling group.
     * </p>
     * <p>
     * When you attach instances, Amazon EC2 Auto Scaling increases the desired capacity of the group by the number of
     * instances being attached. If the number of instances being attached plus the desired capacity of the group
     * exceeds the maximum size of the group, the operation fails.
     * </p>
     * <p>
     * If there is a Classic Load Balancer attached to your Auto Scaling group, the instances are also registered with
     * the load balancer. If there are target groups attached to your Auto Scaling group, the instances are also
     * registered with the target groups.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/attach-instance-asg.html">Attach EC2 instances to
     * your Auto Scaling group</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param attachInstancesRequest
     * @return Result of the AttachInstances operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws ServiceLinkedRoleFailureException
     *         The service-linked role is not yet ready for use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.AttachInstances
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/AttachInstances" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public AttachInstancesResponse attachInstances(AttachInstancesRequest attachInstancesRequest)
            throws ResourceContentionException, ServiceLinkedRoleFailureException, AwsServiceException, SdkClientException,
            AutoScalingException {

        HttpResponseHandler<AttachInstancesResponse> responseHandler = protocolFactory
                .createResponseHandler(AttachInstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, attachInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AttachInstances");

            return clientHandler.execute(new ClientExecutionParams<AttachInstancesRequest, AttachInstancesResponse>()
                    .withOperationName("AttachInstances").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(attachInstancesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AttachInstancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Attaches one or more target groups to the specified Auto Scaling group.
     * </p>
     * <p>
     * This operation is used with the following load balancer types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Application Load Balancer - Operates at the application layer (layer 7) and supports HTTP and HTTPS.
     * </p>
     * </li>
     * <li>
     * <p>
     * Network Load Balancer - Operates at the transport layer (layer 4) and supports TCP, TLS, and UDP.
     * </p>
     * </li>
     * <li>
     * <p>
     * Gateway Load Balancer - Operates at the network layer (layer 3).
     * </p>
     * </li>
     * </ul>
     * <p>
     * To describe the target groups for an Auto Scaling group, call the <a>DescribeLoadBalancerTargetGroups</a> API. To
     * detach the target group from the Auto Scaling group, call the <a>DetachLoadBalancerTargetGroups</a> API.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/autoscaling-load-balancer.html">Elastic Load
     * Balancing and Amazon EC2 Auto Scaling</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param attachLoadBalancerTargetGroupsRequest
     * @return Result of the AttachLoadBalancerTargetGroups operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws ServiceLinkedRoleFailureException
     *         The service-linked role is not yet ready for use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.AttachLoadBalancerTargetGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/AttachLoadBalancerTargetGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AttachLoadBalancerTargetGroupsResponse attachLoadBalancerTargetGroups(
            AttachLoadBalancerTargetGroupsRequest attachLoadBalancerTargetGroupsRequest) throws ResourceContentionException,
            ServiceLinkedRoleFailureException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<AttachLoadBalancerTargetGroupsResponse> responseHandler = protocolFactory
                .createResponseHandler(AttachLoadBalancerTargetGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                attachLoadBalancerTargetGroupsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AttachLoadBalancerTargetGroups");

            return clientHandler
                    .execute(new ClientExecutionParams<AttachLoadBalancerTargetGroupsRequest, AttachLoadBalancerTargetGroupsResponse>()
                            .withOperationName("AttachLoadBalancerTargetGroups").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(attachLoadBalancerTargetGroupsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AttachLoadBalancerTargetGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <note>
     * <p>
     * To attach an Application Load Balancer, Network Load Balancer, or Gateway Load Balancer, use the
     * <a>AttachLoadBalancerTargetGroups</a> API operation instead.
     * </p>
     * </note>
     * <p>
     * Attaches one or more Classic Load Balancers to the specified Auto Scaling group. Amazon EC2 Auto Scaling
     * registers the running instances with these Classic Load Balancers.
     * </p>
     * <p>
     * To describe the load balancers for an Auto Scaling group, call the <a>DescribeLoadBalancers</a> API. To detach
     * the load balancer from the Auto Scaling group, call the <a>DetachLoadBalancers</a> API.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/autoscaling-load-balancer.html">Elastic Load
     * Balancing and Amazon EC2 Auto Scaling</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param attachLoadBalancersRequest
     * @return Result of the AttachLoadBalancers operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws ServiceLinkedRoleFailureException
     *         The service-linked role is not yet ready for use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.AttachLoadBalancers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/AttachLoadBalancers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AttachLoadBalancersResponse attachLoadBalancers(AttachLoadBalancersRequest attachLoadBalancersRequest)
            throws ResourceContentionException, ServiceLinkedRoleFailureException, AwsServiceException, SdkClientException,
            AutoScalingException {

        HttpResponseHandler<AttachLoadBalancersResponse> responseHandler = protocolFactory
                .createResponseHandler(AttachLoadBalancersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, attachLoadBalancersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AttachLoadBalancers");

            return clientHandler.execute(new ClientExecutionParams<AttachLoadBalancersRequest, AttachLoadBalancersResponse>()
                    .withOperationName("AttachLoadBalancers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(attachLoadBalancersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AttachLoadBalancersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes one or more scheduled actions for the specified Auto Scaling group.
     * </p>
     *
     * @param batchDeleteScheduledActionRequest
     * @return Result of the BatchDeleteScheduledAction operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.BatchDeleteScheduledAction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/BatchDeleteScheduledAction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchDeleteScheduledActionResponse batchDeleteScheduledAction(
            BatchDeleteScheduledActionRequest batchDeleteScheduledActionRequest) throws ResourceContentionException,
            AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<BatchDeleteScheduledActionResponse> responseHandler = protocolFactory
                .createResponseHandler(BatchDeleteScheduledActionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDeleteScheduledActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDeleteScheduledAction");

            return clientHandler
                    .execute(new ClientExecutionParams<BatchDeleteScheduledActionRequest, BatchDeleteScheduledActionResponse>()
                            .withOperationName("BatchDeleteScheduledAction").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(batchDeleteScheduledActionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new BatchDeleteScheduledActionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates one or more scheduled scaling actions for an Auto Scaling group. If you leave a parameter
     * unspecified when updating a scheduled scaling action, the corresponding value remains unchanged.
     * </p>
     *
     * @param batchPutScheduledUpdateGroupActionRequest
     * @return Result of the BatchPutScheduledUpdateGroupAction operation returned by the service.
     * @throws AlreadyExistsException
     *         You already have an Auto Scaling group or launch configuration with this name.
     * @throws LimitExceededException
     *         You have already reached a limit for your Amazon EC2 Auto Scaling resources (for example, Auto Scaling
     *         groups, launch configurations, or lifecycle hooks). For more information, see <a
     *         href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html"
     *         >DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.BatchPutScheduledUpdateGroupAction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/BatchPutScheduledUpdateGroupAction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchPutScheduledUpdateGroupActionResponse batchPutScheduledUpdateGroupAction(
            BatchPutScheduledUpdateGroupActionRequest batchPutScheduledUpdateGroupActionRequest) throws AlreadyExistsException,
            LimitExceededException, ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<BatchPutScheduledUpdateGroupActionResponse> responseHandler = protocolFactory
                .createResponseHandler(BatchPutScheduledUpdateGroupActionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                batchPutScheduledUpdateGroupActionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchPutScheduledUpdateGroupAction");

            return clientHandler
                    .execute(new ClientExecutionParams<BatchPutScheduledUpdateGroupActionRequest, BatchPutScheduledUpdateGroupActionResponse>()
                            .withOperationName("BatchPutScheduledUpdateGroupAction").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(batchPutScheduledUpdateGroupActionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new BatchPutScheduledUpdateGroupActionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Cancels an instance refresh operation in progress. Cancellation does not roll back any replacements that have
     * already been completed, but it prevents new replacements from being started.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-instance-refresh.html">Replacing Auto Scaling
     * Instances Based on an Instance Refresh</a>.
     * </p>
     *
     * @param cancelInstanceRefreshRequest
     * @return Result of the CancelInstanceRefresh operation returned by the service.
     * @throws LimitExceededException
     *         You have already reached a limit for your Amazon EC2 Auto Scaling resources (for example, Auto Scaling
     *         groups, launch configurations, or lifecycle hooks). For more information, see <a
     *         href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html"
     *         >DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws ActiveInstanceRefreshNotFoundException
     *         The request failed because an active instance refresh for the specified Auto Scaling group was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.CancelInstanceRefresh
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/CancelInstanceRefresh"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CancelInstanceRefreshResponse cancelInstanceRefresh(CancelInstanceRefreshRequest cancelInstanceRefreshRequest)
            throws LimitExceededException, ResourceContentionException, ActiveInstanceRefreshNotFoundException,
            AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<CancelInstanceRefreshResponse> responseHandler = protocolFactory
                .createResponseHandler(CancelInstanceRefreshResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelInstanceRefreshRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelInstanceRefresh");

            return clientHandler.execute(new ClientExecutionParams<CancelInstanceRefreshRequest, CancelInstanceRefreshResponse>()
                    .withOperationName("CancelInstanceRefresh").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(cancelInstanceRefreshRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelInstanceRefreshRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Completes the lifecycle action for the specified token or instance with the specified result.
     * </p>
     * <p>
     * This step is a part of the procedure for adding a lifecycle hook to an Auto Scaling group:
     * </p>
     * <ol>
     * <li>
     * <p>
     * (Optional) Create a Lambda function and a rule that allows CloudWatch Events to invoke your Lambda function when
     * Amazon EC2 Auto Scaling launches or terminates instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * (Optional) Create a notification target and an IAM role. The target can be either an Amazon SQS queue or an
     * Amazon SNS topic. The role allows Amazon EC2 Auto Scaling to publish lifecycle notifications to the target.
     * </p>
     * </li>
     * <li>
     * <p>
     * Create the lifecycle hook. Specify whether the hook is used when the instances launch or terminate.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you need more time, record the lifecycle action heartbeat to keep the instance in a pending state.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>If you finish before the timeout period ends, complete the lifecycle action.</b>
     * </p>
     * </li>
     * </ol>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/lifecycle-hooks.html">Amazon EC2 Auto Scaling
     * lifecycle hooks</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param completeLifecycleActionRequest
     * @return Result of the CompleteLifecycleAction operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.CompleteLifecycleAction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/CompleteLifecycleAction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompleteLifecycleActionResponse completeLifecycleAction(CompleteLifecycleActionRequest completeLifecycleActionRequest)
            throws ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<CompleteLifecycleActionResponse> responseHandler = protocolFactory
                .createResponseHandler(CompleteLifecycleActionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, completeLifecycleActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CompleteLifecycleAction");

            return clientHandler
                    .execute(new ClientExecutionParams<CompleteLifecycleActionRequest, CompleteLifecycleActionResponse>()
                            .withOperationName("CompleteLifecycleAction").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(completeLifecycleActionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CompleteLifecycleActionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an Auto Scaling group with the specified name and attributes.
     * </p>
     * <p>
     * If you exceed your maximum limit of Auto Scaling groups, the call fails. To query this limit, call the
     * <a>DescribeAccountLimits</a> API. For information about updating this limit, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-account-limits.html">Amazon EC2 Auto Scaling
     * service quotas</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * For introductory exercises for creating an Auto Scaling group, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/GettingStartedTutorial.html">Getting started with
     * Amazon EC2 Auto Scaling</a> and <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-register-lbs-with-asg.html">Tutorial: Set up a
     * scaled and load-balanced application</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>. For more information,
     * see <a href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/AutoScalingGroup.html">Auto Scaling groups</a>
     * in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * Every Auto Scaling group has three size parameters (<code>DesiredCapacity</code>, <code>MaxSize</code>, and
     * <code>MinSize</code>). Usually, you set these sizes based on a specific number of instances. However, if you
     * configure a mixed instances policy that defines weights for the instance types, you must specify these sizes with
     * the same units that you use for weighting instances.
     * </p>
     *
     * @param createAutoScalingGroupRequest
     * @return Result of the CreateAutoScalingGroup operation returned by the service.
     * @throws AlreadyExistsException
     *         You already have an Auto Scaling group or launch configuration with this name.
     * @throws LimitExceededException
     *         You have already reached a limit for your Amazon EC2 Auto Scaling resources (for example, Auto Scaling
     *         groups, launch configurations, or lifecycle hooks). For more information, see <a
     *         href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html"
     *         >DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws ServiceLinkedRoleFailureException
     *         The service-linked role is not yet ready for use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.CreateAutoScalingGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/CreateAutoScalingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateAutoScalingGroupResponse createAutoScalingGroup(CreateAutoScalingGroupRequest createAutoScalingGroupRequest)
            throws AlreadyExistsException, LimitExceededException, ResourceContentionException,
            ServiceLinkedRoleFailureException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<CreateAutoScalingGroupResponse> responseHandler = protocolFactory
                .createResponseHandler(CreateAutoScalingGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAutoScalingGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAutoScalingGroup");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateAutoScalingGroupRequest, CreateAutoScalingGroupResponse>()
                            .withOperationName("CreateAutoScalingGroup").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createAutoScalingGroupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateAutoScalingGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a launch configuration.
     * </p>
     * <p>
     * If you exceed your maximum limit of launch configurations, the call fails. To query this limit, call the
     * <a>DescribeAccountLimits</a> API. For information about updating this limit, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-account-limits.html">Amazon EC2 Auto Scaling
     * service quotas</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/LaunchConfiguration.html">Launch configurations</a>
     * in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param createLaunchConfigurationRequest
     * @return Result of the CreateLaunchConfiguration operation returned by the service.
     * @throws AlreadyExistsException
     *         You already have an Auto Scaling group or launch configuration with this name.
     * @throws LimitExceededException
     *         You have already reached a limit for your Amazon EC2 Auto Scaling resources (for example, Auto Scaling
     *         groups, launch configurations, or lifecycle hooks). For more information, see <a
     *         href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html"
     *         >DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.CreateLaunchConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/CreateLaunchConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateLaunchConfigurationResponse createLaunchConfiguration(
            CreateLaunchConfigurationRequest createLaunchConfigurationRequest) throws AlreadyExistsException,
            LimitExceededException, ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<CreateLaunchConfigurationResponse> responseHandler = protocolFactory
                .createResponseHandler(CreateLaunchConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLaunchConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLaunchConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateLaunchConfigurationRequest, CreateLaunchConfigurationResponse>()
                            .withOperationName("CreateLaunchConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createLaunchConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateLaunchConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates tags for the specified Auto Scaling group.
     * </p>
     * <p>
     * When you specify a tag with a key that already exists, the operation overwrites the previous tag definition, and
     * you do not get an error message.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/autoscaling-tagging.html">Tagging Auto Scaling groups
     * and instances</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param createOrUpdateTagsRequest
     * @return Result of the CreateOrUpdateTags operation returned by the service.
     * @throws LimitExceededException
     *         You have already reached a limit for your Amazon EC2 Auto Scaling resources (for example, Auto Scaling
     *         groups, launch configurations, or lifecycle hooks). For more information, see <a
     *         href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html"
     *         >DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.
     * @throws AlreadyExistsException
     *         You already have an Auto Scaling group or launch configuration with this name.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws ResourceInUseException
     *         The operation can't be performed because the resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.CreateOrUpdateTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/CreateOrUpdateTags" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateOrUpdateTagsResponse createOrUpdateTags(CreateOrUpdateTagsRequest createOrUpdateTagsRequest)
            throws LimitExceededException, AlreadyExistsException, ResourceContentionException, ResourceInUseException,
            AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<CreateOrUpdateTagsResponse> responseHandler = protocolFactory
                .createResponseHandler(CreateOrUpdateTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createOrUpdateTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateOrUpdateTags");

            return clientHandler.execute(new ClientExecutionParams<CreateOrUpdateTagsRequest, CreateOrUpdateTagsResponse>()
                    .withOperationName("CreateOrUpdateTags").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createOrUpdateTagsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateOrUpdateTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified Auto Scaling group.
     * </p>
     * <p>
     * If the group has instances or scaling activities in progress, you must specify the option to force the deletion
     * in order for it to succeed.
     * </p>
     * <p>
     * If the group has policies, deleting the group deletes the policies, the underlying alarm actions, and any alarm
     * that no longer has an associated action.
     * </p>
     * <p>
     * To remove instances from the Auto Scaling group before deleting it, call the <a>DetachInstances</a> API with the
     * list of instances and the option to decrement the desired capacity. This ensures that Amazon EC2 Auto Scaling
     * does not launch replacement instances.
     * </p>
     * <p>
     * To terminate all instances before deleting the Auto Scaling group, call the <a>UpdateAutoScalingGroup</a> API and
     * set the minimum size and desired capacity of the Auto Scaling group to zero.
     * </p>
     *
     * @param deleteAutoScalingGroupRequest
     * @return Result of the DeleteAutoScalingGroup operation returned by the service.
     * @throws ScalingActivityInProgressException
     *         The operation can't be performed because there are scaling activities in progress.
     * @throws ResourceInUseException
     *         The operation can't be performed because the resource is in use.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DeleteAutoScalingGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DeleteAutoScalingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteAutoScalingGroupResponse deleteAutoScalingGroup(DeleteAutoScalingGroupRequest deleteAutoScalingGroupRequest)
            throws ScalingActivityInProgressException, ResourceInUseException, ResourceContentionException, AwsServiceException,
            SdkClientException, AutoScalingException {

        HttpResponseHandler<DeleteAutoScalingGroupResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteAutoScalingGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAutoScalingGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAutoScalingGroup");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteAutoScalingGroupRequest, DeleteAutoScalingGroupResponse>()
                            .withOperationName("DeleteAutoScalingGroup").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteAutoScalingGroupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteAutoScalingGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified launch configuration.
     * </p>
     * <p>
     * The launch configuration must not be attached to an Auto Scaling group. When this call completes, the launch
     * configuration is no longer available for use.
     * </p>
     *
     * @param deleteLaunchConfigurationRequest
     * @return Result of the DeleteLaunchConfiguration operation returned by the service.
     * @throws ResourceInUseException
     *         The operation can't be performed because the resource is in use.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DeleteLaunchConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DeleteLaunchConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteLaunchConfigurationResponse deleteLaunchConfiguration(
            DeleteLaunchConfigurationRequest deleteLaunchConfigurationRequest) throws ResourceInUseException,
            ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DeleteLaunchConfigurationResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteLaunchConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLaunchConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLaunchConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteLaunchConfigurationRequest, DeleteLaunchConfigurationResponse>()
                            .withOperationName("DeleteLaunchConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteLaunchConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteLaunchConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified lifecycle hook.
     * </p>
     * <p>
     * If there are any outstanding lifecycle actions, they are completed first (<code>ABANDON</code> for launching
     * instances, <code>CONTINUE</code> for terminating instances).
     * </p>
     *
     * @param deleteLifecycleHookRequest
     * @return Result of the DeleteLifecycleHook operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DeleteLifecycleHook
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DeleteLifecycleHook"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteLifecycleHookResponse deleteLifecycleHook(DeleteLifecycleHookRequest deleteLifecycleHookRequest)
            throws ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DeleteLifecycleHookResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteLifecycleHookResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLifecycleHookRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLifecycleHook");

            return clientHandler.execute(new ClientExecutionParams<DeleteLifecycleHookRequest, DeleteLifecycleHookResponse>()
                    .withOperationName("DeleteLifecycleHook").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteLifecycleHookRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteLifecycleHookRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified notification.
     * </p>
     *
     * @param deleteNotificationConfigurationRequest
     * @return Result of the DeleteNotificationConfiguration operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DeleteNotificationConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DeleteNotificationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteNotificationConfigurationResponse deleteNotificationConfiguration(
            DeleteNotificationConfigurationRequest deleteNotificationConfigurationRequest) throws ResourceContentionException,
            AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DeleteNotificationConfigurationResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteNotificationConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteNotificationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNotificationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteNotificationConfigurationRequest, DeleteNotificationConfigurationResponse>()
                            .withOperationName("DeleteNotificationConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteNotificationConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteNotificationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified scaling policy.
     * </p>
     * <p>
     * Deleting either a step scaling policy or a simple scaling policy deletes the underlying alarm action, but does
     * not delete the alarm, even if it no longer has an associated action.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/deleting-scaling-policy.html">Deleting a scaling
     * policy</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param deletePolicyRequest
     * @return Result of the DeletePolicy operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws ServiceLinkedRoleFailureException
     *         The service-linked role is not yet ready for use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DeletePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DeletePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeletePolicyResponse deletePolicy(DeletePolicyRequest deletePolicyRequest) throws ResourceContentionException,
            ServiceLinkedRoleFailureException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DeletePolicyResponse> responseHandler = protocolFactory
                .createResponseHandler(DeletePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePolicy");

            return clientHandler.execute(new ClientExecutionParams<DeletePolicyRequest, DeletePolicyResponse>()
                    .withOperationName("DeletePolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deletePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeletePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified scheduled action.
     * </p>
     *
     * @param deleteScheduledActionRequest
     * @return Result of the DeleteScheduledAction operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DeleteScheduledAction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DeleteScheduledAction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteScheduledActionResponse deleteScheduledAction(DeleteScheduledActionRequest deleteScheduledActionRequest)
            throws ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DeleteScheduledActionResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteScheduledActionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteScheduledActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteScheduledAction");

            return clientHandler.execute(new ClientExecutionParams<DeleteScheduledActionRequest, DeleteScheduledActionResponse>()
                    .withOperationName("DeleteScheduledAction").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteScheduledActionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteScheduledActionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified tags.
     * </p>
     *
     * @param deleteTagsRequest
     * @return Result of the DeleteTags operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws ResourceInUseException
     *         The operation can't be performed because the resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DeleteTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DeleteTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteTagsResponse deleteTags(DeleteTagsRequest deleteTagsRequest) throws ResourceContentionException,
            ResourceInUseException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DeleteTagsResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTags");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteTagsRequest, DeleteTagsResponse>().withOperationName("DeleteTags")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteTagsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the current Amazon EC2 Auto Scaling resource quotas for your AWS account.
     * </p>
     * <p>
     * For information about requesting an increase, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-account-limits.html">Amazon EC2 Auto Scaling
     * service quotas</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param describeAccountLimitsRequest
     * @return Result of the DescribeAccountLimits operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeAccountLimits
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeAccountLimits"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAccountLimitsResponse describeAccountLimits(DescribeAccountLimitsRequest describeAccountLimitsRequest)
            throws ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DescribeAccountLimitsResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeAccountLimitsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAccountLimitsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAccountLimits");

            return clientHandler.execute(new ClientExecutionParams<DescribeAccountLimitsRequest, DescribeAccountLimitsResponse>()
                    .withOperationName("DescribeAccountLimits").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeAccountLimitsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeAccountLimitsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the available adjustment types for Amazon EC2 Auto Scaling scaling policies. These settings apply to
     * step scaling policies and simple scaling policies; they do not apply to target tracking scaling policies.
     * </p>
     * <p>
     * The following adjustment types are supported:
     * </p>
     * <ul>
     * <li>
     * <p>
     * ChangeInCapacity
     * </p>
     * </li>
     * <li>
     * <p>
     * ExactCapacity
     * </p>
     * </li>
     * <li>
     * <p>
     * PercentChangeInCapacity
     * </p>
     * </li>
     * </ul>
     *
     * @param describeAdjustmentTypesRequest
     * @return Result of the DescribeAdjustmentTypes operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeAdjustmentTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeAdjustmentTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAdjustmentTypesResponse describeAdjustmentTypes(DescribeAdjustmentTypesRequest describeAdjustmentTypesRequest)
            throws ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DescribeAdjustmentTypesResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeAdjustmentTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAdjustmentTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAdjustmentTypes");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAdjustmentTypesRequest, DescribeAdjustmentTypesResponse>()
                            .withOperationName("DescribeAdjustmentTypes").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeAdjustmentTypesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAdjustmentTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more Auto Scaling groups.
     * </p>
     *
     * @param describeAutoScalingGroupsRequest
     * @return Result of the DescribeAutoScalingGroups operation returned by the service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> value is not valid.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeAutoScalingGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeAutoScalingGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAutoScalingGroupsResponse describeAutoScalingGroups(
            DescribeAutoScalingGroupsRequest describeAutoScalingGroupsRequest) throws InvalidNextTokenException,
            ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DescribeAutoScalingGroupsResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeAutoScalingGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAutoScalingGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAutoScalingGroups");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAutoScalingGroupsRequest, DescribeAutoScalingGroupsResponse>()
                            .withOperationName("DescribeAutoScalingGroups").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeAutoScalingGroupsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAutoScalingGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more Auto Scaling groups.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAutoScalingGroups(software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingGroupsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.autoscaling.paginators.DescribeAutoScalingGroupsIterable responses = client.describeAutoScalingGroupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.autoscaling.paginators.DescribeAutoScalingGroupsIterable responses = client
     *             .describeAutoScalingGroupsPaginator(request);
     *     for (software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingGroupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.autoscaling.paginators.DescribeAutoScalingGroupsIterable responses = client.describeAutoScalingGroupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxRecords won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAutoScalingGroups(software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingGroupsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAutoScalingGroupsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> value is not valid.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeAutoScalingGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeAutoScalingGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAutoScalingGroupsIterable describeAutoScalingGroupsPaginator(
            DescribeAutoScalingGroupsRequest describeAutoScalingGroupsRequest) throws InvalidNextTokenException,
            ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {
        return new DescribeAutoScalingGroupsIterable(this, applyPaginatorUserAgent(describeAutoScalingGroupsRequest));
    }

    /**
     * <p>
     * Describes one or more Auto Scaling instances.
     * </p>
     *
     * @param describeAutoScalingInstancesRequest
     * @return Result of the DescribeAutoScalingInstances operation returned by the service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> value is not valid.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeAutoScalingInstances
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeAutoScalingInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAutoScalingInstancesResponse describeAutoScalingInstances(
            DescribeAutoScalingInstancesRequest describeAutoScalingInstancesRequest) throws InvalidNextTokenException,
            ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DescribeAutoScalingInstancesResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeAutoScalingInstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAutoScalingInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAutoScalingInstances");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAutoScalingInstancesRequest, DescribeAutoScalingInstancesResponse>()
                            .withOperationName("DescribeAutoScalingInstances").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeAutoScalingInstancesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAutoScalingInstancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more Auto Scaling instances.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAutoScalingInstances(software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingInstancesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.autoscaling.paginators.DescribeAutoScalingInstancesIterable responses = client.describeAutoScalingInstancesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.autoscaling.paginators.DescribeAutoScalingInstancesIterable responses = client
     *             .describeAutoScalingInstancesPaginator(request);
     *     for (software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingInstancesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.autoscaling.paginators.DescribeAutoScalingInstancesIterable responses = client.describeAutoScalingInstancesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxRecords won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAutoScalingInstances(software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingInstancesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAutoScalingInstancesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> value is not valid.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeAutoScalingInstances
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeAutoScalingInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAutoScalingInstancesIterable describeAutoScalingInstancesPaginator(
            DescribeAutoScalingInstancesRequest describeAutoScalingInstancesRequest) throws InvalidNextTokenException,
            ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {
        return new DescribeAutoScalingInstancesIterable(this, applyPaginatorUserAgent(describeAutoScalingInstancesRequest));
    }

    /**
     * <p>
     * Describes the notification types that are supported by Amazon EC2 Auto Scaling.
     * </p>
     *
     * @param describeAutoScalingNotificationTypesRequest
     * @return Result of the DescribeAutoScalingNotificationTypes operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeAutoScalingNotificationTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeAutoScalingNotificationTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAutoScalingNotificationTypesResponse describeAutoScalingNotificationTypes(
            DescribeAutoScalingNotificationTypesRequest describeAutoScalingNotificationTypesRequest)
            throws ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DescribeAutoScalingNotificationTypesResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeAutoScalingNotificationTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeAutoScalingNotificationTypesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAutoScalingNotificationTypes");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAutoScalingNotificationTypesRequest, DescribeAutoScalingNotificationTypesResponse>()
                            .withOperationName("DescribeAutoScalingNotificationTypes").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeAutoScalingNotificationTypesRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAutoScalingNotificationTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more instance refreshes.
     * </p>
     * <p>
     * You can determine the status of a request by looking at the <code>Status</code> parameter. The following are the
     * possible statuses:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Pending</code> - The request was created, but the operation has not started.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>InProgress</code> - The operation is in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Successful</code> - The operation completed successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Failed</code> - The operation failed to complete. You can troubleshoot using the status reason and the
     * scaling activities.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Cancelling</code> - An ongoing operation is being cancelled. Cancellation does not roll back any
     * replacements that have already been completed, but it prevents new replacements from being started.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Cancelled</code> - The operation is cancelled.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-instance-refresh.html">Replacing Auto Scaling
     * Instances Based on an Instance Refresh</a>.
     * </p>
     *
     * @param describeInstanceRefreshesRequest
     * @return Result of the DescribeInstanceRefreshes operation returned by the service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> value is not valid.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeInstanceRefreshes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeInstanceRefreshes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeInstanceRefreshesResponse describeInstanceRefreshes(
            DescribeInstanceRefreshesRequest describeInstanceRefreshesRequest) throws InvalidNextTokenException,
            ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DescribeInstanceRefreshesResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeInstanceRefreshesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeInstanceRefreshesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeInstanceRefreshes");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeInstanceRefreshesRequest, DescribeInstanceRefreshesResponse>()
                            .withOperationName("DescribeInstanceRefreshes").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeInstanceRefreshesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeInstanceRefreshesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more launch configurations.
     * </p>
     *
     * @param describeLaunchConfigurationsRequest
     * @return Result of the DescribeLaunchConfigurations operation returned by the service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> value is not valid.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeLaunchConfigurations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeLaunchConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeLaunchConfigurationsResponse describeLaunchConfigurations(
            DescribeLaunchConfigurationsRequest describeLaunchConfigurationsRequest) throws InvalidNextTokenException,
            ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DescribeLaunchConfigurationsResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeLaunchConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLaunchConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLaunchConfigurations");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeLaunchConfigurationsRequest, DescribeLaunchConfigurationsResponse>()
                            .withOperationName("DescribeLaunchConfigurations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeLaunchConfigurationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeLaunchConfigurationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more launch configurations.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeLaunchConfigurations(software.amazon.awssdk.services.autoscaling.model.DescribeLaunchConfigurationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.autoscaling.paginators.DescribeLaunchConfigurationsIterable responses = client.describeLaunchConfigurationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.autoscaling.paginators.DescribeLaunchConfigurationsIterable responses = client
     *             .describeLaunchConfigurationsPaginator(request);
     *     for (software.amazon.awssdk.services.autoscaling.model.DescribeLaunchConfigurationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.autoscaling.paginators.DescribeLaunchConfigurationsIterable responses = client.describeLaunchConfigurationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxRecords won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeLaunchConfigurations(software.amazon.awssdk.services.autoscaling.model.DescribeLaunchConfigurationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeLaunchConfigurationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> value is not valid.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeLaunchConfigurations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeLaunchConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeLaunchConfigurationsIterable describeLaunchConfigurationsPaginator(
            DescribeLaunchConfigurationsRequest describeLaunchConfigurationsRequest) throws InvalidNextTokenException,
            ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {
        return new DescribeLaunchConfigurationsIterable(this, applyPaginatorUserAgent(describeLaunchConfigurationsRequest));
    }

    /**
     * <p>
     * Describes the available types of lifecycle hooks.
     * </p>
     * <p>
     * The following hook types are supported:
     * </p>
     * <ul>
     * <li>
     * <p>
     * autoscaling:EC2_INSTANCE_LAUNCHING
     * </p>
     * </li>
     * <li>
     * <p>
     * autoscaling:EC2_INSTANCE_TERMINATING
     * </p>
     * </li>
     * </ul>
     *
     * @param describeLifecycleHookTypesRequest
     * @return Result of the DescribeLifecycleHookTypes operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeLifecycleHookTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeLifecycleHookTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeLifecycleHookTypesResponse describeLifecycleHookTypes(
            DescribeLifecycleHookTypesRequest describeLifecycleHookTypesRequest) throws ResourceContentionException,
            AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DescribeLifecycleHookTypesResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeLifecycleHookTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLifecycleHookTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLifecycleHookTypes");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeLifecycleHookTypesRequest, DescribeLifecycleHookTypesResponse>()
                            .withOperationName("DescribeLifecycleHookTypes").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeLifecycleHookTypesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeLifecycleHookTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the lifecycle hooks for the specified Auto Scaling group.
     * </p>
     *
     * @param describeLifecycleHooksRequest
     * @return Result of the DescribeLifecycleHooks operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeLifecycleHooks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeLifecycleHooks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeLifecycleHooksResponse describeLifecycleHooks(DescribeLifecycleHooksRequest describeLifecycleHooksRequest)
            throws ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DescribeLifecycleHooksResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeLifecycleHooksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLifecycleHooksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLifecycleHooks");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeLifecycleHooksRequest, DescribeLifecycleHooksResponse>()
                            .withOperationName("DescribeLifecycleHooks").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeLifecycleHooksRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeLifecycleHooksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the target groups for the specified Auto Scaling group.
     * </p>
     *
     * @param describeLoadBalancerTargetGroupsRequest
     * @return Result of the DescribeLoadBalancerTargetGroups operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeLoadBalancerTargetGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeLoadBalancerTargetGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeLoadBalancerTargetGroupsResponse describeLoadBalancerTargetGroups(
            DescribeLoadBalancerTargetGroupsRequest describeLoadBalancerTargetGroupsRequest) throws ResourceContentionException,
            AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DescribeLoadBalancerTargetGroupsResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeLoadBalancerTargetGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeLoadBalancerTargetGroupsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLoadBalancerTargetGroups");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeLoadBalancerTargetGroupsRequest, DescribeLoadBalancerTargetGroupsResponse>()
                            .withOperationName("DescribeLoadBalancerTargetGroups").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeLoadBalancerTargetGroupsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeLoadBalancerTargetGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the load balancers for the specified Auto Scaling group.
     * </p>
     * <p>
     * This operation describes only Classic Load Balancers. If you have Application Load Balancers, Network Load
     * Balancers, or Gateway Load Balancers, use the <a>DescribeLoadBalancerTargetGroups</a> API instead.
     * </p>
     *
     * @param describeLoadBalancersRequest
     * @return Result of the DescribeLoadBalancers operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeLoadBalancers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeLoadBalancers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeLoadBalancersResponse describeLoadBalancers(DescribeLoadBalancersRequest describeLoadBalancersRequest)
            throws ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DescribeLoadBalancersResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeLoadBalancersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLoadBalancersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLoadBalancers");

            return clientHandler.execute(new ClientExecutionParams<DescribeLoadBalancersRequest, DescribeLoadBalancersResponse>()
                    .withOperationName("DescribeLoadBalancers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeLoadBalancersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeLoadBalancersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the available CloudWatch metrics for Amazon EC2 Auto Scaling.
     * </p>
     * <p>
     * The <code>GroupStandbyInstances</code> metric is not returned by default. You must explicitly request this metric
     * when calling the <a>EnableMetricsCollection</a> API.
     * </p>
     *
     * @param describeMetricCollectionTypesRequest
     * @return Result of the DescribeMetricCollectionTypes operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeMetricCollectionTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeMetricCollectionTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeMetricCollectionTypesResponse describeMetricCollectionTypes(
            DescribeMetricCollectionTypesRequest describeMetricCollectionTypesRequest) throws ResourceContentionException,
            AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DescribeMetricCollectionTypesResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeMetricCollectionTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeMetricCollectionTypesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMetricCollectionTypes");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeMetricCollectionTypesRequest, DescribeMetricCollectionTypesResponse>()
                            .withOperationName("DescribeMetricCollectionTypes").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeMetricCollectionTypesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeMetricCollectionTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the notification actions associated with the specified Auto Scaling group.
     * </p>
     *
     * @param describeNotificationConfigurationsRequest
     * @return Result of the DescribeNotificationConfigurations operation returned by the service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> value is not valid.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeNotificationConfigurations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeNotificationConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeNotificationConfigurationsResponse describeNotificationConfigurations(
            DescribeNotificationConfigurationsRequest describeNotificationConfigurationsRequest)
            throws InvalidNextTokenException, ResourceContentionException, AwsServiceException, SdkClientException,
            AutoScalingException {

        HttpResponseHandler<DescribeNotificationConfigurationsResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeNotificationConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeNotificationConfigurationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeNotificationConfigurations");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeNotificationConfigurationsRequest, DescribeNotificationConfigurationsResponse>()
                            .withOperationName("DescribeNotificationConfigurations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeNotificationConfigurationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeNotificationConfigurationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the notification actions associated with the specified Auto Scaling group.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeNotificationConfigurations(software.amazon.awssdk.services.autoscaling.model.DescribeNotificationConfigurationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.autoscaling.paginators.DescribeNotificationConfigurationsIterable responses = client.describeNotificationConfigurationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.autoscaling.paginators.DescribeNotificationConfigurationsIterable responses = client
     *             .describeNotificationConfigurationsPaginator(request);
     *     for (software.amazon.awssdk.services.autoscaling.model.DescribeNotificationConfigurationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.autoscaling.paginators.DescribeNotificationConfigurationsIterable responses = client.describeNotificationConfigurationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxRecords won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeNotificationConfigurations(software.amazon.awssdk.services.autoscaling.model.DescribeNotificationConfigurationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeNotificationConfigurationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> value is not valid.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeNotificationConfigurations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeNotificationConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeNotificationConfigurationsIterable describeNotificationConfigurationsPaginator(
            DescribeNotificationConfigurationsRequest describeNotificationConfigurationsRequest)
            throws InvalidNextTokenException, ResourceContentionException, AwsServiceException, SdkClientException,
            AutoScalingException {
        return new DescribeNotificationConfigurationsIterable(this,
                applyPaginatorUserAgent(describeNotificationConfigurationsRequest));
    }

    /**
     * <p>
     * Describes the policies for the specified Auto Scaling group.
     * </p>
     *
     * @param describePoliciesRequest
     * @return Result of the DescribePolicies operation returned by the service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> value is not valid.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws ServiceLinkedRoleFailureException
     *         The service-linked role is not yet ready for use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribePolicies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribePolicies" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribePoliciesResponse describePolicies(DescribePoliciesRequest describePoliciesRequest)
            throws InvalidNextTokenException, ResourceContentionException, ServiceLinkedRoleFailureException,
            AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DescribePoliciesResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribePoliciesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePolicies");

            return clientHandler.execute(new ClientExecutionParams<DescribePoliciesRequest, DescribePoliciesResponse>()
                    .withOperationName("DescribePolicies").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describePoliciesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribePoliciesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the policies for the specified Auto Scaling group.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describePolicies(software.amazon.awssdk.services.autoscaling.model.DescribePoliciesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.autoscaling.paginators.DescribePoliciesIterable responses = client.describePoliciesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.autoscaling.paginators.DescribePoliciesIterable responses = client
     *             .describePoliciesPaginator(request);
     *     for (software.amazon.awssdk.services.autoscaling.model.DescribePoliciesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.autoscaling.paginators.DescribePoliciesIterable responses = client.describePoliciesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxRecords won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describePolicies(software.amazon.awssdk.services.autoscaling.model.DescribePoliciesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describePoliciesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> value is not valid.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws ServiceLinkedRoleFailureException
     *         The service-linked role is not yet ready for use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribePolicies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribePolicies" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribePoliciesIterable describePoliciesPaginator(DescribePoliciesRequest describePoliciesRequest)
            throws InvalidNextTokenException, ResourceContentionException, ServiceLinkedRoleFailureException,
            AwsServiceException, SdkClientException, AutoScalingException {
        return new DescribePoliciesIterable(this, applyPaginatorUserAgent(describePoliciesRequest));
    }

    /**
     * <p>
     * Describes one or more scaling activities for the specified Auto Scaling group.
     * </p>
     *
     * @param describeScalingActivitiesRequest
     * @return Result of the DescribeScalingActivities operation returned by the service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> value is not valid.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeScalingActivities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeScalingActivities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeScalingActivitiesResponse describeScalingActivities(
            DescribeScalingActivitiesRequest describeScalingActivitiesRequest) throws InvalidNextTokenException,
            ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DescribeScalingActivitiesResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeScalingActivitiesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeScalingActivitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeScalingActivities");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeScalingActivitiesRequest, DescribeScalingActivitiesResponse>()
                            .withOperationName("DescribeScalingActivities").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeScalingActivitiesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeScalingActivitiesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more scaling activities for the specified Auto Scaling group.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeScalingActivities(software.amazon.awssdk.services.autoscaling.model.DescribeScalingActivitiesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.autoscaling.paginators.DescribeScalingActivitiesIterable responses = client.describeScalingActivitiesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.autoscaling.paginators.DescribeScalingActivitiesIterable responses = client
     *             .describeScalingActivitiesPaginator(request);
     *     for (software.amazon.awssdk.services.autoscaling.model.DescribeScalingActivitiesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.autoscaling.paginators.DescribeScalingActivitiesIterable responses = client.describeScalingActivitiesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxRecords won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeScalingActivities(software.amazon.awssdk.services.autoscaling.model.DescribeScalingActivitiesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeScalingActivitiesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> value is not valid.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeScalingActivities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeScalingActivities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeScalingActivitiesIterable describeScalingActivitiesPaginator(
            DescribeScalingActivitiesRequest describeScalingActivitiesRequest) throws InvalidNextTokenException,
            ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {
        return new DescribeScalingActivitiesIterable(this, applyPaginatorUserAgent(describeScalingActivitiesRequest));
    }

    /**
     * <p>
     * Describes the scaling process types for use with the <a>ResumeProcesses</a> and <a>SuspendProcesses</a> APIs.
     * </p>
     *
     * @param describeScalingProcessTypesRequest
     * @return Result of the DescribeScalingProcessTypes operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeScalingProcessTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeScalingProcessTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeScalingProcessTypesResponse describeScalingProcessTypes(
            DescribeScalingProcessTypesRequest describeScalingProcessTypesRequest) throws ResourceContentionException,
            AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DescribeScalingProcessTypesResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeScalingProcessTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeScalingProcessTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeScalingProcessTypes");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeScalingProcessTypesRequest, DescribeScalingProcessTypesResponse>()
                            .withOperationName("DescribeScalingProcessTypes").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeScalingProcessTypesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeScalingProcessTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the actions scheduled for your Auto Scaling group that haven't run or that have not reached their end
     * time. To describe the actions that have already run, call the <a>DescribeScalingActivities</a> API.
     * </p>
     *
     * @param describeScheduledActionsRequest
     * @return Result of the DescribeScheduledActions operation returned by the service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> value is not valid.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeScheduledActions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeScheduledActions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeScheduledActionsResponse describeScheduledActions(
            DescribeScheduledActionsRequest describeScheduledActionsRequest) throws InvalidNextTokenException,
            ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DescribeScheduledActionsResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeScheduledActionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeScheduledActionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeScheduledActions");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeScheduledActionsRequest, DescribeScheduledActionsResponse>()
                            .withOperationName("DescribeScheduledActions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeScheduledActionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeScheduledActionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the actions scheduled for your Auto Scaling group that haven't run or that have not reached their end
     * time. To describe the actions that have already run, call the <a>DescribeScalingActivities</a> API.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeScheduledActions(software.amazon.awssdk.services.autoscaling.model.DescribeScheduledActionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.autoscaling.paginators.DescribeScheduledActionsIterable responses = client.describeScheduledActionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.autoscaling.paginators.DescribeScheduledActionsIterable responses = client
     *             .describeScheduledActionsPaginator(request);
     *     for (software.amazon.awssdk.services.autoscaling.model.DescribeScheduledActionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.autoscaling.paginators.DescribeScheduledActionsIterable responses = client.describeScheduledActionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxRecords won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeScheduledActions(software.amazon.awssdk.services.autoscaling.model.DescribeScheduledActionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeScheduledActionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> value is not valid.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeScheduledActions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeScheduledActions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeScheduledActionsIterable describeScheduledActionsPaginator(
            DescribeScheduledActionsRequest describeScheduledActionsRequest) throws InvalidNextTokenException,
            ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {
        return new DescribeScheduledActionsIterable(this, applyPaginatorUserAgent(describeScheduledActionsRequest));
    }

    /**
     * <p>
     * Describes the specified tags.
     * </p>
     * <p>
     * You can use filters to limit the results. For example, you can query for the tags for a specific Auto Scaling
     * group. You can specify multiple values for a filter. A tag must match at least one of the specified values for it
     * to be included in the results.
     * </p>
     * <p>
     * You can also specify multiple filters. The result includes information for a particular tag only if it matches
     * all the filters. If there's no match, no special message is returned.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/autoscaling-tagging.html">Tagging Auto Scaling groups
     * and instances</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param describeTagsRequest
     * @return Result of the DescribeTags operation returned by the service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> value is not valid.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeTagsResponse describeTags(DescribeTagsRequest describeTagsRequest) throws InvalidNextTokenException,
            ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DescribeTagsResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTags");

            return clientHandler.execute(new ClientExecutionParams<DescribeTagsRequest, DescribeTagsResponse>()
                    .withOperationName("DescribeTags").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeTagsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the specified tags.
     * </p>
     * <p>
     * You can use filters to limit the results. For example, you can query for the tags for a specific Auto Scaling
     * group. You can specify multiple values for a filter. A tag must match at least one of the specified values for it
     * to be included in the results.
     * </p>
     * <p>
     * You can also specify multiple filters. The result includes information for a particular tag only if it matches
     * all the filters. If there's no match, no special message is returned.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/autoscaling-tagging.html">Tagging Auto Scaling groups
     * and instances</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #describeTags(software.amazon.awssdk.services.autoscaling.model.DescribeTagsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.autoscaling.paginators.DescribeTagsIterable responses = client.describeTagsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.autoscaling.paginators.DescribeTagsIterable responses = client.describeTagsPaginator(request);
     *     for (software.amazon.awssdk.services.autoscaling.model.DescribeTagsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.autoscaling.paginators.DescribeTagsIterable responses = client.describeTagsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxRecords won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeTags(software.amazon.awssdk.services.autoscaling.model.DescribeTagsRequest)} operation.</b>
     * </p>
     *
     * @param describeTagsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> value is not valid.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeTagsIterable describeTagsPaginator(DescribeTagsRequest describeTagsRequest) throws InvalidNextTokenException,
            ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {
        return new DescribeTagsIterable(this, applyPaginatorUserAgent(describeTagsRequest));
    }

    /**
     * <p>
     * Describes the termination policies supported by Amazon EC2 Auto Scaling.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html">Controlling which Auto
     * Scaling instances terminate during scale in</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param describeTerminationPolicyTypesRequest
     * @return Result of the DescribeTerminationPolicyTypes operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DescribeTerminationPolicyTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeTerminationPolicyTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeTerminationPolicyTypesResponse describeTerminationPolicyTypes(
            DescribeTerminationPolicyTypesRequest describeTerminationPolicyTypesRequest) throws ResourceContentionException,
            AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DescribeTerminationPolicyTypesResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeTerminationPolicyTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeTerminationPolicyTypesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTerminationPolicyTypes");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeTerminationPolicyTypesRequest, DescribeTerminationPolicyTypesResponse>()
                            .withOperationName("DescribeTerminationPolicyTypes").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeTerminationPolicyTypesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeTerminationPolicyTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more instances from the specified Auto Scaling group.
     * </p>
     * <p>
     * After the instances are detached, you can manage them independent of the Auto Scaling group.
     * </p>
     * <p>
     * If you do not specify the option to decrement the desired capacity, Amazon EC2 Auto Scaling launches instances to
     * replace the ones that are detached.
     * </p>
     * <p>
     * If there is a Classic Load Balancer attached to the Auto Scaling group, the instances are deregistered from the
     * load balancer. If there are target groups attached to the Auto Scaling group, the instances are deregistered from
     * the target groups.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/detach-instance-asg.html">Detach EC2 instances from
     * your Auto Scaling group</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param detachInstancesRequest
     * @return Result of the DetachInstances operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DetachInstances
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DetachInstances" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DetachInstancesResponse detachInstances(DetachInstancesRequest detachInstancesRequest)
            throws ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DetachInstancesResponse> responseHandler = protocolFactory
                .createResponseHandler(DetachInstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, detachInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DetachInstances");

            return clientHandler.execute(new ClientExecutionParams<DetachInstancesRequest, DetachInstancesResponse>()
                    .withOperationName("DetachInstances").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(detachInstancesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DetachInstancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Detaches one or more target groups from the specified Auto Scaling group.
     * </p>
     *
     * @param detachLoadBalancerTargetGroupsRequest
     * @return Result of the DetachLoadBalancerTargetGroups operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DetachLoadBalancerTargetGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DetachLoadBalancerTargetGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DetachLoadBalancerTargetGroupsResponse detachLoadBalancerTargetGroups(
            DetachLoadBalancerTargetGroupsRequest detachLoadBalancerTargetGroupsRequest) throws ResourceContentionException,
            AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DetachLoadBalancerTargetGroupsResponse> responseHandler = protocolFactory
                .createResponseHandler(DetachLoadBalancerTargetGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                detachLoadBalancerTargetGroupsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DetachLoadBalancerTargetGroups");

            return clientHandler
                    .execute(new ClientExecutionParams<DetachLoadBalancerTargetGroupsRequest, DetachLoadBalancerTargetGroupsResponse>()
                            .withOperationName("DetachLoadBalancerTargetGroups").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(detachLoadBalancerTargetGroupsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DetachLoadBalancerTargetGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Detaches one or more Classic Load Balancers from the specified Auto Scaling group.
     * </p>
     * <p>
     * This operation detaches only Classic Load Balancers. If you have Application Load Balancers, Network Load
     * Balancers, or Gateway Load Balancers, use the <a>DetachLoadBalancerTargetGroups</a> API instead.
     * </p>
     * <p>
     * When you detach a load balancer, it enters the <code>Removing</code> state while deregistering the instances in
     * the group. When all instances are deregistered, then you can no longer describe the load balancer using the
     * <a>DescribeLoadBalancers</a> API call. The instances remain running.
     * </p>
     *
     * @param detachLoadBalancersRequest
     * @return Result of the DetachLoadBalancers operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DetachLoadBalancers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DetachLoadBalancers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DetachLoadBalancersResponse detachLoadBalancers(DetachLoadBalancersRequest detachLoadBalancersRequest)
            throws ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DetachLoadBalancersResponse> responseHandler = protocolFactory
                .createResponseHandler(DetachLoadBalancersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, detachLoadBalancersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DetachLoadBalancers");

            return clientHandler.execute(new ClientExecutionParams<DetachLoadBalancersRequest, DetachLoadBalancersResponse>()
                    .withOperationName("DetachLoadBalancers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(detachLoadBalancersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DetachLoadBalancersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disables group metrics for the specified Auto Scaling group.
     * </p>
     *
     * @param disableMetricsCollectionRequest
     * @return Result of the DisableMetricsCollection operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.DisableMetricsCollection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DisableMetricsCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisableMetricsCollectionResponse disableMetricsCollection(
            DisableMetricsCollectionRequest disableMetricsCollectionRequest) throws ResourceContentionException,
            AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<DisableMetricsCollectionResponse> responseHandler = protocolFactory
                .createResponseHandler(DisableMetricsCollectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableMetricsCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableMetricsCollection");

            return clientHandler
                    .execute(new ClientExecutionParams<DisableMetricsCollectionRequest, DisableMetricsCollectionResponse>()
                            .withOperationName("DisableMetricsCollection").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disableMetricsCollectionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisableMetricsCollectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables group metrics for the specified Auto Scaling group. For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-monitoring.html">Monitoring CloudWatch
     * metrics for your Auto Scaling groups and instances</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param enableMetricsCollectionRequest
     * @return Result of the EnableMetricsCollection operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.EnableMetricsCollection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/EnableMetricsCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public EnableMetricsCollectionResponse enableMetricsCollection(EnableMetricsCollectionRequest enableMetricsCollectionRequest)
            throws ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<EnableMetricsCollectionResponse> responseHandler = protocolFactory
                .createResponseHandler(EnableMetricsCollectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableMetricsCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableMetricsCollection");

            return clientHandler
                    .execute(new ClientExecutionParams<EnableMetricsCollectionRequest, EnableMetricsCollectionResponse>()
                            .withOperationName("EnableMetricsCollection").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(enableMetricsCollectionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new EnableMetricsCollectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Moves the specified instances into the standby state.
     * </p>
     * <p>
     * If you choose to decrement the desired capacity of the Auto Scaling group, the instances can enter standby as
     * long as the desired capacity of the Auto Scaling group after the instances are placed into standby is equal to or
     * greater than the minimum capacity of the group.
     * </p>
     * <p>
     * If you choose not to decrement the desired capacity of the Auto Scaling group, the Auto Scaling group launches
     * new instances to replace the instances on standby.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-enter-exit-standby.html">Temporarily removing
     * instances from your Auto Scaling group</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param enterStandbyRequest
     * @return Result of the EnterStandby operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.EnterStandby
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/EnterStandby" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public EnterStandbyResponse enterStandby(EnterStandbyRequest enterStandbyRequest) throws ResourceContentionException,
            AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<EnterStandbyResponse> responseHandler = protocolFactory
                .createResponseHandler(EnterStandbyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enterStandbyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnterStandby");

            return clientHandler.execute(new ClientExecutionParams<EnterStandbyRequest, EnterStandbyResponse>()
                    .withOperationName("EnterStandby").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(enterStandbyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new EnterStandbyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Executes the specified policy. This can be useful for testing the design of your scaling policy.
     * </p>
     *
     * @param executePolicyRequest
     * @return Result of the ExecutePolicy operation returned by the service.
     * @throws ScalingActivityInProgressException
     *         The operation can't be performed because there are scaling activities in progress.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.ExecutePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/ExecutePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ExecutePolicyResponse executePolicy(ExecutePolicyRequest executePolicyRequest)
            throws ScalingActivityInProgressException, ResourceContentionException, AwsServiceException, SdkClientException,
            AutoScalingException {

        HttpResponseHandler<ExecutePolicyResponse> responseHandler = protocolFactory
                .createResponseHandler(ExecutePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, executePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExecutePolicy");

            return clientHandler.execute(new ClientExecutionParams<ExecutePolicyRequest, ExecutePolicyResponse>()
                    .withOperationName("ExecutePolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(executePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ExecutePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Moves the specified instances out of the standby state.
     * </p>
     * <p>
     * After you put the instances back in service, the desired capacity is incremented.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-enter-exit-standby.html">Temporarily removing
     * instances from your Auto Scaling group</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param exitStandbyRequest
     * @return Result of the ExitStandby operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.ExitStandby
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/ExitStandby" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ExitStandbyResponse exitStandby(ExitStandbyRequest exitStandbyRequest) throws ResourceContentionException,
            AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<ExitStandbyResponse> responseHandler = protocolFactory
                .createResponseHandler(ExitStandbyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exitStandbyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExitStandby");

            return clientHandler.execute(new ClientExecutionParams<ExitStandbyRequest, ExitStandbyResponse>()
                    .withOperationName("ExitStandby").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(exitStandbyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ExitStandbyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates a lifecycle hook for the specified Auto Scaling group.
     * </p>
     * <p>
     * A lifecycle hook tells Amazon EC2 Auto Scaling to perform an action on an instance when the instance launches
     * (before it is put into service) or as the instance terminates (before it is fully terminated).
     * </p>
     * <p>
     * This step is a part of the procedure for adding a lifecycle hook to an Auto Scaling group:
     * </p>
     * <ol>
     * <li>
     * <p>
     * (Optional) Create a Lambda function and a rule that allows CloudWatch Events to invoke your Lambda function when
     * Amazon EC2 Auto Scaling launches or terminates instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * (Optional) Create a notification target and an IAM role. The target can be either an Amazon SQS queue or an
     * Amazon SNS topic. The role allows Amazon EC2 Auto Scaling to publish lifecycle notifications to the target.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Create the lifecycle hook. Specify whether the hook is used when the instances launch or terminate.</b>
     * </p>
     * </li>
     * <li>
     * <p>
     * If you need more time, record the lifecycle action heartbeat to keep the instance in a pending state using the
     * <a>RecordLifecycleActionHeartbeat</a> API call.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you finish before the timeout period ends, complete the lifecycle action using the
     * <a>CompleteLifecycleAction</a> API call.
     * </p>
     * </li>
     * </ol>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/lifecycle-hooks.html">Amazon EC2 Auto Scaling
     * lifecycle hooks</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * If you exceed your maximum limit of lifecycle hooks, which by default is 50 per Auto Scaling group, the call
     * fails.
     * </p>
     * <p>
     * You can view the lifecycle hooks for an Auto Scaling group using the <a>DescribeLifecycleHooks</a> API call. If
     * you are no longer using a lifecycle hook, you can delete it by calling the <a>DeleteLifecycleHook</a> API.
     * </p>
     *
     * @param putLifecycleHookRequest
     * @return Result of the PutLifecycleHook operation returned by the service.
     * @throws LimitExceededException
     *         You have already reached a limit for your Amazon EC2 Auto Scaling resources (for example, Auto Scaling
     *         groups, launch configurations, or lifecycle hooks). For more information, see <a
     *         href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html"
     *         >DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.PutLifecycleHook
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/PutLifecycleHook" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutLifecycleHookResponse putLifecycleHook(PutLifecycleHookRequest putLifecycleHookRequest)
            throws LimitExceededException, ResourceContentionException, AwsServiceException, SdkClientException,
            AutoScalingException {

        HttpResponseHandler<PutLifecycleHookResponse> responseHandler = protocolFactory
                .createResponseHandler(PutLifecycleHookResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putLifecycleHookRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutLifecycleHook");

            return clientHandler.execute(new ClientExecutionParams<PutLifecycleHookRequest, PutLifecycleHookResponse>()
                    .withOperationName("PutLifecycleHook").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putLifecycleHookRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutLifecycleHookRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Configures an Auto Scaling group to send notifications when specified events take place. Subscribers to the
     * specified topic can have messages delivered to an endpoint such as a web server or an email address.
     * </p>
     * <p>
     * This configuration overwrites any existing configuration.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/ASGettingNotifications.html">Getting Amazon SNS
     * notifications when your Auto Scaling group scales</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * If you exceed your maximum limit of SNS topics, which is 10 per Auto Scaling group, the call fails.
     * </p>
     *
     * @param putNotificationConfigurationRequest
     * @return Result of the PutNotificationConfiguration operation returned by the service.
     * @throws LimitExceededException
     *         You have already reached a limit for your Amazon EC2 Auto Scaling resources (for example, Auto Scaling
     *         groups, launch configurations, or lifecycle hooks). For more information, see <a
     *         href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html"
     *         >DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws ServiceLinkedRoleFailureException
     *         The service-linked role is not yet ready for use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.PutNotificationConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/PutNotificationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutNotificationConfigurationResponse putNotificationConfiguration(
            PutNotificationConfigurationRequest putNotificationConfigurationRequest) throws LimitExceededException,
            ResourceContentionException, ServiceLinkedRoleFailureException, AwsServiceException, SdkClientException,
            AutoScalingException {

        HttpResponseHandler<PutNotificationConfigurationResponse> responseHandler = protocolFactory
                .createResponseHandler(PutNotificationConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putNotificationConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutNotificationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<PutNotificationConfigurationRequest, PutNotificationConfigurationResponse>()
                            .withOperationName("PutNotificationConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(putNotificationConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutNotificationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates a scaling policy for an Auto Scaling group.
     * </p>
     * <p>
     * For more information about using scaling policies to scale your Auto Scaling group, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-scaling-target-tracking.html">Target tracking
     * scaling policies</a> and <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-scaling-simple-step.html">Step and simple scaling
     * policies</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param putScalingPolicyRequest
     * @return Result of the PutScalingPolicy operation returned by the service.
     * @throws LimitExceededException
     *         You have already reached a limit for your Amazon EC2 Auto Scaling resources (for example, Auto Scaling
     *         groups, launch configurations, or lifecycle hooks). For more information, see <a
     *         href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html"
     *         >DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws ServiceLinkedRoleFailureException
     *         The service-linked role is not yet ready for use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.PutScalingPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/PutScalingPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutScalingPolicyResponse putScalingPolicy(PutScalingPolicyRequest putScalingPolicyRequest)
            throws LimitExceededException, ResourceContentionException, ServiceLinkedRoleFailureException, AwsServiceException,
            SdkClientException, AutoScalingException {

        HttpResponseHandler<PutScalingPolicyResponse> responseHandler = protocolFactory
                .createResponseHandler(PutScalingPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putScalingPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutScalingPolicy");

            return clientHandler.execute(new ClientExecutionParams<PutScalingPolicyRequest, PutScalingPolicyResponse>()
                    .withOperationName("PutScalingPolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putScalingPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutScalingPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates a scheduled scaling action for an Auto Scaling group. If you leave a parameter unspecified
     * when updating a scheduled scaling action, the corresponding value remains unchanged.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/schedule_time.html">Scheduled scaling</a> in the
     * <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param putScheduledUpdateGroupActionRequest
     * @return Result of the PutScheduledUpdateGroupAction operation returned by the service.
     * @throws AlreadyExistsException
     *         You already have an Auto Scaling group or launch configuration with this name.
     * @throws LimitExceededException
     *         You have already reached a limit for your Amazon EC2 Auto Scaling resources (for example, Auto Scaling
     *         groups, launch configurations, or lifecycle hooks). For more information, see <a
     *         href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html"
     *         >DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.PutScheduledUpdateGroupAction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/PutScheduledUpdateGroupAction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutScheduledUpdateGroupActionResponse putScheduledUpdateGroupAction(
            PutScheduledUpdateGroupActionRequest putScheduledUpdateGroupActionRequest) throws AlreadyExistsException,
            LimitExceededException, ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<PutScheduledUpdateGroupActionResponse> responseHandler = protocolFactory
                .createResponseHandler(PutScheduledUpdateGroupActionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putScheduledUpdateGroupActionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutScheduledUpdateGroupAction");

            return clientHandler
                    .execute(new ClientExecutionParams<PutScheduledUpdateGroupActionRequest, PutScheduledUpdateGroupActionResponse>()
                            .withOperationName("PutScheduledUpdateGroupAction").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(putScheduledUpdateGroupActionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutScheduledUpdateGroupActionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Records a heartbeat for the lifecycle action associated with the specified token or instance. This extends the
     * timeout by the length of time defined using the <a>PutLifecycleHook</a> API call.
     * </p>
     * <p>
     * This step is a part of the procedure for adding a lifecycle hook to an Auto Scaling group:
     * </p>
     * <ol>
     * <li>
     * <p>
     * (Optional) Create a Lambda function and a rule that allows CloudWatch Events to invoke your Lambda function when
     * Amazon EC2 Auto Scaling launches or terminates instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * (Optional) Create a notification target and an IAM role. The target can be either an Amazon SQS queue or an
     * Amazon SNS topic. The role allows Amazon EC2 Auto Scaling to publish lifecycle notifications to the target.
     * </p>
     * </li>
     * <li>
     * <p>
     * Create the lifecycle hook. Specify whether the hook is used when the instances launch or terminate.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>If you need more time, record the lifecycle action heartbeat to keep the instance in a pending state.</b>
     * </p>
     * </li>
     * <li>
     * <p>
     * If you finish before the timeout period ends, complete the lifecycle action.
     * </p>
     * </li>
     * </ol>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/AutoScalingGroupLifecycle.html">Auto Scaling
     * lifecycle</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param recordLifecycleActionHeartbeatRequest
     * @return Result of the RecordLifecycleActionHeartbeat operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.RecordLifecycleActionHeartbeat
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/RecordLifecycleActionHeartbeat"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RecordLifecycleActionHeartbeatResponse recordLifecycleActionHeartbeat(
            RecordLifecycleActionHeartbeatRequest recordLifecycleActionHeartbeatRequest) throws ResourceContentionException,
            AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<RecordLifecycleActionHeartbeatResponse> responseHandler = protocolFactory
                .createResponseHandler(RecordLifecycleActionHeartbeatResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                recordLifecycleActionHeartbeatRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RecordLifecycleActionHeartbeat");

            return clientHandler
                    .execute(new ClientExecutionParams<RecordLifecycleActionHeartbeatRequest, RecordLifecycleActionHeartbeatResponse>()
                            .withOperationName("RecordLifecycleActionHeartbeat").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(recordLifecycleActionHeartbeatRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RecordLifecycleActionHeartbeatRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Resumes the specified suspended auto scaling processes, or all suspended process, for the specified Auto Scaling
     * group.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-suspend-resume-processes.html">Suspending and
     * resuming scaling processes</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param resumeProcessesRequest
     * @return Result of the ResumeProcesses operation returned by the service.
     * @throws ResourceInUseException
     *         The operation can't be performed because the resource is in use.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.ResumeProcesses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/ResumeProcesses" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ResumeProcessesResponse resumeProcesses(ResumeProcessesRequest resumeProcessesRequest) throws ResourceInUseException,
            ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<ResumeProcessesResponse> responseHandler = protocolFactory
                .createResponseHandler(ResumeProcessesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resumeProcessesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResumeProcesses");

            return clientHandler.execute(new ClientExecutionParams<ResumeProcessesRequest, ResumeProcessesResponse>()
                    .withOperationName("ResumeProcesses").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(resumeProcessesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ResumeProcessesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sets the size of the specified Auto Scaling group.
     * </p>
     * <p>
     * If a scale-in activity occurs as a result of a new <code>DesiredCapacity</code> value that is lower than the
     * current size of the group, the Auto Scaling group uses its termination policy to determine which instances to
     * terminate.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-manual-scaling.html">Manual scaling</a> in the
     * <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param setDesiredCapacityRequest
     * @return Result of the SetDesiredCapacity operation returned by the service.
     * @throws ScalingActivityInProgressException
     *         The operation can't be performed because there are scaling activities in progress.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.SetDesiredCapacity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/SetDesiredCapacity" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public SetDesiredCapacityResponse setDesiredCapacity(SetDesiredCapacityRequest setDesiredCapacityRequest)
            throws ScalingActivityInProgressException, ResourceContentionException, AwsServiceException, SdkClientException,
            AutoScalingException {

        HttpResponseHandler<SetDesiredCapacityResponse> responseHandler = protocolFactory
                .createResponseHandler(SetDesiredCapacityResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setDesiredCapacityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetDesiredCapacity");

            return clientHandler.execute(new ClientExecutionParams<SetDesiredCapacityRequest, SetDesiredCapacityResponse>()
                    .withOperationName("SetDesiredCapacity").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(setDesiredCapacityRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SetDesiredCapacityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sets the health status of the specified instance.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/healthcheck.html">Health
     * checks for Auto Scaling instances</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param setInstanceHealthRequest
     * @return Result of the SetInstanceHealth operation returned by the service.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.SetInstanceHealth
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/SetInstanceHealth" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public SetInstanceHealthResponse setInstanceHealth(SetInstanceHealthRequest setInstanceHealthRequest)
            throws ResourceContentionException, AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<SetInstanceHealthResponse> responseHandler = protocolFactory
                .createResponseHandler(SetInstanceHealthResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setInstanceHealthRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetInstanceHealth");

            return clientHandler.execute(new ClientExecutionParams<SetInstanceHealthRequest, SetInstanceHealthResponse>()
                    .withOperationName("SetInstanceHealth").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(setInstanceHealthRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SetInstanceHealthRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the instance protection settings of the specified instances.
     * </p>
     * <p>
     * For more information about preventing instances that are part of an Auto Scaling group from terminating on scale
     * in, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html#instance-protection"
     * >Instance scale-in protection</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * If you exceed your maximum limit of instance IDs, which is 50 per Auto Scaling group, the call fails.
     * </p>
     *
     * @param setInstanceProtectionRequest
     * @return Result of the SetInstanceProtection operation returned by the service.
     * @throws LimitExceededException
     *         You have already reached a limit for your Amazon EC2 Auto Scaling resources (for example, Auto Scaling
     *         groups, launch configurations, or lifecycle hooks). For more information, see <a
     *         href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html"
     *         >DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.SetInstanceProtection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/SetInstanceProtection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SetInstanceProtectionResponse setInstanceProtection(SetInstanceProtectionRequest setInstanceProtectionRequest)
            throws LimitExceededException, ResourceContentionException, AwsServiceException, SdkClientException,
            AutoScalingException {

        HttpResponseHandler<SetInstanceProtectionResponse> responseHandler = protocolFactory
                .createResponseHandler(SetInstanceProtectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setInstanceProtectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetInstanceProtection");

            return clientHandler.execute(new ClientExecutionParams<SetInstanceProtectionRequest, SetInstanceProtectionResponse>()
                    .withOperationName("SetInstanceProtection").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(setInstanceProtectionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SetInstanceProtectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a new instance refresh operation, which triggers a rolling replacement of all previously launched
     * instances in the Auto Scaling group with a new group of instances.
     * </p>
     * <p>
     * If successful, this call creates a new instance refresh request with a unique ID that you can use to track its
     * progress. To query its status, call the <a>DescribeInstanceRefreshes</a> API. To describe the instance refreshes
     * that have already run, call the <a>DescribeInstanceRefreshes</a> API. To cancel an instance refresh operation in
     * progress, use the <a>CancelInstanceRefresh</a> API.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-instance-refresh.html">Replacing Auto Scaling
     * Instances Based on an Instance Refresh</a>.
     * </p>
     *
     * @param startInstanceRefreshRequest
     * @return Result of the StartInstanceRefresh operation returned by the service.
     * @throws LimitExceededException
     *         You have already reached a limit for your Amazon EC2 Auto Scaling resources (for example, Auto Scaling
     *         groups, launch configurations, or lifecycle hooks). For more information, see <a
     *         href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html"
     *         >DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws InstanceRefreshInProgressException
     *         The request failed because an active instance refresh operation already exists for the specified Auto
     *         Scaling group.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.StartInstanceRefresh
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/StartInstanceRefresh"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartInstanceRefreshResponse startInstanceRefresh(StartInstanceRefreshRequest startInstanceRefreshRequest)
            throws LimitExceededException, ResourceContentionException, InstanceRefreshInProgressException, AwsServiceException,
            SdkClientException, AutoScalingException {

        HttpResponseHandler<StartInstanceRefreshResponse> responseHandler = protocolFactory
                .createResponseHandler(StartInstanceRefreshResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startInstanceRefreshRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartInstanceRefresh");

            return clientHandler.execute(new ClientExecutionParams<StartInstanceRefreshRequest, StartInstanceRefreshResponse>()
                    .withOperationName("StartInstanceRefresh").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startInstanceRefreshRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartInstanceRefreshRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Suspends the specified auto scaling processes, or all processes, for the specified Auto Scaling group.
     * </p>
     * <p>
     * If you suspend either the <code>Launch</code> or <code>Terminate</code> process types, it can prevent other
     * process types from functioning properly. For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-suspend-resume-processes.html">Suspending and
     * resuming scaling processes</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * To resume processes that have been suspended, call the <a>ResumeProcesses</a> API.
     * </p>
     *
     * @param suspendProcessesRequest
     * @return Result of the SuspendProcesses operation returned by the service.
     * @throws ResourceInUseException
     *         The operation can't be performed because the resource is in use.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.SuspendProcesses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/SuspendProcesses" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public SuspendProcessesResponse suspendProcesses(SuspendProcessesRequest suspendProcessesRequest)
            throws ResourceInUseException, ResourceContentionException, AwsServiceException, SdkClientException,
            AutoScalingException {

        HttpResponseHandler<SuspendProcessesResponse> responseHandler = protocolFactory
                .createResponseHandler(SuspendProcessesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, suspendProcessesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SuspendProcesses");

            return clientHandler.execute(new ClientExecutionParams<SuspendProcessesRequest, SuspendProcessesResponse>()
                    .withOperationName("SuspendProcesses").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(suspendProcessesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SuspendProcessesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Terminates the specified instance and optionally adjusts the desired group size.
     * </p>
     * <p>
     * This call simply makes a termination request. The instance is not terminated immediately. When an instance is
     * terminated, the instance status changes to <code>terminated</code>. You can't connect to or start an instance
     * after you've terminated it.
     * </p>
     * <p>
     * If you do not specify the option to decrement the desired capacity, Amazon EC2 Auto Scaling launches instances to
     * replace the ones that are terminated.
     * </p>
     * <p>
     * By default, Amazon EC2 Auto Scaling balances instances across all Availability Zones. If you decrement the
     * desired capacity, your Auto Scaling group can become unbalanced between Availability Zones. Amazon EC2 Auto
     * Scaling tries to rebalance the group, and rebalancing might terminate instances in other zones. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/ec2/userguide/auto-scaling-benefits.html#AutoScalingBehavior.InstanceUsage"
     * >Rebalancing activities</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param terminateInstanceInAutoScalingGroupRequest
     * @return Result of the TerminateInstanceInAutoScalingGroup operation returned by the service.
     * @throws ScalingActivityInProgressException
     *         The operation can't be performed because there are scaling activities in progress.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.TerminateInstanceInAutoScalingGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/TerminateInstanceInAutoScalingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TerminateInstanceInAutoScalingGroupResponse terminateInstanceInAutoScalingGroup(
            TerminateInstanceInAutoScalingGroupRequest terminateInstanceInAutoScalingGroupRequest)
            throws ScalingActivityInProgressException, ResourceContentionException, AwsServiceException, SdkClientException,
            AutoScalingException {

        HttpResponseHandler<TerminateInstanceInAutoScalingGroupResponse> responseHandler = protocolFactory
                .createResponseHandler(TerminateInstanceInAutoScalingGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                terminateInstanceInAutoScalingGroupRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TerminateInstanceInAutoScalingGroup");

            return clientHandler
                    .execute(new ClientExecutionParams<TerminateInstanceInAutoScalingGroupRequest, TerminateInstanceInAutoScalingGroupResponse>()
                            .withOperationName("TerminateInstanceInAutoScalingGroup").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(terminateInstanceInAutoScalingGroupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new TerminateInstanceInAutoScalingGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the configuration for the specified Auto Scaling group.
     * </p>
     * <p>
     * To update an Auto Scaling group, specify the name of the group and the parameter that you want to change. Any
     * parameters that you don't specify are not changed by this update request. The new settings take effect on any
     * scaling activities after this call returns.
     * </p>
     * <p>
     * If you associate a new launch configuration or template with an Auto Scaling group, all new instances will get
     * the updated configuration. Existing instances continue to run with the configuration that they were originally
     * launched with. When you update a group to specify a mixed instances policy instead of a launch configuration or
     * template, existing instances may be replaced to match the new purchasing options that you specified in the
     * policy. For example, if the group currently has 100% On-Demand capacity and the policy specifies 50% Spot
     * capacity, this means that half of your instances will be gradually terminated and relaunched as Spot Instances.
     * When replacing instances, Amazon EC2 Auto Scaling launches new instances before terminating the old ones, so that
     * updating your group does not compromise the performance or availability of your application.
     * </p>
     * <p>
     * Note the following about changing <code>DesiredCapacity</code>, <code>MaxSize</code>, or <code>MinSize</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If a scale-in activity occurs as a result of a new <code>DesiredCapacity</code> value that is lower than the
     * current size of the group, the Auto Scaling group uses its termination policy to determine which instances to
     * terminate.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify a new value for <code>MinSize</code> without specifying a value for <code>DesiredCapacity</code>,
     * and the new <code>MinSize</code> is larger than the current size of the group, this sets the group's
     * <code>DesiredCapacity</code> to the new <code>MinSize</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify a new value for <code>MaxSize</code> without specifying a value for <code>DesiredCapacity</code>,
     * and the new <code>MaxSize</code> is smaller than the current size of the group, this sets the group's
     * <code>DesiredCapacity</code> to the new <code>MaxSize</code> value.
     * </p>
     * </li>
     * </ul>
     * <p>
     * To see which parameters have been set, call the <a>DescribeAutoScalingGroups</a> API. To view the scaling
     * policies for an Auto Scaling group, call the <a>DescribePolicies</a> API. If the group has scaling policies, you
     * can update them by calling the <a>PutScalingPolicy</a> API.
     * </p>
     *
     * @param updateAutoScalingGroupRequest
     * @return Result of the UpdateAutoScalingGroup operation returned by the service.
     * @throws ScalingActivityInProgressException
     *         The operation can't be performed because there are scaling activities in progress.
     * @throws ResourceContentionException
     *         You already have a pending update to an Amazon EC2 Auto Scaling resource (for example, an Auto Scaling
     *         group, instance, or load balancer).
     * @throws ServiceLinkedRoleFailureException
     *         The service-linked role is not yet ready for use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingClient.UpdateAutoScalingGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/UpdateAutoScalingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateAutoScalingGroupResponse updateAutoScalingGroup(UpdateAutoScalingGroupRequest updateAutoScalingGroupRequest)
            throws ScalingActivityInProgressException, ResourceContentionException, ServiceLinkedRoleFailureException,
            AwsServiceException, SdkClientException, AutoScalingException {

        HttpResponseHandler<UpdateAutoScalingGroupResponse> responseHandler = protocolFactory
                .createResponseHandler(UpdateAutoScalingGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAutoScalingGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAutoScalingGroup");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateAutoScalingGroupRequest, UpdateAutoScalingGroupResponse>()
                            .withOperationName("UpdateAutoScalingGroup").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateAutoScalingGroupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateAutoScalingGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private AwsQueryProtocolFactory init() {
        return AwsQueryProtocolFactory
                .builder()
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUse")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InstanceRefreshInProgress")
                                .exceptionBuilderSupplier(InstanceRefreshInProgressException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ScalingActivityInProgress")
                                .exceptionBuilderSupplier(ScalingActivityInProgressException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextToken")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceeded")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AlreadyExists")
                                .exceptionBuilderSupplier(AlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ActiveInstanceRefreshNotFound")
                                .exceptionBuilderSupplier(ActiveInstanceRefreshNotFoundException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceContention")
                                .exceptionBuilderSupplier(ResourceContentionException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceLinkedRoleFailure")
                                .exceptionBuilderSupplier(ServiceLinkedRoleFailureException::builder).httpStatusCode(500).build())
                .clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(AutoScalingException::builder).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends AutoScalingRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public AutoScalingWaiter waiter() {
        return AutoScalingWaiter.builder().client(this).build();
    }
}
