/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscaling.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a specific metric.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Metric implements SdkPojo, Serializable, ToCopyableBuilder<Metric.Builder, Metric> {
    private static final SdkField<String> NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Namespace").getter(getter(Metric::namespace)).setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Namespace").build()).build();

    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MetricName").getter(getter(Metric::metricName)).setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricName").build()).build();

    private static final SdkField<List<MetricDimension>> DIMENSIONS_FIELD = SdkField
            .<List<MetricDimension>> builder(MarshallingType.LIST)
            .memberName("Dimensions")
            .getter(getter(Metric::dimensions))
            .setter(setter(Builder::dimensions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Dimensions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MetricDimension> builder(MarshallingType.SDK_POJO)
                                            .constructor(MetricDimension::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAMESPACE_FIELD,
            METRIC_NAME_FIELD, DIMENSIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String namespace;

    private final String metricName;

    private final List<MetricDimension> dimensions;

    private Metric(BuilderImpl builder) {
        this.namespace = builder.namespace;
        this.metricName = builder.metricName;
        this.dimensions = builder.dimensions;
    }

    /**
     * <p>
     * The namespace of the metric. For more information, see the table in <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/aws-services-cloudwatch-metrics.html">Amazon
     * Web Services services that publish CloudWatch metrics </a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     * 
     * @return The namespace of the metric. For more information, see the table in <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/aws-services-cloudwatch-metrics.html"
     *         >Amazon Web Services services that publish CloudWatch metrics </a> in the <i>Amazon CloudWatch User
     *         Guide</i>.
     */
    public final String namespace() {
        return namespace;
    }

    /**
     * <p>
     * The name of the metric.
     * </p>
     * 
     * @return The name of the metric.
     */
    public final String metricName() {
        return metricName;
    }

    /**
     * For responses, this returns true if the service returned a value for the Dimensions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDimensions() {
        return dimensions != null && !(dimensions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The dimensions for the metric. For the list of available dimensions, see the Amazon Web Services documentation
     * available from the table in <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/aws-services-cloudwatch-metrics.html">Amazon
     * Web Services services that publish CloudWatch metrics </a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     * <p>
     * Conditional: If you published your metric with dimensions, you must specify the same dimensions in your scaling
     * policy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDimensions} method.
     * </p>
     * 
     * @return The dimensions for the metric. For the list of available dimensions, see the Amazon Web Services
     *         documentation available from the table in <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/aws-services-cloudwatch-metrics.html"
     *         >Amazon Web Services services that publish CloudWatch metrics </a> in the <i>Amazon CloudWatch User
     *         Guide</i>. </p>
     *         <p>
     *         Conditional: If you published your metric with dimensions, you must specify the same dimensions in your
     *         scaling policy.
     */
    public final List<MetricDimension> dimensions() {
        return dimensions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(metricName());
        hashCode = 31 * hashCode + Objects.hashCode(hasDimensions() ? dimensions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Metric)) {
            return false;
        }
        Metric other = (Metric) obj;
        return Objects.equals(namespace(), other.namespace()) && Objects.equals(metricName(), other.metricName())
                && hasDimensions() == other.hasDimensions() && Objects.equals(dimensions(), other.dimensions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Metric").add("Namespace", namespace()).add("MetricName", metricName())
                .add("Dimensions", hasDimensions() ? dimensions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "MetricName":
            return Optional.ofNullable(clazz.cast(metricName()));
        case "Dimensions":
            return Optional.ofNullable(clazz.cast(dimensions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Metric, T> g) {
        return obj -> g.apply((Metric) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Metric> {
        /**
         * <p>
         * The namespace of the metric. For more information, see the table in <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/aws-services-cloudwatch-metrics.html"
         * >Amazon Web Services services that publish CloudWatch metrics </a> in the <i>Amazon CloudWatch User
         * Guide</i>.
         * </p>
         * 
         * @param namespace
         *        The namespace of the metric. For more information, see the table in <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/aws-services-cloudwatch-metrics.html"
         *        >Amazon Web Services services that publish CloudWatch metrics </a> in the <i>Amazon CloudWatch User
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * The name of the metric.
         * </p>
         * 
         * @param metricName
         *        The name of the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * The dimensions for the metric. For the list of available dimensions, see the Amazon Web Services
         * documentation available from the table in <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/aws-services-cloudwatch-metrics.html"
         * >Amazon Web Services services that publish CloudWatch metrics </a> in the <i>Amazon CloudWatch User
         * Guide</i>.
         * </p>
         * <p>
         * Conditional: If you published your metric with dimensions, you must specify the same dimensions in your
         * scaling policy.
         * </p>
         * 
         * @param dimensions
         *        The dimensions for the metric. For the list of available dimensions, see the Amazon Web Services
         *        documentation available from the table in <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/aws-services-cloudwatch-metrics.html"
         *        >Amazon Web Services services that publish CloudWatch metrics </a> in the <i>Amazon CloudWatch User
         *        Guide</i>. </p>
         *        <p>
         *        Conditional: If you published your metric with dimensions, you must specify the same dimensions in
         *        your scaling policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Collection<MetricDimension> dimensions);

        /**
         * <p>
         * The dimensions for the metric. For the list of available dimensions, see the Amazon Web Services
         * documentation available from the table in <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/aws-services-cloudwatch-metrics.html"
         * >Amazon Web Services services that publish CloudWatch metrics </a> in the <i>Amazon CloudWatch User
         * Guide</i>.
         * </p>
         * <p>
         * Conditional: If you published your metric with dimensions, you must specify the same dimensions in your
         * scaling policy.
         * </p>
         * 
         * @param dimensions
         *        The dimensions for the metric. For the list of available dimensions, see the Amazon Web Services
         *        documentation available from the table in <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/aws-services-cloudwatch-metrics.html"
         *        >Amazon Web Services services that publish CloudWatch metrics </a> in the <i>Amazon CloudWatch User
         *        Guide</i>. </p>
         *        <p>
         *        Conditional: If you published your metric with dimensions, you must specify the same dimensions in
         *        your scaling policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(MetricDimension... dimensions);

        /**
         * <p>
         * The dimensions for the metric. For the list of available dimensions, see the Amazon Web Services
         * documentation available from the table in <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/aws-services-cloudwatch-metrics.html"
         * >Amazon Web Services services that publish CloudWatch metrics </a> in the <i>Amazon CloudWatch User
         * Guide</i>.
         * </p>
         * <p>
         * Conditional: If you published your metric with dimensions, you must specify the same dimensions in your
         * scaling policy.
         * </p>
         * This is a convenience that creates an instance of the {@link List<MetricDimension>.Builder} avoiding the need
         * to create one manually via {@link List<MetricDimension>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<MetricDimension>.Builder#build()} is called immediately and
         * its result is passed to {@link #dimensions(List<MetricDimension>)}.
         * 
         * @param dimensions
         *        a consumer that will call methods on {@link List<MetricDimension>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dimensions(List<MetricDimension>)
         */
        Builder dimensions(Consumer<MetricDimension.Builder>... dimensions);
    }

    static final class BuilderImpl implements Builder {
        private String namespace;

        private String metricName;

        private List<MetricDimension> dimensions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Metric model) {
            namespace(model.namespace);
            metricName(model.metricName);
            dimensions(model.dimensions);
        }

        public final String getNamespace() {
            return namespace;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        @Override
        @Transient
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final String getMetricName() {
            return metricName;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        @Override
        @Transient
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        public final List<MetricDimension.Builder> getDimensions() {
            List<MetricDimension.Builder> result = MetricDimensionsCopier.copyToBuilder(this.dimensions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDimensions(Collection<MetricDimension.BuilderImpl> dimensions) {
            this.dimensions = MetricDimensionsCopier.copyFromBuilder(dimensions);
        }

        @Override
        @Transient
        public final Builder dimensions(Collection<MetricDimension> dimensions) {
            this.dimensions = MetricDimensionsCopier.copy(dimensions);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder dimensions(MetricDimension... dimensions) {
            dimensions(Arrays.asList(dimensions));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder dimensions(Consumer<MetricDimension.Builder>... dimensions) {
            dimensions(Stream.of(dimensions).map(c -> MetricDimension.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Metric build() {
            return new Metric(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
