/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscaling.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a predictive scaling policy configuration to use with Amazon EC2 Auto Scaling.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PredictiveScalingConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<PredictiveScalingConfiguration.Builder, PredictiveScalingConfiguration> {
    private static final SdkField<List<PredictiveScalingMetricSpecification>> METRIC_SPECIFICATIONS_FIELD = SdkField
            .<List<PredictiveScalingMetricSpecification>> builder(MarshallingType.LIST)
            .memberName("MetricSpecifications")
            .getter(getter(PredictiveScalingConfiguration::metricSpecifications))
            .setter(setter(Builder::metricSpecifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricSpecifications").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PredictiveScalingMetricSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(PredictiveScalingMetricSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Mode")
            .getter(getter(PredictiveScalingConfiguration::modeAsString)).setter(setter(Builder::mode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Mode").build()).build();

    private static final SdkField<Integer> SCHEDULING_BUFFER_TIME_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("SchedulingBufferTime").getter(getter(PredictiveScalingConfiguration::schedulingBufferTime))
            .setter(setter(Builder::schedulingBufferTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SchedulingBufferTime").build())
            .build();

    private static final SdkField<String> MAX_CAPACITY_BREACH_BEHAVIOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MaxCapacityBreachBehavior")
            .getter(getter(PredictiveScalingConfiguration::maxCapacityBreachBehaviorAsString))
            .setter(setter(Builder::maxCapacityBreachBehavior))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxCapacityBreachBehavior").build())
            .build();

    private static final SdkField<Integer> MAX_CAPACITY_BUFFER_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxCapacityBuffer").getter(getter(PredictiveScalingConfiguration::maxCapacityBuffer))
            .setter(setter(Builder::maxCapacityBuffer))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxCapacityBuffer").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(METRIC_SPECIFICATIONS_FIELD,
            MODE_FIELD, SCHEDULING_BUFFER_TIME_FIELD, MAX_CAPACITY_BREACH_BEHAVIOR_FIELD, MAX_CAPACITY_BUFFER_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<PredictiveScalingMetricSpecification> metricSpecifications;

    private final String mode;

    private final Integer schedulingBufferTime;

    private final String maxCapacityBreachBehavior;

    private final Integer maxCapacityBuffer;

    private PredictiveScalingConfiguration(BuilderImpl builder) {
        this.metricSpecifications = builder.metricSpecifications;
        this.mode = builder.mode;
        this.schedulingBufferTime = builder.schedulingBufferTime;
        this.maxCapacityBreachBehavior = builder.maxCapacityBreachBehavior;
        this.maxCapacityBuffer = builder.maxCapacityBuffer;
    }

    /**
     * For responses, this returns true if the service returned a value for the MetricSpecifications property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMetricSpecifications() {
        return metricSpecifications != null && !(metricSpecifications instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * This structure includes the metrics and target utilization to use for predictive scaling.
     * </p>
     * <p>
     * This is an array, but we currently only support a single metric specification. That is, you can specify a target
     * value and a single metric pair, or a target value and one scaling metric and one load metric.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetricSpecifications} method.
     * </p>
     * 
     * @return This structure includes the metrics and target utilization to use for predictive scaling. </p>
     *         <p>
     *         This is an array, but we currently only support a single metric specification. That is, you can specify a
     *         target value and a single metric pair, or a target value and one scaling metric and one load metric.
     */
    public final List<PredictiveScalingMetricSpecification> metricSpecifications() {
        return metricSpecifications;
    }

    /**
     * <p>
     * The predictive scaling mode. Defaults to <code>ForecastOnly</code> if not specified.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mode} will return
     * {@link PredictiveScalingMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modeAsString}.
     * </p>
     * 
     * @return The predictive scaling mode. Defaults to <code>ForecastOnly</code> if not specified.
     * @see PredictiveScalingMode
     */
    public final PredictiveScalingMode mode() {
        return PredictiveScalingMode.fromValue(mode);
    }

    /**
     * <p>
     * The predictive scaling mode. Defaults to <code>ForecastOnly</code> if not specified.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mode} will return
     * {@link PredictiveScalingMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modeAsString}.
     * </p>
     * 
     * @return The predictive scaling mode. Defaults to <code>ForecastOnly</code> if not specified.
     * @see PredictiveScalingMode
     */
    public final String modeAsString() {
        return mode;
    }

    /**
     * <p>
     * The amount of time, in seconds, by which the instance launch time can be advanced. For example, the forecast says
     * to add capacity at 10:00 AM, and you choose to pre-launch instances by 5 minutes. In that case, the instances
     * will be launched at 9:55 AM. The intention is to give resources time to be provisioned. It can take a few minutes
     * to launch an EC2 instance. The actual amount of time required depends on several factors, such as the size of the
     * instance and whether there are startup scripts to complete.
     * </p>
     * <p>
     * The value must be less than the forecast interval duration of 3600 seconds (60 minutes). Defaults to 300 seconds
     * if not specified.
     * </p>
     * 
     * @return The amount of time, in seconds, by which the instance launch time can be advanced. For example, the
     *         forecast says to add capacity at 10:00 AM, and you choose to pre-launch instances by 5 minutes. In that
     *         case, the instances will be launched at 9:55 AM. The intention is to give resources time to be
     *         provisioned. It can take a few minutes to launch an EC2 instance. The actual amount of time required
     *         depends on several factors, such as the size of the instance and whether there are startup scripts to
     *         complete. </p>
     *         <p>
     *         The value must be less than the forecast interval duration of 3600 seconds (60 minutes). Defaults to 300
     *         seconds if not specified.
     */
    public final Integer schedulingBufferTime() {
        return schedulingBufferTime;
    }

    /**
     * <p>
     * Defines the behavior that should be applied if the forecast capacity approaches or exceeds the maximum capacity
     * of the Auto Scaling group. Defaults to <code>HonorMaxCapacity</code> if not specified.
     * </p>
     * <p>
     * The following are possible values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>HonorMaxCapacity</code> - Amazon EC2 Auto Scaling cannot scale out capacity higher than the maximum
     * capacity. The maximum capacity is enforced as a hard limit.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IncreaseMaxCapacity</code> - Amazon EC2 Auto Scaling can scale out capacity higher than the maximum
     * capacity when the forecast capacity is close to or exceeds the maximum capacity. The upper limit is determined by
     * the forecasted capacity and the value for <code>MaxCapacityBuffer</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #maxCapacityBreachBehavior} will return
     * {@link PredictiveScalingMaxCapacityBreachBehavior#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #maxCapacityBreachBehaviorAsString}.
     * </p>
     * 
     * @return Defines the behavior that should be applied if the forecast capacity approaches or exceeds the maximum
     *         capacity of the Auto Scaling group. Defaults to <code>HonorMaxCapacity</code> if not specified.</p>
     *         <p>
     *         The following are possible values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>HonorMaxCapacity</code> - Amazon EC2 Auto Scaling cannot scale out capacity higher than the maximum
     *         capacity. The maximum capacity is enforced as a hard limit.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IncreaseMaxCapacity</code> - Amazon EC2 Auto Scaling can scale out capacity higher than the maximum
     *         capacity when the forecast capacity is close to or exceeds the maximum capacity. The upper limit is
     *         determined by the forecasted capacity and the value for <code>MaxCapacityBuffer</code>.
     *         </p>
     *         </li>
     * @see PredictiveScalingMaxCapacityBreachBehavior
     */
    public final PredictiveScalingMaxCapacityBreachBehavior maxCapacityBreachBehavior() {
        return PredictiveScalingMaxCapacityBreachBehavior.fromValue(maxCapacityBreachBehavior);
    }

    /**
     * <p>
     * Defines the behavior that should be applied if the forecast capacity approaches or exceeds the maximum capacity
     * of the Auto Scaling group. Defaults to <code>HonorMaxCapacity</code> if not specified.
     * </p>
     * <p>
     * The following are possible values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>HonorMaxCapacity</code> - Amazon EC2 Auto Scaling cannot scale out capacity higher than the maximum
     * capacity. The maximum capacity is enforced as a hard limit.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IncreaseMaxCapacity</code> - Amazon EC2 Auto Scaling can scale out capacity higher than the maximum
     * capacity when the forecast capacity is close to or exceeds the maximum capacity. The upper limit is determined by
     * the forecasted capacity and the value for <code>MaxCapacityBuffer</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #maxCapacityBreachBehavior} will return
     * {@link PredictiveScalingMaxCapacityBreachBehavior#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #maxCapacityBreachBehaviorAsString}.
     * </p>
     * 
     * @return Defines the behavior that should be applied if the forecast capacity approaches or exceeds the maximum
     *         capacity of the Auto Scaling group. Defaults to <code>HonorMaxCapacity</code> if not specified.</p>
     *         <p>
     *         The following are possible values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>HonorMaxCapacity</code> - Amazon EC2 Auto Scaling cannot scale out capacity higher than the maximum
     *         capacity. The maximum capacity is enforced as a hard limit.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IncreaseMaxCapacity</code> - Amazon EC2 Auto Scaling can scale out capacity higher than the maximum
     *         capacity when the forecast capacity is close to or exceeds the maximum capacity. The upper limit is
     *         determined by the forecasted capacity and the value for <code>MaxCapacityBuffer</code>.
     *         </p>
     *         </li>
     * @see PredictiveScalingMaxCapacityBreachBehavior
     */
    public final String maxCapacityBreachBehaviorAsString() {
        return maxCapacityBreachBehavior;
    }

    /**
     * <p>
     * The size of the capacity buffer to use when the forecast capacity is close to or exceeds the maximum capacity.
     * The value is specified as a percentage relative to the forecast capacity. For example, if the buffer is 10, this
     * means a 10 percent buffer, such that if the forecast capacity is 50, and the maximum capacity is 40, then the
     * effective maximum capacity is 55.
     * </p>
     * <p>
     * If set to 0, Amazon EC2 Auto Scaling may scale capacity higher than the maximum capacity to equal but not exceed
     * forecast capacity.
     * </p>
     * <p>
     * Required if the <code>MaxCapacityBreachBehavior</code> property is set to <code>IncreaseMaxCapacity</code>, and
     * cannot be used otherwise.
     * </p>
     * 
     * @return The size of the capacity buffer to use when the forecast capacity is close to or exceeds the maximum
     *         capacity. The value is specified as a percentage relative to the forecast capacity. For example, if the
     *         buffer is 10, this means a 10 percent buffer, such that if the forecast capacity is 50, and the maximum
     *         capacity is 40, then the effective maximum capacity is 55.</p>
     *         <p>
     *         If set to 0, Amazon EC2 Auto Scaling may scale capacity higher than the maximum capacity to equal but not
     *         exceed forecast capacity.
     *         </p>
     *         <p>
     *         Required if the <code>MaxCapacityBreachBehavior</code> property is set to
     *         <code>IncreaseMaxCapacity</code>, and cannot be used otherwise.
     */
    public final Integer maxCapacityBuffer() {
        return maxCapacityBuffer;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasMetricSpecifications() ? metricSpecifications() : null);
        hashCode = 31 * hashCode + Objects.hashCode(modeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(schedulingBufferTime());
        hashCode = 31 * hashCode + Objects.hashCode(maxCapacityBreachBehaviorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(maxCapacityBuffer());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PredictiveScalingConfiguration)) {
            return false;
        }
        PredictiveScalingConfiguration other = (PredictiveScalingConfiguration) obj;
        return hasMetricSpecifications() == other.hasMetricSpecifications()
                && Objects.equals(metricSpecifications(), other.metricSpecifications())
                && Objects.equals(modeAsString(), other.modeAsString())
                && Objects.equals(schedulingBufferTime(), other.schedulingBufferTime())
                && Objects.equals(maxCapacityBreachBehaviorAsString(), other.maxCapacityBreachBehaviorAsString())
                && Objects.equals(maxCapacityBuffer(), other.maxCapacityBuffer());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PredictiveScalingConfiguration")
                .add("MetricSpecifications", hasMetricSpecifications() ? metricSpecifications() : null)
                .add("Mode", modeAsString()).add("SchedulingBufferTime", schedulingBufferTime())
                .add("MaxCapacityBreachBehavior", maxCapacityBreachBehaviorAsString())
                .add("MaxCapacityBuffer", maxCapacityBuffer()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MetricSpecifications":
            return Optional.ofNullable(clazz.cast(metricSpecifications()));
        case "Mode":
            return Optional.ofNullable(clazz.cast(modeAsString()));
        case "SchedulingBufferTime":
            return Optional.ofNullable(clazz.cast(schedulingBufferTime()));
        case "MaxCapacityBreachBehavior":
            return Optional.ofNullable(clazz.cast(maxCapacityBreachBehaviorAsString()));
        case "MaxCapacityBuffer":
            return Optional.ofNullable(clazz.cast(maxCapacityBuffer()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PredictiveScalingConfiguration, T> g) {
        return obj -> g.apply((PredictiveScalingConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PredictiveScalingConfiguration> {
        /**
         * <p>
         * This structure includes the metrics and target utilization to use for predictive scaling.
         * </p>
         * <p>
         * This is an array, but we currently only support a single metric specification. That is, you can specify a
         * target value and a single metric pair, or a target value and one scaling metric and one load metric.
         * </p>
         * 
         * @param metricSpecifications
         *        This structure includes the metrics and target utilization to use for predictive scaling. </p>
         *        <p>
         *        This is an array, but we currently only support a single metric specification. That is, you can
         *        specify a target value and a single metric pair, or a target value and one scaling metric and one load
         *        metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricSpecifications(Collection<PredictiveScalingMetricSpecification> metricSpecifications);

        /**
         * <p>
         * This structure includes the metrics and target utilization to use for predictive scaling.
         * </p>
         * <p>
         * This is an array, but we currently only support a single metric specification. That is, you can specify a
         * target value and a single metric pair, or a target value and one scaling metric and one load metric.
         * </p>
         * 
         * @param metricSpecifications
         *        This structure includes the metrics and target utilization to use for predictive scaling. </p>
         *        <p>
         *        This is an array, but we currently only support a single metric specification. That is, you can
         *        specify a target value and a single metric pair, or a target value and one scaling metric and one load
         *        metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricSpecifications(PredictiveScalingMetricSpecification... metricSpecifications);

        /**
         * <p>
         * This structure includes the metrics and target utilization to use for predictive scaling.
         * </p>
         * <p>
         * This is an array, but we currently only support a single metric specification. That is, you can specify a
         * target value and a single metric pair, or a target value and one scaling metric and one load metric.
         * </p>
         * This is a convenience method that creates an instance of the {@link List
         * <PredictiveScalingMetricSpecification>.Builder} avoiding the need to create one manually via {@link List
         * <PredictiveScalingMetricSpecification>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<PredictiveScalingMetricSpecification>.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #metricSpecifications(List<PredictiveScalingMetricSpecification>)}.
         * 
         * @param metricSpecifications
         *        a consumer that will call methods on {@link List<PredictiveScalingMetricSpecification>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metricSpecifications(List<PredictiveScalingMetricSpecification>)
         */
        Builder metricSpecifications(Consumer<PredictiveScalingMetricSpecification.Builder>... metricSpecifications);

        /**
         * <p>
         * The predictive scaling mode. Defaults to <code>ForecastOnly</code> if not specified.
         * </p>
         * 
         * @param mode
         *        The predictive scaling mode. Defaults to <code>ForecastOnly</code> if not specified.
         * @see PredictiveScalingMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PredictiveScalingMode
         */
        Builder mode(String mode);

        /**
         * <p>
         * The predictive scaling mode. Defaults to <code>ForecastOnly</code> if not specified.
         * </p>
         * 
         * @param mode
         *        The predictive scaling mode. Defaults to <code>ForecastOnly</code> if not specified.
         * @see PredictiveScalingMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PredictiveScalingMode
         */
        Builder mode(PredictiveScalingMode mode);

        /**
         * <p>
         * The amount of time, in seconds, by which the instance launch time can be advanced. For example, the forecast
         * says to add capacity at 10:00 AM, and you choose to pre-launch instances by 5 minutes. In that case, the
         * instances will be launched at 9:55 AM. The intention is to give resources time to be provisioned. It can take
         * a few minutes to launch an EC2 instance. The actual amount of time required depends on several factors, such
         * as the size of the instance and whether there are startup scripts to complete.
         * </p>
         * <p>
         * The value must be less than the forecast interval duration of 3600 seconds (60 minutes). Defaults to 300
         * seconds if not specified.
         * </p>
         * 
         * @param schedulingBufferTime
         *        The amount of time, in seconds, by which the instance launch time can be advanced. For example, the
         *        forecast says to add capacity at 10:00 AM, and you choose to pre-launch instances by 5 minutes. In
         *        that case, the instances will be launched at 9:55 AM. The intention is to give resources time to be
         *        provisioned. It can take a few minutes to launch an EC2 instance. The actual amount of time required
         *        depends on several factors, such as the size of the instance and whether there are startup scripts to
         *        complete. </p>
         *        <p>
         *        The value must be less than the forecast interval duration of 3600 seconds (60 minutes). Defaults to
         *        300 seconds if not specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schedulingBufferTime(Integer schedulingBufferTime);

        /**
         * <p>
         * Defines the behavior that should be applied if the forecast capacity approaches or exceeds the maximum
         * capacity of the Auto Scaling group. Defaults to <code>HonorMaxCapacity</code> if not specified.
         * </p>
         * <p>
         * The following are possible values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HonorMaxCapacity</code> - Amazon EC2 Auto Scaling cannot scale out capacity higher than the maximum
         * capacity. The maximum capacity is enforced as a hard limit.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IncreaseMaxCapacity</code> - Amazon EC2 Auto Scaling can scale out capacity higher than the maximum
         * capacity when the forecast capacity is close to or exceeds the maximum capacity. The upper limit is
         * determined by the forecasted capacity and the value for <code>MaxCapacityBuffer</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param maxCapacityBreachBehavior
         *        Defines the behavior that should be applied if the forecast capacity approaches or exceeds the maximum
         *        capacity of the Auto Scaling group. Defaults to <code>HonorMaxCapacity</code> if not specified.</p>
         *        <p>
         *        The following are possible values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>HonorMaxCapacity</code> - Amazon EC2 Auto Scaling cannot scale out capacity higher than the
         *        maximum capacity. The maximum capacity is enforced as a hard limit.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IncreaseMaxCapacity</code> - Amazon EC2 Auto Scaling can scale out capacity higher than the
         *        maximum capacity when the forecast capacity is close to or exceeds the maximum capacity. The upper
         *        limit is determined by the forecasted capacity and the value for <code>MaxCapacityBuffer</code>.
         *        </p>
         *        </li>
         * @see PredictiveScalingMaxCapacityBreachBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PredictiveScalingMaxCapacityBreachBehavior
         */
        Builder maxCapacityBreachBehavior(String maxCapacityBreachBehavior);

        /**
         * <p>
         * Defines the behavior that should be applied if the forecast capacity approaches or exceeds the maximum
         * capacity of the Auto Scaling group. Defaults to <code>HonorMaxCapacity</code> if not specified.
         * </p>
         * <p>
         * The following are possible values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HonorMaxCapacity</code> - Amazon EC2 Auto Scaling cannot scale out capacity higher than the maximum
         * capacity. The maximum capacity is enforced as a hard limit.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IncreaseMaxCapacity</code> - Amazon EC2 Auto Scaling can scale out capacity higher than the maximum
         * capacity when the forecast capacity is close to or exceeds the maximum capacity. The upper limit is
         * determined by the forecasted capacity and the value for <code>MaxCapacityBuffer</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param maxCapacityBreachBehavior
         *        Defines the behavior that should be applied if the forecast capacity approaches or exceeds the maximum
         *        capacity of the Auto Scaling group. Defaults to <code>HonorMaxCapacity</code> if not specified.</p>
         *        <p>
         *        The following are possible values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>HonorMaxCapacity</code> - Amazon EC2 Auto Scaling cannot scale out capacity higher than the
         *        maximum capacity. The maximum capacity is enforced as a hard limit.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IncreaseMaxCapacity</code> - Amazon EC2 Auto Scaling can scale out capacity higher than the
         *        maximum capacity when the forecast capacity is close to or exceeds the maximum capacity. The upper
         *        limit is determined by the forecasted capacity and the value for <code>MaxCapacityBuffer</code>.
         *        </p>
         *        </li>
         * @see PredictiveScalingMaxCapacityBreachBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PredictiveScalingMaxCapacityBreachBehavior
         */
        Builder maxCapacityBreachBehavior(PredictiveScalingMaxCapacityBreachBehavior maxCapacityBreachBehavior);

        /**
         * <p>
         * The size of the capacity buffer to use when the forecast capacity is close to or exceeds the maximum
         * capacity. The value is specified as a percentage relative to the forecast capacity. For example, if the
         * buffer is 10, this means a 10 percent buffer, such that if the forecast capacity is 50, and the maximum
         * capacity is 40, then the effective maximum capacity is 55.
         * </p>
         * <p>
         * If set to 0, Amazon EC2 Auto Scaling may scale capacity higher than the maximum capacity to equal but not
         * exceed forecast capacity.
         * </p>
         * <p>
         * Required if the <code>MaxCapacityBreachBehavior</code> property is set to <code>IncreaseMaxCapacity</code>,
         * and cannot be used otherwise.
         * </p>
         * 
         * @param maxCapacityBuffer
         *        The size of the capacity buffer to use when the forecast capacity is close to or exceeds the maximum
         *        capacity. The value is specified as a percentage relative to the forecast capacity. For example, if
         *        the buffer is 10, this means a 10 percent buffer, such that if the forecast capacity is 50, and the
         *        maximum capacity is 40, then the effective maximum capacity is 55.</p>
         *        <p>
         *        If set to 0, Amazon EC2 Auto Scaling may scale capacity higher than the maximum capacity to equal but
         *        not exceed forecast capacity.
         *        </p>
         *        <p>
         *        Required if the <code>MaxCapacityBreachBehavior</code> property is set to
         *        <code>IncreaseMaxCapacity</code>, and cannot be used otherwise.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxCapacityBuffer(Integer maxCapacityBuffer);
    }

    static final class BuilderImpl implements Builder {
        private List<PredictiveScalingMetricSpecification> metricSpecifications = DefaultSdkAutoConstructList.getInstance();

        private String mode;

        private Integer schedulingBufferTime;

        private String maxCapacityBreachBehavior;

        private Integer maxCapacityBuffer;

        private BuilderImpl() {
        }

        private BuilderImpl(PredictiveScalingConfiguration model) {
            metricSpecifications(model.metricSpecifications);
            mode(model.mode);
            schedulingBufferTime(model.schedulingBufferTime);
            maxCapacityBreachBehavior(model.maxCapacityBreachBehavior);
            maxCapacityBuffer(model.maxCapacityBuffer);
        }

        public final List<PredictiveScalingMetricSpecification.Builder> getMetricSpecifications() {
            List<PredictiveScalingMetricSpecification.Builder> result = PredictiveScalingMetricSpecificationsCopier
                    .copyToBuilder(this.metricSpecifications);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMetricSpecifications(
                Collection<PredictiveScalingMetricSpecification.BuilderImpl> metricSpecifications) {
            this.metricSpecifications = PredictiveScalingMetricSpecificationsCopier.copyFromBuilder(metricSpecifications);
        }

        @Override
        public final Builder metricSpecifications(Collection<PredictiveScalingMetricSpecification> metricSpecifications) {
            this.metricSpecifications = PredictiveScalingMetricSpecificationsCopier.copy(metricSpecifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricSpecifications(PredictiveScalingMetricSpecification... metricSpecifications) {
            metricSpecifications(Arrays.asList(metricSpecifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricSpecifications(Consumer<PredictiveScalingMetricSpecification.Builder>... metricSpecifications) {
            metricSpecifications(Stream.of(metricSpecifications)
                    .map(c -> PredictiveScalingMetricSpecification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getMode() {
            return mode;
        }

        public final void setMode(String mode) {
            this.mode = mode;
        }

        @Override
        public final Builder mode(String mode) {
            this.mode = mode;
            return this;
        }

        @Override
        public final Builder mode(PredictiveScalingMode mode) {
            this.mode(mode == null ? null : mode.toString());
            return this;
        }

        public final Integer getSchedulingBufferTime() {
            return schedulingBufferTime;
        }

        public final void setSchedulingBufferTime(Integer schedulingBufferTime) {
            this.schedulingBufferTime = schedulingBufferTime;
        }

        @Override
        public final Builder schedulingBufferTime(Integer schedulingBufferTime) {
            this.schedulingBufferTime = schedulingBufferTime;
            return this;
        }

        public final String getMaxCapacityBreachBehavior() {
            return maxCapacityBreachBehavior;
        }

        public final void setMaxCapacityBreachBehavior(String maxCapacityBreachBehavior) {
            this.maxCapacityBreachBehavior = maxCapacityBreachBehavior;
        }

        @Override
        public final Builder maxCapacityBreachBehavior(String maxCapacityBreachBehavior) {
            this.maxCapacityBreachBehavior = maxCapacityBreachBehavior;
            return this;
        }

        @Override
        public final Builder maxCapacityBreachBehavior(PredictiveScalingMaxCapacityBreachBehavior maxCapacityBreachBehavior) {
            this.maxCapacityBreachBehavior(maxCapacityBreachBehavior == null ? null : maxCapacityBreachBehavior.toString());
            return this;
        }

        public final Integer getMaxCapacityBuffer() {
            return maxCapacityBuffer;
        }

        public final void setMaxCapacityBuffer(Integer maxCapacityBuffer) {
            this.maxCapacityBuffer = maxCapacityBuffer;
        }

        @Override
        public final Builder maxCapacityBuffer(Integer maxCapacityBuffer) {
            this.maxCapacityBuffer = maxCapacityBuffer;
            return this;
        }

        @Override
        public PredictiveScalingConfiguration build() {
            return new PredictiveScalingConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
