/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscaling.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the preferences for an instance refresh.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RefreshPreferences implements SdkPojo, Serializable,
        ToCopyableBuilder<RefreshPreferences.Builder, RefreshPreferences> {
    private static final SdkField<Integer> MIN_HEALTHY_PERCENTAGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MinHealthyPercentage").getter(getter(RefreshPreferences::minHealthyPercentage))
            .setter(setter(Builder::minHealthyPercentage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinHealthyPercentage").build())
            .build();

    private static final SdkField<Integer> INSTANCE_WARMUP_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("InstanceWarmup").getter(getter(RefreshPreferences::instanceWarmup))
            .setter(setter(Builder::instanceWarmup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceWarmup").build()).build();

    private static final SdkField<List<Integer>> CHECKPOINT_PERCENTAGES_FIELD = SdkField
            .<List<Integer>> builder(MarshallingType.LIST)
            .memberName("CheckpointPercentages")
            .getter(getter(RefreshPreferences::checkpointPercentages))
            .setter(setter(Builder::checkpointPercentages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CheckpointPercentages").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Integer> builder(MarshallingType.INTEGER)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> CHECKPOINT_DELAY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("CheckpointDelay").getter(getter(RefreshPreferences::checkpointDelay))
            .setter(setter(Builder::checkpointDelay))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CheckpointDelay").build()).build();

    private static final SdkField<Boolean> SKIP_MATCHING_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SkipMatching").getter(getter(RefreshPreferences::skipMatching)).setter(setter(Builder::skipMatching))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SkipMatching").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MIN_HEALTHY_PERCENTAGE_FIELD,
            INSTANCE_WARMUP_FIELD, CHECKPOINT_PERCENTAGES_FIELD, CHECKPOINT_DELAY_FIELD, SKIP_MATCHING_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer minHealthyPercentage;

    private final Integer instanceWarmup;

    private final List<Integer> checkpointPercentages;

    private final Integer checkpointDelay;

    private final Boolean skipMatching;

    private RefreshPreferences(BuilderImpl builder) {
        this.minHealthyPercentage = builder.minHealthyPercentage;
        this.instanceWarmup = builder.instanceWarmup;
        this.checkpointPercentages = builder.checkpointPercentages;
        this.checkpointDelay = builder.checkpointDelay;
        this.skipMatching = builder.skipMatching;
    }

    /**
     * <p>
     * The amount of capacity in the Auto Scaling group that must remain healthy during an instance refresh to allow the
     * operation to continue. The value is expressed as a percentage of the desired capacity of the Auto Scaling group
     * (rounded up to the nearest integer). The default is <code>90</code>.
     * </p>
     * <p>
     * Setting the minimum healthy percentage to 100 percent limits the rate of replacement to one instance at a time.
     * In contrast, setting it to 0 percent has the effect of replacing all instances at the same time.
     * </p>
     * 
     * @return The amount of capacity in the Auto Scaling group that must remain healthy during an instance refresh to
     *         allow the operation to continue. The value is expressed as a percentage of the desired capacity of the
     *         Auto Scaling group (rounded up to the nearest integer). The default is <code>90</code>.</p>
     *         <p>
     *         Setting the minimum healthy percentage to 100 percent limits the rate of replacement to one instance at a
     *         time. In contrast, setting it to 0 percent has the effect of replacing all instances at the same time.
     */
    public final Integer minHealthyPercentage() {
        return minHealthyPercentage;
    }

    /**
     * <p>
     * The number of seconds until a newly launched instance is configured and ready to use. During this time, Amazon
     * EC2 Auto Scaling does not immediately move on to the next replacement. The default is to use the value for the
     * health check grace period defined for the group.
     * </p>
     * 
     * @return The number of seconds until a newly launched instance is configured and ready to use. During this time,
     *         Amazon EC2 Auto Scaling does not immediately move on to the next replacement. The default is to use the
     *         value for the health check grace period defined for the group.
     */
    public final Integer instanceWarmup() {
        return instanceWarmup;
    }

    /**
     * For responses, this returns true if the service returned a value for the CheckpointPercentages property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasCheckpointPercentages() {
        return checkpointPercentages != null && !(checkpointPercentages instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Threshold values for each checkpoint in ascending order. Each number must be unique. To replace all instances in
     * the Auto Scaling group, the last number in the array must be <code>100</code>.
     * </p>
     * <p>
     * For usage examples, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-adding-checkpoints-instance-refresh.html">Adding
     * checkpoints to an instance refresh</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCheckpointPercentages} method.
     * </p>
     * 
     * @return Threshold values for each checkpoint in ascending order. Each number must be unique. To replace all
     *         instances in the Auto Scaling group, the last number in the array must be <code>100</code>.</p>
     *         <p>
     *         For usage examples, see <a href=
     *         "https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-adding-checkpoints-instance-refresh.html"
     *         >Adding checkpoints to an instance refresh</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     */
    public final List<Integer> checkpointPercentages() {
        return checkpointPercentages;
    }

    /**
     * <p>
     * The amount of time, in seconds, to wait after a checkpoint before continuing. This property is optional, but if
     * you specify a value for it, you must also specify a value for <code>CheckpointPercentages</code>. If you specify
     * a value for <code>CheckpointPercentages</code> and not for <code>CheckpointDelay</code>, the
     * <code>CheckpointDelay</code> defaults to <code>3600</code> (1 hour).
     * </p>
     * 
     * @return The amount of time, in seconds, to wait after a checkpoint before continuing. This property is optional,
     *         but if you specify a value for it, you must also specify a value for <code>CheckpointPercentages</code>.
     *         If you specify a value for <code>CheckpointPercentages</code> and not for <code>CheckpointDelay</code>,
     *         the <code>CheckpointDelay</code> defaults to <code>3600</code> (1 hour).
     */
    public final Integer checkpointDelay() {
        return checkpointDelay;
    }

    /**
     * <p>
     * A boolean value that indicates whether skip matching is enabled. If true, then Amazon EC2 Auto Scaling skips
     * replacing instances that match the desired configuration. If no desired configuration is specified, then it skips
     * replacing instances that have the same configuration that is already set on the group. The default is
     * <code>false</code>.
     * </p>
     * 
     * @return A boolean value that indicates whether skip matching is enabled. If true, then Amazon EC2 Auto Scaling
     *         skips replacing instances that match the desired configuration. If no desired configuration is specified,
     *         then it skips replacing instances that have the same configuration that is already set on the group. The
     *         default is <code>false</code>.
     */
    public final Boolean skipMatching() {
        return skipMatching;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(minHealthyPercentage());
        hashCode = 31 * hashCode + Objects.hashCode(instanceWarmup());
        hashCode = 31 * hashCode + Objects.hashCode(hasCheckpointPercentages() ? checkpointPercentages() : null);
        hashCode = 31 * hashCode + Objects.hashCode(checkpointDelay());
        hashCode = 31 * hashCode + Objects.hashCode(skipMatching());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RefreshPreferences)) {
            return false;
        }
        RefreshPreferences other = (RefreshPreferences) obj;
        return Objects.equals(minHealthyPercentage(), other.minHealthyPercentage())
                && Objects.equals(instanceWarmup(), other.instanceWarmup())
                && hasCheckpointPercentages() == other.hasCheckpointPercentages()
                && Objects.equals(checkpointPercentages(), other.checkpointPercentages())
                && Objects.equals(checkpointDelay(), other.checkpointDelay())
                && Objects.equals(skipMatching(), other.skipMatching());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RefreshPreferences").add("MinHealthyPercentage", minHealthyPercentage())
                .add("InstanceWarmup", instanceWarmup())
                .add("CheckpointPercentages", hasCheckpointPercentages() ? checkpointPercentages() : null)
                .add("CheckpointDelay", checkpointDelay()).add("SkipMatching", skipMatching()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MinHealthyPercentage":
            return Optional.ofNullable(clazz.cast(minHealthyPercentage()));
        case "InstanceWarmup":
            return Optional.ofNullable(clazz.cast(instanceWarmup()));
        case "CheckpointPercentages":
            return Optional.ofNullable(clazz.cast(checkpointPercentages()));
        case "CheckpointDelay":
            return Optional.ofNullable(clazz.cast(checkpointDelay()));
        case "SkipMatching":
            return Optional.ofNullable(clazz.cast(skipMatching()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RefreshPreferences, T> g) {
        return obj -> g.apply((RefreshPreferences) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RefreshPreferences> {
        /**
         * <p>
         * The amount of capacity in the Auto Scaling group that must remain healthy during an instance refresh to allow
         * the operation to continue. The value is expressed as a percentage of the desired capacity of the Auto Scaling
         * group (rounded up to the nearest integer). The default is <code>90</code>.
         * </p>
         * <p>
         * Setting the minimum healthy percentage to 100 percent limits the rate of replacement to one instance at a
         * time. In contrast, setting it to 0 percent has the effect of replacing all instances at the same time.
         * </p>
         * 
         * @param minHealthyPercentage
         *        The amount of capacity in the Auto Scaling group that must remain healthy during an instance refresh
         *        to allow the operation to continue. The value is expressed as a percentage of the desired capacity of
         *        the Auto Scaling group (rounded up to the nearest integer). The default is <code>90</code>.</p>
         *        <p>
         *        Setting the minimum healthy percentage to 100 percent limits the rate of replacement to one instance
         *        at a time. In contrast, setting it to 0 percent has the effect of replacing all instances at the same
         *        time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minHealthyPercentage(Integer minHealthyPercentage);

        /**
         * <p>
         * The number of seconds until a newly launched instance is configured and ready to use. During this time,
         * Amazon EC2 Auto Scaling does not immediately move on to the next replacement. The default is to use the value
         * for the health check grace period defined for the group.
         * </p>
         * 
         * @param instanceWarmup
         *        The number of seconds until a newly launched instance is configured and ready to use. During this
         *        time, Amazon EC2 Auto Scaling does not immediately move on to the next replacement. The default is to
         *        use the value for the health check grace period defined for the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceWarmup(Integer instanceWarmup);

        /**
         * <p>
         * Threshold values for each checkpoint in ascending order. Each number must be unique. To replace all instances
         * in the Auto Scaling group, the last number in the array must be <code>100</code>.
         * </p>
         * <p>
         * For usage examples, see <a
         * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-adding-checkpoints-instance-refresh.html"
         * >Adding checkpoints to an instance refresh</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
         * </p>
         * 
         * @param checkpointPercentages
         *        Threshold values for each checkpoint in ascending order. Each number must be unique. To replace all
         *        instances in the Auto Scaling group, the last number in the array must be <code>100</code>.</p>
         *        <p>
         *        For usage examples, see <a href=
         *        "https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-adding-checkpoints-instance-refresh.html"
         *        >Adding checkpoints to an instance refresh</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder checkpointPercentages(Collection<Integer> checkpointPercentages);

        /**
         * <p>
         * Threshold values for each checkpoint in ascending order. Each number must be unique. To replace all instances
         * in the Auto Scaling group, the last number in the array must be <code>100</code>.
         * </p>
         * <p>
         * For usage examples, see <a
         * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-adding-checkpoints-instance-refresh.html"
         * >Adding checkpoints to an instance refresh</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
         * </p>
         * 
         * @param checkpointPercentages
         *        Threshold values for each checkpoint in ascending order. Each number must be unique. To replace all
         *        instances in the Auto Scaling group, the last number in the array must be <code>100</code>.</p>
         *        <p>
         *        For usage examples, see <a href=
         *        "https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-adding-checkpoints-instance-refresh.html"
         *        >Adding checkpoints to an instance refresh</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder checkpointPercentages(Integer... checkpointPercentages);

        /**
         * <p>
         * The amount of time, in seconds, to wait after a checkpoint before continuing. This property is optional, but
         * if you specify a value for it, you must also specify a value for <code>CheckpointPercentages</code>. If you
         * specify a value for <code>CheckpointPercentages</code> and not for <code>CheckpointDelay</code>, the
         * <code>CheckpointDelay</code> defaults to <code>3600</code> (1 hour).
         * </p>
         * 
         * @param checkpointDelay
         *        The amount of time, in seconds, to wait after a checkpoint before continuing. This property is
         *        optional, but if you specify a value for it, you must also specify a value for
         *        <code>CheckpointPercentages</code>. If you specify a value for <code>CheckpointPercentages</code> and
         *        not for <code>CheckpointDelay</code>, the <code>CheckpointDelay</code> defaults to <code>3600</code>
         *        (1 hour).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder checkpointDelay(Integer checkpointDelay);

        /**
         * <p>
         * A boolean value that indicates whether skip matching is enabled. If true, then Amazon EC2 Auto Scaling skips
         * replacing instances that match the desired configuration. If no desired configuration is specified, then it
         * skips replacing instances that have the same configuration that is already set on the group. The default is
         * <code>false</code>.
         * </p>
         * 
         * @param skipMatching
         *        A boolean value that indicates whether skip matching is enabled. If true, then Amazon EC2 Auto Scaling
         *        skips replacing instances that match the desired configuration. If no desired configuration is
         *        specified, then it skips replacing instances that have the same configuration that is already set on
         *        the group. The default is <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder skipMatching(Boolean skipMatching);
    }

    static final class BuilderImpl implements Builder {
        private Integer minHealthyPercentage;

        private Integer instanceWarmup;

        private List<Integer> checkpointPercentages = DefaultSdkAutoConstructList.getInstance();

        private Integer checkpointDelay;

        private Boolean skipMatching;

        private BuilderImpl() {
        }

        private BuilderImpl(RefreshPreferences model) {
            minHealthyPercentage(model.minHealthyPercentage);
            instanceWarmup(model.instanceWarmup);
            checkpointPercentages(model.checkpointPercentages);
            checkpointDelay(model.checkpointDelay);
            skipMatching(model.skipMatching);
        }

        public final Integer getMinHealthyPercentage() {
            return minHealthyPercentage;
        }

        public final void setMinHealthyPercentage(Integer minHealthyPercentage) {
            this.minHealthyPercentage = minHealthyPercentage;
        }

        @Override
        public final Builder minHealthyPercentage(Integer minHealthyPercentage) {
            this.minHealthyPercentage = minHealthyPercentage;
            return this;
        }

        public final Integer getInstanceWarmup() {
            return instanceWarmup;
        }

        public final void setInstanceWarmup(Integer instanceWarmup) {
            this.instanceWarmup = instanceWarmup;
        }

        @Override
        public final Builder instanceWarmup(Integer instanceWarmup) {
            this.instanceWarmup = instanceWarmup;
            return this;
        }

        public final Collection<Integer> getCheckpointPercentages() {
            if (checkpointPercentages instanceof SdkAutoConstructList) {
                return null;
            }
            return checkpointPercentages;
        }

        public final void setCheckpointPercentages(Collection<Integer> checkpointPercentages) {
            this.checkpointPercentages = CheckpointPercentagesCopier.copy(checkpointPercentages);
        }

        @Override
        public final Builder checkpointPercentages(Collection<Integer> checkpointPercentages) {
            this.checkpointPercentages = CheckpointPercentagesCopier.copy(checkpointPercentages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder checkpointPercentages(Integer... checkpointPercentages) {
            checkpointPercentages(Arrays.asList(checkpointPercentages));
            return this;
        }

        public final Integer getCheckpointDelay() {
            return checkpointDelay;
        }

        public final void setCheckpointDelay(Integer checkpointDelay) {
            this.checkpointDelay = checkpointDelay;
        }

        @Override
        public final Builder checkpointDelay(Integer checkpointDelay) {
            this.checkpointDelay = checkpointDelay;
            return this;
        }

        public final Boolean getSkipMatching() {
            return skipMatching;
        }

        public final void setSkipMatching(Boolean skipMatching) {
            this.skipMatching = skipMatching;
        }

        @Override
        public final Builder skipMatching(Boolean skipMatching) {
            this.skipMatching = skipMatching;
            return this;
        }

        @Override
        public RefreshPreferences build() {
            return new RefreshPreferences(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
