/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscaling.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a filter that is used to return a more specific list of results from a describe operation.
 * </p>
 * <p>
 * If you specify multiple filters, the filters are automatically logically joined with an <code>AND</code>, and the
 * request returns only the results that match all of the specified filters.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/autoscaling-tagging.html">Tagging Auto Scaling groups and
 * instances</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Filter implements SdkPojo, Serializable, ToCopyableBuilder<Filter.Builder, Filter> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Filter::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<String>> VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Values")
            .getter(getter(Filter::values))
            .setter(setter(Builder::values))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Values").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, VALUES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final List<String> values;

    private Filter(BuilderImpl builder) {
        this.name = builder.name;
        this.values = builder.values;
    }

    /**
     * <p>
     * The name of the filter.
     * </p>
     * <p>
     * The valid values for <code>Name</code> depend on which API operation you're using with the filter
     * (<a>DescribeAutoScalingGroups</a> or <a>DescribeTags</a>).
     * </p>
     * <p>
     * <b>DescribeAutoScalingGroups</b>
     * </p>
     * <p>
     * Valid values for <code>Name</code> include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>tag-key</code> - Accepts tag keys. The results only include information about the Auto Scaling groups
     * associated with these tag keys.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>tag-value</code> - Accepts tag values. The results only include information about the Auto Scaling groups
     * associated with these tag values.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>tag:&lt;key&gt;</code> - Accepts the key/value combination of the tag. Use the tag key in the filter name
     * and the tag value as the filter value. The results only include information about the Auto Scaling groups
     * associated with the specified key/value combination.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>DescribeTags</b>
     * </p>
     * <p>
     * Valid values for <code>Name</code> include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>auto-scaling-group</code> - Accepts the names of Auto Scaling groups. The results only include information
     * about the tags associated with these Auto Scaling groups.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>key</code> - Accepts tag keys. The results only include information about the tags associated with these
     * tag keys.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>value</code> - Accepts tag values. The results only include information about the tags associated with
     * these tag values.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>propagate-at-launch</code> - Accepts a Boolean value, which specifies whether tags propagate to instances
     * at launch. The results only include information about the tags associated with the specified Boolean value.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The name of the filter.</p>
     *         <p>
     *         The valid values for <code>Name</code> depend on which API operation you're using with the filter
     *         (<a>DescribeAutoScalingGroups</a> or <a>DescribeTags</a>).
     *         </p>
     *         <p>
     *         <b>DescribeAutoScalingGroups</b>
     *         </p>
     *         <p>
     *         Valid values for <code>Name</code> include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>tag-key</code> - Accepts tag keys. The results only include information about the Auto Scaling
     *         groups associated with these tag keys.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>tag-value</code> - Accepts tag values. The results only include information about the Auto Scaling
     *         groups associated with these tag values.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>tag:&lt;key&gt;</code> - Accepts the key/value combination of the tag. Use the tag key in the
     *         filter name and the tag value as the filter value. The results only include information about the Auto
     *         Scaling groups associated with the specified key/value combination.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>DescribeTags</b>
     *         </p>
     *         <p>
     *         Valid values for <code>Name</code> include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>auto-scaling-group</code> - Accepts the names of Auto Scaling groups. The results only include
     *         information about the tags associated with these Auto Scaling groups.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>key</code> - Accepts tag keys. The results only include information about the tags associated with
     *         these tag keys.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>value</code> - Accepts tag values. The results only include information about the tags associated
     *         with these tag values.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>propagate-at-launch</code> - Accepts a Boolean value, which specifies whether tags propagate to
     *         instances at launch. The results only include information about the tags associated with the specified
     *         Boolean value.
     *         </p>
     *         </li>
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Values property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasValues() {
        return values != null && !(values instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more filter values. Filter values are case-sensitive.
     * </p>
     * <p>
     * If you specify multiple values for a filter, the values are automatically logically joined with an
     * <code>OR</code>, and the request returns all results that match any of the specified values. For example, specify
     * "tag:environment" for the filter name and "production,development" for the filter values to find Auto Scaling
     * groups with the tag "environment=production" or "environment=development".
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasValues} method.
     * </p>
     * 
     * @return One or more filter values. Filter values are case-sensitive. </p>
     *         <p>
     *         If you specify multiple values for a filter, the values are automatically logically joined with an
     *         <code>OR</code>, and the request returns all results that match any of the specified values. For example,
     *         specify "tag:environment" for the filter name and "production,development" for the filter values to find
     *         Auto Scaling groups with the tag "environment=production" or "environment=development".
     */
    public final List<String> values() {
        return values;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasValues() ? values() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Filter)) {
            return false;
        }
        Filter other = (Filter) obj;
        return Objects.equals(name(), other.name()) && hasValues() == other.hasValues()
                && Objects.equals(values(), other.values());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Filter").add("Name", name()).add("Values", hasValues() ? values() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Values":
            return Optional.ofNullable(clazz.cast(values()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Filter, T> g) {
        return obj -> g.apply((Filter) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Filter> {
        /**
         * <p>
         * The name of the filter.
         * </p>
         * <p>
         * The valid values for <code>Name</code> depend on which API operation you're using with the filter
         * (<a>DescribeAutoScalingGroups</a> or <a>DescribeTags</a>).
         * </p>
         * <p>
         * <b>DescribeAutoScalingGroups</b>
         * </p>
         * <p>
         * Valid values for <code>Name</code> include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>tag-key</code> - Accepts tag keys. The results only include information about the Auto Scaling groups
         * associated with these tag keys.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>tag-value</code> - Accepts tag values. The results only include information about the Auto Scaling
         * groups associated with these tag values.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>tag:&lt;key&gt;</code> - Accepts the key/value combination of the tag. Use the tag key in the filter
         * name and the tag value as the filter value. The results only include information about the Auto Scaling
         * groups associated with the specified key/value combination.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>DescribeTags</b>
         * </p>
         * <p>
         * Valid values for <code>Name</code> include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>auto-scaling-group</code> - Accepts the names of Auto Scaling groups. The results only include
         * information about the tags associated with these Auto Scaling groups.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>key</code> - Accepts tag keys. The results only include information about the tags associated with
         * these tag keys.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>value</code> - Accepts tag values. The results only include information about the tags associated with
         * these tag values.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>propagate-at-launch</code> - Accepts a Boolean value, which specifies whether tags propagate to
         * instances at launch. The results only include information about the tags associated with the specified
         * Boolean value.
         * </p>
         * </li>
         * </ul>
         * 
         * @param name
         *        The name of the filter.</p>
         *        <p>
         *        The valid values for <code>Name</code> depend on which API operation you're using with the filter
         *        (<a>DescribeAutoScalingGroups</a> or <a>DescribeTags</a>).
         *        </p>
         *        <p>
         *        <b>DescribeAutoScalingGroups</b>
         *        </p>
         *        <p>
         *        Valid values for <code>Name</code> include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>tag-key</code> - Accepts tag keys. The results only include information about the Auto Scaling
         *        groups associated with these tag keys.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>tag-value</code> - Accepts tag values. The results only include information about the Auto
         *        Scaling groups associated with these tag values.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>tag:&lt;key&gt;</code> - Accepts the key/value combination of the tag. Use the tag key in the
         *        filter name and the tag value as the filter value. The results only include information about the Auto
         *        Scaling groups associated with the specified key/value combination.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>DescribeTags</b>
         *        </p>
         *        <p>
         *        Valid values for <code>Name</code> include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>auto-scaling-group</code> - Accepts the names of Auto Scaling groups. The results only include
         *        information about the tags associated with these Auto Scaling groups.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>key</code> - Accepts tag keys. The results only include information about the tags associated
         *        with these tag keys.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>value</code> - Accepts tag values. The results only include information about the tags
         *        associated with these tag values.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>propagate-at-launch</code> - Accepts a Boolean value, which specifies whether tags propagate to
         *        instances at launch. The results only include information about the tags associated with the specified
         *        Boolean value.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * One or more filter values. Filter values are case-sensitive.
         * </p>
         * <p>
         * If you specify multiple values for a filter, the values are automatically logically joined with an
         * <code>OR</code>, and the request returns all results that match any of the specified values. For example,
         * specify "tag:environment" for the filter name and "production,development" for the filter values to find Auto
         * Scaling groups with the tag "environment=production" or "environment=development".
         * </p>
         * 
         * @param values
         *        One or more filter values. Filter values are case-sensitive. </p>
         *        <p>
         *        If you specify multiple values for a filter, the values are automatically logically joined with an
         *        <code>OR</code>, and the request returns all results that match any of the specified values. For
         *        example, specify "tag:environment" for the filter name and "production,development" for the filter
         *        values to find Auto Scaling groups with the tag "environment=production" or "environment=development".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Collection<String> values);

        /**
         * <p>
         * One or more filter values. Filter values are case-sensitive.
         * </p>
         * <p>
         * If you specify multiple values for a filter, the values are automatically logically joined with an
         * <code>OR</code>, and the request returns all results that match any of the specified values. For example,
         * specify "tag:environment" for the filter name and "production,development" for the filter values to find Auto
         * Scaling groups with the tag "environment=production" or "environment=development".
         * </p>
         * 
         * @param values
         *        One or more filter values. Filter values are case-sensitive. </p>
         *        <p>
         *        If you specify multiple values for a filter, the values are automatically logically joined with an
         *        <code>OR</code>, and the request returns all results that match any of the specified values. For
         *        example, specify "tag:environment" for the filter name and "production,development" for the filter
         *        values to find Auto Scaling groups with the tag "environment=production" or "environment=development".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(String... values);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private List<String> values = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Filter model) {
            name(model.name);
            values(model.values);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Collection<String> getValues() {
            if (values instanceof SdkAutoConstructList) {
                return null;
            }
            return values;
        }

        public final void setValues(Collection<String> values) {
            this.values = ValuesCopier.copy(values);
        }

        @Override
        @Transient
        public final Builder values(Collection<String> values) {
            this.values = ValuesCopier.copy(values);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder values(String... values) {
            values(Arrays.asList(values));
            return this;
        }

        @Override
        public Filter build() {
            return new Filter(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
