/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscaling;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.query.AwsQueryProtocolFactory;
import software.amazon.awssdk.services.autoscaling.internal.AutoScalingServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.autoscaling.model.ActiveInstanceRefreshNotFoundException;
import software.amazon.awssdk.services.autoscaling.model.AlreadyExistsException;
import software.amazon.awssdk.services.autoscaling.model.AttachInstancesRequest;
import software.amazon.awssdk.services.autoscaling.model.AttachInstancesResponse;
import software.amazon.awssdk.services.autoscaling.model.AttachLoadBalancerTargetGroupsRequest;
import software.amazon.awssdk.services.autoscaling.model.AttachLoadBalancerTargetGroupsResponse;
import software.amazon.awssdk.services.autoscaling.model.AttachLoadBalancersRequest;
import software.amazon.awssdk.services.autoscaling.model.AttachLoadBalancersResponse;
import software.amazon.awssdk.services.autoscaling.model.AttachTrafficSourcesRequest;
import software.amazon.awssdk.services.autoscaling.model.AttachTrafficSourcesResponse;
import software.amazon.awssdk.services.autoscaling.model.AutoScalingException;
import software.amazon.awssdk.services.autoscaling.model.BatchDeleteScheduledActionRequest;
import software.amazon.awssdk.services.autoscaling.model.BatchDeleteScheduledActionResponse;
import software.amazon.awssdk.services.autoscaling.model.BatchPutScheduledUpdateGroupActionRequest;
import software.amazon.awssdk.services.autoscaling.model.BatchPutScheduledUpdateGroupActionResponse;
import software.amazon.awssdk.services.autoscaling.model.CancelInstanceRefreshRequest;
import software.amazon.awssdk.services.autoscaling.model.CancelInstanceRefreshResponse;
import software.amazon.awssdk.services.autoscaling.model.CompleteLifecycleActionRequest;
import software.amazon.awssdk.services.autoscaling.model.CompleteLifecycleActionResponse;
import software.amazon.awssdk.services.autoscaling.model.CreateAutoScalingGroupRequest;
import software.amazon.awssdk.services.autoscaling.model.CreateAutoScalingGroupResponse;
import software.amazon.awssdk.services.autoscaling.model.CreateLaunchConfigurationRequest;
import software.amazon.awssdk.services.autoscaling.model.CreateLaunchConfigurationResponse;
import software.amazon.awssdk.services.autoscaling.model.CreateOrUpdateTagsRequest;
import software.amazon.awssdk.services.autoscaling.model.CreateOrUpdateTagsResponse;
import software.amazon.awssdk.services.autoscaling.model.DeleteAutoScalingGroupRequest;
import software.amazon.awssdk.services.autoscaling.model.DeleteAutoScalingGroupResponse;
import software.amazon.awssdk.services.autoscaling.model.DeleteLaunchConfigurationRequest;
import software.amazon.awssdk.services.autoscaling.model.DeleteLaunchConfigurationResponse;
import software.amazon.awssdk.services.autoscaling.model.DeleteLifecycleHookRequest;
import software.amazon.awssdk.services.autoscaling.model.DeleteLifecycleHookResponse;
import software.amazon.awssdk.services.autoscaling.model.DeleteNotificationConfigurationRequest;
import software.amazon.awssdk.services.autoscaling.model.DeleteNotificationConfigurationResponse;
import software.amazon.awssdk.services.autoscaling.model.DeletePolicyRequest;
import software.amazon.awssdk.services.autoscaling.model.DeletePolicyResponse;
import software.amazon.awssdk.services.autoscaling.model.DeleteScheduledActionRequest;
import software.amazon.awssdk.services.autoscaling.model.DeleteScheduledActionResponse;
import software.amazon.awssdk.services.autoscaling.model.DeleteTagsRequest;
import software.amazon.awssdk.services.autoscaling.model.DeleteTagsResponse;
import software.amazon.awssdk.services.autoscaling.model.DeleteWarmPoolRequest;
import software.amazon.awssdk.services.autoscaling.model.DeleteWarmPoolResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeAccountLimitsRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeAccountLimitsResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeAdjustmentTypesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeAdjustmentTypesResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingGroupsRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingGroupsResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingInstancesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingInstancesResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingNotificationTypesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingNotificationTypesResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeInstanceRefreshesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeInstanceRefreshesResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeLaunchConfigurationsRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeLaunchConfigurationsResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeLifecycleHookTypesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeLifecycleHookTypesResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeLifecycleHooksRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeLifecycleHooksResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeLoadBalancerTargetGroupsRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeLoadBalancerTargetGroupsResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeLoadBalancersRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeLoadBalancersResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeMetricCollectionTypesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeMetricCollectionTypesResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeNotificationConfigurationsRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeNotificationConfigurationsResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribePoliciesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribePoliciesResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeScalingActivitiesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeScalingActivitiesResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeScalingProcessTypesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeScalingProcessTypesResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeScheduledActionsRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeScheduledActionsResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeTagsRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeTagsResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeTerminationPolicyTypesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeTerminationPolicyTypesResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeTrafficSourcesRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeTrafficSourcesResponse;
import software.amazon.awssdk.services.autoscaling.model.DescribeWarmPoolRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeWarmPoolResponse;
import software.amazon.awssdk.services.autoscaling.model.DetachInstancesRequest;
import software.amazon.awssdk.services.autoscaling.model.DetachInstancesResponse;
import software.amazon.awssdk.services.autoscaling.model.DetachLoadBalancerTargetGroupsRequest;
import software.amazon.awssdk.services.autoscaling.model.DetachLoadBalancerTargetGroupsResponse;
import software.amazon.awssdk.services.autoscaling.model.DetachLoadBalancersRequest;
import software.amazon.awssdk.services.autoscaling.model.DetachLoadBalancersResponse;
import software.amazon.awssdk.services.autoscaling.model.DetachTrafficSourcesRequest;
import software.amazon.awssdk.services.autoscaling.model.DetachTrafficSourcesResponse;
import software.amazon.awssdk.services.autoscaling.model.DisableMetricsCollectionRequest;
import software.amazon.awssdk.services.autoscaling.model.DisableMetricsCollectionResponse;
import software.amazon.awssdk.services.autoscaling.model.EnableMetricsCollectionRequest;
import software.amazon.awssdk.services.autoscaling.model.EnableMetricsCollectionResponse;
import software.amazon.awssdk.services.autoscaling.model.EnterStandbyRequest;
import software.amazon.awssdk.services.autoscaling.model.EnterStandbyResponse;
import software.amazon.awssdk.services.autoscaling.model.ExecutePolicyRequest;
import software.amazon.awssdk.services.autoscaling.model.ExecutePolicyResponse;
import software.amazon.awssdk.services.autoscaling.model.ExitStandbyRequest;
import software.amazon.awssdk.services.autoscaling.model.ExitStandbyResponse;
import software.amazon.awssdk.services.autoscaling.model.GetPredictiveScalingForecastRequest;
import software.amazon.awssdk.services.autoscaling.model.GetPredictiveScalingForecastResponse;
import software.amazon.awssdk.services.autoscaling.model.InstanceRefreshInProgressException;
import software.amazon.awssdk.services.autoscaling.model.InvalidNextTokenException;
import software.amazon.awssdk.services.autoscaling.model.IrreversibleInstanceRefreshException;
import software.amazon.awssdk.services.autoscaling.model.LimitExceededException;
import software.amazon.awssdk.services.autoscaling.model.PutLifecycleHookRequest;
import software.amazon.awssdk.services.autoscaling.model.PutLifecycleHookResponse;
import software.amazon.awssdk.services.autoscaling.model.PutNotificationConfigurationRequest;
import software.amazon.awssdk.services.autoscaling.model.PutNotificationConfigurationResponse;
import software.amazon.awssdk.services.autoscaling.model.PutScalingPolicyRequest;
import software.amazon.awssdk.services.autoscaling.model.PutScalingPolicyResponse;
import software.amazon.awssdk.services.autoscaling.model.PutScheduledUpdateGroupActionRequest;
import software.amazon.awssdk.services.autoscaling.model.PutScheduledUpdateGroupActionResponse;
import software.amazon.awssdk.services.autoscaling.model.PutWarmPoolRequest;
import software.amazon.awssdk.services.autoscaling.model.PutWarmPoolResponse;
import software.amazon.awssdk.services.autoscaling.model.RecordLifecycleActionHeartbeatRequest;
import software.amazon.awssdk.services.autoscaling.model.RecordLifecycleActionHeartbeatResponse;
import software.amazon.awssdk.services.autoscaling.model.ResourceContentionException;
import software.amazon.awssdk.services.autoscaling.model.ResourceInUseException;
import software.amazon.awssdk.services.autoscaling.model.ResumeProcessesRequest;
import software.amazon.awssdk.services.autoscaling.model.ResumeProcessesResponse;
import software.amazon.awssdk.services.autoscaling.model.RollbackInstanceRefreshRequest;
import software.amazon.awssdk.services.autoscaling.model.RollbackInstanceRefreshResponse;
import software.amazon.awssdk.services.autoscaling.model.ScalingActivityInProgressException;
import software.amazon.awssdk.services.autoscaling.model.ServiceLinkedRoleFailureException;
import software.amazon.awssdk.services.autoscaling.model.SetDesiredCapacityRequest;
import software.amazon.awssdk.services.autoscaling.model.SetDesiredCapacityResponse;
import software.amazon.awssdk.services.autoscaling.model.SetInstanceHealthRequest;
import software.amazon.awssdk.services.autoscaling.model.SetInstanceHealthResponse;
import software.amazon.awssdk.services.autoscaling.model.SetInstanceProtectionRequest;
import software.amazon.awssdk.services.autoscaling.model.SetInstanceProtectionResponse;
import software.amazon.awssdk.services.autoscaling.model.StartInstanceRefreshRequest;
import software.amazon.awssdk.services.autoscaling.model.StartInstanceRefreshResponse;
import software.amazon.awssdk.services.autoscaling.model.SuspendProcessesRequest;
import software.amazon.awssdk.services.autoscaling.model.SuspendProcessesResponse;
import software.amazon.awssdk.services.autoscaling.model.TerminateInstanceInAutoScalingGroupRequest;
import software.amazon.awssdk.services.autoscaling.model.TerminateInstanceInAutoScalingGroupResponse;
import software.amazon.awssdk.services.autoscaling.model.UpdateAutoScalingGroupRequest;
import software.amazon.awssdk.services.autoscaling.model.UpdateAutoScalingGroupResponse;
import software.amazon.awssdk.services.autoscaling.transform.AttachInstancesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.AttachLoadBalancerTargetGroupsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.AttachLoadBalancersRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.AttachTrafficSourcesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.BatchDeleteScheduledActionRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.BatchPutScheduledUpdateGroupActionRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.CancelInstanceRefreshRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.CompleteLifecycleActionRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.CreateAutoScalingGroupRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.CreateLaunchConfigurationRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.CreateOrUpdateTagsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DeleteAutoScalingGroupRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DeleteLaunchConfigurationRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DeleteLifecycleHookRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DeleteNotificationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DeletePolicyRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DeleteScheduledActionRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DeleteTagsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DeleteWarmPoolRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeAccountLimitsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeAdjustmentTypesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeAutoScalingGroupsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeAutoScalingInstancesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeAutoScalingNotificationTypesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeInstanceRefreshesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeLaunchConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeLifecycleHookTypesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeLifecycleHooksRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeLoadBalancerTargetGroupsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeLoadBalancersRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeMetricCollectionTypesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeNotificationConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribePoliciesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeScalingActivitiesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeScalingProcessTypesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeScheduledActionsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeTagsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeTerminationPolicyTypesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeTrafficSourcesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DescribeWarmPoolRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DetachInstancesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DetachLoadBalancerTargetGroupsRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DetachLoadBalancersRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DetachTrafficSourcesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.DisableMetricsCollectionRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.EnableMetricsCollectionRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.EnterStandbyRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.ExecutePolicyRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.ExitStandbyRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.GetPredictiveScalingForecastRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.PutLifecycleHookRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.PutNotificationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.PutScalingPolicyRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.PutScheduledUpdateGroupActionRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.PutWarmPoolRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.RecordLifecycleActionHeartbeatRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.ResumeProcessesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.RollbackInstanceRefreshRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.SetDesiredCapacityRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.SetInstanceHealthRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.SetInstanceProtectionRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.StartInstanceRefreshRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.SuspendProcessesRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.TerminateInstanceInAutoScalingGroupRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.transform.UpdateAutoScalingGroupRequestMarshaller;
import software.amazon.awssdk.services.autoscaling.waiters.AutoScalingAsyncWaiter;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AutoScalingAsyncClient}.
 *
 * @see AutoScalingAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAutoScalingAsyncClient implements AutoScalingAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAutoScalingAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsQueryProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final AutoScalingServiceClientConfiguration serviceClientConfiguration;

    private final ScheduledExecutorService executorService;

    protected DefaultAutoScalingAsyncClient(AutoScalingServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init();
        this.executorService = clientConfiguration.option(SdkClientOption.SCHEDULED_EXECUTOR_SERVICE);
    }

    /**
     * <p>
     * Attaches one or more EC2 instances to the specified Auto Scaling group.
     * </p>
     * <p>
     * When you attach instances, Amazon EC2 Auto Scaling increases the desired capacity of the group by the number of
     * instances being attached. If the number of instances being attached plus the desired capacity of the group
     * exceeds the maximum size of the group, the operation fails.
     * </p>
     * <p>
     * If there is a Classic Load Balancer attached to your Auto Scaling group, the instances are also registered with
     * the load balancer. If there are target groups attached to your Auto Scaling group, the instances are also
     * registered with the target groups.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/attach-instance-asg.html">Attach EC2 instances to
     * your Auto Scaling group</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param attachInstancesRequest
     * @return A Java Future containing the result of the AttachInstances operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>ServiceLinkedRoleFailureException The service-linked role is not yet ready for use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.AttachInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/AttachInstances" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AttachInstancesResponse> attachInstances(AttachInstancesRequest attachInstancesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(attachInstancesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, attachInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AttachInstances");

            HttpResponseHandler<AttachInstancesResponse> responseHandler = protocolFactory
                    .createResponseHandler(AttachInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<AttachInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AttachInstancesRequest, AttachInstancesResponse>()
                            .withOperationName("AttachInstances")
                            .withMarshaller(new AttachInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(attachInstancesRequest));
            CompletableFuture<AttachInstancesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This API operation is superseded by <a>AttachTrafficSources</a>, which can attach multiple traffic sources types.
     * We recommend using <code>AttachTrafficSources</code> to simplify how you manage traffic sources. However, we
     * continue to support <code>AttachLoadBalancerTargetGroups</code>. You can use both the original
     * <code>AttachLoadBalancerTargetGroups</code> API operation and <code>AttachTrafficSources</code> on the same Auto
     * Scaling group.
     * </p>
     * </note>
     * <p>
     * Attaches one or more target groups to the specified Auto Scaling group.
     * </p>
     * <p>
     * This operation is used with the following load balancer types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Application Load Balancer - Operates at the application layer (layer 7) and supports HTTP and HTTPS.
     * </p>
     * </li>
     * <li>
     * <p>
     * Network Load Balancer - Operates at the transport layer (layer 4) and supports TCP, TLS, and UDP.
     * </p>
     * </li>
     * <li>
     * <p>
     * Gateway Load Balancer - Operates at the network layer (layer 3).
     * </p>
     * </li>
     * </ul>
     * <p>
     * To describe the target groups for an Auto Scaling group, call the <a>DescribeLoadBalancerTargetGroups</a> API. To
     * detach the target group from the Auto Scaling group, call the <a>DetachLoadBalancerTargetGroups</a> API.
     * </p>
     * <p>
     * This operation is additive and does not detach existing target groups or Classic Load Balancers from the Auto
     * Scaling group.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/autoscaling-load-balancer.html">Use Elastic Load
     * Balancing to distribute traffic across the instances in your Auto Scaling group</a> in the <i>Amazon EC2 Auto
     * Scaling User Guide</i>.
     * </p>
     *
     * @param attachLoadBalancerTargetGroupsRequest
     * @return A Java Future containing the result of the AttachLoadBalancerTargetGroups operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>ServiceLinkedRoleFailureException The service-linked role is not yet ready for use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.AttachLoadBalancerTargetGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/AttachLoadBalancerTargetGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AttachLoadBalancerTargetGroupsResponse> attachLoadBalancerTargetGroups(
            AttachLoadBalancerTargetGroupsRequest attachLoadBalancerTargetGroupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(attachLoadBalancerTargetGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                attachLoadBalancerTargetGroupsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AttachLoadBalancerTargetGroups");

            HttpResponseHandler<AttachLoadBalancerTargetGroupsResponse> responseHandler = protocolFactory
                    .createResponseHandler(AttachLoadBalancerTargetGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<AttachLoadBalancerTargetGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AttachLoadBalancerTargetGroupsRequest, AttachLoadBalancerTargetGroupsResponse>()
                            .withOperationName("AttachLoadBalancerTargetGroups")
                            .withMarshaller(new AttachLoadBalancerTargetGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(attachLoadBalancerTargetGroupsRequest));
            CompletableFuture<AttachLoadBalancerTargetGroupsResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This API operation is superseded by <a>AttachTrafficSources</a>, which can attach multiple traffic sources types.
     * We recommend using <code>AttachTrafficSources</code> to simplify how you manage traffic sources. However, we
     * continue to support <code>AttachLoadBalancers</code>. You can use both the original
     * <code>AttachLoadBalancers</code> API operation and <code>AttachTrafficSources</code> on the same Auto Scaling
     * group.
     * </p>
     * </note>
     * <p>
     * Attaches one or more Classic Load Balancers to the specified Auto Scaling group. Amazon EC2 Auto Scaling
     * registers the running instances with these Classic Load Balancers.
     * </p>
     * <p>
     * To describe the load balancers for an Auto Scaling group, call the <a>DescribeLoadBalancers</a> API. To detach a
     * load balancer from the Auto Scaling group, call the <a>DetachLoadBalancers</a> API.
     * </p>
     * <p>
     * This operation is additive and does not detach existing Classic Load Balancers or target groups from the Auto
     * Scaling group.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/autoscaling-load-balancer.html">Use Elastic Load
     * Balancing to distribute traffic across the instances in your Auto Scaling group</a> in the <i>Amazon EC2 Auto
     * Scaling User Guide</i>.
     * </p>
     *
     * @param attachLoadBalancersRequest
     * @return A Java Future containing the result of the AttachLoadBalancers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>ServiceLinkedRoleFailureException The service-linked role is not yet ready for use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.AttachLoadBalancers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/AttachLoadBalancers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AttachLoadBalancersResponse> attachLoadBalancers(
            AttachLoadBalancersRequest attachLoadBalancersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(attachLoadBalancersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, attachLoadBalancersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AttachLoadBalancers");

            HttpResponseHandler<AttachLoadBalancersResponse> responseHandler = protocolFactory
                    .createResponseHandler(AttachLoadBalancersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<AttachLoadBalancersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AttachLoadBalancersRequest, AttachLoadBalancersResponse>()
                            .withOperationName("AttachLoadBalancers")
                            .withMarshaller(new AttachLoadBalancersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(attachLoadBalancersRequest));
            CompletableFuture<AttachLoadBalancersResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Attaches one or more traffic sources to the specified Auto Scaling group.
     * </p>
     * <p>
     * You can use any of the following as traffic sources for an Auto Scaling group:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Application Load Balancer
     * </p>
     * </li>
     * <li>
     * <p>
     * Classic Load Balancer
     * </p>
     * </li>
     * <li>
     * <p>
     * Gateway Load Balancer
     * </p>
     * </li>
     * <li>
     * <p>
     * Network Load Balancer
     * </p>
     * </li>
     * <li>
     * <p>
     * VPC Lattice
     * </p>
     * </li>
     * </ul>
     * <p>
     * This operation is additive and does not detach existing traffic sources from the Auto Scaling group.
     * </p>
     * <p>
     * After the operation completes, use the <a>DescribeTrafficSources</a> API to return details about the state of the
     * attachments between traffic sources and your Auto Scaling group. To detach a traffic source from the Auto Scaling
     * group, call the <a>DetachTrafficSources</a> API.
     * </p>
     *
     * @param attachTrafficSourcesRequest
     * @return A Java Future containing the result of the AttachTrafficSources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>ServiceLinkedRoleFailureException The service-linked role is not yet ready for use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.AttachTrafficSources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/AttachTrafficSources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AttachTrafficSourcesResponse> attachTrafficSources(
            AttachTrafficSourcesRequest attachTrafficSourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(attachTrafficSourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, attachTrafficSourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AttachTrafficSources");

            HttpResponseHandler<AttachTrafficSourcesResponse> responseHandler = protocolFactory
                    .createResponseHandler(AttachTrafficSourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<AttachTrafficSourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AttachTrafficSourcesRequest, AttachTrafficSourcesResponse>()
                            .withOperationName("AttachTrafficSources")
                            .withMarshaller(new AttachTrafficSourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(attachTrafficSourcesRequest));
            CompletableFuture<AttachTrafficSourcesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes one or more scheduled actions for the specified Auto Scaling group.
     * </p>
     *
     * @param batchDeleteScheduledActionRequest
     * @return A Java Future containing the result of the BatchDeleteScheduledAction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.BatchDeleteScheduledAction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/BatchDeleteScheduledAction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDeleteScheduledActionResponse> batchDeleteScheduledAction(
            BatchDeleteScheduledActionRequest batchDeleteScheduledActionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchDeleteScheduledActionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDeleteScheduledActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDeleteScheduledAction");

            HttpResponseHandler<BatchDeleteScheduledActionResponse> responseHandler = protocolFactory
                    .createResponseHandler(BatchDeleteScheduledActionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<BatchDeleteScheduledActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDeleteScheduledActionRequest, BatchDeleteScheduledActionResponse>()
                            .withOperationName("BatchDeleteScheduledAction")
                            .withMarshaller(new BatchDeleteScheduledActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchDeleteScheduledActionRequest));
            CompletableFuture<BatchDeleteScheduledActionResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates one or more scheduled scaling actions for an Auto Scaling group.
     * </p>
     *
     * @param batchPutScheduledUpdateGroupActionRequest
     * @return A Java Future containing the result of the BatchPutScheduledUpdateGroupAction operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AlreadyExistsException You already have an Auto Scaling group or launch configuration with this name.
     *         </li>
     *         <li>LimitExceededException You have already reached a limit for your Amazon EC2 Auto Scaling resources
     *         (for example, Auto Scaling groups, launch configurations, or lifecycle hooks). For more information, see
     *         <a href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html">
     *         DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.BatchPutScheduledUpdateGroupAction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/BatchPutScheduledUpdateGroupAction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchPutScheduledUpdateGroupActionResponse> batchPutScheduledUpdateGroupAction(
            BatchPutScheduledUpdateGroupActionRequest batchPutScheduledUpdateGroupActionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchPutScheduledUpdateGroupActionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                batchPutScheduledUpdateGroupActionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchPutScheduledUpdateGroupAction");

            HttpResponseHandler<BatchPutScheduledUpdateGroupActionResponse> responseHandler = protocolFactory
                    .createResponseHandler(BatchPutScheduledUpdateGroupActionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<BatchPutScheduledUpdateGroupActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchPutScheduledUpdateGroupActionRequest, BatchPutScheduledUpdateGroupActionResponse>()
                            .withOperationName("BatchPutScheduledUpdateGroupAction")
                            .withMarshaller(new BatchPutScheduledUpdateGroupActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchPutScheduledUpdateGroupActionRequest));
            CompletableFuture<BatchPutScheduledUpdateGroupActionResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels an instance refresh or rollback that is in progress. If an instance refresh or rollback is not in
     * progress, an <code>ActiveInstanceRefreshNotFound</code> error occurs.
     * </p>
     * <p>
     * This operation is part of the <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-instance-refresh.html">instance refresh
     * feature</a> in Amazon EC2 Auto Scaling, which helps you update instances in your Auto Scaling group after you
     * make configuration changes.
     * </p>
     * <p>
     * When you cancel an instance refresh, this does not roll back any changes that it made. Use the
     * <a>RollbackInstanceRefresh</a> API to roll back instead.
     * </p>
     *
     * @param cancelInstanceRefreshRequest
     * @return A Java Future containing the result of the CancelInstanceRefresh operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You have already reached a limit for your Amazon EC2 Auto Scaling resources
     *         (for example, Auto Scaling groups, launch configurations, or lifecycle hooks). For more information, see
     *         <a href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html">
     *         DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>ActiveInstanceRefreshNotFoundException The request failed because an active instance refresh or
     *         rollback for the specified Auto Scaling group was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.CancelInstanceRefresh
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/CancelInstanceRefresh"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelInstanceRefreshResponse> cancelInstanceRefresh(
            CancelInstanceRefreshRequest cancelInstanceRefreshRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelInstanceRefreshRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelInstanceRefreshRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelInstanceRefresh");

            HttpResponseHandler<CancelInstanceRefreshResponse> responseHandler = protocolFactory
                    .createResponseHandler(CancelInstanceRefreshResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CancelInstanceRefreshResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelInstanceRefreshRequest, CancelInstanceRefreshResponse>()
                            .withOperationName("CancelInstanceRefresh")
                            .withMarshaller(new CancelInstanceRefreshRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(cancelInstanceRefreshRequest));
            CompletableFuture<CancelInstanceRefreshResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Completes the lifecycle action for the specified token or instance with the specified result.
     * </p>
     * <p>
     * This step is a part of the procedure for adding a lifecycle hook to an Auto Scaling group:
     * </p>
     * <ol>
     * <li>
     * <p>
     * (Optional) Create a launch template or launch configuration with a user data script that runs while an instance
     * is in a wait state due to a lifecycle hook.
     * </p>
     * </li>
     * <li>
     * <p>
     * (Optional) Create a Lambda function and a rule that allows Amazon EventBridge to invoke your Lambda function when
     * an instance is put into a wait state due to a lifecycle hook.
     * </p>
     * </li>
     * <li>
     * <p>
     * (Optional) Create a notification target and an IAM role. The target can be either an Amazon SQS queue or an
     * Amazon SNS topic. The role allows Amazon EC2 Auto Scaling to publish lifecycle notifications to the target.
     * </p>
     * </li>
     * <li>
     * <p>
     * Create the lifecycle hook. Specify whether the hook is used when the instances launch or terminate.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you need more time, record the lifecycle action heartbeat to keep the instance in a wait state.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>If you finish before the timeout period ends, send a callback by using the <a>CompleteLifecycleAction</a> API
     * call.</b>
     * </p>
     * </li>
     * </ol>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/completing-lifecycle-hooks.html">Complete a lifecycle
     * action</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param completeLifecycleActionRequest
     * @return A Java Future containing the result of the CompleteLifecycleAction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.CompleteLifecycleAction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/CompleteLifecycleAction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CompleteLifecycleActionResponse> completeLifecycleAction(
            CompleteLifecycleActionRequest completeLifecycleActionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(completeLifecycleActionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, completeLifecycleActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CompleteLifecycleAction");

            HttpResponseHandler<CompleteLifecycleActionResponse> responseHandler = protocolFactory
                    .createResponseHandler(CompleteLifecycleActionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CompleteLifecycleActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CompleteLifecycleActionRequest, CompleteLifecycleActionResponse>()
                            .withOperationName("CompleteLifecycleAction")
                            .withMarshaller(new CompleteLifecycleActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(completeLifecycleActionRequest));
            CompletableFuture<CompleteLifecycleActionResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <b>We strongly recommend using a launch template when calling this operation to ensure full functionality for
     * Amazon EC2 Auto Scaling and Amazon EC2.</b>
     * </p>
     * <p>
     * Creates an Auto Scaling group with the specified name and attributes.
     * </p>
     * <p>
     * If you exceed your maximum limit of Auto Scaling groups, the call fails. To query this limit, call the
     * <a>DescribeAccountLimits</a> API. For information about updating this limit, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-quotas.html">Quotas for Amazon EC2
     * Auto Scaling</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * For introductory exercises for creating an Auto Scaling group, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/GettingStartedTutorial.html">Getting started with
     * Amazon EC2 Auto Scaling</a> and <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-register-lbs-with-asg.html">Tutorial: Set up a
     * scaled and load-balanced application</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>. For more information,
     * see <a href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/AutoScalingGroup.html">Auto Scaling groups</a>
     * in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * Every Auto Scaling group has three size properties (<code>DesiredCapacity</code>, <code>MaxSize</code>, and
     * <code>MinSize</code>). Usually, you set these sizes based on a specific number of instances. However, if you
     * configure a mixed instances policy that defines weights for the instance types, you must specify these sizes with
     * the same units that you use for weighting instances.
     * </p>
     *
     * @param createAutoScalingGroupRequest
     * @return A Java Future containing the result of the CreateAutoScalingGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AlreadyExistsException You already have an Auto Scaling group or launch configuration with this name.
     *         </li>
     *         <li>LimitExceededException You have already reached a limit for your Amazon EC2 Auto Scaling resources
     *         (for example, Auto Scaling groups, launch configurations, or lifecycle hooks). For more information, see
     *         <a href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html">
     *         DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>ServiceLinkedRoleFailureException The service-linked role is not yet ready for use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.CreateAutoScalingGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/CreateAutoScalingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAutoScalingGroupResponse> createAutoScalingGroup(
            CreateAutoScalingGroupRequest createAutoScalingGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAutoScalingGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAutoScalingGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAutoScalingGroup");

            HttpResponseHandler<CreateAutoScalingGroupResponse> responseHandler = protocolFactory
                    .createResponseHandler(CreateAutoScalingGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CreateAutoScalingGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAutoScalingGroupRequest, CreateAutoScalingGroupResponse>()
                            .withOperationName("CreateAutoScalingGroup")
                            .withMarshaller(new CreateAutoScalingGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAutoScalingGroupRequest));
            CompletableFuture<CreateAutoScalingGroupResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a launch configuration.
     * </p>
     * <p>
     * If you exceed your maximum limit of launch configurations, the call fails. To query this limit, call the
     * <a>DescribeAccountLimits</a> API. For information about updating this limit, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-quotas.html">Quotas for Amazon EC2
     * Auto Scaling</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/LaunchConfiguration.html">Launch configurations</a>
     * in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <note>
     * <p>
     * Amazon EC2 Auto Scaling configures instances launched as part of an Auto Scaling group using either a launch
     * template or a launch configuration. We strongly recommend that you do not use launch configurations. They do not
     * provide full functionality for Amazon EC2 Auto Scaling or Amazon EC2. For information about using launch
     * templates, see <a href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/launch-templates.html">Launch
     * templates</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * </note>
     *
     * @param createLaunchConfigurationRequest
     * @return A Java Future containing the result of the CreateLaunchConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AlreadyExistsException You already have an Auto Scaling group or launch configuration with this name.
     *         </li>
     *         <li>LimitExceededException You have already reached a limit for your Amazon EC2 Auto Scaling resources
     *         (for example, Auto Scaling groups, launch configurations, or lifecycle hooks). For more information, see
     *         <a href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html">
     *         DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.CreateLaunchConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/CreateLaunchConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLaunchConfigurationResponse> createLaunchConfiguration(
            CreateLaunchConfigurationRequest createLaunchConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createLaunchConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLaunchConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLaunchConfiguration");

            HttpResponseHandler<CreateLaunchConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(CreateLaunchConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CreateLaunchConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLaunchConfigurationRequest, CreateLaunchConfigurationResponse>()
                            .withOperationName("CreateLaunchConfiguration")
                            .withMarshaller(new CreateLaunchConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createLaunchConfigurationRequest));
            CompletableFuture<CreateLaunchConfigurationResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates tags for the specified Auto Scaling group.
     * </p>
     * <p>
     * When you specify a tag with a key that already exists, the operation overwrites the previous tag definition, and
     * you do not get an error message.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-tagging.html">Tag Auto Scaling
     * groups and instances</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param createOrUpdateTagsRequest
     * @return A Java Future containing the result of the CreateOrUpdateTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You have already reached a limit for your Amazon EC2 Auto Scaling resources
     *         (for example, Auto Scaling groups, launch configurations, or lifecycle hooks). For more information, see
     *         <a href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html">
     *         DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.</li>
     *         <li>AlreadyExistsException You already have an Auto Scaling group or launch configuration with this name.
     *         </li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>ResourceInUseException The operation can't be performed because the resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.CreateOrUpdateTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/CreateOrUpdateTags"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateOrUpdateTagsResponse> createOrUpdateTags(CreateOrUpdateTagsRequest createOrUpdateTagsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createOrUpdateTagsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createOrUpdateTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateOrUpdateTags");

            HttpResponseHandler<CreateOrUpdateTagsResponse> responseHandler = protocolFactory
                    .createResponseHandler(CreateOrUpdateTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CreateOrUpdateTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateOrUpdateTagsRequest, CreateOrUpdateTagsResponse>()
                            .withOperationName("CreateOrUpdateTags")
                            .withMarshaller(new CreateOrUpdateTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createOrUpdateTagsRequest));
            CompletableFuture<CreateOrUpdateTagsResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified Auto Scaling group.
     * </p>
     * <p>
     * If the group has instances or scaling activities in progress, you must specify the option to force the deletion
     * in order for it to succeed. The force delete operation will also terminate the EC2 instances. If the group has a
     * warm pool, the force delete option also deletes the warm pool.
     * </p>
     * <p>
     * To remove instances from the Auto Scaling group before deleting it, call the <a>DetachInstances</a> API with the
     * list of instances and the option to decrement the desired capacity. This ensures that Amazon EC2 Auto Scaling
     * does not launch replacement instances.
     * </p>
     * <p>
     * To terminate all instances before deleting the Auto Scaling group, call the <a>UpdateAutoScalingGroup</a> API and
     * set the minimum size and desired capacity of the Auto Scaling group to zero.
     * </p>
     * <p>
     * If the group has scaling policies, deleting the group deletes the policies, the underlying alarm actions, and any
     * alarm that no longer has an associated action.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-process-shutdown.html">Delete your Auto Scaling
     * infrastructure</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param deleteAutoScalingGroupRequest
     * @return A Java Future containing the result of the DeleteAutoScalingGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ScalingActivityInProgressException The operation can't be performed because there are scaling
     *         activities in progress.</li>
     *         <li>ResourceInUseException The operation can't be performed because the resource is in use.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DeleteAutoScalingGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DeleteAutoScalingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAutoScalingGroupResponse> deleteAutoScalingGroup(
            DeleteAutoScalingGroupRequest deleteAutoScalingGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAutoScalingGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAutoScalingGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAutoScalingGroup");

            HttpResponseHandler<DeleteAutoScalingGroupResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeleteAutoScalingGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteAutoScalingGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAutoScalingGroupRequest, DeleteAutoScalingGroupResponse>()
                            .withOperationName("DeleteAutoScalingGroup")
                            .withMarshaller(new DeleteAutoScalingGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAutoScalingGroupRequest));
            CompletableFuture<DeleteAutoScalingGroupResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified launch configuration.
     * </p>
     * <p>
     * The launch configuration must not be attached to an Auto Scaling group. When this call completes, the launch
     * configuration is no longer available for use.
     * </p>
     *
     * @param deleteLaunchConfigurationRequest
     * @return A Java Future containing the result of the DeleteLaunchConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException The operation can't be performed because the resource is in use.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DeleteLaunchConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DeleteLaunchConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLaunchConfigurationResponse> deleteLaunchConfiguration(
            DeleteLaunchConfigurationRequest deleteLaunchConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteLaunchConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLaunchConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLaunchConfiguration");

            HttpResponseHandler<DeleteLaunchConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeleteLaunchConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteLaunchConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLaunchConfigurationRequest, DeleteLaunchConfigurationResponse>()
                            .withOperationName("DeleteLaunchConfiguration")
                            .withMarshaller(new DeleteLaunchConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteLaunchConfigurationRequest));
            CompletableFuture<DeleteLaunchConfigurationResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified lifecycle hook.
     * </p>
     * <p>
     * If there are any outstanding lifecycle actions, they are completed first (<code>ABANDON</code> for launching
     * instances, <code>CONTINUE</code> for terminating instances).
     * </p>
     *
     * @param deleteLifecycleHookRequest
     * @return A Java Future containing the result of the DeleteLifecycleHook operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DeleteLifecycleHook
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DeleteLifecycleHook"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLifecycleHookResponse> deleteLifecycleHook(
            DeleteLifecycleHookRequest deleteLifecycleHookRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteLifecycleHookRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLifecycleHookRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLifecycleHook");

            HttpResponseHandler<DeleteLifecycleHookResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeleteLifecycleHookResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteLifecycleHookResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLifecycleHookRequest, DeleteLifecycleHookResponse>()
                            .withOperationName("DeleteLifecycleHook")
                            .withMarshaller(new DeleteLifecycleHookRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteLifecycleHookRequest));
            CompletableFuture<DeleteLifecycleHookResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified notification.
     * </p>
     *
     * @param deleteNotificationConfigurationRequest
     * @return A Java Future containing the result of the DeleteNotificationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DeleteNotificationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DeleteNotificationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteNotificationConfigurationResponse> deleteNotificationConfiguration(
            DeleteNotificationConfigurationRequest deleteNotificationConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteNotificationConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteNotificationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNotificationConfiguration");

            HttpResponseHandler<DeleteNotificationConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeleteNotificationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteNotificationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteNotificationConfigurationRequest, DeleteNotificationConfigurationResponse>()
                            .withOperationName("DeleteNotificationConfiguration")
                            .withMarshaller(new DeleteNotificationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteNotificationConfigurationRequest));
            CompletableFuture<DeleteNotificationConfigurationResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified scaling policy.
     * </p>
     * <p>
     * Deleting either a step scaling policy or a simple scaling policy deletes the underlying alarm action, but does
     * not delete the alarm, even if it no longer has an associated action.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/deleting-scaling-policy.html">Deleting a scaling
     * policy</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param deletePolicyRequest
     * @return A Java Future containing the result of the DeletePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>ServiceLinkedRoleFailureException The service-linked role is not yet ready for use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DeletePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DeletePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePolicyResponse> deletePolicy(DeletePolicyRequest deletePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePolicyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePolicy");

            HttpResponseHandler<DeletePolicyResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeletePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeletePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePolicyRequest, DeletePolicyResponse>()
                            .withOperationName("DeletePolicy").withMarshaller(new DeletePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePolicyRequest));
            CompletableFuture<DeletePolicyResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified scheduled action.
     * </p>
     *
     * @param deleteScheduledActionRequest
     * @return A Java Future containing the result of the DeleteScheduledAction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DeleteScheduledAction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DeleteScheduledAction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteScheduledActionResponse> deleteScheduledAction(
            DeleteScheduledActionRequest deleteScheduledActionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteScheduledActionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteScheduledActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteScheduledAction");

            HttpResponseHandler<DeleteScheduledActionResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeleteScheduledActionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteScheduledActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteScheduledActionRequest, DeleteScheduledActionResponse>()
                            .withOperationName("DeleteScheduledAction")
                            .withMarshaller(new DeleteScheduledActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteScheduledActionRequest));
            CompletableFuture<DeleteScheduledActionResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified tags.
     * </p>
     *
     * @param deleteTagsRequest
     * @return A Java Future containing the result of the DeleteTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>ResourceInUseException The operation can't be performed because the resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DeleteTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DeleteTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTagsResponse> deleteTags(DeleteTagsRequest deleteTagsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTagsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTags");

            HttpResponseHandler<DeleteTagsResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeleteTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTagsRequest, DeleteTagsResponse>().withOperationName("DeleteTags")
                            .withMarshaller(new DeleteTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTagsRequest));
            CompletableFuture<DeleteTagsResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the warm pool for the specified Auto Scaling group.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-warm-pools.html">Warm pools for
     * Amazon EC2 Auto Scaling</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param deleteWarmPoolRequest
     * @return A Java Future containing the result of the DeleteWarmPool operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You have already reached a limit for your Amazon EC2 Auto Scaling resources
     *         (for example, Auto Scaling groups, launch configurations, or lifecycle hooks). For more information, see
     *         <a href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html">
     *         DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>ScalingActivityInProgressException The operation can't be performed because there are scaling
     *         activities in progress.</li>
     *         <li>ResourceInUseException The operation can't be performed because the resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DeleteWarmPool
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DeleteWarmPool" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWarmPoolResponse> deleteWarmPool(DeleteWarmPoolRequest deleteWarmPoolRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteWarmPoolRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWarmPoolRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWarmPool");

            HttpResponseHandler<DeleteWarmPoolResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeleteWarmPoolResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteWarmPoolResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWarmPoolRequest, DeleteWarmPoolResponse>()
                            .withOperationName("DeleteWarmPool")
                            .withMarshaller(new DeleteWarmPoolRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteWarmPoolRequest));
            CompletableFuture<DeleteWarmPoolResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the current Amazon EC2 Auto Scaling resource quotas for your account.
     * </p>
     * <p>
     * When you establish an Amazon Web Services account, the account has initial quotas on the maximum number of Auto
     * Scaling groups and launch configurations that you can create in a given Region. For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-quotas.html">Quotas for Amazon EC2
     * Auto Scaling</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param describeAccountLimitsRequest
     * @return A Java Future containing the result of the DescribeAccountLimits operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribeAccountLimits
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeAccountLimits"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAccountLimitsResponse> describeAccountLimits(
            DescribeAccountLimitsRequest describeAccountLimitsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAccountLimitsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAccountLimitsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAccountLimits");

            HttpResponseHandler<DescribeAccountLimitsResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeAccountLimitsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeAccountLimitsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAccountLimitsRequest, DescribeAccountLimitsResponse>()
                            .withOperationName("DescribeAccountLimits")
                            .withMarshaller(new DescribeAccountLimitsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAccountLimitsRequest));
            CompletableFuture<DescribeAccountLimitsResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the available adjustment types for step scaling and simple scaling policies.
     * </p>
     * <p>
     * The following adjustment types are supported:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ChangeInCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ExactCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PercentChangeInCapacity</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeAdjustmentTypesRequest
     * @return A Java Future containing the result of the DescribeAdjustmentTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribeAdjustmentTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeAdjustmentTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAdjustmentTypesResponse> describeAdjustmentTypes(
            DescribeAdjustmentTypesRequest describeAdjustmentTypesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAdjustmentTypesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAdjustmentTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAdjustmentTypes");

            HttpResponseHandler<DescribeAdjustmentTypesResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeAdjustmentTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeAdjustmentTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAdjustmentTypesRequest, DescribeAdjustmentTypesResponse>()
                            .withOperationName("DescribeAdjustmentTypes")
                            .withMarshaller(new DescribeAdjustmentTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAdjustmentTypesRequest));
            CompletableFuture<DescribeAdjustmentTypesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the Auto Scaling groups in the account and Region.
     * </p>
     * <p>
     * If you specify Auto Scaling group names, the output includes information for only the specified Auto Scaling
     * groups. If you specify filters, the output includes information for only those Auto Scaling groups that meet the
     * filter criteria. If you do not specify group names or filters, the output includes information for all Auto
     * Scaling groups.
     * </p>
     * <p>
     * This operation also returns information about instances in Auto Scaling groups. To retrieve information about the
     * instances in a warm pool, you must call the <a>DescribeWarmPool</a> API.
     * </p>
     *
     * @param describeAutoScalingGroupsRequest
     * @return A Java Future containing the result of the DescribeAutoScalingGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribeAutoScalingGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeAutoScalingGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAutoScalingGroupsResponse> describeAutoScalingGroups(
            DescribeAutoScalingGroupsRequest describeAutoScalingGroupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAutoScalingGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAutoScalingGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAutoScalingGroups");

            HttpResponseHandler<DescribeAutoScalingGroupsResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeAutoScalingGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeAutoScalingGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAutoScalingGroupsRequest, DescribeAutoScalingGroupsResponse>()
                            .withOperationName("DescribeAutoScalingGroups")
                            .withMarshaller(new DescribeAutoScalingGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAutoScalingGroupsRequest));
            CompletableFuture<DescribeAutoScalingGroupsResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the Auto Scaling instances in the account and Region.
     * </p>
     *
     * @param describeAutoScalingInstancesRequest
     * @return A Java Future containing the result of the DescribeAutoScalingInstances operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribeAutoScalingInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeAutoScalingInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAutoScalingInstancesResponse> describeAutoScalingInstances(
            DescribeAutoScalingInstancesRequest describeAutoScalingInstancesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAutoScalingInstancesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAutoScalingInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAutoScalingInstances");

            HttpResponseHandler<DescribeAutoScalingInstancesResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeAutoScalingInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeAutoScalingInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAutoScalingInstancesRequest, DescribeAutoScalingInstancesResponse>()
                            .withOperationName("DescribeAutoScalingInstances")
                            .withMarshaller(new DescribeAutoScalingInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAutoScalingInstancesRequest));
            CompletableFuture<DescribeAutoScalingInstancesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the notification types that are supported by Amazon EC2 Auto Scaling.
     * </p>
     *
     * @param describeAutoScalingNotificationTypesRequest
     * @return A Java Future containing the result of the DescribeAutoScalingNotificationTypes operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribeAutoScalingNotificationTypes
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeAutoScalingNotificationTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAutoScalingNotificationTypesResponse> describeAutoScalingNotificationTypes(
            DescribeAutoScalingNotificationTypesRequest describeAutoScalingNotificationTypesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAutoScalingNotificationTypesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeAutoScalingNotificationTypesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAutoScalingNotificationTypes");

            HttpResponseHandler<DescribeAutoScalingNotificationTypesResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeAutoScalingNotificationTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeAutoScalingNotificationTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAutoScalingNotificationTypesRequest, DescribeAutoScalingNotificationTypesResponse>()
                            .withOperationName("DescribeAutoScalingNotificationTypes")
                            .withMarshaller(new DescribeAutoScalingNotificationTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAutoScalingNotificationTypesRequest));
            CompletableFuture<DescribeAutoScalingNotificationTypesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the instance refreshes for the specified Auto Scaling group.
     * </p>
     * <p>
     * This operation is part of the <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-instance-refresh.html">instance refresh
     * feature</a> in Amazon EC2 Auto Scaling, which helps you update instances in your Auto Scaling group after you
     * make configuration changes.
     * </p>
     * <p>
     * To help you determine the status of an instance refresh, Amazon EC2 Auto Scaling returns information about the
     * instance refreshes you previously initiated, including their status, start time, end time, the percentage of the
     * instance refresh that is complete, and the number of instances remaining to update before the instance refresh is
     * complete. If a rollback is initiated while an instance refresh is in progress, Amazon EC2 Auto Scaling also
     * returns information about the rollback of the instance refresh.
     * </p>
     *
     * @param describeInstanceRefreshesRequest
     * @return A Java Future containing the result of the DescribeInstanceRefreshes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribeInstanceRefreshes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeInstanceRefreshes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeInstanceRefreshesResponse> describeInstanceRefreshes(
            DescribeInstanceRefreshesRequest describeInstanceRefreshesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeInstanceRefreshesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeInstanceRefreshesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeInstanceRefreshes");

            HttpResponseHandler<DescribeInstanceRefreshesResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeInstanceRefreshesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeInstanceRefreshesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeInstanceRefreshesRequest, DescribeInstanceRefreshesResponse>()
                            .withOperationName("DescribeInstanceRefreshes")
                            .withMarshaller(new DescribeInstanceRefreshesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeInstanceRefreshesRequest));
            CompletableFuture<DescribeInstanceRefreshesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the launch configurations in the account and Region.
     * </p>
     *
     * @param describeLaunchConfigurationsRequest
     * @return A Java Future containing the result of the DescribeLaunchConfigurations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribeLaunchConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeLaunchConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLaunchConfigurationsResponse> describeLaunchConfigurations(
            DescribeLaunchConfigurationsRequest describeLaunchConfigurationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeLaunchConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLaunchConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLaunchConfigurations");

            HttpResponseHandler<DescribeLaunchConfigurationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeLaunchConfigurationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeLaunchConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLaunchConfigurationsRequest, DescribeLaunchConfigurationsResponse>()
                            .withOperationName("DescribeLaunchConfigurations")
                            .withMarshaller(new DescribeLaunchConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeLaunchConfigurationsRequest));
            CompletableFuture<DescribeLaunchConfigurationsResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the available types of lifecycle hooks.
     * </p>
     * <p>
     * The following hook types are supported:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>autoscaling:EC2_INSTANCE_LAUNCHING</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>autoscaling:EC2_INSTANCE_TERMINATING</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeLifecycleHookTypesRequest
     * @return A Java Future containing the result of the DescribeLifecycleHookTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribeLifecycleHookTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeLifecycleHookTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLifecycleHookTypesResponse> describeLifecycleHookTypes(
            DescribeLifecycleHookTypesRequest describeLifecycleHookTypesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeLifecycleHookTypesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLifecycleHookTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLifecycleHookTypes");

            HttpResponseHandler<DescribeLifecycleHookTypesResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeLifecycleHookTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeLifecycleHookTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLifecycleHookTypesRequest, DescribeLifecycleHookTypesResponse>()
                            .withOperationName("DescribeLifecycleHookTypes")
                            .withMarshaller(new DescribeLifecycleHookTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeLifecycleHookTypesRequest));
            CompletableFuture<DescribeLifecycleHookTypesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the lifecycle hooks for the specified Auto Scaling group.
     * </p>
     *
     * @param describeLifecycleHooksRequest
     * @return A Java Future containing the result of the DescribeLifecycleHooks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribeLifecycleHooks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeLifecycleHooks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLifecycleHooksResponse> describeLifecycleHooks(
            DescribeLifecycleHooksRequest describeLifecycleHooksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeLifecycleHooksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLifecycleHooksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLifecycleHooks");

            HttpResponseHandler<DescribeLifecycleHooksResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeLifecycleHooksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeLifecycleHooksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLifecycleHooksRequest, DescribeLifecycleHooksResponse>()
                            .withOperationName("DescribeLifecycleHooks")
                            .withMarshaller(new DescribeLifecycleHooksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeLifecycleHooksRequest));
            CompletableFuture<DescribeLifecycleHooksResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This API operation is superseded by <a>DescribeTrafficSources</a>, which can describe multiple traffic sources
     * types. We recommend using <code>DetachTrafficSources</code> to simplify how you manage traffic sources. However,
     * we continue to support <code>DescribeLoadBalancerTargetGroups</code>. You can use both the original
     * <code>DescribeLoadBalancerTargetGroups</code> API operation and <code>DescribeTrafficSources</code> on the same
     * Auto Scaling group.
     * </p>
     * </note>
     * <p>
     * Gets information about the Elastic Load Balancing target groups for the specified Auto Scaling group.
     * </p>
     * <p>
     * To determine the attachment status of the target group, use the <code>State</code> element in the response. When
     * you attach a target group to an Auto Scaling group, the initial <code>State</code> value is <code>Adding</code>.
     * The state transitions to <code>Added</code> after all Auto Scaling instances are registered with the target
     * group. If Elastic Load Balancing health checks are enabled for the Auto Scaling group, the state transitions to
     * <code>InService</code> after at least one Auto Scaling instance passes the health check. When the target group is
     * in the <code>InService</code> state, Amazon EC2 Auto Scaling can terminate and replace any instances that are
     * reported as unhealthy. If no registered instances pass the health checks, the target group doesn't enter the
     * <code>InService</code> state.
     * </p>
     * <p>
     * Target groups also have an <code>InService</code> state if you attach them in the <a>CreateAutoScalingGroup</a>
     * API call. If your target group state is <code>InService</code>, but it is not working properly, check the scaling
     * activities by calling <a>DescribeScalingActivities</a> and take any corrective actions necessary.
     * </p>
     * <p>
     * For help with failed health checks, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/ts-as-healthchecks.html">Troubleshooting Amazon EC2
     * Auto Scaling: Health checks</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>. For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/autoscaling-load-balancer.html">Use Elastic Load
     * Balancing to distribute traffic across the instances in your Auto Scaling group</a> in the <i>Amazon EC2 Auto
     * Scaling User Guide</i>.
     * </p>
     * <note>
     * <p>
     * You can use this operation to describe target groups that were attached by using
     * <a>AttachLoadBalancerTargetGroups</a>, but not for target groups that were attached by using
     * <a>AttachTrafficSources</a>.
     * </p>
     * </note>
     *
     * @param describeLoadBalancerTargetGroupsRequest
     * @return A Java Future containing the result of the DescribeLoadBalancerTargetGroups operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribeLoadBalancerTargetGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeLoadBalancerTargetGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLoadBalancerTargetGroupsResponse> describeLoadBalancerTargetGroups(
            DescribeLoadBalancerTargetGroupsRequest describeLoadBalancerTargetGroupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeLoadBalancerTargetGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeLoadBalancerTargetGroupsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLoadBalancerTargetGroups");

            HttpResponseHandler<DescribeLoadBalancerTargetGroupsResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeLoadBalancerTargetGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeLoadBalancerTargetGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLoadBalancerTargetGroupsRequest, DescribeLoadBalancerTargetGroupsResponse>()
                            .withOperationName("DescribeLoadBalancerTargetGroups")
                            .withMarshaller(new DescribeLoadBalancerTargetGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeLoadBalancerTargetGroupsRequest));
            CompletableFuture<DescribeLoadBalancerTargetGroupsResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This API operation is superseded by <a>DescribeTrafficSources</a>, which can describe multiple traffic sources
     * types. We recommend using <code>DescribeTrafficSources</code> to simplify how you manage traffic sources.
     * However, we continue to support <code>DescribeLoadBalancers</code>. You can use both the original
     * <code>DescribeLoadBalancers</code> API operation and <code>DescribeTrafficSources</code> on the same Auto Scaling
     * group.
     * </p>
     * </note>
     * <p>
     * Gets information about the load balancers for the specified Auto Scaling group.
     * </p>
     * <p>
     * This operation describes only Classic Load Balancers. If you have Application Load Balancers, Network Load
     * Balancers, or Gateway Load Balancers, use the <a>DescribeLoadBalancerTargetGroups</a> API instead.
     * </p>
     * <p>
     * To determine the attachment status of the load balancer, use the <code>State</code> element in the response. When
     * you attach a load balancer to an Auto Scaling group, the initial <code>State</code> value is <code>Adding</code>.
     * The state transitions to <code>Added</code> after all Auto Scaling instances are registered with the load
     * balancer. If Elastic Load Balancing health checks are enabled for the Auto Scaling group, the state transitions
     * to <code>InService</code> after at least one Auto Scaling instance passes the health check. When the load
     * balancer is in the <code>InService</code> state, Amazon EC2 Auto Scaling can terminate and replace any instances
     * that are reported as unhealthy. If no registered instances pass the health checks, the load balancer doesn't
     * enter the <code>InService</code> state.
     * </p>
     * <p>
     * Load balancers also have an <code>InService</code> state if you attach them in the <a>CreateAutoScalingGroup</a>
     * API call. If your load balancer state is <code>InService</code>, but it is not working properly, check the
     * scaling activities by calling <a>DescribeScalingActivities</a> and take any corrective actions necessary.
     * </p>
     * <p>
     * For help with failed health checks, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/ts-as-healthchecks.html">Troubleshooting Amazon EC2
     * Auto Scaling: Health checks</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>. For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/autoscaling-load-balancer.html">Use Elastic Load
     * Balancing to distribute traffic across the instances in your Auto Scaling group</a> in the <i>Amazon EC2 Auto
     * Scaling User Guide</i>.
     * </p>
     *
     * @param describeLoadBalancersRequest
     * @return A Java Future containing the result of the DescribeLoadBalancers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribeLoadBalancers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeLoadBalancers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLoadBalancersResponse> describeLoadBalancers(
            DescribeLoadBalancersRequest describeLoadBalancersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeLoadBalancersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLoadBalancersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLoadBalancers");

            HttpResponseHandler<DescribeLoadBalancersResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeLoadBalancersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeLoadBalancersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLoadBalancersRequest, DescribeLoadBalancersResponse>()
                            .withOperationName("DescribeLoadBalancers")
                            .withMarshaller(new DescribeLoadBalancersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeLoadBalancersRequest));
            CompletableFuture<DescribeLoadBalancersResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the available CloudWatch metrics for Amazon EC2 Auto Scaling.
     * </p>
     *
     * @param describeMetricCollectionTypesRequest
     * @return A Java Future containing the result of the DescribeMetricCollectionTypes operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribeMetricCollectionTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeMetricCollectionTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMetricCollectionTypesResponse> describeMetricCollectionTypes(
            DescribeMetricCollectionTypesRequest describeMetricCollectionTypesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeMetricCollectionTypesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeMetricCollectionTypesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMetricCollectionTypes");

            HttpResponseHandler<DescribeMetricCollectionTypesResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeMetricCollectionTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeMetricCollectionTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeMetricCollectionTypesRequest, DescribeMetricCollectionTypesResponse>()
                            .withOperationName("DescribeMetricCollectionTypes")
                            .withMarshaller(new DescribeMetricCollectionTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeMetricCollectionTypesRequest));
            CompletableFuture<DescribeMetricCollectionTypesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the Amazon SNS notifications that are configured for one or more Auto Scaling groups.
     * </p>
     *
     * @param describeNotificationConfigurationsRequest
     * @return A Java Future containing the result of the DescribeNotificationConfigurations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribeNotificationConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeNotificationConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeNotificationConfigurationsResponse> describeNotificationConfigurations(
            DescribeNotificationConfigurationsRequest describeNotificationConfigurationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeNotificationConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeNotificationConfigurationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeNotificationConfigurations");

            HttpResponseHandler<DescribeNotificationConfigurationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeNotificationConfigurationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeNotificationConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeNotificationConfigurationsRequest, DescribeNotificationConfigurationsResponse>()
                            .withOperationName("DescribeNotificationConfigurations")
                            .withMarshaller(new DescribeNotificationConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeNotificationConfigurationsRequest));
            CompletableFuture<DescribeNotificationConfigurationsResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the scaling policies in the account and Region.
     * </p>
     *
     * @param describePoliciesRequest
     * @return A Java Future containing the result of the DescribePolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>ServiceLinkedRoleFailureException The service-linked role is not yet ready for use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribePolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribePolicies" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePoliciesResponse> describePolicies(DescribePoliciesRequest describePoliciesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describePoliciesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePolicies");

            HttpResponseHandler<DescribePoliciesResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribePoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribePoliciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePoliciesRequest, DescribePoliciesResponse>()
                            .withOperationName("DescribePolicies")
                            .withMarshaller(new DescribePoliciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describePoliciesRequest));
            CompletableFuture<DescribePoliciesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the scaling activities in the account and Region.
     * </p>
     * <p>
     * When scaling events occur, you see a record of the scaling activity in the scaling activities. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-verify-scaling-activity.html">Verifying a scaling
     * activity for an Auto Scaling group</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * If the scaling event succeeds, the value of the <code>StatusCode</code> element in the response is
     * <code>Successful</code>. If an attempt to launch instances failed, the <code>StatusCode</code> value is
     * <code>Failed</code> or <code>Cancelled</code> and the <code>StatusMessage</code> element in the response
     * indicates the cause of the failure. For help interpreting the <code>StatusMessage</code>, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/CHAP_Troubleshooting.html">Troubleshooting Amazon EC2
     * Auto Scaling</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param describeScalingActivitiesRequest
     * @return A Java Future containing the result of the DescribeScalingActivities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribeScalingActivities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeScalingActivities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeScalingActivitiesResponse> describeScalingActivities(
            DescribeScalingActivitiesRequest describeScalingActivitiesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeScalingActivitiesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeScalingActivitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeScalingActivities");

            HttpResponseHandler<DescribeScalingActivitiesResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeScalingActivitiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeScalingActivitiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeScalingActivitiesRequest, DescribeScalingActivitiesResponse>()
                            .withOperationName("DescribeScalingActivities")
                            .withMarshaller(new DescribeScalingActivitiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeScalingActivitiesRequest));
            CompletableFuture<DescribeScalingActivitiesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the scaling process types for use with the <a>ResumeProcesses</a> and <a>SuspendProcesses</a> APIs.
     * </p>
     *
     * @param describeScalingProcessTypesRequest
     * @return A Java Future containing the result of the DescribeScalingProcessTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribeScalingProcessTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeScalingProcessTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeScalingProcessTypesResponse> describeScalingProcessTypes(
            DescribeScalingProcessTypesRequest describeScalingProcessTypesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeScalingProcessTypesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeScalingProcessTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeScalingProcessTypes");

            HttpResponseHandler<DescribeScalingProcessTypesResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeScalingProcessTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeScalingProcessTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeScalingProcessTypesRequest, DescribeScalingProcessTypesResponse>()
                            .withOperationName("DescribeScalingProcessTypes")
                            .withMarshaller(new DescribeScalingProcessTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeScalingProcessTypesRequest));
            CompletableFuture<DescribeScalingProcessTypesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the scheduled actions that haven't run or that have not reached their end time.
     * </p>
     * <p>
     * To describe the scaling activities for scheduled actions that have already run, call the
     * <a>DescribeScalingActivities</a> API.
     * </p>
     *
     * @param describeScheduledActionsRequest
     * @return A Java Future containing the result of the DescribeScheduledActions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribeScheduledActions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeScheduledActions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeScheduledActionsResponse> describeScheduledActions(
            DescribeScheduledActionsRequest describeScheduledActionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeScheduledActionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeScheduledActionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeScheduledActions");

            HttpResponseHandler<DescribeScheduledActionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeScheduledActionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeScheduledActionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeScheduledActionsRequest, DescribeScheduledActionsResponse>()
                            .withOperationName("DescribeScheduledActions")
                            .withMarshaller(new DescribeScheduledActionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeScheduledActionsRequest));
            CompletableFuture<DescribeScheduledActionsResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the specified tags.
     * </p>
     * <p>
     * You can use filters to limit the results. For example, you can query for the tags for a specific Auto Scaling
     * group. You can specify multiple values for a filter. A tag must match at least one of the specified values for it
     * to be included in the results.
     * </p>
     * <p>
     * You can also specify multiple filters. The result includes information for a particular tag only if it matches
     * all the filters. If there's no match, no special message is returned.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-tagging.html">Tag Auto Scaling
     * groups and instances</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param describeTagsRequest
     * @return A Java Future containing the result of the DescribeTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribeTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTagsResponse> describeTags(DescribeTagsRequest describeTagsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeTagsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTags");

            HttpResponseHandler<DescribeTagsResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTagsRequest, DescribeTagsResponse>()
                            .withOperationName("DescribeTags").withMarshaller(new DescribeTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeTagsRequest));
            CompletableFuture<DescribeTagsResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the termination policies supported by Amazon EC2 Auto Scaling.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-termination-policies.html">Work with
     * Amazon EC2 Auto Scaling termination policies</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param describeTerminationPolicyTypesRequest
     * @return A Java Future containing the result of the DescribeTerminationPolicyTypes operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribeTerminationPolicyTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeTerminationPolicyTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTerminationPolicyTypesResponse> describeTerminationPolicyTypes(
            DescribeTerminationPolicyTypesRequest describeTerminationPolicyTypesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeTerminationPolicyTypesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeTerminationPolicyTypesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTerminationPolicyTypes");

            HttpResponseHandler<DescribeTerminationPolicyTypesResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeTerminationPolicyTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeTerminationPolicyTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTerminationPolicyTypesRequest, DescribeTerminationPolicyTypesResponse>()
                            .withOperationName("DescribeTerminationPolicyTypes")
                            .withMarshaller(new DescribeTerminationPolicyTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeTerminationPolicyTypesRequest));
            CompletableFuture<DescribeTerminationPolicyTypesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the traffic sources for the specified Auto Scaling group.
     * </p>
     * <p>
     * You can optionally provide a traffic source type. If you provide a traffic source type, then the results only
     * include that traffic source type.
     * </p>
     * <p>
     * If you do not provide a traffic source type, then the results include all the traffic sources for the specified
     * Auto Scaling group.
     * </p>
     *
     * @param describeTrafficSourcesRequest
     * @return A Java Future containing the result of the DescribeTrafficSources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribeTrafficSources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeTrafficSources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTrafficSourcesResponse> describeTrafficSources(
            DescribeTrafficSourcesRequest describeTrafficSourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeTrafficSourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTrafficSourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTrafficSources");

            HttpResponseHandler<DescribeTrafficSourcesResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeTrafficSourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeTrafficSourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTrafficSourcesRequest, DescribeTrafficSourcesResponse>()
                            .withOperationName("DescribeTrafficSources")
                            .withMarshaller(new DescribeTrafficSourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeTrafficSourcesRequest));
            CompletableFuture<DescribeTrafficSourcesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a warm pool and its instances.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-warm-pools.html">Warm pools for
     * Amazon EC2 Auto Scaling</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param describeWarmPoolRequest
     * @return A Java Future containing the result of the DescribeWarmPool operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>LimitExceededException You have already reached a limit for your Amazon EC2 Auto Scaling resources
     *         (for example, Auto Scaling groups, launch configurations, or lifecycle hooks). For more information, see
     *         <a href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html">
     *         DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DescribeWarmPool
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DescribeWarmPool" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeWarmPoolResponse> describeWarmPool(DescribeWarmPoolRequest describeWarmPoolRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeWarmPoolRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeWarmPoolRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeWarmPool");

            HttpResponseHandler<DescribeWarmPoolResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeWarmPoolResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeWarmPoolResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeWarmPoolRequest, DescribeWarmPoolResponse>()
                            .withOperationName("DescribeWarmPool")
                            .withMarshaller(new DescribeWarmPoolRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeWarmPoolRequest));
            CompletableFuture<DescribeWarmPoolResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more instances from the specified Auto Scaling group.
     * </p>
     * <p>
     * After the instances are detached, you can manage them independent of the Auto Scaling group.
     * </p>
     * <p>
     * If you do not specify the option to decrement the desired capacity, Amazon EC2 Auto Scaling launches instances to
     * replace the ones that are detached.
     * </p>
     * <p>
     * If there is a Classic Load Balancer attached to the Auto Scaling group, the instances are deregistered from the
     * load balancer. If there are target groups attached to the Auto Scaling group, the instances are deregistered from
     * the target groups.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/detach-instance-asg.html">Detach EC2 instances from
     * your Auto Scaling group</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param detachInstancesRequest
     * @return A Java Future containing the result of the DetachInstances operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DetachInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DetachInstances" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DetachInstancesResponse> detachInstances(DetachInstancesRequest detachInstancesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(detachInstancesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, detachInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DetachInstances");

            HttpResponseHandler<DetachInstancesResponse> responseHandler = protocolFactory
                    .createResponseHandler(DetachInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DetachInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DetachInstancesRequest, DetachInstancesResponse>()
                            .withOperationName("DetachInstances")
                            .withMarshaller(new DetachInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(detachInstancesRequest));
            CompletableFuture<DetachInstancesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This API operation is superseded by <a>DetachTrafficSources</a>, which can detach multiple traffic sources types.
     * We recommend using <code>DetachTrafficSources</code> to simplify how you manage traffic sources. However, we
     * continue to support <code>DetachLoadBalancerTargetGroups</code>. You can use both the original
     * <code>DetachLoadBalancerTargetGroups</code> API operation and <code>DetachTrafficSources</code> on the same Auto
     * Scaling group.
     * </p>
     * </note>
     * <p>
     * Detaches one or more target groups from the specified Auto Scaling group.
     * </p>
     * <p>
     * When you detach a target group, it enters the <code>Removing</code> state while deregistering the instances in
     * the group. When all instances are deregistered, then you can no longer describe the target group using the
     * <a>DescribeLoadBalancerTargetGroups</a> API call. The instances remain running.
     * </p>
     * <note>
     * <p>
     * You can use this operation to detach target groups that were attached by using
     * <a>AttachLoadBalancerTargetGroups</a>, but not for target groups that were attached by using
     * <a>AttachTrafficSources</a>.
     * </p>
     * </note>
     *
     * @param detachLoadBalancerTargetGroupsRequest
     * @return A Java Future containing the result of the DetachLoadBalancerTargetGroups operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DetachLoadBalancerTargetGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DetachLoadBalancerTargetGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DetachLoadBalancerTargetGroupsResponse> detachLoadBalancerTargetGroups(
            DetachLoadBalancerTargetGroupsRequest detachLoadBalancerTargetGroupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(detachLoadBalancerTargetGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                detachLoadBalancerTargetGroupsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DetachLoadBalancerTargetGroups");

            HttpResponseHandler<DetachLoadBalancerTargetGroupsResponse> responseHandler = protocolFactory
                    .createResponseHandler(DetachLoadBalancerTargetGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DetachLoadBalancerTargetGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DetachLoadBalancerTargetGroupsRequest, DetachLoadBalancerTargetGroupsResponse>()
                            .withOperationName("DetachLoadBalancerTargetGroups")
                            .withMarshaller(new DetachLoadBalancerTargetGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(detachLoadBalancerTargetGroupsRequest));
            CompletableFuture<DetachLoadBalancerTargetGroupsResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This API operation is superseded by <a>DetachTrafficSources</a>, which can detach multiple traffic sources types.
     * We recommend using <code>DetachTrafficSources</code> to simplify how you manage traffic sources. However, we
     * continue to support <code>DetachLoadBalancers</code>. You can use both the original
     * <code>DetachLoadBalancers</code> API operation and <code>DetachTrafficSources</code> on the same Auto Scaling
     * group.
     * </p>
     * </note>
     * <p>
     * Detaches one or more Classic Load Balancers from the specified Auto Scaling group.
     * </p>
     * <p>
     * This operation detaches only Classic Load Balancers. If you have Application Load Balancers, Network Load
     * Balancers, or Gateway Load Balancers, use the <a>DetachLoadBalancerTargetGroups</a> API instead.
     * </p>
     * <p>
     * When you detach a load balancer, it enters the <code>Removing</code> state while deregistering the instances in
     * the group. When all instances are deregistered, then you can no longer describe the load balancer using the
     * <a>DescribeLoadBalancers</a> API call. The instances remain running.
     * </p>
     *
     * @param detachLoadBalancersRequest
     * @return A Java Future containing the result of the DetachLoadBalancers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DetachLoadBalancers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DetachLoadBalancers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DetachLoadBalancersResponse> detachLoadBalancers(
            DetachLoadBalancersRequest detachLoadBalancersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(detachLoadBalancersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, detachLoadBalancersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DetachLoadBalancers");

            HttpResponseHandler<DetachLoadBalancersResponse> responseHandler = protocolFactory
                    .createResponseHandler(DetachLoadBalancersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DetachLoadBalancersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DetachLoadBalancersRequest, DetachLoadBalancersResponse>()
                            .withOperationName("DetachLoadBalancers")
                            .withMarshaller(new DetachLoadBalancersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(detachLoadBalancersRequest));
            CompletableFuture<DetachLoadBalancersResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Detaches one or more traffic sources from the specified Auto Scaling group.
     * </p>
     * <p>
     * When you detach a traffic source, it enters the <code>Removing</code> state while deregistering the instances in
     * the group. When all instances are deregistered, then you can no longer describe the traffic source using the
     * <a>DescribeTrafficSources</a> API call. The instances continue to run.
     * </p>
     *
     * @param detachTrafficSourcesRequest
     * @return A Java Future containing the result of the DetachTrafficSources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DetachTrafficSources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DetachTrafficSources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DetachTrafficSourcesResponse> detachTrafficSources(
            DetachTrafficSourcesRequest detachTrafficSourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(detachTrafficSourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, detachTrafficSourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DetachTrafficSources");

            HttpResponseHandler<DetachTrafficSourcesResponse> responseHandler = protocolFactory
                    .createResponseHandler(DetachTrafficSourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DetachTrafficSourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DetachTrafficSourcesRequest, DetachTrafficSourcesResponse>()
                            .withOperationName("DetachTrafficSources")
                            .withMarshaller(new DetachTrafficSourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(detachTrafficSourcesRequest));
            CompletableFuture<DetachTrafficSourcesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables group metrics collection for the specified Auto Scaling group.
     * </p>
     *
     * @param disableMetricsCollectionRequest
     * @return A Java Future containing the result of the DisableMetricsCollection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.DisableMetricsCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/DisableMetricsCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisableMetricsCollectionResponse> disableMetricsCollection(
            DisableMetricsCollectionRequest disableMetricsCollectionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disableMetricsCollectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableMetricsCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableMetricsCollection");

            HttpResponseHandler<DisableMetricsCollectionResponse> responseHandler = protocolFactory
                    .createResponseHandler(DisableMetricsCollectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DisableMetricsCollectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableMetricsCollectionRequest, DisableMetricsCollectionResponse>()
                            .withOperationName("DisableMetricsCollection")
                            .withMarshaller(new DisableMetricsCollectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disableMetricsCollectionRequest));
            CompletableFuture<DisableMetricsCollectionResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables group metrics collection for the specified Auto Scaling group.
     * </p>
     * <p>
     * You can use these metrics to track changes in an Auto Scaling group and to set alarms on threshold values. You
     * can view group metrics using the Amazon EC2 Auto Scaling console or the CloudWatch console. For more information,
     * see <a href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-cloudwatch-monitoring.html">
     * Monitor CloudWatch metrics for your Auto Scaling groups and instances</a> in the <i>Amazon EC2 Auto Scaling User
     * Guide</i>.
     * </p>
     *
     * @param enableMetricsCollectionRequest
     * @return A Java Future containing the result of the EnableMetricsCollection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.EnableMetricsCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/EnableMetricsCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableMetricsCollectionResponse> enableMetricsCollection(
            EnableMetricsCollectionRequest enableMetricsCollectionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableMetricsCollectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableMetricsCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableMetricsCollection");

            HttpResponseHandler<EnableMetricsCollectionResponse> responseHandler = protocolFactory
                    .createResponseHandler(EnableMetricsCollectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<EnableMetricsCollectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableMetricsCollectionRequest, EnableMetricsCollectionResponse>()
                            .withOperationName("EnableMetricsCollection")
                            .withMarshaller(new EnableMetricsCollectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(enableMetricsCollectionRequest));
            CompletableFuture<EnableMetricsCollectionResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Moves the specified instances into the standby state.
     * </p>
     * <p>
     * If you choose to decrement the desired capacity of the Auto Scaling group, the instances can enter standby as
     * long as the desired capacity of the Auto Scaling group after the instances are placed into standby is equal to or
     * greater than the minimum capacity of the group.
     * </p>
     * <p>
     * If you choose not to decrement the desired capacity of the Auto Scaling group, the Auto Scaling group launches
     * new instances to replace the instances on standby.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-enter-exit-standby.html">Temporarily removing
     * instances from your Auto Scaling group</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param enterStandbyRequest
     * @return A Java Future containing the result of the EnterStandby operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.EnterStandby
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/EnterStandby" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<EnterStandbyResponse> enterStandby(EnterStandbyRequest enterStandbyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enterStandbyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enterStandbyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnterStandby");

            HttpResponseHandler<EnterStandbyResponse> responseHandler = protocolFactory
                    .createResponseHandler(EnterStandbyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<EnterStandbyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnterStandbyRequest, EnterStandbyResponse>()
                            .withOperationName("EnterStandby").withMarshaller(new EnterStandbyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(enterStandbyRequest));
            CompletableFuture<EnterStandbyResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Executes the specified policy. This can be useful for testing the design of your scaling policy.
     * </p>
     *
     * @param executePolicyRequest
     * @return A Java Future containing the result of the ExecutePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ScalingActivityInProgressException The operation can't be performed because there are scaling
     *         activities in progress.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.ExecutePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/ExecutePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ExecutePolicyResponse> executePolicy(ExecutePolicyRequest executePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(executePolicyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, executePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExecutePolicy");

            HttpResponseHandler<ExecutePolicyResponse> responseHandler = protocolFactory
                    .createResponseHandler(ExecutePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ExecutePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExecutePolicyRequest, ExecutePolicyResponse>()
                            .withOperationName("ExecutePolicy")
                            .withMarshaller(new ExecutePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(executePolicyRequest));
            CompletableFuture<ExecutePolicyResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Moves the specified instances out of the standby state.
     * </p>
     * <p>
     * After you put the instances back in service, the desired capacity is incremented.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-enter-exit-standby.html">Temporarily removing
     * instances from your Auto Scaling group</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param exitStandbyRequest
     * @return A Java Future containing the result of the ExitStandby operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.ExitStandby
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/ExitStandby" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ExitStandbyResponse> exitStandby(ExitStandbyRequest exitStandbyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exitStandbyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exitStandbyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExitStandby");

            HttpResponseHandler<ExitStandbyResponse> responseHandler = protocolFactory
                    .createResponseHandler(ExitStandbyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ExitStandbyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExitStandbyRequest, ExitStandbyResponse>()
                            .withOperationName("ExitStandby").withMarshaller(new ExitStandbyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(exitStandbyRequest));
            CompletableFuture<ExitStandbyResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the forecast data for a predictive scaling policy.
     * </p>
     * <p>
     * Load forecasts are predictions of the hourly load values using historical load data from CloudWatch and an
     * analysis of historical trends. Capacity forecasts are represented as predicted values for the minimum capacity
     * that is needed on an hourly basis, based on the hourly load forecast.
     * </p>
     * <p>
     * A minimum of 24 hours of data is required to create the initial forecasts. However, having a full 14 days of
     * historical data results in more accurate forecasts.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-predictive-scaling.html">Predictive
     * scaling for Amazon EC2 Auto Scaling</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param getPredictiveScalingForecastRequest
     * @return A Java Future containing the result of the GetPredictiveScalingForecast operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.GetPredictiveScalingForecast
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/GetPredictiveScalingForecast"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPredictiveScalingForecastResponse> getPredictiveScalingForecast(
            GetPredictiveScalingForecastRequest getPredictiveScalingForecastRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPredictiveScalingForecastRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPredictiveScalingForecastRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPredictiveScalingForecast");

            HttpResponseHandler<GetPredictiveScalingForecastResponse> responseHandler = protocolFactory
                    .createResponseHandler(GetPredictiveScalingForecastResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetPredictiveScalingForecastResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPredictiveScalingForecastRequest, GetPredictiveScalingForecastResponse>()
                            .withOperationName("GetPredictiveScalingForecast")
                            .withMarshaller(new GetPredictiveScalingForecastRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPredictiveScalingForecastRequest));
            CompletableFuture<GetPredictiveScalingForecastResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates a lifecycle hook for the specified Auto Scaling group.
     * </p>
     * <p>
     * Lifecycle hooks let you create solutions that are aware of events in the Auto Scaling instance lifecycle, and
     * then perform a custom action on instances when the corresponding lifecycle event occurs.
     * </p>
     * <p>
     * This step is a part of the procedure for adding a lifecycle hook to an Auto Scaling group:
     * </p>
     * <ol>
     * <li>
     * <p>
     * (Optional) Create a launch template or launch configuration with a user data script that runs while an instance
     * is in a wait state due to a lifecycle hook.
     * </p>
     * </li>
     * <li>
     * <p>
     * (Optional) Create a Lambda function and a rule that allows Amazon EventBridge to invoke your Lambda function when
     * an instance is put into a wait state due to a lifecycle hook.
     * </p>
     * </li>
     * <li>
     * <p>
     * (Optional) Create a notification target and an IAM role. The target can be either an Amazon SQS queue or an
     * Amazon SNS topic. The role allows Amazon EC2 Auto Scaling to publish lifecycle notifications to the target.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Create the lifecycle hook. Specify whether the hook is used when the instances launch or terminate.</b>
     * </p>
     * </li>
     * <li>
     * <p>
     * If you need more time, record the lifecycle action heartbeat to keep the instance in a wait state using the
     * <a>RecordLifecycleActionHeartbeat</a> API call.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you finish before the timeout period ends, send a callback by using the <a>CompleteLifecycleAction</a> API
     * call.
     * </p>
     * </li>
     * </ol>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/lifecycle-hooks.html">Amazon EC2 Auto Scaling
     * lifecycle hooks</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * If you exceed your maximum limit of lifecycle hooks, which by default is 50 per Auto Scaling group, the call
     * fails.
     * </p>
     * <p>
     * You can view the lifecycle hooks for an Auto Scaling group using the <a>DescribeLifecycleHooks</a> API call. If
     * you are no longer using a lifecycle hook, you can delete it by calling the <a>DeleteLifecycleHook</a> API.
     * </p>
     *
     * @param putLifecycleHookRequest
     * @return A Java Future containing the result of the PutLifecycleHook operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You have already reached a limit for your Amazon EC2 Auto Scaling resources
     *         (for example, Auto Scaling groups, launch configurations, or lifecycle hooks). For more information, see
     *         <a href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html">
     *         DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.PutLifecycleHook
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/PutLifecycleHook" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutLifecycleHookResponse> putLifecycleHook(PutLifecycleHookRequest putLifecycleHookRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putLifecycleHookRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putLifecycleHookRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutLifecycleHook");

            HttpResponseHandler<PutLifecycleHookResponse> responseHandler = protocolFactory
                    .createResponseHandler(PutLifecycleHookResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<PutLifecycleHookResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutLifecycleHookRequest, PutLifecycleHookResponse>()
                            .withOperationName("PutLifecycleHook")
                            .withMarshaller(new PutLifecycleHookRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putLifecycleHookRequest));
            CompletableFuture<PutLifecycleHookResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Configures an Auto Scaling group to send notifications when specified events take place. Subscribers to the
     * specified topic can have messages delivered to an endpoint such as a web server or an email address.
     * </p>
     * <p>
     * This configuration overwrites any existing configuration.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/ASGettingNotifications.html">Getting Amazon SNS
     * notifications when your Auto Scaling group scales</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * If you exceed your maximum limit of SNS topics, which is 10 per Auto Scaling group, the call fails.
     * </p>
     *
     * @param putNotificationConfigurationRequest
     * @return A Java Future containing the result of the PutNotificationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You have already reached a limit for your Amazon EC2 Auto Scaling resources
     *         (for example, Auto Scaling groups, launch configurations, or lifecycle hooks). For more information, see
     *         <a href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html">
     *         DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>ServiceLinkedRoleFailureException The service-linked role is not yet ready for use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.PutNotificationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/PutNotificationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutNotificationConfigurationResponse> putNotificationConfiguration(
            PutNotificationConfigurationRequest putNotificationConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putNotificationConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putNotificationConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutNotificationConfiguration");

            HttpResponseHandler<PutNotificationConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(PutNotificationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<PutNotificationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutNotificationConfigurationRequest, PutNotificationConfigurationResponse>()
                            .withOperationName("PutNotificationConfiguration")
                            .withMarshaller(new PutNotificationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putNotificationConfigurationRequest));
            CompletableFuture<PutNotificationConfigurationResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates a scaling policy for an Auto Scaling group. Scaling policies are used to scale an Auto Scaling
     * group based on configurable metrics. If no policies are defined, the dynamic scaling and predictive scaling
     * features are not used.
     * </p>
     * <p>
     * For more information about using dynamic scaling, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-scaling-target-tracking.html">Target tracking
     * scaling policies</a> and <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-scaling-simple-step.html">Step and simple scaling
     * policies</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * For more information about using predictive scaling, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-predictive-scaling.html">Predictive
     * scaling for Amazon EC2 Auto Scaling</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * You can view the scaling policies for an Auto Scaling group using the <a>DescribePolicies</a> API call. If you
     * are no longer using a scaling policy, you can delete it by calling the <a>DeletePolicy</a> API.
     * </p>
     *
     * @param putScalingPolicyRequest
     * @return A Java Future containing the result of the PutScalingPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You have already reached a limit for your Amazon EC2 Auto Scaling resources
     *         (for example, Auto Scaling groups, launch configurations, or lifecycle hooks). For more information, see
     *         <a href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html">
     *         DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>ServiceLinkedRoleFailureException The service-linked role is not yet ready for use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.PutScalingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/PutScalingPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutScalingPolicyResponse> putScalingPolicy(PutScalingPolicyRequest putScalingPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putScalingPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putScalingPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutScalingPolicy");

            HttpResponseHandler<PutScalingPolicyResponse> responseHandler = protocolFactory
                    .createResponseHandler(PutScalingPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<PutScalingPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutScalingPolicyRequest, PutScalingPolicyResponse>()
                            .withOperationName("PutScalingPolicy")
                            .withMarshaller(new PutScalingPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putScalingPolicyRequest));
            CompletableFuture<PutScalingPolicyResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates a scheduled scaling action for an Auto Scaling group.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/schedule_time.html">Scheduled scaling</a> in the
     * <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * You can view the scheduled actions for an Auto Scaling group using the <a>DescribeScheduledActions</a> API call.
     * If you are no longer using a scheduled action, you can delete it by calling the <a>DeleteScheduledAction</a> API.
     * </p>
     * <p>
     * If you try to schedule your action in the past, Amazon EC2 Auto Scaling returns an error message.
     * </p>
     *
     * @param putScheduledUpdateGroupActionRequest
     * @return A Java Future containing the result of the PutScheduledUpdateGroupAction operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AlreadyExistsException You already have an Auto Scaling group or launch configuration with this name.
     *         </li>
     *         <li>LimitExceededException You have already reached a limit for your Amazon EC2 Auto Scaling resources
     *         (for example, Auto Scaling groups, launch configurations, or lifecycle hooks). For more information, see
     *         <a href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html">
     *         DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.PutScheduledUpdateGroupAction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/PutScheduledUpdateGroupAction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutScheduledUpdateGroupActionResponse> putScheduledUpdateGroupAction(
            PutScheduledUpdateGroupActionRequest putScheduledUpdateGroupActionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putScheduledUpdateGroupActionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putScheduledUpdateGroupActionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutScheduledUpdateGroupAction");

            HttpResponseHandler<PutScheduledUpdateGroupActionResponse> responseHandler = protocolFactory
                    .createResponseHandler(PutScheduledUpdateGroupActionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<PutScheduledUpdateGroupActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutScheduledUpdateGroupActionRequest, PutScheduledUpdateGroupActionResponse>()
                            .withOperationName("PutScheduledUpdateGroupAction")
                            .withMarshaller(new PutScheduledUpdateGroupActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putScheduledUpdateGroupActionRequest));
            CompletableFuture<PutScheduledUpdateGroupActionResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates a warm pool for the specified Auto Scaling group. A warm pool is a pool of pre-initialized EC2
     * instances that sits alongside the Auto Scaling group. Whenever your application needs to scale out, the Auto
     * Scaling group can draw on the warm pool to meet its new desired capacity. For more information and example
     * configurations, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-warm-pools.html">Warm pools for
     * Amazon EC2 Auto Scaling</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * This operation must be called from the Region in which the Auto Scaling group was created. This operation cannot
     * be called on an Auto Scaling group that has a mixed instances policy or a launch template or launch configuration
     * that requests Spot Instances.
     * </p>
     * <p>
     * You can view the instances in the warm pool using the <a>DescribeWarmPool</a> API call. If you are no longer
     * using a warm pool, you can delete it by calling the <a>DeleteWarmPool</a> API.
     * </p>
     *
     * @param putWarmPoolRequest
     * @return A Java Future containing the result of the PutWarmPool operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You have already reached a limit for your Amazon EC2 Auto Scaling resources
     *         (for example, Auto Scaling groups, launch configurations, or lifecycle hooks). For more information, see
     *         <a href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html">
     *         DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.PutWarmPool
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/PutWarmPool" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutWarmPoolResponse> putWarmPool(PutWarmPoolRequest putWarmPoolRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putWarmPoolRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putWarmPoolRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutWarmPool");

            HttpResponseHandler<PutWarmPoolResponse> responseHandler = protocolFactory
                    .createResponseHandler(PutWarmPoolResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<PutWarmPoolResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutWarmPoolRequest, PutWarmPoolResponse>()
                            .withOperationName("PutWarmPool").withMarshaller(new PutWarmPoolRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putWarmPoolRequest));
            CompletableFuture<PutWarmPoolResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Records a heartbeat for the lifecycle action associated with the specified token or instance. This extends the
     * timeout by the length of time defined using the <a>PutLifecycleHook</a> API call.
     * </p>
     * <p>
     * This step is a part of the procedure for adding a lifecycle hook to an Auto Scaling group:
     * </p>
     * <ol>
     * <li>
     * <p>
     * (Optional) Create a launch template or launch configuration with a user data script that runs while an instance
     * is in a wait state due to a lifecycle hook.
     * </p>
     * </li>
     * <li>
     * <p>
     * (Optional) Create a Lambda function and a rule that allows Amazon EventBridge to invoke your Lambda function when
     * an instance is put into a wait state due to a lifecycle hook.
     * </p>
     * </li>
     * <li>
     * <p>
     * (Optional) Create a notification target and an IAM role. The target can be either an Amazon SQS queue or an
     * Amazon SNS topic. The role allows Amazon EC2 Auto Scaling to publish lifecycle notifications to the target.
     * </p>
     * </li>
     * <li>
     * <p>
     * Create the lifecycle hook. Specify whether the hook is used when the instances launch or terminate.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>If you need more time, record the lifecycle action heartbeat to keep the instance in a wait state.</b>
     * </p>
     * </li>
     * <li>
     * <p>
     * If you finish before the timeout period ends, send a callback by using the <a>CompleteLifecycleAction</a> API
     * call.
     * </p>
     * </li>
     * </ol>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/lifecycle-hooks.html">Amazon EC2 Auto Scaling
     * lifecycle hooks</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param recordLifecycleActionHeartbeatRequest
     * @return A Java Future containing the result of the RecordLifecycleActionHeartbeat operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.RecordLifecycleActionHeartbeat
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/RecordLifecycleActionHeartbeat"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RecordLifecycleActionHeartbeatResponse> recordLifecycleActionHeartbeat(
            RecordLifecycleActionHeartbeatRequest recordLifecycleActionHeartbeatRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(recordLifecycleActionHeartbeatRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                recordLifecycleActionHeartbeatRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RecordLifecycleActionHeartbeat");

            HttpResponseHandler<RecordLifecycleActionHeartbeatResponse> responseHandler = protocolFactory
                    .createResponseHandler(RecordLifecycleActionHeartbeatResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<RecordLifecycleActionHeartbeatResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RecordLifecycleActionHeartbeatRequest, RecordLifecycleActionHeartbeatResponse>()
                            .withOperationName("RecordLifecycleActionHeartbeat")
                            .withMarshaller(new RecordLifecycleActionHeartbeatRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(recordLifecycleActionHeartbeatRequest));
            CompletableFuture<RecordLifecycleActionHeartbeatResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Resumes the specified suspended auto scaling processes, or all suspended process, for the specified Auto Scaling
     * group.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-suspend-resume-processes.html">Suspending and
     * resuming scaling processes</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param resumeProcessesRequest
     * @return A Java Future containing the result of the ResumeProcesses operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException The operation can't be performed because the resource is in use.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.ResumeProcesses
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/ResumeProcesses" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ResumeProcessesResponse> resumeProcesses(ResumeProcessesRequest resumeProcessesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(resumeProcessesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resumeProcessesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResumeProcesses");

            HttpResponseHandler<ResumeProcessesResponse> responseHandler = protocolFactory
                    .createResponseHandler(ResumeProcessesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ResumeProcessesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ResumeProcessesRequest, ResumeProcessesResponse>()
                            .withOperationName("ResumeProcesses")
                            .withMarshaller(new ResumeProcessesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(resumeProcessesRequest));
            CompletableFuture<ResumeProcessesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels an instance refresh that is in progress and rolls back any changes that it made. Amazon EC2 Auto Scaling
     * replaces any instances that were replaced during the instance refresh. This restores your Auto Scaling group to
     * the configuration that it was using before the start of the instance refresh.
     * </p>
     * <p>
     * This operation is part of the <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-instance-refresh.html">instance refresh
     * feature</a> in Amazon EC2 Auto Scaling, which helps you update instances in your Auto Scaling group after you
     * make configuration changes.
     * </p>
     * <p>
     * A rollback is not supported in the following situations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * There is no desired configuration specified for the instance refresh.
     * </p>
     * </li>
     * <li>
     * <p>
     * The Auto Scaling group has a launch template that uses an Amazon Web Services Systems Manager parameter instead
     * of an AMI ID for the <code>ImageId</code> property.
     * </p>
     * </li>
     * <li>
     * <p>
     * The Auto Scaling group uses the launch template's <code>&#36Latest</code> or <code>&#36Default</code> version.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When you receive a successful response from this operation, Amazon EC2 Auto Scaling immediately begins replacing
     * instances. You can check the status of this operation through the <a>DescribeInstanceRefreshes</a> API operation.
     * </p>
     *
     * @param rollbackInstanceRefreshRequest
     * @return A Java Future containing the result of the RollbackInstanceRefresh operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You have already reached a limit for your Amazon EC2 Auto Scaling resources
     *         (for example, Auto Scaling groups, launch configurations, or lifecycle hooks). For more information, see
     *         <a href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html">
     *         DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>ActiveInstanceRefreshNotFoundException The request failed because an active instance refresh or
     *         rollback for the specified Auto Scaling group was not found.</li>
     *         <li>IrreversibleInstanceRefreshException The request failed because a desired configuration was not found
     *         or an incompatible launch template (uses a Systems Manager parameter instead of an AMI ID) or launch
     *         template version (<code>&#36Latest</code> or <code>&#36Default</code>) is present on the Auto Scaling
     *         group.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.RollbackInstanceRefresh
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/RollbackInstanceRefresh"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RollbackInstanceRefreshResponse> rollbackInstanceRefresh(
            RollbackInstanceRefreshRequest rollbackInstanceRefreshRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(rollbackInstanceRefreshRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rollbackInstanceRefreshRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RollbackInstanceRefresh");

            HttpResponseHandler<RollbackInstanceRefreshResponse> responseHandler = protocolFactory
                    .createResponseHandler(RollbackInstanceRefreshResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<RollbackInstanceRefreshResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RollbackInstanceRefreshRequest, RollbackInstanceRefreshResponse>()
                            .withOperationName("RollbackInstanceRefresh")
                            .withMarshaller(new RollbackInstanceRefreshRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(rollbackInstanceRefreshRequest));
            CompletableFuture<RollbackInstanceRefreshResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the size of the specified Auto Scaling group.
     * </p>
     * <p>
     * If a scale-in activity occurs as a result of a new <code>DesiredCapacity</code> value that is lower than the
     * current size of the group, the Auto Scaling group uses its termination policy to determine which instances to
     * terminate.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-manual-scaling.html">Manual scaling</a> in the
     * <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param setDesiredCapacityRequest
     * @return A Java Future containing the result of the SetDesiredCapacity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ScalingActivityInProgressException The operation can't be performed because there are scaling
     *         activities in progress.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.SetDesiredCapacity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/SetDesiredCapacity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SetDesiredCapacityResponse> setDesiredCapacity(SetDesiredCapacityRequest setDesiredCapacityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setDesiredCapacityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setDesiredCapacityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetDesiredCapacity");

            HttpResponseHandler<SetDesiredCapacityResponse> responseHandler = protocolFactory
                    .createResponseHandler(SetDesiredCapacityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<SetDesiredCapacityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetDesiredCapacityRequest, SetDesiredCapacityResponse>()
                            .withOperationName("SetDesiredCapacity")
                            .withMarshaller(new SetDesiredCapacityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(setDesiredCapacityRequest));
            CompletableFuture<SetDesiredCapacityResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the health status of the specified instance.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/healthcheck.html">Health
     * checks for Auto Scaling instances</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param setInstanceHealthRequest
     * @return A Java Future containing the result of the SetInstanceHealth operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.SetInstanceHealth
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/SetInstanceHealth" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<SetInstanceHealthResponse> setInstanceHealth(SetInstanceHealthRequest setInstanceHealthRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setInstanceHealthRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setInstanceHealthRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetInstanceHealth");

            HttpResponseHandler<SetInstanceHealthResponse> responseHandler = protocolFactory
                    .createResponseHandler(SetInstanceHealthResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<SetInstanceHealthResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetInstanceHealthRequest, SetInstanceHealthResponse>()
                            .withOperationName("SetInstanceHealth")
                            .withMarshaller(new SetInstanceHealthRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(setInstanceHealthRequest));
            CompletableFuture<SetInstanceHealthResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the instance protection settings of the specified instances. This operation cannot be called on instances
     * in a warm pool.
     * </p>
     * <p>
     * For more information about preventing instances that are part of an Auto Scaling group from terminating on scale
     * in, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-instance-protection.html">Using
     * instance scale-in protection</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * If you exceed your maximum limit of instance IDs, which is 50 per Auto Scaling group, the call fails.
     * </p>
     *
     * @param setInstanceProtectionRequest
     * @return A Java Future containing the result of the SetInstanceProtection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You have already reached a limit for your Amazon EC2 Auto Scaling resources
     *         (for example, Auto Scaling groups, launch configurations, or lifecycle hooks). For more information, see
     *         <a href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html">
     *         DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.SetInstanceProtection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/SetInstanceProtection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SetInstanceProtectionResponse> setInstanceProtection(
            SetInstanceProtectionRequest setInstanceProtectionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setInstanceProtectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setInstanceProtectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetInstanceProtection");

            HttpResponseHandler<SetInstanceProtectionResponse> responseHandler = protocolFactory
                    .createResponseHandler(SetInstanceProtectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<SetInstanceProtectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetInstanceProtectionRequest, SetInstanceProtectionResponse>()
                            .withOperationName("SetInstanceProtection")
                            .withMarshaller(new SetInstanceProtectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(setInstanceProtectionRequest));
            CompletableFuture<SetInstanceProtectionResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts an instance refresh. During an instance refresh, Amazon EC2 Auto Scaling performs a rolling update of
     * instances in an Auto Scaling group. Instances are terminated first and then replaced, which temporarily reduces
     * the capacity available within your Auto Scaling group.
     * </p>
     * <p>
     * This operation is part of the <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-instance-refresh.html">instance refresh
     * feature</a> in Amazon EC2 Auto Scaling, which helps you update instances in your Auto Scaling group. This feature
     * is helpful, for example, when you have a new AMI or a new user data script. You just need to create a new launch
     * template that specifies the new AMI or user data script. Then start an instance refresh to immediately begin the
     * process of updating instances in the group.
     * </p>
     * <p>
     * If successful, the request's response contains a unique ID that you can use to track the progress of the instance
     * refresh. To query its status, call the <a>DescribeInstanceRefreshes</a> API. To describe the instance refreshes
     * that have already run, call the <a>DescribeInstanceRefreshes</a> API. To cancel an instance refresh that is in
     * progress, use the <a>CancelInstanceRefresh</a> API.
     * </p>
     * <p>
     * An instance refresh might fail for several reasons, such as EC2 launch failures, misconfigured health checks, or
     * not ignoring or allowing the termination of instances that are in <code>Standby</code> state or protected from
     * scale in. You can monitor for failed EC2 launches using the scaling activities. To find the scaling activities,
     * call the <a>DescribeScalingActivities</a> API.
     * </p>
     * <p>
     * If you enable auto rollback, your Auto Scaling group will be rolled back automatically when the instance refresh
     * fails. You can enable this feature before starting an instance refresh by specifying the
     * <code>AutoRollback</code> property in the instance refresh preferences. Otherwise, to roll back an instance
     * refresh before it finishes, use the <a>RollbackInstanceRefresh</a> API.
     * </p>
     *
     * @param startInstanceRefreshRequest
     * @return A Java Future containing the result of the StartInstanceRefresh operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You have already reached a limit for your Amazon EC2 Auto Scaling resources
     *         (for example, Auto Scaling groups, launch configurations, or lifecycle hooks). For more information, see
     *         <a href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_DescribeAccountLimits.html">
     *         DescribeAccountLimits</a> in the <i>Amazon EC2 Auto Scaling API Reference</i>.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>InstanceRefreshInProgressException The request failed because an active instance refresh already
     *         exists for the specified Auto Scaling group.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.StartInstanceRefresh
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/StartInstanceRefresh"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartInstanceRefreshResponse> startInstanceRefresh(
            StartInstanceRefreshRequest startInstanceRefreshRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startInstanceRefreshRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startInstanceRefreshRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartInstanceRefresh");

            HttpResponseHandler<StartInstanceRefreshResponse> responseHandler = protocolFactory
                    .createResponseHandler(StartInstanceRefreshResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<StartInstanceRefreshResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartInstanceRefreshRequest, StartInstanceRefreshResponse>()
                            .withOperationName("StartInstanceRefresh")
                            .withMarshaller(new StartInstanceRefreshRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startInstanceRefreshRequest));
            CompletableFuture<StartInstanceRefreshResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Suspends the specified auto scaling processes, or all processes, for the specified Auto Scaling group.
     * </p>
     * <p>
     * If you suspend either the <code>Launch</code> or <code>Terminate</code> process types, it can prevent other
     * process types from functioning properly. For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-suspend-resume-processes.html">Suspending and
     * resuming scaling processes</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * To resume processes that have been suspended, call the <a>ResumeProcesses</a> API.
     * </p>
     *
     * @param suspendProcessesRequest
     * @return A Java Future containing the result of the SuspendProcesses operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException The operation can't be performed because the resource is in use.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.SuspendProcesses
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/SuspendProcesses" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<SuspendProcessesResponse> suspendProcesses(SuspendProcessesRequest suspendProcessesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(suspendProcessesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, suspendProcessesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SuspendProcesses");

            HttpResponseHandler<SuspendProcessesResponse> responseHandler = protocolFactory
                    .createResponseHandler(SuspendProcessesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<SuspendProcessesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SuspendProcessesRequest, SuspendProcessesResponse>()
                            .withOperationName("SuspendProcesses")
                            .withMarshaller(new SuspendProcessesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(suspendProcessesRequest));
            CompletableFuture<SuspendProcessesResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Terminates the specified instance and optionally adjusts the desired group size. This operation cannot be called
     * on instances in a warm pool.
     * </p>
     * <p>
     * This call simply makes a termination request. The instance is not terminated immediately. When an instance is
     * terminated, the instance status changes to <code>terminated</code>. You can't connect to or start an instance
     * after you've terminated it.
     * </p>
     * <p>
     * If you do not specify the option to decrement the desired capacity, Amazon EC2 Auto Scaling launches instances to
     * replace the ones that are terminated.
     * </p>
     * <p>
     * By default, Amazon EC2 Auto Scaling balances instances across all Availability Zones. If you decrement the
     * desired capacity, your Auto Scaling group can become unbalanced between Availability Zones. Amazon EC2 Auto
     * Scaling tries to rebalance the group, and rebalancing might terminate instances in other zones. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/ec2/userguide/auto-scaling-benefits.html#AutoScalingBehavior.InstanceUsage"
     * >Rebalancing activities</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     *
     * @param terminateInstanceInAutoScalingGroupRequest
     * @return A Java Future containing the result of the TerminateInstanceInAutoScalingGroup operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ScalingActivityInProgressException The operation can't be performed because there are scaling
     *         activities in progress.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.TerminateInstanceInAutoScalingGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/TerminateInstanceInAutoScalingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TerminateInstanceInAutoScalingGroupResponse> terminateInstanceInAutoScalingGroup(
            TerminateInstanceInAutoScalingGroupRequest terminateInstanceInAutoScalingGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(terminateInstanceInAutoScalingGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                terminateInstanceInAutoScalingGroupRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TerminateInstanceInAutoScalingGroup");

            HttpResponseHandler<TerminateInstanceInAutoScalingGroupResponse> responseHandler = protocolFactory
                    .createResponseHandler(TerminateInstanceInAutoScalingGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<TerminateInstanceInAutoScalingGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TerminateInstanceInAutoScalingGroupRequest, TerminateInstanceInAutoScalingGroupResponse>()
                            .withOperationName("TerminateInstanceInAutoScalingGroup")
                            .withMarshaller(new TerminateInstanceInAutoScalingGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(terminateInstanceInAutoScalingGroupRequest));
            CompletableFuture<TerminateInstanceInAutoScalingGroupResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <b>We strongly recommend that all Auto Scaling groups use launch templates to ensure full functionality for
     * Amazon EC2 Auto Scaling and Amazon EC2.</b>
     * </p>
     * <p>
     * Updates the configuration for the specified Auto Scaling group.
     * </p>
     * <p>
     * To update an Auto Scaling group, specify the name of the group and the property that you want to change. Any
     * properties that you don't specify are not changed by this update request. The new settings take effect on any
     * scaling activities after this call returns.
     * </p>
     * <p>
     * If you associate a new launch configuration or template with an Auto Scaling group, all new instances will get
     * the updated configuration. Existing instances continue to run with the configuration that they were originally
     * launched with. When you update a group to specify a mixed instances policy instead of a launch configuration or
     * template, existing instances may be replaced to match the new purchasing options that you specified in the
     * policy. For example, if the group currently has 100% On-Demand capacity and the policy specifies 50% Spot
     * capacity, this means that half of your instances will be gradually terminated and relaunched as Spot Instances.
     * When replacing instances, Amazon EC2 Auto Scaling launches new instances before terminating the old ones, so that
     * updating your group does not compromise the performance or availability of your application.
     * </p>
     * <p>
     * Note the following about changing <code>DesiredCapacity</code>, <code>MaxSize</code>, or <code>MinSize</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If a scale-in activity occurs as a result of a new <code>DesiredCapacity</code> value that is lower than the
     * current size of the group, the Auto Scaling group uses its termination policy to determine which instances to
     * terminate.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify a new value for <code>MinSize</code> without specifying a value for <code>DesiredCapacity</code>,
     * and the new <code>MinSize</code> is larger than the current size of the group, this sets the group's
     * <code>DesiredCapacity</code> to the new <code>MinSize</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify a new value for <code>MaxSize</code> without specifying a value for <code>DesiredCapacity</code>,
     * and the new <code>MaxSize</code> is smaller than the current size of the group, this sets the group's
     * <code>DesiredCapacity</code> to the new <code>MaxSize</code> value.
     * </p>
     * </li>
     * </ul>
     * <p>
     * To see which properties have been set, call the <a>DescribeAutoScalingGroups</a> API. To view the scaling
     * policies for an Auto Scaling group, call the <a>DescribePolicies</a> API. If the group has scaling policies, you
     * can update them by calling the <a>PutScalingPolicy</a> API.
     * </p>
     *
     * @param updateAutoScalingGroupRequest
     * @return A Java Future containing the result of the UpdateAutoScalingGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ScalingActivityInProgressException The operation can't be performed because there are scaling
     *         activities in progress.</li>
     *         <li>ResourceContentionException You already have a pending update to an Amazon EC2 Auto Scaling resource
     *         (for example, an Auto Scaling group, instance, or load balancer).</li>
     *         <li>ServiceLinkedRoleFailureException The service-linked role is not yet ready for use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AutoScalingAsyncClient.UpdateAutoScalingGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/UpdateAutoScalingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAutoScalingGroupResponse> updateAutoScalingGroup(
            UpdateAutoScalingGroupRequest updateAutoScalingGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAutoScalingGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAutoScalingGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAutoScalingGroup");

            HttpResponseHandler<UpdateAutoScalingGroupResponse> responseHandler = protocolFactory
                    .createResponseHandler(UpdateAutoScalingGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<UpdateAutoScalingGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAutoScalingGroupRequest, UpdateAutoScalingGroupResponse>()
                            .withOperationName("UpdateAutoScalingGroup")
                            .withMarshaller(new UpdateAutoScalingGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAutoScalingGroupRequest));
            CompletableFuture<UpdateAutoScalingGroupResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public AutoScalingAsyncWaiter waiter() {
        return AutoScalingAsyncWaiter.builder().client(this).scheduledExecutorService(executorService).build();
    }

    @Override
    public final AutoScalingServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private AwsQueryProtocolFactory init() {
        return AwsQueryProtocolFactory
                .builder()
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InstanceRefreshInProgress")
                                .exceptionBuilderSupplier(InstanceRefreshInProgressException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ScalingActivityInProgress")
                                .exceptionBuilderSupplier(ScalingActivityInProgressException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextToken")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceeded")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AlreadyExists")
                                .exceptionBuilderSupplier(AlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ActiveInstanceRefreshNotFound")
                                .exceptionBuilderSupplier(ActiveInstanceRefreshNotFoundException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceContention")
                                .exceptionBuilderSupplier(ResourceContentionException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceLinkedRoleFailure")
                                .exceptionBuilderSupplier(ServiceLinkedRoleFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUse")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IrreversibleInstanceRefresh")
                                .exceptionBuilderSupplier(IrreversibleInstanceRefreshException::builder).httpStatusCode(400)
                                .build()).clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AutoScalingException::builder).build();
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        AutoScalingServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = AutoScalingServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
