/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscaling.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.autoscaling.AutoScalingClient;
import software.amazon.awssdk.services.autoscaling.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.autoscaling.model.AutoScalingRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingGroupsRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingGroupsResponse;
import software.amazon.awssdk.services.autoscaling.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultAutoScalingWaiter implements AutoScalingWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final AutoScalingClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeAutoScalingGroupsResponse> groupExistsWaiter;

    private final Waiter<DescribeAutoScalingGroupsResponse> groupInServiceWaiter;

    private final Waiter<DescribeAutoScalingGroupsResponse> groupNotExistsWaiter;

    private DefaultAutoScalingWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = AutoScalingClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.groupExistsWaiter = Waiter.builder(DescribeAutoScalingGroupsResponse.class).acceptors(groupExistsWaiterAcceptors())
                .overrideConfiguration(groupExistsWaiterConfig(builder.overrideConfiguration)).build();
        this.groupInServiceWaiter = Waiter.builder(DescribeAutoScalingGroupsResponse.class)
                .acceptors(groupInServiceWaiterAcceptors())
                .overrideConfiguration(groupInServiceWaiterConfig(builder.overrideConfiguration)).build();
        this.groupNotExistsWaiter = Waiter.builder(DescribeAutoScalingGroupsResponse.class)
                .acceptors(groupNotExistsWaiterAcceptors())
                .overrideConfiguration(groupNotExistsWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeAutoScalingGroupsResponse> waitUntilGroupExists(
            DescribeAutoScalingGroupsRequest describeAutoScalingGroupsRequest) {
        return groupExistsWaiter.run(() -> client
                .describeAutoScalingGroups(applyWaitersUserAgent(describeAutoScalingGroupsRequest)));
    }

    @Override
    public WaiterResponse<DescribeAutoScalingGroupsResponse> waitUntilGroupExists(
            DescribeAutoScalingGroupsRequest describeAutoScalingGroupsRequest, WaiterOverrideConfiguration overrideConfig) {
        return groupExistsWaiter.run(
                () -> client.describeAutoScalingGroups(applyWaitersUserAgent(describeAutoScalingGroupsRequest)),
                groupExistsWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeAutoScalingGroupsResponse> waitUntilGroupInService(
            DescribeAutoScalingGroupsRequest describeAutoScalingGroupsRequest) {
        return groupInServiceWaiter.run(() -> client
                .describeAutoScalingGroups(applyWaitersUserAgent(describeAutoScalingGroupsRequest)));
    }

    @Override
    public WaiterResponse<DescribeAutoScalingGroupsResponse> waitUntilGroupInService(
            DescribeAutoScalingGroupsRequest describeAutoScalingGroupsRequest, WaiterOverrideConfiguration overrideConfig) {
        return groupInServiceWaiter.run(
                () -> client.describeAutoScalingGroups(applyWaitersUserAgent(describeAutoScalingGroupsRequest)),
                groupInServiceWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeAutoScalingGroupsResponse> waitUntilGroupNotExists(
            DescribeAutoScalingGroupsRequest describeAutoScalingGroupsRequest) {
        return groupNotExistsWaiter.run(() -> client
                .describeAutoScalingGroups(applyWaitersUserAgent(describeAutoScalingGroupsRequest)));
    }

    @Override
    public WaiterResponse<DescribeAutoScalingGroupsResponse> waitUntilGroupNotExists(
            DescribeAutoScalingGroupsRequest describeAutoScalingGroupsRequest, WaiterOverrideConfiguration overrideConfig) {
        return groupNotExistsWaiter.run(
                () -> client.describeAutoScalingGroups(applyWaitersUserAgent(describeAutoScalingGroupsRequest)),
                groupNotExistsWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeAutoScalingGroupsResponse>> groupExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAutoScalingGroupsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("AutoScalingGroups").length().compare(">", input.constant(0)).value(), true);
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("AutoScalingGroups").length().compare(">", input.constant(0)).value(), false);
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeAutoScalingGroupsResponse>> groupInServiceWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAutoScalingGroupsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(
                    input.field("AutoScalingGroups")
                            .flatten()
                            .multiSelectList(
                                    x0 -> x0.field("Instances")
                                            .filter(x1 -> x1.field("LifecycleState").compare("==", x1.constant("InService")))
                                            .length().compare(">=", x0.field("MinSize"))).flatten()
                            .contains(input.constant(false)).value(), false);
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(
                    input.field("AutoScalingGroups")
                            .flatten()
                            .multiSelectList(
                                    x0 -> x0.field("Instances")
                                            .filter(x1 -> x1.field("LifecycleState").compare("==", x1.constant("InService")))
                                            .length().compare(">=", x0.field("MinSize"))).flatten()
                            .contains(input.constant(false)).value(), true);
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeAutoScalingGroupsResponse>> groupNotExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAutoScalingGroupsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("AutoScalingGroups").length().compare(">", input.constant(0)).value(), false);
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("AutoScalingGroups").length().compare(">", input.constant(0)).value(), true);
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration groupExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(10);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration groupInServiceWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration groupNotExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static AutoScalingWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends AutoScalingRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements AutoScalingWaiter.Builder {
        private AutoScalingClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public AutoScalingWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public AutoScalingWaiter.Builder client(AutoScalingClient client) {
            this.client = client;
            return this;
        }

        public AutoScalingWaiter build() {
            return new DefaultAutoScalingWaiter(this);
        }
    }
}
