/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscaling.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Use this structure to specify the distribution of On-Demand Instances and Spot Instances and the allocation
 * strategies used to fulfill On-Demand and Spot capacities for a mixed instances policy.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstancesDistribution implements SdkPojo, Serializable,
        ToCopyableBuilder<InstancesDistribution.Builder, InstancesDistribution> {
    private static final SdkField<String> ON_DEMAND_ALLOCATION_STRATEGY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OnDemandAllocationStrategy")
            .getter(getter(InstancesDistribution::onDemandAllocationStrategy))
            .setter(setter(Builder::onDemandAllocationStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OnDemandAllocationStrategy").build())
            .build();

    private static final SdkField<Integer> ON_DEMAND_BASE_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("OnDemandBaseCapacity").getter(getter(InstancesDistribution::onDemandBaseCapacity))
            .setter(setter(Builder::onDemandBaseCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OnDemandBaseCapacity").build())
            .build();

    private static final SdkField<Integer> ON_DEMAND_PERCENTAGE_ABOVE_BASE_CAPACITY_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("OnDemandPercentageAboveBaseCapacity")
            .getter(getter(InstancesDistribution::onDemandPercentageAboveBaseCapacity))
            .setter(setter(Builder::onDemandPercentageAboveBaseCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("OnDemandPercentageAboveBaseCapacity").build()).build();

    private static final SdkField<String> SPOT_ALLOCATION_STRATEGY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SpotAllocationStrategy").getter(getter(InstancesDistribution::spotAllocationStrategy))
            .setter(setter(Builder::spotAllocationStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SpotAllocationStrategy").build())
            .build();

    private static final SdkField<Integer> SPOT_INSTANCE_POOLS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("SpotInstancePools").getter(getter(InstancesDistribution::spotInstancePools))
            .setter(setter(Builder::spotInstancePools))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SpotInstancePools").build()).build();

    private static final SdkField<String> SPOT_MAX_PRICE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SpotMaxPrice").getter(getter(InstancesDistribution::spotMaxPrice)).setter(setter(Builder::spotMaxPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SpotMaxPrice").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            ON_DEMAND_ALLOCATION_STRATEGY_FIELD, ON_DEMAND_BASE_CAPACITY_FIELD, ON_DEMAND_PERCENTAGE_ABOVE_BASE_CAPACITY_FIELD,
            SPOT_ALLOCATION_STRATEGY_FIELD, SPOT_INSTANCE_POOLS_FIELD, SPOT_MAX_PRICE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("OnDemandAllocationStrategy", ON_DEMAND_ALLOCATION_STRATEGY_FIELD);
                    put("OnDemandBaseCapacity", ON_DEMAND_BASE_CAPACITY_FIELD);
                    put("OnDemandPercentageAboveBaseCapacity", ON_DEMAND_PERCENTAGE_ABOVE_BASE_CAPACITY_FIELD);
                    put("SpotAllocationStrategy", SPOT_ALLOCATION_STRATEGY_FIELD);
                    put("SpotInstancePools", SPOT_INSTANCE_POOLS_FIELD);
                    put("SpotMaxPrice", SPOT_MAX_PRICE_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String onDemandAllocationStrategy;

    private final Integer onDemandBaseCapacity;

    private final Integer onDemandPercentageAboveBaseCapacity;

    private final String spotAllocationStrategy;

    private final Integer spotInstancePools;

    private final String spotMaxPrice;

    private InstancesDistribution(BuilderImpl builder) {
        this.onDemandAllocationStrategy = builder.onDemandAllocationStrategy;
        this.onDemandBaseCapacity = builder.onDemandBaseCapacity;
        this.onDemandPercentageAboveBaseCapacity = builder.onDemandPercentageAboveBaseCapacity;
        this.spotAllocationStrategy = builder.spotAllocationStrategy;
        this.spotInstancePools = builder.spotInstancePools;
        this.spotMaxPrice = builder.spotMaxPrice;
    }

    /**
     * <p>
     * The allocation strategy to apply to your On-Demand Instances when they are launched. Possible instance types are
     * determined by the launch template overrides that you specify.
     * </p>
     * <p>
     * The following lists the valid values:
     * </p>
     * <dl>
     * <dt>lowest-price</dt>
     * <dd>
     * <p>
     * Uses price to determine which instance types are the highest priority, launching the lowest priced instance types
     * within an Availability Zone first. This is the default value for Auto Scaling groups that specify <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_InstanceRequirements.html"
     * >InstanceRequirements</a>.
     * </p>
     * </dd>
     * <dt>prioritized</dt>
     * <dd>
     * <p>
     * You set the order of instance types for the launch template overrides from highest to lowest priority (from first
     * to last in the list). Amazon EC2 Auto Scaling launches your highest priority instance types first. If all your
     * On-Demand capacity cannot be fulfilled using your highest priority instance type, then Amazon EC2 Auto Scaling
     * launches the remaining capacity using the second priority instance type, and so on. This is the default value for
     * Auto Scaling groups that don't specify <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_InstanceRequirements.html"
     * >InstanceRequirements</a> and cannot be used for groups that do.
     * </p>
     * </dd>
     * </dl>
     * 
     * @return The allocation strategy to apply to your On-Demand Instances when they are launched. Possible instance
     *         types are determined by the launch template overrides that you specify.</p>
     *         <p>
     *         The following lists the valid values:
     *         </p>
     *         <dl>
     *         <dt>lowest-price</dt>
     *         <dd>
     *         <p>
     *         Uses price to determine which instance types are the highest priority, launching the lowest priced
     *         instance types within an Availability Zone first. This is the default value for Auto Scaling groups that
     *         specify <a href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_InstanceRequirements.html">
     *         InstanceRequirements</a>.
     *         </p>
     *         </dd>
     *         <dt>prioritized</dt>
     *         <dd>
     *         <p>
     *         You set the order of instance types for the launch template overrides from highest to lowest priority
     *         (from first to last in the list). Amazon EC2 Auto Scaling launches your highest priority instance types
     *         first. If all your On-Demand capacity cannot be fulfilled using your highest priority instance type, then
     *         Amazon EC2 Auto Scaling launches the remaining capacity using the second priority instance type, and so
     *         on. This is the default value for Auto Scaling groups that don't specify <a
     *         href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_InstanceRequirements.html"
     *         >InstanceRequirements</a> and cannot be used for groups that do.
     *         </p>
     *         </dd>
     */
    public final String onDemandAllocationStrategy() {
        return onDemandAllocationStrategy;
    }

    /**
     * <p>
     * The minimum amount of the Auto Scaling group's capacity that must be fulfilled by On-Demand Instances. This base
     * portion is launched first as your group scales.
     * </p>
     * <p>
     * This number has the same unit of measurement as the group's desired capacity. If you change the default unit of
     * measurement (number of instances) by specifying weighted capacity values in your launch template overrides list,
     * or by changing the default desired capacity type setting of the group, you must specify this number using the
     * same unit of measurement.
     * </p>
     * <p>
     * Default: 0
     * </p>
     * 
     * @return The minimum amount of the Auto Scaling group's capacity that must be fulfilled by On-Demand Instances.
     *         This base portion is launched first as your group scales.</p>
     *         <p>
     *         This number has the same unit of measurement as the group's desired capacity. If you change the default
     *         unit of measurement (number of instances) by specifying weighted capacity values in your launch template
     *         overrides list, or by changing the default desired capacity type setting of the group, you must specify
     *         this number using the same unit of measurement.
     *         </p>
     *         <p>
     *         Default: 0
     */
    public final Integer onDemandBaseCapacity() {
        return onDemandBaseCapacity;
    }

    /**
     * <p>
     * Controls the percentages of On-Demand Instances and Spot Instances for your additional capacity beyond
     * <code>OnDemandBaseCapacity</code>. Expressed as a number (for example, 20 specifies 20% On-Demand Instances, 80%
     * Spot Instances). If set to 100, only On-Demand Instances are used.
     * </p>
     * <p>
     * Default: 100
     * </p>
     * 
     * @return Controls the percentages of On-Demand Instances and Spot Instances for your additional capacity beyond
     *         <code>OnDemandBaseCapacity</code>. Expressed as a number (for example, 20 specifies 20% On-Demand
     *         Instances, 80% Spot Instances). If set to 100, only On-Demand Instances are used.</p>
     *         <p>
     *         Default: 100
     */
    public final Integer onDemandPercentageAboveBaseCapacity() {
        return onDemandPercentageAboveBaseCapacity;
    }

    /**
     * <p>
     * The allocation strategy to apply to your Spot Instances when they are launched. Possible instance types are
     * determined by the launch template overrides that you specify.
     * </p>
     * <p>
     * The following lists the valid values:
     * </p>
     * <dl>
     * <dt>capacity-optimized</dt>
     * <dd>
     * <p>
     * Requests Spot Instances using pools that are optimally chosen based on the available Spot capacity. This strategy
     * has the lowest risk of interruption. To give certain instance types a higher chance of launching first, use
     * <code>capacity-optimized-prioritized</code>.
     * </p>
     * </dd>
     * <dt>capacity-optimized-prioritized</dt>
     * <dd>
     * <p>
     * You set the order of instance types for the launch template overrides from highest to lowest priority (from first
     * to last in the list). Amazon EC2 Auto Scaling honors the instance type priorities on a best effort basis but
     * optimizes for capacity first. Note that if the On-Demand allocation strategy is set to <code>prioritized</code>,
     * the same priority is applied when fulfilling On-Demand capacity. This is not a valid value for Auto Scaling
     * groups that specify <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_InstanceRequirements.html"
     * >InstanceRequirements</a>.
     * </p>
     * </dd>
     * <dt>lowest-price</dt>
     * <dd>
     * <p>
     * Requests Spot Instances using the lowest priced pools within an Availability Zone, across the number of Spot
     * pools that you specify for the <code>SpotInstancePools</code> property. To ensure that your desired capacity is
     * met, you might receive Spot Instances from several pools. This is the default value, but it might lead to high
     * interruption rates because this strategy only considers instance price and not available capacity.
     * </p>
     * </dd>
     * <dt>price-capacity-optimized (recommended)</dt>
     * <dd>
     * <p>
     * The price and capacity optimized allocation strategy looks at both price and capacity to select the Spot Instance
     * pools that are the least likely to be interrupted and have the lowest possible price.
     * </p>
     * </dd>
     * </dl>
     * 
     * @return The allocation strategy to apply to your Spot Instances when they are launched. Possible instance types
     *         are determined by the launch template overrides that you specify.</p>
     *         <p>
     *         The following lists the valid values:
     *         </p>
     *         <dl>
     *         <dt>capacity-optimized</dt>
     *         <dd>
     *         <p>
     *         Requests Spot Instances using pools that are optimally chosen based on the available Spot capacity. This
     *         strategy has the lowest risk of interruption. To give certain instance types a higher chance of launching
     *         first, use <code>capacity-optimized-prioritized</code>.
     *         </p>
     *         </dd>
     *         <dt>capacity-optimized-prioritized</dt>
     *         <dd>
     *         <p>
     *         You set the order of instance types for the launch template overrides from highest to lowest priority
     *         (from first to last in the list). Amazon EC2 Auto Scaling honors the instance type priorities on a best
     *         effort basis but optimizes for capacity first. Note that if the On-Demand allocation strategy is set to
     *         <code>prioritized</code>, the same priority is applied when fulfilling On-Demand capacity. This is not a
     *         valid value for Auto Scaling groups that specify <a
     *         href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_InstanceRequirements.html"
     *         >InstanceRequirements</a>.
     *         </p>
     *         </dd>
     *         <dt>lowest-price</dt>
     *         <dd>
     *         <p>
     *         Requests Spot Instances using the lowest priced pools within an Availability Zone, across the number of
     *         Spot pools that you specify for the <code>SpotInstancePools</code> property. To ensure that your desired
     *         capacity is met, you might receive Spot Instances from several pools. This is the default value, but it
     *         might lead to high interruption rates because this strategy only considers instance price and not
     *         available capacity.
     *         </p>
     *         </dd>
     *         <dt>price-capacity-optimized (recommended)</dt>
     *         <dd>
     *         <p>
     *         The price and capacity optimized allocation strategy looks at both price and capacity to select the Spot
     *         Instance pools that are the least likely to be interrupted and have the lowest possible price.
     *         </p>
     *         </dd>
     */
    public final String spotAllocationStrategy() {
        return spotAllocationStrategy;
    }

    /**
     * <p>
     * The number of Spot Instance pools across which to allocate your Spot Instances. The Spot pools are determined
     * from the different instance types in the overrides. Valid only when the <code>SpotAllocationStrategy</code> is
     * <code>lowest-price</code>. Value must be in the range of 1–20.
     * </p>
     * <p>
     * Default: 2
     * </p>
     * 
     * @return The number of Spot Instance pools across which to allocate your Spot Instances. The Spot pools are
     *         determined from the different instance types in the overrides. Valid only when the
     *         <code>SpotAllocationStrategy</code> is <code>lowest-price</code>. Value must be in the range of 1–20.</p>
     *         <p>
     *         Default: 2
     */
    public final Integer spotInstancePools() {
        return spotInstancePools;
    }

    /**
     * <p>
     * The maximum price per unit hour that you are willing to pay for a Spot Instance. If your maximum price is lower
     * than the Spot price for the instance types that you selected, your Spot Instances are not launched. We do not
     * recommend specifying a maximum price because it can lead to increased interruptions. When Spot Instances launch,
     * you pay the current Spot price. To remove a maximum price that you previously set, include the property but
     * specify an empty string ("") for the value.
     * </p>
     * <important>
     * <p>
     * If you specify a maximum price, your instances will be interrupted more frequently than if you do not specify
     * one.
     * </p>
     * </important>
     * <p>
     * Valid Range: Minimum value of 0.001
     * </p>
     * 
     * @return The maximum price per unit hour that you are willing to pay for a Spot Instance. If your maximum price is
     *         lower than the Spot price for the instance types that you selected, your Spot Instances are not launched.
     *         We do not recommend specifying a maximum price because it can lead to increased interruptions. When Spot
     *         Instances launch, you pay the current Spot price. To remove a maximum price that you previously set,
     *         include the property but specify an empty string ("") for the value.</p> <important>
     *         <p>
     *         If you specify a maximum price, your instances will be interrupted more frequently than if you do not
     *         specify one.
     *         </p>
     *         </important>
     *         <p>
     *         Valid Range: Minimum value of 0.001
     */
    public final String spotMaxPrice() {
        return spotMaxPrice;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(onDemandAllocationStrategy());
        hashCode = 31 * hashCode + Objects.hashCode(onDemandBaseCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(onDemandPercentageAboveBaseCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(spotAllocationStrategy());
        hashCode = 31 * hashCode + Objects.hashCode(spotInstancePools());
        hashCode = 31 * hashCode + Objects.hashCode(spotMaxPrice());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstancesDistribution)) {
            return false;
        }
        InstancesDistribution other = (InstancesDistribution) obj;
        return Objects.equals(onDemandAllocationStrategy(), other.onDemandAllocationStrategy())
                && Objects.equals(onDemandBaseCapacity(), other.onDemandBaseCapacity())
                && Objects.equals(onDemandPercentageAboveBaseCapacity(), other.onDemandPercentageAboveBaseCapacity())
                && Objects.equals(spotAllocationStrategy(), other.spotAllocationStrategy())
                && Objects.equals(spotInstancePools(), other.spotInstancePools())
                && Objects.equals(spotMaxPrice(), other.spotMaxPrice());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstancesDistribution").add("OnDemandAllocationStrategy", onDemandAllocationStrategy())
                .add("OnDemandBaseCapacity", onDemandBaseCapacity())
                .add("OnDemandPercentageAboveBaseCapacity", onDemandPercentageAboveBaseCapacity())
                .add("SpotAllocationStrategy", spotAllocationStrategy()).add("SpotInstancePools", spotInstancePools())
                .add("SpotMaxPrice", spotMaxPrice()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OnDemandAllocationStrategy":
            return Optional.ofNullable(clazz.cast(onDemandAllocationStrategy()));
        case "OnDemandBaseCapacity":
            return Optional.ofNullable(clazz.cast(onDemandBaseCapacity()));
        case "OnDemandPercentageAboveBaseCapacity":
            return Optional.ofNullable(clazz.cast(onDemandPercentageAboveBaseCapacity()));
        case "SpotAllocationStrategy":
            return Optional.ofNullable(clazz.cast(spotAllocationStrategy()));
        case "SpotInstancePools":
            return Optional.ofNullable(clazz.cast(spotInstancePools()));
        case "SpotMaxPrice":
            return Optional.ofNullable(clazz.cast(spotMaxPrice()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<InstancesDistribution, T> g) {
        return obj -> g.apply((InstancesDistribution) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstancesDistribution> {
        /**
         * <p>
         * The allocation strategy to apply to your On-Demand Instances when they are launched. Possible instance types
         * are determined by the launch template overrides that you specify.
         * </p>
         * <p>
         * The following lists the valid values:
         * </p>
         * <dl>
         * <dt>lowest-price</dt>
         * <dd>
         * <p>
         * Uses price to determine which instance types are the highest priority, launching the lowest priced instance
         * types within an Availability Zone first. This is the default value for Auto Scaling groups that specify <a
         * href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_InstanceRequirements.html">
         * InstanceRequirements</a>.
         * </p>
         * </dd>
         * <dt>prioritized</dt>
         * <dd>
         * <p>
         * You set the order of instance types for the launch template overrides from highest to lowest priority (from
         * first to last in the list). Amazon EC2 Auto Scaling launches your highest priority instance types first. If
         * all your On-Demand capacity cannot be fulfilled using your highest priority instance type, then Amazon EC2
         * Auto Scaling launches the remaining capacity using the second priority instance type, and so on. This is the
         * default value for Auto Scaling groups that don't specify <a
         * href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_InstanceRequirements.html"
         * >InstanceRequirements</a> and cannot be used for groups that do.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param onDemandAllocationStrategy
         *        The allocation strategy to apply to your On-Demand Instances when they are launched. Possible instance
         *        types are determined by the launch template overrides that you specify.</p>
         *        <p>
         *        The following lists the valid values:
         *        </p>
         *        <dl>
         *        <dt>lowest-price</dt>
         *        <dd>
         *        <p>
         *        Uses price to determine which instance types are the highest priority, launching the lowest priced
         *        instance types within an Availability Zone first. This is the default value for Auto Scaling groups
         *        that specify <a
         *        href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_InstanceRequirements.html"
         *        >InstanceRequirements</a>.
         *        </p>
         *        </dd>
         *        <dt>prioritized</dt>
         *        <dd>
         *        <p>
         *        You set the order of instance types for the launch template overrides from highest to lowest priority
         *        (from first to last in the list). Amazon EC2 Auto Scaling launches your highest priority instance
         *        types first. If all your On-Demand capacity cannot be fulfilled using your highest priority instance
         *        type, then Amazon EC2 Auto Scaling launches the remaining capacity using the second priority instance
         *        type, and so on. This is the default value for Auto Scaling groups that don't specify <a
         *        href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_InstanceRequirements.html"
         *        >InstanceRequirements</a> and cannot be used for groups that do.
         *        </p>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder onDemandAllocationStrategy(String onDemandAllocationStrategy);

        /**
         * <p>
         * The minimum amount of the Auto Scaling group's capacity that must be fulfilled by On-Demand Instances. This
         * base portion is launched first as your group scales.
         * </p>
         * <p>
         * This number has the same unit of measurement as the group's desired capacity. If you change the default unit
         * of measurement (number of instances) by specifying weighted capacity values in your launch template overrides
         * list, or by changing the default desired capacity type setting of the group, you must specify this number
         * using the same unit of measurement.
         * </p>
         * <p>
         * Default: 0
         * </p>
         * 
         * @param onDemandBaseCapacity
         *        The minimum amount of the Auto Scaling group's capacity that must be fulfilled by On-Demand Instances.
         *        This base portion is launched first as your group scales.</p>
         *        <p>
         *        This number has the same unit of measurement as the group's desired capacity. If you change the
         *        default unit of measurement (number of instances) by specifying weighted capacity values in your
         *        launch template overrides list, or by changing the default desired capacity type setting of the group,
         *        you must specify this number using the same unit of measurement.
         *        </p>
         *        <p>
         *        Default: 0
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder onDemandBaseCapacity(Integer onDemandBaseCapacity);

        /**
         * <p>
         * Controls the percentages of On-Demand Instances and Spot Instances for your additional capacity beyond
         * <code>OnDemandBaseCapacity</code>. Expressed as a number (for example, 20 specifies 20% On-Demand Instances,
         * 80% Spot Instances). If set to 100, only On-Demand Instances are used.
         * </p>
         * <p>
         * Default: 100
         * </p>
         * 
         * @param onDemandPercentageAboveBaseCapacity
         *        Controls the percentages of On-Demand Instances and Spot Instances for your additional capacity beyond
         *        <code>OnDemandBaseCapacity</code>. Expressed as a number (for example, 20 specifies 20% On-Demand
         *        Instances, 80% Spot Instances). If set to 100, only On-Demand Instances are used.</p>
         *        <p>
         *        Default: 100
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder onDemandPercentageAboveBaseCapacity(Integer onDemandPercentageAboveBaseCapacity);

        /**
         * <p>
         * The allocation strategy to apply to your Spot Instances when they are launched. Possible instance types are
         * determined by the launch template overrides that you specify.
         * </p>
         * <p>
         * The following lists the valid values:
         * </p>
         * <dl>
         * <dt>capacity-optimized</dt>
         * <dd>
         * <p>
         * Requests Spot Instances using pools that are optimally chosen based on the available Spot capacity. This
         * strategy has the lowest risk of interruption. To give certain instance types a higher chance of launching
         * first, use <code>capacity-optimized-prioritized</code>.
         * </p>
         * </dd>
         * <dt>capacity-optimized-prioritized</dt>
         * <dd>
         * <p>
         * You set the order of instance types for the launch template overrides from highest to lowest priority (from
         * first to last in the list). Amazon EC2 Auto Scaling honors the instance type priorities on a best effort
         * basis but optimizes for capacity first. Note that if the On-Demand allocation strategy is set to
         * <code>prioritized</code>, the same priority is applied when fulfilling On-Demand capacity. This is not a
         * valid value for Auto Scaling groups that specify <a
         * href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_InstanceRequirements.html"
         * >InstanceRequirements</a>.
         * </p>
         * </dd>
         * <dt>lowest-price</dt>
         * <dd>
         * <p>
         * Requests Spot Instances using the lowest priced pools within an Availability Zone, across the number of Spot
         * pools that you specify for the <code>SpotInstancePools</code> property. To ensure that your desired capacity
         * is met, you might receive Spot Instances from several pools. This is the default value, but it might lead to
         * high interruption rates because this strategy only considers instance price and not available capacity.
         * </p>
         * </dd>
         * <dt>price-capacity-optimized (recommended)</dt>
         * <dd>
         * <p>
         * The price and capacity optimized allocation strategy looks at both price and capacity to select the Spot
         * Instance pools that are the least likely to be interrupted and have the lowest possible price.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param spotAllocationStrategy
         *        The allocation strategy to apply to your Spot Instances when they are launched. Possible instance
         *        types are determined by the launch template overrides that you specify.</p>
         *        <p>
         *        The following lists the valid values:
         *        </p>
         *        <dl>
         *        <dt>capacity-optimized</dt>
         *        <dd>
         *        <p>
         *        Requests Spot Instances using pools that are optimally chosen based on the available Spot capacity.
         *        This strategy has the lowest risk of interruption. To give certain instance types a higher chance of
         *        launching first, use <code>capacity-optimized-prioritized</code>.
         *        </p>
         *        </dd>
         *        <dt>capacity-optimized-prioritized</dt>
         *        <dd>
         *        <p>
         *        You set the order of instance types for the launch template overrides from highest to lowest priority
         *        (from first to last in the list). Amazon EC2 Auto Scaling honors the instance type priorities on a
         *        best effort basis but optimizes for capacity first. Note that if the On-Demand allocation strategy is
         *        set to <code>prioritized</code>, the same priority is applied when fulfilling On-Demand capacity. This
         *        is not a valid value for Auto Scaling groups that specify <a
         *        href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_InstanceRequirements.html"
         *        >InstanceRequirements</a>.
         *        </p>
         *        </dd>
         *        <dt>lowest-price</dt>
         *        <dd>
         *        <p>
         *        Requests Spot Instances using the lowest priced pools within an Availability Zone, across the number
         *        of Spot pools that you specify for the <code>SpotInstancePools</code> property. To ensure that your
         *        desired capacity is met, you might receive Spot Instances from several pools. This is the default
         *        value, but it might lead to high interruption rates because this strategy only considers instance
         *        price and not available capacity.
         *        </p>
         *        </dd>
         *        <dt>price-capacity-optimized (recommended)</dt>
         *        <dd>
         *        <p>
         *        The price and capacity optimized allocation strategy looks at both price and capacity to select the
         *        Spot Instance pools that are the least likely to be interrupted and have the lowest possible price.
         *        </p>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder spotAllocationStrategy(String spotAllocationStrategy);

        /**
         * <p>
         * The number of Spot Instance pools across which to allocate your Spot Instances. The Spot pools are determined
         * from the different instance types in the overrides. Valid only when the <code>SpotAllocationStrategy</code>
         * is <code>lowest-price</code>. Value must be in the range of 1–20.
         * </p>
         * <p>
         * Default: 2
         * </p>
         * 
         * @param spotInstancePools
         *        The number of Spot Instance pools across which to allocate your Spot Instances. The Spot pools are
         *        determined from the different instance types in the overrides. Valid only when the
         *        <code>SpotAllocationStrategy</code> is <code>lowest-price</code>. Value must be in the range of
         *        1–20.</p>
         *        <p>
         *        Default: 2
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder spotInstancePools(Integer spotInstancePools);

        /**
         * <p>
         * The maximum price per unit hour that you are willing to pay for a Spot Instance. If your maximum price is
         * lower than the Spot price for the instance types that you selected, your Spot Instances are not launched. We
         * do not recommend specifying a maximum price because it can lead to increased interruptions. When Spot
         * Instances launch, you pay the current Spot price. To remove a maximum price that you previously set, include
         * the property but specify an empty string ("") for the value.
         * </p>
         * <important>
         * <p>
         * If you specify a maximum price, your instances will be interrupted more frequently than if you do not specify
         * one.
         * </p>
         * </important>
         * <p>
         * Valid Range: Minimum value of 0.001
         * </p>
         * 
         * @param spotMaxPrice
         *        The maximum price per unit hour that you are willing to pay for a Spot Instance. If your maximum price
         *        is lower than the Spot price for the instance types that you selected, your Spot Instances are not
         *        launched. We do not recommend specifying a maximum price because it can lead to increased
         *        interruptions. When Spot Instances launch, you pay the current Spot price. To remove a maximum price
         *        that you previously set, include the property but specify an empty string ("") for the value.</p>
         *        <important>
         *        <p>
         *        If you specify a maximum price, your instances will be interrupted more frequently than if you do not
         *        specify one.
         *        </p>
         *        </important>
         *        <p>
         *        Valid Range: Minimum value of 0.001
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder spotMaxPrice(String spotMaxPrice);
    }

    static final class BuilderImpl implements Builder {
        private String onDemandAllocationStrategy;

        private Integer onDemandBaseCapacity;

        private Integer onDemandPercentageAboveBaseCapacity;

        private String spotAllocationStrategy;

        private Integer spotInstancePools;

        private String spotMaxPrice;

        private BuilderImpl() {
        }

        private BuilderImpl(InstancesDistribution model) {
            onDemandAllocationStrategy(model.onDemandAllocationStrategy);
            onDemandBaseCapacity(model.onDemandBaseCapacity);
            onDemandPercentageAboveBaseCapacity(model.onDemandPercentageAboveBaseCapacity);
            spotAllocationStrategy(model.spotAllocationStrategy);
            spotInstancePools(model.spotInstancePools);
            spotMaxPrice(model.spotMaxPrice);
        }

        public final String getOnDemandAllocationStrategy() {
            return onDemandAllocationStrategy;
        }

        public final void setOnDemandAllocationStrategy(String onDemandAllocationStrategy) {
            this.onDemandAllocationStrategy = onDemandAllocationStrategy;
        }

        @Override
        public final Builder onDemandAllocationStrategy(String onDemandAllocationStrategy) {
            this.onDemandAllocationStrategy = onDemandAllocationStrategy;
            return this;
        }

        public final Integer getOnDemandBaseCapacity() {
            return onDemandBaseCapacity;
        }

        public final void setOnDemandBaseCapacity(Integer onDemandBaseCapacity) {
            this.onDemandBaseCapacity = onDemandBaseCapacity;
        }

        @Override
        public final Builder onDemandBaseCapacity(Integer onDemandBaseCapacity) {
            this.onDemandBaseCapacity = onDemandBaseCapacity;
            return this;
        }

        public final Integer getOnDemandPercentageAboveBaseCapacity() {
            return onDemandPercentageAboveBaseCapacity;
        }

        public final void setOnDemandPercentageAboveBaseCapacity(Integer onDemandPercentageAboveBaseCapacity) {
            this.onDemandPercentageAboveBaseCapacity = onDemandPercentageAboveBaseCapacity;
        }

        @Override
        public final Builder onDemandPercentageAboveBaseCapacity(Integer onDemandPercentageAboveBaseCapacity) {
            this.onDemandPercentageAboveBaseCapacity = onDemandPercentageAboveBaseCapacity;
            return this;
        }

        public final String getSpotAllocationStrategy() {
            return spotAllocationStrategy;
        }

        public final void setSpotAllocationStrategy(String spotAllocationStrategy) {
            this.spotAllocationStrategy = spotAllocationStrategy;
        }

        @Override
        public final Builder spotAllocationStrategy(String spotAllocationStrategy) {
            this.spotAllocationStrategy = spotAllocationStrategy;
            return this;
        }

        public final Integer getSpotInstancePools() {
            return spotInstancePools;
        }

        public final void setSpotInstancePools(Integer spotInstancePools) {
            this.spotInstancePools = spotInstancePools;
        }

        @Override
        public final Builder spotInstancePools(Integer spotInstancePools) {
            this.spotInstancePools = spotInstancePools;
            return this;
        }

        public final String getSpotMaxPrice() {
            return spotMaxPrice;
        }

        public final void setSpotMaxPrice(String spotMaxPrice) {
            this.spotMaxPrice = spotMaxPrice;
        }

        @Override
        public final Builder spotMaxPrice(String spotMaxPrice) {
            this.spotMaxPrice = spotMaxPrice;
            return this;
        }

        @Override
        public InstancesDistribution build() {
            return new InstancesDistribution(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
