/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscaling.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the preferences for an instance refresh.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RefreshPreferences implements SdkPojo, Serializable,
        ToCopyableBuilder<RefreshPreferences.Builder, RefreshPreferences> {
    private static final SdkField<Integer> MIN_HEALTHY_PERCENTAGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MinHealthyPercentage").getter(getter(RefreshPreferences::minHealthyPercentage))
            .setter(setter(Builder::minHealthyPercentage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinHealthyPercentage").build())
            .build();

    private static final SdkField<Integer> INSTANCE_WARMUP_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("InstanceWarmup").getter(getter(RefreshPreferences::instanceWarmup))
            .setter(setter(Builder::instanceWarmup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceWarmup").build()).build();

    private static final SdkField<List<Integer>> CHECKPOINT_PERCENTAGES_FIELD = SdkField
            .<List<Integer>> builder(MarshallingType.LIST)
            .memberName("CheckpointPercentages")
            .getter(getter(RefreshPreferences::checkpointPercentages))
            .setter(setter(Builder::checkpointPercentages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CheckpointPercentages").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Integer> builder(MarshallingType.INTEGER)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> CHECKPOINT_DELAY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("CheckpointDelay").getter(getter(RefreshPreferences::checkpointDelay))
            .setter(setter(Builder::checkpointDelay))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CheckpointDelay").build()).build();

    private static final SdkField<Boolean> SKIP_MATCHING_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SkipMatching").getter(getter(RefreshPreferences::skipMatching)).setter(setter(Builder::skipMatching))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SkipMatching").build()).build();

    private static final SdkField<Boolean> AUTO_ROLLBACK_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AutoRollback").getter(getter(RefreshPreferences::autoRollback)).setter(setter(Builder::autoRollback))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoRollback").build()).build();

    private static final SdkField<String> SCALE_IN_PROTECTED_INSTANCES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScaleInProtectedInstances").getter(getter(RefreshPreferences::scaleInProtectedInstancesAsString))
            .setter(setter(Builder::scaleInProtectedInstances))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScaleInProtectedInstances").build())
            .build();

    private static final SdkField<String> STANDBY_INSTANCES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StandbyInstances").getter(getter(RefreshPreferences::standbyInstancesAsString))
            .setter(setter(Builder::standbyInstances))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StandbyInstances").build()).build();

    private static final SdkField<AlarmSpecification> ALARM_SPECIFICATION_FIELD = SdkField
            .<AlarmSpecification> builder(MarshallingType.SDK_POJO).memberName("AlarmSpecification")
            .getter(getter(RefreshPreferences::alarmSpecification)).setter(setter(Builder::alarmSpecification))
            .constructor(AlarmSpecification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlarmSpecification").build())
            .build();

    private static final SdkField<Integer> MAX_HEALTHY_PERCENTAGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxHealthyPercentage").getter(getter(RefreshPreferences::maxHealthyPercentage))
            .setter(setter(Builder::maxHealthyPercentage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxHealthyPercentage").build())
            .build();

    private static final SdkField<Integer> BAKE_TIME_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("BakeTime").getter(getter(RefreshPreferences::bakeTime)).setter(setter(Builder::bakeTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BakeTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MIN_HEALTHY_PERCENTAGE_FIELD,
            INSTANCE_WARMUP_FIELD, CHECKPOINT_PERCENTAGES_FIELD, CHECKPOINT_DELAY_FIELD, SKIP_MATCHING_FIELD,
            AUTO_ROLLBACK_FIELD, SCALE_IN_PROTECTED_INSTANCES_FIELD, STANDBY_INSTANCES_FIELD, ALARM_SPECIFICATION_FIELD,
            MAX_HEALTHY_PERCENTAGE_FIELD, BAKE_TIME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Integer minHealthyPercentage;

    private final Integer instanceWarmup;

    private final List<Integer> checkpointPercentages;

    private final Integer checkpointDelay;

    private final Boolean skipMatching;

    private final Boolean autoRollback;

    private final String scaleInProtectedInstances;

    private final String standbyInstances;

    private final AlarmSpecification alarmSpecification;

    private final Integer maxHealthyPercentage;

    private final Integer bakeTime;

    private RefreshPreferences(BuilderImpl builder) {
        this.minHealthyPercentage = builder.minHealthyPercentage;
        this.instanceWarmup = builder.instanceWarmup;
        this.checkpointPercentages = builder.checkpointPercentages;
        this.checkpointDelay = builder.checkpointDelay;
        this.skipMatching = builder.skipMatching;
        this.autoRollback = builder.autoRollback;
        this.scaleInProtectedInstances = builder.scaleInProtectedInstances;
        this.standbyInstances = builder.standbyInstances;
        this.alarmSpecification = builder.alarmSpecification;
        this.maxHealthyPercentage = builder.maxHealthyPercentage;
        this.bakeTime = builder.bakeTime;
    }

    /**
     * <p>
     * Specifies the minimum percentage of the group to keep in service, healthy, and ready to use to support your
     * workload to allow the operation to continue. The value is expressed as a percentage of the desired capacity of
     * the Auto Scaling group. Value range is 0 to 100.
     * </p>
     * <p>
     * If you do not specify this property, the default is 90 percent, or the percentage set in the instance maintenance
     * policy for the Auto Scaling group, if defined.
     * </p>
     * 
     * @return Specifies the minimum percentage of the group to keep in service, healthy, and ready to use to support
     *         your workload to allow the operation to continue. The value is expressed as a percentage of the desired
     *         capacity of the Auto Scaling group. Value range is 0 to 100.</p>
     *         <p>
     *         If you do not specify this property, the default is 90 percent, or the percentage set in the instance
     *         maintenance policy for the Auto Scaling group, if defined.
     */
    public final Integer minHealthyPercentage() {
        return minHealthyPercentage;
    }

    /**
     * <p>
     * A time period, in seconds, during which an instance refresh waits before moving on to replacing the next instance
     * after a new instance enters the <code>InService</code> state.
     * </p>
     * <p>
     * This property is not required for normal usage. Instead, use the <code>DefaultInstanceWarmup</code> property of
     * the Auto Scaling group. The <code>InstanceWarmup</code> and <code>DefaultInstanceWarmup</code> properties work
     * the same way. Only specify this property if you must override the <code>DefaultInstanceWarmup</code> property.
     * </p>
     * <p>
     * If you do not specify this property, the instance warmup by default is the value of the
     * <code>DefaultInstanceWarmup</code> property, if defined (which is recommended in all cases), or the
     * <code>HealthCheckGracePeriod</code> property otherwise.
     * </p>
     * 
     * @return A time period, in seconds, during which an instance refresh waits before moving on to replacing the next
     *         instance after a new instance enters the <code>InService</code> state.</p>
     *         <p>
     *         This property is not required for normal usage. Instead, use the <code>DefaultInstanceWarmup</code>
     *         property of the Auto Scaling group. The <code>InstanceWarmup</code> and
     *         <code>DefaultInstanceWarmup</code> properties work the same way. Only specify this property if you must
     *         override the <code>DefaultInstanceWarmup</code> property.
     *         </p>
     *         <p>
     *         If you do not specify this property, the instance warmup by default is the value of the
     *         <code>DefaultInstanceWarmup</code> property, if defined (which is recommended in all cases), or the
     *         <code>HealthCheckGracePeriod</code> property otherwise.
     */
    public final Integer instanceWarmup() {
        return instanceWarmup;
    }

    /**
     * For responses, this returns true if the service returned a value for the CheckpointPercentages property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasCheckpointPercentages() {
        return checkpointPercentages != null && !(checkpointPercentages instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * (Optional) Threshold values for each checkpoint in ascending order. Each number must be unique. To replace all
     * instances in the Auto Scaling group, the last number in the array must be <code>100</code>.
     * </p>
     * <p>
     * For usage examples, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-adding-checkpoints-instance-refresh.html">Add
     * checkpoints to an instance refresh</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCheckpointPercentages} method.
     * </p>
     * 
     * @return (Optional) Threshold values for each checkpoint in ascending order. Each number must be unique. To
     *         replace all instances in the Auto Scaling group, the last number in the array must be <code>100</code>
     *         .</p>
     *         <p>
     *         For usage examples, see <a href=
     *         "https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-adding-checkpoints-instance-refresh.html">Add
     *         checkpoints to an instance refresh</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     */
    public final List<Integer> checkpointPercentages() {
        return checkpointPercentages;
    }

    /**
     * <p>
     * (Optional) The amount of time, in seconds, to wait after a checkpoint before continuing. This property is
     * optional, but if you specify a value for it, you must also specify a value for <code>CheckpointPercentages</code>
     * . If you specify a value for <code>CheckpointPercentages</code> and not for <code>CheckpointDelay</code>, the
     * <code>CheckpointDelay</code> defaults to <code>3600</code> (1 hour).
     * </p>
     * 
     * @return (Optional) The amount of time, in seconds, to wait after a checkpoint before continuing. This property is
     *         optional, but if you specify a value for it, you must also specify a value for
     *         <code>CheckpointPercentages</code>. If you specify a value for <code>CheckpointPercentages</code> and not
     *         for <code>CheckpointDelay</code>, the <code>CheckpointDelay</code> defaults to <code>3600</code> (1
     *         hour).
     */
    public final Integer checkpointDelay() {
        return checkpointDelay;
    }

    /**
     * <p>
     * (Optional) Indicates whether skip matching is enabled. If enabled (<code>true</code>), then Amazon EC2 Auto
     * Scaling skips replacing instances that match the desired configuration. If no desired configuration is specified,
     * then it skips replacing instances that have the same launch template and instance types that the Auto Scaling
     * group was using before the start of the instance refresh. The default is <code>false</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-instance-refresh-skip-matching.html">Use an
     * instance refresh with skip matching</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * 
     * @return (Optional) Indicates whether skip matching is enabled. If enabled (<code>true</code>), then Amazon EC2
     *         Auto Scaling skips replacing instances that match the desired configuration. If no desired configuration
     *         is specified, then it skips replacing instances that have the same launch template and instance types
     *         that the Auto Scaling group was using before the start of the instance refresh. The default is
     *         <code>false</code>.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-instance-refresh-skip-matching.html">Use
     *         an instance refresh with skip matching</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     */
    public final Boolean skipMatching() {
        return skipMatching;
    }

    /**
     * <p>
     * (Optional) Indicates whether to roll back the Auto Scaling group to its previous configuration if the instance
     * refresh fails or a CloudWatch alarm threshold is met. The default is <code>false</code>.
     * </p>
     * <p>
     * A rollback is not supported in the following situations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * There is no desired configuration specified for the instance refresh.
     * </p>
     * </li>
     * <li>
     * <p>
     * The Auto Scaling group has a launch template that uses an Amazon Web Services Systems Manager parameter instead
     * of an AMI ID for the <code>ImageId</code> property.
     * </p>
     * </li>
     * <li>
     * <p>
     * The Auto Scaling group uses the launch template's <code>$Latest</code> or <code>$Default</code> version.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/instance-refresh-rollback.html">Undo changes with a
     * rollback</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     * </p>
     * 
     * @return (Optional) Indicates whether to roll back the Auto Scaling group to its previous configuration if the
     *         instance refresh fails or a CloudWatch alarm threshold is met. The default is <code>false</code>.</p>
     *         <p>
     *         A rollback is not supported in the following situations:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         There is no desired configuration specified for the instance refresh.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The Auto Scaling group has a launch template that uses an Amazon Web Services Systems Manager parameter
     *         instead of an AMI ID for the <code>ImageId</code> property.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The Auto Scaling group uses the launch template's <code>$Latest</code> or <code>$Default</code> version.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/instance-refresh-rollback.html">Undo changes
     *         with a rollback</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
     */
    public final Boolean autoRollback() {
        return autoRollback;
    }

    /**
     * <p>
     * Choose the behavior that you want Amazon EC2 Auto Scaling to use if instances protected from scale in are found.
     * </p>
     * <p>
     * The following lists the valid values:
     * </p>
     * <dl>
     * <dt>Refresh</dt>
     * <dd>
     * <p>
     * Amazon EC2 Auto Scaling replaces instances that are protected from scale in.
     * </p>
     * </dd>
     * <dt>Ignore</dt>
     * <dd>
     * <p>
     * Amazon EC2 Auto Scaling ignores instances that are protected from scale in and continues to replace instances
     * that are not protected.
     * </p>
     * </dd>
     * <dt>Wait (default)</dt>
     * <dd>
     * <p>
     * Amazon EC2 Auto Scaling waits one hour for you to remove scale-in protection. Otherwise, the instance refresh
     * will fail.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #scaleInProtectedInstances} will return {@link ScaleInProtectedInstances#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #scaleInProtectedInstancesAsString}.
     * </p>
     * 
     * @return Choose the behavior that you want Amazon EC2 Auto Scaling to use if instances protected from scale in are
     *         found. </p>
     *         <p>
     *         The following lists the valid values:
     *         </p>
     *         <dl>
     *         <dt>Refresh</dt>
     *         <dd>
     *         <p>
     *         Amazon EC2 Auto Scaling replaces instances that are protected from scale in.
     *         </p>
     *         </dd>
     *         <dt>Ignore</dt>
     *         <dd>
     *         <p>
     *         Amazon EC2 Auto Scaling ignores instances that are protected from scale in and continues to replace
     *         instances that are not protected.
     *         </p>
     *         </dd>
     *         <dt>Wait (default)</dt>
     *         <dd>
     *         <p>
     *         Amazon EC2 Auto Scaling waits one hour for you to remove scale-in protection. Otherwise, the instance
     *         refresh will fail.
     *         </p>
     *         </dd>
     * @see ScaleInProtectedInstances
     */
    public final ScaleInProtectedInstances scaleInProtectedInstances() {
        return ScaleInProtectedInstances.fromValue(scaleInProtectedInstances);
    }

    /**
     * <p>
     * Choose the behavior that you want Amazon EC2 Auto Scaling to use if instances protected from scale in are found.
     * </p>
     * <p>
     * The following lists the valid values:
     * </p>
     * <dl>
     * <dt>Refresh</dt>
     * <dd>
     * <p>
     * Amazon EC2 Auto Scaling replaces instances that are protected from scale in.
     * </p>
     * </dd>
     * <dt>Ignore</dt>
     * <dd>
     * <p>
     * Amazon EC2 Auto Scaling ignores instances that are protected from scale in and continues to replace instances
     * that are not protected.
     * </p>
     * </dd>
     * <dt>Wait (default)</dt>
     * <dd>
     * <p>
     * Amazon EC2 Auto Scaling waits one hour for you to remove scale-in protection. Otherwise, the instance refresh
     * will fail.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #scaleInProtectedInstances} will return {@link ScaleInProtectedInstances#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #scaleInProtectedInstancesAsString}.
     * </p>
     * 
     * @return Choose the behavior that you want Amazon EC2 Auto Scaling to use if instances protected from scale in are
     *         found. </p>
     *         <p>
     *         The following lists the valid values:
     *         </p>
     *         <dl>
     *         <dt>Refresh</dt>
     *         <dd>
     *         <p>
     *         Amazon EC2 Auto Scaling replaces instances that are protected from scale in.
     *         </p>
     *         </dd>
     *         <dt>Ignore</dt>
     *         <dd>
     *         <p>
     *         Amazon EC2 Auto Scaling ignores instances that are protected from scale in and continues to replace
     *         instances that are not protected.
     *         </p>
     *         </dd>
     *         <dt>Wait (default)</dt>
     *         <dd>
     *         <p>
     *         Amazon EC2 Auto Scaling waits one hour for you to remove scale-in protection. Otherwise, the instance
     *         refresh will fail.
     *         </p>
     *         </dd>
     * @see ScaleInProtectedInstances
     */
    public final String scaleInProtectedInstancesAsString() {
        return scaleInProtectedInstances;
    }

    /**
     * <p>
     * Choose the behavior that you want Amazon EC2 Auto Scaling to use if instances in <code>Standby</code> state are
     * found.
     * </p>
     * <p>
     * The following lists the valid values:
     * </p>
     * <dl>
     * <dt>Terminate</dt>
     * <dd>
     * <p>
     * Amazon EC2 Auto Scaling terminates instances that are in <code>Standby</code>.
     * </p>
     * </dd>
     * <dt>Ignore</dt>
     * <dd>
     * <p>
     * Amazon EC2 Auto Scaling ignores instances that are in <code>Standby</code> and continues to replace instances
     * that are in the <code>InService</code> state.
     * </p>
     * </dd>
     * <dt>Wait (default)</dt>
     * <dd>
     * <p>
     * Amazon EC2 Auto Scaling waits one hour for you to return the instances to service. Otherwise, the instance
     * refresh will fail.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #standbyInstances}
     * will return {@link StandbyInstances#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #standbyInstancesAsString}.
     * </p>
     * 
     * @return Choose the behavior that you want Amazon EC2 Auto Scaling to use if instances in <code>Standby</code>
     *         state are found.</p>
     *         <p>
     *         The following lists the valid values:
     *         </p>
     *         <dl>
     *         <dt>Terminate</dt>
     *         <dd>
     *         <p>
     *         Amazon EC2 Auto Scaling terminates instances that are in <code>Standby</code>.
     *         </p>
     *         </dd>
     *         <dt>Ignore</dt>
     *         <dd>
     *         <p>
     *         Amazon EC2 Auto Scaling ignores instances that are in <code>Standby</code> and continues to replace
     *         instances that are in the <code>InService</code> state.
     *         </p>
     *         </dd>
     *         <dt>Wait (default)</dt>
     *         <dd>
     *         <p>
     *         Amazon EC2 Auto Scaling waits one hour for you to return the instances to service. Otherwise, the
     *         instance refresh will fail.
     *         </p>
     *         </dd>
     * @see StandbyInstances
     */
    public final StandbyInstances standbyInstances() {
        return StandbyInstances.fromValue(standbyInstances);
    }

    /**
     * <p>
     * Choose the behavior that you want Amazon EC2 Auto Scaling to use if instances in <code>Standby</code> state are
     * found.
     * </p>
     * <p>
     * The following lists the valid values:
     * </p>
     * <dl>
     * <dt>Terminate</dt>
     * <dd>
     * <p>
     * Amazon EC2 Auto Scaling terminates instances that are in <code>Standby</code>.
     * </p>
     * </dd>
     * <dt>Ignore</dt>
     * <dd>
     * <p>
     * Amazon EC2 Auto Scaling ignores instances that are in <code>Standby</code> and continues to replace instances
     * that are in the <code>InService</code> state.
     * </p>
     * </dd>
     * <dt>Wait (default)</dt>
     * <dd>
     * <p>
     * Amazon EC2 Auto Scaling waits one hour for you to return the instances to service. Otherwise, the instance
     * refresh will fail.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #standbyInstances}
     * will return {@link StandbyInstances#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #standbyInstancesAsString}.
     * </p>
     * 
     * @return Choose the behavior that you want Amazon EC2 Auto Scaling to use if instances in <code>Standby</code>
     *         state are found.</p>
     *         <p>
     *         The following lists the valid values:
     *         </p>
     *         <dl>
     *         <dt>Terminate</dt>
     *         <dd>
     *         <p>
     *         Amazon EC2 Auto Scaling terminates instances that are in <code>Standby</code>.
     *         </p>
     *         </dd>
     *         <dt>Ignore</dt>
     *         <dd>
     *         <p>
     *         Amazon EC2 Auto Scaling ignores instances that are in <code>Standby</code> and continues to replace
     *         instances that are in the <code>InService</code> state.
     *         </p>
     *         </dd>
     *         <dt>Wait (default)</dt>
     *         <dd>
     *         <p>
     *         Amazon EC2 Auto Scaling waits one hour for you to return the instances to service. Otherwise, the
     *         instance refresh will fail.
     *         </p>
     *         </dd>
     * @see StandbyInstances
     */
    public final String standbyInstancesAsString() {
        return standbyInstances;
    }

    /**
     * <p>
     * (Optional) The CloudWatch alarm specification. CloudWatch alarms can be used to identify any issues and fail the
     * operation if an alarm threshold is met.
     * </p>
     * 
     * @return (Optional) The CloudWatch alarm specification. CloudWatch alarms can be used to identify any issues and
     *         fail the operation if an alarm threshold is met.
     */
    public final AlarmSpecification alarmSpecification() {
        return alarmSpecification;
    }

    /**
     * <p>
     * Specifies the maximum percentage of the group that can be in service and healthy, or pending, to support your
     * workload when replacing instances. The value is expressed as a percentage of the desired capacity of the Auto
     * Scaling group. Value range is 100 to 200.
     * </p>
     * <p>
     * If you specify <code>MaxHealthyPercentage</code>, you must also specify <code>MinHealthyPercentage</code>, and
     * the difference between them cannot be greater than 100. A larger range increases the number of instances that can
     * be replaced at the same time.
     * </p>
     * <p>
     * If you do not specify this property, the default is 100 percent, or the percentage set in the instance
     * maintenance policy for the Auto Scaling group, if defined.
     * </p>
     * 
     * @return Specifies the maximum percentage of the group that can be in service and healthy, or pending, to support
     *         your workload when replacing instances. The value is expressed as a percentage of the desired capacity of
     *         the Auto Scaling group. Value range is 100 to 200.</p>
     *         <p>
     *         If you specify <code>MaxHealthyPercentage</code>, you must also specify <code>MinHealthyPercentage</code>
     *         , and the difference between them cannot be greater than 100. A larger range increases the number of
     *         instances that can be replaced at the same time.
     *         </p>
     *         <p>
     *         If you do not specify this property, the default is 100 percent, or the percentage set in the instance
     *         maintenance policy for the Auto Scaling group, if defined.
     */
    public final Integer maxHealthyPercentage() {
        return maxHealthyPercentage;
    }

    /**
     * <p>
     * The amount of time, in seconds, to wait at the end of an instance refresh before the instance refresh is
     * considered complete.
     * </p>
     * 
     * @return The amount of time, in seconds, to wait at the end of an instance refresh before the instance refresh is
     *         considered complete.
     */
    public final Integer bakeTime() {
        return bakeTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(minHealthyPercentage());
        hashCode = 31 * hashCode + Objects.hashCode(instanceWarmup());
        hashCode = 31 * hashCode + Objects.hashCode(hasCheckpointPercentages() ? checkpointPercentages() : null);
        hashCode = 31 * hashCode + Objects.hashCode(checkpointDelay());
        hashCode = 31 * hashCode + Objects.hashCode(skipMatching());
        hashCode = 31 * hashCode + Objects.hashCode(autoRollback());
        hashCode = 31 * hashCode + Objects.hashCode(scaleInProtectedInstancesAsString());
        hashCode = 31 * hashCode + Objects.hashCode(standbyInstancesAsString());
        hashCode = 31 * hashCode + Objects.hashCode(alarmSpecification());
        hashCode = 31 * hashCode + Objects.hashCode(maxHealthyPercentage());
        hashCode = 31 * hashCode + Objects.hashCode(bakeTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RefreshPreferences)) {
            return false;
        }
        RefreshPreferences other = (RefreshPreferences) obj;
        return Objects.equals(minHealthyPercentage(), other.minHealthyPercentage())
                && Objects.equals(instanceWarmup(), other.instanceWarmup())
                && hasCheckpointPercentages() == other.hasCheckpointPercentages()
                && Objects.equals(checkpointPercentages(), other.checkpointPercentages())
                && Objects.equals(checkpointDelay(), other.checkpointDelay())
                && Objects.equals(skipMatching(), other.skipMatching()) && Objects.equals(autoRollback(), other.autoRollback())
                && Objects.equals(scaleInProtectedInstancesAsString(), other.scaleInProtectedInstancesAsString())
                && Objects.equals(standbyInstancesAsString(), other.standbyInstancesAsString())
                && Objects.equals(alarmSpecification(), other.alarmSpecification())
                && Objects.equals(maxHealthyPercentage(), other.maxHealthyPercentage())
                && Objects.equals(bakeTime(), other.bakeTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RefreshPreferences").add("MinHealthyPercentage", minHealthyPercentage())
                .add("InstanceWarmup", instanceWarmup())
                .add("CheckpointPercentages", hasCheckpointPercentages() ? checkpointPercentages() : null)
                .add("CheckpointDelay", checkpointDelay()).add("SkipMatching", skipMatching())
                .add("AutoRollback", autoRollback()).add("ScaleInProtectedInstances", scaleInProtectedInstancesAsString())
                .add("StandbyInstances", standbyInstancesAsString()).add("AlarmSpecification", alarmSpecification())
                .add("MaxHealthyPercentage", maxHealthyPercentage()).add("BakeTime", bakeTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MinHealthyPercentage":
            return Optional.ofNullable(clazz.cast(minHealthyPercentage()));
        case "InstanceWarmup":
            return Optional.ofNullable(clazz.cast(instanceWarmup()));
        case "CheckpointPercentages":
            return Optional.ofNullable(clazz.cast(checkpointPercentages()));
        case "CheckpointDelay":
            return Optional.ofNullable(clazz.cast(checkpointDelay()));
        case "SkipMatching":
            return Optional.ofNullable(clazz.cast(skipMatching()));
        case "AutoRollback":
            return Optional.ofNullable(clazz.cast(autoRollback()));
        case "ScaleInProtectedInstances":
            return Optional.ofNullable(clazz.cast(scaleInProtectedInstancesAsString()));
        case "StandbyInstances":
            return Optional.ofNullable(clazz.cast(standbyInstancesAsString()));
        case "AlarmSpecification":
            return Optional.ofNullable(clazz.cast(alarmSpecification()));
        case "MaxHealthyPercentage":
            return Optional.ofNullable(clazz.cast(maxHealthyPercentage()));
        case "BakeTime":
            return Optional.ofNullable(clazz.cast(bakeTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("MinHealthyPercentage", MIN_HEALTHY_PERCENTAGE_FIELD);
        map.put("InstanceWarmup", INSTANCE_WARMUP_FIELD);
        map.put("CheckpointPercentages", CHECKPOINT_PERCENTAGES_FIELD);
        map.put("CheckpointDelay", CHECKPOINT_DELAY_FIELD);
        map.put("SkipMatching", SKIP_MATCHING_FIELD);
        map.put("AutoRollback", AUTO_ROLLBACK_FIELD);
        map.put("ScaleInProtectedInstances", SCALE_IN_PROTECTED_INSTANCES_FIELD);
        map.put("StandbyInstances", STANDBY_INSTANCES_FIELD);
        map.put("AlarmSpecification", ALARM_SPECIFICATION_FIELD);
        map.put("MaxHealthyPercentage", MAX_HEALTHY_PERCENTAGE_FIELD);
        map.put("BakeTime", BAKE_TIME_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RefreshPreferences, T> g) {
        return obj -> g.apply((RefreshPreferences) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RefreshPreferences> {
        /**
         * <p>
         * Specifies the minimum percentage of the group to keep in service, healthy, and ready to use to support your
         * workload to allow the operation to continue. The value is expressed as a percentage of the desired capacity
         * of the Auto Scaling group. Value range is 0 to 100.
         * </p>
         * <p>
         * If you do not specify this property, the default is 90 percent, or the percentage set in the instance
         * maintenance policy for the Auto Scaling group, if defined.
         * </p>
         * 
         * @param minHealthyPercentage
         *        Specifies the minimum percentage of the group to keep in service, healthy, and ready to use to support
         *        your workload to allow the operation to continue. The value is expressed as a percentage of the
         *        desired capacity of the Auto Scaling group. Value range is 0 to 100.</p>
         *        <p>
         *        If you do not specify this property, the default is 90 percent, or the percentage set in the instance
         *        maintenance policy for the Auto Scaling group, if defined.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minHealthyPercentage(Integer minHealthyPercentage);

        /**
         * <p>
         * A time period, in seconds, during which an instance refresh waits before moving on to replacing the next
         * instance after a new instance enters the <code>InService</code> state.
         * </p>
         * <p>
         * This property is not required for normal usage. Instead, use the <code>DefaultInstanceWarmup</code> property
         * of the Auto Scaling group. The <code>InstanceWarmup</code> and <code>DefaultInstanceWarmup</code> properties
         * work the same way. Only specify this property if you must override the <code>DefaultInstanceWarmup</code>
         * property.
         * </p>
         * <p>
         * If you do not specify this property, the instance warmup by default is the value of the
         * <code>DefaultInstanceWarmup</code> property, if defined (which is recommended in all cases), or the
         * <code>HealthCheckGracePeriod</code> property otherwise.
         * </p>
         * 
         * @param instanceWarmup
         *        A time period, in seconds, during which an instance refresh waits before moving on to replacing the
         *        next instance after a new instance enters the <code>InService</code> state.</p>
         *        <p>
         *        This property is not required for normal usage. Instead, use the <code>DefaultInstanceWarmup</code>
         *        property of the Auto Scaling group. The <code>InstanceWarmup</code> and
         *        <code>DefaultInstanceWarmup</code> properties work the same way. Only specify this property if you
         *        must override the <code>DefaultInstanceWarmup</code> property.
         *        </p>
         *        <p>
         *        If you do not specify this property, the instance warmup by default is the value of the
         *        <code>DefaultInstanceWarmup</code> property, if defined (which is recommended in all cases), or the
         *        <code>HealthCheckGracePeriod</code> property otherwise.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceWarmup(Integer instanceWarmup);

        /**
         * <p>
         * (Optional) Threshold values for each checkpoint in ascending order. Each number must be unique. To replace
         * all instances in the Auto Scaling group, the last number in the array must be <code>100</code>.
         * </p>
         * <p>
         * For usage examples, see <a
         * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-adding-checkpoints-instance-refresh.html">Add
         * checkpoints to an instance refresh</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
         * </p>
         * 
         * @param checkpointPercentages
         *        (Optional) Threshold values for each checkpoint in ascending order. Each number must be unique. To
         *        replace all instances in the Auto Scaling group, the last number in the array must be <code>100</code>
         *        .</p>
         *        <p>
         *        For usage examples, see <a href=
         *        "https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-adding-checkpoints-instance-refresh.html"
         *        >Add checkpoints to an instance refresh</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder checkpointPercentages(Collection<Integer> checkpointPercentages);

        /**
         * <p>
         * (Optional) Threshold values for each checkpoint in ascending order. Each number must be unique. To replace
         * all instances in the Auto Scaling group, the last number in the array must be <code>100</code>.
         * </p>
         * <p>
         * For usage examples, see <a
         * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-adding-checkpoints-instance-refresh.html">Add
         * checkpoints to an instance refresh</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
         * </p>
         * 
         * @param checkpointPercentages
         *        (Optional) Threshold values for each checkpoint in ascending order. Each number must be unique. To
         *        replace all instances in the Auto Scaling group, the last number in the array must be <code>100</code>
         *        .</p>
         *        <p>
         *        For usage examples, see <a href=
         *        "https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-adding-checkpoints-instance-refresh.html"
         *        >Add checkpoints to an instance refresh</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder checkpointPercentages(Integer... checkpointPercentages);

        /**
         * <p>
         * (Optional) The amount of time, in seconds, to wait after a checkpoint before continuing. This property is
         * optional, but if you specify a value for it, you must also specify a value for
         * <code>CheckpointPercentages</code>. If you specify a value for <code>CheckpointPercentages</code> and not for
         * <code>CheckpointDelay</code>, the <code>CheckpointDelay</code> defaults to <code>3600</code> (1 hour).
         * </p>
         * 
         * @param checkpointDelay
         *        (Optional) The amount of time, in seconds, to wait after a checkpoint before continuing. This property
         *        is optional, but if you specify a value for it, you must also specify a value for
         *        <code>CheckpointPercentages</code>. If you specify a value for <code>CheckpointPercentages</code> and
         *        not for <code>CheckpointDelay</code>, the <code>CheckpointDelay</code> defaults to <code>3600</code>
         *        (1 hour).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder checkpointDelay(Integer checkpointDelay);

        /**
         * <p>
         * (Optional) Indicates whether skip matching is enabled. If enabled (<code>true</code>), then Amazon EC2 Auto
         * Scaling skips replacing instances that match the desired configuration. If no desired configuration is
         * specified, then it skips replacing instances that have the same launch template and instance types that the
         * Auto Scaling group was using before the start of the instance refresh. The default is <code>false</code>.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-instance-refresh-skip-matching.html">Use an
         * instance refresh with skip matching</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
         * </p>
         * 
         * @param skipMatching
         *        (Optional) Indicates whether skip matching is enabled. If enabled (<code>true</code>), then Amazon EC2
         *        Auto Scaling skips replacing instances that match the desired configuration. If no desired
         *        configuration is specified, then it skips replacing instances that have the same launch template and
         *        instance types that the Auto Scaling group was using before the start of the instance refresh. The
         *        default is <code>false</code>.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-instance-refresh-skip-matching.html"
         *        >Use an instance refresh with skip matching</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder skipMatching(Boolean skipMatching);

        /**
         * <p>
         * (Optional) Indicates whether to roll back the Auto Scaling group to its previous configuration if the
         * instance refresh fails or a CloudWatch alarm threshold is met. The default is <code>false</code>.
         * </p>
         * <p>
         * A rollback is not supported in the following situations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * There is no desired configuration specified for the instance refresh.
         * </p>
         * </li>
         * <li>
         * <p>
         * The Auto Scaling group has a launch template that uses an Amazon Web Services Systems Manager parameter
         * instead of an AMI ID for the <code>ImageId</code> property.
         * </p>
         * </li>
         * <li>
         * <p>
         * The Auto Scaling group uses the launch template's <code>$Latest</code> or <code>$Default</code> version.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/instance-refresh-rollback.html">Undo changes with
         * a rollback</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
         * </p>
         * 
         * @param autoRollback
         *        (Optional) Indicates whether to roll back the Auto Scaling group to its previous configuration if the
         *        instance refresh fails or a CloudWatch alarm threshold is met. The default is <code>false</code>.</p>
         *        <p>
         *        A rollback is not supported in the following situations:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        There is no desired configuration specified for the instance refresh.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The Auto Scaling group has a launch template that uses an Amazon Web Services Systems Manager
         *        parameter instead of an AMI ID for the <code>ImageId</code> property.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The Auto Scaling group uses the launch template's <code>$Latest</code> or <code>$Default</code>
         *        version.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/instance-refresh-rollback.html">Undo
         *        changes with a rollback</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoRollback(Boolean autoRollback);

        /**
         * <p>
         * Choose the behavior that you want Amazon EC2 Auto Scaling to use if instances protected from scale in are
         * found.
         * </p>
         * <p>
         * The following lists the valid values:
         * </p>
         * <dl>
         * <dt>Refresh</dt>
         * <dd>
         * <p>
         * Amazon EC2 Auto Scaling replaces instances that are protected from scale in.
         * </p>
         * </dd>
         * <dt>Ignore</dt>
         * <dd>
         * <p>
         * Amazon EC2 Auto Scaling ignores instances that are protected from scale in and continues to replace instances
         * that are not protected.
         * </p>
         * </dd>
         * <dt>Wait (default)</dt>
         * <dd>
         * <p>
         * Amazon EC2 Auto Scaling waits one hour for you to remove scale-in protection. Otherwise, the instance refresh
         * will fail.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param scaleInProtectedInstances
         *        Choose the behavior that you want Amazon EC2 Auto Scaling to use if instances protected from scale in
         *        are found. </p>
         *        <p>
         *        The following lists the valid values:
         *        </p>
         *        <dl>
         *        <dt>Refresh</dt>
         *        <dd>
         *        <p>
         *        Amazon EC2 Auto Scaling replaces instances that are protected from scale in.
         *        </p>
         *        </dd>
         *        <dt>Ignore</dt>
         *        <dd>
         *        <p>
         *        Amazon EC2 Auto Scaling ignores instances that are protected from scale in and continues to replace
         *        instances that are not protected.
         *        </p>
         *        </dd>
         *        <dt>Wait (default)</dt>
         *        <dd>
         *        <p>
         *        Amazon EC2 Auto Scaling waits one hour for you to remove scale-in protection. Otherwise, the instance
         *        refresh will fail.
         *        </p>
         *        </dd>
         * @see ScaleInProtectedInstances
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScaleInProtectedInstances
         */
        Builder scaleInProtectedInstances(String scaleInProtectedInstances);

        /**
         * <p>
         * Choose the behavior that you want Amazon EC2 Auto Scaling to use if instances protected from scale in are
         * found.
         * </p>
         * <p>
         * The following lists the valid values:
         * </p>
         * <dl>
         * <dt>Refresh</dt>
         * <dd>
         * <p>
         * Amazon EC2 Auto Scaling replaces instances that are protected from scale in.
         * </p>
         * </dd>
         * <dt>Ignore</dt>
         * <dd>
         * <p>
         * Amazon EC2 Auto Scaling ignores instances that are protected from scale in and continues to replace instances
         * that are not protected.
         * </p>
         * </dd>
         * <dt>Wait (default)</dt>
         * <dd>
         * <p>
         * Amazon EC2 Auto Scaling waits one hour for you to remove scale-in protection. Otherwise, the instance refresh
         * will fail.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param scaleInProtectedInstances
         *        Choose the behavior that you want Amazon EC2 Auto Scaling to use if instances protected from scale in
         *        are found. </p>
         *        <p>
         *        The following lists the valid values:
         *        </p>
         *        <dl>
         *        <dt>Refresh</dt>
         *        <dd>
         *        <p>
         *        Amazon EC2 Auto Scaling replaces instances that are protected from scale in.
         *        </p>
         *        </dd>
         *        <dt>Ignore</dt>
         *        <dd>
         *        <p>
         *        Amazon EC2 Auto Scaling ignores instances that are protected from scale in and continues to replace
         *        instances that are not protected.
         *        </p>
         *        </dd>
         *        <dt>Wait (default)</dt>
         *        <dd>
         *        <p>
         *        Amazon EC2 Auto Scaling waits one hour for you to remove scale-in protection. Otherwise, the instance
         *        refresh will fail.
         *        </p>
         *        </dd>
         * @see ScaleInProtectedInstances
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScaleInProtectedInstances
         */
        Builder scaleInProtectedInstances(ScaleInProtectedInstances scaleInProtectedInstances);

        /**
         * <p>
         * Choose the behavior that you want Amazon EC2 Auto Scaling to use if instances in <code>Standby</code> state
         * are found.
         * </p>
         * <p>
         * The following lists the valid values:
         * </p>
         * <dl>
         * <dt>Terminate</dt>
         * <dd>
         * <p>
         * Amazon EC2 Auto Scaling terminates instances that are in <code>Standby</code>.
         * </p>
         * </dd>
         * <dt>Ignore</dt>
         * <dd>
         * <p>
         * Amazon EC2 Auto Scaling ignores instances that are in <code>Standby</code> and continues to replace instances
         * that are in the <code>InService</code> state.
         * </p>
         * </dd>
         * <dt>Wait (default)</dt>
         * <dd>
         * <p>
         * Amazon EC2 Auto Scaling waits one hour for you to return the instances to service. Otherwise, the instance
         * refresh will fail.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param standbyInstances
         *        Choose the behavior that you want Amazon EC2 Auto Scaling to use if instances in <code>Standby</code>
         *        state are found.</p>
         *        <p>
         *        The following lists the valid values:
         *        </p>
         *        <dl>
         *        <dt>Terminate</dt>
         *        <dd>
         *        <p>
         *        Amazon EC2 Auto Scaling terminates instances that are in <code>Standby</code>.
         *        </p>
         *        </dd>
         *        <dt>Ignore</dt>
         *        <dd>
         *        <p>
         *        Amazon EC2 Auto Scaling ignores instances that are in <code>Standby</code> and continues to replace
         *        instances that are in the <code>InService</code> state.
         *        </p>
         *        </dd>
         *        <dt>Wait (default)</dt>
         *        <dd>
         *        <p>
         *        Amazon EC2 Auto Scaling waits one hour for you to return the instances to service. Otherwise, the
         *        instance refresh will fail.
         *        </p>
         *        </dd>
         * @see StandbyInstances
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StandbyInstances
         */
        Builder standbyInstances(String standbyInstances);

        /**
         * <p>
         * Choose the behavior that you want Amazon EC2 Auto Scaling to use if instances in <code>Standby</code> state
         * are found.
         * </p>
         * <p>
         * The following lists the valid values:
         * </p>
         * <dl>
         * <dt>Terminate</dt>
         * <dd>
         * <p>
         * Amazon EC2 Auto Scaling terminates instances that are in <code>Standby</code>.
         * </p>
         * </dd>
         * <dt>Ignore</dt>
         * <dd>
         * <p>
         * Amazon EC2 Auto Scaling ignores instances that are in <code>Standby</code> and continues to replace instances
         * that are in the <code>InService</code> state.
         * </p>
         * </dd>
         * <dt>Wait (default)</dt>
         * <dd>
         * <p>
         * Amazon EC2 Auto Scaling waits one hour for you to return the instances to service. Otherwise, the instance
         * refresh will fail.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param standbyInstances
         *        Choose the behavior that you want Amazon EC2 Auto Scaling to use if instances in <code>Standby</code>
         *        state are found.</p>
         *        <p>
         *        The following lists the valid values:
         *        </p>
         *        <dl>
         *        <dt>Terminate</dt>
         *        <dd>
         *        <p>
         *        Amazon EC2 Auto Scaling terminates instances that are in <code>Standby</code>.
         *        </p>
         *        </dd>
         *        <dt>Ignore</dt>
         *        <dd>
         *        <p>
         *        Amazon EC2 Auto Scaling ignores instances that are in <code>Standby</code> and continues to replace
         *        instances that are in the <code>InService</code> state.
         *        </p>
         *        </dd>
         *        <dt>Wait (default)</dt>
         *        <dd>
         *        <p>
         *        Amazon EC2 Auto Scaling waits one hour for you to return the instances to service. Otherwise, the
         *        instance refresh will fail.
         *        </p>
         *        </dd>
         * @see StandbyInstances
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StandbyInstances
         */
        Builder standbyInstances(StandbyInstances standbyInstances);

        /**
         * <p>
         * (Optional) The CloudWatch alarm specification. CloudWatch alarms can be used to identify any issues and fail
         * the operation if an alarm threshold is met.
         * </p>
         * 
         * @param alarmSpecification
         *        (Optional) The CloudWatch alarm specification. CloudWatch alarms can be used to identify any issues
         *        and fail the operation if an alarm threshold is met.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmSpecification(AlarmSpecification alarmSpecification);

        /**
         * <p>
         * (Optional) The CloudWatch alarm specification. CloudWatch alarms can be used to identify any issues and fail
         * the operation if an alarm threshold is met.
         * </p>
         * This is a convenience method that creates an instance of the {@link AlarmSpecification.Builder} avoiding the
         * need to create one manually via {@link AlarmSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AlarmSpecification.Builder#build()} is called immediately and its
         * result is passed to {@link #alarmSpecification(AlarmSpecification)}.
         * 
         * @param alarmSpecification
         *        a consumer that will call methods on {@link AlarmSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #alarmSpecification(AlarmSpecification)
         */
        default Builder alarmSpecification(Consumer<AlarmSpecification.Builder> alarmSpecification) {
            return alarmSpecification(AlarmSpecification.builder().applyMutation(alarmSpecification).build());
        }

        /**
         * <p>
         * Specifies the maximum percentage of the group that can be in service and healthy, or pending, to support your
         * workload when replacing instances. The value is expressed as a percentage of the desired capacity of the Auto
         * Scaling group. Value range is 100 to 200.
         * </p>
         * <p>
         * If you specify <code>MaxHealthyPercentage</code>, you must also specify <code>MinHealthyPercentage</code>,
         * and the difference between them cannot be greater than 100. A larger range increases the number of instances
         * that can be replaced at the same time.
         * </p>
         * <p>
         * If you do not specify this property, the default is 100 percent, or the percentage set in the instance
         * maintenance policy for the Auto Scaling group, if defined.
         * </p>
         * 
         * @param maxHealthyPercentage
         *        Specifies the maximum percentage of the group that can be in service and healthy, or pending, to
         *        support your workload when replacing instances. The value is expressed as a percentage of the desired
         *        capacity of the Auto Scaling group. Value range is 100 to 200.</p>
         *        <p>
         *        If you specify <code>MaxHealthyPercentage</code>, you must also specify
         *        <code>MinHealthyPercentage</code>, and the difference between them cannot be greater than 100. A
         *        larger range increases the number of instances that can be replaced at the same time.
         *        </p>
         *        <p>
         *        If you do not specify this property, the default is 100 percent, or the percentage set in the instance
         *        maintenance policy for the Auto Scaling group, if defined.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxHealthyPercentage(Integer maxHealthyPercentage);

        /**
         * <p>
         * The amount of time, in seconds, to wait at the end of an instance refresh before the instance refresh is
         * considered complete.
         * </p>
         * 
         * @param bakeTime
         *        The amount of time, in seconds, to wait at the end of an instance refresh before the instance refresh
         *        is considered complete.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bakeTime(Integer bakeTime);
    }

    static final class BuilderImpl implements Builder {
        private Integer minHealthyPercentage;

        private Integer instanceWarmup;

        private List<Integer> checkpointPercentages = DefaultSdkAutoConstructList.getInstance();

        private Integer checkpointDelay;

        private Boolean skipMatching;

        private Boolean autoRollback;

        private String scaleInProtectedInstances;

        private String standbyInstances;

        private AlarmSpecification alarmSpecification;

        private Integer maxHealthyPercentage;

        private Integer bakeTime;

        private BuilderImpl() {
        }

        private BuilderImpl(RefreshPreferences model) {
            minHealthyPercentage(model.minHealthyPercentage);
            instanceWarmup(model.instanceWarmup);
            checkpointPercentages(model.checkpointPercentages);
            checkpointDelay(model.checkpointDelay);
            skipMatching(model.skipMatching);
            autoRollback(model.autoRollback);
            scaleInProtectedInstances(model.scaleInProtectedInstances);
            standbyInstances(model.standbyInstances);
            alarmSpecification(model.alarmSpecification);
            maxHealthyPercentage(model.maxHealthyPercentage);
            bakeTime(model.bakeTime);
        }

        public final Integer getMinHealthyPercentage() {
            return minHealthyPercentage;
        }

        public final void setMinHealthyPercentage(Integer minHealthyPercentage) {
            this.minHealthyPercentage = minHealthyPercentage;
        }

        @Override
        public final Builder minHealthyPercentage(Integer minHealthyPercentage) {
            this.minHealthyPercentage = minHealthyPercentage;
            return this;
        }

        public final Integer getInstanceWarmup() {
            return instanceWarmup;
        }

        public final void setInstanceWarmup(Integer instanceWarmup) {
            this.instanceWarmup = instanceWarmup;
        }

        @Override
        public final Builder instanceWarmup(Integer instanceWarmup) {
            this.instanceWarmup = instanceWarmup;
            return this;
        }

        public final Collection<Integer> getCheckpointPercentages() {
            if (checkpointPercentages instanceof SdkAutoConstructList) {
                return null;
            }
            return checkpointPercentages;
        }

        public final void setCheckpointPercentages(Collection<Integer> checkpointPercentages) {
            this.checkpointPercentages = CheckpointPercentagesCopier.copy(checkpointPercentages);
        }

        @Override
        public final Builder checkpointPercentages(Collection<Integer> checkpointPercentages) {
            this.checkpointPercentages = CheckpointPercentagesCopier.copy(checkpointPercentages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder checkpointPercentages(Integer... checkpointPercentages) {
            checkpointPercentages(Arrays.asList(checkpointPercentages));
            return this;
        }

        public final Integer getCheckpointDelay() {
            return checkpointDelay;
        }

        public final void setCheckpointDelay(Integer checkpointDelay) {
            this.checkpointDelay = checkpointDelay;
        }

        @Override
        public final Builder checkpointDelay(Integer checkpointDelay) {
            this.checkpointDelay = checkpointDelay;
            return this;
        }

        public final Boolean getSkipMatching() {
            return skipMatching;
        }

        public final void setSkipMatching(Boolean skipMatching) {
            this.skipMatching = skipMatching;
        }

        @Override
        public final Builder skipMatching(Boolean skipMatching) {
            this.skipMatching = skipMatching;
            return this;
        }

        public final Boolean getAutoRollback() {
            return autoRollback;
        }

        public final void setAutoRollback(Boolean autoRollback) {
            this.autoRollback = autoRollback;
        }

        @Override
        public final Builder autoRollback(Boolean autoRollback) {
            this.autoRollback = autoRollback;
            return this;
        }

        public final String getScaleInProtectedInstances() {
            return scaleInProtectedInstances;
        }

        public final void setScaleInProtectedInstances(String scaleInProtectedInstances) {
            this.scaleInProtectedInstances = scaleInProtectedInstances;
        }

        @Override
        public final Builder scaleInProtectedInstances(String scaleInProtectedInstances) {
            this.scaleInProtectedInstances = scaleInProtectedInstances;
            return this;
        }

        @Override
        public final Builder scaleInProtectedInstances(ScaleInProtectedInstances scaleInProtectedInstances) {
            this.scaleInProtectedInstances(scaleInProtectedInstances == null ? null : scaleInProtectedInstances.toString());
            return this;
        }

        public final String getStandbyInstances() {
            return standbyInstances;
        }

        public final void setStandbyInstances(String standbyInstances) {
            this.standbyInstances = standbyInstances;
        }

        @Override
        public final Builder standbyInstances(String standbyInstances) {
            this.standbyInstances = standbyInstances;
            return this;
        }

        @Override
        public final Builder standbyInstances(StandbyInstances standbyInstances) {
            this.standbyInstances(standbyInstances == null ? null : standbyInstances.toString());
            return this;
        }

        public final AlarmSpecification.Builder getAlarmSpecification() {
            return alarmSpecification != null ? alarmSpecification.toBuilder() : null;
        }

        public final void setAlarmSpecification(AlarmSpecification.BuilderImpl alarmSpecification) {
            this.alarmSpecification = alarmSpecification != null ? alarmSpecification.build() : null;
        }

        @Override
        public final Builder alarmSpecification(AlarmSpecification alarmSpecification) {
            this.alarmSpecification = alarmSpecification;
            return this;
        }

        public final Integer getMaxHealthyPercentage() {
            return maxHealthyPercentage;
        }

        public final void setMaxHealthyPercentage(Integer maxHealthyPercentage) {
            this.maxHealthyPercentage = maxHealthyPercentage;
        }

        @Override
        public final Builder maxHealthyPercentage(Integer maxHealthyPercentage) {
            this.maxHealthyPercentage = maxHealthyPercentage;
            return this;
        }

        public final Integer getBakeTime() {
            return bakeTime;
        }

        public final void setBakeTime(Integer bakeTime) {
            this.bakeTime = bakeTime;
        }

        @Override
        public final Builder bakeTime(Integer bakeTime) {
            this.bakeTime = bakeTime;
            return this;
        }

        @Override
        public RefreshPreferences build() {
            return new RefreshPreferences(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
