/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscaling.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a CloudWatch metric of your choosing for a target tracking scaling policy to use with Amazon EC2 Auto
 * Scaling.
 * </p>
 * <p>
 * To create your customized metric specification:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Add values for each required property from CloudWatch. You can use an existing metric, or a new metric that you
 * create. To use your own metric, you must first publish the metric to CloudWatch. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/publishingMetrics.html">Publish custom
 * metrics</a> in the <i>Amazon CloudWatch User Guide</i>.
 * </p>
 * </li>
 * <li>
 * <p>
 * Choose a metric that changes proportionally with capacity. The value of the metric should increase or decrease in
 * inverse proportion to the number of capacity units. That is, the value of the metric should decrease when capacity
 * increases.
 * </p>
 * </li>
 * </ul>
 * <p>
 * For more information about the CloudWatch terminology below, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html">Amazon CloudWatch
 * concepts</a>.
 * </p>
 * <note>
 * <p>
 * Each individual service provides information about the metrics, namespace, and dimensions they use. For more
 * information, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/aws-services-cloudwatch-metrics.html">Amazon Web
 * Services services that publish CloudWatch metrics</a> in the <i>Amazon CloudWatch User Guide</i>.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CustomizedMetricSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<CustomizedMetricSpecification.Builder, CustomizedMetricSpecification> {
    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MetricName").getter(getter(CustomizedMetricSpecification::metricName))
            .setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricName").build()).build();

    private static final SdkField<String> NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Namespace").getter(getter(CustomizedMetricSpecification::namespace)).setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Namespace").build()).build();

    private static final SdkField<List<MetricDimension>> DIMENSIONS_FIELD = SdkField
            .<List<MetricDimension>> builder(MarshallingType.LIST)
            .memberName("Dimensions")
            .getter(getter(CustomizedMetricSpecification::dimensions))
            .setter(setter(Builder::dimensions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Dimensions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MetricDimension> builder(MarshallingType.SDK_POJO)
                                            .constructor(MetricDimension::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATISTIC_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Statistic").getter(getter(CustomizedMetricSpecification::statisticAsString))
            .setter(setter(Builder::statistic))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Statistic").build()).build();

    private static final SdkField<String> UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Unit")
            .getter(getter(CustomizedMetricSpecification::unit)).setter(setter(Builder::unit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Unit").build()).build();

    private static final SdkField<Integer> PERIOD_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Period").getter(getter(CustomizedMetricSpecification::period)).setter(setter(Builder::period))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Period").build()).build();

    private static final SdkField<List<TargetTrackingMetricDataQuery>> METRICS_FIELD = SdkField
            .<List<TargetTrackingMetricDataQuery>> builder(MarshallingType.LIST)
            .memberName("Metrics")
            .getter(getter(CustomizedMetricSpecification::metrics))
            .setter(setter(Builder::metrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Metrics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TargetTrackingMetricDataQuery> builder(MarshallingType.SDK_POJO)
                                            .constructor(TargetTrackingMetricDataQuery::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(METRIC_NAME_FIELD,
            NAMESPACE_FIELD, DIMENSIONS_FIELD, STATISTIC_FIELD, UNIT_FIELD, PERIOD_FIELD, METRICS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String metricName;

    private final String namespace;

    private final List<MetricDimension> dimensions;

    private final String statistic;

    private final String unit;

    private final Integer period;

    private final List<TargetTrackingMetricDataQuery> metrics;

    private CustomizedMetricSpecification(BuilderImpl builder) {
        this.metricName = builder.metricName;
        this.namespace = builder.namespace;
        this.dimensions = builder.dimensions;
        this.statistic = builder.statistic;
        this.unit = builder.unit;
        this.period = builder.period;
        this.metrics = builder.metrics;
    }

    /**
     * <p>
     * The name of the metric. To get the exact metric name, namespace, and dimensions, inspect the <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_Metric.html">Metric</a> object that is
     * returned by a call to <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListMetrics.html">ListMetrics</a>.
     * </p>
     * 
     * @return The name of the metric. To get the exact metric name, namespace, and dimensions, inspect the <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_Metric.html">Metric</a> object
     *         that is returned by a call to <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListMetrics.html"
     *         >ListMetrics</a>.
     */
    public final String metricName() {
        return metricName;
    }

    /**
     * <p>
     * The namespace of the metric.
     * </p>
     * 
     * @return The namespace of the metric.
     */
    public final String namespace() {
        return namespace;
    }

    /**
     * For responses, this returns true if the service returned a value for the Dimensions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDimensions() {
        return dimensions != null && !(dimensions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The dimensions of the metric.
     * </p>
     * <p>
     * Conditional: If you published your metric with dimensions, you must specify the same dimensions in your scaling
     * policy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDimensions} method.
     * </p>
     * 
     * @return The dimensions of the metric.</p>
     *         <p>
     *         Conditional: If you published your metric with dimensions, you must specify the same dimensions in your
     *         scaling policy.
     */
    public final List<MetricDimension> dimensions() {
        return dimensions;
    }

    /**
     * <p>
     * The statistic of the metric.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statistic} will
     * return {@link MetricStatistic#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statisticAsString}.
     * </p>
     * 
     * @return The statistic of the metric.
     * @see MetricStatistic
     */
    public final MetricStatistic statistic() {
        return MetricStatistic.fromValue(statistic);
    }

    /**
     * <p>
     * The statistic of the metric.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statistic} will
     * return {@link MetricStatistic#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statisticAsString}.
     * </p>
     * 
     * @return The statistic of the metric.
     * @see MetricStatistic
     */
    public final String statisticAsString() {
        return statistic;
    }

    /**
     * <p>
     * The unit of the metric. For a complete list of the units that CloudWatch supports, see the <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_MetricDatum.html">MetricDatum</a> data
     * type in the <i>Amazon CloudWatch API Reference</i>.
     * </p>
     * 
     * @return The unit of the metric. For a complete list of the units that CloudWatch supports, see the <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_MetricDatum.html"
     *         >MetricDatum</a> data type in the <i>Amazon CloudWatch API Reference</i>.
     */
    public final String unit() {
        return unit;
    }

    /**
     * <p>
     * The period of the metric in seconds. The default value is 60. Accepted values are 10, 30, and 60. For high
     * resolution metric, set the value to less than 60. For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/policy-creating-high-resolution-metrics.html">Create
     * a target tracking policy using high-resolution metrics for faster response</a>.
     * </p>
     * 
     * @return The period of the metric in seconds. The default value is 60. Accepted values are 10, 30, and 60. For
     *         high resolution metric, set the value to less than 60. For more information, see <a href=
     *         "https://docs.aws.amazon.com/autoscaling/ec2/userguide/policy-creating-high-resolution-metrics.html"
     *         >Create a target tracking policy using high-resolution metrics for faster response</a>.
     */
    public final Integer period() {
        return period;
    }

    /**
     * For responses, this returns true if the service returned a value for the Metrics property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasMetrics() {
        return metrics != null && !(metrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The metrics to include in the target tracking scaling policy, as a metric data query. This can include both raw
     * metric and metric math expressions.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetrics} method.
     * </p>
     * 
     * @return The metrics to include in the target tracking scaling policy, as a metric data query. This can include
     *         both raw metric and metric math expressions.
     */
    public final List<TargetTrackingMetricDataQuery> metrics() {
        return metrics;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(metricName());
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(hasDimensions() ? dimensions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statisticAsString());
        hashCode = 31 * hashCode + Objects.hashCode(unit());
        hashCode = 31 * hashCode + Objects.hashCode(period());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetrics() ? metrics() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CustomizedMetricSpecification)) {
            return false;
        }
        CustomizedMetricSpecification other = (CustomizedMetricSpecification) obj;
        return Objects.equals(metricName(), other.metricName()) && Objects.equals(namespace(), other.namespace())
                && hasDimensions() == other.hasDimensions() && Objects.equals(dimensions(), other.dimensions())
                && Objects.equals(statisticAsString(), other.statisticAsString()) && Objects.equals(unit(), other.unit())
                && Objects.equals(period(), other.period()) && hasMetrics() == other.hasMetrics()
                && Objects.equals(metrics(), other.metrics());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CustomizedMetricSpecification").add("MetricName", metricName()).add("Namespace", namespace())
                .add("Dimensions", hasDimensions() ? dimensions() : null).add("Statistic", statisticAsString())
                .add("Unit", unit()).add("Period", period()).add("Metrics", hasMetrics() ? metrics() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MetricName":
            return Optional.ofNullable(clazz.cast(metricName()));
        case "Namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "Dimensions":
            return Optional.ofNullable(clazz.cast(dimensions()));
        case "Statistic":
            return Optional.ofNullable(clazz.cast(statisticAsString()));
        case "Unit":
            return Optional.ofNullable(clazz.cast(unit()));
        case "Period":
            return Optional.ofNullable(clazz.cast(period()));
        case "Metrics":
            return Optional.ofNullable(clazz.cast(metrics()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("MetricName", METRIC_NAME_FIELD);
        map.put("Namespace", NAMESPACE_FIELD);
        map.put("Dimensions", DIMENSIONS_FIELD);
        map.put("Statistic", STATISTIC_FIELD);
        map.put("Unit", UNIT_FIELD);
        map.put("Period", PERIOD_FIELD);
        map.put("Metrics", METRICS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CustomizedMetricSpecification, T> g) {
        return obj -> g.apply((CustomizedMetricSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CustomizedMetricSpecification> {
        /**
         * <p>
         * The name of the metric. To get the exact metric name, namespace, and dimensions, inspect the <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_Metric.html">Metric</a> object
         * that is returned by a call to <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListMetrics.html">ListMetrics</a>.
         * </p>
         * 
         * @param metricName
         *        The name of the metric. To get the exact metric name, namespace, and dimensions, inspect the <a
         *        href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_Metric.html">Metric</a>
         *        object that is returned by a call to <a
         *        href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListMetrics.html"
         *        >ListMetrics</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * The namespace of the metric.
         * </p>
         * 
         * @param namespace
         *        The namespace of the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * The dimensions of the metric.
         * </p>
         * <p>
         * Conditional: If you published your metric with dimensions, you must specify the same dimensions in your
         * scaling policy.
         * </p>
         * 
         * @param dimensions
         *        The dimensions of the metric.</p>
         *        <p>
         *        Conditional: If you published your metric with dimensions, you must specify the same dimensions in
         *        your scaling policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Collection<MetricDimension> dimensions);

        /**
         * <p>
         * The dimensions of the metric.
         * </p>
         * <p>
         * Conditional: If you published your metric with dimensions, you must specify the same dimensions in your
         * scaling policy.
         * </p>
         * 
         * @param dimensions
         *        The dimensions of the metric.</p>
         *        <p>
         *        Conditional: If you published your metric with dimensions, you must specify the same dimensions in
         *        your scaling policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(MetricDimension... dimensions);

        /**
         * <p>
         * The dimensions of the metric.
         * </p>
         * <p>
         * Conditional: If you published your metric with dimensions, you must specify the same dimensions in your
         * scaling policy.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.autoscaling.model.MetricDimension.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.autoscaling.model.MetricDimension#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.autoscaling.model.MetricDimension.Builder#build()} is called
         * immediately and its result is passed to {@link #dimensions(List<MetricDimension>)}.
         * 
         * @param dimensions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.autoscaling.model.MetricDimension.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dimensions(java.util.Collection<MetricDimension>)
         */
        Builder dimensions(Consumer<MetricDimension.Builder>... dimensions);

        /**
         * <p>
         * The statistic of the metric.
         * </p>
         * 
         * @param statistic
         *        The statistic of the metric.
         * @see MetricStatistic
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricStatistic
         */
        Builder statistic(String statistic);

        /**
         * <p>
         * The statistic of the metric.
         * </p>
         * 
         * @param statistic
         *        The statistic of the metric.
         * @see MetricStatistic
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricStatistic
         */
        Builder statistic(MetricStatistic statistic);

        /**
         * <p>
         * The unit of the metric. For a complete list of the units that CloudWatch supports, see the <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_MetricDatum.html">MetricDatum</a>
         * data type in the <i>Amazon CloudWatch API Reference</i>.
         * </p>
         * 
         * @param unit
         *        The unit of the metric. For a complete list of the units that CloudWatch supports, see the <a
         *        href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_MetricDatum.html"
         *        >MetricDatum</a> data type in the <i>Amazon CloudWatch API Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unit(String unit);

        /**
         * <p>
         * The period of the metric in seconds. The default value is 60. Accepted values are 10, 30, and 60. For high
         * resolution metric, set the value to less than 60. For more information, see <a
         * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/policy-creating-high-resolution-metrics.html"
         * >Create a target tracking policy using high-resolution metrics for faster response</a>.
         * </p>
         * 
         * @param period
         *        The period of the metric in seconds. The default value is 60. Accepted values are 10, 30, and 60. For
         *        high resolution metric, set the value to less than 60. For more information, see <a href=
         *        "https://docs.aws.amazon.com/autoscaling/ec2/userguide/policy-creating-high-resolution-metrics.html"
         *        >Create a target tracking policy using high-resolution metrics for faster response</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder period(Integer period);

        /**
         * <p>
         * The metrics to include in the target tracking scaling policy, as a metric data query. This can include both
         * raw metric and metric math expressions.
         * </p>
         * 
         * @param metrics
         *        The metrics to include in the target tracking scaling policy, as a metric data query. This can include
         *        both raw metric and metric math expressions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metrics(Collection<TargetTrackingMetricDataQuery> metrics);

        /**
         * <p>
         * The metrics to include in the target tracking scaling policy, as a metric data query. This can include both
         * raw metric and metric math expressions.
         * </p>
         * 
         * @param metrics
         *        The metrics to include in the target tracking scaling policy, as a metric data query. This can include
         *        both raw metric and metric math expressions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metrics(TargetTrackingMetricDataQuery... metrics);

        /**
         * <p>
         * The metrics to include in the target tracking scaling policy, as a metric data query. This can include both
         * raw metric and metric math expressions.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.autoscaling.model.TargetTrackingMetricDataQuery.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.autoscaling.model.TargetTrackingMetricDataQuery#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.autoscaling.model.TargetTrackingMetricDataQuery.Builder#build()} is
         * called immediately and its result is passed to {@link #metrics(List<TargetTrackingMetricDataQuery>)}.
         * 
         * @param metrics
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.autoscaling.model.TargetTrackingMetricDataQuery.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metrics(java.util.Collection<TargetTrackingMetricDataQuery>)
         */
        Builder metrics(Consumer<TargetTrackingMetricDataQuery.Builder>... metrics);
    }

    static final class BuilderImpl implements Builder {
        private String metricName;

        private String namespace;

        private List<MetricDimension> dimensions = DefaultSdkAutoConstructList.getInstance();

        private String statistic;

        private String unit;

        private Integer period;

        private List<TargetTrackingMetricDataQuery> metrics = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CustomizedMetricSpecification model) {
            metricName(model.metricName);
            namespace(model.namespace);
            dimensions(model.dimensions);
            statistic(model.statistic);
            unit(model.unit);
            period(model.period);
            metrics(model.metrics);
        }

        public final String getMetricName() {
            return metricName;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        public final String getNamespace() {
            return namespace;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final List<MetricDimension.Builder> getDimensions() {
            List<MetricDimension.Builder> result = MetricDimensionsCopier.copyToBuilder(this.dimensions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDimensions(Collection<MetricDimension.BuilderImpl> dimensions) {
            this.dimensions = MetricDimensionsCopier.copyFromBuilder(dimensions);
        }

        @Override
        public final Builder dimensions(Collection<MetricDimension> dimensions) {
            this.dimensions = MetricDimensionsCopier.copy(dimensions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(MetricDimension... dimensions) {
            dimensions(Arrays.asList(dimensions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Consumer<MetricDimension.Builder>... dimensions) {
            dimensions(Stream.of(dimensions).map(c -> MetricDimension.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getStatistic() {
            return statistic;
        }

        public final void setStatistic(String statistic) {
            this.statistic = statistic;
        }

        @Override
        public final Builder statistic(String statistic) {
            this.statistic = statistic;
            return this;
        }

        @Override
        public final Builder statistic(MetricStatistic statistic) {
            this.statistic(statistic == null ? null : statistic.toString());
            return this;
        }

        public final String getUnit() {
            return unit;
        }

        public final void setUnit(String unit) {
            this.unit = unit;
        }

        @Override
        public final Builder unit(String unit) {
            this.unit = unit;
            return this;
        }

        public final Integer getPeriod() {
            return period;
        }

        public final void setPeriod(Integer period) {
            this.period = period;
        }

        @Override
        public final Builder period(Integer period) {
            this.period = period;
            return this;
        }

        public final List<TargetTrackingMetricDataQuery.Builder> getMetrics() {
            List<TargetTrackingMetricDataQuery.Builder> result = TargetTrackingMetricDataQueriesCopier
                    .copyToBuilder(this.metrics);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMetrics(Collection<TargetTrackingMetricDataQuery.BuilderImpl> metrics) {
            this.metrics = TargetTrackingMetricDataQueriesCopier.copyFromBuilder(metrics);
        }

        @Override
        public final Builder metrics(Collection<TargetTrackingMetricDataQuery> metrics) {
            this.metrics = TargetTrackingMetricDataQueriesCopier.copy(metrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metrics(TargetTrackingMetricDataQuery... metrics) {
            metrics(Arrays.asList(metrics));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metrics(Consumer<TargetTrackingMetricDataQuery.Builder>... metrics) {
            metrics(Stream.of(metrics).map(c -> TargetTrackingMetricDataQuery.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public CustomizedMetricSpecification build() {
            return new CustomizedMetricSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
