/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscalingplans.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a scalable resource.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ScalingPlanResource implements SdkPojo, Serializable,
        ToCopyableBuilder<ScalingPlanResource.Builder, ScalingPlanResource> {
    private static final SdkField<String> SCALING_PLAN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScalingPlanName").getter(getter(ScalingPlanResource::scalingPlanName))
            .setter(setter(Builder::scalingPlanName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalingPlanName").build()).build();

    private static final SdkField<Long> SCALING_PLAN_VERSION_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("ScalingPlanVersion").getter(getter(ScalingPlanResource::scalingPlanVersion))
            .setter(setter(Builder::scalingPlanVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalingPlanVersion").build())
            .build();

    private static final SdkField<String> SERVICE_NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServiceNamespace").getter(getter(ScalingPlanResource::serviceNamespaceAsString))
            .setter(setter(Builder::serviceNamespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceNamespace").build()).build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceId").getter(getter(ScalingPlanResource::resourceId)).setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceId").build()).build();

    private static final SdkField<String> SCALABLE_DIMENSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScalableDimension").getter(getter(ScalingPlanResource::scalableDimensionAsString))
            .setter(setter(Builder::scalableDimension))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalableDimension").build()).build();

    private static final SdkField<List<ScalingPolicy>> SCALING_POLICIES_FIELD = SdkField
            .<List<ScalingPolicy>> builder(MarshallingType.LIST)
            .memberName("ScalingPolicies")
            .getter(getter(ScalingPlanResource::scalingPolicies))
            .setter(setter(Builder::scalingPolicies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalingPolicies").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ScalingPolicy> builder(MarshallingType.SDK_POJO)
                                            .constructor(ScalingPolicy::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SCALING_STATUS_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScalingStatusCode").getter(getter(ScalingPlanResource::scalingStatusCodeAsString))
            .setter(setter(Builder::scalingStatusCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalingStatusCode").build()).build();

    private static final SdkField<String> SCALING_STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScalingStatusMessage").getter(getter(ScalingPlanResource::scalingStatusMessage))
            .setter(setter(Builder::scalingStatusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalingStatusMessage").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SCALING_PLAN_NAME_FIELD,
            SCALING_PLAN_VERSION_FIELD, SERVICE_NAMESPACE_FIELD, RESOURCE_ID_FIELD, SCALABLE_DIMENSION_FIELD,
            SCALING_POLICIES_FIELD, SCALING_STATUS_CODE_FIELD, SCALING_STATUS_MESSAGE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String scalingPlanName;

    private final Long scalingPlanVersion;

    private final String serviceNamespace;

    private final String resourceId;

    private final String scalableDimension;

    private final List<ScalingPolicy> scalingPolicies;

    private final String scalingStatusCode;

    private final String scalingStatusMessage;

    private ScalingPlanResource(BuilderImpl builder) {
        this.scalingPlanName = builder.scalingPlanName;
        this.scalingPlanVersion = builder.scalingPlanVersion;
        this.serviceNamespace = builder.serviceNamespace;
        this.resourceId = builder.resourceId;
        this.scalableDimension = builder.scalableDimension;
        this.scalingPolicies = builder.scalingPolicies;
        this.scalingStatusCode = builder.scalingStatusCode;
        this.scalingStatusMessage = builder.scalingStatusMessage;
    }

    /**
     * <p>
     * The name of the scaling plan.
     * </p>
     * 
     * @return The name of the scaling plan.
     */
    public final String scalingPlanName() {
        return scalingPlanName;
    }

    /**
     * <p>
     * The version number of the scaling plan.
     * </p>
     * 
     * @return The version number of the scaling plan.
     */
    public final Long scalingPlanVersion() {
        return scalingPlanVersion;
    }

    /**
     * <p>
     * The namespace of the AWS service.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #serviceNamespace}
     * will return {@link ServiceNamespace#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #serviceNamespaceAsString}.
     * </p>
     * 
     * @return The namespace of the AWS service.
     * @see ServiceNamespace
     */
    public final ServiceNamespace serviceNamespace() {
        return ServiceNamespace.fromValue(serviceNamespace);
    }

    /**
     * <p>
     * The namespace of the AWS service.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #serviceNamespace}
     * will return {@link ServiceNamespace#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #serviceNamespaceAsString}.
     * </p>
     * 
     * @return The namespace of the AWS service.
     * @see ServiceNamespace
     */
    public final String serviceNamespaceAsString() {
        return serviceNamespace;
    }

    /**
     * <p>
     * The ID of the resource. This string consists of the resource type and unique identifier.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Auto Scaling group - The resource type is <code>autoScalingGroup</code> and the unique identifier is the name of
     * the Auto Scaling group. Example: <code>autoScalingGroup/my-asg</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * ECS service - The resource type is <code>service</code> and the unique identifier is the cluster name and service
     * name. Example: <code>service/default/sample-webapp</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Spot Fleet request - The resource type is <code>spot-fleet-request</code> and the unique identifier is the Spot
     * Fleet request ID. Example: <code>spot-fleet-request/sfr-73fbd2ce-aa30-494c-8788-1cee4EXAMPLE</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * DynamoDB table - The resource type is <code>table</code> and the unique identifier is the resource ID. Example:
     * <code>table/my-table</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * DynamoDB global secondary index - The resource type is <code>index</code> and the unique identifier is the
     * resource ID. Example: <code>table/my-table/index/my-table-index</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Aurora DB cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster name.
     * Example: <code>cluster:my-db-cluster</code>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The ID of the resource. This string consists of the resource type and unique identifier.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Auto Scaling group - The resource type is <code>autoScalingGroup</code> and the unique identifier is the
     *         name of the Auto Scaling group. Example: <code>autoScalingGroup/my-asg</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ECS service - The resource type is <code>service</code> and the unique identifier is the cluster name and
     *         service name. Example: <code>service/default/sample-webapp</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Spot Fleet request - The resource type is <code>spot-fleet-request</code> and the unique identifier is
     *         the Spot Fleet request ID. Example:
     *         <code>spot-fleet-request/sfr-73fbd2ce-aa30-494c-8788-1cee4EXAMPLE</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DynamoDB table - The resource type is <code>table</code> and the unique identifier is the resource ID.
     *         Example: <code>table/my-table</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DynamoDB global secondary index - The resource type is <code>index</code> and the unique identifier is
     *         the resource ID. Example: <code>table/my-table/index/my-table-index</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Aurora DB cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster
     *         name. Example: <code>cluster:my-db-cluster</code>.
     *         </p>
     *         </li>
     */
    public final String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * The scalable dimension for the resource.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>autoscaling:autoScalingGroup:DesiredCapacity</code> - The desired capacity of an Auto Scaling group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global secondary
     * index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global secondary
     * index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster. Available for
     * Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scalableDimension}
     * will return {@link ScalableDimension#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #scalableDimensionAsString}.
     * </p>
     * 
     * @return The scalable dimension for the resource.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>autoscaling:autoScalingGroup:DesiredCapacity</code> - The desired capacity of an Auto Scaling
     *         group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global
     *         secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
     *         secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster.
     *         Available for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
     *         </p>
     *         </li>
     * @see ScalableDimension
     */
    public final ScalableDimension scalableDimension() {
        return ScalableDimension.fromValue(scalableDimension);
    }

    /**
     * <p>
     * The scalable dimension for the resource.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>autoscaling:autoScalingGroup:DesiredCapacity</code> - The desired capacity of an Auto Scaling group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global secondary
     * index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global secondary
     * index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster. Available for
     * Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scalableDimension}
     * will return {@link ScalableDimension#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #scalableDimensionAsString}.
     * </p>
     * 
     * @return The scalable dimension for the resource.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>autoscaling:autoScalingGroup:DesiredCapacity</code> - The desired capacity of an Auto Scaling
     *         group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global
     *         secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
     *         secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster.
     *         Available for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
     *         </p>
     *         </li>
     * @see ScalableDimension
     */
    public final String scalableDimensionAsString() {
        return scalableDimension;
    }

    /**
     * For responses, this returns true if the service returned a value for the ScalingPolicies property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasScalingPolicies() {
        return scalingPolicies != null && !(scalingPolicies instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The scaling policies.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasScalingPolicies} method.
     * </p>
     * 
     * @return The scaling policies.
     */
    public final List<ScalingPolicy> scalingPolicies() {
        return scalingPolicies;
    }

    /**
     * <p>
     * The scaling status of the resource.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Active</code> - The scaling configuration is active.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Inactive</code> - The scaling configuration is not active because the scaling plan is being created or the
     * scaling configuration could not be applied. Check the status message for more information.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartiallyActive</code> - The scaling configuration is partially active because the scaling plan is being
     * created or deleted or the scaling configuration could not be fully applied. Check the status message for more
     * information.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scalingStatusCode}
     * will return {@link ScalingStatusCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #scalingStatusCodeAsString}.
     * </p>
     * 
     * @return The scaling status of the resource.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Active</code> - The scaling configuration is active.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Inactive</code> - The scaling configuration is not active because the scaling plan is being created
     *         or the scaling configuration could not be applied. Check the status message for more information.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PartiallyActive</code> - The scaling configuration is partially active because the scaling plan is
     *         being created or deleted or the scaling configuration could not be fully applied. Check the status
     *         message for more information.
     *         </p>
     *         </li>
     * @see ScalingStatusCode
     */
    public final ScalingStatusCode scalingStatusCode() {
        return ScalingStatusCode.fromValue(scalingStatusCode);
    }

    /**
     * <p>
     * The scaling status of the resource.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Active</code> - The scaling configuration is active.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Inactive</code> - The scaling configuration is not active because the scaling plan is being created or the
     * scaling configuration could not be applied. Check the status message for more information.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartiallyActive</code> - The scaling configuration is partially active because the scaling plan is being
     * created or deleted or the scaling configuration could not be fully applied. Check the status message for more
     * information.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scalingStatusCode}
     * will return {@link ScalingStatusCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #scalingStatusCodeAsString}.
     * </p>
     * 
     * @return The scaling status of the resource.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Active</code> - The scaling configuration is active.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Inactive</code> - The scaling configuration is not active because the scaling plan is being created
     *         or the scaling configuration could not be applied. Check the status message for more information.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PartiallyActive</code> - The scaling configuration is partially active because the scaling plan is
     *         being created or deleted or the scaling configuration could not be fully applied. Check the status
     *         message for more information.
     *         </p>
     *         </li>
     * @see ScalingStatusCode
     */
    public final String scalingStatusCodeAsString() {
        return scalingStatusCode;
    }

    /**
     * <p>
     * A simple message about the current scaling status of the resource.
     * </p>
     * 
     * @return A simple message about the current scaling status of the resource.
     */
    public final String scalingStatusMessage() {
        return scalingStatusMessage;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(scalingPlanName());
        hashCode = 31 * hashCode + Objects.hashCode(scalingPlanVersion());
        hashCode = 31 * hashCode + Objects.hashCode(serviceNamespaceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(scalableDimensionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasScalingPolicies() ? scalingPolicies() : null);
        hashCode = 31 * hashCode + Objects.hashCode(scalingStatusCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(scalingStatusMessage());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ScalingPlanResource)) {
            return false;
        }
        ScalingPlanResource other = (ScalingPlanResource) obj;
        return Objects.equals(scalingPlanName(), other.scalingPlanName())
                && Objects.equals(scalingPlanVersion(), other.scalingPlanVersion())
                && Objects.equals(serviceNamespaceAsString(), other.serviceNamespaceAsString())
                && Objects.equals(resourceId(), other.resourceId())
                && Objects.equals(scalableDimensionAsString(), other.scalableDimensionAsString())
                && hasScalingPolicies() == other.hasScalingPolicies()
                && Objects.equals(scalingPolicies(), other.scalingPolicies())
                && Objects.equals(scalingStatusCodeAsString(), other.scalingStatusCodeAsString())
                && Objects.equals(scalingStatusMessage(), other.scalingStatusMessage());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ScalingPlanResource").add("ScalingPlanName", scalingPlanName())
                .add("ScalingPlanVersion", scalingPlanVersion()).add("ServiceNamespace", serviceNamespaceAsString())
                .add("ResourceId", resourceId()).add("ScalableDimension", scalableDimensionAsString())
                .add("ScalingPolicies", hasScalingPolicies() ? scalingPolicies() : null)
                .add("ScalingStatusCode", scalingStatusCodeAsString()).add("ScalingStatusMessage", scalingStatusMessage())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ScalingPlanName":
            return Optional.ofNullable(clazz.cast(scalingPlanName()));
        case "ScalingPlanVersion":
            return Optional.ofNullable(clazz.cast(scalingPlanVersion()));
        case "ServiceNamespace":
            return Optional.ofNullable(clazz.cast(serviceNamespaceAsString()));
        case "ResourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "ScalableDimension":
            return Optional.ofNullable(clazz.cast(scalableDimensionAsString()));
        case "ScalingPolicies":
            return Optional.ofNullable(clazz.cast(scalingPolicies()));
        case "ScalingStatusCode":
            return Optional.ofNullable(clazz.cast(scalingStatusCodeAsString()));
        case "ScalingStatusMessage":
            return Optional.ofNullable(clazz.cast(scalingStatusMessage()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ScalingPlanResource, T> g) {
        return obj -> g.apply((ScalingPlanResource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ScalingPlanResource> {
        /**
         * <p>
         * The name of the scaling plan.
         * </p>
         * 
         * @param scalingPlanName
         *        The name of the scaling plan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scalingPlanName(String scalingPlanName);

        /**
         * <p>
         * The version number of the scaling plan.
         * </p>
         * 
         * @param scalingPlanVersion
         *        The version number of the scaling plan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scalingPlanVersion(Long scalingPlanVersion);

        /**
         * <p>
         * The namespace of the AWS service.
         * </p>
         * 
         * @param serviceNamespace
         *        The namespace of the AWS service.
         * @see ServiceNamespace
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceNamespace
         */
        Builder serviceNamespace(String serviceNamespace);

        /**
         * <p>
         * The namespace of the AWS service.
         * </p>
         * 
         * @param serviceNamespace
         *        The namespace of the AWS service.
         * @see ServiceNamespace
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceNamespace
         */
        Builder serviceNamespace(ServiceNamespace serviceNamespace);

        /**
         * <p>
         * The ID of the resource. This string consists of the resource type and unique identifier.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Auto Scaling group - The resource type is <code>autoScalingGroup</code> and the unique identifier is the name
         * of the Auto Scaling group. Example: <code>autoScalingGroup/my-asg</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * ECS service - The resource type is <code>service</code> and the unique identifier is the cluster name and
         * service name. Example: <code>service/default/sample-webapp</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Spot Fleet request - The resource type is <code>spot-fleet-request</code> and the unique identifier is the
         * Spot Fleet request ID. Example: <code>spot-fleet-request/sfr-73fbd2ce-aa30-494c-8788-1cee4EXAMPLE</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * DynamoDB table - The resource type is <code>table</code> and the unique identifier is the resource ID.
         * Example: <code>table/my-table</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * DynamoDB global secondary index - The resource type is <code>index</code> and the unique identifier is the
         * resource ID. Example: <code>table/my-table/index/my-table-index</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Aurora DB cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster name.
         * Example: <code>cluster:my-db-cluster</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param resourceId
         *        The ID of the resource. This string consists of the resource type and unique identifier.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Auto Scaling group - The resource type is <code>autoScalingGroup</code> and the unique identifier is
         *        the name of the Auto Scaling group. Example: <code>autoScalingGroup/my-asg</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ECS service - The resource type is <code>service</code> and the unique identifier is the cluster name
         *        and service name. Example: <code>service/default/sample-webapp</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Spot Fleet request - The resource type is <code>spot-fleet-request</code> and the unique identifier is
         *        the Spot Fleet request ID. Example:
         *        <code>spot-fleet-request/sfr-73fbd2ce-aa30-494c-8788-1cee4EXAMPLE</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DynamoDB table - The resource type is <code>table</code> and the unique identifier is the resource ID.
         *        Example: <code>table/my-table</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DynamoDB global secondary index - The resource type is <code>index</code> and the unique identifier is
         *        the resource ID. Example: <code>table/my-table/index/my-table-index</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Aurora DB cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster
         *        name. Example: <code>cluster:my-db-cluster</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * The scalable dimension for the resource.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>autoscaling:autoScalingGroup:DesiredCapacity</code> - The desired capacity of an Auto Scaling group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global secondary
         * index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
         * secondary index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster. Available
         * for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
         * </p>
         * </li>
         * </ul>
         * 
         * @param scalableDimension
         *        The scalable dimension for the resource.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>autoscaling:autoScalingGroup:DesiredCapacity</code> - The desired capacity of an Auto Scaling
         *        group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global
         *        secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
         *        secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster.
         *        Available for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
         *        </p>
         *        </li>
         * @see ScalableDimension
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalableDimension
         */
        Builder scalableDimension(String scalableDimension);

        /**
         * <p>
         * The scalable dimension for the resource.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>autoscaling:autoScalingGroup:DesiredCapacity</code> - The desired capacity of an Auto Scaling group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global secondary
         * index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
         * secondary index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster. Available
         * for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
         * </p>
         * </li>
         * </ul>
         * 
         * @param scalableDimension
         *        The scalable dimension for the resource.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>autoscaling:autoScalingGroup:DesiredCapacity</code> - The desired capacity of an Auto Scaling
         *        group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global
         *        secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
         *        secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster.
         *        Available for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
         *        </p>
         *        </li>
         * @see ScalableDimension
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalableDimension
         */
        Builder scalableDimension(ScalableDimension scalableDimension);

        /**
         * <p>
         * The scaling policies.
         * </p>
         * 
         * @param scalingPolicies
         *        The scaling policies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scalingPolicies(Collection<ScalingPolicy> scalingPolicies);

        /**
         * <p>
         * The scaling policies.
         * </p>
         * 
         * @param scalingPolicies
         *        The scaling policies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scalingPolicies(ScalingPolicy... scalingPolicies);

        /**
         * <p>
         * The scaling policies.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ScalingPolicy>.Builder} avoiding the need
         * to create one manually via {@link List<ScalingPolicy>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ScalingPolicy>.Builder#build()} is called immediately and
         * its result is passed to {@link #scalingPolicies(List<ScalingPolicy>)}.
         * 
         * @param scalingPolicies
         *        a consumer that will call methods on {@link List<ScalingPolicy>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scalingPolicies(List<ScalingPolicy>)
         */
        Builder scalingPolicies(Consumer<ScalingPolicy.Builder>... scalingPolicies);

        /**
         * <p>
         * The scaling status of the resource.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Active</code> - The scaling configuration is active.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Inactive</code> - The scaling configuration is not active because the scaling plan is being created or
         * the scaling configuration could not be applied. Check the status message for more information.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PartiallyActive</code> - The scaling configuration is partially active because the scaling plan is
         * being created or deleted or the scaling configuration could not be fully applied. Check the status message
         * for more information.
         * </p>
         * </li>
         * </ul>
         * 
         * @param scalingStatusCode
         *        The scaling status of the resource.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Active</code> - The scaling configuration is active.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Inactive</code> - The scaling configuration is not active because the scaling plan is being
         *        created or the scaling configuration could not be applied. Check the status message for more
         *        information.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PartiallyActive</code> - The scaling configuration is partially active because the scaling plan
         *        is being created or deleted or the scaling configuration could not be fully applied. Check the status
         *        message for more information.
         *        </p>
         *        </li>
         * @see ScalingStatusCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalingStatusCode
         */
        Builder scalingStatusCode(String scalingStatusCode);

        /**
         * <p>
         * The scaling status of the resource.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Active</code> - The scaling configuration is active.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Inactive</code> - The scaling configuration is not active because the scaling plan is being created or
         * the scaling configuration could not be applied. Check the status message for more information.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PartiallyActive</code> - The scaling configuration is partially active because the scaling plan is
         * being created or deleted or the scaling configuration could not be fully applied. Check the status message
         * for more information.
         * </p>
         * </li>
         * </ul>
         * 
         * @param scalingStatusCode
         *        The scaling status of the resource.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Active</code> - The scaling configuration is active.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Inactive</code> - The scaling configuration is not active because the scaling plan is being
         *        created or the scaling configuration could not be applied. Check the status message for more
         *        information.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PartiallyActive</code> - The scaling configuration is partially active because the scaling plan
         *        is being created or deleted or the scaling configuration could not be fully applied. Check the status
         *        message for more information.
         *        </p>
         *        </li>
         * @see ScalingStatusCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalingStatusCode
         */
        Builder scalingStatusCode(ScalingStatusCode scalingStatusCode);

        /**
         * <p>
         * A simple message about the current scaling status of the resource.
         * </p>
         * 
         * @param scalingStatusMessage
         *        A simple message about the current scaling status of the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scalingStatusMessage(String scalingStatusMessage);
    }

    static final class BuilderImpl implements Builder {
        private String scalingPlanName;

        private Long scalingPlanVersion;

        private String serviceNamespace;

        private String resourceId;

        private String scalableDimension;

        private List<ScalingPolicy> scalingPolicies = DefaultSdkAutoConstructList.getInstance();

        private String scalingStatusCode;

        private String scalingStatusMessage;

        private BuilderImpl() {
        }

        private BuilderImpl(ScalingPlanResource model) {
            scalingPlanName(model.scalingPlanName);
            scalingPlanVersion(model.scalingPlanVersion);
            serviceNamespace(model.serviceNamespace);
            resourceId(model.resourceId);
            scalableDimension(model.scalableDimension);
            scalingPolicies(model.scalingPolicies);
            scalingStatusCode(model.scalingStatusCode);
            scalingStatusMessage(model.scalingStatusMessage);
        }

        public final String getScalingPlanName() {
            return scalingPlanName;
        }

        public final void setScalingPlanName(String scalingPlanName) {
            this.scalingPlanName = scalingPlanName;
        }

        @Override
        @Transient
        public final Builder scalingPlanName(String scalingPlanName) {
            this.scalingPlanName = scalingPlanName;
            return this;
        }

        public final Long getScalingPlanVersion() {
            return scalingPlanVersion;
        }

        public final void setScalingPlanVersion(Long scalingPlanVersion) {
            this.scalingPlanVersion = scalingPlanVersion;
        }

        @Override
        @Transient
        public final Builder scalingPlanVersion(Long scalingPlanVersion) {
            this.scalingPlanVersion = scalingPlanVersion;
            return this;
        }

        public final String getServiceNamespace() {
            return serviceNamespace;
        }

        public final void setServiceNamespace(String serviceNamespace) {
            this.serviceNamespace = serviceNamespace;
        }

        @Override
        @Transient
        public final Builder serviceNamespace(String serviceNamespace) {
            this.serviceNamespace = serviceNamespace;
            return this;
        }

        @Override
        @Transient
        public final Builder serviceNamespace(ServiceNamespace serviceNamespace) {
            this.serviceNamespace(serviceNamespace == null ? null : serviceNamespace.toString());
            return this;
        }

        public final String getResourceId() {
            return resourceId;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        @Override
        @Transient
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final String getScalableDimension() {
            return scalableDimension;
        }

        public final void setScalableDimension(String scalableDimension) {
            this.scalableDimension = scalableDimension;
        }

        @Override
        @Transient
        public final Builder scalableDimension(String scalableDimension) {
            this.scalableDimension = scalableDimension;
            return this;
        }

        @Override
        @Transient
        public final Builder scalableDimension(ScalableDimension scalableDimension) {
            this.scalableDimension(scalableDimension == null ? null : scalableDimension.toString());
            return this;
        }

        public final List<ScalingPolicy.Builder> getScalingPolicies() {
            List<ScalingPolicy.Builder> result = ScalingPoliciesCopier.copyToBuilder(this.scalingPolicies);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setScalingPolicies(Collection<ScalingPolicy.BuilderImpl> scalingPolicies) {
            this.scalingPolicies = ScalingPoliciesCopier.copyFromBuilder(scalingPolicies);
        }

        @Override
        @Transient
        public final Builder scalingPolicies(Collection<ScalingPolicy> scalingPolicies) {
            this.scalingPolicies = ScalingPoliciesCopier.copy(scalingPolicies);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder scalingPolicies(ScalingPolicy... scalingPolicies) {
            scalingPolicies(Arrays.asList(scalingPolicies));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder scalingPolicies(Consumer<ScalingPolicy.Builder>... scalingPolicies) {
            scalingPolicies(Stream.of(scalingPolicies).map(c -> ScalingPolicy.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getScalingStatusCode() {
            return scalingStatusCode;
        }

        public final void setScalingStatusCode(String scalingStatusCode) {
            this.scalingStatusCode = scalingStatusCode;
        }

        @Override
        @Transient
        public final Builder scalingStatusCode(String scalingStatusCode) {
            this.scalingStatusCode = scalingStatusCode;
            return this;
        }

        @Override
        @Transient
        public final Builder scalingStatusCode(ScalingStatusCode scalingStatusCode) {
            this.scalingStatusCode(scalingStatusCode == null ? null : scalingStatusCode.toString());
            return this;
        }

        public final String getScalingStatusMessage() {
            return scalingStatusMessage;
        }

        public final void setScalingStatusMessage(String scalingStatusMessage) {
            this.scalingStatusMessage = scalingStatusMessage;
        }

        @Override
        @Transient
        public final Builder scalingStatusMessage(String scalingStatusMessage) {
            this.scalingStatusMessage = scalingStatusMessage;
            return this;
        }

        @Override
        public ScalingPlanResource build() {
            return new ScalingPlanResource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
