/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.b2bi.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Returns the capability summary details. A trading capability contains the information required to transform incoming
 * EDI documents into JSON or XML outputs.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CapabilitySummary implements SdkPojo, Serializable,
        ToCopyableBuilder<CapabilitySummary.Builder, CapabilitySummary> {
    private static final SdkField<String> CAPABILITY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("capabilityId").getter(getter(CapabilitySummary::capabilityId)).setter(setter(Builder::capabilityId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("capabilityId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(CapabilitySummary::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(CapabilitySummary::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(CapabilitySummary::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> MODIFIED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("modifiedAt")
            .getter(getter(CapabilitySummary::modifiedAt))
            .setter(setter(Builder::modifiedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modifiedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CAPABILITY_ID_FIELD,
            NAME_FIELD, TYPE_FIELD, CREATED_AT_FIELD, MODIFIED_AT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String capabilityId;

    private final String name;

    private final String type;

    private final Instant createdAt;

    private final Instant modifiedAt;

    private CapabilitySummary(BuilderImpl builder) {
        this.capabilityId = builder.capabilityId;
        this.name = builder.name;
        this.type = builder.type;
        this.createdAt = builder.createdAt;
        this.modifiedAt = builder.modifiedAt;
    }

    /**
     * <p>
     * Returns a system-assigned unique identifier for the capability.
     * </p>
     * 
     * @return Returns a system-assigned unique identifier for the capability.
     */
    public final String capabilityId() {
        return capabilityId;
    }

    /**
     * <p>
     * The display name of the capability.
     * </p>
     * 
     * @return The display name of the capability.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * Returns the type of the capability. Currently, only <code>edi</code> is supported.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CapabilityType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Returns the type of the capability. Currently, only <code>edi</code> is supported.
     * @see CapabilityType
     */
    public final CapabilityType type() {
        return CapabilityType.fromValue(type);
    }

    /**
     * <p>
     * Returns the type of the capability. Currently, only <code>edi</code> is supported.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CapabilityType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Returns the type of the capability. Currently, only <code>edi</code> is supported.
     * @see CapabilityType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * Returns a timestamp for creation date and time of the capability.
     * </p>
     * 
     * @return Returns a timestamp for creation date and time of the capability.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * Returns a timestamp that identifies the most recent date and time that the capability was modified.
     * </p>
     * 
     * @return Returns a timestamp that identifies the most recent date and time that the capability was modified.
     */
    public final Instant modifiedAt() {
        return modifiedAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(capabilityId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(modifiedAt());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CapabilitySummary)) {
            return false;
        }
        CapabilitySummary other = (CapabilitySummary) obj;
        return Objects.equals(capabilityId(), other.capabilityId()) && Objects.equals(name(), other.name())
                && Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(modifiedAt(), other.modifiedAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CapabilitySummary").add("CapabilityId", capabilityId()).add("Name", name())
                .add("Type", typeAsString()).add("CreatedAt", createdAt()).add("ModifiedAt", modifiedAt()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "capabilityId":
            return Optional.ofNullable(clazz.cast(capabilityId()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "modifiedAt":
            return Optional.ofNullable(clazz.cast(modifiedAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CapabilitySummary, T> g) {
        return obj -> g.apply((CapabilitySummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CapabilitySummary> {
        /**
         * <p>
         * Returns a system-assigned unique identifier for the capability.
         * </p>
         * 
         * @param capabilityId
         *        Returns a system-assigned unique identifier for the capability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilityId(String capabilityId);

        /**
         * <p>
         * The display name of the capability.
         * </p>
         * 
         * @param name
         *        The display name of the capability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Returns the type of the capability. Currently, only <code>edi</code> is supported.
         * </p>
         * 
         * @param type
         *        Returns the type of the capability. Currently, only <code>edi</code> is supported.
         * @see CapabilityType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapabilityType
         */
        Builder type(String type);

        /**
         * <p>
         * Returns the type of the capability. Currently, only <code>edi</code> is supported.
         * </p>
         * 
         * @param type
         *        Returns the type of the capability. Currently, only <code>edi</code> is supported.
         * @see CapabilityType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapabilityType
         */
        Builder type(CapabilityType type);

        /**
         * <p>
         * Returns a timestamp for creation date and time of the capability.
         * </p>
         * 
         * @param createdAt
         *        Returns a timestamp for creation date and time of the capability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * Returns a timestamp that identifies the most recent date and time that the capability was modified.
         * </p>
         * 
         * @param modifiedAt
         *        Returns a timestamp that identifies the most recent date and time that the capability was modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modifiedAt(Instant modifiedAt);
    }

    static final class BuilderImpl implements Builder {
        private String capabilityId;

        private String name;

        private String type;

        private Instant createdAt;

        private Instant modifiedAt;

        private BuilderImpl() {
        }

        private BuilderImpl(CapabilitySummary model) {
            capabilityId(model.capabilityId);
            name(model.name);
            type(model.type);
            createdAt(model.createdAt);
            modifiedAt(model.modifiedAt);
        }

        public final String getCapabilityId() {
            return capabilityId;
        }

        public final void setCapabilityId(String capabilityId) {
            this.capabilityId = capabilityId;
        }

        @Override
        public final Builder capabilityId(String capabilityId) {
            this.capabilityId = capabilityId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(CapabilityType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getModifiedAt() {
            return modifiedAt;
        }

        public final void setModifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
        }

        @Override
        public final Builder modifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
            return this;
        }

        @Override
        public CapabilitySummary build() {
            return new CapabilitySummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
