/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.b2bi;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.b2bi.internal.B2BiServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.b2bi.model.AccessDeniedException;
import software.amazon.awssdk.services.b2bi.model.B2BiException;
import software.amazon.awssdk.services.b2bi.model.ConflictException;
import software.amazon.awssdk.services.b2bi.model.CreateCapabilityRequest;
import software.amazon.awssdk.services.b2bi.model.CreateCapabilityResponse;
import software.amazon.awssdk.services.b2bi.model.CreatePartnershipRequest;
import software.amazon.awssdk.services.b2bi.model.CreatePartnershipResponse;
import software.amazon.awssdk.services.b2bi.model.CreateProfileRequest;
import software.amazon.awssdk.services.b2bi.model.CreateProfileResponse;
import software.amazon.awssdk.services.b2bi.model.CreateTransformerRequest;
import software.amazon.awssdk.services.b2bi.model.CreateTransformerResponse;
import software.amazon.awssdk.services.b2bi.model.DeleteCapabilityRequest;
import software.amazon.awssdk.services.b2bi.model.DeleteCapabilityResponse;
import software.amazon.awssdk.services.b2bi.model.DeletePartnershipRequest;
import software.amazon.awssdk.services.b2bi.model.DeletePartnershipResponse;
import software.amazon.awssdk.services.b2bi.model.DeleteProfileRequest;
import software.amazon.awssdk.services.b2bi.model.DeleteProfileResponse;
import software.amazon.awssdk.services.b2bi.model.DeleteTransformerRequest;
import software.amazon.awssdk.services.b2bi.model.DeleteTransformerResponse;
import software.amazon.awssdk.services.b2bi.model.GetCapabilityRequest;
import software.amazon.awssdk.services.b2bi.model.GetCapabilityResponse;
import software.amazon.awssdk.services.b2bi.model.GetPartnershipRequest;
import software.amazon.awssdk.services.b2bi.model.GetPartnershipResponse;
import software.amazon.awssdk.services.b2bi.model.GetProfileRequest;
import software.amazon.awssdk.services.b2bi.model.GetProfileResponse;
import software.amazon.awssdk.services.b2bi.model.GetTransformerJobRequest;
import software.amazon.awssdk.services.b2bi.model.GetTransformerJobResponse;
import software.amazon.awssdk.services.b2bi.model.GetTransformerRequest;
import software.amazon.awssdk.services.b2bi.model.GetTransformerResponse;
import software.amazon.awssdk.services.b2bi.model.InternalServerException;
import software.amazon.awssdk.services.b2bi.model.ListCapabilitiesRequest;
import software.amazon.awssdk.services.b2bi.model.ListCapabilitiesResponse;
import software.amazon.awssdk.services.b2bi.model.ListPartnershipsRequest;
import software.amazon.awssdk.services.b2bi.model.ListPartnershipsResponse;
import software.amazon.awssdk.services.b2bi.model.ListProfilesRequest;
import software.amazon.awssdk.services.b2bi.model.ListProfilesResponse;
import software.amazon.awssdk.services.b2bi.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.b2bi.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.b2bi.model.ListTransformersRequest;
import software.amazon.awssdk.services.b2bi.model.ListTransformersResponse;
import software.amazon.awssdk.services.b2bi.model.ResourceNotFoundException;
import software.amazon.awssdk.services.b2bi.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.b2bi.model.StartTransformerJobRequest;
import software.amazon.awssdk.services.b2bi.model.StartTransformerJobResponse;
import software.amazon.awssdk.services.b2bi.model.TagResourceRequest;
import software.amazon.awssdk.services.b2bi.model.TagResourceResponse;
import software.amazon.awssdk.services.b2bi.model.TestMappingRequest;
import software.amazon.awssdk.services.b2bi.model.TestMappingResponse;
import software.amazon.awssdk.services.b2bi.model.TestParsingRequest;
import software.amazon.awssdk.services.b2bi.model.TestParsingResponse;
import software.amazon.awssdk.services.b2bi.model.ThrottlingException;
import software.amazon.awssdk.services.b2bi.model.UntagResourceRequest;
import software.amazon.awssdk.services.b2bi.model.UntagResourceResponse;
import software.amazon.awssdk.services.b2bi.model.UpdateCapabilityRequest;
import software.amazon.awssdk.services.b2bi.model.UpdateCapabilityResponse;
import software.amazon.awssdk.services.b2bi.model.UpdatePartnershipRequest;
import software.amazon.awssdk.services.b2bi.model.UpdatePartnershipResponse;
import software.amazon.awssdk.services.b2bi.model.UpdateProfileRequest;
import software.amazon.awssdk.services.b2bi.model.UpdateProfileResponse;
import software.amazon.awssdk.services.b2bi.model.UpdateTransformerRequest;
import software.amazon.awssdk.services.b2bi.model.UpdateTransformerResponse;
import software.amazon.awssdk.services.b2bi.model.ValidationException;
import software.amazon.awssdk.services.b2bi.transform.CreateCapabilityRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.CreatePartnershipRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.CreateProfileRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.CreateTransformerRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.DeleteCapabilityRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.DeletePartnershipRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.DeleteProfileRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.DeleteTransformerRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.GetCapabilityRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.GetPartnershipRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.GetProfileRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.GetTransformerJobRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.GetTransformerRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.ListCapabilitiesRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.ListPartnershipsRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.ListProfilesRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.ListTransformersRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.StartTransformerJobRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.TestMappingRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.TestParsingRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.UpdateCapabilityRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.UpdatePartnershipRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.UpdateProfileRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.UpdateTransformerRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link B2BiAsyncClient}.
 *
 * @see B2BiAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultB2BiAsyncClient implements B2BiAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultB2BiAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultB2BiAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Instantiates a capability based on the specified parameters. A trading capability contains the information
     * required to transform incoming EDI documents into JSON or XML outputs.
     * </p>
     *
     * @param createCapabilityRequest
     * @return A Java Future containing the result of the CreateCapability operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException A conflict exception is thrown when you attempt to delete a resource (such as a
     *         profile or a capability) that is being used by other resources.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>ServiceQuotaExceededException Occurs when the calling command attempts to exceed one of the service
     *         quotas, for example trying to create a capability when you already have the maximum number of
     *         capabilities allowed.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.CreateCapability
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/CreateCapability" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCapabilityResponse> createCapability(CreateCapabilityRequest createCapabilityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createCapabilityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCapabilityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCapability");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCapabilityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateCapabilityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateCapabilityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCapabilityRequest, CreateCapabilityResponse>()
                            .withOperationName("CreateCapability").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateCapabilityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createCapabilityRequest));
            CompletableFuture<CreateCapabilityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a partnership between a customer and a trading partner, based on the supplied parameters. A partnership
     * represents the connection between you and your trading partner. It ties together a profile and one or more
     * trading capabilities.
     * </p>
     *
     * @param createPartnershipRequest
     * @return A Java Future containing the result of the CreatePartnership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException A conflict exception is thrown when you attempt to delete a resource (such as a
     *         profile or a capability) that is being used by other resources.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>ServiceQuotaExceededException Occurs when the calling command attempts to exceed one of the service
     *         quotas, for example trying to create a capability when you already have the maximum number of
     *         capabilities allowed.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.CreatePartnership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/CreatePartnership" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePartnershipResponse> createPartnership(CreatePartnershipRequest createPartnershipRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPartnershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPartnershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePartnership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePartnershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePartnershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePartnershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePartnershipRequest, CreatePartnershipResponse>()
                            .withOperationName("CreatePartnership").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePartnershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPartnershipRequest));
            CompletableFuture<CreatePartnershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a customer profile. You can have up to five customer profiles, each representing a distinct private
     * network. A profile is the mechanism used to create the concept of a private network.
     * </p>
     *
     * @param createProfileRequest
     * @return A Java Future containing the result of the CreateProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException A conflict exception is thrown when you attempt to delete a resource (such as a
     *         profile or a capability) that is being used by other resources.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>ServiceQuotaExceededException Occurs when the calling command attempts to exceed one of the service
     *         quotas, for example trying to create a capability when you already have the maximum number of
     *         capabilities allowed.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.CreateProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/CreateProfile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProfileResponse> createProfile(CreateProfileRequest createProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProfileRequest, CreateProfileResponse>()
                            .withOperationName("CreateProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createProfileRequest));
            CompletableFuture<CreateProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a transformer. A transformer describes how to process the incoming EDI documents and extract the
     * necessary information to the output file.
     * </p>
     *
     * @param createTransformerRequest
     * @return A Java Future containing the result of the CreateTransformer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException A conflict exception is thrown when you attempt to delete a resource (such as a
     *         profile or a capability) that is being used by other resources.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>ServiceQuotaExceededException Occurs when the calling command attempts to exceed one of the service
     *         quotas, for example trying to create a capability when you already have the maximum number of
     *         capabilities allowed.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.CreateTransformer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/CreateTransformer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTransformerResponse> createTransformer(CreateTransformerRequest createTransformerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTransformerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTransformerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTransformer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTransformerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateTransformerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTransformerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTransformerRequest, CreateTransformerResponse>()
                            .withOperationName("CreateTransformer").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateTransformerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createTransformerRequest));
            CompletableFuture<CreateTransformerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified capability. A trading capability contains the information required to transform incoming
     * EDI documents into JSON or XML outputs.
     * </p>
     *
     * @param deleteCapabilityRequest
     * @return A Java Future containing the result of the DeleteCapability operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException A conflict exception is thrown when you attempt to delete a resource (such as a
     *         profile or a capability) that is being used by other resources.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.DeleteCapability
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/DeleteCapability" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCapabilityResponse> deleteCapability(DeleteCapabilityRequest deleteCapabilityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCapabilityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCapabilityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCapability");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCapabilityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteCapabilityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteCapabilityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCapabilityRequest, DeleteCapabilityResponse>()
                            .withOperationName("DeleteCapability").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteCapabilityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteCapabilityRequest));
            CompletableFuture<DeleteCapabilityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified partnership. A partnership represents the connection between you and your trading partner.
     * It ties together a profile and one or more trading capabilities.
     * </p>
     *
     * @param deletePartnershipRequest
     * @return A Java Future containing the result of the DeletePartnership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException A conflict exception is thrown when you attempt to delete a resource (such as a
     *         profile or a capability) that is being used by other resources.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.DeletePartnership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/DeletePartnership" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePartnershipResponse> deletePartnership(DeletePartnershipRequest deletePartnershipRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePartnershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePartnershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePartnership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePartnershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePartnershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePartnershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePartnershipRequest, DeletePartnershipResponse>()
                            .withOperationName("DeletePartnership").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePartnershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePartnershipRequest));
            CompletableFuture<DeletePartnershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified profile. A profile is the mechanism used to create the concept of a private network.
     * </p>
     *
     * @param deleteProfileRequest
     * @return A Java Future containing the result of the DeleteProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException A conflict exception is thrown when you attempt to delete a resource (such as a
     *         profile or a capability) that is being used by other resources.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.DeleteProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/DeleteProfile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProfileResponse> deleteProfile(DeleteProfileRequest deleteProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProfileRequest, DeleteProfileResponse>()
                            .withOperationName("DeleteProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteProfileRequest));
            CompletableFuture<DeleteProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified transformer. A transformer describes how to process the incoming EDI documents and extract
     * the necessary information to the output file.
     * </p>
     *
     * @param deleteTransformerRequest
     * @return A Java Future containing the result of the DeleteTransformer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException A conflict exception is thrown when you attempt to delete a resource (such as a
     *         profile or a capability) that is being used by other resources.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.DeleteTransformer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/DeleteTransformer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTransformerResponse> deleteTransformer(DeleteTransformerRequest deleteTransformerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTransformerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTransformerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTransformer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTransformerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteTransformerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTransformerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTransformerRequest, DeleteTransformerResponse>()
                            .withOperationName("DeleteTransformer").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteTransformerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTransformerRequest));
            CompletableFuture<DeleteTransformerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the details for the specified capability. A trading capability contains the information required to
     * transform incoming EDI documents into JSON or XML outputs.
     * </p>
     *
     * @param getCapabilityRequest
     * @return A Java Future containing the result of the GetCapability operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.GetCapability
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/GetCapability" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetCapabilityResponse> getCapability(GetCapabilityRequest getCapabilityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCapabilityRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCapabilityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCapability");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCapabilityResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetCapabilityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCapabilityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCapabilityRequest, GetCapabilityResponse>()
                            .withOperationName("GetCapability").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetCapabilityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getCapabilityRequest));
            CompletableFuture<GetCapabilityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the details for a partnership, based on the partner and profile IDs specified. A partnership represents
     * the connection between you and your trading partner. It ties together a profile and one or more trading
     * capabilities.
     * </p>
     *
     * @param getPartnershipRequest
     * @return A Java Future containing the result of the GetPartnership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.GetPartnership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/GetPartnership" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetPartnershipResponse> getPartnership(GetPartnershipRequest getPartnershipRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPartnershipRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPartnershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPartnership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPartnershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPartnershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPartnershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPartnershipRequest, GetPartnershipResponse>()
                            .withOperationName("GetPartnership").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPartnershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPartnershipRequest));
            CompletableFuture<GetPartnershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the details for the profile specified by the profile ID. A profile is the mechanism used to create the
     * concept of a private network.
     * </p>
     *
     * @param getProfileRequest
     * @return A Java Future containing the result of the GetProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.GetProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/GetProfile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetProfileResponse> getProfile(GetProfileRequest getProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetProfileRequest, GetProfileResponse>().withOperationName("GetProfile")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getProfileRequest));
            CompletableFuture<GetProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the details for the transformer specified by the transformer ID. A transformer describes how to process
     * the incoming EDI documents and extract the necessary information to the output file.
     * </p>
     *
     * @param getTransformerRequest
     * @return A Java Future containing the result of the GetTransformer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.GetTransformer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/GetTransformer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTransformerResponse> getTransformer(GetTransformerRequest getTransformerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTransformerRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTransformerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTransformer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTransformerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTransformerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTransformerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTransformerRequest, GetTransformerResponse>()
                            .withOperationName("GetTransformer").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTransformerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTransformerRequest));
            CompletableFuture<GetTransformerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the details of the transformer run, based on the Transformer job ID.
     * </p>
     *
     * @param getTransformerJobRequest
     * @return A Java Future containing the result of the GetTransformerJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.GetTransformerJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/GetTransformerJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTransformerJobResponse> getTransformerJob(GetTransformerJobRequest getTransformerJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTransformerJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTransformerJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTransformerJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTransformerJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTransformerJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTransformerJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTransformerJobRequest, GetTransformerJobResponse>()
                            .withOperationName("GetTransformerJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTransformerJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTransformerJobRequest));
            CompletableFuture<GetTransformerJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the capabilities associated with your Amazon Web Services account for your current or specified region. A
     * trading capability contains the information required to transform incoming EDI documents into JSON or XML
     * outputs.
     * </p>
     *
     * @param listCapabilitiesRequest
     * @return A Java Future containing the result of the ListCapabilities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.ListCapabilities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/ListCapabilities" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListCapabilitiesResponse> listCapabilities(ListCapabilitiesRequest listCapabilitiesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCapabilitiesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCapabilitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCapabilities");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCapabilitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCapabilitiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCapabilitiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCapabilitiesRequest, ListCapabilitiesResponse>()
                            .withOperationName("ListCapabilities").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListCapabilitiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listCapabilitiesRequest));
            CompletableFuture<ListCapabilitiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the partnerships associated with your Amazon Web Services account for your current or specified region. A
     * partnership represents the connection between you and your trading partner. It ties together a profile and one or
     * more trading capabilities.
     * </p>
     *
     * @param listPartnershipsRequest
     * @return A Java Future containing the result of the ListPartnerships operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.ListPartnerships
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/ListPartnerships" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPartnershipsResponse> listPartnerships(ListPartnershipsRequest listPartnershipsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPartnershipsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPartnershipsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPartnerships");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPartnershipsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPartnershipsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPartnershipsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPartnershipsRequest, ListPartnershipsResponse>()
                            .withOperationName("ListPartnerships").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPartnershipsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPartnershipsRequest));
            CompletableFuture<ListPartnershipsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the profiles associated with your Amazon Web Services account for your current or specified region. A
     * profile is the mechanism used to create the concept of a private network.
     * </p>
     *
     * @param listProfilesRequest
     * @return A Java Future containing the result of the ListProfiles operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.ListProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/ListProfiles" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListProfilesResponse> listProfiles(ListProfilesRequest listProfilesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProfilesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProfiles");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProfilesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListProfilesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProfilesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProfilesRequest, ListProfilesResponse>()
                            .withOperationName("ListProfiles").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListProfilesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listProfilesRequest));
            CompletableFuture<ListProfilesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the tags associated with the Amazon Resource Name (ARN) that you specify. The resource can be a
     * capability, partnership, profile, or transformer.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the available transformers. A transformer describes how to process the incoming EDI documents and extract
     * the necessary information to the output file.
     * </p>
     *
     * @param listTransformersRequest
     * @return A Java Future containing the result of the ListTransformers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.ListTransformers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/ListTransformers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTransformersResponse> listTransformers(ListTransformersRequest listTransformersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTransformersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTransformersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTransformers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTransformersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTransformersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTransformersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTransformersRequest, ListTransformersResponse>()
                            .withOperationName("ListTransformers").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTransformersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTransformersRequest));
            CompletableFuture<ListTransformersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Runs a job, using a transformer, to parse input EDI (electronic data interchange) file into the output structures
     * used by Amazon Web Services B2BI Data Interchange.
     * </p>
     * <p>
     * If you only want to transform EDI (electronic data interchange) documents, you don't need to create profiles,
     * partnerships or capabilities. Just create and configure a transformer, and then run the
     * <code>StartTransformerJob</code> API to process your files.
     * </p>
     *
     * @param startTransformerJobRequest
     * @return A Java Future containing the result of the StartTransformerJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.StartTransformerJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/StartTransformerJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartTransformerJobResponse> startTransformerJob(
            StartTransformerJobRequest startTransformerJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startTransformerJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startTransformerJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartTransformerJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartTransformerJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartTransformerJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartTransformerJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartTransformerJobRequest, StartTransformerJobResponse>()
                            .withOperationName("StartTransformerJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartTransformerJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startTransformerJobRequest));
            CompletableFuture<StartTransformerJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Attaches a key-value pair to a resource, as identified by its Amazon Resource Name (ARN). Resources are
     * capability, partnership, profile, transformers and other entities.
     * </p>
     * <p>
     * There is no response returned from this call.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Maps the input file according to the provided template file. The API call downloads the file contents from the
     * Amazon S3 location, and passes the contents in as a string, to the <code>inputFileContent</code> parameter.
     * </p>
     *
     * @param testMappingRequest
     * @return A Java Future containing the result of the TestMapping operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.TestMapping
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/TestMapping" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TestMappingResponse> testMapping(TestMappingRequest testMappingRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(testMappingRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, testMappingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TestMapping");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TestMappingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TestMappingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TestMappingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TestMappingRequest, TestMappingResponse>()
                            .withOperationName("TestMapping").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TestMappingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(testMappingRequest));
            CompletableFuture<TestMappingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Parses the input EDI (electronic data interchange) file. The input file has a file size limit of 250 KB.
     * </p>
     *
     * @param testParsingRequest
     * @return A Java Future containing the result of the TestParsing operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.TestParsing
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/TestParsing" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TestParsingResponse> testParsing(TestParsingRequest testParsingRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(testParsingRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, testParsingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TestParsing");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TestParsingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TestParsingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TestParsingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TestParsingRequest, TestParsingResponse>()
                            .withOperationName("TestParsing").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TestParsingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(testParsingRequest));
            CompletableFuture<TestParsingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Detaches a key-value pair from the specified resource, as identified by its Amazon Resource Name (ARN). Resources
     * are capability, partnership, profile, transformers and other entities.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates some of the parameters for a capability, based on the specified parameters. A trading capability contains
     * the information required to transform incoming EDI documents into JSON or XML outputs.
     * </p>
     *
     * @param updateCapabilityRequest
     * @return A Java Future containing the result of the UpdateCapability operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException A conflict exception is thrown when you attempt to delete a resource (such as a
     *         profile or a capability) that is being used by other resources.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>ServiceQuotaExceededException Occurs when the calling command attempts to exceed one of the service
     *         quotas, for example trying to create a capability when you already have the maximum number of
     *         capabilities allowed.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.UpdateCapability
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/UpdateCapability" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateCapabilityResponse> updateCapability(UpdateCapabilityRequest updateCapabilityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateCapabilityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateCapabilityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCapability");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateCapabilityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateCapabilityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateCapabilityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateCapabilityRequest, UpdateCapabilityResponse>()
                            .withOperationName("UpdateCapability").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateCapabilityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateCapabilityRequest));
            CompletableFuture<UpdateCapabilityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates some of the parameters for a partnership between a customer and trading partner. A partnership represents
     * the connection between you and your trading partner. It ties together a profile and one or more trading
     * capabilities.
     * </p>
     *
     * @param updatePartnershipRequest
     * @return A Java Future containing the result of the UpdatePartnership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException A conflict exception is thrown when you attempt to delete a resource (such as a
     *         profile or a capability) that is being used by other resources.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>ServiceQuotaExceededException Occurs when the calling command attempts to exceed one of the service
     *         quotas, for example trying to create a capability when you already have the maximum number of
     *         capabilities allowed.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.UpdatePartnership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/UpdatePartnership" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePartnershipResponse> updatePartnership(UpdatePartnershipRequest updatePartnershipRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePartnershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePartnershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePartnership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePartnershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePartnershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePartnershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePartnershipRequest, UpdatePartnershipResponse>()
                            .withOperationName("UpdatePartnership").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdatePartnershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePartnershipRequest));
            CompletableFuture<UpdatePartnershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified parameters for a profile. A profile is the mechanism used to create the concept of a
     * private network.
     * </p>
     *
     * @param updateProfileRequest
     * @return A Java Future containing the result of the UpdateProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException A conflict exception is thrown when you attempt to delete a resource (such as a
     *         profile or a capability) that is being used by other resources.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>ServiceQuotaExceededException Occurs when the calling command attempts to exceed one of the service
     *         quotas, for example trying to create a capability when you already have the maximum number of
     *         capabilities allowed.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.UpdateProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/UpdateProfile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateProfileResponse> updateProfile(UpdateProfileRequest updateProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateProfileRequest, UpdateProfileResponse>()
                            .withOperationName("UpdateProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateProfileRequest));
            CompletableFuture<UpdateProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified parameters for a transformer. A transformer describes how to process the incoming EDI
     * documents and extract the necessary information to the output file.
     * </p>
     *
     * @param updateTransformerRequest
     * @return A Java Future containing the result of the UpdateTransformer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException A conflict exception is thrown when you attempt to delete a resource (such as a
     *         profile or a capability) that is being used by other resources.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException Occurs when a B2BI object cannot be validated against a request from another
     *         object.</li>
     *         <li>ThrottlingException The request was denied due to throttling: the data speed and rendering may be
     *         limited depending on various parameters and conditions.</li>
     *         <li>ResourceNotFoundException Occurs when the requested resource does not exist, or cannot be found. In
     *         some cases, the resource exists in a region other than the region specified in the API call.</li>
     *         <li>ServiceQuotaExceededException Occurs when the calling command attempts to exceed one of the service
     *         quotas, for example trying to create a capability when you already have the maximum number of
     *         capabilities allowed.</li>
     *         <li>InternalServerException This exception is thrown when an error occurs in the Amazon Web Services B2B
     *         Data Interchange service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>B2BiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample B2BiAsyncClient.UpdateTransformer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/UpdateTransformer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTransformerResponse> updateTransformer(UpdateTransformerRequest updateTransformerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateTransformerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTransformerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTransformer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTransformerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateTransformerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateTransformerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTransformerRequest, UpdateTransformerResponse>()
                            .withOperationName("UpdateTransformer").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateTransformerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateTransformerRequest));
            CompletableFuture<UpdateTransformerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final B2BiServiceClientConfiguration serviceClientConfiguration() {
        return new B2BiServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(B2BiException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        B2BiServiceClientConfigurationBuilder serviceConfigBuilder = new B2BiServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
